/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Carsten Driesner <carsten.driesner@open-xchange.com>
 * @author Mario Schroeder <mario.schroeder@open-xchange.com>
 */

define('io.ox/office/editframework/app/rtconnectionviewer', [
    'io.ox/core/uuids',
    'io.ox/office/tk/io',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/logger',
    'io.ox/office/tk/utils/scheduler',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/baseframework/utils/errorcode',
    'io.ox/office/editframework/utils/editconfig'
], function (UUID, IO, Utils, Logger, Scheduler, TriggerObject, ErrorCode, Config) {

    'use strict';

    // private static functions ===============================================

    // class RTConnectionViewer =====================================================

    /**
     * Represents the connection to the real-time framework, used to send
     * message to and receive message from the server.
     *
     * Triggers the following events:
     * - 'update': Update notifications containing actions created by remote
     *  clients, and state information for the application (read-only, name and
     *  identifier of the current editor, etc.).
     *
     * @extends TriggerObject
     *
     * @param {EditApplication} app
     *  The application containing this operations queue.
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     */
    function RTConnectionViewer(app /*, initOptions*/) {

        // self reference
        //var self = this;

        // file descriptor of the edited document
        var file = app.getFileDescriptor();

        var traceActions = {};

        // is encrypted document files
        var isEncrypted = false;

        var uuid = UUID.randomUUID();

        var self = this;

        // base constructor ---------------------------------------------------

        TriggerObject.call(this, app);

        // private methods ----------------------------------------------------

        // methods ------------------------------------------------------------

        /**
         * Connects the real-time group and sends the initial request for the
         * document import operations.
         *
         * @param {Object} connectData
         *  Contains properties to be sent while joining to the
         *  document real-time connection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved with the getActions results.
         */
        this.connect = function (connectData) {

            var connectDeferred = Scheduler.createDeferred(app, 'RTConnectionViewer.connect');
            var getOpsData = { action: 'getoperations', id: file.id, folder_id: file.folder_id };

            if (_.isString(connectData.auth_code) && !_.isEmpty(connectData.auth_code)) {
                isEncrypted = true;
                _.extend(getOpsData, { cryptoAuth: connectData.auth_code });
            }

            app.sendRequest(IO.FILTER_MODULE_NAME, getOpsData).done(function (data) {
                var error = new ErrorCode(data);
                var dataRemaining = null;

                if (!error.isError()) {
                    data.clientId = 'dummy';
                    if (_.isObject(data.preview)) {
                        dataRemaining = {};
                        dataRemaining.clientId = 'dummy';
                        dataRemaining.actions = [{ operations: data.operations }];
                        dataRemaining.error = data.error;
                    } else {
                        data.syncLoad = true;
                        data.actions = [{ operations: data.operations }];
                    }
                    delete data.operations;
                    connectDeferred.resolve(data);

                    if (dataRemaining) {
                        dataRemaining.finalLoadUpdate = true;

                        app.executeDelayed(function () {
                            self.trigger('update', dataRemaining);

                        }, 'RTConnectionViewer.connect()', { delay: 100 });
                    }
                } else {
                    connectDeferred.reject(data);
                }
            });

            // Remember *connect* deferred to be able to reject them if user
            // closes document while we are still connecting to the realtime group!
            return this.createAbortablePromise(connectDeferred);
        };

        /**
         * Send the given operations to the connected RT object
         * for further distribution to other connected clients
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the ACK of the internal send
         *  arrives.
         */
        this.sendActions = function () {
            return $.when();
        };

        /**
         * Sends user data update to the server. This user update data will
         * broadcasted to all collaborating clients.
         *
         * @returns {jQuery.Promise}
         */
        this.updateUserData =  function () {
            return $.when();
        };

        /**
         * Sends a real-time request to the server and waits for a response.
         *
         * @param {String} action
         *  The identifier of the action sent to the server.
         *
         * @param {Number} [timeout]
         *  If specified, the delay time in milliseconds, after the Promise
         *  returned by this method will be rejected, if the query is still
         *  pending.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved with the answer of the server
         *  request, or rejected on error or timeout.
         */
        this.sendQuery =  function () {
            return $.when();
        };

        /**
         * Initiates flushing of the document using a synchronous request.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the answer for flushDocument
         *  from the server arrives.
         */
        this.flushDocument =  function () {
            return $.when();
        };

        /**
         * Send our wish to acquire the edit rights to the connected RT object
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the ACK of the internal send
         *  arrives.
         */
        this.acquireEditRights = function () {
            return $.when();
        };

        /**
         * Acknowledge to the server that our preparation to lose the
         * edit rights are completed.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the ACK of the internal send
         *  arrives.
         */
        this.canLoseEditRights = function () {
            return $.when();
        };

        /**
         * Sends client log message to the server so it can be written to the
         * server side log files for later debugging.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the ACK of the internal send
         *  arrives.
         */
        this.sendLogMessage =  function () {
            return $.when();
        };

        /**
         * Sends a sync request to the server, which will answer with a update
         * notification. This can be used to synchronize client and server
         * after an off-/online scenario.
         */
        this.sync =  function () {
            return $.when();
        };

        /**
         * Sends a "alive" answer to the server, which extends the time
         * for client to make necessary changes until a "edit rights" switch will
         * be forced due to time-out.
         *
         * @return {jQuery.Promise}
         *  A promise that will be resolved when the ACK of the internal send
         *  arrives.
         */
        this.alive =  function () {
            return $.when();
        };

        /**
         * Close the document at the connected RT object
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved with the closing document status of
         *  the request.
         */
        this.closeDocument = function () {
            if (isEncrypted) {
                var getOpsData = { action: 'getoperations', subaction: 'close', id: file.id, folder_id: file.folder_id };

                return app.sendRequest(IO.FILTER_MODULE_NAME, getOpsData);
            } else {
                return $.when();
            }
        };

        /**
         * Provides the unique real-time id used by this connection.
         *
         * @returns {String}
         *  The unique real-time id used by this connection.
         */
        this.getUuid = function () {
            return uuid;
        };

        /**
         * Retrieves the pending actions, not processed by the backend. These
         * actions can be provided to a restoreDocument() function to try to
         * restore the document.
         *
         * @returns {Object}
         *  The result object contains the property "actions" which is an Array
         *  with all pending operations (without server-side ack).
         */
        this.getPendingActions = function () {
            return { actions: [] };
        };

        /**
         * Determines, if we have pending actions which have not been acknowledge by
         * the OX Documents backend.
         *
         * @returns {Boolean} TRUE, if there are pending actions, otherwise FALSE.
         */
        this.hasPendingActions = function () {
            return false;
        };

        // initialization -----------------------------------------------------

        RTConnectionViewer.log('RTConnectionViewer initialization');

        // read traceable actions from page URL
        if (_.url.hash('office:trace')) {
            _.each(_.url.hash('office:trace').split(/\s*,\s*/), function (action) {
                traceActions[action.toLowerCase()] = true;
            });
        }

        // disconnect from the RT object, no further calls should be made at this object from now on
        this.registerDestructor(function () {

            RTConnectionViewer.log('RTConnectionViewer.destroy()');

            // remove references
            file = traceActions = null;
        });

    } // class RTConnectionViewer

    // global initialization ==================================================

    // whether to show console output of this RTConnectionViewer class
    RTConnectionViewer.debug = Config.DEBUG && Config.getDebugState('module/realtimedebugging', 'office:log-realtime');

    // extend statically with logging methods
    Logger.extend(RTConnectionViewer, { enable: RTConnectionViewer.debug, prefix: 'RT' });

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: RTConnectionViewer });

});
