/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Mario Schroeder <mario.schroeder@open-xchange.com>
 */

/* eslint no-new: 0 */

define('io.ox/office/editframework/app/pimactions', [
    'io.ox/core/extensions',
    'io.ox/core/extPatterns/links',
    'io.ox/core/notifications',
    'io.ox/core/api/attachment',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/driveutils',
    'io.ox/office/tk/io',
    'io.ox/office/baseframework/utils/infomessages',
    'io.ox/office/baseframework/app/extensionregistry',
    'gettext!io.ox/office/editframework/actions'
], function (ext, links, CoreNotification, attachmentAPI, Utils, DriveUtils, IO, InfoMessages, ExtensionRegistry, gt) {

    'use strict';

    // static class PIMActions ===============================================

    var PIMActions = {};

    // methods ----------------------------------------------------------------

    /**
     * Creates all required actions and links in the Calendar and Tasks
     * application for a specific OX Documents application.
     *
     * @param {String} appBaseName
     *  The base name of the application (name of the application source code
     *  directory).
     *
     * @param {FileExtensions} fileExtensions
     *  A singleton object containing configuration settings of all file
     *  extensions supported by this application instance.
     *
     */
    PIMActions.create = function (appBaseName) {

        // the pim extension point
        var ACTION_POINT = 'io.ox/pim/office/' + appBaseName;
        // the module name of the edit application
        var MODULE_NAME  = 'io.ox/office/' + appBaseName;

        // private methods ----------------------------------------------------

        /**
         * Launches a new OX Documents application with the passed options.
         */
        function launchApplication(options) {
            ox.launch(MODULE_NAME + '/main', options);
        }

        /**
         * Handler for AttachmentAPI.save errors.
         */
        function showSaveAttachmentError(error) {
            Utils.error('AttachmentAPI.save makes an error', error);
            require(['io.ox/office/baseframework/utils/errormessages']).done(function (ErrorMessages) {
                CoreNotification.yell('error', ErrorMessages.getMessageData('GENERAL_SERVER_COMPONENT_NOT_WORKING_ERROR').message);
            });
        }

        // initialization -----------------------------------------------------

        // Copy PIM attachment to Files and edit
        new links.Action(ACTION_POINT + '-edit-asnew', {
            id: 'save',
            capabilities: 'infostore',

            requires: function (e) {
                // show the "edit as new" link for single selection only
                if (!e.collection.has('one', 'read')) {
                    return false;
                }

                // e.context can be an object or array dependent on the context
                var context = _.isArray(e.context) ? _.first(e.context) : e.context;
                // filename of the attachment
                var filename = context && context.filename;
                // whether the "edit as new" link should be shown for the attachment
                var editable = ExtensionRegistry.isEditable(filename, MODULE_NAME);

                // prefetch application source code
                if (editable) {
                    IO.prefetchModuleSource(MODULE_NAME);
                }

                return editable;
            },

            action: function (baton) {
                // the id of the target folder in Drive
                var targetFolderId = DriveUtils.getStandardDocumentsFolderId();

                attachmentAPI.save(baton.data, targetFolderId)
                .then(function (fileId) {
                    return DriveUtils.getFile({ id: fileId, folder_id: targetFolderId });

                }, showSaveAttachmentError)
                .then(function (file) {
                    // show a notification to the user that a new file has been created in Drive
                    CoreNotification.yell('success', InfoMessages.getMessageData('INFO_DOC_STORED_IN_DEFAULTFOLDER_AS', { fullFileName: file.filename }).message);

                    // launch the OX Documents editor application
                    if (ExtensionRegistry.isNative(file.filename)) {
                        launchApplication({ action: 'load', file: file });
                    } else {
                        launchApplication({ action: 'convert', target_folder_id: file.folder_id, templateFile: file, preserveFileName: true });
                    }
                });
            }

        });

        var link = {
            id: appBaseName + '-edit',
            index: 1000,
            prio: 'lo',
            mobile: 'lo',
            label: gt('Edit as new'),
            ref: ACTION_POINT + '-edit-asnew'
        };

        // an action link to edit documents attached to a PIM entry
        ext.point('io.ox/core/tk/attachment/links').extend(new links.Link(link));

        // an action link to edit documents attached to a PIM entry
        ext.point('io.ox/core/viewer/toolbar/links/pim').extend(new links.Link(link));

    };

    // exports ================================================================

    return PIMActions;

});
