/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/drawinglayer/view/drawingcontrols', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/controls',
    'io.ox/office/baseframework/view/control/compoundbutton',
    'io.ox/office/baseframework/view/control/compoundsplitbutton',
    'io.ox/office/drawinglayer/view/control/arrowstylepicker',
    'io.ox/office/drawinglayer/view/control/shapetypepicker',
    'io.ox/office/drawinglayer/view/control/charttypepicker',
    'io.ox/office/drawinglayer/view/control/chartcolorsetpicker',
    'io.ox/office/drawinglayer/view/control/chartstylesetpicker',
    'io.ox/office/drawinglayer/view/control/chartlegendpicker',
    'io.ox/office/drawinglayer/view/drawinglabels',
    'gettext!io.ox/office/drawinglayer/main'
], function (Utils, Controls, CompoundButton, CompoundSplitButton, ArrowStylePicker, ShapeTypePicker, ChartTypePicker, ChartColorSetPicker, ChartStyleSetPicker, ChartLegendPicker, DrawingLabels, gt) {

    'use strict';

    // convenience shortcuts
    var Button = Controls.Button;

    // private global functions ===============================================

    /**
     * Inserts all buttons needed to change the Z order of drawing objects to
     * the passed drop-down control.
     *
     * @param {CompoundMenuMixin} control
     *  The drop-down control to add the button controls to.
     */
    function addDrawingOrderButtons(docView, control) {

        control.addGroup('drawing/order', new Button(docView, {
            value: 'front',
            icon: 'docs-arrange-to-front',
            label: /*#. change the order of drawing objects */ gt.pgettext('drawing-pos', 'Bring to front'),
            tooltip: /*#. change the order of drawing objects */ gt.pgettext('drawing-pos', 'Bring object to the front')
        }));

        control.addGroup('drawing/order', new Button(docView, {
            value: 'forward',
            icon: 'docs-arrange-forward',
            label: /*#. change the order of drawing objects */ gt.pgettext('drawing-pos', 'Bring forward'),
            tooltip: /*#. change the order of drawing objects */ gt.pgettext('drawing-pos', 'Move object up one placement to the front')
        }));

        control.addGroup('drawing/order', new Button(docView, {
            value: 'backward',
            icon: 'docs-arrange-backward',
            label: /*#. change the order of drawing objects */ gt.pgettext('drawing-pos', 'Send backward'),
            tooltip: /*#. change the order of drawing objects */ gt.pgettext('drawing-pos', 'Send object one placement to the back')
        }));

        control.addGroup('drawing/order', new Button(docView, {
            value: 'back',
            icon: 'docs-arrange-to-back',
            label: /*#. change the order of drawing objects */ gt.pgettext('drawing-pos', 'Send to back'),
            tooltip: /*#. change the order of drawing objects */ gt.pgettext('drawing-pos', 'Send object to the back')
        }));
    }

    /**
     * Inserts all buttons needed to change the rotation of drawing objects to
     * the passed drop-down control.
     *
     * @param {CompoundMenuMixin} control
     *  The drop-down control to add the button controls to.
     */
    function addDrawingRotateButtons(docView, control) {

        control.addGroup('drawing/transform/rotate', new Button(docView, {
            value: 90,
            icon: 'docs-drawing-rotate-right',
            label: /*#. rotate drawing objects */ gt.pgettext('drawing-pos', 'Rotate right 90\xb0'),
            tooltip: /*#. rotate drawing objects */ gt.pgettext('drawing-pos', 'Rotate right 90 degrees')
        }));

        control.addGroup('drawing/transform/rotate', new Button(docView, {
            value: -90,
            icon: 'docs-drawing-rotate-left',
            label: /*#. rotate drawing objects */ gt.pgettext('drawing-pos', 'Rotate left 90\xb0'),
            tooltip: /*#. rotate drawing objects */ gt.pgettext('drawing-pos', 'Rotate left 90 degrees')
        }));

        control.addGroup('drawing/transform/flipv', new Button(docView, {
            icon: 'docs-drawing-flip-v',
            label: /*#. rotate drawing objects */ gt.pgettext('drawing-pos', 'Flip vertical'),
            tooltip: /*#. rotate drawing objects */ gt.pgettext('drawing-pos', 'Flip object vertical')
        }));

        control.addGroup('drawing/transform/fliph', new Button(docView, {
            icon: 'docs-drawing-flip-h',
            label: /*#. rotate drawing objects */ gt.pgettext('drawing-pos', 'Flip horizontal'),
            tooltip: /*#. rotate drawing objects */ gt.pgettext('drawing-pos', 'Flip object horizontal')
        }));
    }

    /**
     * Inserts all buttons needed to change the horizontal alignment of drawing
     * objects to the passed drop-down control.
     *
     * @param {CompoundMenuMixin} control
     *  The drop-down control to add the button controls to.
     */
    function addDrawingHorAlignButtons(docView, control) {

        control.addGroup('drawing/align', new Button(docView, {
            value: 'left',
            icon: 'docs-drawing-h-align-left',
            label: /*#. horizontal alignment of drawing objects */ gt.pgettext('drawing-pos', 'Align left')
        }));

        control.addGroup('drawing/align', new Button(docView, {
            value: 'center',
            icon: 'docs-drawing-h-align-center',
            label: /*#. horizontal alignment of drawing objects */ gt.pgettext('drawing-pos', 'Align center')
        }));

        control.addGroup('drawing/align', new Button(docView, {
            value: 'right',
            icon: 'docs-drawing-h-align-right',
            label: /*#. horizontal alignment of drawing objects */ gt.pgettext('drawing-pos', 'Align right')
        }));
    }

    /**
     * Inserts all buttons needed to change the vertical alignment of drawing
     * objects to the passed drop-down control.
     *
     * @param {CompoundMenuMixin} control
     *  The drop-down control to add the button controls to.
     */
    function addDrawingVertAlignButtons(docView, control) {

        control.addGroup('drawing/align', new Button(docView, {
            value: 'top',
            icon: 'docs-drawing-v-align-top',
            label: /*#. vertical alignment of drawing objects */ gt.pgettext('drawing-pos', 'Align top')
        }));

        control.addGroup('drawing/align', new Button(docView, {
            value: 'middle',
            icon: 'docs-drawing-v-align-middle',
            label: /*#. vertical alignment of drawing objects */ gt.pgettext('drawing-pos', 'Align middle')
        }));

        control.addGroup('drawing/align', new Button(docView, {
            value: 'bottom',
            icon: 'docs-drawing-v-align-bottom',
            label: /*#. vertical alignment of drawing objects */ gt.pgettext('drawing-pos', 'Align bottom')
        }));
    }

    /**
     * Inserts all buttons needed to distribute multiple drawing objects to the
     * passed drop-down control.
     *
     * @param {CompoundMenuMixin} control
     *  The drop-down control to add the button controls to.
     */
    function addDrawingDistributionButtons(docView, control) {

        control.addGroup('drawing/distribute', new Button(docView, {
            value: 'horzSlide',
            icon: 'docs-distribute-slide-h',
            label: gt.pgettext('drawing-pos', 'Distribute horizontally on slide'),
            tooltip: gt('Distribute selected objects horizontally on slide')
        }));

        control.addGroup('drawing/distribute', new Button(docView, {
            value: 'vertSlide',
            icon: 'docs-distribute-slide-v',
            label: gt.pgettext('drawing-pos', 'Distribute vertically on slide'),
            tooltip: gt('Distribute selected objects vertically on slide')
        }));

        control.addGroup('drawing/distributeamong', new Button(docView, {
            value: 'horzSelection',
            icon: 'docs-distribute-objects-h',
            label: gt.pgettext('drawing-pos', 'Distribute horizontally among objects'),
            tooltip: gt('Distribute selected objects horizontally among themselves')
        }));

        control.addGroup('drawing/distributeamong', new Button(docView, {
            value: 'vertSelection',
            icon: 'docs-distribute-objects-v',
            label: gt.pgettext('drawing-pos', 'Distribute vertically among objects'),
            tooltip: gt('Distribute selected objects vertically among themselves')
        }));
    }

    // static class DrawingControls ===========================================

    /**
     * Provides different classes for GUI form controls for drawing objects in
     * a single map for convenience.
     */
    var DrawingControls = {
        ArrowStylePicker: ArrowStylePicker,
        ChartTypePicker: ChartTypePicker,
        ChartColorSetPicker: ChartColorSetPicker,
        ChartStyleSetPicker: ChartStyleSetPicker,
        ChartLegendPicker: ChartLegendPicker,
        ShapeTypePicker: ShapeTypePicker
    };

    // class ArrowPresetStylePicker ===========================================

    // predefined arrow styles (start type, colon, end type)
    var ARROW_PRESET_STYLES = [
        { value: 'none:none' },
        { value: 'none:arrow' },
        { value: 'arrow:none' },
        { value: 'arrow:arrow' },
        { value: 'none:triangle' },
        { value: 'triangle:none' },
        { value: 'triangle:triangle' },
        { value: 'diamond:triangle' },
        { value: 'oval:triangle' },
        { value: 'diamond:diamond' },
        { value: 'oval:oval' }
    ];

    /**
     * A drop-down list control for line end styles used in OOXML files.
     *
     * @constructor
     *
     * @extends ArrowStylePicker
     */
    DrawingControls.ArrowPresetStylePicker = ArrowStylePicker.extend({ constructor: function (docView) {

        ArrowStylePicker.call(this, docView, ARROW_PRESET_STYLES);

    } }); // class ArrowPresetStylePicker

    // class DrawingOrderPicker ===============================================

    /**
     * Drop-down menu with commands to change the Z order of drawing objects.
     *
     * @constructor
     *
     * @extends CompoundButton
     *
     * @param {EditView} docView
     *  The document view instance owning this control.
     */
    DrawingControls.DrawingOrderPicker = CompoundButton.extend({ constructor: function (docView, initOptions) {

        // base constructor ---------------------------------------------------

        CompoundButton.call(this, docView, Utils.extendOptions({
            //#. drop-down menu to change the Z order of drawing objects
            label: gt.pgettext('drawing-pos', 'Reorder'),
            icon: 'docs-z-order',
            smallerVersion: { hideLabel: true }
        }, initOptions));

        // initialization -----------------------------------------------------

        addDrawingOrderButtons(docView, this);

    } });

    // class DrawingArrangementPicker =========================================

    /**
     * Drop-down menu with commands to change the Z order of drawing objects,
     * and additional rotation and flipping commands.
     *
     * @constructor
     *
     * @extends CompoundButton
     *
     * @param {EditView} docView
     *  The document view instance owning this control.
     */
    DrawingControls.DrawingArrangementPicker = CompoundButton.extend({ constructor: function (docView, initOptions) {

        // base constructor ---------------------------------------------------

        CompoundButton.call(this, docView, Utils.extendOptions({
            //#. drop-down menu to change the Z order and rotation of drawing objects
            label: gt.pgettext('drawing-pos', 'Arrange'),
            icon: 'docs-z-order',
            smallerVersion: { hideLabel: true }
        }, initOptions));

        // initialization -----------------------------------------------------

        this.addSectionLabel(gt('Reorder'));
        addDrawingOrderButtons(docView, this);

        this.addSectionLabel(gt('Rotate'));
        addDrawingRotateButtons(docView, this);

    } }); // class DrawingArrangementPicker

    // class DrawingAlignmentPicker ===========================================

    /**
     * Drop-down menu with commands to change the alignment of drawing objects
     * relative to each other.
     *
     * @constructor
     *
     * @extends CompoundButton
     *
     * @param {EditView} docView
     *  The document view instance owning this control.
     */
    DrawingControls.DrawingAlignmentPicker = CompoundButton.extend({ constructor: function (docView, initOptions) {

        // base constructor ---------------------------------------------------

        CompoundButton.call(this, docView, Utils.extendOptions({
            //#. drop-down menu to change the alignment of drawing objects
            label: gt.pgettext('drawing-pos', 'Align'),
            icon: 'docs-drawing-h-align-left',
            smallerVersion: { hideLabel: true }
        }, initOptions));

        // initialization -----------------------------------------------------

        addDrawingHorAlignButtons(docView, this);
        this.addSeparator();
        addDrawingVertAlignButtons(docView, this);
        this.addSeparator();
        addDrawingDistributionButtons(docView, this);

    } }); // class DrawingAlignmentPicker

    // class InsertTextFrameButton ============================================

    DrawingControls.InsertTextFrameButton = Button.extend({ constructor: function (docView, initOptions) {

        Button.call(this, docView, Utils.extendOptions({
            icon: DrawingLabels.getDrawingTypeIcon('textframe'),
            label: DrawingLabels.getDrawingTypeLabel('textframe'),
            tooltip: gt.pgettext('drawing', 'Insert a text frame'),
            smallerVersion: { css: { width: 35 }, hideLabel: true }
        }, initOptions));

    } });

    // class ImagePicker ======================================================

    DrawingControls.ImagePicker = CompoundSplitButton.extend({ constructor: function (docView, initOptions) {

        // base constructor ---------------------------------------------------

        CompoundSplitButton.call(this, docView, Utils.extendOptions({
            icon: 'docs-image-insert',
            label: DrawingLabels.INSERT_DRAWING_LABEL,
            tooltip: DrawingLabels.INSERT_DRAWING_TOOLTIP,
            smallerVersion: {
                css: { width: 35 },
                hideLabel: true
            },
            value: 'drive'
        }, initOptions));

        var itemKey = Utils.getStringOption(initOptions, 'itemKey', 'image/insert/dialog');
        this.addGroup(itemKey, new Button(docView, { value: 'local', label: /*#. Label for insert image from the local file system. */ gt('Local file') }));
        this.addGroup(itemKey, new Button(docView, { value: 'drive', label: /*#. Label for insert image from drive. Please do not translate the productname 'Drive' */ gt('From Drive') }));
        this.addGroup(itemKey, new Button(docView, { value: 'url', label:  /*#. Label for insert image from an url. */ gt('From URL') }));

    } }); // class ImagePicker

    // class CompactImagePicker ===============================================

    /**
     * Control to pick image from local file or from drive.
     *
     * @constructor
     *
     * @extends CompoundButton
     */
    DrawingControls.CompactImagePicker = CompoundButton.extend({ constructor: function (docView, initOptions) {

        CompoundButton.call(this, docView, Utils.extendOptions({
            icon: 'docs-image-insert',
            label: DrawingLabels.INSERT_DRAWING_LABEL,
            tooltip: DrawingLabels.INSERT_DRAWING_TOOLTIP,
            smallerVersion: {
                hideLabel: true
            }
        }, initOptions));

        var itemKey = Utils.getStringOption(initOptions, 'itemKey', 'image/insert/dialog');
        this.addGroup(itemKey, new Button(docView, { value: 'local', label: /*#. Label for insert image from the local file system. */ gt('Local file') }));
        this.addGroup(itemKey, new Button(docView, { value: 'drive', label: /*#. Label for insert image from drive. Please do not translate the productname 'Drive' */ gt('From Drive') }));

    } }); // class CompactImagePicker

    // class ImageCropPosition ======================================================

    DrawingControls.ImageCropPosition = CompoundSplitButton.extend({ constructor: function (docView, initOptions) {

        // base constructor ---------------------------------------------------

        CompoundSplitButton.call(this, docView, Utils.extendOptions({
            icon: 'fa-crop',
            label: DrawingLabels.CROP_IMAGE_LABEL,
            tooltip: DrawingLabels.CROP_IMAGE_TOOLTIP,
            smallerVersion: {
                css: { width: 35 },
                hideLabel: true
            },
            toggle: true
        }, initOptions));

        this.addGroup('drawing/crop', new Controls.CheckBox(docView, { value: 'crop', label: /*#. Label for cropping selected image. */ gt('Crop'), tooltip: gt('Crop image'), boxed: true }));
        this.addSeparator();
        this.addGroup('drawing/cropposition/fill', new Button(docView, { value: 'fill', label: /*#. Label for fill selected image completely over frame. */ gt('Fill'), tooltip: gt('Scale the image to fill the frame. The image aspect ratio is maintained. The image might be cropped') }));
        this.addGroup('drawing/cropposition/fit', new Button(docView, { value: 'fit', label: /*#. Label for fit selected image completely into frame. */ gt('Fit'), tooltip: gt('Scale the image to fit the frame. The image aspect ratio is maintained. There might be empty space inside the frame.') }));
        this.addSeparator();
        this.addGroup('drawing/cropposition/dialog', new Button(docView, { value: 'cropposition', label: /*#. Label for openinig crop position dialog. */ gt('Position ...'), tooltip: gt('Open the crop position dialog') }));
    } }); // class ImageCropPosition

    // exports ================================================================

    return DrawingControls;

});
