/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/drawinglayer/view/dialog/insertimagefiledialog', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/io',
    'io.ox/office/tk/utils/driveutils',
    'io.ox/office/tk/dialog/basedialog',
    'io.ox/office/drawinglayer/utils/imageutils',
    'gettext!io.ox/office/drawinglayer/main'
], function (Utils, IO, DriveUtils, BaseDialog, ImageUtils, gt) {

    'use strict';

    // private global functions ===============================================

    // create a file picker widget node
    function createFilePickerWidget() {

        var guid = _.uniqueId('form-control-');

        var rootNode = $('<div class="file-picker form-group" style="margin-bottom:0;">');

        var buttonFile = $('<span class="btn btn-default btn-file">').text(gt('Select file'));

        var labelFile = $('<label class="sr-only" for="' + guid + '">').text(gt('Select file'));

        var inputFile = $('<input id="' + guid + '" name="file" type="file" style="margin-top:-20px;margin-bottom:-20px;" accept="image/*">');

        var buttonGroupFile = $('<span class="input-group-btn btn-file" role="button">').append(buttonFile, labelFile, inputFile);

        var inputText = $('<input type="text" class="form-control" readonly="" aria-readonly="true" style="pointer-events: none;">')
                .attr({ placeholder: gt('No file selected'), 'aria-label': gt('file name') });

        // set tabindex depending on browser
        (_.browser.IE ? buttonFile : buttonGroupFile).attr('tabindex', '1');

        rootNode.append($('<div class="input-group">').append(buttonGroupFile, inputText));

        rootNode.on('keypress', function (e) {
            // press SPACE (ENTER isn't active, cause of the special 'send'-occupancy)
            if (e.which === 32) {
                Utils.setFocus(inputFile); // BUG #34034: FF needs to focus the input-element first
                inputFile.trigger('click');
            }
        });

        inputFile.on('change', function (event) {
            var fileName = inputFile[0].value;
            var name = (event.target && event.target.files && event.target.files[0] && event.target.files[0].name) || (fileName && fileName.substring(fileName.lastIndexOf('\\') + 1)) || '';
            inputText.val(name);

            // bug 53214: Safari is not able to remove the placeholder by itselfs, so we have to do it.
            // Because it's not critical, we remove this attribute for all (not only for safari)
            if (name !== '') {
                inputText.attr('placeholder', '');
            }
        });

        return rootNode;
    }

    // class InsertImageFileDialog ============================================

    /**
     * A dialog that provides a local file picker for a new image to be
     * inserted into the document.
     *
     * @constructor
     *
     * @extends BaseDialog
     *
     * @param {EditView} docView
     *  The document view that is creating the dialog.
     */
    var InsertImageFileDialog = BaseDialog.extend(function (docView) {

        // base constructor
        BaseDialog.call(this, docView, {
            title: gt('Insert Image'),
            width: 800
        });

        // the application instance
        var app = docView.getApp();
        // the file picker widget
        var filePicker = createFilePickerWidget();
        // the <input> field to pick a file
        var fileInput = filePicker.find('input[type="file"]');
        // the file descriptor of the file currently selected
        var uploadFile = null;

        // prepare the 'Upload File' tab panel
        var uploadForm = $('<form>').append(filePicker);
        // IE has a special handling for single inputs inside a form.
        // Pressing enter directly submits the form. Adding a hidden input solves this.
        if (_.browser.IE) {
            uploadForm.append('<input type="text" value="" style="display:none;">');
        }
        this.append(uploadForm);

        this.setFocusNode(fileInput);

        // close dialog automatically after losing edit rights
        docView.closeDialogOnReadOnlyMode(this);

        // register a change handler at the file input that extracts the file descriptor
        fileInput.on('change', function (event) {
            uploadFile = (event.target && event.target.files && event.target.files[0]) || null;  // requires IE 10+
            if (uploadFile && !ImageUtils.hasFileImageExtension(uploadFile.name)) {
                app.rejectEditAttempt('image');
                uploadFile = null;
            }
            this.validate();
        }.bind(this));

        // add validator that checks existence of a file descriptor
        this.addOkValidator(function () { return uploadFile; });

        // handler for the OK button of the dialog
        this.setOkHandler(function () {
            var hasFile = _.isObject(uploadFile) || _.isString(uploadFile);
            if (!hasFile) { throw new Error('missing file'); }
            return ImageUtils.insertFile(app, uploadFile);
        });

    }); // class InsertImageFileDialog

    // exports ================================================================

    return InsertImageFileDialog;

});
