/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/drawinglayer/view/chartstyleutil', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/color',
    'gettext!io.ox/office/drawinglayer/main'
], function (Utils, Color, gt) {

    'use strict';

    var BACKGROUNDTRANSFORMATION = [{ type: 'satMod', value: 25000 }, { type: 'lumOff', value: 35000 }];

    var STYLESET = [
        { bg: { type: 'scheme', value: 'light1' }, bevelEnabled: false },
        { bg: { type: 'scheme', value: 'light1' }, bevelEnabled: false },
        { bg: { type: 'scheme', value: 'light1' }, bevelEnabled: false },
        { bg: { type: 'scheme', value: 'light1' }, bevelEnabled: true },
        { bg: null, bevelEnabled: false },
        { bg: { type: 'scheme', value: 'text1', transformations: [{ type: 'lumOff', value: 25000 }] }, bevelEnabled: true }
    ];

    var COLORSET = [];

    (function () {
        var allAccents = [];
        for (var i = 1; i <= 6; i++) {
            allAccents.push({ type: 'scheme', value: 'accent' + i });
        }

        function getAccentColorSetName(accent) {
            //#. predefined formatting style for chart objects: single accent color from the current color scheme
            //#. Example result: "Accent 1", "Accent 2"
            //#. %1$d is the index of the accent color
            //#, c-format
            return gt.pgettext('chart-format', 'Accent %1$d', _.noI18n(accent));
        }

        COLORSET.push({
            //#. predefined formatting style for chart objects: black/gray/white only
            name: gt.pgettext('chart-format', 'Grayscale'),
            colors: [{ type: 'rgb', value: '555555' }, { type: 'rgb', value: '9E9E9E' }, { type: 'rgb', value: '727272' }, { type: 'rgb', value: '464646' }, { type: 'rgb', value: '838383' }, { type: 'rgb', value: 'C1C1C1' }],
            type: 'group'
        });

        COLORSET.push({
             //#. predefined formatting style for chart objects: all accent colors from the current color scheme
            name: gt.pgettext('chart-format', 'Accents'),
            colors: allAccents,
            type: 'group'
        });

        _.each(allAccents, function (color, index) {
            COLORSET.push({
                name: getAccentColorSetName(index + 1),
                colors: [color],
                type: 'single'
            });
        });
    }());

    // static class ChartStyleUtil ============================================

    /**
     * an infoholder for the 48 chartstyles, a combination of colorpatterns, backgroundcolors and highlighting
     */
    var ChartStyleUtil = {};

    // public methods ---------------------------------------------------------

    /**
     * return a list of different combinations of backgrounds and highlights
     */
    ChartStyleUtil.getStyleSet = function () {
        return STYLESET;
    };

    /**
     * return a list of different combinations of grayscales and accents
     */
    ChartStyleUtil.getColorSet = function () {
        return COLORSET;
    };

    /**
     * returns the color-transformations for desaturate the first color in the used pattern
     */
    ChartStyleUtil.getBackgroundTransformation = function () {
        return BACKGROUNDTRANSFORMATION;
    };

    ChartStyleUtil.isAutoShape = function (shape) {
        if (!shape) { return true; }
        if (shape.gradient && shape.gradient.colorStops) {
            return ChartStyleUtil.isAutoColor(shape.gradient.colorStops[0].color);
        } else {
            return ChartStyleUtil.isAutoColor(shape.color);
        }
    };

    ChartStyleUtil.isAutoColor = function (color) {
        return !color || color.type === 'auto' || color.value === 'phClr';
    };

    function makeShadeTintTrans(value) {
        if (value > 0) {
            return [{
                type: 'tint',
                value: Math.round((1 - value) * 100000)
            }];
        } else if (value < 0) {
            return [{
                type: 'shade',
                value: Math.round((1 + value) * 100000)
            }];
        }
    }

    ChartStyleUtil.getColorOfPattern = function (meth, type, index, schemeColors, count, docModel) {

        var colorIndex = null;
        var relVariation = null;

        if (type === 'single') {
            relVariation = (index / count) - 0.5;
        } else if (type === 'group') {
            var group = Math.floor(index / schemeColors.length);
            var all = Math.floor(count / schemeColors.length);
            relVariation = group / all;

            if (count <= schemeColors.length * 2) {
                //TODO: workaround for small charts, should be calculated correctly
                relVariation *= 0.5;
            } else {
                relVariation -= 0.5;
            }
        } else {
            Utils.warn('cant handle colorize type:' + type);
            return null;
        }

        if (meth === 'cycle') {
            colorIndex = index % schemeColors.length;
        } else {
            Utils.warn('cant handle color meth:' + meth);
            return null;
        }

        var color = schemeColors[colorIndex];

        var tmpColor = {};
        tmpColor.type = color.type;
        tmpColor.value = color.value;

        if (!relVariation) {
            relVariation = 0;
        }
        tmpColor.transformations = makeShadeTintTrans(relVariation * 1.4);

        if (docModel) {
            tmpColor.fallbackValue = docModel.parseAndResolveColor(tmpColor, 'fill').hex;
        }
        return tmpColor;
    };

    function getLineColorForAuto(docModel, chartModel) {
        var colorDesc = docModel.parseAndResolveColor(chartModel.getBackgroundColor(), 'fill');
        return colorDesc.dark ? 'white' : 'black';
    }

    /**
     * sets all character-model-properties to the target,
     * target is a CanvasJS-data-model
     * @param {Object} character character-model comes from operations
     * @param {Object} target target-object is a inner object of CanvasJS
     * @param {Object} prefix necessary if there are more than one character-holder on the target object
     */
    ChartStyleUtil.handleCharacterProps = function (chartModel, character, target, prefix) {
        var app = chartModel.getApp();
        var docModel = app.getModel();

        var usePrefix;
        if (prefix && prefix.length) {
            usePrefix = prefix + 'Font';
        } else {
            usePrefix = 'font';
        }

        var cssColor = null;
        var color = character.color;
        if (ChartStyleUtil.isAutoColor(color)) {
            cssColor = getLineColorForAuto(docModel, chartModel);
        } else {
            cssColor = docModel.getCssColor(color);
        }

        var targetJSON = target[usePrefix] || {};
        target[usePrefix] = targetJSON;

        targetJSON.Color  = cssColor;
        targetJSON.Family = docModel.getCssFontFamily(character.fontName);
        targetJSON.Size   = ChartStyleUtil.getHeightForCharacterProps(chartModel, character);
        targetJSON.Weight = character.bold ? 'bold' : 'normal';
        targetJSON.Style  = character.italic ? 'italic' : 'normal';

        _.each(targetJSON, function (entry, key) {
            target[usePrefix + key] = entry;
        });
    };

    /**
     * returns calculates pixel height from character's fontSize and current zoom factor from docView
     *
     * @param {Object} character character-model comes from operations
     */
    ChartStyleUtil.getHeightForCharacterProps = function (chartModel, character) {
        var app = chartModel.getApp();
        var docView = app.getView();
        return docView.getZoomFactor() * Utils.convertLength(character.fontSize, 'pt', 'px');
    };

    /**
     * sets all lineshape-model-properties to the target,
     * target is a CanvasJS-data-model
     * @app {BaseApplication}
     * @character {object} lineshape-model comes from operations
     * @target {object} target-object is a inner object of CanvasJS
     * @prefix {object} necessary for more than one character-holder on the target object
     */
    ChartStyleUtil.handleLineProps = function (chartModel, line, target, prefix) {

        var app = chartModel.getApp();
        var docModel = app.getModel();
        var docView = app.getView();
        var w = line.width ? line.width * (72 / 2540) : 1;
        var usePrefix = prefix;

        var targetJSON = target[usePrefix] || {};
        target[usePrefix] = targetJSON;

        if (line.type === 'none') {
            targetJSON.Color = 'transparent';
        } else {
            var cssColor = null;
            var color = line.color;
            if (ChartStyleUtil.isAutoColor(color)) {
                cssColor = getLineColorForAuto(docModel, chartModel);
            } else {
                cssColor = docModel.getCssColor(color);
            }

            targetJSON.Color     = cssColor;
            targetJSON.Thickness = Math.max(1, Math.round(docView.getZoomFactor() * w));
        }

        _.each(targetJSON, function (entry, key) {
            target[usePrefix + key] = entry;
        });
    };

    ChartStyleUtil.toStyleSetIndex = function (chartStyleId) {
        var styleId = chartStyleId - 1;
        var styleSet = (styleId / 8) | 0;

        if (styleSet < 4) {
            return 0;
        } else {
            return styleSet;
        }
    };

    ChartStyleUtil.toColorSetIndex = function (chartStyleId) {
        if (!chartStyleId) {
            return 0;
        }
        var styleId = chartStyleId - 1;
        return styleId % 8;
    };

    ChartStyleUtil.toChartStyleId = function (colorSetIndex, styleSetIndex) {
        return styleSetIndex * 8 + colorSetIndex + 1;
    };

    // exports ================================================================

    return ChartStyleUtil;

});
