/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/drawinglayer/model/drawingcollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/iterator',
    'io.ox/office/tk/container/valueset',
    'io.ox/office/baseframework/model/modelobject'
], function (Utils, Iterator, ValueSet, ModelObject) {

    'use strict';

    // convenience shortcuts
    var SingleIterator = Iterator.SingleIterator;
    var FilterIterator = Iterator.FilterIterator;
    var NestedIterator = Iterator.NestedIterator;
    var SerialIterator = Iterator.SerialIterator;

    // class DrawingCollection ================================================

    /**
     * A collection containing models of drawing objects.
     *
     * Triggers the following events:
     * - 'insert:drawing'
     *      After a new drawing model has been inserted into this collection.
     *      Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {DrawingModel} drawingModel
     *          The new drawing model inserted into this collection.
     * - 'delete:drawing'
     *      Before a drawing model will be removed from this collection. Event
     *      handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {DrawingModel} drawingModel
     *          The drawing model to be deleted from this collection.
     * - 'change:drawing'
     *      After some contents of a drawing model in this collection have been
     *      changed (dependent on the type of the drawing object), including
     *      the formatting attributes of the drawing object. Event handlers
     *      receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {DrawingModel} drawingModel
     *          The drawing model that caused the change event.
     *      (3) {String} type
     *          The exact type of the change event. The possible values are
     *          dependent on the type of the drawing object. Will be set to
     *          'attributes', if the formatting attributes of the drawing
     *          objects have been changed.
     *      (4) {Any} [...]
     *          Additional data dependent on the type of the change event. if
     *          the change event type is 'attributes', the fifth parameter
     *          contains the new merged attribute set, and the sixth parameter
     *          contains the old merged attribute set.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {EditModel} docModel
     *  The document model containing this instance.
     *
     * @param {Function} modelFactory
     *  A factory function that constructs drawing model instances for the
     *  specified drawing type. Receives the following parameters:
     *  (1) {DrawingCollection} parentCollection
     *      The parent drawing collection that will contain the new drawing
     *      object (may be this instance, or an embedded drawing collection of
     *      a group object).
     *  (2) {String} drawingType
     *      The type of the drawing model to be created.
     *  (3) {Object|Null} attributeSet
     *      The initial formatting attributes for the new drawing model.
     *  (4) {Object} [options]
     *      Additional model settings to be passed to the model constructor.
     *  Must return a new instance of a drawing model according to the passed
     *  type. May return null for unsupported drawing types.
     *
     * @param {Object} initOptions
     *  Optional or required parameters:
     *  - {DrawingModel} [initOptions.parentModel]
     *      The parent drawing model owning this instance as embedded drawing
     *      collection. If omitted, this instance will be a root collection
     *      without parent drawing model.
     *  - {Function} initOptions.clientResolveModelHandler
     *      A function that receives an arbitrary document position, and must
     *      return the model of a drawing object at that position, if
     *      available; otherwise null.
     *  - {Function} initOptions.clientResolvePositionHandler
     *      A function that receives an arbitrary drawing model, and must
     *      return its document position, if available; otherwise null.
     *  - {Function} initOptions.clientResolveIndexHandler
     *      A function that receives an arbitrary drawing model, and must
     *      return its own array index in the list of all its siblings, which
     *      is used to define the rendering order ("Z order").
     *  its document position, if available; otherwise null.
     *  - {Function} initOptions.clientRegisterModelHandler
     *      A function that receives a new drawing model that has just been
     *      inserted into this collection, and the document position of the new
     *      drawing model. The function has to perform client-side registration
     *      of the drawing model according to the document position, and
     *      returns whether registration of the drawing model at the passed
     *      position was successful.
     *  - {Function} initOptions.clientUnregisterModelHandler
     *      A function that receives a drawing model that has just been removed
     *      from this collection, and the old document position of the drawing
     *      model. The function has to perform client-side deregistration of
     *      the drawing model according to the document position, and returns
     *      whether deregistration of the drawing model at the passed position
     *      was successful.
     */
    var DrawingCollection = ModelObject.extend({ constructor: function (docModel, modelFactory, initOptions) {

        // self reference
        var self = this;

        // the parent drawing model (if this instance is an embedded collection of a group object)
        var parentModel = Utils.getOption(initOptions, 'parentModel', null);

        // the root drawing collection
        var rootCollection = parentModel ? parentModel.getRootCollection() : this;

        // the client resolves document positions to models
        var clientResolveModelHandler = Utils.getFunctionOption(initOptions, 'clientResolveModelHandler', _.constant(null));

        // the client resolves models to document positions
        var clientResolvePositionHandler = Utils.getFunctionOption(initOptions, 'clientResolvePositionHandler', _.constant(null));

        // the client resolves models to Z order indexes
        var clientResolveIndexHandler = Utils.getFunctionOption(initOptions, 'clientResolveIndexHandler', _.constant(null));

        // the client has to register new drawing models
        var clientRegisterModelHandler = Utils.getFunctionOption(initOptions, 'clientRegisterModelHandler', _.constant(false));

        // the client has to register new drawing models
        var clientUnregisterModelHandler = Utils.getFunctionOption(initOptions, 'clientUnregisterModelHandler', _.constant(false));

        // all drawing models, mapped by internal unique identifier
        var drawingModels = new ValueSet('getUid()');

        // cached number of drawing models in this collection (faster than counting map properties)
        var modelCount = 0;

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel, { trigger: 'always' });

        // private methods ----------------------------------------------------

        /**
         * Tries to find the root drawing object inside this drawing collection
         * specified by some leading elements of the passed position array.
         *
         * @param {Array<Number>} position
         *  The document position of a drawing model, with arbitrary length.
         *
         * @returns {Object|Null}
         *  The value null, if no root drawing object has been found; otherwise
         *  a result object with the following properties:
         *  - {DrawingModel} model
         *      The model of a drawing object inside this collection.
         *  - {Array<Number>} position
         *      The document position of the root drawing object (some leading
         *      elements of the passed document position). The remaining array
         *      elements may address an embedded drawing object (e.g. in group
         *      objects), or other embedded content inside the drawing object
         *      (e.g. text contents), or both.
         *  - {Array<Number>} remaining
         *      The remaining part of the passed document position (the
         *      trailing array elements).
         */
        function getRootModel(position) {

            for (var length = 1; length <= position.length; length += 1) {

                // the leading part of the passed drawing position
                var rootPos = position.slice(0, length);

                // let the client resolve the drawing model by its position
                var model = clientResolveModelHandler.call(self, rootPos);
                if (!model) { continue; }

                // ensure that the model is part of this collection
                if (!drawingModels.has(model)) {
                    Utils.warn('DrawingCollection.getRootModel() - unregistered drawing model');
                    return null;
                }

                // return the drawing model and its exact position
                return { model: model, position: rootPos, remaining: position.slice(length) };
            }

            return null;
        }

        // protected methods --------------------------------------------------

        /**
         * Creates the specified drawing model by calling the callback handler
         * function passed to the constructor of this instance.
         *
         * @param {DrawingCollection} parentCollection
         *  The parent drawing collection that will contain the new drawing
         *  object (may be this instance, or an embedded drawing collection of
         *  a group object).
         *
         * @param {String} drawingType
         *  The type of the drawing model to be created.
         *
         * @param {Object|Null} [attributeSet]
         *  The initial formatting attributes for the drawing model.
         *
         * @param {Object} [options]
         *  Additional model settings to be passed to the model factory.
         *
         * @returns {DrawingModel|Null}
         *  The new instance of a drawing model according to the passed type;
         *  or null for unsupported drawing types.
         */
        this.implCreateModel = function (parentCollection, drawingType, attributeSet, options) {
            return modelFactory(parentCollection, drawingType, attributeSet || null, options);
        };

        /**
         * Stores the passed model of a new drawing object into this drawing
         * collection.
         *
         * @param {Array<Number>} position
         *  The document position of the new drawing model.
         *
         * @param {DrawingModel} model
         *  The model instance of the new drawing object. This collection takes
         *  ownership of this drawing model!
         *
         * @returns {Boolean}
         *  Whether the new drawing model has been inserted successfully.
         */
        this.implInsertModel = function (position, model) {

            // register the new drawing model at the client application
            if (!clientRegisterModelHandler.call(this, model, position)) { return false; }

            // store the new drawing model
            drawingModels.insert(model);
            modelCount += 1;

            // notify all listeners
            this.trigger('insert:drawing', model);

            // forward attribute change events of the drawing model to own listeners
            model.on('change:attributes', function () {
                self.trigger.apply(self, ['change:drawing', model, 'attributes'].concat(_.toArray(arguments).slice(1)));
            });

            // forward custom change events of the drawing model to own listeners
            model.on('change:drawing', function () {
                self.trigger.apply(self, ['change:drawing', model].concat(_.toArray(arguments).slice(1)));
            });

            // forward events of the embedded drawing collection to own listeners
            model.withChildCollection(this.forwardEvents, this);

            return true;
        };

        /**
         * Returns the document position of the specified drawing model in this
         * drawing collection.
         *
         * @param {DrawingModel} model
         *  The drawing model whose position will be calculated.
         *
         * @returns {Array<Number>|Null}
         *  The position of the drawing object, if it is contained in this
         *  collection; otherwise null.
         */
        this.implResolvePosition = function (model) {
            return clientResolvePositionHandler.call(this, model);
        };

        /**
         * Returns the own array index of the specified drawing model in the
         * list of all its siblings in this drawing collection. This index is
         * used to define the rendering order ("Z order").
         *
         * @param {DrawingModel} model
         *  The drawing model whose Z order index will be calculated.
         *
         * @returns {Number|Null}
         *  The own index in the list of the siblings of this drawing model; or
         *  null on any error.
         */
        this.implResolveIndex = function (drawingModel) {
            return clientResolveIndexHandler.call(this, drawingModel);
        };

        /**
         * Implementation helper for subclasses that creates the effective
         * model iterator filtering and extending the passed raw child model
         * iterator according to the passed options.
         *
         * @param {Any} iterable
         *  The source sequence to be iterated over. Will be passed to the
         *  static method Iterator.from() to create an iterator object.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.deep=false]
         *      If set to true, all drawing objects embedded in other drawing
         *      objects will be visited too.
         *  - {Boolean} [options.visible=false]
         *      If set to true, only visible drawing objects will be visited.
         *
         * @returns {Iterator}
         *  The new iterator that filters and extends the passed iterator if
         *  necessary.
         */
        this.implCreateModelIterator = function (iterable, options) {

            // create an iterator
            var iterator = Iterator.from(iterable);

            // filter for visible drawing models
            if (Utils.getBooleanOption(options, 'visible', false)) {
                iterator = new FilterIterator(iterator, function (drawingModel) {
                    return drawingModel.isVisible();
                });
            }

            // extend iterator to visit the embedded models of each direct child model
            if (Utils.getBooleanOption(options, 'deep', false)) {
                iterator = new NestedIterator(iterator, function (drawingModel, iterResult) {
                    // an iterator that provides the drawing model
                    var iterator1 = new SingleIterator(iterResult);
                    // an iterator that visits all children of the drawing model (recursively)
                    var iterator2 = drawingModel.createChildModelIterator(options);
                    // create a serialized iterator that visits both iterators
                    return new SerialIterator(iterator1, iterator2);
                });
            }

            return iterator;
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the parent drawing model owning this instance as embedded
         * drawing collection.
         *
         * @returns {DrawingModel|Null}
         *  The parent drawing model owning this instance as embedded drawing
         *  collection; or null, if this instance is a root collection without
         *  parent drawing model.
         */
        this.getParentModel = function () {
            return parentModel;
        };

        /**
         * Returns the direct parent drawing collection that contains the
         * drawing model owning this embedded drawing collection.
         *
         * @returns {DrawingCollection|Null}
         *  The parent drawing collection, if this instance is an embedded
         *  drawing collection; or null, if this instance is a root collection.
         */
        this.getParentCollection = function () {
            return parentModel ? parentModel.getParentCollection() : null;
        };

        /**
         * Returns the ancestor top-level drawing collection that contains all
         * top-level drawing models.
         *
         * @returns {DrawingCollection}
         *  The ancestor top-level drawing collection that contains all
         *  top-level drawing models; or a reference to this instance, if it is
         *  a top-level drawing collection by itself.
         */
        this.getRootCollection = function () {
            return rootCollection;
        };

        /**
         * Returns the number of drawings contained in this drawing collection.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.deep=false]
         *      If set to true, all drawing objects embedded deeply in other
         *      drawing objects will be counted too.
         *
         * @returns {Number}
         *  The number of drawing objects contained in this collection.
         */
        this.getModelCount = function (options) {

            // the resulting model count (start with the cached number of own drawing models)
            var totalCount = modelCount;

            // count all deeply embedded models if specified
            if (Utils.getBooleanOption(options, 'deep', false)) {
                drawingModels.forEach(function (model) {
                    model.withChildCollection(function (childCollection) {
                        totalCount += childCollection.getModelCount({ deep: true });
                    });
                });
            }

            return totalCount;
        };

        /**
         * Returns detailed information for the drawing model at the passed
         * document position.
         *
         * @param {Array<Number>} position
         *  The document position of the drawing model. May specify a position
         *  inside another drawing object (group objects), and may specify a
         *  child component inside a drawing object (e.g. text contents). This
         *  method traverses as deep as possible into the document position
         *  array, as long as the current drawing object contains child drawing
         *  objects.
         *
         * @returns {Object|Null}
         *  The value null, if the passed document position is invalid;
         *  otherwise a descriptor object with the following properties:
         *  - {DrawingModel} model
         *      The model of a drawing object at the specified document
         *      position, either directly from this collection, or from an
         *      embedded drawing object.
         *  - {Array<Number>} position
         *      The exact document position of the drawing object (some leading
         *      elements of the passed document position) in this collection.
         *  - {Array<Number>} remaining
         *      The remaining part of the passed document position (the
         *      trailing array elements) addressing embedded content inside the
         *      drawing object (e.g. text contents).
         */
        this.getModelDescriptor = function (position) {

            // find the root drawing object in this collection
            var rootDesc = getRootModel(position);
            if (!rootDesc) { return null; }

            // no remaining elements in the position array: resulting drawing model found
            if (rootDesc.remaining.length === 0) { return rootDesc; }

            // resolve embedded drawing models, if there are remaining elements in the position array
            var childCollection = rootDesc.model.getChildCollection();
            if (!childCollection) { return rootDesc; }

            // resolve an embedded drawing model (return null, if there is no such embedded drawing model)
            var embedDesc = childCollection.getModelDescriptor(rootDesc.remaining);
            if (!embedDesc) { return null; }

            // restore the complete position of the drawing model
            embedDesc.position = rootDesc.position.concat(embedDesc.position);

            return embedDesc;
        };

        /**
         * Returns the drawing model at the passed document position.
         *
         * @param {Array<Number>} position
         *  The exact document position of the drawing model. May specify a
         *  position inside another drawing object (group objects).
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {String} [options.type]
         *      If specified, the drawing model must be of the specified type.
         *      If a drawing model with another type exists at the specified
         *      document position, this method will return null.
         *
         * @returns {DrawingModel|Null}
         *  The drawing model at the specified document position; or null, if
         *  no drawing model has been found.
         */
        this.getModel = function (position, options) {

            // find the drawing model (do not accept document position arrays with trailing elements)
            var modelDesc = this.getModelDescriptor(position);
            if (!modelDesc || (modelDesc.remaining.length > 0)) { return null; }

            // check the type of the drawing object if specified
            var type = Utils.getStringOption(options, 'type', null);
            if (type && (type !== modelDesc.model.getType())) { return null; }

            // return the model of the drawing object directly
            return modelDesc.model;
        };

        /**
         * Creates an iterator that visits all drawing models in this drawing
         * collection. The drawing models will be visited in no specific order.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.deep=false]
         *      If set to true, all drawing objects embedded in other drawing
         *      objects will be visited too.
         *  - {Boolean} [options.visible=false]
         *      If set to true, only visible drawing objects will be visited.
         *
         * @returns {Iterator}
         *  The new iterator. The result objects will contain the drawing model
         *  (class DrawingModel) currently visited as value.
         */
        this.createModelIterator = function (options) {
            return this.implCreateModelIterator(drawingModels, options);
        };

        /**
         * Creates and stores the model of a new drawing object based on the
         * passed settings.
         *
         * @param {Array<Number>} position
         *  The document position of the new drawing model. May specify a
         *  position inside another drawing object (if the array without the
         *  last element points to an existing drawing object). If that drawing
         *  model supports embedding drawing objects, the new drawing model
         *  will be inserted into the drawing collection of that drawing object
         *  instead into this collection.
         *
         * @param {String} drawingType
         *  The type of the drawing object to be created.
         *
         * @param {Object|Null} [attributeSet]
         *  An attribute set with initial formatting attributes for the drawing
         *  object.
         *
         * @param {Object} [options]
         *  Additional model settings to be passed to the model factory.
         *
         * @returns {Boolean}
         *  Whether the new drawing model has been created successfully.
         */
        this.insertModel = function (position, drawingType, attributeSet, options) {

            // try to find a root drawing object in this collection to insert the new drawing into
            var modelDesc = getRootModel(position);
            if (modelDesc && (modelDesc.remaining.length > 0)) {
                var childCollection = modelDesc.model.getChildCollection();
                return !!childCollection && childCollection.insertModel(modelDesc.remaining, drawingType, attributeSet);
            }

            // create and store the new drawing model
            var model = this.implCreateModel(this, drawingType, attributeSet, options);
            return (!!model && this.implInsertModel(position, model)) ? model : null;
        };

        /**
         * Removes the model of a drawing object located at the passed document
         * position.
         *
         * @param {Array<Number>} position
         *  The document position of the drawing model. May specify a position
         *  inside another drawing object (if the array without the last
         *  element points to an existing drawing object).
         *
         * @returns {Boolean}
         *  Whether the drawing model has been removed successfully.
         */
        this.deleteModel = function (position) {

            // find the root drawing object in this collection
            var modelDesc = getRootModel(position);
            if (!modelDesc) { return false; }

            // remaining elements: delete drawing model in an embedded collection
            if (modelDesc.remaining.length > 0) {
                var childCollection = modelDesc.model.getChildCollection();
                return !!childCollection && childCollection.deleteModel(modelDesc.remaining);
            }

            // notify all listeners
            this.trigger('delete:drawing', modelDesc.model);

            // let the client application unregister the deleted drawing model
            if (!clientUnregisterModelHandler.call(this, modelDesc.model, modelDesc.position)) { return false; }

            // remove and destroy the drawing model
            drawingModels.remove(modelDesc.model);
            modelDesc.model.destroy();
            modelCount -= 1;

            return true;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            drawingModels.destroyElements();
            self = docModel = parentModel = rootCollection = drawingModels = null;
            modelFactory = clientResolveModelHandler = clientResolvePositionHandler = null;
            clientRegisterModelHandler = clientUnregisterModelHandler = null;
        });

    } }); // class DrawingCollection

    // exports ================================================================

    return DrawingCollection;

});
