/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/baseframework/view/control/dynamiclabel', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/control/label'
], function (Utils, Label) {

    'use strict';

    // class DynamicLabel =====================================================

    /**
     * A label control that updates its text dynamically via its current
     * internal value.
     *
     * @constructor
     *
     * @extends Label
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {Function} [initOptions.captionResolver]
     *      A callback function that receives the control value, and returns a
     *      string to be shown as label, or an options object containing label
     *      and/or icon options. Will be called in the context of this label
     *      instance. If omitted, the current value will be used. If it is an
     *      object, it will be used as caption options object, if it is null,
     *      the label text will be cleared, otherwise it will be converted to a
     *      string.
     *  @param {Boolean} [initOptions.setToolTip=false]
     *      If set to true, the caption text of this label control will be set
     *      as tool-tip too.
     */
    function DynamicLabel(docView, initOptions) {

        // self reference
        var self = this;

        // resolves the value to a display string, or a caption options object
        var captionResolver = Utils.getFunctionOption(initOptions, 'captionResolver');

        // whether to set the caption as tool-tip
        var setToolTip = Utils.getOption(initOptions, 'setToolTip', false);

        // base constructor ---------------------------------------------------

        Label.call(this, docView, initOptions);

        // private methods ----------------------------------------------------

        /**
         * Updates the label text.
         */
        function updateHandler(value) {
            var label = captionResolver ? captionResolver.call(self, value) : value;
            if (_.isObject(label)) {
                self.setCaption(label);
            } else {
                self.setLabel(label || '');
                if (setToolTip) {
                    self.setToolTip(label || '');
                }
            }
        }

        // initialization -----------------------------------------------------

        // additional options for an ARIA live region for dynamic labels
        this.getLabelNode().attr({
            role: 'log',
            'aria-live': 'assertive',
            'aria-relevant': 'additions',
            'aria-atomic': true,
            'aria-readonly': true
        });

        // update label text according to current control value
        this.registerUpdateHandler(updateHandler);

        // destroy all class members
        this.registerDestructor(function () {
            initOptions = self = captionResolver = null;
        });

    } // class DynamicLabel

    // exports ================================================================

    // derive this class from class Label
    return Label.extend({ constructor: DynamicLabel });

});
