/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define([
    'globals/apphelper',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/model/tableoperationmixin'
], function (AppHelper, AttributeUtils, Position, TableOperationMixin) {

    'use strict';

    // class TableOperationMixin =============================================

    describe('TextFramework class TableOperationMixin', function () {

        // private helpers ----------------------------------------------------

        // the operations to be applied by the document model
        var OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { defaultTabStop: 1270, zoom: { value: 100 } }, page: { width: 21590, height: 27940, marginLeft: 2540, marginTop: 2540, marginRight: 2540, marginBottom: 2540, marginHeader: 1248, marginFooter: 1248 } } },
            { name: 'insertParagraph', start: [0] },
            { name: 'insertText', start: [0, 0], text: 'Hello World!' },
            { name: 'insertTable', start: [1], attrs: { table: { tableGrid: [1000, 1000], width: 'auto', exclude: ['lastRow', 'lastCol', 'bandsVert'] }, styleId: 'TableGrid' } },
            { name: 'insertRows', start: [1, 0], count: 1, insertDefaultCells: true },
            { name: 'insertParagraph', start: [1, 0, 0, 0] },
            { name: 'insertText', start: [1, 0, 0, 0, 0], text: 'Hello Earth!' },
            { name: 'setAttributes', start: [1, 0, 0, 0, 0], attrs: { character: { fontSize: 15, bold: true } }, end: [1, 0, 0, 0, 11] },
            { name: 'setAttributes', start: [1, 0, 1, 0], attrs: { character: { underline: true } } }
        ];

        var model;
        AppHelper.createTextApp('ooxml', OPERATIONS).done(function (app) {
            model = app.getModel();
        });

        // existence check ----------------------------------------------------

        it('should exist', function () {
            expect(TableOperationMixin).to.be.a('function');
        });

        // public methods -----------------------------------------------------

        describe('method "insertRowHandler"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('insertRowHandler');
            });
            it('should insert a row', function () {
                expect(model.insertRowHandler({ name: 'insertRow', start: [2, 0], count: 1, insertDefaultCells: false, referenceRow: 0, attrs: {} })).to.equal(false);
                expect(model.insertRowHandler({ name: 'insertRow', start: [1, 0], count: 1, insertDefaultCells: false, referenceRow: 0, attrs: {} })).to.equal(true);
            });
        });

        describe('method "insertTableHandler"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('insertTableHandler');
            });
            it('should insert a table', function () {
                expect(model.insertTableHandler({ name: 'insertTable', start: [3] })).to.equal(false);
                expect(model.insertTableHandler({ name: 'insertTable', start: [2] })).to.equal(true);
            });
        });

        // DOCS-797 Inherit text and paragraph attributes on insert row/column
        describe('method "insertRow"', function () {
            before(function (done) {
                model.getSelection().setTextSelection([1, 1, 0, 0, 0]);
                model.insertRow();
                done();
            });

            it('should exist', function () {
                expect(model).to.respondTo('insertRow');
            });

            it('should insert new row with inherited attributes', function () {
                var par1 = Position.getParagraphElement(model.getNode(), [1, 1, 0, 0]);
                var par1Child = par1 ? par1.firstChild : null;
                var attrsPara1 = AttributeUtils.getExplicitAttributes(par1Child);
                var charAttrs1 = attrsPara1 ? attrsPara1.character : {};
                var par2 = Position.getParagraphElement(model.getNode(), [1, 2, 0, 0]);
                var par2Child = par2 ? par2.firstChild : null;
                var attrsPara2 = AttributeUtils.getExplicitAttributes(par2Child);
                var charAttrs2 = attrsPara2 ? attrsPara2.character : {};

                var par3 = Position.getParagraphElement(model.getNode(), [1, 1, 1, 0]);
                var par3Child = par3 ? par3.firstChild : null;
                var attrsPara3 = AttributeUtils.getExplicitAttributes(par3Child);
                var charAttrs3 = attrsPara3 ? attrsPara3.character : {};
                var par4 = Position.getParagraphElement(model.getNode(), [1, 2, 1, 0]);
                var par4Child = par4 ? par4.firstChild : null;
                var attrsPara4 = AttributeUtils.getExplicitAttributes(par4Child);
                var charAttrs4 = attrsPara4 ? attrsPara4.character : {};

                expect(charAttrs1.bold).to.equal(charAttrs2.bold);
                expect(charAttrs1.fontSize).to.equal(charAttrs2.fontSize);
                expect(charAttrs3.underline).to.equal(charAttrs4.underline);
            });
        });

        // DOCS-797 Inherit text and paragraph attributes on insert row/column
        describe('method "insertColumn"', function () {
            before(function (done) {
                model.getSelection().setTextSelection([1, 0, 0, 0, 0]);
                model.insertColumn();
                done();
            });

            it('should exist', function () {
                expect(model).to.respondTo('insertColumn');
            });

            it('should insert new column with inherited attributes', function () {
                var par1 = Position.getParagraphElement(model.getNode(), [1, 2, 0, 0]);
                var par1Child = par1 ? par1.firstChild : null;
                var attrsPara1 = AttributeUtils.getExplicitAttributes(par1Child);
                var charAttrs1 = attrsPara1 ? attrsPara1.character : {};
                var par2 = Position.getParagraphElement(model.getNode(), [1, 2, 1, 0]);
                var par2Child = par2 ? par2.firstChild : null;
                var attrsPara2 = AttributeUtils.getExplicitAttributes(par2Child);
                var charAttrs2 = attrsPara2 ? attrsPara2.character : {};
                expect(charAttrs1.bold).to.equal(charAttrs2.bold);
                expect(charAttrs1.fontSize).to.equal(charAttrs2.fontSize);
            });
        });
    });

    // ========================================================================
});
