/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define([
    'globals/apphelper',
    'io.ox/office/text/components/drawing/drawingresize',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/tk/utils'
], function (AppHelper, DrawingResize, DOM, Utils) {

    'use strict';

    // class DrawingResize ==============================================

    describe('Text class DrawingResize', function () {

        // private helpers ----------------------------------------------------
        var model = null;
        var selection = null;

        // the operations to be applied by the document model
        var OPERATIONS = [
            { name: 'setDocumentAttributes',
                attrs: {
                    document:  { defaultTabStop: 1270, zoom: { value: 100 } },
                    page:      { width: 21590, height: 27940, marginLeft: 2540, marginTop: 2540, marginRight: 2540, marginBottom: 2540, marginHeader: 1248, marginFooter: 1248 },
                    character: { fontName: 'Arial', fontSize: 11, language: 'en-US', languageEa: 'en-US', languageBidi: 'ar-SA' },
                    paragraph: { lineHeight: { type: 'percent', value: 119 }, marginBottom: 352 }
                }
            },
            { name: 'insertStyleSheet', type: 'paragraph', styleId: 'Heading1', styleName: 'heading 1',
                attrs: {
                    character: { bold: true, fontName: 'Times New Roman', fontSize: 14, color: { transformations: [{ type: 'shade', value: 74902 }], type: 'scheme', value: 'accent1' } },
                    paragraph: { marginTop: 846, outlineLevel: 0, nextStyleId: 'Normal' }
                },
                parent: 'Normal',
                uiPriority: 9
            },
            { name: 'insertParagraph', start: [0] },
            { name: 'insertText', text: 'Hello World.', start: [0, 0] },
            { name: 'insertDrawing', start: [0, 0], type: 'shape', attrs: { drawing: { name: 'TextBox 1', width: 5000, height: 1000 }, geometry: { presetShape: 'rect', avList: {} }, fill: { type: 'none' } } },
            { name: 'setAttributes', start: [0, 0], attrs: { drawing: { inline: false, anchorHorBase: 'column', anchorVertBase: 'paragraph', anchorHorAlign: 'right', anchorHorOffset: 0, anchorVertOffset: 0 } } },
            { name: 'insertParagraph', start: [0, 0, 0], attrs: { character: { color: { type: 'scheme', value: 'light1' } }, paragraph: { alignment: 'center' } } }
        ];

        AppHelper.createTextApp('ooxml', OPERATIONS).done(function (app) {
            model = app.getModel();
            selection = model.getSelection();
        });

        function triggerTrackingEvent(node, type, x, y, options) {
            var positionNode = Utils.getObjectOption(options, 'drawingNode', node);
            var isShiftKey = Utils.getBooleanOption(options, 'shiftKey', false);
            var startX = parseFloat(positionNode.css('left'));
            var startY = parseFloat(positionNode.css('top'));
            var event = new $.Event(type, {
                which: 1,
                startX: startX,
                startY: startY,
                pageX: x + startX,
                pageY: y + startY,
                shiftKey: isShiftKey
            });
            node.trigger(event);
        }

        // existence check ----------------------------------------------------

        it('should exist', function () {
            expect(DrawingResize).to.be.a('object');
        });

        // public methods -----------------------------------------------------

        describe('method "DrawingResize.drawDrawingSelection"', function () {

            it('should exist', function () {
                expect(DrawingResize).to.respondTo('drawDrawingSelection');
            });

            it('should rotate drawing correctly', function () {
                selection.setTextSelection([0, 0], [0, 1]);

                var firstDrawing = selection.getSelectedDrawing();
                var selectionDrawing = firstDrawing.data('selection');
                var rotateHandle = selectionDrawing.find('.rotate-handle');
                var oldAttributes = firstDrawing.data('attributes');
                var rotateHandleHint = rotateHandle.text();

                firstDrawing.css({ position: 'absolute', left: 0 });

                expect(oldAttributes.drawing.anchorHorOffset).to.be.closeTo(0, 20);
                expect(oldAttributes.drawing.anchorVertOffset).to.be.closeTo(0, 20);
                expect(oldAttributes.drawing.rotation).not.exist;
                expect(rotateHandleHint).to.be.equal('');

                triggerTrackingEvent(rotateHandle, 'tracking:start', 0, 0, { drawingNode: firstDrawing });
                triggerTrackingEvent(rotateHandle, 'tracking:move', 500, 19, { drawingNode: firstDrawing }); // height 38px, so divide /2
                rotateHandleHint = rotateHandle.text();
                expect(rotateHandleHint).to.equal('90°');
                triggerTrackingEvent(rotateHandle, 'tracking:end', 500, 19, { drawingNode: firstDrawing });

                var newAttributes = firstDrawing.data('attributes');
                expect(newAttributes.drawing.rotation).not.to.be.equal(0);
                expect(newAttributes.drawing.rotation).to.be.equal(90);

                expect(newAttributes.drawing.anchorHorOffset).to.be.closeTo(0, 20);
                expect(newAttributes.drawing.anchorVertOffset).to.be.closeTo(0, 20);

                firstDrawing.css({ position: 'absolute', left: 0 });

                triggerTrackingEvent(rotateHandle, 'tracking:start', 79.5, 0, { drawingNode: firstDrawing }); // 159px is drawing width, /2 because rotate handle is in middle
                triggerTrackingEvent(rotateHandle, 'tracking:move', 79.5, 1000, { drawingNode: firstDrawing });
                rotateHandleHint = rotateHandle.text();
                expect(rotateHandleHint).to.equal('180°');
                triggerTrackingEvent(rotateHandle, 'tracking:end', 79.5, 1000, { drawingNode: firstDrawing });

                newAttributes = firstDrawing.data('attributes');

                expect(newAttributes.drawing.rotation).not.to.be.equal(90);
                expect(newAttributes.drawing.rotation).to.be.equal(180);

                firstDrawing.css({ position: 'absolute', left: 0 });

                triggerTrackingEvent(rotateHandle, 'tracking:start', 0, 0, { drawingNode: firstDrawing });
                triggerTrackingEvent(rotateHandle, 'tracking:move', -100, 19, { drawingNode: firstDrawing });
                rotateHandleHint = rotateHandle.text();
                expect(rotateHandleHint).to.equal('270°');
                triggerTrackingEvent(rotateHandle, 'tracking:end', -100, 19, { drawingNode: firstDrawing });

                newAttributes = firstDrawing.data('attributes');
                rotateHandleHint = rotateHandle.text();
                expect(rotateHandleHint).to.equal('');

                expect(newAttributes.drawing.rotation).not.to.be.equal(180);
                expect(newAttributes.drawing.rotation).to.be.equal(270);
            });

            it('should rotate drawing correctly with step 15 degree (shift key)', function () {
                selection.setTextSelection([0, 0], [0, 1]);

                var firstDrawing = selection.getSelectedDrawing();
                var selectionDrawing = firstDrawing.data('selection');
                var rotateHandle = selectionDrawing.find('.rotate-handle');

                firstDrawing.css({ position: 'absolute', left: 0 });

                // shift key works as rounding, as it will round to closest step 15 angle
                triggerTrackingEvent(rotateHandle, 'tracking:start', 0, 0, { drawingNode: firstDrawing, shiftKey: true });
                triggerTrackingEvent(rotateHandle, 'tracking:move', 500, 15, { drawingNode: firstDrawing, shiftKey: true }); // height 38px, so divide /2
                triggerTrackingEvent(rotateHandle, 'tracking:end', 500, 15, { drawingNode: firstDrawing, shiftKey: true });

                var newAttributes = firstDrawing.data('attributes');
                expect(newAttributes.drawing.rotation).not.to.be.equal(0);
                expect(newAttributes.drawing.rotation).to.be.equal(90);

                expect(newAttributes.drawing.anchorHorOffset).to.be.closeTo(0, 20);
                expect(newAttributes.drawing.anchorVertOffset).to.be.closeTo(0, 20);

                firstDrawing.css({ position: 'absolute', left: 0 });

                triggerTrackingEvent(rotateHandle, 'tracking:start', 75.5, 0, { drawingNode: firstDrawing, shiftKey: true }); // 159px is drawing width, /2 because rotate handle is in middle
                triggerTrackingEvent(rotateHandle, 'tracking:move', 75.5, 1000, { drawingNode: firstDrawing, shiftKey: true });
                triggerTrackingEvent(rotateHandle, 'tracking:end', 75.5, 1000, { drawingNode: firstDrawing, shiftKey: true });

                newAttributes = firstDrawing.data('attributes');

                expect(newAttributes.drawing.rotation).not.to.be.equal(90);
                expect(newAttributes.drawing.rotation).to.be.equal(180);

                firstDrawing.css({ position: 'absolute', left: 0 });

                triggerTrackingEvent(rotateHandle, 'tracking:start', 0, 0, { drawingNode: firstDrawing, shiftKey: true });
                triggerTrackingEvent(rotateHandle, 'tracking:move', -90, 25, { drawingNode: firstDrawing, shiftKey: true });
                triggerTrackingEvent(rotateHandle, 'tracking:end', -90, 25, { drawingNode: firstDrawing, shiftKey: true });

                newAttributes = firstDrawing.data('attributes');
                expect(newAttributes.drawing.rotation).not.to.be.equal(180);
                expect(newAttributes.drawing.rotation).to.be.equal(270);
            });

            it('should generate a selection inside the selection overlay node', function () {

                selection.setTextSelection([0, 0], [0, 1]);

                var selectionOverlayNode = model.getNode().children('.drawingselection-overlay');

                // there must be one (and only one) overlay node for the selection
                expect(selectionOverlayNode.length).to.equal(1);

                // there must be one drawing selection inside this overlay node
                var drawingSelection = selectionOverlayNode.children();
                expect(drawingSelection.length).to.equal(1);

                var firstDrawing = selection.getSelectedDrawing();
                var selectionDrawing = firstDrawing.data('selection');

                // one drawing selected with one registered drawing selection
                expect(firstDrawing.length).to.equal(1);
                expect(selectionDrawing.length).to.equal(1);

                // the one existing drawing selection must be registered at the drawing in the document
                expect(selectionDrawing[0] === drawingSelection[0]).to.equal(true);
            });

            it('should remove the selection in the selection overlay node when drawing is inline', function () {

                selection.setTextSelection([0, 0], [0, 1]);

                var selectionOverlayNode = model.getNode().children('.drawingselection-overlay');

                // there must be one (and only one) overlay node for the selection
                expect(selectionOverlayNode.length).to.equal(1);

                // switching the selected drawing to be inline
                model.anchorDrawingTo('inline');

                var firstDrawing = selection.getSelectedDrawing();
                expect(firstDrawing.length).to.equal(1);

                // there must be no selection inside the overlay node for 'inline' drawings
                expect(selectionOverlayNode.children().length).to.equal(0);

            });

            it('should insert the selection into the selection overlay node when drawing is aligned with page', function () {

                selection.setTextSelection([0, 0], [0, 1]);

                var selectionOverlayNode = model.getNode().children('.drawingselection-overlay');

                // there must be one (and only one) overlay node for the selection
                expect(selectionOverlayNode.length).to.equal(1);

                // switching the selected drawing to be aligned with page
                model.anchorDrawingTo('page');

                // there must be one drawing selection inside this overlay node
                var drawingSelection = selectionOverlayNode.children();
                expect(drawingSelection.length).to.equal(1);

                var firstDrawing = selection.getSelectedDrawing();
                var selectionDrawing = firstDrawing.data('selection');

                // one drawing selected with one registered drawing selection
                expect(firstDrawing.length).to.equal(1);
                expect(selectionDrawing.length).to.equal(1);

                // the one existing drawing selection must be registered at the drawing in the document
                expect(selectionDrawing[0] === drawingSelection[0]).to.equal(true);
            });

            it('should insert the selection into the selection overlay node when drawing is aligned with paragraph', function () {

                selection.setTextSelection([0, 0], [0, 1]);

                var selectionOverlayNode = model.getNode().children('.drawingselection-overlay');

                // there must be one (and only one) overlay node for the selection
                expect(selectionOverlayNode.length).to.equal(1);

                // switching the selected drawing to be aligned with paragraph
                model.anchorDrawingTo('paragraph');

                // there must be one drawing selection inside this overlay node
                var drawingSelection = selectionOverlayNode.children();
                expect(drawingSelection.length).to.equal(1);

                var firstDrawing = selection.getSelectedDrawing();
                var selectionDrawing = firstDrawing.data('selection');

                // one drawing selected with one registered drawing selection
                expect(firstDrawing.length).to.equal(1);
                expect(selectionDrawing.length).to.equal(1);

                // the one existing drawing selection must be registered at the drawing in the document
                expect(selectionDrawing[0] === drawingSelection[0]).to.equal(true);
            });

            it('should remove the drawing selection from selection overlay if there is a cursor selection', function () {

                selection.setTextSelection([0, 0], [0, 1]); // -> text selection

                var selectionOverlayNode = model.getNode().children('.drawingselection-overlay');

                // there must be one (and only one) overlay node for the selection
                expect(selectionOverlayNode.length).to.equal(1);

                // there must be one drawing selection inside this overlay node
                expect(selectionOverlayNode.children().length).to.equal(1);

                // setting text selection
                selection.setTextSelection([0, 0], [0, 0]);

                // no more drawing selection
                expect(model.getSelection().isAnyDrawingSelection()).to.equal(false);

                // there must be no drawing selection inside this overlay node
                expect(selectionOverlayNode.children().length).to.equal(0);
            });

            it('should remove the drawing selection from selection overlay if the drawing is deleted', function (done) {

                var // the delete promise
                    delPromise = null;

                selection.setTextSelection([0, 0], [0, 1]); // -> selecting the drawing

                var selectionOverlayNode = model.getNode().children('.drawingselection-overlay');

                // there must be one (and only one) overlay node for the selection
                expect(selectionOverlayNode.length).to.equal(1);

                // switching the selected drawing to be aligned with paragraph
                model.anchorDrawingTo('paragraph');

                // there must be one drawing selection inside this overlay node
                var drawingSelection = selectionOverlayNode.children();
                expect(drawingSelection.length).to.equal(1);

                var selectedDrawing = selection.getSelectedDrawing();
                var selectionDrawing = selectedDrawing.data('selection');

                // one drawing selected with one registered drawing selection
                expect(selectedDrawing.length).to.equal(1);
                expect(selectionDrawing.length).to.equal(1);

                // the one existing drawing selection must be registered at the drawing in the document
                expect(selectionDrawing[0] === drawingSelection[0]).to.equal(true);

                // deleting the selected drawing
                delPromise = model.deleteSelected({ deleteKey: true });

                delPromise.always(function () {

                    expect(delPromise.state()).to.equal('resolved');

                    // there must be no more drawing selection inside the overlay node
                    expect(selectionOverlayNode.children().length).to.equal(0);

                    done();
                });
            });

            it('should restore the drawing selection in the selection overlay after undoing the deleting of the drawing', function (done) {

                var // the undo promise
                    undoPromise = null;

                selection.setTextSelection([0, 0], [0, 0]); // -> text selection

                var selectionOverlayNode = model.getNode().children('.drawingselection-overlay');

                // there must be one (and only one) overlay node for the selection
                expect(selectionOverlayNode.length).to.equal(1);

                // no more drawing selection
                expect(model.getSelection().isAnyDrawingSelection()).to.equal(false);

                // there must be no drawing selection inside this overlay node
                expect(selectionOverlayNode.children().length).to.equal(0);

                // restoring the drawing and the drawing selection via undo
                undoPromise = model.getUndoManager().undo();
                undoPromise.always(function () {
                    expect(undoPromise.state()).to.equal('resolved');

                    // there must be a drawing selection after undo of deleting a drawing
                    expect(model.getSelection().isAnyDrawingSelection()).to.equal(true);

                    // there must be one drawing selection inside this overlay node
                    var drawingSelection = selectionOverlayNode.children();
                    expect(drawingSelection.length).to.equal(1);

                    var selectedDrawing = selection.getSelectedDrawing();
                    var selectionDrawing = selectedDrawing.data('selection');

                    // one drawing selected with one registered drawing selection
                    expect(selectedDrawing.length).to.equal(1);
                    expect(selectionDrawing.length).to.equal(1);

                    // the one existing drawing selection must be registered at the drawing in the document
                    expect(selectionDrawing[0] === drawingSelection[0]).to.equal(true);

                    done();
                });
            });
        });
    });

    // ========================================================================
});
