/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'globals/sheethelper',
    'io.ox/office/spreadsheet/utils/intervalarray',
    'io.ox/office/spreadsheet/utils/intervalset'
], function (SheetHelper, IntervalArray, IntervalSet) {

    'use strict';

    // class IntervalSet ======================================================

    describe('Spreadsheet class IntervalSet', function () {

        it('should exist', function () {
            expect(IntervalSet).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var i = SheetHelper.i;
        var ia = SheetHelper.ia;
        var unorderedMatcher = SheetHelper.unorderedIntervalsMatcher;

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create an interval set', function () {
                var set = new IntervalSet();
                expect(set).to.be.an.instanceof(IntervalSet);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "empty"', function () {
            it('should exist', function () {
                expect(IntervalSet).to.respondTo('empty');
            });
            it('should return true for an empty set', function () {
                var set = new IntervalSet();
                expect(set.empty()).to.equal(true);
            });
        });

        describe('method "clear"', function () {
            it('should exist', function () {
                expect(IntervalSet).to.respondTo('clear');
            });
            it('should clear the set', function () {
                var set = new IntervalSet();
                set.insert(i('1:2'));
                expect(set.empty()).to.equal(false);
                expect(set.clear()).to.equal(set);
                expect(set.empty()).to.equal(true);
            });
        });

        describe('method "values"', function () {
            it('should exist', function () {
                expect(IntervalSet).to.respondTo('values');
            });
            it('should return all values', function () {
                var set = new IntervalSet();
                var i1 = i('1:2'), i2 = i('4:7'), i3 = i('9:10');
                set.insert(i1);
                set.insert(i2);
                set.insert(i3);
                var values = set.values();
                expect(values).to.be.an.instanceof(IntervalArray);
                expect(values).to.have.length(3);
                expect(values.indexOf(i1)).to.be.at.least(0);
                expect(values.indexOf(i2)).to.be.at.least(0);
                expect(values.indexOf(i3)).to.be.at.least(0);
            });
        });

        describe('method "clone"', function () {
            it('should exist', function () {
                expect(IntervalSet).to.respondTo('clone');
            });
            var set = new IntervalSet();
            var i1 = i('1:2'), i2 = i('4:7'), i3 = i('9:10');
            set.insert(i1);
            set.insert(i2);
            set.insert(i3);
            it('should create a shallow clone', function () {
                var clone = set.clone();
                expect(clone).to.be.an.instanceof(IntervalSet);
                var values1 = set.values().sort();
                var values2 = clone.values().sort();
                expect(values2).to.deep.equal(values1);
                expect(values2[0]).to.equal(values1[0]);
            });
            it('should create a deep clone', function () {
                var clone = set.clone(true);
                expect(clone).to.be.an.instanceof(IntervalSet);
                var values1 = set.values().sort();
                var values2 = clone.values().sort();
                expect(values2).to.deep.equal(values1);
                expect(values2[0]).to.not.equal(values1[0]);
            });
            it('should create a deep clone with callback', function () {
                var context = {}, spy = sinon.spy(), clone = set.clone(spy, context);
                expect(clone).to.be.an.instanceof(IntervalSet);
                var values1 = set.values().sort();
                var values2 = clone.values().sort();
                expect(values2).to.deep.equal(values1);
                expect(values2[0]).to.not.equal(values1[0]);
                sinon.assert.alwaysCalledOn(spy, context);
                sinon.assert.callCount(spy, 3);
            });
        });

        describe('method "has"', function () {
            it('should exist', function () {
                expect(IntervalSet).to.respondTo('has');
            });
            it('should return false for an empty set', function () {
                var set = new IntervalSet();
                expect(set.has(i('1:10'))).to.equal(false);
            });
        });

        describe('method "get"', function () {
            it('should exist', function () {
                expect(IntervalSet).to.respondTo('get');
            });
            it('should return null for an empty set', function () {
                var set = new IntervalSet();
                expect(set.get(i('1:10'))).to.equal(null);
            });
        });

        describe('method "containsIndex"', function () {
            it('should exist', function () {
                expect(IntervalSet).to.respondTo('containsIndex');
            });
            it('should return false for an empty set', function () {
                var set = new IntervalSet();
                expect(set.containsIndex(0)).to.equal(false);
                expect(set.containsIndex(9)).to.equal(false);
            });
        });

        describe('method "findByIndex"', function () {
            it('should exist', function () {
                expect(IntervalSet).to.respondTo('findByIndex');
            });
            it('should return an empty array for an empty set', function () {
                var set = new IntervalSet();
                expect(set.findByIndex(0)).to.deep.equal(ia());
                expect(set.findByIndex(9)).to.deep.equal(ia());
            });
        });

        describe('method "overlaps"', function () {
            it('should exist', function () {
                expect(IntervalSet).to.respondTo('overlaps');
            });
            it('should return false for an empty set', function () {
                var set = new IntervalSet();
                expect(set.overlaps(i('1:10'))).to.equal(false);
            });
        });

        describe('method "findIntervals"', function () {
            it('should exist', function () {
                expect(IntervalSet).to.respondTo('findIntervals');
            });
            it('should return an empty array for an empty set', function () {
                var set = new IntervalSet();
                expect(set.findIntervals(i('1:10'))).to.deep.equal(ia());
            });
        });

        describe('method "forEach"', function () {
            it('should exist', function () {
                expect(IntervalSet).to.respondTo('forEach');
            });
            it('should visit all values', function () {
                var set = new IntervalSet();
                set.insert(i('1:2'));
                set.insert(i('4:7'));
                set.insert(i('9:10'));
                var spy = sinon.spy(), context = {};
                set.forEach(spy, context);
                sinon.assert.alwaysCalledOn(spy, context);
                sinon.assert.callCount(spy, 3);
            });
        });

        describe('method "iterator"', function () {
            it('should exist', function () {
                expect(IntervalSet).to.respondTo('iterator');
            });
            it('should visit all intervals', function () {
                var set = new IntervalSet();
                set.insert(i('1:2'));
                set.insert(i('4:7'));
                set.insert(i('9:10'));
                var it = set.iterator();
                expect(it).to.respondTo('next');
                expect(it.next().done).to.equal(false);
                expect(it.next().done).to.equal(false);
                expect(it.next().done).to.equal(false);
                expect(it.next().done).to.equal(true);
            });
        });

        describe('method "insert"', function () {
            it('should exist', function () {
                expect(IntervalSet).to.respondTo('insert');
            });
            var set = new IntervalSet();
            it('should insert a new interval', function () {
                var i1 = i('1:3'), i2 = i('1:3');
                expect(set.insert(i1)).to.equal(i1);
                expect(set.has(i1)).to.equal(true);
                expect(set.has(i2)).to.equal(true);
                expect(set.get(i1)).to.equal(i1);
                expect(set.get(i2)).to.equal(i1);
                expect(set.values().length).to.equal(1);
                expect(set.insert(i2)).to.equal(i1);
                expect(set.has(i1)).to.equal(true);
                expect(set.has(i2)).to.equal(true);
                expect(set.get(i1)).to.equal(i1);
                expect(set.get(i2)).to.equal(i1);
                expect(set.values().length).to.equal(1);
                expect(set.insert(i2, true)).to.equal(i2);
                expect(set.has(i1)).to.equal(true);
                expect(set.has(i2)).to.equal(true);
                expect(set.get(i1)).to.equal(i2);
                expect(set.get(i2)).to.equal(i2);
                expect(set.values().length).to.equal(1);
            });
            it('should detect the intervals by index', function () {
                expect(set.containsIndex(0)).to.equal(true);
                expect(set.containsIndex(1)).to.equal(true);
                expect(set.containsIndex(2)).to.equal(true);
                expect(set.containsIndex(3)).to.equal(false);
                expect(set.containsIndex(9)).to.equal(false);
            });
            it('should detect the intervals by overlap test', function () {
                expect(set.overlaps(i('1:2'))).to.equal(true);
                expect(set.overlaps(i('2:3'))).to.equal(true);
                expect(set.overlaps(i('3:4'))).to.equal(true);
                expect(set.overlaps(i('4:5'))).to.equal(false);
            });
            it('should insert and find another interval', function () {
                set.insert(i('5:9'));
                expect(set.containsIndex(0)).to.equal(true);
                expect(set.containsIndex(2)).to.equal(true);
                expect(set.containsIndex(3)).to.equal(false);
                expect(set.containsIndex(4)).to.equal(true);
                expect(set.containsIndex(6)).to.equal(true);
                expect(set.containsIndex(8)).to.equal(true);
                expect(set.containsIndex(9)).to.equal(false);
                expect(set.overlaps(i('1:2'))).to.equal(true);
                expect(set.overlaps(i('2:3'))).to.equal(true);
                expect(set.overlaps(i('3:4'))).to.equal(true);
                expect(set.overlaps(i('4:5'))).to.equal(true);
                expect(set.overlaps(i('5:6'))).to.equal(true);
                expect(set.overlaps(i('9:10'))).to.equal(true);
                expect(set.overlaps(i('10:10'))).to.equal(false);
                expect(set.overlaps(i('1:10'))).to.equal(true);
                expect(set.values().length).to.equal(2);
            });
            it('should insert and find multiple intervals', function () {
                set.insert(i('7:8'));
                set.insert(i('7:9'));
                set.insert(i('1:1'));
                set.insert(i('6:8'));
                set.insert(i('6:10'));
                set.insert(i('2:3'));
                expect(set.findByIndex(0)).to.satisfy(unorderedMatcher('1:3 1:1'));
                expect(set.findByIndex(1)).to.satisfy(unorderedMatcher('1:3 2:3'));
                expect(set.findByIndex(2)).to.satisfy(unorderedMatcher('1:3 2:3'));
                expect(set.findByIndex(3)).to.satisfy(unorderedMatcher(''));
                expect(set.findByIndex(4)).to.satisfy(unorderedMatcher('5:9'));
                expect(set.findByIndex(5)).to.satisfy(unorderedMatcher('5:9 6:8 6:10'));
                expect(set.findByIndex(6)).to.satisfy(unorderedMatcher('5:9 7:8 7:9 6:8 6:10'));
                expect(set.findByIndex(7)).to.satisfy(unorderedMatcher('5:9 7:8 7:9 6:8 6:10'));
                expect(set.findByIndex(8)).to.satisfy(unorderedMatcher('5:9 7:9 6:10'));
                expect(set.findByIndex(9)).to.satisfy(unorderedMatcher('6:10'));
                expect(set.findIntervals(i('1:2'))).to.satisfy(unorderedMatcher('1:3 1:1 2:3'));
                expect(set.findIntervals(i('2:3'))).to.satisfy(unorderedMatcher('1:3 2:3'));
                expect(set.findIntervals(i('3:5'))).to.satisfy(unorderedMatcher('1:3 5:9 2:3'));
                expect(set.findIntervals(i('4:6'))).to.satisfy(unorderedMatcher('5:9 6:8 6:10'));
                expect(set.findIntervals(i('8:8'))).to.satisfy(unorderedMatcher('5:9 7:8 7:9 6:8 6:10'));
                expect(set.values().length).to.equal(8);
            });
        });

        describe('method "remove"', function () {
            it('should exist', function () {
                expect(IntervalSet).to.respondTo('remove');
            });
            it('should remove intervals', function () {
                var set = new IntervalSet();
                var i1 = i('1:3');
                set.insert(i1);
                set.insert(i('2:3'));
                set.insert(i('3:5'));
                expect(set.values().length).to.equal(3);
                expect(set.remove(i('1:2'))).to.equal(null);
                expect(set.containsIndex(0)).to.equal(true);
                expect(set.values().length).to.equal(3);
                expect(set.remove(i('1:3'))).to.equal(i1);
                expect(set.containsIndex(0)).to.equal(false);
                expect(set.values().length).to.equal(2);
            });
        });

        describe('method "merge"', function () {
            it('should exist', function () {
                expect(IntervalSet).to.respondTo('merge');
            });
            it('should merge with an object', function () {
                var set = new IntervalSet();
                set.insert(i('1:2'));
                set.insert(i('4:7'));
                set.insert(i('9:10'));
                var i1 = i('1:2');
                var map = { a: i1, b: i('7:9') };
                set.merge(map);
                var values = set.values();
                expect(values).to.satisfy(unorderedMatcher('1:2 4:7 9:10 7:9'));
                expect(values.indexOf(i1)).to.equal(-1);
                set.merge(map, true);
                values = set.values();
                expect(values).to.satisfy(unorderedMatcher('1:2 4:7 9:10 7:9'));
                expect(values.indexOf(i1)).to.be.at.least(0);
            });
            it('should merge two interval sets', function () {
                var set1 = new IntervalSet();
                set1.insert(i('1:2'));
                set1.insert(i('4:7'));
                set1.insert(i('9:10'));
                var set2 = new IntervalSet();
                var i1 = i('1:2');
                set2.insert(i1);
                set2.insert(i('7:9'));
                set1.merge(set2);
                var values = set1.values();
                expect(values).to.satisfy(unorderedMatcher('1:2 4:7 9:10 7:9'));
                expect(values.indexOf(i1)).to.equal(-1);
                set1.merge(set2, true);
                values = set1.values();
                expect(values).to.satisfy(unorderedMatcher('1:2 4:7 9:10 7:9'));
                expect(values.indexOf(i1)).to.be.at.least(0);
            });
        });

        // more tests covering bugs found in the implementation
        describe('real-life cases', function () {
            var set = new IntervalSet();
            set.insert(i('5:5'));
            set.insert(i('8:8'));
            set.insert(i('11:11'));
            it('should return whether intervals overlap', function () {
                expect(set.overlaps(i('11:11'))).to.equal(true);
                expect(set.overlaps(i('11:13'))).to.equal(true);
                expect(set.overlaps(i('9:11'))).to.equal(true);
                expect(set.overlaps(i('9:13'))).to.equal(true);
            });
            it('should return overlapping intervals', function () {
                expect(set.findIntervals(i('11:11'))).to.satisfy(unorderedMatcher('11:11'));
                expect(set.findIntervals(i('11:13'))).to.satisfy(unorderedMatcher('11:11'));
                expect(set.findIntervals(i('9:11'))).to.satisfy(unorderedMatcher('11:11'));
                expect(set.findIntervals(i('9:13'))).to.satisfy(unorderedMatcher('11:11'));
            });
        });
    });

    // ========================================================================
});
