/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/control/radiogroup', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/control/buttongroup'
], function (Utils, Forms, ButtonGroup) {

    'use strict';

    // class RadioGroup =======================================================

    /**
     * Creates a control that contains a set of radio buttons. At most one of
     * the buttons can be selected at a time. The value of this control group
     * is equal to the value of the selected button.
     *
     * @constructor
     *
     * @extends ButtonGroup
     *
     * @param {String|Object} windowId
     *  The identifier of the root window of the context application owning the
     *  radio group object, or an object with a method 'getWindowId' that
     *  returns such a window identifier. Used for debugging and logging of
     *  running timers in automated test environments.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the base class
     *  ButtonGroup. Additionally, the following options are supported:
     *  - {Any} [initOptions.toggleValue]
     *      If set to a value different to null or undefined, the option button
     *      that is currently selected can be clicked to be switched off. In
     *      that case, this radio group will activate the button associated to
     *      the value specified in this option (if existing), and the action
     *      handler will return this value instead of the value of the button
     *      that has been switched off.
     *  - {Boolean} [initOptions.radioIcons=false]
     *      If set to true, the option buttons will contain leading radio icons
     *      (empty circle for unselected option buttons, circle with a dot for
     *      selected option buttons). By default, the selected state will be
     *      visualized by a background fill effect.
     */
    function RadioGroup(windowId, initOptions) {

        // self reference
        var self = this;

        // comparator for list item values
        var matcher = Utils.getFunctionOption(initOptions, 'matcher');

        // fall-back value for toggle click
        var toggleValue = Utils.getOption(initOptions, 'toggleValue');

        // show radio icons instead of selection style
        var design = Utils.getBooleanOption(initOptions, 'radioIcons', false) ? 'radio' : 'select';

        // whether the radiobutton-value should change it's value or not
        var active = true;

        // base constructor ---------------------------------------------------

        ButtonGroup.call(this, windowId, Utils.extendOptions({ role: 'radiogroup' }, initOptions));

        // private methods ----------------------------------------------------

        /**
         * Activates an option button in this radio group.
         *
         * @param {Any} value
         *  The value associated to the button to be activated. If set to null,
         *  does not activate any button (ambiguous state), but shows all
         *  buttons in the ambiguous state.
         */
        function updateHandler(value) {

            // all existing option buttons
            var optionButtons = self.getOptionButtons();

            // set all buttons to ambiguous state for null value
            if (value === null) {
                Forms.checkButtonNodes(optionButtons, null, { design: design, ambiguous: true });
            } else {
                Forms.checkMatchingButtonNodes(optionButtons, value, { matcher: matcher, design: design });
            }
        }

        // initialization -----------------------------------------------------

        // update handler for selected state
        this.registerUpdateHandler(updateHandler);

        // add ARIA role to buttons
        this.on('create:button', function (event, buttonNode) {
            buttonNode.attr('role', 'radio');
        });

        // prevent changing the radiobutton-value on "taphold" to stay open the contextmenu
        this.getNode().on('taphold', function () {
            active = false;
        });

        // option button click handler (convert ENTER and SPACE keys to click events)
        Forms.setButtonKeyHandler(this.getNode());
        Forms.touchAwareListener({ node: this.getNode(), delegate: Forms.OPTION_BUTTON_SELECTOR }, function (event) {

            // get out, if "taphold" was triggered before. In this case we want to (stay) open the contextmenu and not
            // change the value of the radiobutton
            if (!active) { active = true; return; }

            // check whether a click on a selected button toggles state back
            var toggleClick = Forms.isCheckedButtonNode(this) && (toggleValue !== null) && !_.isUndefined(toggleValue);
            // the new control value
            var value = toggleClick ? toggleValue : Forms.getButtonValue(this);

            self.triggerChange(value, { sourceEvent: event });
        });

        // handler for 'remote' events at the group root node to activate a specific option button
        this.getNode().on('remote', function (event, command, value) {
            if (command === 'click') {
                self.findOptionButtons(value).first().trigger(Forms.DEFAULT_CLICK_TYPE);
            }
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            initOptions = self = toggleValue = matcher = null;
        });

    } // class RadioGroup

    // exports ================================================================

    // derive this class from class ButtonGroup
    return ButtonGroup.extend({ constructor: RadioGroup });

});
