/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/textframework/view/control/paragraphstylepicker', [
    'io.ox/office/tk/forms',
    'io.ox/office/tk/control/button',
    'io.ox/office/editframework/view/control/stylesheetpicker',
    'io.ox/office/baseframework/view/popup/contextmenu',
    'io.ox/office/textframework/view/labels'
], function (Forms, Button, StyleSheetPicker, ContextMenu, Labels) {

    'use strict';

    // class ParagraphStylePicker =============================================

    var SECTION_FUNCTIONS = 'functions';

    /**
     * A drop-down menu control for paragraph style sheets.
     *
     * @constructor
     *
     * @extends StyleSheetPicker
     */
    function ParagraphStylePicker(docView) {

        // self reference
        var self = this;
        // paragraph style sheets
        var paragraphStyles = docView.getDocModel().getStyleCollection('paragraph');
        // menu for changing styles
        var contextMenu = null;
        // whether the context menu is active
        var isVisibleContextMenu = false;
        // the button node belonging to the context menu
        var buttonNode = null;

        // base constructor ---------------------------------------------------

        StyleSheetPicker.call(this, docView, 'paragraph', {
            width: 175,
            icon: 'docs-pilcrow',
            tooltip: /*#. tool tip: predefined styles for paragraphs */ Labels.PARAGRAPH_STYLE_LABEL,
            gridColumns: 4,
            i18nModulePath: 'io.ox/office/textframework/resource/paragraphstylenames',
            smallerVersion: { css: { width: 50 }, hideLabel: true }
        });

        // private methods ----------------------------------------------------

        /**
         * Formats the passed DOM element according to the specified paragraph
         * style sheet.
         */
        function setParagraphFormatting(buttonNode, styleId) {
            var captionNode = Forms.getCaptionNode(buttonNode);
            paragraphStyles.setElementAttributes(captionNode, { styleId: styleId }, { preview: true });
        }

        /**
         * Initializes a new list item according to the paragraph style sheet.
         */
        function createItemHandler(event, buttonNode, styleId, options) {
            //TODO: hacked code for functions like "create new paragraph stylesheet"
            if (options && options.section === SECTION_FUNCTIONS) {
                //no paragraph styling here
                return;
            }
            setParagraphFormatting(buttonNode, styleId);
        }

        /**
         * Updates the drop-down menu button.
         */
        function updateHandler(styleId) {
            var menuButton = self.getMenuButton();

            setParagraphFormatting(menuButton, styleId);
            Forms.getCaptionNode(menuButton).find('>span').css({
                color: Forms.LessVar.TEXT_COLOR
            });

            // repairing the selection that could be accidentally modified by controller update, while context menu was open (58002)
            if (isVisibleContextMenu) {
                clearSelected();
                buttonNode.addClass('selected');
            }
        }

        //TODO: hacked code for functions like "create new paragraph stylesheet"
        function initSections() {
            self.createMenuSection('', { gridColumns: 4 });
            self.createMenuSection(SECTION_FUNCTIONS, { gridColumns: 1 });
        }

        function clearSelected() {
            self.getMenu().getNode().find('.selected').removeClass('selected');
        }

        function initContextMenu() {

            contextMenu = new ContextMenu(docView, self.getMenu().getNode(), {
                selector: '.section-container:not([data-section="' + SECTION_FUNCTIONS + '"]) a'
            });

            self.getMenu().registerFocusableNodes(contextMenu.getNode());

            contextMenu.on('contextmenu:prepare', function (evt) {

                buttonNode = $(evt.sourceEvent.target).closest('a');
                var styleId = Forms.getButtonValue(buttonNode);
                var custom = paragraphStyles.isCustom(styleId);

                // adding the groups to the context menu dynamically, because 'display: none' is not always reliable (58002)
                contextMenu.addGroup('paragraph/stylesheet',       new Button(docView, { label: Labels.APPLY_STYLE_LABEL }));
                contextMenu.addGroup('paragraph/changestylesheet', new Button(docView, { label: Labels.CHANGE_STYLE_LABEL }));

                if (custom) {
                    contextMenu.addGroup('paragraph/renamestylesheet', new Button(docView, { label: Labels.RENAME_STYLE_LABEL }));
                    contextMenu.addGroup('paragraph/deletestylesheet', new Button(docView, { label: Labels.DELETE_STYLE_LABEL }));
                }

                contextMenu.iterateGroups(function (group) {
                    var link = group.getNode().find('a');
                    group.show();
                    group.setValue(styleId);
                    link.data('value', styleId);
                });

                clearSelected();
                buttonNode.addClass('selected');

                isVisibleContextMenu = true;
            });

            contextMenu.on('popup:hide', function () {
                contextMenu.iterateGroups(function (group) {
                    contextMenu.removeGroup(group);
                });

                clearSelected();
                self.getMenu().getNode().find('a[aria-selected="true"]').parent().addClass('selected');
                isVisibleContextMenu = false;
                buttonNode = null;
            });

        }

        // initialization -----------------------------------------------------

        initContextMenu();
        initSections();
        this.getMenu().on('popup:afterclear', initSections);

        // TODO: hacked code for functions like "create new paragraph stylesheet"
        paragraphStyles.createDirtyStyleSheet('\x00paragraph/createstylesheet', Labels.CREATE_STYLE_LABEL, {}, { category: SECTION_FUNCTIONS, priority: 1 });

        this.getMenu().on('create:item', createItemHandler);
        this.registerUpdateHandler(updateHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            contextMenu.destroy();
            self = docView = paragraphStyles = contextMenu = null;
        });

    } // class ParagraphStylePicker

    // exports ================================================================

    // derive this class from class StyleSheetPicker
    return StyleSheetPicker.extend({ constructor: ParagraphStylePicker });

});
