/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/textframework/utils/config', [
    'io.ox/office/editframework/utils/editconfig'
], function (EditConfig) {

    'use strict';

    // static class TextConfig ================================================

    var TextConfig = _.clone(EditConfig);

    /**
     * A Boolean flag specifying whether online spelling is enabled.
     *
     * @constant
     * @type Boolean
     */
    TextConfig.SPELLING_ENABLED = EditConfig.getFlag('module/spellingEnabled', true);

    /**
     * The maximum page number that is allowed to be pasted as inline-html content
     * into a newly created e-mail.
     * - If the server configuration item is missing, this fallback defaults to 10.
     * - A server configuration value beneath 0 will be ignored, it defaults to 0.
     * - A server configuration or default value of 0 disables the send-as-html feature.
     *
     * @constant
     * @type Number
     */
    TextConfig.SEND_AS_HTML_MAX_PAGE_COUNT = EditConfig.getInt('module/sendAsHtmlMaxPageCount', 10);

    /**
     * The maximum number of cells supported in a text table. If the server
     * configuration item is missing, the default of 1500 used.
     * ATTENTION: If you change this value, please make sure that the server
     * side is also changed.
     *
     * @constant
     * @type Number
     */
    TextConfig.MAX_TABLE_CELLS = EditConfig.getInt('module/maxTableCells', 1500);

    /**
     * The maximum number of rows supported in a text table. If the server
     * configuration item is missing, the default of 1500 is used.
     * ATTENTION: If you change this value, please make sure that the server
     * side is also changed.
     *
     * @constant
     * @type Number
     */
    TextConfig.MAX_TABLE_ROWS = EditConfig.getInt('module/maxTableRows', 1500);

    /**
     * The maximum number of columns supported in a text table. If the server
     * configuration item is missing, the default of 15 is used.
     * ATTENTION: If you change this value, please make sure that the server
     * side is also changed.
     *
     * @constant
     * @type Number
     */
    TextConfig.MAX_TABLE_COLUMNS = EditConfig.getInt('module/maxTableColumns', 15);

    // global initialization --------------------------------------------------

    // restrict MAX_TABLE_COLUMNS and MAX_TABLE_ROWS according to MAX_TABLE_CELLS
    (function () {

        var maxCells = TextConfig.MAX_TABLE_CELLS;
        var maxCols = TextConfig.MAX_TABLE_COLUMNS;
        var maxRows = TextConfig.MAX_TABLE_ROWS;

        if (maxCells < (maxCols * maxRows)) {

            // start with the square root of the cell count for both limits (columns and rows)
            var maxCount = Math.floor(Math.sqrt(maxCells));

            // respect the original limits for columns and rows
            if (maxCount > maxRows) {
                TextConfig.MAX_TABLE_COLUMNS = Math.floor(maxCells / maxRows);
            } else if (maxCount > maxCols) {
                TextConfig.MAX_TABLE_ROWS = Math.floor(maxCells / maxCols);
            } else {
                TextConfig.MAX_TABLE_COLUMNS = TextConfig.MAX_TABLE_ROWS = maxCount;
            }
        }
    }());

    // exports ================================================================

    return TextConfig;

});
