/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Malte Timmermann <malte.timmermann@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Carsten Driesner <carsten.driesner@open-xchange.com>
 * @author Oliver Specht <oliver.specht@open-xchange.com>
 */

define('io.ox/office/textframework/model/editor', [
    'io.ox/office/tk/keycodes',
    'io.ox/office/textframework/utils/textutils',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/model/editmodel',
    'io.ox/office/editframework/utils/operationutils',
    'io.ox/office/drawinglayer/utils/drawingutils',
    'io.ox/office/drawinglayer/utils/imageutils',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/textframework/components/drawing/imagecropframe',
    'io.ox/office/tk/render/rectangle',
    'io.ox/office/textframework/utils/config',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/selection/selection',
    'io.ox/office/textframework/selection/remoteselection',
    'io.ox/office/textframework/components/table/table',
    'io.ox/office/textframework/components/hyperlink/hyperlink',
    'io.ox/office/textframework/model/operationgenerator',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/utils/textutils',
    'io.ox/office/textframework/components/rangemarker/rangemarker',
    'io.ox/office/textframework/components/changetrack/changetrack',
    'io.ox/office/textframework/components/comment/commentlayer',
    'io.ox/office/textframework/components/drawing/drawinglayer',
    'io.ox/office/textframework/format/tablestyles',
    'io.ox/office/textframework/model/clipboardmixin',
    'io.ox/office/textframework/model/clipboardhandlermixin',
    'io.ox/office/textframework/model/stringconvertermixin',
    'io.ox/office/textframework/model/storageoperationmixin',
    'io.ox/office/textframework/model/tableoperationmixin',
    'io.ox/office/textframework/model/groupoperationmixin',
    'io.ox/office/textframework/model/attributeoperationmixin',
    'io.ox/office/textframework/model/deleteoperationmixin',
    'io.ox/office/textframework/model/paragraphmixin',
    'io.ox/office/textframework/model/hardbreakmixin',
    'io.ox/office/textframework/model/hyperlinkmixin',
    'io.ox/office/textframework/model/keyhandlingmixin',
    'io.ox/office/textframework/components/pagelayout/pagelayout',
    'io.ox/office/textframework/components/searchhandler/searchhandler',
    'io.ox/office/textframework/components/spellcheck/spellchecker',
    'gettext!io.ox/office/textframework/main' // OperationUtils
], function (KeyCodes, Utils, Color, Border, AttributeUtils, EditModel, OperationUtils, DrawingUtils, ImageUtils, DrawingFrame, ImageCropFrame, Rectangle, Config, Operations, DOM, Selection, RemoteSelection, Table, Hyperlink, TextOperationGenerator, Position, TextUtils, RangeMarker, ChangeTrack, CommentLayer, DrawingLayer, TableStyles, ClipboardMixin, ClipboardHandlerMixin, StorageOperationMixin, StringConverterMixin, TableOperationMixin, GroupOperationMixin, AttributeOperationMixin, DeleteOperationMixin, ParagraphMixin, HardBreakMixin, HyperlinkMixin, KeyHandlingMixin, PageLayout, SearchHandler, SpellChecker, gt) {

    'use strict';

    var // style attributes for heading 1 -6 based on latent styles
        HEADINGS_CHARATTRIBUTES = [
                { color: { type: 'scheme', value: 'accent1', transformations: [{ type: 'shade', value: 74902 }], fallbackValue: '376092' }, bold: true, fontSize: 14 },
                { color: { type: 'scheme', value: 'accent1', fallbackValue: '4F81BD' }, bold: true, fontSize: 13 },
                { color: { type: 'scheme', value: 'accent1', fallbackValue: '4F81BD' }, bold: true },
                { color: { type: 'scheme', value: 'accent1', fallbackValue: '4F81BD' }, bold: true, italic: true },
                { color: { type: 'scheme', value: 'accent1', transformations: [{ type: 'shade', value: 49804 }], fallbackValue: '244061' } },
                { color: { type: 'scheme', value: 'accent1', transformations: [{ type: 'shade', value: 49804 }], fallbackValue: '244061' }, italic: true }
        ],

        DEFAULT_PARAGRAPH_DEFINTIONS = { default: true, styleId: 'Standard', styleName: 'Normal' },

        // style attributes for lateral table style
        DEFAULT_LATERAL_TABLE_DEFINITIONS = { default: true, styleId: 'TableGrid', styleName: 'Table Grid', uiPriority: 59 },
        DEFAULT_LATERAL_TABLE_ATTRIBUTES = {
            wholeTable: {
                paragraph: { lineHeight: { type: 'percent', value: 100 }, marginBottom: 0 },
                table: {
                    borderTop:        { color: Color.AUTO, width: 17, style: 'single' },
                    borderBottom:     { color: Color.AUTO, width: 17, style: 'single' },
                    borderInsideHor:  { color: Color.AUTO, width: 17, style: 'single' },
                    borderInsideVert: { color: Color.AUTO, width: 17, style: 'single' },
                    borderLeft:       { color: Color.AUTO, width: 17, style: 'single' },
                    borderRight:      { color: Color.AUTO, width: 17, style: 'single' },
                    paddingBottom: 0,
                    paddingTop: 0,
                    paddingLeft: 190,
                    paddingRight: 190
                }
            }
        },

        // style attributes for lateral table style, used for ODF
        // because in ODF 'TableGrid' can already exists, and it has no borders defined
        DEFAULT_LATERAL_TABLE_OX_DEFINITIONS = { default: true, styleId: 'TableGridOx', styleName: 'Table Grid Ox', uiPriority: 59 },
        DEFAULT_LATERAL_TABLE_OX_ATTRIBUTES = {
            wholeTable: {
                paragraph: { lineHeight: { type: 'percent', value: 100 }, marginBottom: 0 },
                table: {
                    borderTop:        { color: Color.AUTO, width: 17, style: 'single' },
                    borderBottom:     { color: Color.AUTO, width: 17, style: 'single' },
                    borderInsideHor:  { color: Color.AUTO, width: 17, style: 'single' },
                    borderInsideVert: { color: Color.AUTO, width: 17, style: 'single' },
                    borderLeft:       { color: Color.AUTO, width: 17, style: 'single' },
                    borderRight:      { color: Color.AUTO, width: 17, style: 'single' },
                    paddingBottom: 0,
                    paddingTop: 0,
                    paddingLeft: 190,
                    paddingRight: 190
                }
            }
        },

        // style attributes for lateral table style
        DEFAULT_LATERAL_TABLE_DEFINITIONS_PRESENTATION = { default: true, styleId: '{5C22544A-7EE6-4342-B048-85BDC9FD1C3A}', styleName: 'Medium Style 2 - Accent 1', uiPriority: 59 },

        // style attributes for lateral comment style
        DEFAULT_LATERAL_COMMENT_DEFINITIONS = { default: false, styleId: 'annotation text', styleName: 'annotation text', parent: 'Standard', uiPriority: 99 },
        DEFAULT_LATERAL_COMMENT_ATTRIBUTES = { character: { fontSize: 10 }, paragraph: { lineHeight: { type: 'percent', value: 100 }, marginBottom: 0 } },

        // style attributes for lateral header style
        DEFAULT_LATERAL_HEADER_DEFINITIONS = { default: false, styleId: 'Header', styleName: 'Header', parent: 'HeaderFooter', uiPriority: 99 },

        // style attributes for lateral footer style
        DEFAULT_LATERAL_FOOTER_DEFINITIONS = { default: false, styleId: 'Footer', styleName: 'Footer', parent: 'HeaderFooter', uiPriority: 99 },

        DEFAULT_HYPERLINK_DEFINTIONS = { default: false, styleId: 'Hyperlink', styleName: 'Hyperlink', uiPriority: 99 },
        DEFAULT_HYPERLINK_CHARATTRIBUTES = { color: _.extend({ fallbackValue: '0080C0' }, Color.HYPERLINK), underline: true },

        DEFAULT_DRAWING_MARGINS = { marginTop: 317, marginLeft: 317, marginBottom: 317, marginRight: 317 },

        DEFAULT_DRAWING_DEFINITION = { default: true, styleId: 'default_drawing_style', styleName: 'Default Graphic Style', uiPriority: 50 },
        DEFAULT_DRAWING_ATTRS = { line: { color: { type: 'rgb', value: '3465a4' } } },

        DEFAULT_DRAWING_TEXTFRAME_DEFINITION = { default: false, styleId: 'Frame', uiPriority: 60 },
        DEFAULT_DRAWING_TEXTFRAME_ATTRS = {
            line: { style: 'solid', type: 'solid', width: 2, color: { type: 'rgb', value: '000000' } },
            shape: { paddingLeft: 150, paddingTop: 150, paddingRight: 150, paddingBottom: 150 },
            drawing: { anchorVertAlign: 'top', marginLeft: 201, indentLeft: 201, marginTop: 201, marginRight: 201, indentRight: 201, marginBottom: 201, textWrapMode: 'square', textWrapSide: 'both', anchorHorAlign: 'center', anchorHorBase: 'column', anchorVertBase: 'paragraph' }
        },
        DEFAULT_DRAWING_TEXTFRAME_ATTRS_ODF_SLIDE = { // 52317
            line: { style: 'solid', type: 'solid', width: 2, color: { type: 'rgb', value: '000000' } },
            shape: { paddingLeft: 250, paddingTop: 125, paddingRight: 250, paddingBottom: 125 },
            drawing: { anchorVertAlign: 'top', marginLeft: 201, indentLeft: 201, marginTop: 201, marginRight: 201, indentRight: 201, marginBottom: 201, textWrapMode: 'square', textWrapSide: 'both', anchorHorAlign: 'center', anchorHorBase: 'column', anchorVertBase: 'paragraph' }
        },

        // internal clipboard
        clipboardOperations = [],
        clipboardId = '',

        // if set to true, events caught by the editor will be logged to console
        LOG_EVENTS = false,

        // Performance: List of operation names, that cache the paragraph in the selection
        PARAGRAPH_CACHE_OPERATIONS = {};

    // Fill performance objects with key value pairs
    (function () {
        PARAGRAPH_CACHE_OPERATIONS[Operations.TEXT_INSERT] = true;
        PARAGRAPH_CACHE_OPERATIONS[Operations.PARA_INSERT] = true;
        PARAGRAPH_CACHE_OPERATIONS[Operations.PARA_SPLIT] = true;
        PARAGRAPH_CACHE_OPERATIONS[Operations.SET_ATTRIBUTES] = true;
    }());

    // private global functions ===============================================

    /**
     * Returns whether the passed document operations refer to the same parent
     * document component (i.e. their 'start' properties contain equal indexes
     * except for the last array element), and refer to the same document
     * target (i.e. their 'target' properties are equal).
     *
     * @param {Object} operation1
     *  The first document operation to be checked.
     *
     * @param {Object} operation2
     *  The second document operation to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed document operations refer to the same parent
     *  document component.
     */
    function operationsHaveSameParentComponent(operation1, operation2) {
        return _.isArray(operation1.start) && _.isArray(operation2.start) &&
            Position.hasSameParentComponent(operation1.start, operation2.start) &&
            (operation1.target === operation2.target);
    }

    /**
     * Returns whether the passed undo action is an 'insertText' action (an
     * action with a 'insertText' redo operation and a 'deleteText' undo
     * operation).
     *
     * @param {UndoAction} action
     *  The action to be tested.
     *
     * @param {Boolean} single
     *  If true, the action must contain exactly one 'deleteText' undo
     *  operation and exactly one 'insertText' redo operation. Otherwise, the
     *  last operations of each of the arrays are checked, and the arrays may
     *  contain other operations.
     */
    function isInsertTextAction(action, single) {
        return (single ? (action.undoOperations.length === 1) : (action.undoOperations.length >= 1)) && (_.first(action.undoOperations).name === Operations.DELETE) &&
            (single ? (action.redoOperations.length === 1) : (action.redoOperations.length >= 1)) && (_.last(action.redoOperations).name === Operations.TEXT_INSERT);
    }

    /**
     * Tries to merge the passed undo actions. Merging works only if both
     * actions represent a single 'insertText' operation, and the passed action
     * appends a single character directly after the text of this action.
     *
     * @param {UndoAction} lastAction
     *  The existing action that will be extended if possible.
     *
     * @param {UndoAction} nextAction
     *  The new action that will be tried to merge into the existing action.
     *
     * @returns {Boolean}
     *  Whether the passed new undo action has been merged successfully into
     *  the existing undo action.
     */
    function mergeUndoActionHandler(lastAction, nextAction) {

        // check if this and the passed action is an 'insertText' action
        var validActions = isInsertTextAction(lastAction, false) && isInsertTextAction(nextAction, true);

        // the redo operation of this action and the passed action
        var lastRedo = _.last(lastAction.redoOperations);
        var nextRedo = nextAction.redoOperations[0];

        // check that the operations are valid for merging the actions
        if (validActions && (nextRedo.text.length >= 1) && operationsHaveSameParentComponent(lastRedo, nextRedo)) {

            // check that the new action adds the character directly after the text of this action and does not change the attributes
            // check that the last character of this action is not a space character (merge actions word by word)
            // check that the operations can be merged corresponding to their attributes
            if ((_.last(lastRedo.start) + lastRedo.text.length === _.last(nextRedo.start)) && (lastRedo.text.substr(-1) !== ' ') && Utils.canMergeNeighboringOperations(lastRedo, nextRedo)) {
                // merge undo operation (delete one more character)
                lastAction.undoOperations[0].end[lastAction.undoOperations[0].end.length - 1] += nextRedo.text.length;
                // merge redo operation (add the character)
                lastRedo.text += nextRedo.text;
                return true;
            }
        }
        return false;
    }

    // class Editor ===========================================================

    /**
     * The textframework editor model. Contains and manages the entire DOM of the edited
     * text document. Implements execution of all supported operations.
     *
     * Triggers the events supported by the base class EditModel, and the
     * following additional events:
     * - 'selection': When the editor selection has been changed. Event
     *      handlers receive a reference to the selection object (instance of
     *      class Selection).
     * - 'changeTrack:stateInfo': When the change tracking state of the document
     *      is modified. This event is triggered with the option 'state: true' after
     *      loading the document and the document contains change tracked elements
     *      or after receiving an operation with change track attributes.
     *      The event is triggered with 'state: false', if no change tracked element was
     *      found inside the document.
     * - 'drawingHeight:update': When the height of a drawing is changed. This can
     *      be used to update the borders of the drawing and surrounding groups.
     * - 'update:absoluteElements': When the document is modified in that way, that
     *      it is necessary to update all absolute positioned elements.
     * - 'cacheBuffer:flush': When edit rights are transfered, and there are still
     *      not registerd operations in buffer, like for update of date fields after load.
     * - 'change:pageSettings': This event is triggered after a change of the documents
     *      page settings.
     *
     * @constructor
     *
     * @extends EditModel
     * @extends ClipboardMixin
     * @extends ClipboardHandlerMixin
     * @extends StringConverterMixin
     * @extends StorageOperationMixin
     * @extends TableOperationMixin
     * @extends GroupOperationMixin
     * @extends AttributeOperationMixin
     * @extends DeleteOperationMixin
     * @extends ParagraphMixin
     * @extends HardBreakMixin
     * @extends HyperlinkMixin
     * @extends KeyHandlingMixin
     *
     * @param {EditApplication} app
     *  The application containing this editor instance.
     *
     * @param {Object} [initOptions]
     *  Optional parameters passed to the base class constructor. Additionally,
     *  the following options are supported:
     *  @param {Boolean} [initOptions.spellingSupported=false]
     *      Whether the spell checker will be enabled by this instance.
     */
    function Editor(app, initOptions) {

        var // self reference for local functions
            self = this,

            // the root element for the document contents
            editdiv = DOM.createPageNode().attr({ contenteditable: true, tabindex: 0, 'data-focus-role': 'page', role: 'main' }).addClass('user-select-text noI18n f6-target'),

            // the undo manager of this document
            undoManager = null,

            // the logical selection, synchronizes with browser DOM selection
            selection = null,

            // the remote selection handling collaborative overlays
            remoteSelection = null,

            // the change track handler
            changeTrack = null,

            // the handler object for fields
            fieldManager = null,

            // the drawing layer for all drawings assigned to the page
            drawingLayer = null,

            // the comment layer for all comments
            commentLayer = null,

            // the handler for all range markers in the document
            rangeMarker = null,

            // instance handling visual page layout, like page breaks, headers/footers, etc.
            pageLayout = null,

            // instance handling search functions
            searchHandler = null,

            // the spell checker object
            spellChecker = null,

            // instance of text's NumberFormatter class
            numberFormatter = null,

            // shortcuts for style sheet containers
            characterStyles = null,
            paragraphStyles = null,
            slideStyles = null,
            tableStyles = null,
            tableRowStyles = null,
            tableCellStyles = null,
            drawingStyles = null,
            pageStyles = null,

            // shortcuts for other format containers
            listCollection = null,

            // values needed for pagebreaks calculus
            pageAttributes,
            //pageMaxHeight,
            pagePaddingLeft,
            //pagePaddingTop,
            //pagePaddingBottom,
            pageWidth,
            pbState = true,
            quitFromPageBreak = false,
            blockOnInsertPageBreaks = false,

            // whether the user selects 'View in draft mode'. This value is not set, when draft mode
            // is force on small devices. See TODO at function 'isDraftMode()'.
            draftModeState = false,

            // attributes that were set without a selection and are only set for a single character
            preselectedAttributes = null,

            // whether the preselected attributes must not be set to null (for example after splitParagraph)
            keepPreselectedAttributes = false,

            // whether mouse down occured in the editdiv (and no mouse up followed)
            activeMouseDownEvent = false,

            // whether a group of operations requires, that the undo stack must be deleted
            deleteUndoStack = false,

            // whether an operation is part of an undo group
            isInUndoGroup = false,

            undoRedoRunning = false,

            // the promise that controls an already active (deferred) text input
            inputTextPromise = null,

            // a paragraph node, that is implicit and located behind a table, might temporarely get an increased height
            increasedParagraphNode = null,

            // all paragraphs (and their background-color) that are artificially highlighted
            highlightedParagraphs = [],

            // whether an update of element formatting is required
            requiresElementFormattingUpdate = true,

            // whether an operation was triggered by a GUI action (for table operations)
            guiTriggeredOperation = false,

            // whether the keyboard events need to be blocked, for example during pasting
            // of internal clipboard
            blockKeyboardEvent = false,

            // whether an internal or external clipboard paste is currently running
            pasteInProgress = false,

            // the current event from the keyDown handler
            lastKeyDownEvent,

            // the last key down key code
            lastKeyDownKeyCode = null,

            // the last key down modifiers
            lastKeyDownModifiers = null,

            // whether several repeated key down events without keypress occured
            repeatedKeyDown = false,

            // the logical position where the operation ends, can be used for following cursor positioning
            lastOperationEnd,     // Need to decide: Should the last operation modify this member, or should the selection be passed up the whole call chain?!

            // whether an undo shall set the selection range after applying the operation
            useSelectionRangeInUndo = false,

            // indicates an active ime session
            imeActive = false,
            // ime objects now stored in a queue as we can have more than
            // one ime processing at a time. due to asynchronous processing
            imeStateQueue = [],
            // ime update cache
            imeUpdateText = null,
            // ime update text must be used
            imeUseUpdateText = false,

            // mutation observer
            editDivObserver = null,
            // iPad read-only paragraph
            roIOSParagraph = null,

            // operations recording
            recordOperations = true,

            // collects information for debounced page break insertion
            currentProcessingNodeCollection = [],

            // collects information for debounced automatic resizing of text frame nodes
            currentProcessingTextFrameNode = null,

            // collects information for debounced target node update
            targetNodesForUpdate = $(),

            // collects information for debounced repeated table rows update
            tableRepeatNodes = $(),

            // whether the paragraph cache can be used (only for internal operations)
            useParagraphCache = false,

            // Performance: Caching the synchronously used paragraph element
            paragraphCache = null,

            // handling double click and triple click in IE
            // whether there is a double click event waiting for a third click
            doubleClickEventWaiting = false,
            // whether the third click occured after a double click, activating the tripleclick event
            tripleClickActive = false,
            // the period of time, that a double click event 'waits' for a further click (in ms).
            doubleClickTimeOut = 60,

            //Android Chrome Code
            androidTimeouter = null,
            androidTyping = false,

            // global var holding info are header and footer in state of editing
            headerFooterEditState = false,

            // global var holding container root node for header and footer state
            $headFootRootNode = $(),

            // reference to parent node of header footer root node
            $parentOfHeadFootRootNode = $(),

            // reference to index of header footer root node, if parent is detached
            rootNodeIndex = null,

            // string holding current editable element's target id
            activeTarget = '',

            // whether the document was loaded with in 'fastLoad' process
            fastLoadImport = false,

            // whether the document was loaded with in 'localStorage' process
            localStorageImport = false,

            // whether there are additional operations loaded from the server during loading document from local storage
            externalLocalStorageOperationsExist = false,

            // whether a current change of selection is triggered by a remote client
            updatingRemoteSelection = false,

            // public listeners for IPAD workaround, all important listeners on editdiv are save, that we can reuse them
            listenerList = null,

            // handler that iterates debounced over _all_ paragraphs and updates numbering symbols and index
            updateListsDebounced = $.noop,

            // handler that iterates over _all_ paragraphs and updates numbering symbols and index
            updateLists = $.noop,

            // the maximum number of top level nodes that defines a large selection (that need to be handled asynchronously)
            MAX_TOP_LEVEL_NODES = 100,

            // whether the minimum time for a move operation shall be ignored. This is required in Unit tests.
            ignoreMinMoveTime = false,

            // the target string that can be used to avoid automatic target setting at operations. This string must not be used as target.
            operationTargetBlocker = 'TARGET_BLOCKER_VALUE',

            // debounced functions, that will be set after successful document load
            implParagraphChanged = $.noop,
            implTableChanged = $.noop,
            insertPageBreaksDebounced = $.noop,
            updateEditingHeaderFooterDebounced = $.noop,
            updateRepeatedTableRowsDebounced = $.noop,
            updateHighligtedRangesDebounced = $.noop,
            updateDrawingsDebounced = $.noop,

            // dumping the passed event object to the browser console.
            dumpEventObject = LOG_EVENTS ? function (event) {
                Utils.log('type=' + event.type + ' keyCode=' + event.keyCode + ' charCode=' + event.charCode + ' shift=' + event.shiftKey + ' ctrl=' + event.ctrlKey + ' alt=' + event.altKey + ' data=' + event.originalEvent.data);
            } : $.noop,

            // a store that contains paragraphs, which needs to be formatted with colored margins or borders later
            paragraphQueueForColoredMarginAndBorder = [];

        // base constructor ---------------------------------------------------

        EditModel.call(this, app, _.extend({
            generatorClass: TextOperationGenerator,
            mergeUndoActionHandler: mergeUndoActionHandler,
            operationsFinalizer: finalizeOperations,
            storageOperationCollector: saveStorageOperationHandler
        }, initOptions));

        // adding the mixin classes
        ClipboardMixin.call(this, app);
        ClipboardHandlerMixin.call(this, app, initOptions);
        StringConverterMixin.call(this, app);
        StorageOperationMixin.call(this, app);
        TableOperationMixin.call(this, app);
        GroupOperationMixin.call(this, app);
        AttributeOperationMixin.call(this, app);
        DeleteOperationMixin.call(this, app);
        ParagraphMixin.call(this, app);
        HardBreakMixin.call(this, app);
        HyperlinkMixin.call(this, app);
        KeyHandlingMixin.call(this, app);

        // public methods -----------------------------------------------------

        /**
         * Returns the root DOM element representing this editor.
         *
         * @returns {jQuery}
         *  The root node containing the document model and representation, as
         *  jQuery object.
         */
        this.getNode = function () {
            return editdiv;
        };

        // /**
        //  * Returns the root DOM element representing the page content.
        //  *
        //  * @returns {jQuery}
        //  *  The jQuery root node that just holds the page content part of model and representation
        //  */
        // this.getPageContent = function () {
        //     return editdiv.children(DOM.PAGECONTENT_NODE_SELECTOR).eq(0);
        //
        //   //DOM.getPageContentNode(editdiv)
        // };

        this.isEmptyPage = function () { // quite expensive, please handle with care.
            var
                $pageContent  = DOM.getPageContentNode(editdiv),
                //$pageContent  = this.getPageContent(),

                elmHeaderWrapper = editdiv.children(DOM.HEADER_WRAPPER_SELECTOR)[0],
                elmFooterWrapper = editdiv.children(DOM.FOOTER_WRAPPER_SELECTOR)[0];

            //  please have a look into
            //
            //  - http://caniuse.com/#feat=innertext
            //  - http://perfectionkills.com/the-poor-misunderstood-innerText/
            //
            return (
                (!!elmHeaderWrapper && ($.trim(elmHeaderWrapper.innerText) === '')) &&
                (!!elmFooterWrapper && ($.trim(elmFooterWrapper.innerText) === '')) &&
                ($.trim($pageContent[0].innerText) === '')
            );
        };

        /**
         * Selects the entire document.
         *
         * @returns {Editor}
         *  A reference to this instance.
         */
        this.selectAll = function () {
            selection.selectAll();
            return this;
        };

        /**
         * Returning the drawing layer, that handles all drawings
         * located inside the drawing layer node.
         *
         * @returns {Object}
         *  A reference to the drawing layer object.
         */
        this.getDrawingLayer = function () {
            return drawingLayer;
        };

        /**
         * Returning the comment layer object, that handles all comments
         * in the document.
         *
         * @returns {Object}
         *  A reference to the change track object.
         */
        this.getCommentLayer = function () {
            return commentLayer;
        };

        /**
         * Returning the range marker object, that handles all range marker
         * nodes in the document.
         *
         * @returns {Object}
         *  A reference to the range marker object.
         */
        this.getRangeMarker = function () {
            return rangeMarker;
        };

        /**
         * Returning the complex field object, that handles all complex fields
         * in the document.
         *
         * @returns {Object}
         *  A reference to the complex field object.
         */
        this.getFieldManager = function () {
            return fieldManager;
        };

        /**
         * Setting the handler for field manager. This handler is different in the
         * several applications.
         *
         * @params {BaseObject} handler
         *  The application specific field manager for updating lists.
         */
        this.setFieldManager = function (handler) {
            fieldManager = handler;
        };

        /**
         * Returning the change track object, that handles all change
         * tracks in the document.
         *
         * @returns {Object}
         *  A reference to the change track object.
         */
        this.getChangeTrack = function () {
            return changeTrack;
        };

        /**
         * Getter for pageLayout object instance.
         *
         * @returns {Object}
         *  A reference to the page layout object.
         */
        this.getPageLayout = function () {
            return pageLayout;
        };

        /**
         * Returns instance of NumberFormatter class.
         *
         * @returns {Object}
         */
        this.getNumberFormatter = function () {
            return numberFormatter;
        };

        /**
         * Getting the blocker target, that avoids, that the active target is automatically
         * added to the operations in the 'finalizeOperations' handler. This blocker is
         * removed in the 'finalizeOperations' handler, so that the operation will not
         * have a target at all. If another target than the active target shall be used
         * in an operation, it must be set during the creation of the operation.
         *
         * @returns {String}
         *  A blocker target string that cannot be used as a valid target. It is automatically
         *  removed in the 'finalizeOperations' handler.
         */
        this.getOperationTargetBlocker = function () {
            return operationTargetBlocker;
        };

        /**
         * Setting the handler for number formatter. This handler is different in the
         * several applications.
         *
         * @params {Function} handler
         *  The application specific handler function for updating lists.
         */
        this.setNumberFormatter = function (handler) {
            numberFormatter = handler;
        };

        /**
         * Whether the document was loaded from local storage.
         *
         * @returns {Boolean}
         *  Returns whether the document was loaded from local storage.
         */
        this.isLocalStorageImport = function () {
            return localStorageImport;
        };

        /**
         * Whether the document was loaded with 'fast load' process.
         *
         * @returns {Boolean}
         *  Returns whether the document was loaded with 'fast load' process.
         */
        this.isFastLoadImport = function () {
            return fastLoadImport;
        };

        /**
         * Setting the global variable 'localStorageImport'.
         *
         * @param {Boolean} value
         *  The value for 'localStorageImport'.
         */
        this.setLocalStorageImport = function (value) {
            localStorageImport = value;
        };

        /**
         * Setting the global variable 'externalLocalStorageOperationsExist'.
         * This is set to true, if there are additional operations sent from
         * the server during loading document with local storage.
         * This reduces the performance, because more parts of the document
         * need to be formatted.
         *
         * @param {Boolean} value
         *  The value for 'externalLocalStorageOperationsExist'.
         */
        this.setExternalLocalStorageOperationsExist = function (value) {
            externalLocalStorageOperationsExist = value;
        };

        /**
         * Whether there were additional operations sent from the server during
         * loading the document from the local storage.
         *
         * @returns {Boolean}
         *  Returns whether there were additional operations sent from the
         *  server during loading the document from the local storage.
         */
        this.externalLocalStorageOperationsExist = function () {
            return externalLocalStorageOperationsExist;
        };

        /**
         * Setting the global variable 'fastLoadImport'.
         *
         * @param {Boolean} value
         *  The value for 'fastLoadImport'.
         */
        this.setFastLoadImport = function (value) {
            fastLoadImport = value;
        };

        /**
         * Getting the global 'clipboardOperations'.
         *
         * @returns {Object[]}
         *  The list with operations in the clip board.
         */
        this.getClipboardOperations = function () {
            return clipboardOperations;
        };

        /**
         * Setting the global value of 'clipbardOperations'.
         *
         * @param {Object[]}
         *  The list with operations in the clip board.
         */
        this.setClipboardOperations = function (ops) {
            clipboardOperations = this.cleanClipboardOperations(ops);
        };

        /**
         * Setting the value for the global clipboardId.
         *
         * @param {String}
         *  The new value for the global clipboard id.
         */
        this.setClipboardId = function (value) {
            clipboardId = value;
        };

        /**
         * Returns the value of global clipboardId.
         *
         * @returns {String}
         *  Returns the global clipboardId.
         */
        this.getClipboardId = function () {
            return clipboardId;
        };

        /**
         * Public helper function for calling the private function.
         * Clean up after the busy mode of a local client with edit prileges. After the
         * asynchronous operations were executed, the busy mode can be left.
         */
        this.leaveAsyncBusy = function () {
            return leaveAsyncBusy();
        };

        /**
         * Public helper function.
         * Calculates the size of the image defined by the given url,
         * limiting the size to the paragraph size of the current selection.
         *
         * @param {String} url
         *  The image url or base64 data url
         *
         * @returns {jQuery.Promise}
         *  A promise where the resolve result contains the size object
         *  including width and height in Hmm {width: 123, height: 456}
         */
        this.getImageSize = function (url) {
            return getImageSize(url);
        };

        /**
         * Public helper function
         * Updates all paragraphs that are part of any bullet or numbering
         * list.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.async=false]
         *      If set to true, all lists are updated asynchronously. This
         *      should happen only once when the document is loaded.
         *
         * @param {jQuery} [paragraphs]
         *  Optional set of paragraphs, that need to be updated. If not specified
         *  all paragraphs in the document need to be updated.
         *  Important: This parameter is currently only supported in Presentation app.
         *
         * @returns {jQuery.Promise}
         *  A promise that is resolved, when all paragraphs are updated.
         */
        this.updateLists = function (options, paragraphs) {
            return updateLists(options, paragraphs);
        };

        /**
         * Public helper function.
         * Check the stored table styles of a document and adds a 'missing'
         * default table style. This ensures that we can insert tables that
         * are based on a reasonable default style.
         */
        this.insertMissingTableStyles = function () {
            return tableStyles.insertMissingTableStyles();
        };

        /**
         * Public helper function.
         * Has to be called after changing the structure of a paragraph node, if
         * the update has to be done immediately (synchronously).
         *
         * @param {jQuery} paragraphs
         *  All paragraph nodes that need to be updated included into a
         *  jQuery object
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.suppressFontSize=false]
         *      If set to true, the dynamic font size calculation is NOT
         *      started during formatting of a complete slide.
         *  @param {Boolean} [options.restoreSelection=true]
         *      If set to false, the selection will not be restored via
         *      restoreBrowserSelection. This is useful, if the DOM was
         *      modfified without setting the new selection.
         */
        this.implParagraphChangedSync = function (paragraphs, options) {
            return implParagraphChangedSync(paragraphs, options);
        };

        /**
         * Getting the object with the preselected attributes. This are attributes that
         * were set without a selection and are only set for a single character.
         *
         * @returns {Object}
         *  An object, that contains the preselected attributes.
         */
        this.getPreselectedAttributes = function () {
            return preselectedAttributes;
        };

        /**
         * Whether the preselected attributes must not be set to null
         * (for example after splitParagraph).
         *
         * @returns {Boolean}
         *  Whether the preselected attributes must not be set to null.
         */
        this.keepPreselectedAttributes = function () {
            return keepPreselectedAttributes;
        };

        /**
         * Setting the global value, whether the preselected attributes
         * must not be set to null (for example after splitParagraph).
         *
         * @param {Boolean} value
         *  Whether the preselected attributes must not be set to null.
         */
        this.setKeepPreselectedAttributes = function (value) {
            keepPreselectedAttributes = value;
        };

        /**
         * Adding the explicit attributes of a specified family of a specified node
         * to the preselected attributes.
         *
         * @param {Node|jQuery|Number[]} node
         *  The node (mostly paragraph node), whose descendants are investigated.
         *
         * @param {Number|Number[]} start
         *  The number or complete logical position to find the child node relative to
         *  the specified node.
         *
         * @param {String} family
         *  The family for the explicit attributes. Only the attributes of the specified
         *  family are saved in the explicit attributes.
         */
        this.addCharacterAttributesToPreselectedAttributes = function (node, start, family) {

            var // the dom point at the specified position
                domPos = Position.getDOMPosition(node, _.isNumber(start) ? [start] : start, true),
                // the explicit attributes of the specified child node
                attrs = null,
                // the attributes of the specified family
                familyAttrs = null;

            if (domPos && domPos.node) {

                attrs = AttributeUtils.getExplicitAttributes(domPos.node, { family: family });

                if (attrs) {
                    familyAttrs = {};
                    familyAttrs[family] = attrs;
                    self.addPreselectedAttributes(familyAttrs);
                }
            }

        };

        /**
         * Adds the passed attributes to the set of preselected attributes that
         * will be applied to the next inserted text contents without moving
         * the text cursor or executing any other action.
         *
         * @param {Object} attributes
         *  A character attribute set that will be added to the set of
         *  preselected attributes.
         *
         * @returns {Editor}
         *  A reference to this instance.
         */
        this.addPreselectedAttributes = function (attributes) {
            preselectedAttributes = preselectedAttributes || {};
            this.getAttributePool().extendAttributeSet(preselectedAttributes, attributes);
            return self;
        };

        /**
         * Whether undo or redo operations are currently running.
         *
         * @returns {Boolean}
         *  Returns whether undo or redo operations are currently running.
         */
        this.isUndoRedoRunning = function () {
            return undoRedoRunning;
        };

        /**
         * Whether an operation is part of an undo group.
         *
         * @returns {Boolean}
         *  Returns whether an operation is part of an undo group.
         */
        this.isInUndoGroup = function () {
            return isInUndoGroup;
        };

        /**
         * Setting the promise that controls an already active (deferred) text input.
         *
         * @param {Object|Null} value
         *  The new value for the global inputTextPromise.
         */
        this.setInputTextPromise = function (value) {
            inputTextPromise = value;
        };

        /**
         * Getting the promise that controls an already active (deferred) text input.
         *
         * @returns {Object|Null}
         *  The global inputTextPromise.
         */
        this.getInputTextPromise = function () {
            return inputTextPromise;
        };

        /**
         * Whether the paragraph cache can be used.
         *
         * @returns {Boolean}
         *  Returns whether the paragraph cache can be used.
         */
        this.useParagraphCache = function () {
            return useParagraphCache;
        };

        /**
         * Setting, whether the paragraph cache can be used.
         *
         * @param {Boolean} value
         *  Whether the paragraph cache can be used.
         */
        this.setUseParagraphCache = function (value) {
            useParagraphCache = value;
        };

        /**
         * Returns the content of the paragraph cache. Or null, if there is no content. It
         * is important to return the node, because it might be assigned directly.
         *
         * @returns {Node|Null}
         *  Returns the content of the paragraph cache. Or null, if there is no content.
         */
        this.getParagraphCache = function () {
            return paragraphCache;
        };

        /**
         * Setting the paragraph cache node.
         *
         * @param {Node|Null}
         *  Setting the content of the paragraph cache.
         */
        this.setParagraphCache = function (node) {
            paragraphCache = node;
        };

        /**
         * Setting the global variable lastOperationEnd. This is necessary,
         * because more and more operation handlers are handled in external
         * modules and they need to set the end of their operation.
         *
         * @param {Number[]} position
         *  The logical position.
         */
        this.setLastOperationEnd = function (pos) {
            lastOperationEnd = pos;
        };

        /**
         * Getting the global variable lastOperationEnd. This is necessary,
         * because more and more operation are generated in external
         * modules and they need to set the selection after applying the
         * operations.
         *
         * @returns {Array<Number>}
         *  The logical position.
         */
        this.getLastOperationEnd = function () {
            return lastOperationEnd;
        };

        /**
         * Setting the key down event.
         *
         * @param {jQuery.Event} event
         *  The key down event.
         */
        this.setLastKeyDownEvent = function (event) {
            lastKeyDownEvent = event;
        };

        /**
         * Getting the keydown event.
         *
         * @returns {jQuery.Event}
         *  The keydown event.
         */
        this.getLastKeyDownEvent = function () {
            return lastKeyDownEvent;
        };

        /**
         * Setting the last key code from key down event.
         *
         * @param {Number} keyCode
         *  The key code number.
         */
        this.setLastKeyDownKeyCode = function (keyCode) {
            lastKeyDownKeyCode = keyCode;
        };

        /**
         * Getting the last key code from key down event.
         *
         * @returns {Number}
         *  The key code number.
         */
        this.getLastKeyDownKeyCode = function () {
            return lastKeyDownKeyCode;
        };

        /**
         * Setting the last key modifiers from the key down event
         * @param {Boolean} ctrlKey
         *  The state of the ctrl key
         *
         * @param {Boolean} shiftKey
         *  The state of the shift key
         *
         * @param {Boolean} altKey
         *  The tate of the alt key
         *
         * @param {Boolean} metaKey
         *  The state of the meta key
         */
        this.setLastKeyDownModifiers = function (ctrlKey, shiftKey, altKey, metaKey) {
            lastKeyDownModifiers = { ctrlKey: ctrlKey, shiftKey: shiftKey, altKey: altKey, metaKey: metaKey };
        };

        /*
         * Retrieves the last modifiers from the key down event
         *
         * @returns {Object}
         *  The key modifiers as stored in the last key down event
         */
        this.getLastKeyDownModifiers = function () {
            return lastKeyDownModifiers;
        };

        /**
         * Setting the repeated key down flag.
         *
         * @param {Boolean} value
         *  The flag for repeated key down.
         */
        this.setRepeatedKeyDown = function (value) {
            repeatedKeyDown = value;
        };

        /**
         * Whether the key down event was triggered several times
         * without keypress event in between (46659).
         *
         * @returns {Boolean}
         *  Whether the key down event was triggered several times
         *  without keypress event.
         */
        this.isRepeatedKeyDown = function () {
            return repeatedKeyDown;
        };

        /**
         * Returns whether the undo stack must be deleted. The value is saved
         * in the global 'deleteUndoStack'.
         *
         * @returns {Boolean}
         *  Returns whether the undo stack must be deleted.
         */
        this.deleteUndoStack = function () {
            return deleteUndoStack;
        };

        /**
         * Setting that the undo stack must be deleted. The value is saved
         * in the global 'deleteUndoStack'.
         *
         * @param {Boolean}
         *  Whether the undo stack must be deleted.
         */
        this.setDeleteUndoStack = function (value) {
            deleteUndoStack = value;
        };

        /**
         * Getting the list of artificially highlighted paragraph nodes.
         *
         * @returns {HTMLElement[]}
         *  The list of artificially highlighted paragraphs.
         */
        this.getHighlightedParagraphs = function () {
            return highlightedParagraphs;
        };

        /**
         * Getting the implicit paragraph node with increased height.
         *
         * @returns {HTMLElement}
         *  The implicit paragraph node with increase height.
         */
        this.getIncreasedParagraphNode = function () {
            return increasedParagraphNode;
        };

        /**
         * Setting the global implicit paragraph node with increase height.
         *
         * @params {HTMLElement}
         *  The implicit paragraph node with increase height.
         */
        this.setIncreasedParagraphNode = function (node) {
            increasedParagraphNode = node;
        };

        /**
         * Setting the global pageAttributes.
         *
         * @params {Object} value
         *  The page attributes object.
         */
        this.setPageAttributes = function (value) {
            pageAttributes = value;
        };

        /**
         * Getting the global pageAttributes.
         *
         * @returns {Object}
         *  The page attributes object.
         */
        this.getPageAttributes = function () {
            return pageAttributes;
        };

        /**
         * Setting the maximum page height.
         *
         * @params {Number} value
         *  The maximum page height.
         */
        this.setPageMaxHeight = function (/*value*/) {
            //pageMaxHeight = value;
        };

        /**
         * Setting the page's left padding.
         *
         * @params {Number} value
         *  The page's left padding.
         */
        this.setPagePaddingLeft = function (value) {
            pagePaddingLeft = value;
        };

        /**
         * Getting the page's left padding.
         *
         * @returns {Number}
         *  The page's left padding.
         */
        this.getPagePaddingLeft = function () {
            return pagePaddingLeft;
        };

        /**
         * Setting the page's top padding.
         *
         * @params {Number} value
         *  The page's top padding.
         */
        this.setPagePaddingTop = function (/*value*/) {
            //pagePaddingTop = value;
        };

        /**
         * Setting the page's bottom padding.
         *
         * @params {Number} value
         *  The page's bottom padding.
         */
        this.setPagePaddingBottom = function (/*value*/) {
            //pagePaddingBottom = value;
        };

        /**
         * Setting the page's width.
         *
         * @params {Number} value
         *  The page's width.
         */
        this.setPageWidth = function (value) {
            pageWidth = value;
        };

        /**
         * Getting the page's width.
         *
         * @returns {Number}
         *  The page's width.
         */
        this.getPageWidth = function () {
            return pageWidth;
        };

        /**
         * Specifying whether the minimum time for a valid move operation shall be ignored.
         * This minimum time is checked, to distinguish move operations from only selecting
         * drawing. This check must not be done in unit tests.
         *
         * @params {Boolean} value
         *  Setting, if the minimum move time shall be ignored.
         */
        this.setIgnoreMinMoveTime = function (value) {
            ignoreMinMoveTime = value;
        };

        /**
         * Returning whether the minimum move time shall be ignored. This minimum time is checked,
         * to distinguish move operations from only selecting drawing. This check must not be done
         * in unit tests.
         *
         * @returns {Boolean}
         *  Whether the minimum move time shall be ignored.
         */
        this.ignoreMinMoveTime = function () {
            return ignoreMinMoveTime;
        };

        /**
         * Getter functions for the styles objects.
         *
         * @returns {Object}
         *  The styles object.
         */
        this.getCharacterStyles = function () { return characterStyles; };
        this.getParagraphStyles = function () { return paragraphStyles; };
        this.getSlideStyles = function () { return slideStyles; };
        this.getTableStyles = function () { return tableStyles; };
        this.getTableRowStyles = function () { return tableRowStyles; };
        this.getTableCellStyles = function () { return tableCellStyles; };
        this.getDrawingStyles = function () { return drawingStyles; };
        this.getPageStyles = function () { return pageStyles; };

        /**
         * Setting the handler for debounced updating of lists. This handler
         * is different in the several applications.
         *
         * @params {Function} handler
         *  The debounced application specific handler function for updating lists.
         */
        this.setUpdateListsDebounced = function (handler) {
            updateListsDebounced = handler;
        };

        /**
         * Setting the handler for updating lists. This handler is different in the
         * several applications.
         *
         * @params {Function} handler
         *  The application specific handler function for updating lists.
         */
        this.setUpdateLists = function (handler) {
            updateLists = handler;
        };

        /**
         * Getter for the global doubleClickEventWaiting.
         *
         * @returns {Boolean}
         *  The global variable doubleClickEventWaiting.
         */
        this.getDoubleClickEventWaiting = function () {
            return doubleClickEventWaiting;
        };

        /**
         * Setter for the global tripleClickActive.
         *
         * @params {Boolean} value
         *  The new value for the global variable tripleClickActive.
         */
        this.setTripleClickActive = function (value) {
            tripleClickActive = value;
        };

        /**
         * Setter for the global activeMouseDownEvent.
         *
         * @params {Boolean} value
         *  The new value for the global variable activeMouseDownEvent.
         */
        this.setActiveMouseDownEvent = function (value) {
            activeMouseDownEvent = value;
        };

        // public helper functions that are required by the presentation application
        // TODO: This functions need to be removed, if presentation and text share common base

        this.repairEmptyTextNodes = function (element, options) {
            return repairEmptyTextNodes(element, options);
        };

        this.validateParagraphNode = function (element) {
            return validateParagraphNode(element);
        };

        this.implParagraphChanged = function (nodes) {
            return implParagraphChanged(nodes);
        };

        /**
         * Debounced function for updating registered drawings.
         *
         * @params {HTMLElement|jQuery} nodes
         *  The drawing nodes to be registered for debounced formatting.
         */
        this.updateDrawingsDebounced = function (nodes) {
            return updateDrawingsDebounced(nodes);
        };

        this.implTableChanged = function (nodes) {
            return implTableChanged(nodes);
        };

        this.checkChangesMode = function (attrs) {
            return checkChangesMode(attrs);
        };

        this.handleTriggeringListUpdate = function (node, options) {
            return handleTriggeringListUpdate(node, options);
        };

        this.insertContentNode = function (position, node, target) {
            return insertContentNode(position, node, target);
        };

        this.deletePageLayout = function () {
            pageLayout.insertPageBreaks = $.noop;
        };

        this.disableDrawingLayer = function () {
            drawingLayer.stopUpdatingAbsoluteElements();
        };

        this.getParagraphCacheOperation = function () {
            return PARAGRAPH_CACHE_OPERATIONS;
        };

        this.getMaxTopLevelNodes = function () {
            return MAX_TOP_LEVEL_NODES;
        };

        this.applyTextOperationsAsync = function (generator, label, options) {
            return applyTextOperationsAsync(generator, label, options);
        };

        this.getRoIOSParagraph = function () {
            return roIOSParagraph;
        };

        this.setRoIOSParagraph = function (value) {
            roIOSParagraph = value;
        };

        this.isImeActive = function () {
            return imeActive;
        };

        this.dumpEventObject = function (event) {
            return dumpEventObject(event);
        };

        // end of functions introduced for presentation app

        /**
         * Returns true if a clipboard paste is still in progress, otherwise false.
         * And then sets the state to true.
         * and the processing of this paste
         *
         * @returns {Boolean}
         *  Wether a clipboard paste is in progress
         */
        this.checkSetClipboardPasteInProgress = function () {
            var state = pasteInProgress;
            pasteInProgress = true;
            return state;
        };

        /**
         * Removing the artificial selection of empty paragraphs
         */
        this.removeArtificalHighlighting = function () {
            _.each(highlightedParagraphs, function (element) {
                $(element).removeClass('selectionHighlighting');
            });
            highlightedParagraphs = [];
        };

        /**
         * Creates and returns an implicit paragraph node, that can be directly
         * included into the dom (validated and updated).
         *
         * @param {Object} [paraAttrs]
         *  Optional attributes that can be assigned to the implicit paragraph.
         *
         * @returns {HTMLElement}
         *  The implicit paragraph
         */
        this.getValidImplicitParagraphNode = function (paraAttrs) {
            var paragraph = DOM.createImplicitParagraphNode();
            if (paraAttrs) { paragraphStyles.setElementAttributes(paragraph, paraAttrs); }
            validateParagraphNode(paragraph);
            paragraphStyles.updateElementFormatting(paragraph);
            return paragraph;
        };

        /**
         * Prepares a 'real' paragraph after insertion of text, tab, drawing, ...
         * by exchanging an 'implicit' paragraph (in empty documents, empty cells,
         * behind tables, ...) with the help of an operation. Therefore the server is
         * always informed about creation and removal of paragraphs and implicit
         * paragraphs are only required for user input in the browser.
         *
         * @param {Number[]} position
         *  The logical text position.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.ignoreLength=false]
         *      If set to true, no check for the paragraph length is done. Normally
         *      a implicit paragraph must be empty. In special cases this check
         *      must be omitted.
         */
        this.doCheckImplicitParagraph = function (position, options) {
            handleImplicitParagraph(position, options);
        };

        /**
         * Public API function for the private function with the same name.
         * Shows a warning dialog with Yes/No buttons before deleting document
         * contents that cannot be restored.
         *
         * @param {String} title
         *  The title of the dialog box.
         *
         * @param {String} message
         *  The message text shown in the dialog box.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the Yes button has been pressed,
         *  or rejected if the No button has been pressed.
         */
        this.showDeleteWarningDialog = function (title, message) {
            return showDeleteWarningDialog(title, message);
        };

        /**
         * Check, wheter a given paragraph style id is adding list style information
         * to a paragraph (bullet or numbering list).
         *
         * @param {String} styleId
         *  The paragraph style id, that will be checked for a list style id.
         *
         * @returns {Boolean}
         *  Whether the specified styleId contains a list style id.
         */
        this.isParagraphStyleWithListStyle = function (styleId) {

            var // the style attributes
                styleAttrs = paragraphStyles.getStyleSheetAttributeMap(styleId);

            return (styleAttrs && styleAttrs.paragraph && (styleAttrs.paragraph.listStyleId || styleAttrs.paragraph.listLevel));
        };

        /**
         * Check, wheter a given paragraph node describes a paragraph that
         * is part of a list (bullet or numbering list).
         *
         * @param {HTMLElement|jQuery} node
         *  The paragraph whose attributes will be checked. If this object is a
         *  jQuery collection, uses the first DOM node it contains.
         *
         * @param {Object} [attributes]
         *  An optional map of attribute maps (name/value pairs), keyed by attribute.
         *  It this parameter is defined, the parameter 'paragraph' can be omitted.
         *
         * @returns {Boolean}
         *  Whether the content node is a list paragraph node.
         */
        this.isListStyleParagraph = function (node, attributes) {
            return isListStyleParagraph(node, attributes);
        };

        // operations API -----------------------------------------------------

        /**
         * collects all attrs for assigned styleId out of family-stylecollection
         * and writes all data into the assigned generator
         */
        this.generateInsertStyleOp = function (generator, family, styleId, setDefault) {

            var styleCollection = this.getStyleCollection(family);
            var parentId = styleCollection.getParentId(styleId);
            var uiPriority = styleCollection.getUIPriority(styleId);
            var hidden = styleCollection.isHidden(styleId);

            var properties = {
                type: family,
                styleId: styleId,
                styleName: styleCollection.getName(styleId),
                attrs: styleCollection.getStyleSheetAttributeMap(styleId)
            };

            if (parentId) { properties.parent = parentId; }
            if (_.isNumber(uiPriority)) { properties.uiPriority = uiPriority; }
            if (hidden === true) { properties.hidden = true; }
            if (setDefault === true) { properties.default = true; }

            // adding property 'fallbackValue' to themed colors inside the specified attributes
            if (app.isODF() && properties.attrs) { self.addFallbackValueToThemeColors(properties.attrs); }

            generator.generateOperation(Operations.INSERT_STYLESHEET, properties);
            styleCollection.setDirty(styleId, false);
        };

        /**
         * Returns whether the internal clipboard contains operations.
         */
        this.hasInternalClipboard = function () {
            return clipboardOperations.length > 0;
        };

        /**
         * Sets the state whether a cipboard paste is in progress
         *
         * @param {Boolean} state
         *  The clipboard pasting state
         */
        function setClipboardPasteInProgress(state) {
            pasteInProgress = state;
        }

        /**
         * Setting the state whether the operations shall be recorded for debugging reasons.
         *
         * @param {Boolean} state
         *  The state for recording operations
         */
        this.setRecordingOperations = function (state) {
            recordOperations = state;
        };

        /**
         * Getting the state whether the operations are recorded for debugging reasons.
         *
         * @returns {Boolean}
         *  Whether the operations are recorded for debugging reasons.
         */
        this.isRecordingOperations = function () {
            return recordOperations;
        };

        /**
         * Replays the operations provided.
         *
         * @param {Array|String} operations
         *  An array of operations to be replayed (processed) or a string
         *  of operations in JSON format to be replayed.
         *
         * @param {Number} [maxOSN]
         *  An optional maximum number, until that the operations will be
         *  executed.
         *
         * @param {Boolean} external
         *  Specifies if the operations are interal or external.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if there are no more
         *  operations to replay.
         */
        this.replayOperations = function (operations, maxOSN) {

            var // operations to replay
                operationsToReplay = null,
                // delay for operations replay
                delay = 10,
                // the deferred for handling blocking of replaying operations
                replayDef = this.createDeferred('Editor.replayOperations');

            function replayOperationNext(operationArray) {

                var // the next operation if available
                    op = (operationArray.length > 0) ? operationArray.shift() : null,
                    // the result of apply operation
                    result;

                // DOCS-892: Expanding operations before applying them
                if (op) { OperationUtils.handleMinifiedObject(op, null, true); }

                if (op && (!maxOSN || !op.osn || (op.osn && op.osn <= maxOSN))) {
                    result = self.applyOperations(op);
                    if (result) {
                        self.executeDelayed(function () {
                            replayOperationNext(operationArray);
                        }, 'Editor.replayOperations', delay);
                    } else {
                        replayDef.resolve();
                    }
                } else {
                    replayDef.resolve();
                }
            }

            if (operations && operations.length > 0) {
                if (_.isString(operations)) {
                    try {
                        operationsToReplay = JSON.parse(operations);
                    } catch (e) {
                        // nothing to do
                        replayDef.resolve();
                    }
                } else if (_.isArray(operations)) {
                    operationsToReplay = operations;
                }

                if (operationsToReplay) {
                    // now try to process the operations
                    replayOperationNext(operationsToReplay);
                } else {
                    replayDef.resolve();
                }
            } else {
                // This is only a debug/developer feature therefore no need to translate this message!
                replayDef.resolve();
                app.getView().yell({ type: 'error', message: 'Recorder cannot play operations. Format unknown or not supported.' });
            }

            return replayDef.promise();
        };

        /**
         * Returning a list of objects describing the extensions that are necessary
         * to load a document from the local storage.
         * One object can contain the following properties:
         *  extension: This is the required string extension, that is used to save
         *      one layer (page content, drawing layer, ...) in the local
         *      storage.
         *  optional:  Whether this is an optional value from local storage (like
         *      drawing layer) or a required value (like page content).
         *  additionalOptions: Some additional options that can be set to influence
         *      the loading process.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.allApps=false]
         *      If set to true the local storage extensions of all apps are collected.
         *      Default is false, so that only application specific extensions are
         *      returned.
         *
         * @returns {Object[]}
         *  An array with objects describing the strings that are used in the local
         *  storage.
         */
        this.getSupportedStorageExtensions = function (options) {

            var // the list of all supported extensions for the local registry
                allExtensions = [],
                // whether the list shall contain all extensions of all apps
                allApps = Utils.getBooleanOption(options, 'allApps', false);

            // the page content layer
            allExtensions.push({ extension: '', optional: 'false' });

            if (app.isTextApp() || allApps) {
                // the drawing layer
                allExtensions.push({ extension: '_DL', optional: 'true', additionalOptions: { drawingLayer: true } });
                // header&footer layer
                allExtensions.push({ extension: '_HFP', optional: 'true', additionalOptions: { headerFooterLayer: true } });
                // the comment layer
                allExtensions.push({ extension: '_CL', optional: 'true', additionalOptions: { commentLayer: true } });
            }

            if (app.isPresentationApp() || allApps) {
                // the master slide layer
                if (!app.isODF()) { allExtensions.push({ extension: '_MSL', optional: 'false', additionalOptions: { masterSlideLayer: true } }); }
                // the layout slide layer
                allExtensions.push({ extension: '_LSL', optional: 'false', additionalOptions: { layoutSlideLayer: true } });
            }

            return allExtensions;
        };

        // ==================================================================
        // HIGH LEVEL EDITOR API which finally results in Operations
        // and creates Undo Actions.
        // Public functions, that are called from outside the editor
        // and generate operations.
        // ==================================================================

        /**
         * special flag for the very slow android chrome
         */
        this.isAndroidTyping = function () {
            return androidTyping;
        };

        /**
         *
         */
        this.handleAndroidTyping = function () {
            androidTyping = true;
            if (androidTimeouter) {
                window.clearTimeout(androidTimeouter);
            }
            androidTimeouter = self.executeDelayed(function () {
                androidTyping = false;
            }, 'Editor.handleAndroidTyping', 1000);
        };

        /**
         * Inserting a drawing group node into the document. This group can be
         * used as container for drawing frames
         * TODO: This code cannot be triggered via GUI yet. It is currently ony
         * available for testing reasons.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the dialog asking the user, if
         *  he wants to delete the selected content (if any) has been closed
         *  with the default action; or rejected, if the dialog has been
         *  canceled. If no dialog is shown, the promise is resolved
         *  immediately.
         */
        this.insertDrawingGroup = function () {

            var // whether the text frame is included into a drawing group node
                insertIntoGroup = false,
                // the position of the text frame inside a group node
                groupPosition = 0,
                // helper nodes
                startNodeInfo = null, contentNode = null;

            // the undo manager returns the return value of the callback function
            return undoManager.enterUndoGroup(function () {

                // helper function, that generates the operations
                function doInsertDrawingGroup() {

                    var // the operations generator
                        generator = self.createOperationGenerator(),
                        start = selection.getStartPosition(),
                        attrs = { drawing: { width: 6000, height: 3000 } };

                    // special code for inserting into group
                    if (insertIntoGroup) {
                        start.push(groupPosition);  // adding the position inside the group -> TODO: Remove asap
                    } else {
                        handleImplicitParagraph(start);
                    }

                    // modifying the attributes, if changeTracking is activated
                    if (changeTrack.isActiveChangeTracking()) {
                        attrs.changes = { inserted: changeTrack.getChangeTrackInfo(), removed: null };
                    }

                    // text frames are drawings of type shape
                    generator.generateOperation(Operations.INSERT_DRAWING, { attrs: attrs, start: start, type: 'group' });

                    // apply all collected operations
                    self.applyOperations(generator);
                }

                // -> integrating a text frame into a drawing frame of type 'group' by selecting the drawing group
                if (selection.hasRange() && selection.isDrawingFrameSelection()) {
                    startNodeInfo = Position.getDOMPosition(editdiv, selection.getStartPosition(), true);
                    if (startNodeInfo && startNodeInfo.node && DrawingFrame.isGroupDrawingFrame(startNodeInfo.node)) {
                        contentNode = $(startNodeInfo.node).children().first();
                        // checking the number of children inside the content node
                        groupPosition = DrawingFrame.isPlaceHolderNode(contentNode) ? 0 : contentNode.children().length;
                        insertIntoGroup = true;
                    }
                }

                if (!insertIntoGroup && selection.hasRange()) {
                    return self.deleteSelected()
                        .done(function () {
                            doInsertDrawingGroup();
                        });
                }

                doInsertDrawingGroup();
                return $.when();

            }, this);  // enterUndoGroup()
        };

        /**
         * Handling the property 'aspectLocked' of selected drawing nodes.
         * The handling from drawings in a single selection and a multiselection is different.
         *
         * @param {Boolean} state
         *  Whether the property 'aspectLocked' of selected drawings shall be
         *  enabled or disabled.
         */
        this.handleDrawingLockRatio = function (state) {

            var // the operations generator
                generator = self.createOperationGenerator(),
                // the options for the setAttributes operation
                operationOptions = {},
                // selected drawing nodes
                drawingNodes = null,
                // the selection
                selection = self.getSelection();

            // multiselection
            if (selection.isMultiSelectionSupported() && selection.isMultiSelection()) {

                _.each(selection.getMultiSelection(), function (drawingItem) {

                    // collecting the attributes for the operation
                    operationOptions.attrs =  {};
                    operationOptions.attrs.drawing = { aspectLocked: state };
                    operationOptions.start = drawingItem.startPosition;

                    // generate the 'setAttributes' operation
                    generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);
                });

                drawingNodes = selection.getAllSelectedDrawingNodes();

            // normal selection
            } else {

                drawingNodes = selection.getAnyDrawingSelection();

                // collecting the attributes for the operation
                operationOptions.attrs =  {};
                operationOptions.attrs.drawing = { aspectLocked: state };

                if (selection.isAdditionalTextframeSelection()) {
                    operationOptions.start = Position.getOxoPosition(self.getCurrentRootNode(), drawingNodes, 0);
                } else {
                    operationOptions.start = selection.getStartPosition();
                }

                // generate the 'setAttributes' operation
                generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);

                drawingNodes = [drawingNodes];

            }
            // apply all collected operations
            this.applyOperations(generator);

            // refreshing the drawing selection
            _.each(drawingNodes, function (drawing) {
                DrawingFrame.clearSelection(drawing);
                selection.drawDrawingSelection(drawing);
            });

        };

        /**
         * Checking whether currently selected drawing nodes have 'aspectLocked' enabled or not.
         * The handling from drawings in a single selection and a multiselection is different.
         *
         * @returns {Boolean}
         *  Whether a currently selected drawing node has 'aspectLocked' enabled. If selected drawings are in a
         *  a multiselection and have different states, null is returned as a flag for an ambiguous state.
         */
        this.isLockedRatioDrawing = function () {

            var // whether the selected drawing has a locked aspect ratio
                lockState = false,
                // if a drawing with an locked ratio is found in a multiselection
                foundLockedDrawings = false,
                // if a drawing with an unlocked ratio is found in a multiselection
                foundUnlockedDrawings = false,
                // attribute from the current node
                attrs;

            // multiselection: check all nodes in the selection for the 'aspectLocked' attribute to compute the 'lockState' later
            if (selection.isMultiSelectionSupported() && selection.isMultiSelection()) {
                _.each(selection.getMultiSelection(), function (drawingItem) {

                    attrs = AttributeUtils.getExplicitAttributes(drawingItem.drawing);

                    // exist check
                    if (attrs && attrs.drawing) {
                        if (attrs.drawing.aspectLocked) {
                            foundLockedDrawings = true;
                        } else {
                            foundUnlockedDrawings = true;
                        }
                    }
                });

                // compute lockState for multiselection:
                // (1) there are locked and unlocked items,
                // (2) only locked items
                // (3) only unlocked items
                if (foundLockedDrawings && foundUnlockedDrawings) {
                    // 'ambiguous' state, check the checkbox element for details
                    lockState = null;
                } else if (foundLockedDrawings) {
                    lockState = foundLockedDrawings;
                } else if (foundUnlockedDrawings) {
                    lockState = false;
                }

            // normal selection
            } else {

                attrs = AttributeUtils.getExplicitAttributes(selection.getAnyDrawingSelection());
                if (attrs && attrs.drawing && attrs.drawing.aspectLocked) { lockState = true; }

            }

            return lockState;
        };

        /**
         * Checking whether a currently selected text frame node or a specified drawing node has
         * 'autoResizeHeight' enabled or not.
         *
         * @param {Node|jQuery} [drawingNode]
         *  An optional drawing node, that shall be investigated. If not specified, a selected
         *  drawing node is used instead.
         *
         * @returns {Boolean}
         *  Whether a a currently selected text frame node has 'autoResizeHeight' enabled. If no
         *  text frame is selected or specified, false is returned.
         */
        this.isAutoResizableTextFrame = function (drawingNode) {

            var // the selected text frame node (also finding text frames inside groups)
                textFrame = drawingNode || selection.getAnyTextFrameDrawing({ forceTextFrame: true });

            return !!textFrame && (textFrame.length > 0) && DrawingFrame.isAutoResizeHeightDrawingFrame(textFrame);
        };

        /**
         * Checking whether a currently selected text frame node or a specified drawing node has
         * 'autoResizeText' enabled or not.
         *
         * @param {Node|jQuery} [drawingNode]
         *  An optional drawing node, that shall be investigated. If not specified, a selected
         *  drawing node is used instead.
         *
         * @returns {Boolean}
         *  Whether a a currently selected text frame node has 'autoResizeText' enabled. If no
         *  text frame is selected or specified, false is returned.
         */
        this.isAutoTextHeightTextFrame = function (drawingNode) {

            var // the selected text frame node (also finding text frames inside groups)
                textFrame = drawingNode || selection.getAnyTextFrameDrawing({ forceTextFrame: true });

            return !!textFrame && (textFrame.length > 0) && DrawingFrame.isAutoTextHeightDrawingFrame(textFrame);
        };

        /**
         * Getting the auto resize state for the selected drawing(s). Allowed values are 'autofit',
         * 'autotextheight' or 'noautofit'.
         *
         * @returns {String|Null}
         *  The auto resize state of the selected drawing(s). If no unique state can be determined,
         *  Null is returned.
         */
        this.getDrawingAutoResizeState = function () {

            var // the auto resize state of the selected drawings
                state = null,
                // a collector for all resize types of the selected drawings
                allResizeTypes = null,
                // one resize type of one selected drawing
                oneType = null,
                // a selected drawing node
                selectedDrawing = null;

            // helper function to receive the resize state for one specified drawing
            function handleOneDrawingState(drawingNode) {
                var node = $(drawingNode);
                return self.isAutoResizableTextFrame(node) ? 'autofit' : (self.isAutoTextHeightTextFrame(node) ? 'autotextheight' : 'noautofit');
            }

            if (selection.isMultiSelection()) {
                allResizeTypes = {};
                _.each(selection.getAllSelectedDrawingNodes(), function (oneDrawing) {
                    if (DrawingFrame.isTextFrameShapeDrawingFrame(oneDrawing)) { // only checking valid drawings. TODO: Groups
                        oneType = handleOneDrawingState(oneDrawing);
                        allResizeTypes[oneType] = 1;
                    } else if (DrawingFrame.isGroupDrawingFrameWithShape(oneDrawing)) {
                        _.each(DrawingFrame.getAllGroupDrawingChildren(oneDrawing), function (child) {
                            if (DrawingFrame.isTextFrameShapeDrawingFrame(child)) {
                                oneType = handleOneDrawingState(child);
                                allResizeTypes[oneType] = 1;
                            }
                        });
                    }
                });
                state = (_.keys(allResizeTypes).length === 1) ? oneType : null;
            } else {

                selectedDrawing = selection.getSelectedDrawing();

                if (selectedDrawing.length  > 0 && DrawingFrame.isGroupDrawingFrame(selectedDrawing)) { // the group drawing itself is selected
                    allResizeTypes = {};
                    _.each(DrawingFrame.getAllGroupDrawingChildren(selectedDrawing), function (oneDrawing) {
                        if (DrawingFrame.isTextFrameShapeDrawingFrame(oneDrawing)) {
                            oneType = handleOneDrawingState(oneDrawing);
                            allResizeTypes[oneType] = 1;
                        }
                    });
                    state = (_.keys(allResizeTypes).length === 1) ? oneType : null;
                } else {
                    state = handleOneDrawingState(selection.getAnyTextFrameDrawing({ forceTextFrame: true }));
                }

            }

            return state;
        };

        /**
         * Adding a default template text into a specified drawing, if the drawing is
         * an empty text frame and a template text is available.
         *
         * @param {Node|jQuery} drawing
         *  The DOM node to be checked. If this object is a jQuery collection, uses
         *  the first DOM node it contains.
         *
         * @param {Object} [options]
         *  Some additional options that are supported by the model function 'getDefaultTextForTextFrame'.
         *  Additionally supported:
         *  @param {Boolean} [options.ignoreTemplateText=false]
         *      This option is handled within DOM.isEmptyTextFrame to define, that text frames
         *      that contain template text are also empty.
         *
         * @returns {Boolean}
         *  Whether the default template text was inserted into the drawing.
         */
        this.addTemplateTextIntoTextSpan = function (drawing, options) {

            var // the text span inside an empty text frame
                textSpan = null,
                // the paragraph node
                paragraph = null,
                // the default text
                text = null,
                // whether the default text was inserted
                insertedText = false;

            if (self.getDefaultTextForTextFrame && DOM.isEmptyTextframe(drawing, options)) {

                text = self.getDefaultTextForTextFrame(drawing, options); // this is application specific

                if (text) {
                    textSpan = $(drawing).find('div.p > span');
                    textSpan.text(text);
                    textSpan.addClass(DOM.TEMPLATE_TEXT_CLASS);
                    paragraph = textSpan.parent();
                    paragraph.removeClass(DOM.PARAGRAPH_NODE_LIST_EMPTY_CLASS); // removing the marker for empty paragraphs (if it was set)
                    paragraph.addClass(DOM.NO_SPELLCHECK_CLASSNAME); // setting marker for avoiding spell checking of template text
                    insertedText = true;
                }
            }

            return insertedText;
        };

        /**
         * Removing template text in a text frame, if there is the default
         * text active inside the text frame.
         *
         * @param {Node|jQuery} oldTextNode
         *  The text node or the text span that will be checked, if it contains
         *  the default text.
         *
         * @returns {Node|Null}
         *  The new empty text node, if the old text node was removed. Otherwise
         *  null is returned.
         */
        this.removeTemplateTextFromTextSpan = function (oldTextNode) {

            var // the new text node in the span
                newTextNode = null,
                // the dom node
                domNode = Utils.getDomNode(oldTextNode),
                // whether the parameter oldTextNode is a text node (not a span)
                isTextNode = domNode.nodeType === 3,
                // the text span node
                mySpan = isTextNode ? domNode.parentNode : domNode;

            if (!isTextNode) { oldTextNode = domNode.firstChild; }

            if (mySpan && $(mySpan).hasClass(DOM.TEMPLATE_TEXT_CLASS)) {
                $(mySpan).removeClass(DOM.TEMPLATE_TEXT_CLASS);
                newTextNode = document.createTextNode(''); // generating a new empty text node
                mySpan.removeChild(oldTextNode);
                mySpan.appendChild(newTextNode);
                $(mySpan).parent().removeClass(DOM.NO_SPELLCHECK_CLASSNAME); // removing spell check marker
            }

            // Info: Is it necessary to remove neighboring template text spans, too? This should never happen.
            //       It was triggered by er spellchecking.

            return newTextNode;
        };

        /**
         * Inserting a text frame drawing node of type 'shape' into the document.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the dialog asking the user, if
         *  he wants to delete the selected content (if any) has been closed
         *  with the default action; or rejected, if the dialog has been
         *  canceled. If no dialog is shown, the promise is resolved
         *  immediately.
         */
        this.insertTextFrame = function () {

            // the undo manager returns the return value of the callback function
            return undoManager.enterUndoGroup(function () {

                // helper function, that generates the operations
                function doInsertTextFrame() {

                    var // the operations generator
                        generator = self.createOperationGenerator(),
                        // the default width of the textframe in hmm
                        defaultWidth = 5000,
                        // the width of the parent paragraph
                        paraWidth = 0,
                        // the current cursor position
                        start = selection.getStartPosition(),
                        // the default border attributes
                        lineAttrs = { color: { type: 'rgb', value: '000000' }, style: 'single', type: 'solid', width: Border.getWidthForPreset('thin') },
                        // the default fill color attributes
                        fillAttrs = { color: { type: 'auto' }, type: 'solid' },
                        // the default attributes
                        attrs = { drawing: { width: defaultWidth }, shape: { autoResizeHeight: true, noAutoResize: false }, line: lineAttrs, fill: fillAttrs },
                        // the position of the first paragraph inside the text frame
                        paraPos = _.clone(start),
                        // a required style ID for style handling
                        parentStyleId = null,
                        // the paragraph node, in which the text frame will be inserted
                        paraNode = null,
                        // target for operation - if exists, it's for ex. header or footer
                        target = self.getActiveTarget(),
                        // created operation
                        newOperation = null;

                    handleImplicitParagraph(start);

                    // checking the width of the parent paragraph (36836)
                    paraNode = Position.getParagraphElement(editdiv, _.initial(start));

                    if (paraNode) {
                        paraWidth = Utils.convertLengthToHmm($(paraNode).width(), 'px');
                        if (paraWidth < defaultWidth) {
                            attrs.drawing.width = paraWidth;
                        }
                    }

                    paraPos.push(0);

                    // modifying the attributes, if changeTracking is activated
                    if (changeTrack.isActiveChangeTracking()) {
                        attrs.changes = { inserted: changeTrack.getChangeTrackInfo(), removed: null };
                    }

                    // Adding styleId (only for ODT)
                    if (app.isODF()) { attrs.styleId = 'Frame'; }

                    // handling the required styles for the text frame
                    if (_.isString(attrs.styleId) && drawingStyles.isDirty(attrs.styleId)) {

                        // checking the parent style of the specified style
                        parentStyleId = drawingStyles.getParentId(attrs.styleId);

                        if (_.isString(parentStyleId) && drawingStyles.isDirty(parentStyleId)) {
                            // inserting parent of text frame style to document
                            self.generateInsertStyleOp(generator, 'drawing', parentStyleId, true);
                            drawingStyles.setDirty(parentStyleId, false);
                        }

                        // insert text frame style to document
                        self.generateInsertStyleOp(generator, 'drawing', attrs.styleId);
                    }

                    // text frames are drawings of type shape
                    newOperation = { attrs: attrs, start: start, type: 'shape' };
                    extendPropertiesWithTarget(newOperation, target);
                    generator.generateOperation(Operations.INSERT_DRAWING, newOperation);

                    // add a paragraph into the shape, so that the cursor can be set into the text frame
                    // -> an operation is required for this step, so that remote clients are also prepared (-> no implicit paragraph)
                    // TODO: Change tracking of this paragraph?
                    newOperation = { start: paraPos };
                    extendPropertiesWithTarget(newOperation, target);
                    generator.generateOperation(Operations.PARA_INSERT, newOperation);

                    // reducing distance of paragraphs inside the text frame
                    newOperation = { start: paraPos, attrs: { paragraph: { marginBottom: 0 } } };
                    extendPropertiesWithTarget(newOperation, target);
                    generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);

                    // apply all collected operations
                    self.applyOperations(generator);

                    // setting cursor into text frame
                    selection.setTextSelection(lastOperationEnd);
                }

                if (selection.hasRange()) {
                    return self.deleteSelected()
                        .done(function () {
                            doInsertTextFrame();
                        });
                }

                doInsertTextFrame();
                return $.when();

            }, this);  // enterUndoGroup()
        };

        /**
         * Generating an operation to assign a new anchor to a drawing. Supported anchor
         * values are 'inline', 'paragraph' and 'page'.
         *
         * @param {String} anchor
         *  A string describing the new anchor position of the drawing.
         */
        this.anchorDrawingTo = function (anchor) {

            var // the drawing position
                start = null,
                // the pixel position of the drawing relative to the page node
                pixelPos = null,
                // the drawing node info object
                startNodeInfo = null,
                // an optional footer offset in px
                footerOffset = 0,
                // the page number of the drawing
                pageNumber = 1,
                // an options object
                options = {
                    anchorHorBase: null,
                    anchorHorAlign: null,
                    anchorVertBase: null,
                    anchorVertAlign: null
                },
                // currently active root node
                activeRootNode = self.getCurrentRootNode(),
                // drawing attributes required for the order of the drawing
                drawingOrderAttrs = null,
                // a helper variable for the drawing node
                drawingNode = null;

            if (self.isDrawingSelected()) {
                start = selection.getStartPosition();
            } else if (self.getSelection().isAdditionalTextframeSelection()) {
                start = Position.getOxoPosition(activeRootNode, self.getSelection().getSelectedTextFrameDrawing(), 0);
            } else {
                return;
            }

            startNodeInfo = Position.getDOMPosition(activeRootNode, start, true);

            if (startNodeInfo && startNodeInfo.node && DrawingFrame.isDrawingFrame(startNodeInfo.node)) {

                if (anchor === 'inline') {
                    // set anchor inline into the text
                    options.inline = true;

                    // setting other values to their defaults
                    options.anchorHorAlign = null;
                    options.anchorVertAlign = null;
                    options.anchorHorOffset = null;
                    options.anchorVertOffset = null;
                    options.anchorHorBase = null;
                    options.anchorVertBase = null;

                    options.anchorBehindDoc = null;
                    options.anchorLayerOrder = null;

                } else {

                    drawingNode = startNodeInfo.node;

                    // set Anchor to page, margin or paragraph
                    if (anchor === 'page' || anchor === 'margin') {

                        pixelPos = Position.getPixelPositionToRootNodeOffset(activeRootNode, startNodeInfo.node, app.getView().getZoomFactor() * 100);

                        if (DOM.isHeaderNode(activeRootNode)) {
                            pageNumber = 1;  // in header or footer the page number is not required
                        } else if (DOM.isFooterNode(activeRootNode)) {
                            pageNumber = 1;  // in header or footer the page number is not required
                            // if this is a footer, the height between page start and footer need to be added
                            footerOffset = Utils.convertHmmToLength(pageLayout.getPageAttribute('height'), 'px', 1) - Utils.round($(activeRootNode).outerHeight(true), 1);
                            pixelPos.y += footerOffset;
                        } else {
                            if (DOM.isDrawingLayerNode(startNodeInfo.node.parentNode)) {
                                drawingNode = DOM.getDrawingPlaceHolderNode(drawingNode); // using previous sibling (see template doc)
                            }
                            pageNumber = pageLayout.getPageNumber(drawingNode); // maybe this drawing is not on page 1
                            if (pageNumber > 1) { pixelPos.y -= Position.getVerticalPagePixelPosition(activeRootNode, pageLayout, pageNumber, app.getView().getZoomFactor() * 100); }
                        }

                        if (anchor === 'margin') {
                            // handling top and left margin
                            pixelPos.x -= Utils.convertHmmToLength(pageLayout.getPageAttribute('marginLeft'), 'px', 1);
                            pixelPos.y -= Utils.convertHmmToLength(pageLayout.getPageAttribute('marginTop'), 'px', 1);
                        }

                        options.anchorHorBase = anchor;
                        options.anchorVertBase = anchor;

                        options.anchorHorOffset = Utils.convertLengthToHmm(pixelPos.x, 'px');
                        options.anchorVertOffset = Utils.convertLengthToHmm(pixelPos.y, 'px');

                    } else {
                        options.anchorHorBase = 'column';
                        options.anchorVertBase = 'paragraph';

                        options.anchorHorOffset = 0;
                        options.anchorVertOffset = 0;
                    }

                    options.inline = false;
                    options.anchorHorAlign = 'offset';
                    options.anchorVertAlign = 'offset';

                    // getting the operation attributes for 'anchorBehindDoc' and 'anchorLayerOrder', if required
                    if (drawingNode && DOM.isInlineComponentNode(drawingNode)) {
                        drawingOrderAttrs = self.getDrawingLayer().getDrawingOrderAttributes('front', drawingNode);
                        if (drawingOrderAttrs) { _.extend(options, drawingOrderAttrs); }
                    }
                }

                this.setAttributes('drawing', { drawing: options });

                // the selection needs to be repainted. Otherwise it is not possible
                // to move the drawing to the page borders.
                DrawingFrame.clearSelection(startNodeInfo.node);
                selection.drawDrawingSelection(startNodeInfo.node);
            }

        };

        /**
         * Generating an operation to assign a border to a drawing.
         *
         * @param {String} preset
         *  A string describing the new border of the drawing.
         */
        this.setDrawingBorder = function (preset) {

            var // the selection object
                selection = self.getSelection(),
                // new line attributes
                lineAttrs = null,
                // the selected drawings of a multi drawing selection
                selectedDrawings = null,
                // the positions of the selected drawings of a multi drawing selection
                allDrawingPositions = null;

            // helper function to generate operation(s)
            function generateDrawingBorderOperation(localLineAttrs) {

                // the attributes of the currently selected drawing
                var oldLineAttrs = self.getAttributes('drawing').line;

                if (localLineAttrs.type !== 'none' && (!oldLineAttrs || !oldLineAttrs.color || (oldLineAttrs.color.type === 'auto'))) {
                    localLineAttrs.color = { type: 'rgb', value: '000000' };
                }

                self.setAttributes('drawing', { line: localLineAttrs }, { drawingFilterRequired: true });
            }

            // there must be a drawing selection
            if ((!selection.isAnyDrawingSelection()) || (!self.isDrawingSelected() && !selection.isAdditionalTextframeSelection())) { return; }

            lineAttrs = DrawingUtils.resolvePresetBorder(preset);

            // if no border color is available, push default black (but not for multi drawing selections (52421))
            if (!selection.isMultiSelection()) {
                generateDrawingBorderOperation(_.copy(lineAttrs, true));
            } else {

                selectedDrawings = selection.getMultiSelection(); // handling for multi drawing selection 57897
                allDrawingPositions = [];

                undoManager.enterUndoGroup(function () {

                    _.each(selectedDrawings, function (oneSelection) {

                        var startPosition = selection.getStartPositionFromMultiSelection(oneSelection);
                        var endPosition = selection.getEndPositionFromMultiSelection(oneSelection);

                        allDrawingPositions.push(startPosition);

                        // setting each drawing selection individually, so that self.setAttributes can still be used
                        selection.setTextSelection(startPosition, endPosition);

                        generateDrawingBorderOperation(_.copy(lineAttrs, true));

                    });
                });

                // restoring the multi selection
                selection.setMultiDrawingSelectionByPosition(allDrawingPositions);
            }

        };

        /**
         * Generating an operation to assign lineending to a line/connector.
         *
         * @param {String} lineendings
         *  A string describing the new head and tail type of the line/connector.
         */
        this.setLineEnds = function (endings) {
            // there must be a drawing selection
            if (!self.getSelection().isAnyDrawingSelection()) { return; }

            if (!self.isDrawingSelected() && !self.getSelection().isAdditionalTextframeSelection()) {
                return;
            }

            var oldLineAttrs    = self.getAttributes('drawing').line,
                end             = endings.split(':'),
                head            = (end[0] !== 'none') ? end[0] : null,
                tail            = (end[1] !== 'none') ? end[1] : null,
                lineAttrs       = {};

            if (oldLineAttrs.headEndType !== head) { lineAttrs.headEndType = head; }
            if (oldLineAttrs.tailEndType !== tail) { lineAttrs.tailEndType = tail; }

            if (!_.isEmpty(lineAttrs)) {
                this.setAttributes('drawing', { line: lineAttrs });
            }
        };

        /**
         * Generating an operation to assign a color to a drawing border.
         *
         * @param {Object} color
         *  An object describing the new border color of the drawing.
         */
        this.setDrawingBorderColor = function (color) {

            // there must be a drawing selection
            if (!self.getSelection().isAnyDrawingSelection()) { return; }

            if (!self.isDrawingSelected() && !self.getSelection().isAdditionalTextframeSelection()) {
                return;
            }

            if (color.type === 'auto') {
                self.setAttributes('drawing', { line: { type: 'none' } }, { drawingFilterRequired: true });
            } else {
                self.setAttributes('drawing', { line: { type: 'solid', color: color } }, { drawingFilterRequired: true });
            }
        };

        /**
         * Generating an operation to assign a fill color to a drawing.
         *
         * Info: In slide mode the filter does not accept color 'auto'. In this
         *       case it is handled like a removal of any background, if the
         *       user selects 'auto'.
         *
         * @param {Object} color
         *  An object describing the new fill color of the drawing.
         */
        this.setDrawingFillColor = function (color) {
            // there must be a drawing selection
            if (!self.getSelection().isAnyDrawingSelection()) { return; }

            var // the fill attributes for the drawing
                fillAttrs = TextUtils.getEmptyDrawingBackgroundAttributes().fill;

            if (color.type === 'auto') {
                if (self.useSlideMode()) {
                    fillAttrs.type = null;
                } else {
                    fillAttrs.type = 'none';
                    fillAttrs.color = { type: 'auto' };
                }
            } else {
                fillAttrs.color = color;
                fillAttrs.type = 'solid';
            }

            self.setAttributes('drawing', { fill: fillAttrs }, { drawingFilterRequired: true });
        };

        /**
         * Handles any drawing (shape/image) cropping properties.
         *
         * @param {Boolean} state
         *  Set/unset Crop mode.
         */
        this.handleDrawingCrop = function (state) {
            if (state) {
                var drawing = selection.getAnyDrawingSelection();
                var attrs = AttributeUtils.getExplicitAttributes(drawing);
                if (attrs) {
                    var isImageType = selection.getSelectedDrawingType() === 'image';
                    var contentNode = DrawingFrame.getContentNode(drawing);
                    var cropNode = $('<div class="drawing crop">');
                    var zoomFactor = app.getView().getZoomFactor();
                    var scrollNode = app.getView().getContentRootNode();
                    // var presetShapeId = (attrs.geometry && attrs.geometry.presetShape) || null;
                    // var options = { noFill: true };
                    // var shapeSize = { width: drawing.width(), height: drawing.height() };
                    // var widthPx = Utils.convertHmmToLength(attrs.drawing.width, 'px', 1);
                    // var heightPx = Utils.convertHmmToLength(attrs.drawing.height, 'px', 1);

                    selection.setActiveCropNode(drawing);

                    if (isImageType) {
                        var imgClone = contentNode.find('img').clone();
                        var imgPosLeft = parseInt(imgClone.css('left'), 10);
                        var imgPosTop = parseInt(imgClone.css('top'), 10);
                        var imgWidth = parseInt(imgClone.css('width'), 10);
                        var imgHeight = parseInt(imgClone.css('height'), 10);
                        cropNode.css({ marginLeft: imgPosLeft, marginTop: imgPosTop, width: imgWidth, height: imgHeight }).append(imgClone.css({ opacity: 0.5, left: 0, top: 0, position: 'absolute' }));
                    } else { // shape with bitmap fill
                        // var previewCanvas = DrawingFrame.drawPreviewShape(app, drawing, shapeSize, presetShapeId, options);
                        // contentNode.append(previewCanvas);
                        // var bitmapUrlAttr = attrs.fill.bitmap.imageUrl;
                        // var calcPosSize = attrs.fill.bitmap.stretching ? DrawingUtils.getStrechedCoords(attrs.fill.bitmap.stretching, widthPx, heightPx) : attrs.drawing;
                        // var imageUrl = ImageUtil.getFileUrl(app, bitmapUrlAttr);
                        // var fakeDiv = $('<img>').attr('src', imageUrl).css({
                        //     width: calcPosSize.width,
                        //     height: calcPosSize.height,
                        //     opacity: 0.5,
                        //     position: 'absolute',
                        //     left: 0,
                        //     top: 0
                        // });
                        // cropNode.css({ width: calcPosSize.width, height: calcPosSize.height, left: calcPosSize.left, top: calcPosSize.top }).append(fakeDiv);
                    }
                    contentNode.append(cropNode);
                    ImageCropFrame.drawCropFrame(self, drawing, cropNode, zoomFactor, scrollNode);

                } else {
                    Utils.warn('editor.handleDrawingCrop(): no explicit attributes for drawing: ', drawing);
                }
            } else {
                self.exitCropMode();
            }
        };

        /**
         * Exit image crop mode and clean up cropped drawing.
         */
        this.exitCropMode = function () {
            var cropNode = selection.getActiveCropNode();
            ImageCropFrame.clearCropFrame(cropNode);
            selection.setActiveCropNode($());
        };

        /**
         * Publicly available method from model to get selected node and refresh it's crop frame.
         */
        this.refreshCropFrame = function () {
            var cropNode = selection.getActiveCropNode();
            ImageCropFrame.refreshCropFrame(cropNode);
        };

        /**
         * Public method for fill or fit types of image crop,
         * accesable via dropdown toolbar menu.
         *
         * @param {String} type
         *  Can be 'fill' or 'fit'.
         */
        this.cropFillFitSpace = function (type) {
            var drawing = self.getSelection().getSelectedDrawing();
            var explAttrs = AttributeUtils.getExplicitAttributes(drawing);

            ImageCropFrame.cropFillFitSpace(self, explAttrs, type).then(function (cropOps) {
                if (cropOps && !_.isEmpty(cropOps)) {
                    var opProperties = { start: self.getSelection().getStartPosition(), attrs: { image: cropOps } };
                    var generator = self.createOperationGenerator();
                    generator.generateOperation(Operations.SET_ATTRIBUTES, opProperties);
                    self.applyOperations(generator);
                }
            });
        };

        /**
         * Opens cropping image dialog.
         */
        this.openCropPositionDialog = function () {
            if (selection.isCropMode()) { self.handleDrawingCrop(false); }

            ImageCropFrame.openCropDialog(app.getView());
        };

        /**
         * Shows an insert image dialog from drive, local or URL dialog
         *
         * @param {Object} imageHolder
         *  The object containing the image attributes. This are 'url' or 'substring'.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number[]} [options.position=null]
         *   The logical position at that the drawing shall be inserted.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved or rejected after the image is inserted or
         *  if an error occured during inserting the image.
         */
        this.insertImageURL = function (imageHolder, options) {

            var attrs = { drawing: {}, image: {}, line: { type: 'none' } };

            // 56351: Setting explicit autoResizeHeight property for ODT
            if (app.isODF() && app.isTextApp()) { attrs.shape = { autoResizeHeight: false }; }

            if (imageHolder.url) {
                attrs.image.imageUrl = imageHolder.url;
                attrs.drawing.name = imageHolder.name;
            } else if (imageHolder.substring(0, 10) === 'data:image') {
                attrs.image.imageData = imageHolder;
            } else {
                attrs.image.imageUrl = imageHolder;
            }

            return getImageSize(attrs.image.imageUrl || attrs.image.imageData).then(function (size) {

                var def = self.createDeferred('Editor.insertImageURL'),
                    result = false,
                    // created operation
                    newOperation = null,
                    // the logical position of the inserted drawing
                    start = null;

                // exit silently if we lost the edit rights
                if (!app.isEditable()) {
                    return def.resolve();
                }

                undoManager.enterUndoGroup(function () {

                    // the logical position for the insertDrawing operation
                    start = Utils.getArrayOption(options, 'position', null) || selection.getStartPosition();

                    _.extend(attrs.drawing, size, DEFAULT_DRAWING_MARGINS);

                    handleImplicitParagraph(start);

                    // modifying the attributes, if changeTracking is activated
                    if (changeTrack.isActiveChangeTracking()) {
                        attrs.changes = { inserted: changeTrack.getChangeTrackInfo(), removed: null };
                    }

                    newOperation = {
                        name: Operations.INSERT_DRAWING,
                        start: start,
                        type: 'image',
                        attrs: attrs
                    };
                    extendPropertiesWithTarget(newOperation, self.getActiveTarget());

                    result = self.applyOperations(newOperation);
                }); // enterUndoGroup()

                // setting the cursor position (selecting the inserted drawing)
                selection.setTextSelection(_.copy(start), Position.increaseLastIndex(start));

                return result ? def.resolve() : def.reject();
            })
                .fail(function () {
                    app.rejectEditAttempt('image');
                });
        };

        /**
         * Inserting a tab stopp.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the dialog has been closed with
         *  the default action; or rejected, if the dialog has been canceled.
         *  If no dialog is shown, the promise is resolved immediately.
         */
        this.insertTab = function () {

            // the undo manager returns the return value of the callback function
            return undoManager.enterUndoGroup(function () {

                function doInsertTabAndSetCursor() {
                    var start = selection.getStartPosition(),
                        operation = { name: Operations.TAB_INSERT, start: start },
                        // target for operation - if exists, it's for ex. header or footer
                        target = self.getActiveTarget();

                    extendPropertiesWithTarget(operation, target);
                    handleImplicitParagraph(start);
                    if (_.isObject(preselectedAttributes)) { operation.attrs = preselectedAttributes; }
                    // modifying the attributes, if changeTracking is activated
                    if (changeTrack.isActiveChangeTracking()) {
                        operation.attrs = operation.attrs || {};
                        operation.attrs.changes = { inserted: changeTrack.getChangeTrackInfo(), removed: null };
                    }
                    self.applyOperations(operation);

                    if (Utils.IOS && self.isHeaderFooterEditState()) {
                        //header footer workaround (Bug 39976)
                        self.applyOperations({ name: Operations.TEXT_INSERT, text: ' ', start: Position.increaseLastIndex(start) });
                        self.deleteRange(Position.increaseLastIndex(start));
                    } else {
                        selection.setTextSelection(lastOperationEnd);
                    }
                }

                if (selection.hasRange()) {
                    return self.deleteSelected()
                    .done(function () {
                        doInsertTabAndSetCursor();
                    });
                }

                doInsertTabAndSetCursor();

                return $.when();

            }, this);

        };

        this.insertText = function (text, position, attrs) {

            var // target for operation - if exists, it's for ex. header or footer
                target = self.getActiveTarget(),
                // current element container node
                rootNode = self.getCurrentRootNode(target);

            // In IE and on Android text input in slide selection happens sometimes (focus not set fast enough into clipboard)
            if ((_.browser.IE || _.browser.Android) && self.useSlideMode() && position.length === 2) {
                Utils.error('Editor.insertText: Invalid operation: ' + JSON.stringify(position) + ' : ' + text);
            }

            // adding changes attributes -> but for performance reasons only, if the change tracking is active!
            if (changeTrack.isActiveChangeTracking()) {
                attrs = attrs || {};
                attrs.changes = { inserted: changeTrack.getChangeTrackInfo(), removed: null };
            }

            undoManager.enterUndoGroup(function () {

                var operation = { name: Operations.TEXT_INSERT, text: text, start: _.clone(position) };

                handleImplicitParagraph(position);
                if (_.isObject(attrs)) { operation.attrs = _.copy(attrs, true); }
                guiTriggeredOperation = true;  // Fix for 30587, remote client requires synchronous attribute setting
                this.applyOperations(operation);
                guiTriggeredOperation = false;

                // Bug 25077: Chrome cannot set curser after a trailing SPACE
                // character, need to validate the paragraph node immediately
                // (which converts trailing SPACE characters to NBSP)
                // -> only do this, if it is really necessary (performance)
                if (_.browser.WebKit && (text === ' ')) {
                    validateParagraphNode(Position.getParagraphElement(rootNode, position.slice(0, -1)));
                }

            }, this);
        };

        // style sheets and formatting attributes -----------------------------

        /**
         * Returns the collection of all list definitions.
         */
        this.getListCollection = function () {
            return listCollection;
        };

        /**
         * Setting the application specific list collection.
         */
        this.setListCollection = function (collection) {
            listCollection = collection;
        };

        /**
         * Returns the current selection.
         */
        this.getSelection = function () {
            return selection;
        };

        /**
         * Returns the remote selection.
         */
        this.getRemoteSelection = function () {
            return remoteSelection;
        };

        /**
         * Returns whether the current selection selects any text. This
         * includes the rectangular table cell selection mode.
         */
        this.isTextSelected = function () {
            return selection.getSelectionType() !== 'drawing';
        };

        /**
         * Returns whether the editor contains a selection range (text,
         * drawings, or table cells) instead of a simple text cursor.
         */
        this.hasSelectedRange = function () {
            return selection.hasRange();
        };

        /**
         * Returns whether the editor contains a selection within a
         * single paragraph or not.
         */
        this.hasEnclosingParagraph = function () {
            return selection.getEnclosingParagraph() !== null;
        };

        // PUBLIC TABLE METHODS

        /**
         * Returns whether the editor contains a selection within a
         * table or not.
         */
        this.isPositionInTable = function () {
            return !_.isNull(selection.getEnclosingTable());
        };

        /**
         * Returns whether the editor contains a selection over one or multiple
         * cells within a table
         */
        this.isCellRangeSelected = function () {
            // Firefox has beautiful cell-range-selection
            if (_.browser.Firefox) {
                var currentSelection = selection.getBrowserSelection();
                return (currentSelection && currentSelection.active) ? DOM.isCellRangeSelected(currentSelection.active.start.node, currentSelection.active.end.node) : false;

            // every other browser has to check the selection
            } else {
                return Position.positionsInTowCellsInSameTable(self.getCurrentRootNode(), selection.getStartPosition(), selection.getEndPosition());
            }
        };

        /**
         * Returns the number of columns, if the editor contains a selection within a
         * table.
         */
        this.getNumberOfColumns = function () {
            var table = selection.getEnclosingTable() || selection.isTableDrawingSelection();
            return table && Table.getColumnCount(table);
        };

        /**
         * Returns the number of rows, if the editor contains a selection within a
         * table.
         */
        this.getNumberOfRows = function () {
            var table = selection.getEnclosingTable() || selection.isTableDrawingSelection();
            return table && Table.getRowCount(table);
        };

        /**
         * Returns whether a further row can be added, if the editor contains a
         * selection within a table. The maximum number of rows and cells can
         * be defined in the configuration.
         */
        this.isRowAddable = function () {

            var // the number of rows in the current table
                rowCount = this.getNumberOfRows(),
                // the number of columns in the current table
                colCount = this.getNumberOfColumns();

            // check that cursor is located in a table, check maximum row count and maximum cell count
            return _.isNumber(rowCount) && _.isNumber(colCount) &&
                (rowCount < Config.MAX_TABLE_ROWS) &&
                ((rowCount + 1) * colCount <= Config.MAX_TABLE_CELLS);
        };

        /**
         * Returns whether a further column can be added, if the editor contains a selection within a
         * table. The maximum number of columns and cells can be defined in the configuration.
         */
        this.isColumnAddable = function () {

            var // the number of rows in the current table
                rowCount = this.getNumberOfRows(),
                // the number of columns in the current table
                colCount = this.getNumberOfColumns();

            // check that cursor is located in a table, check maximum column count and maximum cell count
            return _.isNumber(rowCount) && _.isNumber(colCount) &&
                (colCount < Config.MAX_TABLE_COLUMNS) &&
                ((colCount + 1) * rowCount <= Config.MAX_TABLE_CELLS);
        };

        // PUBLIC DRAWING METHODS

        /**
         * Returns whether the current selection selects one or more drawings.
         *
         * @returns {Boolean}
         */
        this.isDrawingSelected = function () {
            return selection.getSelectionType() === 'drawing';
        };

        /**
         * Returns whether the current selection is a text selection inside a shape
         * in an odf document. These 'shapes with text' need to be distinguished from
         * 'classical' text frames in odf format, because much less functionality is
         * available. It is also handled correctly, if the selection is inside a grouped
         * shape.
         *
         * @returns {Boolean}
         */
        this.isReducedOdfTextframeFunctionality = function () {
            var selectedTextFrame = selection.getSelectedTextFrameDrawing();

            return app.isODF() && selection.isAdditionalTextframeSelection() &&
                (DrawingFrame.isReducedOdfTextframeNode(selectedTextFrame) ||
                (DrawingFrame.isGroupDrawingFrame(selectedTextFrame) && DrawingFrame.isReducedOdfTextframeNode(Position.getClosestDrawingAtPosition(selection.getRootNode(), selection.getStartPosition()))));
        };

        /**
         * Returns whether the current selection is a text selection inside a 'classic'
         * text frame in an odf document. These 'classic' text frames need to be
         * distinguished from 'shapes with text', because more functionality is
         * available. It is also handled correctly, if the selection is inside a grouped
         * shape.
         *
         * @returns {Boolean}
         */
        this.isFullOdfTextframeFunctionality = function () {
            var selectedTextFrame = selection.getSelectedTextFrameDrawing();

            return app.isODF() && selection.isAdditionalTextframeSelection() &&
                (DrawingFrame.isFullOdfTextframeNode(selectedTextFrame) ||
                (DrawingFrame.isGroupDrawingFrame(selectedTextFrame) && DrawingFrame.isFullOdfTextframeNode(Position.getClosestDrawingAtPosition(selection.getRootNode(), selection.getStartPosition()))));
        };

        /**
         * Returns true if specified node is inside a text frame node inside a drawing.
         * This function also returns true, if the specified node is a text frame node
         * itself.
         *
         * @param {jQuery|Node} node
         *  The node, that is checked.
         *
         * @returns {Boolean}
         *  Whether the specified node is inside a text frame node in a drawing or is
         *  the text frame node itself.
         */
        this.isTextframeOrInsideTextframe = function (node) {
            return $(node).parentsUntil(DOM.PAGE_NODE_SELECTOR, DrawingFrame.TEXTFRAME_NODE_SELECTOR).length > 0;
        };

        /**
         * Returns true if specified node is inside a text frame node inside a drawing.
         * This function does not (!) returns true, if the specified node is a text
         * frame node itself.
         *
         * @param {jQuery|Node} node
         *  The node, that is checked.
         *
         * @returns {Boolean}
         *  Whether the specified node is inside a text frame node in a drawing.
         */
        this.isNodeInsideTextframe = function (node) {
            return !DrawingFrame.isTextFrameNode(node) && $(node).parentsUntil(DOM.PAGE_NODE_SELECTOR, DrawingFrame.TEXTFRAME_NODE_SELECTOR).length > 0;
        };

        /**
         * Returns whether the current selection is a selection inside a comment
         * in an odf document.
         *
         * @returns {Boolean}
         */
        this.isOdfCommentFunctionality = function () {
            return app.isODF() && self.isCommentFunctionality();
        };

        /**
         * Returns whether the current selection is a selection for inserting a shape.
         * -> only main document or header/footer is active.
         * In all other cases (for example text selection inside a text frame) a valid
         * text position can be calculated.
         *
         * @returns {Boolean}
         */
        this.isInsertShapePosition = function () {
            return !self.getActiveTarget() || self.isHeaderFooterEditState();
        };

        /**
         * Returns whether the current selection is a selection inside a comment.
         *
         * @returns {Boolean}
         */
        this.isCommentFunctionality = function () {
            var activeTarget = self.getActiveTarget();

            return activeTarget ? !_.isNull(commentLayer.getCommentRootNode(activeTarget)) : false;
        };

        /**
         * Returns whether the current selection is a selection inside header/footer.
         *
         * @returns {Boolean}
         */
        this.isHeaderFooterFunctionality = function () {
            return self.getActiveTarget() ? self.isHeaderFooterEditState() : false;
        };

        /**
         * Returns whether the current selection is a selection inside a comment or a header.
         *
         * @returns {Boolean}
         */
        this.isTargetFunctionality = function () {
            return self.getActiveTarget() ? (self.isCommentFunctionality() || self.isHeaderFooterEditState()) : false;
        };

        /**
         * Returns whether the current selection selects text, not cells and
         * not drawings.
         *
         * @returns {Boolean}
         */
        this.isTextOnlySelected = function () {
            return selection.getSelectionType() === 'text';
        };

        /**
         * Returns whether clipboard paste is in progress.
         *
         * @returns {Boolean}
         */
        this.isClipboardPasteInProgress = function () {
            return pasteInProgress;
        };

        /**
         * Returns the default lateral heading character styles
         *
         * @returns {Array}
         */
        this.getDefaultHeadingCharacterStyles = function () {
            return HEADINGS_CHARATTRIBUTES;
        };

        /**
         * Returns the default lateral paragraph style
         *
         * @returns {Object}
         */
        this.getDefaultParagraphStyleDefinition = function () {
            return DEFAULT_PARAGRAPH_DEFINTIONS;
        };

        /**
         * Returns the default lateral table definiton
         *
         * @returns {Object}
         */
        this.getDefaultLateralTableDefinition = function () {
            return self.useSlideMode() ? DEFAULT_LATERAL_TABLE_DEFINITIONS_PRESENTATION : DEFAULT_LATERAL_TABLE_DEFINITIONS;
        };

        /**
         * Returns the default lateral table attributes
         *
         * @returns {Object}
         */
        this.getDefaultLateralTableAttributes = function () {
            return self.useSlideMode() ? TableStyles.getMediumStyle2TableStyleAttributesPresentation('accent1', 'light1', 'dark1') : DEFAULT_LATERAL_TABLE_ATTRIBUTES;
        };

        /**
         * Returns the default lateral table definiton for ODF only
         *
         * @returns {Object}
         */
        this.getDefaultLateralTableODFDefinition = function () {
            return DEFAULT_LATERAL_TABLE_OX_DEFINITIONS;
        };

        /**
         * Returns the default lateral table attributes for ODF only
         *
         * @returns {Object}
         */
        this.getDefaultLateralTableODFAttributes = function () {
            return DEFAULT_LATERAL_TABLE_OX_ATTRIBUTES;
        };

        /**
         * Returns default lateral hyperlink definition.
         *
         * @returns {Object}
         */
        this.getDefaultLateralHyperlinkDefinition = function () {
            return DEFAULT_HYPERLINK_DEFINTIONS;
        };

        /**
         * Returns default lateral hyperlink character attributes.
         *
         * @returns {Object}
         */
        this.getDefaultLateralHyperlinkAttributes = function () {
            return DEFAULT_HYPERLINK_CHARATTRIBUTES;
        };

        /**
         * Returns the default lateral drawing definition.
         *
         * @returns {Object}
         */
        this.getDefaultDrawingDefintion = function () {
            return DEFAULT_DRAWING_DEFINITION;
        };

        /**
         * Returns the default drawing attributes.
         *
         * @returns {Object}
         */
        this.getDefaultDrawingAttributes = function () {
            return DEFAULT_DRAWING_ATTRS;
        };

        /**
         * Returns the default drawing margins.
         *
         * @returns {Object}
         */
        this.getDefaultDrawingMargins = function () {
            return DEFAULT_DRAWING_MARGINS;
        };

        /**
         * Returns the default drawing textframe definition.
         *
         * @returns {Object}
         */
        this.getDefaultDrawingTextFrameDefintion = function () {
            return DEFAULT_DRAWING_TEXTFRAME_DEFINITION;
        };

        /**
         * Returns the default drawing text frame attributes.
         *
         * @returns {Object}
         */
        this.getDefaultDrawingTextFrameAttributes = function () {
            return (self.useSlideMode() && app.isODF()) ? DEFAULT_DRAWING_TEXTFRAME_ATTRS_ODF_SLIDE : DEFAULT_DRAWING_TEXTFRAME_ATTRS;
        };

        /**
         * Returns the default lateral comment definitions.
         *
         * @returns {Object}
         */
        this.getDefaultCommentTextDefintion = function () {
            return DEFAULT_LATERAL_COMMENT_DEFINITIONS;
        };

        /**
         * Returns the default lateral header definitions.
         *
         * @returns {Object}
         */
        this.getDefaultHeaderTextDefinition = function () {
            return DEFAULT_LATERAL_HEADER_DEFINITIONS;
        };

        /**
         * Returns the default lateral footer definitions.
         *
         * @returns {Object}
         */
        this.getDefaultFooterTextDefinition = function () {
            return DEFAULT_LATERAL_FOOTER_DEFINITIONS;
        };

        /**
         * Returns the default comment text attributes.
         *
         * @returns {Object}
         */
        this.getDefaultCommentTextAttributes = function () {
            return DEFAULT_LATERAL_COMMENT_ATTRIBUTES;
        };

        /**
         * Returns the document default paragraph stylesheet id
         *
         * @returns {String|Null}
         */
        this.getDefaultUIParagraphStylesheet = function () {
            var styleNames = [],
                styleId = null;

            if (app.isODF()) {
                styleNames = paragraphStyles.getStyleSheetNames();
                _(styleNames).each(function (name, id) {
                    var lowerId = id.toLowerCase();
                    if (lowerId === 'standard' || lowerId === 'normal') {
                        styleId = id;
                    }
                });

                if (!styleId) {
                    styleId = paragraphStyles.getDefaultStyleId();
                }
            } else {
                styleId = paragraphStyles.getDefaultStyleId();
            }

            return styleId;
        };

        /**
         * Returns the document default table stylesheet id
         * which can be used to set attributes for a new
         * table.
         *
         * @returns {String|Null}
         */
        this.getDefaultUITableStylesheet = function () {
            var styleNames = tableStyles.getStyleSheetNames(),
                highestUIPriority = 99,
                tableStyleId = null;

            _(styleNames).each(function (name, id) {
                var uiPriority = tableStyles.getUIPriority(id);

                if (_.isNumber(uiPriority) && (uiPriority < highestUIPriority)) {
                    tableStyleId = id;
                    highestUIPriority = uiPriority;
                }
            });

            return tableStyleId;
        };

        /**
         * Returns the document default hyperlink stylesheet id.
         *
         * @returns {String|Null}
         */
        this.getDefaultUIHyperlinkStylesheet = function () {
            var styleNames = characterStyles.getStyleSheetNames(),
                hyperlinkId = null;

            _(styleNames).find(function (name, id) {
                var lowerName = name.toLowerCase();
                if (lowerName.indexOf('hyperlink') >= 0) {
                    hyperlinkId = id;
                    return true;
                }
                return false;
            });

            return hyperlinkId;
        };

        /**
         * Returns the document default paragraph list stylesheet id.
         *
         * @returns {String|Null}
         */
        this.getDefaultUIParagraphListStylesheet = function () {
            var styleNames = paragraphStyles.getStyleSheetNames(),
                paragraphListId = null;

            _(styleNames).find(function (name, id) {
                var lowerName = name.toLowerCase();
                if (lowerName.indexOf('list paragraph') === 0 || lowerName.indexOf('listparagraph') === 0) {
                    paragraphListId = id;
                    return true;
                }
            });

            return paragraphListId;
        };

        /**
         * Returns an 'insertText' operation from the passed action, if it
         * exists and is the only operation in the action.
         *
         * @param {Object} action
         *  A document action with an operations array.
         *
         * @returns {Object|Null}
         *  An 'insertText' operation from the passed action, if it exists and
         *  is the only operation in the action; otherwise null.
         */
        function getSingleInsertTextOperation(action) {
            var singleOp = (_.isObject(action) && _.isArray(action.operations) && (action.operations.length === 1)) ? action.operations[0] : null;
            return (singleOp && (singleOp.name === Operations.TEXT_INSERT)) ? singleOp : null;
        }

        /**
         * Optimizes the actions before they will be sent to the server.
         * Especially consecutive 'insertText' operations can be merged. This
         * function is a callback that is started from 'sendActions' in the
         * application.
         *
         * @param {Array<Object>} actions
         *  An array with all actions to be optimized.
         *
         * @returns {Array}
         *  An array with optimized actions.
         */
        this.optimizeActions = function (actions) {

            // the resulting merged actions
            var newActions = [];

            // process all passed actions
            actions.forEach(function (nextAction) {

                // the 'insertText' operations to be merged (either variable may become null)
                var lastTextOp = getSingleInsertTextOperation(_.last(newActions));
                var nextTextOp = getSingleInsertTextOperation(nextAction);

                // check that the operations are valid for merging:
                // - both operations must refer to the same paragraph in the same document target
                // - next operation has no attributes and the previous operation is not a change track operation,
                //      or the following two operations both are change track operations
                if (lastTextOp && nextTextOp && operationsHaveSameParentComponent(lastTextOp, nextTextOp) && Utils.canMergeNeighboringOperations(lastTextOp, nextTextOp)) {

                    // the text contents of the last and next operation
                    var lastText = lastTextOp.text;
                    var nextText = nextTextOp.text;

                    // the distance between start character indexes of the last and next operation
                    var distance = _.last(nextTextOp.start) - _.last(lastTextOp.start);

                    // new text can be inserted between existing text (splice new text into the preceding text),
                    // this includes the special case that the new text will be appended to the preceding text
                    if ((distance >= 0) && (distance <= lastText.length)) {

                        // splice the new text into the existing text
                        lastTextOp.text = lastText.slice(0, distance) + nextText + lastText.slice(distance);

                        // increasing the operation length
                        if (lastTextOp.opl && nextTextOp.opl) {
                            lastTextOp.opl += nextTextOp.opl;
                        }

                        // do not add the new action to the result array
                        return;
                    }
                }

                newActions.push(nextAction);
            });

            return newActions;
        };

        /**
         * Getting the global property 'requiresElementFormattingUpdate'.
         *
         * @returns {Boolean}
         *  Returning the value of the global property 'requiresElementFormattingUpdate'.
         */
        this.requiresElementFormattingUpdate = function () {
            return requiresElementFormattingUpdate;
        };

        /**
         * Setting the global property 'requiresElementFormattingUpdate'.
         *
         * @param {Boolean} state
         *  Whether an update of element formatting is required.
         */
        this.setRequiresElementFormattingUpdate = function (state) {
            requiresElementFormattingUpdate = state;
        };

        /**
         * Getting the global property 'guiTriggeredOperation'.
         *
         * @returns {Boolean}
         *  Returning the value of the global property 'guiTriggeredOperation'.
         */
        this.isGUITriggeredOperation = function () {
            return guiTriggeredOperation;
        };

        /**
         * Setting the global property 'guiTriggeredOperation'.
         *
         * @param {Boolean} state
         *  Whether the running process was triggered locally via GUI.
         */
        this.setGUITriggeredOperation = function (state) {
            guiTriggeredOperation = state;
        };

        /**
         * returns the preselected attributed, needed for external call of "interText"
         */
        this.getPreselectedAttributes = function () {
            return preselectedAttributes;
        };

        /**
         * Returns wheter editor is in page breaks mode, or without them.
         *
         * @returns {Boolean}
         *
         */
        this.isPageBreakMode = function () {
            return pbState;
        };

        /**
         * Returns whether editor is in draft mode (limited functionality mode for small devices).
         *
         * TODO: The value of 'draftModeState' is not handled by small devices! Small devices use
         *       draft mode, but this is forced and not handled by the state 'draftModeState'. This
         *       state is only set, when the user selects 'View in draft mode' from the 'View' top
         *       bar pane. For small devices a further check of 'Utils.SMALL_DEVICE' is typically
         *       required. The latter should be made superfluous in the future, so that 'draftModeState'
         *       is also set correctly on small devices.
         *
         * @returns {Boolean}
         */
        this.isDraftMode = function () {
            return draftModeState;
        };

        /**
         * Method that toggles and simulates Draftmode in web browser.
         *
         * @param {Boolean} state
         *  New state. If false, toggles draftmode off.
         *
         * @returns {Editor}
         *  A reference to this instance.
         *
         */
        this.toggleDraftMode = function (state) {
            var pageContentNode = DOM.getPageContentNode(editdiv);

            if (draftModeState === state) { return this; }
            draftModeState = state;
            if (draftModeState) {
                //clear zoom level that might interfere with normal mode
                app.getView().increaseZoomLevel(100, true);
                app.getView().getContentRootNode().addClass('draft-mode');
                this.togglePageBreakMode(false);
                this.toggleForceInlineDrawings(false);
            } else {
                //return drawings style to float
                app.getView().getContentRootNode().removeClass('draft-mode');
                //clear zoom level from draft mode, which is different from zoom in normal mode
                Utils.setCssAttributeWithPrefixes(pageContentNode, 'transform', 'scale(1)');
                Utils.setCssAttributeWithPrefixes(pageContentNode, 'transform-origin', '');
                pageContentNode.css({ width: '100%', backgroundColor: '' });
                app.getView().increaseZoomLevel(100);
                this.togglePageBreakMode(true);
                this.toggleForceInlineDrawings(true);
            }
            return this;
        };

        /**
         * Turn page breaks on or off. Removes them from document, and makes cleanup.
         *
         * @param {Boolean} state
         *   New state. If false, toggles the page aligned drawings to inline drawings.
         *
         * @returns {Editor}
         *   A reference to this instance.
         */
        this.toggleForceInlineDrawings = function (state) {

            if (state) { // leaving draft-mode
                // update formatting of all drawings that have the marker that they are forced to be inline (the page aligned drawings)
                // and all other absolute positioned drawings (the paragraph aligned drawings)
                var formattingNodes = DOM.getPageContentNode(editdiv).find(DrawingFrame.FORCE_INLINE_SELECTOR + ',' + DOM.ABSOLUTE_DRAWING_SELECTOR);

                // updating all collected drawings
                _.each(formattingNodes, function (drawing) { drawingStyles.updateElementFormatting(drawing); });
                self.trigger('update:absoluteElements');  // updating absolutely positioned drawings
            } else { // entering draft mode
                // updating all collected drawings in drawing layer (page aligned drawings)
                _.each(drawingLayer.getDrawings(), function (drawing) {
                    drawingStyles.updateElementFormatting(drawing);
                    // problem sometimes occurs with 'position: absolute' at image node
                    if (DrawingFrame.getDrawingType(drawing) === 'image') { $(drawing).find('img').css('position', ''); }
                });

                // also removing space maker nodes. These come from paragraph aligned drawings that are forced to be inline via CSS.
                DOM.getPageContentNode(editdiv).find(DOM.DRAWING_SPACEMAKER_NODE_SELECTOR).remove();
            }

            return this;
        };

        /**
         * Turn page breaks on or off. Removes them from document, and makes cleanup.
         *
         * @param {Boolean} state
         *   New state. If false, toggles page breaks off.
         *
         * @returns {Editor}
         *   A reference to this instance.
         */
        this.togglePageBreakMode = function (state) {
            var
                splittedParagraphs,
                pageContentNode = DOM.getPageContentNode(editdiv),
                spans;

            if (pbState === state) { return this; }
            pbState = state;

            if (!pbState) {
                pageContentNode.find('.page-break').remove();
                pageContentNode.find('.pb-row').remove();
                pageContentNode.find('.tb-split-nb').removeClass('tb-split-nb');
                pageContentNode.find('.break-above').removeClass('break-above');
                pageContentNode.find('.last-on-page').removeClass('last-on-page');
                editdiv.children('.header-wrapper, .footer-wrapper').addClass('hiddenVisibility');

                splittedParagraphs = pageContentNode.find('.contains-pagebreak');
                if (splittedParagraphs.length > 0) {
                    _.each(splittedParagraphs, function (splittedItem) {
                        spans = $(splittedItem).find('.break-above-span').removeClass('break-above-span');
                        $(splittedItem).children('.break-above-drawing').removeClass('break-above-drawing');
                        validateParagraphNode(splittedItem);
                        if (spans.length > 0) {
                            _.each(spans, function (span) {
                                Utils.mergeSiblingTextSpans(span);
                                Utils.mergeSiblingTextSpans(span, true);
                            });
                        }
                        $(splittedItem).removeClass('contains-pagebreak');
                    });
                }
                pageContentNode.css({ paddingBottom: '' });
            } else {
                editdiv.children('.header-wrapper, .footer-wrapper').removeClass('hiddenVisibility');
                pageLayout.callInitialPageBreaks();
            }
            app.getView().recalculateDocumentMargin();
            return this;
        };

        /**
         * Setter for global state - if headers and footers are currently edited
         *
         *  @param {Boolean} state
         */
        this.setHeaderFooterEditState = function (state) {
            if (headerFooterEditState !== state) {
                headerFooterEditState = state;
            }
        };

        /**
         * Getter for global state of editing or not headers and footers.
         *
         * @returns {Boolean}
         *  Global state of editing or not headers and footer.
         */
        this.isHeaderFooterEditState = function () {
            return headerFooterEditState;
        };

        /**
         * Checks if rendering of page breaks layout should be skipped or not.
         *
         * @returns {Boolean}
         *  Whether should rendering of the page breaks be skipped or not.
         */
        this.skipPageLayoutRendering = function () {
            return !self.isPageBreakMode() || Utils.SMALL_DEVICE || self.isDraftMode() || self.isTargetFunctionality();
        };

        /**
         * Setter for header/footer container root node
         *
         *  @param {jQuery} $node
         */
        this.setHeaderFooterRootNode = function ($node) {
            $headFootRootNode = $node;
            rootNodeIndex = editdiv.find('.header, .footer').filter('[data-container-id="' + $headFootRootNode.attr('data-container-id') + '"]').index($headFootRootNode);
            $parentOfHeadFootRootNode = $headFootRootNode.parent();
            selection.setNewRootNode($node);
        };

        /**
         * Getter for the parent node of the header/footer container root node
         *
         * @returns {jQuery}
         *  The parent node of the header/footer root node
         */
        this.getParentOfHeaderFooterRootNode = function () {
            return $parentOfHeadFootRootNode;
        };

        /**
         * Getter for header/footer container root node
         *
         * @param {String} [target]
         *  if exists, find node by target id
         *
         * @returns {jQuery}
         */
        this.getHeaderFooterRootNode = function (target) {
            // if node is detached from DOM, fetch from parent reference in DOM
            if (target || !$headFootRootNode.parent().parent().length) {
                // if it comes from operation, it will have target argument
                if (target) {
                    if (target === $headFootRootNode.attr('data-container-id')) { // if target is same as cached node reference, use this node
                        if (self.isHeaderFooterEditState()) {
                            return $headFootRootNode;
                        }
                        $headFootRootNode = editdiv.find('.header, .footer').filter('[data-container-id="' + target + '"]').eq(rootNodeIndex);
                    } else { // otherwise use first node with that target

                        // leaving current edit state, because another header/footer is selected
                        pageLayout.leaveHeaderFooterEditMode($headFootRootNode, $headFootRootNode.parent());

                        // finding new header/footer
                        $headFootRootNode = editdiv.find('.header, .footer').filter('[data-container-id="' + target + '"]').eq(0);
                    }
                } else {
                    if (app.isEditable()) {
                        if (!self.isHeaderFooterEditState()) {
                            Utils.warn('Editor.getHeaderFooterRootNode failed to fetch valid node!');
                            return editdiv;
                        }
                        $headFootRootNode = editdiv.find('.header, .footer').filter('[data-container-id="' + $headFootRootNode.attr('data-container-id') + '"]').eq(rootNodeIndex);
                    } else {
                        // remote clients need to fetch template node
                        $headFootRootNode = pageLayout.getHeaderFooterPlaceHolder().children('[data-container-id="' + target + '"]').first();
                    }
                }
                if (!$headFootRootNode.length) {
                    Utils.warn('Editor.getHeaderFooterRootNode not found!');
                    return editdiv;
                }

                pageLayout.enterHeaderFooterEditMode($headFootRootNode);

                //selection.resetSelection();
                selection.setTextSelection(selection.getFirstDocumentPosition());
            }

            return $headFootRootNode;
        };

        /**
         * Getter method for currently active root container node.
         *
         * @param {String} [target]
         *  ID of root container node
         *
         * @returns {jQuery}
         *  Found node
         */
        this.getCurrentRootNode = function (target) {
            // target for operation - if exists, it's for ex. header or footer
            var currTarget = target || self.getActiveTarget();

            // First checking, if the target is a target for a comment. Then this is handled by the comment layer.
            if (currTarget && commentLayer.getCommentRootNode(currTarget)) { return commentLayer.getCommentRootNode(currTarget); }

            // container root node of header/footer
            return currTarget ? (self.isImportFinished() ? self.getHeaderFooterRootNode(target) : pageLayout.getHeaderFooterPlaceHolder().children('[data-container-id="' + target + '"]')) : editdiv;
        };

        /**
         * Getter method for root container node with passed target,
         * or if target is not passed, default editdiv node.
         *
         * @param {String} [target]
         *  ID of root container node
         *
         * @param {Number} [index]
         *  Cardinal number of target node in the document from top to bottom.
         *
         * @returns {jQuery}
         *  Found node
         */
        this.getRootNode = function (target, index) {

            if (!target) { return editdiv; }

            if (commentLayer.getCommentRootNode(target)) {
                return commentLayer.getCommentRootNode(target);
            }

            // headers&footers
            return pageLayout.getRootNodeTypeHeaderFooter(target, index);
        };

        /**
         * Public method interface for private function extendPropertiesWithTarget.
         *
         * @param {Object} object
         *   object that is being extended
         *
         * @param {Array<String>} target
         *   Id referencing to header or footer node
         */
        this.extendPropertiesWithTarget = function (object, target) {
            return extendPropertiesWithTarget(object, target);
        };

        /**
         * This function is also available as private function. Check is
         * necessary, if this really must be a public function. Using
         * revealing pattern here.
         *
         * Prepares the text span at the specified logical position for
         * insertion of a new text component or character. Splits the text span
         * at the position, if splitting is required. Always splits the span,
         * if the position points between two characters of the span.
         * Additionally splits the span, if there is no previous sibling text
         * span while the position points to the beginning of the span, or if
         * there is no next text span while the position points to the end of
         * the span.
         *
         * @param {Number[]} position
         *  The logical text position.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.isInsertText=false]
         *      If set to true, this function was called from implInsertText.
         *  @param {Boolean} [options.useCache=false]
         *      If set to true, the paragraph element saved in the selection can
         *      be reused.
         *  @param {Boolean} [options.allowDrawingGroup=false]
         *      If set to true, the element can also be inserted into a drawing
         *      frame of type 'group'.
         *
         * @returns {HTMLSpanElement|Null}
         *  The text span that precedes the passed offset. Will be the leading
         *  part of the original text span addressed by the passed position, if
         *  it has been split, or the previous sibling text span, if the passed
         *  position points to the beginning of the span, or the entire text
         *  span, if the passed position points to the end of a text span and
         *  there is a following text span available or if there is no following
         *  sibling at all.
         *  Returns null, if the passed logical position is invalid.
         */
        this.prepareTextSpanForInsertion = function (position, options, target) {
            return prepareTextSpanForInsertion(position, options, target);
        };

        /**
         * Method to check whether page break calculation are blocked and returned from it.
         *
         * @returns {Boolean}
         *  True, if calculation should stop running.
         */
        this.checkQuitFromPageBreaks = function () {
            return quitFromPageBreak;
        };

        /**
         * Set block on running loops inside page breaks calculation, and immediately returns from it.
         * It is called from functions that have bigger priority than page breaks: such as text insert and delete.
         *
         * @param {Boolean} state
         *  New state
         */
        this.setQuitFromPageBreaks = function (state) {
            quitFromPageBreak = state;
        };

        /**
         * Check if debounced method should skip insertPageBreaks function, but run other piggybacked, like updateAbsolutePositionedDrawings.
         * This is different from checkQuitFromPageBreaks, as it block page breaks inside debounced method.
         *
         * @returns {Boolean}
         *  True, if debounced method should skip call of insertPageBreaks.
         */
        this.getBlockOnInsertPageBreaks = function () {
            return blockOnInsertPageBreaks;
        };

        /**
         * Sets block on pageLayout.insertPageBreak function, that is evaluated inside insertPageBreaksDebounced.
         * This is different from setQuitFromPageBreaks, as it block page breaks inside debounced method,
         * not inside insertPageBreaks function.
         *
         * @param {Boolean} state
         *  New state
         */
        this.setBlockOnInsertPageBreaks = function (state) {
            if (self.isImportFinished()) {
                blockOnInsertPageBreaks = state;
            }
        };

        /**
         * Getter to fetch global variable node currentProcessingNode,
         * which is set during direct call of inserting page breaks, registerPageBreaksInsert.
         *
         * @returns {jQuery|Node}
         */
        this.getCurrentProcessingNode = function () {
            var firstInCollection;
            var firstIndex;
            var lastInCollection;
            var lastIndex;
            var currentProcessingNode = null;

            // group of lot of operations, e.g. setAttributes,
            // we need to start pagination not from last, but from first (higher in DOM, with lower index)
            if (currentProcessingNodeCollection.length) {
                firstInCollection = currentProcessingNodeCollection[0];
                lastInCollection = _.last(currentProcessingNodeCollection);
                firstIndex = $(firstInCollection).index();
                lastIndex = $(lastInCollection).index();
                if (firstIndex > -1 && lastIndex > -1) { // both elements are in DOM
                    currentProcessingNode = lastIndex < firstIndex ? lastInCollection : firstInCollection;
                } else {
                    if (firstIndex < 0 && lastIndex > -1) { // first elem is not in DOM & last is
                        currentProcessingNode = lastInCollection;
                    }
                    if (lastIndex < 0 && firstIndex > -1) { // last el is not in DOM & first is
                        currentProcessingNode = firstInCollection;
                    }
                }
            }

            return currentProcessingNode;
        };

        /**
         * Public method to get access to debounced insertPageBreaks call.
         *
         * @param {jQuery|Node} [node]
         *  Node from where to start page breaks calculation (top-down direction).
         *
         * @param {jQuery} [textFrameNode]
         *  An optional text frame node.
         */
        this.insertPageBreaks = function (node, textframe) {
            insertPageBreaksDebounced(node, textframe);
        };

        /**
         * Public method to get access to debounced updateLists call.
         *
         * @param {Object} options
         *  The options supported by 'registerListUpdate'.
         *
         * @param {Object} additionalOptions
         *  Some additional application specific options supported by
         *  'registerListUpdate'.
         */
        this.updateListsDebounced = function (options, additionalOptions) {
            return updateListsDebounced(options, additionalOptions);
        };

        /**
         * Getter for the function 'validateParagraphNode'.
         *
         * @returns {Function}
         *  The function 'validateParagraphNode'.
         */
        this.getValidateParagraphNode = function () {
            return validateParagraphNode;
        };

        /**
         * Getter for global variable containing target of operation (header or footer target id).
         *
         * @returns {String}
         *  activeTarget
         */
        this.getActiveTarget = function () {
            return activeTarget;
        };

        /**
         * Setter for target of operatioprn (header or footer target id).
         *
         * @param {String} value
         *  sets passed value to activeTarget
         */
        this.setActiveTarget = function (value) {
            activeTarget = value || '';
        };

        /**
         * Getter for the keyboard block property.
         */
        this.getBlockKeyboardEvent = function () {
            return blockKeyboardEvent;
        };

        /**
         * Setting a blocker for the incoming keyboard events. This is necessary
         * for long running processes.
         */
        this.setBlockKeyboardEvent = function (value) {
            blockKeyboardEvent = value;
        };

        /**
         * public listeners for IPAD workaround:
         * so we can unlisten all function on editdiv
         * and assign them new on editdivs parent
         */
        this.getListenerList = function () {
            return listenerList;
        };

        /**
         * Check, whether the start position of the current selection is inside
         * the specified (paragraph) node. In this case the start position has
         * to start with the logical position of the node.
         *
         * @param {Node|jQuery} [node]
         *  The DOM node to be checked. If this object is a jQuery collection,
         *  uses the first DOM node it contains.
         *
         * @returns {Boolean}
         *  Whether the start of the selection is located inside the specified
         *  node.
         */
        this.selectionInNode = function (node) {

            var // the node corresponding to the current selection
                selectionPoint = Position.getDOMPosition(self.getCurrentRootNode(), selection.getStartPosition()),
                // whether the selection is inside the specified node
                selectionInsideNode = false;

            if (selectionPoint && selectionPoint.node) {
                selectionInsideNode = Utils.containsNode(node, selectionPoint.node, { allowEqual: true });
            }

            return selectionInsideNode;
        };

        /**
         * Helper function that can be called, if a new valid text position is required. This
         * might be the case after some content is deleted, for example after accepting or
         * rejecting a change track.
         * Before setting a new text selection it is checked, if the current selection is
         * invalid.
         */
        this.setValidSelectionIfRequired = function () {

            var validPos = 0;

            if (!Position.isValidTextPosition(self.getCurrentRootNode(), selection.getStartPosition())) {
                validPos = Position.findValidSelection(self.getCurrentRootNode(), selection.getStartPosition(), selection.getEndPosition(), selection.getLastDocumentPosition(), self.useSlideMode());
                selection.setTextSelection(validPos.start, validPos.end);
            }
        };

        /**
         * Public method to access updateEditingHeaderFooterDebounced function outside of Editor class.
         */
        this.updateEditingHeaderFooterDebounced = function (node) {
            return updateEditingHeaderFooterDebounced(node);
        };

        /**
         * Public method to access updateRepeatedTableRowsDebounced function outside of Editor class.
         */
        this.updateRepeatedTableRowsDebounced = function (node) {
            return updateRepeatedTableRowsDebounced(node);
        };

        /**
         * Unique collection for updating of marginal nodes is emptied, which leads to canceling of debounced update.
         */
        this.cancelDebouncedUpdatingHeaders = function () {
            targetNodesForUpdate = $();
        };

        /**
         * Public method to check if there is pending debounced call to header update.
         */
        this.isPendingDebouncedHeaderUpdate = function () {
            return targetNodesForUpdate.length > 0;
        };

        /**
         * Public method to access searchHandler outside of Editor class.
         */
        this.getSearchHandler = function () {
            return searchHandler;
        };

        /**
         * Public method to access spellchecker outside of Editor class.
         */
        this.getSpellChecker = function () {
            return spellChecker;
        };

        /**
         * Updating the models for drawings in drawing layer, comments, complex fields or range
         * markers. If a node like a paragraph, a table row, a table cell, a complete table or a
         * header or footer is removed, it is necessary to check, if the collections of drawing
         * layer, comments, range markers or complex fields are affected.
         * Additionally the drawings and comments inside their layers need to be removed, if a
         * placeholder is removed.
         *
         * @param {HTMLElement|jQuery} parentNode
         *  The node (paragraph, table, row, cell, ...) that will be checked for specific elements
         *  that are stored in models.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.marginal=false]
         *      If set to true, this updating of models was triggered by a change of header or footer.
         */
        this.updateCollectionModels = function (parentNode, options) {

            var // whehter this is a marginal model update
                marginal = Utils.getBooleanOption(options, 'marginal', false);

            if (self.useSlideMode()) {
                if (drawingStyles.deletePlaceHolderAttributes) { drawingStyles.deletePlaceHolderAttributes(parentNode); }
            }

            // checking, if the paragraph contains comment place holder nodes. In this case the comment
            // needs to be removed from comment layer, too.
            if (!marginal || app.isODF()) { commentLayer.removeAllInsertedCommentsFromCommentLayer(parentNode); }

            // checking, if the paragraph contains drawing place holder nodes. In this case the drawings
            // needs to be removed from drawing layer, too.
            if (!marginal) { drawingLayer.removeAllInsertedDrawingsFromDrawingLayer(parentNode); }

            // checking, if the paragraph contains additional range marker nodes, that need to be removed, too.
            rangeMarker.removeAllInsertedRangeMarker(parentNode);

            // checking, if the paragraph contains additional field nodes, that need to be removed.
            fieldManager.removeAllFieldsInNode(parentNode);

            // updating the slide model
            if (DOM.isSlideNode(parentNode)) { self.removeFromSlideModel(DOM.getTargetContainerId(parentNode)); }
        };

        // public handler function for updating change tracks debounced.
        // This handler invalidates the existing side bar. This needs to be done
        // after operations or after 'refresh:layout' event.
        this.updateChangeTracksDebounced = $.noop;

        // public handler function for updating change tracks debounced.
        // This handler does not invalidate the existing side bar. This can happen
        // for example after 'scroll' events.
        this.updateChangeTracksDebouncedScroll = $.noop;

        /**
         * Adding an paragraph object to the 'paragraphQueueForColoredMarginAndBorder'.
         * This store contains paragraphs, for which the background color
         * and border formatting needs to be checked.
         *
         * @param {jQuery} paragraph
         *  A paragraph that needs to be updated for correct color and border formatting.
         *
         * @param {Object} [options.mergedAttributes=null]
         *  A map of attribute maps (name/value pairs), keyed by attribute
         *  family, containing the effective attribute values merged from style
         *  sheets and explicit attributes.
         */
        this.addParagraphToStore = function (paragraph, mergedAttributes) {

            // add to queue (set attributes to null when not specified)
            paragraphQueueForColoredMarginAndBorder.push({ paragraph: paragraph, mergedAttributes: mergedAttributes || null });
        };

        /**
         * Starting to format all paragraphs that are in 'paragraphQueueForColoredMarginAndBorder'.
         * The queue is cleared afterwards.
         *
         * Note: There are two cases: 1) the queue was filled at undoRedo, 2) the queue was filled document loading
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.filledAtUndoRedo=false]
         *      Whether the 'paragraphQueueForColoredMarginAndBorder' was filled at undoRedo.
         */
        this.startParagraphMarginColorAndBorderFormating = function (options) {

            var undo = Utils.getBooleanOption(options, 'filledAtUndoRedo', false);

            paragraphQueueForColoredMarginAndBorder.forEach(function (paraObj) {
                // when the queue was filled at undoRedo use this
                if (undo) {
                    self.implParagraphChanged(paraObj.paragraph);
                //... at document loading use this
                } else {
                    paragraphStyles.applyElementFormatting(paraObj.paragraph, paraObj.mergedAttributes, { duringImport: true });
                }
            });

            // clear the queue
            paragraphQueueForColoredMarginAndBorder = [];
        };

        /**
         * Create the operations for the new page attributes.
         * @param {Object} pageAttrs the attributes to set
         * @param {Object} initialPageAttrs the initial page attributes
         */
        this.setPaperFormat = function (pageAttrs, initialPageAttrs) {

            var newPageAttrs = {};
            _.each(pageAttrs, function (value, key) {
                if (value !== initialPageAttrs[key]) {
                    newPageAttrs[key] = _.isNumber(value) ? Utils.round(value, 10) : value;
                }
            });

            if (!_.isEmpty(newPageAttrs)) {
                if (self.useSlideMode()) {

                    return self.executeDelayed(function () {
                        this.reformatDocument(newPageAttrs);
                    }, 'Editor.setPaperFormat');

                } else {
                    return self.createAndApplyOperations(function (generator) {

                        generator.generateOperation(Operations.SET_DOCUMENT_ATTRIBUTES, { attrs: { page: newPageAttrs } });

                        var paraStyles = self.getStyleCollection('paragraph');

                        function createTabStopOp(styleId) {

                            if (paraStyles.isDirty(styleId)) { return; }

                            var styleAttrs = paraStyles.getStyleSheetAttributeMap(styleId);
                            if (styleAttrs.paragraph && styleAttrs.paragraph.tabStops) {

                                var oldWidth = initialPageAttrs.width - initialPageAttrs.marginLeft - initialPageAttrs.marginRight;
                                var newWidth = pageAttrs.width - pageAttrs.marginLeft - pageAttrs.marginRight;

                                _.each(styleAttrs.paragraph.tabStops, function (tabStop) {
                                    var relPos = tabStop.pos / oldWidth;
                                    tabStop.pos = Math.round(relPos * newWidth);
                                });

                                generator.generateOperation(Operations.CHANGE_STYLESHEET, { styleId: styleId, attrs: styleAttrs, type: 'paragraph' });
                            }
                        }
                        createTabStopOp(self.getDefaultHeaderTextDefinition().styleId);
                        createTabStopOp(self.getDefaultFooterTextDefinition().styleId);
                    });
                }

            }

            return $.when();
        };

        // ==================================================================
        // END of Editor API
        // ==================================================================

        // ==================================================================
        // Private functions
        // ==================================================================

        /**
         * Checks stored character styles of a document and adds 'missing'
         * character styles (e.g. hyperlink).
         */
        function insertMissingCharacterStyles() {
            var styleNames = characterStyles.getStyleSheetNames(),
                parentId = characterStyles.getDefaultStyleId(),
                hyperlinkMissing = true;

            _(styleNames).find(function (name) {
                var lowerName = name.toLowerCase();
                if (lowerName.indexOf('hyperlink') >= 0) {
                    hyperlinkMissing = false;
                    return true;
                }
                return false;
            });

            if (hyperlinkMissing) {
                var hyperlinkAttr = { character: self.getDefaultLateralHyperlinkAttributes() },
                    hyperlinkDef = self.getDefaultLateralHyperlinkDefinition();
                characterStyles.createDirtyStyleSheet(hyperlinkDef.styleId, hyperlinkDef.styleName, hyperlinkAttr, { parent: parentId, priority: hyperlinkDef.uiPriority });
            }
        }

        /**
         * Check the stored paragraph styles of a document and adds 'missing'
         * heading / default and other paragraph styles.
         */
        function insertMissingParagraphStyles() {

            var // the number extensions for the supported headings
                headings = [0, 1, 2, 3, 4, 5],
                // the paragraph style names
                styleNames = paragraphStyles.getStyleSheetNames(),
                // the default paragraph style id
                parentId = paragraphStyles.getDefaultStyleId(),
                // whether a default style is already defined after loading
                hasDefaultStyle = _.isString(parentId) && (parentId.length > 0),
                // the id of the paragraph list style
                paragraphListStyleId = null,
                // the id of the paragraph comment style
                commentStyleId = null,
                // the id of the paragraph header style
                headerStyleId = null,
                // the id of the paragraph footer style
                footerStyleId = null,
                // the default style id of a visible paragraph
                visibleParagraphDefaultStyleId = null,
                // the default style id of a hidden paragraph
                hiddenDefaultStyle = false,
                // the default paragraph definition
                defParaDef,
                // the comment paragraph definition
                commentDef = null,
                // the comment paragraph attributes
                commentAttrs = null,
                // the header paragraph definition
                headerDef = null,
                // the footer paragraph definition
                footerDef = null,
                // header/footer style tab position right
                rightTabPos = pageLayout.getPageAttribute('width') - pageLayout.getPageAttribute('marginLeft') - pageLayout.getPageAttribute('marginRight'),
                // header/footer style tab position center
                centerTabPos = rightTabPos / 2,
                // the header paragraph attributes
                headerAttrs = null,
                // the footer paragraph attributes
                footerAttrs = null;

            if (!hasDefaultStyle) {
                // add a missing default paragraph style
                defParaDef = self.getDefaultParagraphStyleDefinition();
                paragraphStyles.createDirtyStyleSheet(defParaDef.styleId, defParaDef.styleName, {}, { priority: 1, defStyle: defParaDef.default });
                parentId = defParaDef.styleId;
                visibleParagraphDefaultStyleId = parentId;
            } else {
                hiddenDefaultStyle = paragraphStyles.isHidden(parentId);
                if (!hiddenDefaultStyle) {
                    visibleParagraphDefaultStyleId = parentId;
                }
            }

            // Search through all paragraph styles to find out what styles
            // are missing.
            _(styleNames).each(function (name, id) {
                var styleAttributes = paragraphStyles.getStyleAttributeSet(id).paragraph,
                    outlineLvl = styleAttributes.outlineLevel,
                    lowerName = name.toLowerCase(),
                    lowerId = id.toLowerCase();

                if (_.isNumber(outlineLvl) && (outlineLvl >= 0 && outlineLvl < 6)) {
                    headings = _(headings).without(outlineLvl);
                }

                if (!paragraphListStyleId) {
                    if (lowerName.indexOf('list paragraph') === 0 || lowerName.indexOf('listparagraph') === 0) {
                        paragraphListStyleId = id;
                    }
                }

                if (!commentStyleId) {
                    if (lowerName.indexOf('annotation text') === 0) {
                        commentStyleId = id;
                    }
                }

                if (!headerStyleId) {
                    if (lowerName.indexOf('header') === 0) {
                        headerStyleId = id;
                    }
                }

                if (!footerStyleId) {
                    if (lowerName.indexOf('footer') === 0) {
                        footerStyleId = id;
                    }
                }

                // Check and store the headings style which is used
                // by odf documents as seed for other heading styles.
                if (hiddenDefaultStyle && !visibleParagraphDefaultStyleId) {
                    if (lowerId === 'standard' || lowerId === 'normal') {
                        visibleParagraphDefaultStyleId = id || parentId;
                    }
                }
            });

            // add the missing paragraph heading styles using predefined values
            if (headings.length > 0) {
                var defaultCharStyles = self.getDefaultHeadingCharacterStyles(),
                    headingsParentId = hiddenDefaultStyle ? visibleParagraphDefaultStyleId : parentId,
                    headingsNextId = hiddenDefaultStyle ? visibleParagraphDefaultStyleId : parentId;

                _(headings).each(function (level) {
                    var attr = {},
                        charAttr = defaultCharStyles[level];
                    attr.character = charAttr;
                    attr.paragraph = { outlineLevel: level, nextStyleId: headingsNextId };
                    paragraphStyles.createDirtyStyleSheet('heading' + (level + 1), 'heading ' + (level + 1), attr, { parent: headingsParentId, priority: 9 });
                });
            }

            // add missing paragraph list style
            if (!paragraphListStyleId) {
                paragraphStyles.createDirtyStyleSheet(
                    'ListParagraph',
                    'List Paragraph',
                    { paragraph: { indentLeft: 1270, contextualSpacing: true, nextStyleId: 'ListParagraph' } },
                    { parent: hiddenDefaultStyle ? (visibleParagraphDefaultStyleId || parentId) : parentId, priority: 34 });
            }

            // add missing comment style
            if (!commentStyleId) {
                commentDef = self.getDefaultCommentTextDefintion();
                commentAttrs = self.getDefaultCommentTextAttributes();
                paragraphStyles.createDirtyStyleSheet(commentDef.styleId, commentDef.styleName, commentAttrs, { parent: commentDef.parent, priority: commentDef.uiPriority, defStyle: commentDef.default });
            }

            // add missing header style
            if (!headerStyleId) {
                headerDef = self.getDefaultHeaderTextDefinition();
                headerAttrs = { paragraph: { tabStops: [{ value: 'center', pos: centerTabPos }, { value: 'right', pos: rightTabPos }] } };
                paragraphStyles.createDirtyStyleSheet(headerDef.styleId, headerDef.styleName, headerAttrs, { parent: headerDef.parent, priority: headerDef.uiPriority, defStyle: headerDef.default });
            }

            // add missing footer style
            if (!footerStyleId) {
                footerDef = self.getDefaultFooterTextDefinition();
                footerAttrs = { paragraph: { tabStops: [{ value: 'center', pos: centerTabPos }, { value: 'right', pos: rightTabPos }] } };
                paragraphStyles.createDirtyStyleSheet(footerDef.styleId, footerDef.styleName, footerAttrs, { parent: footerDef.parent, priority: footerDef.uiPriority, defStyle: footerDef.default });
            }
        }

        /**
         * Check the stored drawing styles of a document and adds 'missing' style. This is
         * currently only used for odt documents, where the insertion of a text frame requires
         * a corresponding drawing style.
         */
        function insertMissingDrawingStyles() {

            var // the names of the drawing styles
                styleNames = drawingStyles.getStyleSheetNames(),
                // the default style Id
                parentId = drawingStyles.getDefaultStyleId(),
                // whether a default style is already defined
                hasDefaultStyle = _.isString(parentId) && (parentId.length > 0),
                // whether the style for the text frames is missing
                textframestyleMissing = true,
                // the drawing style definition
                drawingDef = null,
                // the drawing style attributes
                drawingAttrs = null;

            // adding the default style, if required
            if (!hasDefaultStyle) {
                drawingDef = self.getDefaultDrawingDefintion();
                drawingAttrs = self.getDefaultDrawingAttributes();
                drawingStyles.createDirtyStyleSheet(drawingDef.styleId, drawingDef.styleName, drawingAttrs, { priority: drawingDef.uiPriority, defStyle: drawingDef.default });
                parentId = drawingDef.styleId;
            }

            // checking the style used for text frames
            _(styleNames).find(function (name) {
                var lowerName = name.toLowerCase();
                if (lowerName.indexOf('frame') >= 0) {
                    textframestyleMissing = false;
                    return true;
                }
                return false;
            });

            if (textframestyleMissing) {
                drawingDef = self.getDefaultDrawingTextFrameDefintion();
                drawingAttrs = self.getDefaultDrawingTextFrameAttributes();
                // Info: Setting option hidden to true. Otherwise this style will replace the style 'default_drawing_style', so that for example line colors are modified.
                drawingStyles.createDirtyStyleSheet(drawingDef.styleId, drawingDef.styleId, drawingAttrs, { parent: parentId, priority: drawingDef.uiPriority, defStyle: drawingDef.default, hidden: true });
            }

        }

        // ====================================================================
        // Private functions for the hybrid edit mode
        // ====================================================================

        function processMouseUp(event) {

            var // whether the document is opened in read-only mode
                readOnly = !app.isEditable(),
                // a drawing node and a tab node
                drawing = null, drawingNode = null, tabNode = null, fieldNode = null,
                // logical positions
                startPosition = null, oxoPosition = null, fieldPos = null,
                // whether this event was triggered by a right click
                rightClick = (event.button === 2);

            activeMouseDownEvent = false;

            // in Presentation app on touch devices the edit mode might be activated
            if (app.isPresentationApp() && self.getSlideTouchMode()) {
                self.handleTouchEditMode(event);
                return;
            }

            // make sure that the clickable parts in the hyperlink popup are
            // processed by the browser but we don't call selection.processBrowserEvent
            // as otherwise we have a bad browser selection which produce assertions
            // and have other strange effects (especially for IE)
            if (Hyperlink.isClickableNode($(event.target), readOnly)) {
                return;
            }

            // handle mouse events in edit mode only
            if (readOnly) {
                selection.processBrowserEvent(event);
                return;
            }

            tabNode = $(event.target).closest(DOM.TAB_NODE_SELECTOR);
            if (tabNode.length && selection.getBrowserSelection().active && selection.getBrowserSelection().active.isCollapsed()) { // hasRange cannot be used, as endPosition is still not set
                // tabulator handler to decide if cursor goes before or after tab
                if (tabNode.offset().left + tabNode.width() / 2 < event.clientX) {
                    startPosition = Position.getOxoPosition(self.getCurrentRootNode(), tabNode.next(), 0);
                    selection.setTextSelection(startPosition);
                    return;
                } else {
                    startPosition = Position.getOxoPosition(self.getCurrentRootNode(), tabNode, 0);
                    selection.setTextSelection(startPosition);
                    return;
                }
            }
            fieldNode = $(event.target).closest(DOM.FIELD_NODE_SELECTOR);
            if (!rightClick && fieldNode.length && selection.getSelectionType() === 'text') { // #47584
                startPosition = selection.getStartPosition();
                fieldPos = Position.getOxoPosition(self.getCurrentRootNode(), fieldNode, 0);
                if (Utils.compareNumberArrays(fieldPos, startPosition) < 0) {
                    selection.setTextSelection(fieldPos, startPosition);
                } else {
                    selection.setTextSelection(startPosition, Position.increaseLastIndex(fieldPos));
                }
            }

            if (selection.getSelectionType() === 'drawing') {
                drawingNode = $(event.target).closest(DrawingFrame.NODE_SELECTOR);

                if (rightClick && drawingNode.length === 0) {
                    oxoPosition = Position.getOxoPositionFromPixelPosition(editdiv, event.pageX, event.pageY);
                    selection.setTextSelection(oxoPosition.start);
                } else if (!self.useSlideMode()) {
                    // mouse up while drawing selected: selection does not change,
                    // but scroll drawing completely into the visible area (not for presentation app, 49354)
                    drawing = selection.getSelectedDrawing(); // empty for multiple selection (if supported)
                    if (drawing.length > 0) { app.getView().scrollToChildNode(drawing); }
                }

                if (_.browser.IE < 11 || Utils.IOS) {
                    // in IE (< 11) processBrowserEvent was not called in mouseDown. Therefore
                    // it is now necessary to call it in mouseup, even if drawings are
                    // selected. (Fix for 29382: Excluding IE 11)

                    // on the iPad when deselecting an image,
                    // processBrowserEvent is called in mouseDown, but applyBrowserSelection is not.
                    // so we need to call processBrowserEvent also in mouseUp to finally call applyBrowserSelection.
                    selection.processBrowserEvent(event);
                } else if (_.browser.IE === 11 && !self.useSlideMode()) {
                    // Avoiding windows frame around the drawing by repainting OX Text drawing frame
                    // TODO: Is this still necessary?
                    // -> removing this for text frames, because text selection will be disturbed
                    if (!DrawingFrame.isTextFrameShapeDrawingFrame(drawing)) {
                        DrawingFrame.clearSelection(drawing);
                        selection.drawDrawingSelection(drawing);
                    }
                }

            } else {
                if (rightClick && !selection.hasRange() && !selection.isMultiSelection()) {
                    // context menu when clicked on slide in Safari gets killed by setTextSelection
                    // in Firefox slide jumps down when clicking on slide, because Position.getOxoPosition always returns [0,0]
                    if ((_.browser.Safari || _.browser.Firefox) && DOM.isSlideNode(event.target)) { return; }
                    oxoPosition = Position.getOxoPositionFromPixelPosition(editdiv, event.pageX, event.pageY);
                    if (oxoPosition) { selection.setTextSelection(oxoPosition.start); }
                } else if (rightClick && selection.hasRange() && selection.getSelectionType() === 'text') {
                    // context menu gets killed by focusChangeHandler if processBrowserEvent is called in this case, #49518
                } else {
                    // calculate logical selection from browser selection, after
                    // browser has processed the mouse event
                    selection.processBrowserEvent(event);
                }
            }
        }

        /**
         * Event handler for scrolling tables horizontaly in draft mode
         */
        var processTouchEventsForTableDraftMode = (function () {
            var currentTable, // target table
                startXpos = 0, // initial pageX value on touchstart for calculating event offset
                currentXpos = 0, // temp pageX value got on touchmove
                tableCssLeft = 0,
                pageContentWidth = 0,
                currentTableWidth = 0;

            return function (event) {
                pageContentWidth = editdiv.find(DOM.PAGECONTENT_NODE_SELECTOR).width();
                currentTable = $(event.target).parents(DOM.TABLE_NODE_SELECTOR).last();
                currentTableWidth = currentTable.width();

                switch (event.type) {
                    case 'touchstart':
                        if (event.originalEvent && event.originalEvent.changedTouches) {
                            startXpos = event.originalEvent.changedTouches[0].pageX;
                        }
                        tableCssLeft = parseInt(currentTable.css('left'), 10);
                        break;
                    case 'touchmove':
                        if (event.originalEvent && event.originalEvent.changedTouches) {
                            currentXpos = (event.originalEvent.changedTouches[0].pageX - startXpos);
                        }
                        if (Math.abs(currentXpos) < pageContentWidth / 2) {
                            currentTable.css({ left: tableCssLeft + currentXpos });
                        } else {
                            currentTable.css({ left: (currentXpos > 0) ? 0 : (pageContentWidth - currentTableWidth) });
                        }
                        break;
                    case 'touchend':
                        tableCssLeft = parseInt(currentTable.css('left'), 10);
                        if (tableCssLeft > 0) {
                            currentTable.css({ left: 0 });
                        } else if (tableCssLeft < pageContentWidth - currentTableWidth) {
                            currentTable.css({ left: pageContentWidth - currentTableWidth });
                        }
                        currentXpos = 0;
                        break;
                }
            };
        }());

        function processHeaderFooterEdit(event) {
            if (!app.isEditable() || !self.isImportFinished()) {
                return;
            }
            event.stopPropagation();

            var $currTarget = $(event.currentTarget),
                $refToParentNode = $();

            if ($currTarget.hasClass('cover-overlay')) {
                $currTarget = $currTarget.parents('.header, .footer');
            }

            if (self.isHeaderFooterEditState()) {
                if (self.getHeaderFooterRootNode()[0] === $currTarget[0]) {
                    return;
                } else {
                    pageLayout.leaveHeaderFooterEditMode(self.getHeaderFooterRootNode());

                    // store parent element as a reference (element itself is going to be replaced)
                    $refToParentNode = $currTarget.parent();

                    // must be direct call
                    pageLayout.updateEditingHeaderFooter();
                }
            }
            if (DOM.isHeaderOrFooter($currTarget)) {

                if ($refToParentNode.length > 0) {
                    $currTarget = $refToParentNode.children('[data-container-id="' + $currTarget.attr('data-container-id') + '"]');
                }
                if (!$currTarget.length) {
                    // from edit mode of one h/f we doubleclicked on other, which is not created yet
                    $currTarget = $(event.currentTarget);
                    if ($currTarget.hasClass('cover-overlay')) {
                        $currTarget = $currTarget.parents('.header, .footer');
                    }
                }
                if (DOM.isHeaderOrFooter($currTarget)) {
                    pageLayout.enterHeaderFooterEditMode($currTarget);

                    selection.setNewRootNode($currTarget);
                    selection.setTextSelection(selection.getFirstDocumentPosition());
                }
            } else {
                Utils.warn('Editor.processHeaderFooterEdit(): Failed to fetch header or footer');
            }
        }

        function processMouseUpOnDocument(event) {

            if (activeMouseDownEvent) {
                activeMouseDownEvent = false;
                processMouseUp(event);
            }
        }

        // Registering keypress handler at the document for Chrome
        // to avoid problems with direct keypress to document
        // -> This handler is triggered once without event, therefore
        // the check for the event is required.
        function processKeypressOnDocument(event) {

            // Fix for the following scenario:
            // - local client uses Chrome browser
            // - local client selects drawing, without have edit privileges
            // - local client removes the focus from the Chrome browser
            // - remote client removes the selected drawing
            // - local client set browser focus by clicking on top of browser (or side pane)
            // - local client can type a letter direct with keypress into the document
            //   -> the letter is inserted into the document, no operation, no read-only warning
            // After removing the drawing in the remote client, the focus cannot be set correctly
            // in the local client, because the browser does not have the focus. Therefore the focus
            // remains in the clipboard and is then switched to the body element.

            var readOnly = !app.isEditable();

            // special handling for the following scenario:
            // - there is an event (once this handler is called without event)
            // - the target of the event was the body. So we cannot conflict with events triggered in the side pane, that bubble up to the body.
            // - the document is in read-only mode -> this might be removed in the future, if more than one user is able to edit the document.
            // - this edit application is visible
            // - the event is not triggered a second time -> this event is branded with the property '_triggeredAgain' to avoid endless loop.
            // - the currently focused element is the body element (which is a second test similar to the check of the event target).
            if (event && $(event.target).is('body') && readOnly && app.isActive() && !event._triggeredAgain && $(Utils.getActiveElement()).is('body')) {
                event.preventDefault();
                // Trigger the same event again at the div.page once more (-> read-only warning will be displayed)
                event.target = editdiv[0];
                event._triggeredAgain = true;
                editdiv.trigger(event);
            }
        }

        // Fix for 29751: IE supports double click for word selection
        function processDoubleClick(event) {

            var // the boundaries of the double-clicked word
                wordBoundaries = null;

            if (_.browser.IE) {

                if (DOM.isTextSpan(event.target)) {
                    doubleClickEventWaiting = true;
                    // Executing the code deferred, so that a triple click becomes possible.
                    self.executeDelayed(function () {
                        // modifying the double click to a triple click, if a further mousedown happened
                        if (tripleClickActive) { event.type = 'tripleclick'; }
                        doubleClickEventWaiting = false;
                        tripleClickActive = false;
                        selection.processBrowserEvent(event);
                    }, 'Editor.processDoubleClick', doubleClickTimeOut);

                }
            } else if (_.browser.WebKit) { // 39357
                if (DOM.isTextSpan(event.target) && DOM.isListParagraphNode(event.target.parentNode) && _.last(selection.getStartPosition()) > 0) {
                    wordBoundaries = Position.getWordBoundaries(self.getCurrentRootNode(), selection.getStartPosition(), { addFinalSpaces: true });
                    if (_.last(wordBoundaries[0]) === 0) { selection.setTextSelection.apply(selection, wordBoundaries); }
                }
            }
        }

        /**
         * Handler for the keyUp event.
         *
         * @param {jQuery.Event} event
         *  A jQuery keyboard event object.
         */
        function processKeyUp(event) {
            // to be able to distinguish tap on ipad's suggestion word, which triggers only textInput event,
            // from tap on normal letter from virtual keyboard, clear cached event value on key up event.
            lastKeyDownEvent = null;
            dumpEventObject(event);
        }

        /**
         * Handler for the 'compositionstart' event.
         * The event is dispatched before a text composition system begins a new
         * composition session and before the DOM is modified due to the composition
         * process.
         *
         * @param {jQuery.Event} event
         *  The 'compositionstart' event
         */
        function processCompositionStart(event) {
            //Utils.log('processCompositionStart');

            var // the span to be used for special webkit treatment
                insertSpan,
                // ime state object
                imeState = {},
                // root node
                rootNode = self.getCurrentRootNode(),
                // whether this is a FireFox browser
                isFireFox = _.browser.Firefox,
                // the browser selection
                browserSel = isFireFox ? window.getSelection() : null;

            // helper function to check
            function isInvalidSelectionNode(node) {
                return DOM.isPageNode(node) || DOM.isHeaderOrFooterWrapper(node) || DOM.isHeaderOrFooter(node);
            }

            dumpEventObject(event);
            if (_.browser.Android) { return; }

            // restoring the browser selection in Firefox (version > 49), because the browser selection
            // might have been modified (by the browser) before 'processCompositionStart'.
            if (isFireFox && (isInvalidSelectionNode(browserSel.anchorNode) || isInvalidSelectionNode(browserSel.focusNode))) { selection.restoreBrowserSelection(); }

            // no text input in read-only mode (54729) (fix fails for Firefox)
            if (!app.isEditable() && !isFireFox) {
                selection.setFocusIntoClipboardNode();
                event.preventDefault();
                event.stopPropagation();
            }

            imeUseUpdateText = false;
            if (imeStateQueue.length) {
                //fix for Bug 35543 - 3-set korean on chrome, compositionStart comes directly after compositionEnd,
                //but we have deferred the postProcessCompositionEnd, so we have to abort it and call it directly
                var last = _.last(imeStateQueue);
                var delayed = last.delayed;
                if (delayed) {
                    delayed.abort();
                    postProcessCompositionEnd();
                }
            }

            // delete current selection first
            self.deleteSelected();

            imeActive = true;
            // determin the span where the IME text will be inserted into
            imeState.imeStartPos = selection.getStartPosition();
            insertSpan = $(Position.getLastNodeFromPositionByNodeName(rootNode, imeState.imeStartPos, 'span'));

            if (Utils.IOS && lastKeyDownEvent) {
                //Bug 42457
                //IME start can come on an Asian keyboard in two ways:
                //1. press just a -> IME starts for replace a -> (we get two sign, this here is the workaround for that)
                //2. just press on a Asian sign, all is fine, IME works a expected

                var before = Position.increaseLastIndex(selection.getStartPosition(), -1);
                if (_.last(before) >= 0) {
                    self.applyOperations({ name: Operations.DELETE, start: before, end: before });
                    imeState.imeStartPos = before;
                }
            }

            // if the span is empty webkit browsers replace the content of the paragraph node
            // to avoid this we add a non-breaking space charater here which will be removed in processCompositionEnd()
            // this workaround is not needed for Firefox and IE and would lead to runtime errors
            if (_.browser.WebKit && (insertSpan.text().length === 0)) {
                imeState.imeAdditionalChar = true;
                insertSpan.text('\xa0');
                selection.setBrowserSelection(DOM.Range.createRange(insertSpan, 1, insertSpan, 1));
            } else {
                imeState.imeAdditionalChar = false;
            }

            // Special code for iPad. When we detect a compositionstart we have to
            // remove the last char from the activeInputText. Unfortunately we only
            // know now that the char should be processed by the composition system (IME).
            // Therefore we have to remove this char from the active input text.
            if (Utils.IOS) {
                // check and abort running input text promise
                if (inputTextPromise) {
                    inputTextPromise.abort();
                }
                // remove latest character from active input text
                if (self.getActiveInputText()) {
                    self.setActiveInputText(self.getActiveInputText().slice(0, -1));
                }
            }
            // now store the current state into our queue
            imeStateQueue.push(imeState);
        }

        /**
         * Handler for the 'compositionupdate' event.
         * The event is dispatched during a composition session when a text composition
         * system updates its active text passage with a new character.
         *
         * @param {jQuery.Event} event
         *  The 'compositionupdate' event
         */
        function processCompositionUpdate(event) {

            if (_.browser.Android) { return; }

            //Utils.log('processCompositionUpdate');
            dumpEventObject(event);
            // Fix for 34726: Store the latest update text for later use.
            imeUpdateText = event.originalEvent.data;
            //Utils.log('processCompositionUpdate: imeUpdateText=' + imeUpdateText);
        }

        /**
         * Handler for the 'compositionend' event.
         * The event is dispatched after the text composition system completes or cancels
         * the current composition session (e.g., the IME is closed, minimized, switched
         * out of focus, or otherwise dismissed, and the focus switched back to the user
         * agent).
         * Be careful: There are browsers which violate the w3c specification (e.g. Chrome
         * and Safari) and send the compositionend event NOT when the composition system
         * completes, but at a moment where DOM manipulations are NOT allowed.
         *
         * @param {jQuery.Event} event
         *  The 'compositionend' event
         */
        function processCompositionEnd(event) {
            if (_.browser.Android) { return; }

            dumpEventObject(event);
            //Utils.log('processCompositionEnd');

            var // current ime state
                imeState = _.last(imeStateQueue);

            // store the original composition end event
            imeState.event = event;
            imeState.imeUpdateText = imeUpdateText;

            // #51390#
            // Reset ime active flag to make sure that calling further
            // functions can work correctly (there are now functions that
            // check the IME activity and do nothing if there is a composition).
            imeActive = false;

            // #48733 The Chrome implementation of their composition processing must have
            // changed considerably. Due to the fact that we implement the composition code
            // based on behavior & assumptions this now breaks. At least from Chrome 53 the
            // compositionend event is now sent AFTER the last DOM changes were made. This
            // leads to bad behavior in our code as we use asynchronous processing.
            // We now use synchronous processing and can remove some special code for Mac/Chrome.
            if (_.browser.Safari && _.device('macos')) {
                // For Safari we have to use the old, bad way to delay the process
                // the composition end event. This is also necessary for Windows where
                // some IME implementations send two or more input events which are
                // not always behind the DOM manipulation
                imeState.delayed = self.executeDelayed(postProcessCompositionEnd, 'Editor.postProcessCompositionEnd');
            } else if (Utils.IOS) {
                //Ipad can process 'compositionend' synchronously, but to separate "composition" from "just insert text" we use that delay
                imeState.delayed = self.executeDelayed(postProcessCompositionEnd, 'Editor.postProcessCompositionEnd');
            } else {
                // all other desktop browser can process 'compositionend' synchronously.
                postProcessCompositionEnd();
            }

            imeUseUpdateText = false;
        }

        function postProcessCompositionEnd() {
            //Utils.log('postProcessCompositionEnd');

            var // pull the first state from the queue
                imeState = imeStateQueue.shift(),
                // retrieve the IME text from the compositonend event or last good update event
                // see #49194# for more information
                imeText = imeUseUpdateText ? imeState.imeUpdateText : imeState.event.originalEvent.data,
                // determine the start position stored in the imeState
                startPosition = imeState.imeStartPos ? _.clone(imeState.imeStartPos) : selection.getStartPosition(),
                // the end position of the ime inserted text
                endPosition;

            if (_.browser.Safari && _.device('macos')) {
                // Fix for 34726: This is a workaround for a special behaviour of
                // MacOS X. The keyboard switches to composition, if the user presses
                // the acute accent key. If this key is pressed twice, the composition is started,
                // ended and started again. This collides with the asynchronous processing
                // of the composition end due to broken 'compositionend' notification of
                // webkit/blink based browers. So the asynchronous code is executed while
                // the browser started a new composition. Manipulating the DOM while in
                // composition stops the composition immediately and 'compositionend'
                // provides an empty string in event.originalEvnt.data. Therefore we use
                // the last 'compositionupdate' text to get the correct text.
                imeText = (imeText.length === 0) ? imeState.imeUpdateText : imeText;
            }

            if (imeText.length > 0) {

                // remove the inserted IME text from the DOM
                // and also remove the non-breaking space if it has been added in 'processCompositionStart'
                endPosition = Position.increaseLastIndex(startPosition, imeText.length - (imeState.imeAdditionalChar ? 0 : 1));
                self.implDeleteText(startPosition, endPosition, null, self.getActiveTarget());

                // We have to make sure that we always convert a implicit
                // paragraph. In some cases we have overlapping composition
                // events, which can produce an implicit pararaph with content
                // (inserted by the IME directly). So for this special
                // case we have to omit the length check in handleImplicitParagraph
                handleImplicitParagraph(startPosition, { ignoreLength: true });

                // insert the IME text provided by the event data
                self.insertText(imeText, startPosition, preselectedAttributes);

                // set the text selection after the inserted IME text
                selection.setTextSelection(imeState.imeAdditionalChar ? endPosition : Position.increaseLastIndex(endPosition));

                // during the IME input we prevented calling 'restoreBrowserSelection', so we need to call it now
                selection.restoreBrowserSelection();
            } else if ((imeText.length === 0) && imeState.imeAdditionalChar) {
                // #48775#
                // We have to remove the "special" nbsp, even if we don't
                // have any composition text. Otherwise an additional character is
                // inside the DOM!!
                self.implDeleteText(startPosition, startPosition, null, self.getActiveTarget());
                // We have to make sure that we always convert a implicit
                // paragraph. In some cases we have overlapping composition
                // events, which can produce an implicit pararaph with content
                // (inserted by the IME directly). So for this special
                // case we have to omit the length check in handleImplicitParagraph
                handleImplicitParagraph(startPosition, { ignoreLength: true });
                // set the text selection after the inserted IME text
                selection.setTextSelection(startPosition);
                // during the IME input we prevented calling 'restoreBrowserSelection', so we need to call it now
                selection.restoreBrowserSelection();
            }
        }

        function isImeActive() {
            return imeActive || imeStateQueue.length > 0;
        }

        function processTextInput(event) {
            var aLastKeyDownEvent = self.getLastKeyDownEvent();
            var startPosition;
            // whether the text can be inserted at the specified position
            var validPosition = true;
            //Utils.log('processTextInput');

            dumpEventObject(event);

            // Evaluating MacOS marker for multiple directly following keydown events (46659)
            // This check should only be done when no IME is active (49285)
            if (_.browser.MacOS && _.isNumber(self.getLastKeyDownKeyCode()) && !isImeActive()) {
                if (self.isRepeatedKeyDown()) {
                    // textinput event occurred directly after keydown and keyup without keypress
                    // -> 'lastKeyDownKeyCode' was not deleted in keypress
                    self.setLastKeyDownKeyCode(null);
                    self.setRepeatedKeyDown(false);
                    event.preventDefault();
                    return false;
                } else if ((self.getLastKeyDownKeyCode() === KeyCodes.SPACE) &&
                           (self.getLastKeyDownModifiers().ctrlKey && self.getLastKeyDownModifiers().metaKey)) {
                    self.setLastKeyDownKeyCode(null);
                    self.setRepeatedKeyDown(false);
                    event.preventDefault();
                    return false;
                }
            }

            if (!isImeActive() && _.browser.Chrome &&
                (((aLastKeyDownEvent === null) && _.isString(event.originalEvent.data) && event.originalEvent.data.length === 1) ||
                 (_.isObject(aLastKeyDownEvent) && (aLastKeyDownEvent.keyCode === KeyCodes.IME_INPUT) &&
                  _.isString(event.originalEvent.data) && event.originalEvent.data.length === 1))) {
                // This is again a very bad hacking solution for a broken Chrome behaviour with
                // certain IME-implementations, e.g. on Windows with MS Cangjie & on Linux with
                // SunPinyin & Cangjie 3 or 5.
                // 1.) Chrome provides a keydown-event with keycode 229, but there is no composition-event.
                // Only a textinput-event provides any information about the user-input.
                // 2.) There is only a "textInput" which is FOLLOWED by a input, keydown, keyup
                // event. This is a very bad behaviour and we also try to detect it.
                // Therefore this code tries to detect this magically and to save the
                // situation (#49585, see also Chromium issue: 658141)
                // ATTENTION: This fix must be removed as soon the Chromium test resolved the
                // broken behaviour. So 658141 should be monitored.
                startPosition = selection.getStartPosition();

                // avoid typing into the slide selection ()
                if (self.useSlideMode() && startPosition.length === 2) { validPosition = false; }

                if (validPosition) {
                    self.insertText(event.originalEvent.data, startPosition, preselectedAttributes);
                    selection.setTextSelection(Position.increaseLastIndex(startPosition));
                }

                // prevent default handling which inserts the text into the DOM
                event.preventDefault();
                return false;
            }

            if (Utils.IOS && !isImeActive()) {
                var data = event.originalEvent.data;
                // Special code: The prevent default cancels the insertion of a suggested text.
                // Unfortunately the text is inserted for a short fraction and afterwards replaced
                // by the old text. Due to this process the cursor position is sometimes incorrectly
                // set. Previous code tried to restore the latest old cursor position, which was leading to internal errors or broken roundtrip.
                if (data.length && !lastKeyDownEvent) {
                    // detect that user taped on word suggestion, and not on actuall letter:
                    // if there is only textInput event, and no keyDown and keyUp.
                    // Workaround: instead of trying to set cursor position while element is or is not in the dom,
                    // close the ios keyboard by setting focus to a hidden node.
                    Utils.focusHiddenNode(); // #40185
                } else if (data.length && lastKeyDownEvent.keyCode === 0 && Utils.isHangulText(data)) {
                    // Korean keyboard has no feedback for IME-Mode
                    Utils.focusHiddenNode(); // #42667
                }
                event.preventDefault();
                return false;
            }

            if (_.browser.MacOS && !isImeActive()) {
                // Bug 48829 emojis in macos' IME mode
                var orgData = event.originalEvent.data;
                if (orgData.length && !lastKeyDownEvent) {
                    event.preventDefault();
                    return false;
                }
            }
        }

        function processInput(event) {
            dumpEventObject(event);
        }

        /**
         * Sets a paragraph to contentEditable = false while removing it
         * from an optional previous remembered paragraph.
         *
         * paragraph {HTMLElement|null}
         *  The paragraph where contentEditable should be set to false.
         */
        function setIOSROParagraph(paragraph) {
            if (roIOSParagraph !== null) {
                $(roIOSParagraph).removeAttr('contentEditable');
                roIOSParagraph = null;
            }
            if (paragraph && paragraph.isContentEditable) {
                paragraph.contentEditable = 'false';
                roIOSParagraph = paragraph;
            }
        }

        /**
         * Central place to make changes to all sychnronously applied operations. Before the generated operations
         * are applied to the operation handler, this is the point, where global changes for all operations can be
         * made.
         *
         * @param {Array} operations
         *  An array containing all operations that are investigated to be finalized.
         */
        function finalizeOperations(operations) {

            operations.forEach(function (operation) {

                // adding the target to each operation, if it is not already added to the operation
                // and if it is a non-empty string
                if (operation.name && operation.start && !operation.target && activeTarget && !undoRedoRunning) {
                    operation.target = activeTarget;
                } else if (operation.target && operation.target === self.getOperationTargetBlocker()) {
                    // removing the blocker for the operation target
                    delete operation.target;
                }
            });

            return operations;
        }

        /**
         * Load performance: Collecting selected operations in collector for saving them in local storage. This
         * makes it possible to load documents without any operations or fast load string, because all required
         * data is saved in the local storage.
         *
         * @param {Object} operation
         *  An operation object.
         */
        function saveStorageOperationHandler(operation) {
            return self.saveStorageOperation(operation);
        }

        /**
         * Utility method for extending object with target property,
         * but only if its defined.
         *
         * @param {Object} object
         *  object that is being extended
         *
         * @param {String[]} target
         *  Id referencing to header or footer node
         */
        function extendPropertiesWithTarget(object, target) {
            if (target) {
                object.target = target;
            }
        }

        function initDocument() {

            var // container for the top-level paragraphs
                pageContentNode = DOM.getPageContentNode(editdiv),
                // the initial paragraph node in an empty document
                paragraph = DOM.createImplicitParagraphNode();

            // set style containers
            characterStyles = self.getStyleCollection('character');
            paragraphStyles = self.getStyleCollection('paragraph');
            tableStyles = self.getStyleCollection('table');
            tableRowStyles = self.getStyleCollection('row');
            tableCellStyles = self.getStyleCollection('cell');
            drawingStyles = self.getStyleCollection('drawing');
            slideStyles = self.getStyleCollection('slide');
            pageStyles = self.getStyleCollection('page');

            // create empty page with single paragraph
            pageContentNode.empty().append(paragraph);
            validateParagraphNode(paragraph);

            // in Internet Explorer it is necessary to add new empty text nodes in paragraph again
            // newly, also in other browsers (new jQuery version?)
            repairEmptyTextNodes(paragraph);

            // initialize default page formatting
            pageStyles.updateElementFormatting(editdiv);

            // register keyboard event handlers on the clipboard node to handle key strokes when
            // a drawing is selected in which case the browser selection is inside the clipboard node
            selection.getClipboardNode().on({
                keydown: app.isPresentationApp() ? _.bind(self.getPageProcessKeyDownHandlerClipboardNode(), self) : _.bind(self.getPageProcessKeyDownHandler(), self),
                keypress: app.isPresentationApp() ? _.bind(self.getPageProcessKeyPressHandlerClipboardNode(), self) : _.bind(self.getPageProcessKeyPressHandler(), self),
                cut: _.bind(self.cut, self),
                copy: _.bind(self.copy, self),
                paste: _.bind(self.paste, self)
            });
        }

        /**
         * Called when all initial document operations have been processed.
         * Can be used to start post-processing tasks which need a fully
         * processed document.
         */
        function documentLoaded() {

            /**
             * TODO: Remove this as soon as possible.
             * BUG #50749
             * Only Selenium needs this hack. Otherwise it is not possible to click in a already selected table (or textframe or shape).
             */
            if (Config.AUTOTEST && _.browser.Firefox && _.browser.Firefox < 50) {
                var pageContentNode = DOM.getPageContentNode(editdiv);
                $(pageContentNode).find('.drawing').attr('contenteditable', 'true');
            }

            if (changeTrack.updateSideBar()) { self.trigger('changeTrack:stateInfo', { state: true }); }
            insertCollaborativeOverlay();
            insertDrawingSelectionOverlay();

            self.setBlockedGuiUpdate(true); // Performance: The missing style sheets must not trigger button generation in GUI
            insertMissingCharacterStyles();
            insertMissingParagraphStyles();
            tableStyles.insertMissingTableStyles();
            if (app.isODF()) { insertMissingDrawingStyles(); }
            self.setBlockedGuiUpdate(false);

            // Special observer for iPad/Safari mobile. Due to missing events to detect
            // and process DOM changes we have to add special code to handle these cases.
            if (Utils.IOS) {
                var MutationObserver = window.MutationObserver || window.WebKitMutationObserver || window.MozMutationObserver;

                editDivObserver = new MutationObserver(function (mutations) {
                    if (!self) {
                        editDivObserver.disconnect();
                        return;
                    }
                    if (!self.isProcessingOperations()) {
                        // Optimization: we only check DOM mutations when we don't process our own
                        // operations.

                        mutations.forEach(function (mutation) {
                            var i = 0;

                            if (mutation.addedNodes.length > 0) {

                                for (i = 0; i < mutation.addedNodes.length; i++) {
                                    var node = mutation.addedNodes[i];

                                    if (node.nodeType === 1 && node.tagName === 'IMG' && node.className === '-webkit-dictation-result-placeholder') {
                                        // We have detected a Siri input. Switch model to read-only/disconnect and disconnect observer
                                        app.rejectEditAttempt('siri');
                                        editDivObserver.disconnect();
                                    }
                                }
                            }
                        });
                    }
                });

                editDivObserver.observe(editdiv[0], { childList: true, subtree: true });

                // Prevent text suggestions which are available via context menu on iPad/iPhone.
                selection.on('change', function (event, options) {
                    var paragraph, text;

                    // Performance: Ignoring handler for 'simpleTextSelection' operations
                    if (!Utils.getBooleanOption(options, 'simpleTextSelection', false)) {
                        if (selection.hasRange()) {
                            if (selection.getSelectionType() === 'text' && app.isEditable()) {
                                paragraph = selection.getEnclosingParagraph();
                                text = null;

                                if (paragraph) {
                                    text = selection.getSelectedText();
                                    if (text && text.length > 0 && text.indexOf(' ') === -1) {
                                        setIOSROParagraph(paragraph);
                                    } else {
                                        setIOSROParagraph(null);
                                    }
                                }
                            }
                        } else {
                            // in case the text cursor is set to a different position
                            setIOSROParagraph(null);
                        }
                    }
                });
            }
        }

        /**
         * Removes empty text nodes from the passed paragraph, checks whether
         * it needs a dummy text node, and converts consecutive white-space
         * characters.
         *
         * @param {HTMLElement|jQuery} paragraph
         *  The paragraph element to be validated. If this object is a jQuery
         *  collection, uses the first DOM node it contains.
         */
        function validateParagraphNode(paragraph) {

            var // current sequence of sibling text nodes
                siblingTextNodes = [],
                // array of arrays collecting all sequences of sibling text nodes
                allSiblingTextNodes = [siblingTextNodes],
                // whether the paragraph contains any text
                hasText = false,
                // whether the last child node is the dummy element
                hasLastDummy = false,
                // whether we have to add or preserve a dummy element for hard-breaks or template text spans
                mustAddOrPreserveDummy = false;

            if (self.isImeActive()) {
                // #51390: Never manipulate the DOM when a composition is active
                return;
            }

            validateParagraphNode.DBG_COUNT = (validateParagraphNode.DBG_COUNT || 0) + 1;

            // convert parameter to a DOM node
            paragraph = Utils.getDomNode(paragraph);

            // whether last node is the dummy node
            hasLastDummy = DOM.isDummyTextNode(paragraph.lastChild);

            // remove all empty text spans which have sibling text spans, and collect
            // sequences of sibling text spans (needed for white-space handling)
            Position.iterateParagraphChildNodes(paragraph, function (node) {

                // visit all text spans embedded in text container nodes (fields, tabs, ... (NOT numbering labels))
                if (DOM.isTextSpan(node) || DOM.isTextComponentNode(node)) {
                    DOM.iterateTextSpans(node, function (span) {
                        if (DOM.isEmptySpan(span)) {
                            // remove this span, if it is an empty portion and has a sibling text portion (should not happen anymore)
                            if (DOM.isTextSpan(span.previousSibling) || DOM.isTextSpan(span.nextSibling)) {
                                Utils.warn('Editor.validateParagraphNode(): empty text span with sibling text span found');
                                $(span).remove();
                            }
                            // otherwise simply ignore the empty span
                        } else {
                            if (hasLastDummy && DOM.isTextFrameTemplateTextSpan(span)) { mustAddOrPreserveDummy = true; }
                            // append text node to current sequence
                            siblingTextNodes.push(span.firstChild);
                        }
                    });

                // anything else (no text span or text container node): start a new sequence of text nodes
                } else {
                    allSiblingTextNodes.push(siblingTextNodes = []);
                }

            }, undefined, { allNodes: true });

            // Special case for hyperlink formatting and empty paragraph (just one empty textspan)
            // Here we always don't want to have the hyperlink formatting, we hardly reset these attributes
            var url;
            if (DOM.isTextSpan(paragraph.firstElementChild) && (paragraph.children.length === 1) &&
                (paragraph.firstElementChild.textContent.length === 0)) {
                url = characterStyles.getElementAttributes(paragraph.firstElementChild).character.url;
                if ((url !== null) && (url.length > 0)) {
                    characterStyles.setElementAttributes(paragraph.firstElementChild, Hyperlink.CLEAR_ATTRIBUTES);
                }
            } else if (DOM.isListLabelNode(paragraph.firstElementChild) && (paragraph.children.length === 2) &&
                       (paragraph.firstElementChild.nextSibling.textContent.length === 0)) {
                url = characterStyles.getElementAttributes(paragraph.firstElementChild.nextSibling).character.url;
                if ((url !== null) && (url.length > 0)) {
                    characterStyles.setElementAttributes(paragraph.firstElementChild.nextSibling, Hyperlink.CLEAR_ATTRIBUTES);
                }
            }

            // Special case for hard-breaks. Check for a hard-break as a node before the last child which
            // is not a dummy node. We need to have a dummy node for a hard-break at the end of the
            // paragraph to support correct formatting & cursor travelling.
            if (paragraph.children.length >= 2) {
                var prevNode = paragraph.lastChild.previousSibling;

                if (paragraph.children.length >= 3 && hasLastDummy) {
                    // Second case: We have a dummy node and must preserve it.
                    // prevNode is in that case the previous node of prev node.
                    // (hard-break) - (span) - (br)
                    prevNode = prevNode.previousSibling;
                }

                if (prevNode && DOM.isHardBreakNode(prevNode)) {
                    mustAddOrPreserveDummy = true;
                }
            }

            // Convert consecutive white-space characters to sequences of SPACE/NBSP
            // pairs. We cannot use the CSS attribute white-space:pre-wrap, because
            // it breaks the paragraph's CSS attribute text-align:justify. Process
            // each sequence of sibling text nodes for its own (the text node
            // sequences may be interrupted by other elements such as hard line
            // breaks, drawings, or other objects).
            // TODO: handle explicit NBSP inserted by the user (when supported)
            _(allSiblingTextNodes).each(function (siblingTextNodes) {

                var // the complete text of all sibling text nodes
                    text = '',
                    // offset for final text distribution
                    offset = 0;

                // collect the complete text in all text nodes
                _(siblingTextNodes).each(function (textNode) { text += textNode.nodeValue; });

                // ignore empty sequences
                if (text.length > 0) {
                    hasText = true;

                    // process all white-space contained in the text nodes
                    text = text
                        // normalize white-space (convert to SPACE characters)
                        .replace(/\s/g, ' ')
                        // text in the node sequence cannot start with a SPACE character
                        .replace(/^ /, '\xa0')
                        // convert SPACE/SPACE pairs to SPACE/NBSP pairs
                        .replace(/ {2}/g, ' \xa0')
                        // text in the node sequence cannot end with a SPACE character
                        .replace(/ $/, '\xa0');

                    // distribute converted text to the text nodes
                    _(siblingTextNodes).each(function (textNode) {
                        var length = textNode.nodeValue.length;
                        textNode.nodeValue = text.substr(offset, length);
                        offset += length;
                    });
                }
            });

            // determine hasLastDummy again to be sure that there is, after some
            // possible modifications before, a dummy text node
            hasLastDummy = DOM.isDummyTextNode(paragraph.lastChild);

            // insert an empty text span if there is no other content (except the dummy node)
            if (!paragraph.hasChildNodes() || (hasLastDummy && (paragraph.childNodes.length === 1))) {
                $(paragraph).prepend(DOM.createTextSpan());
            }

            // append dummy text node if the paragraph contains no text,
            // or remove it if there is any text
            if (!hasText && !hasLastDummy) {
                $(paragraph).append(DOM.createDummyTextNode());
            } else if (!hasLastDummy && mustAddOrPreserveDummy) {
                // we need to add a dummy text node after a last hard-break
                $(paragraph).append(DOM.createDummyTextNode());
            } else if (hasText && hasLastDummy && !mustAddOrPreserveDummy) {
                $(paragraph.lastChild).remove();
            }
        }

        /**
         * Draws the border(s) of drawings. Needed for localStorage case to draw lines into the canvas
         * or for automatically resized text frames or text frames within groups.
         *
         * @param {jQuery|Node[]} drawingNodes
         *  Collector (jQuery or array) for all drawing nodes, whose border canvas need to be
         *  repainted.
         */
        function updateDrawings(event, drawingNodes) {

            _.each(drawingNodes, function (node) {

                var // one drawing node, must be 'jQuerified'
                    drawing = DrawingFrame.isDrawingFrame(node) ? $(node) : DrawingFrame.getDrawingNode(node),
                    // the attributes of the drawing (also using styles (38058))
                    drawingAttrs = drawingStyles.getElementAttributes(drawing);

                // update is required for table background color (48370)
                if (self.useSlideMode() && DrawingFrame.isTableDrawingFrame(node)) {
                    tableStyles.updateTableDrawingBackground(node);
                }

                DrawingFrame.updateFormatting(app, drawing, drawingAttrs);
            });
        }

        /**
         * Calculates the size of the image defined by the given url,
         * limiting the size to the paragraph size of the current selection.
         *
         * @param {String} url
         *  The image url or base64 data url
         *
         * @returns {jQuery.Promise}
         *  A promise where the resolve result contains the size object
         *  including width and height in Hmm {width: 123, height: 456}
         */
        function getImageSize(url) {

            var // the result deferred
                def = self.createDeferred('Editor.getImageSize'),
                // the image for size rendering
                image = $('<img>'),
                // the image url
                absUrl,
                // the clipboard holding the image
                clipboard = null;

            if (!url) { return def.reject(); }

            absUrl = ImageUtils.getFileUrl(app, url);

            // append the clipboard div to the body and place the image into it
            clipboard = app.getView().createClipboardNode();
            clipboard.append(image);

            // if you set the load handler BEFORE you set the .src property on a new image, you will reliably get the load event.
            image.one('load', function () {

                var width, height, para, maxWidth, maxHeight, factor, start,
                    pageAttributes = null,
                    // a text frame node containing the image
                    textFrameDrawing = null;

                // helper function for calculating drawing width and height
                function resolveImageSize() {
                    if (app.isTextApp()) { // -> adapting the size of the drawing to the paragraph (only in OX Text!)
                        // maybe the paragraph is not so big
                        start = selection.getStartPosition();
                        start.pop();
                        para = Position.getParagraphElement(editdiv, start);
                        if (para) {
                            maxWidth = Utils.convertLengthToHmm($(para).outerWidth(), 'px');
                            pageAttributes = pageStyles.getElementAttributes(editdiv);
                            // reading page attributes, they are always available -> no need to check existence
                            maxHeight = pageAttributes.page.height - pageAttributes.page.marginTop - pageAttributes.page.marginBottom;

                            // adapting width and height also to text frames with fixed height (36329)
                            if (DOM.isNodeInsideTextFrame(para)) {
                                textFrameDrawing = DrawingFrame.getClosestTextFrameDrawingNode(para);
                                if (DrawingFrame.isFixedHeightDrawingFrame(textFrameDrawing)) {
                                    maxHeight = Utils.convertLengthToHmm(DrawingFrame.getTextFrameNode(textFrameDrawing).height(), 'px');
                                }
                            }

                            if (width > maxWidth) {
                                factor = Utils.round(maxWidth / width, 0.01);
                                width = maxWidth;
                                height = Math.round(height * factor);
                            }

                            if ((maxHeight) && (height > maxHeight)) {
                                factor = Utils.round(maxHeight / height, 0.01);
                                height = maxHeight;
                                width = Math.round(width * factor);
                            }
                        }
                    }

                    def.resolve({ width: width, height: height });
                }

                // Workaround for a strange Chrome behavior, even if we use .one() Chrome fires the 'load' event twice.
                // One time for the image node rendered and the other time for a not rendered image node.
                // We check for the rendered image node
                if (Utils.containsNode(clipboard, image)) {
                    width = Utils.convertLengthToHmm(image.width(), 'px');
                    height = Utils.convertLengthToHmm(image.height(), 'px');

                    if (width === 0 && height === 0) {
                        // IE performance problem: The browser sometimes returns wrong values -> waiting a half second (56730)
                        self.executeDelayed(function () {
                            width = Utils.convertLengthToHmm(image.width(), 'px');
                            height = Utils.convertLengthToHmm(image.height(), 'px');
                            resolveImageSize();
                        }, 'Editor: getImageSize', 500);
                    } else {
                        resolveImageSize();
                    }
                }
            })
            .one('error', function () {
                Utils.warn('Editor.getImageSize(): image load error');
                def.reject();
            })
            .attr('src', absUrl);

            // always remove the clipboard again
            def.always(function () {
                clipboard.remove();
            });

            return def.promise();
        }

        // ====================================================================
        // IMPLEMENTATION FUNCTIONS
        // Private methods, that are called from method applyOperations().
        // The operations itself are never generated inside an impl*-function.
        // ====================================================================

        /**
         * Has to be called after changing the structure of a paragraph node, if
         * the update has to be done immediately (synchronously).
         *
         * @param {jQuery} paragraphs
         *  All paragraph nodes that need to be updated included into a
         *  jQuery object
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.suppressFontSize=false]
         *      If set to true, the dynamic font size calculation is NOT started
         *  @param {Boolean} [options.restoreSelection=true]
         *      If set to false, the selection will not be restored via
         *      restoreBrowserSelection. This is useful, if the DOM was
         *      modfified without setting the new selection.
         */
        function implParagraphChangedSync(paragraphs, options) {

            paragraphs.each(function () {

                var // the paragraph node
                    paragraph = this,
                    // the style handler for updating attributes
                    styleHandler = null;

                // the paragraph may have been removed from the DOM in the meantime
                if (Utils.containsNode(editdiv, paragraph)) {
                    spellChecker.resetDirectly(paragraph); // reset to 'not spelled'
                    validateParagraphNode(paragraph);
                    styleHandler = (self.useSlideMode() && DOM.isSlideNode(paragraph)) ? slideStyles : paragraphStyles;
                    styleHandler.updateElementFormatting(paragraph)
                        .done(function () {
                            // -> maybe drawing formatting is required only for selected drawings? (TODO) -> check for implicit or empty paragraph?
                            options = options || {};
                            options.paragraphChanged = true; // setting marker, so that the caller can be recognized in the handler function
                            self.trigger('paragraphUpdate:after', paragraph, options); // -> Performance critical
                        });
                }
            });
            // paragraph validation changes the DOM, restore selection
            // -> Not restoring browser selection, if the edit rights are not available (Task 29049)
            // -> Not restoring browser selection, if this function was started by debounced slide formatting (with 'suppressFontSize')
            // -> Not restoring browser selection, if 'restoreSelection' is explicitely set to false (52750)
            // -> Not restoring browser selection, if this function was started inside block of async operations (#50555)
            if (!imeActive && app.isEditable() && !Utils.getBooleanOption(options, 'suppressFontSize', false) && Utils.getBooleanOption(options, 'restoreSelection', true) && !self.getBlockKeyboardEvent()) {
                selection.restoreBrowserSelection({ operationTriggered: true });
            }
        }

        /**
         * After paragraph modifications it might be necessary to update lists.
         * If this is necessary and which performance properties are required,
         * is checked within this function.
         *
         * @param {Node|jQuery|Null} [node]
         *  The DOM paragraph node to be checked. If this object is a jQuery
         *   collection, uses the first DOM node it contains. If missing or null,
         *   no list update is triggered and false is returned.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.checkSplitInNumberedList=false]
         *      If set to true, an additional check is executed, to make special
         *      performance improvements for numbered lists (32583). In this case
         *      a special marker is set at the following paragraph, if this function
         *      returns 'true'. In this case only all following paragraphs need to
         *      be updated.
         *  @param {Boolean} [options.paraInsert=false]
         *      If set to true, the 'updateList' data attribute is set at the (new)
         *      paragraph. This is necessary, if the paragraph does not already contain
         *      the list label node (DOM.LIST_LABEL_NODE_SELECTOR) and is therefore
         *      ignored in updateLists. Additionally 'updateListsDebounced' is 'informed
         *      about this setting. All this is important for performance reasons.
         *
         * @returns {Boolean}
         *  Whether the calling function can executed some specific code. This is
         *  only used for special options.
         */
        function handleTriggeringListUpdate(node, options) {

            var // paragraph attributes object
                paragraphAttrs = null,
                // the paragraph list style id
                listStyleId = null,
                // the paragraph list level
                listLevel = null,
                // whether a marker for a node is required
                runSpecificCode = false,
                // whether a split happened inside a numbered list (performance)
                splitInNumberedList = false,
                // whether the 'paraInsert' value needs to be set for updateListsDebounced
                paraInsert = Utils.getBooleanOption(options, 'paraInsert', false),
                // whether the property 'splitInNumberedList' needs to be checked
                checkSplitInNumberedList = Utils.getBooleanOption(options, 'checkSplitInNumberedList', false);

            // nothing to do, if the node is not a paragraph
            if (!DOM.isParagraphNode(node)) { return false; }

            // receiving the paragraph attributes
            // fix for Bug 37594: changed from explicit attrs to merged attrs,
            // because some para stylesheets have list styles inside
            paragraphAttrs = paragraphStyles.getElementAttributes(node);

            if (self.useSlideMode()) {
                if (paragraphAttrs.paragraph && 'level' in paragraphAttrs.paragraph) {
                    updateListsDebounced(node);
                }
            } else if (isListStyleParagraph(null, paragraphAttrs)) {
                if (paragraphAttrs && paragraphAttrs.paragraph) {
                    listStyleId = paragraphAttrs.paragraph.listStyleId;
                    listLevel = paragraphAttrs.paragraph.listLevel;

                    if (checkSplitInNumberedList) {
                        // marking paragraph for performance reasons, if this is a numbered list
                        if (listCollection && listCollection.isNumberingList(listStyleId, listLevel)) {
                            runSpecificCode = true;
                            splitInNumberedList = true;
                        }
                    }

                    if (paraInsert) { runSpecificCode = true; }
                }

                // triggering list update debounced
                updateListsDebounced({ useSelectedListStyleIDs: true, listStyleId: listStyleId, listLevel: listLevel, paraInsert: paraInsert, splitInNumberedList: splitInNumberedList });
            }

            return runSpecificCode;
        }

        /**
         * After changing the page settings it is necessary to update specific elements. This are especially drawings
         * at paragraphs and tab stops. Drawings that are anchored to the page are updated by the event 'update:absoluteElements'
         * that comes later. Tables do not need to be updated, if their width is correctly set in percentage or 'auto' If they
         * have a fixed width, this must not be changed.
         */
        function updatePageDocumentFormatting() {

            var // updating tables and drawings at paragraphs (in page and header / footer)
                pageContentNode = DOM.getPageContentNode(editdiv),
                // all paragraph and all drawings
                formattingNodes = pageContentNode.find(DrawingFrame.NODE_SELECTOR + ', ' + DOM.PARAGRAPH_NODE_SELECTOR),
                // header and footer container nodes needs to be updated also, if there is content inside them
                headerFooterFormattingNodes = pageLayout.getHeaderFooterPlaceHolder().find(DrawingFrame.NODE_SELECTOR + ', ' + DOM.PARAGRAPH_NODE_SELECTOR);

            // update of tables is not required: width 'auto' (that is '100%') and percentage values works automatically

            // add content of comment nodes
            formattingNodes = formattingNodes.add(headerFooterFormattingNodes);

            // updating all elements
            _.each(formattingNodes, function (element) {

                if (DrawingFrame.isDrawingFrame(element) && DOM.isFloatingNode(element)) {
                    // Only handle drawings that are anchored to paragraphs
                    // -> absolute positioned drawings are updated later via 'update:absoluteElements'
                    drawingStyles.updateElementFormatting(element);
                    // also updating all paragraphs inside text frames (not searching twice for the drawings)
                    implParagraphChanged($(element).find(DOM.PARAGRAPH_NODE_SELECTOR));

                } else if (DOM.isParagraphNode(element)) {

                    // update the size of all tab stops in this paragraph (but only if the paragraph contains tabs (Performance))
                    if ($(element).find(DOM.TAB_NODE_SELECTOR).length > 0) {
                        paragraphStyles.updateTabStops(element);
                    }

                }
            });
        }

        /**
         * Prepares a 'real' paragraph after insertion of text, tab, drawing, ...
         * by exchanging an 'implicit' paragraph (in empty documents, empty cells,
         * behind tables, ...) with the help of an operation. Therefore the server is
         * always informed about creation and removal of paragraphs and implicit
         * paragraphs are only required for user input in the browser.
         *
         * @param {Number[]} position
         *  The logical text position.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.ignoreLength=false]
         *      If set to true, no check for the paragraph length is done. Normally
         *      a implicit paragraph must be empty. In special cases this check
         *      must be omitted.
         */
        function handleImplicitParagraph(position, options) {

            var // the searched implicit paragraph
                paragraph = null,
                // the new created paragraph
                newParagraph = null,
                // ignore length option
                ignoreLength = Utils.getBooleanOption(options, 'ignoreLength', false),
                // target for operation - if exists, it's for ex. header or footer
                target = self.getActiveTarget(),
                // current element container node
                rootNode = self.getCurrentRootNode(target),
                // created operation
                operation = null,
                // optional attributes for the insertParagraph operation
                attrs = null;

            position = _.clone(position);
            if (position.pop() === 0) {  // is this an empty paragraph?
                paragraph = (useParagraphCache && paragraphCache) || Position.getParagraphElement(rootNode, position);
                if ((paragraph) && (DOM.isImplicitParagraphNode(paragraph)) && ((Position.getParagraphNodeLength(paragraph) === 0) || ignoreLength)) {
                    // adding attributes to the paragraph, if required
                    if (self.getReplaceImplicitParagraphAttrs) { attrs = self.getReplaceImplicitParagraphAttrs(paragraph); } // checking for attributes before removing the implicit paragraph
                    // removing implicit paragraph node
                    $(paragraph).remove();
                    // creating new paragraph explicitely
                    operation = { name: Operations.PARA_INSERT, start: position };
                    if (attrs) { operation.attrs = attrs; }
                    extendPropertiesWithTarget(operation, target);
                    self.applyOperations(operation);
                    // Setting attributes to new paragraph immediately (task 25670)
                    newParagraph = Position.getParagraphElement(rootNode, position);
                    paragraphStyles.updateElementFormatting(newParagraph);
                    // using the new paragraph as global cache
                    if (useParagraphCache && paragraphCache) { paragraphCache = newParagraph; }
                    // if implicit paragraph was marked as marginal, mark newly created also
                    if (DOM.isMarginalNode(paragraph)) {
                        $(newParagraph).addClass(DOM.MARGINAL_NODE_CLASSNAME);
                    }
                }
            }
        }

        /**
         * Prepares the text span at the specified logical position for
         * insertion of a new text component or character. Splits the text span
         * at the position, if splitting is required. Always splits the span,
         * if the position points between two characters of the span.
         * Additionally splits the span, if there is no previous sibling text
         * span while the position points to the beginning of the span, or if
         * there is no next text span while the position points to the end of
         * the span.
         *
         * @param {Number[]} position
         *  The logical text position.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.isInsertText=false]
         *      If set to true, this function was called from implInsertText.
         *  @param {Boolean} [options.useCache=false]
         *      If set to true, the paragraph element saved in the selection can
         *      be reused.
         *  @param {Boolean} [options.allowDrawingGroup=false]
         *      If set to true, the element can also be inserted into a drawing
         *      frame of type 'group'.
         *
         * @param {String|Node} [target]
         *  ID of root node or the root node itself.
         *
         * @returns {HTMLSpanElement|Null}
         *  The text span that precedes the passed offset. Will be the leading
         *  part of the original text span addressed by the passed position, if
         *  it has been split, or the previous sibling text span, if the passed
         *  position points to the beginning of the span, or the entire text
         *  span, if the passed position points to the end of a text span and
         *  there is a following text span available or if there is no following
         *  sibling at all.
         *  Returns null, if the passed logical position is invalid.
         */
        function prepareTextSpanForInsertion(position, options, target) {

            var // node info at passed position (DOM text node level)
                nodeInfo = null,
                // whether this text span is required for inserting text (performance)
                isInsertText = Utils.getBooleanOption(options, 'isInsertText', false),
                // whether the cached paragraph can be reused (performance)
                useCache = Utils.getBooleanOption(options, 'useCache', false),
                // the inline component before an empty text span
                inlineNode = null,
                // the character attributes of a previous inline component
                inlineAttrs = null,
                // the node, in which the content will be inserted
                insertNode = null;

            if (useCache && !target) {
                nodeInfo = Position.getDOMPosition(selection.getParagraphCache().node, [_.last(position)]);
            } else {
                if (target) {
                    if (_.isString(target)) {
                        nodeInfo = Position.getDOMPosition(self.getRootNode(target), position);

                        if (!nodeInfo) { // IE 11 might have empty text nodes in header/footer (58045)
                            repairEmptyTextNodes(self.getRootNode(target), { allNodes: true });
                            nodeInfo = Position.getDOMPosition(self.getRootNode(target), position);
                        }
                    } else {
                        nodeInfo = Position.getDOMPosition(target, position);
                    }
                } else {
                    nodeInfo = Position.getDOMPosition(editdiv, position);
                }
            }

            // check that the parent is a text span
            if (!nodeInfo || !nodeInfo.node || !DOM.isPortionSpan(nodeInfo.node.parentNode)) {

                // maybe the nodeInfo is a drawing in the drawing layer
                if (nodeInfo && nodeInfo.node && DOM.isDrawingLayerNode(nodeInfo.node.parentNode)) {
                    nodeInfo.node = DOM.getDrawingPlaceHolderNode(nodeInfo.node).previousSibling.firstChild; // using previous sibling (see template doc)
                    nodeInfo.offset = 0;
                }

                if (!nodeInfo || !nodeInfo.node || !DOM.isPortionSpan(nodeInfo.node.parentNode)) {
                    Utils.warn('Editor.prepareTextSpanForInsertion(): expecting text span at position ' + JSON.stringify(position));
                    return null;
                }
            }
            nodeInfo.node = nodeInfo.node.parentNode;

            // return current span, if offset points to its end
            // without following node or with following text span
            if (nodeInfo.offset === nodeInfo.node.firstChild.nodeValue.length) {
                if ((isInsertText && !nodeInfo.node.nextSibling) || DOM.isTextSpan(nodeInfo.node.nextSibling)) {
                    if (isInsertText && (nodeInfo.offset === 0) && nodeInfo.node.previousSibling && DOM.isInlineComponentNode(nodeInfo.node.previousSibling)) {
                        // inheriting character attributes from previous inline component node, for example 'tabulator'
                        inlineNode = nodeInfo.node.previousSibling;
                        if (inlineNode.firstChild && DOM.isSpan(inlineNode.firstChild)) {
                            inlineAttrs = AttributeUtils.getExplicitAttributes(inlineNode.firstChild);
                            if (inlineAttrs && inlineAttrs.changes) { delete inlineAttrs.changes; } // -> no inheritance of change track attributes
                            if (!_.isEmpty(inlineAttrs)) {
                                characterStyles.setElementAttributes(nodeInfo.node, inlineAttrs);
                            }
                        }
                    }
                    return nodeInfo.node;
                }
            }

            // do not split at beginning with existing preceding text span
            if ((nodeInfo.offset === 0) && nodeInfo.node.previousSibling) {
                if (DOM.isTextSpan(nodeInfo.node.previousSibling)) {
                    return nodeInfo.node.previousSibling;
                } else if (isInsertText && DOM.isInlineComponentNode(nodeInfo.node.previousSibling)) {
                    // inheriting character attributes from previous inline component node, for example 'tabulator'
                    inlineNode = nodeInfo.node.previousSibling;
                    if (inlineNode.firstChild && DOM.isSpan(inlineNode.firstChild)) {
                        inlineAttrs = AttributeUtils.getExplicitAttributes(inlineNode.firstChild);
                        if (inlineAttrs && inlineAttrs.changes) { delete inlineAttrs.changes; } // -> no inheritance of change track attributes
                    }
                }
            }

            // checking for template text in empty text frames (problem with undo, that inserts text in not-selected empty text frame)
            if (DOM.isTextFrameTemplateTextSpan(nodeInfo.node)) { self.removeTemplateTextFromTextSpan(nodeInfo.node); }

            // otherwise, split the span
            insertNode = DOM.splitTextSpan(nodeInfo.node, nodeInfo.offset)[0];

            // checking, if there are attributes inherited from previous inline components
            if (isInsertText && inlineAttrs && !_.isEmpty(inlineAttrs)) {
                characterStyles.setElementAttributes(insertNode, inlineAttrs);
            }

            return insertNode;
        }

        /**
         * Helper function to insert the changes mode property into the
         * attribute object. This is necessary for performance reasons:
         * The 'mode: null' property and value shall not be sent with
         * every operation.
         * On the other this is necessary, so that an inserted text/tab/
         * field/hardbreak/... can be included into a change track span,
         * without using changeTrack. So the inserted text/tab.... must
         * not be marked as change tracked.
         * -> This is different to other character attributes.
         *
         * @param {Object} [attrs]
         *  Attributes transfered with the operation. These are checked
         *  for change track information and expanded, if necessary.
         *
         * @returns {Object}
         *  Attributes object, that is expanded with the change track
         *  attributes, if required.
         */
        function checkChangesMode(attrs) {

            var // a new object containing attributes
                newAttrs;

            // if no attributes are defined, set empty changes attributes
            if (!attrs) { return { changes: { inserted: null, removed: null, modified: null } }; }

            if (!attrs.changes) {
                newAttrs = _.copy(attrs, true);
                // if attributes are defined, but no change track attributes, also set empty changes attributes
                newAttrs.changes = { inserted: null, removed: null, modified: null };
                return newAttrs;
            }

            return attrs;
        }

        /**
         * Inserts a simple text portion into the document DOM.
         *
         * @param {Number[]} start
         *  The logical start position for the new text portion.
         *
         * @param {String} text
         *  The text to be inserted.
         *
         * @param {Object} [attrs]
         *  Attributes to be applied at the new text portion, as map of
         *  attribute maps (name/value pairs), keyed by attribute family.
         *
         * @param {String} [target]
         *  The target corresponding to the specified logical start position.
         *
         * @param {Boolean} [external]
         *  Will be set to true, if the invocation of this method originates
         *  from an external operation.
         *
         * @returns {Boolean}
         *  Whether the text portion has been inserted successfully.
         */
        function implInsertText(start, text, attrs, target, external) {

            var // text span that will precede the field
                span = null,
                // new new text span
                newSpan = null, newSpanNode = null,
                // variables for page breaks rendering
                currentElement = null, currentHeight = null, directParagraph = null, textFrameNode = null,
                // the height of the paragraph or table in that the text is inserted
                prevHeight = null, prevDirectParaHeight = null,
                // whether the height of a paragraph, table or dynamic text frame has changed
                updatePageBreak = false, updateTextFrame = false,
                // whether the height of a node can be changed by this operation handler
                fixedHeight = false,
                // whether the span for text insertion is a spell error span
                isSpellErrorSpan = false,
                // the paragraph node
                paragraphNode = null,
                // the previous operation before this current operation
                previousOperation = self.getPreviousOperation(),
                // whether the cache can be reused (performance)
                useCache = (!target && previousOperation && PARAGRAPH_CACHE_OPERATIONS[previousOperation.name] &&
                            selection.getParagraphCache() && _.isEqual(_.initial(start), selection.getParagraphCache().pos) && _.isUndefined(previousOperation.target));

            // helper function to determine a node, that can be used for measuring changes of height
            // triggered by the text insertion
            function setValidHeightNode() {

                currentElement = paragraphNode;

                // measuring heights to trigger update of page breaks or auto resizing text frames
                if (!$(currentElement.parentNode).hasClass('pagecontent')) {

                    // check, if the node inside a text frame
                    textFrameNode = DrawingFrame.getDrawingNode(currentElement.parentNode);

                    if (textFrameNode && textFrameNode.length > 0) {

                        if (DOM.isCommentNode(textFrameNode)) {
                            currentElement = textFrameNode[0]; // using the comment text frame for height measurement
                        } else {
                            // check, if the text frame node needs to grow dynamically
                            if (DrawingFrame.isAutoResizeHeightDrawingFrame(textFrameNode)) {
                                directParagraph = currentElement;
                                prevDirectParaHeight = currentElement.offsetHeight;

                                // finding a top level node for page break calculation
                                if (DOM.isInsideDrawingLayerNode(currentElement)) {
                                    currentElement = DOM.getTopLevelDrawingInDrawingLayerNode(currentElement);
                                    currentElement = DOM.getDrawingPlaceHolderNode(currentElement);
                                }

                                // searching for the top level paragraph or table
                                currentElement = $(currentElement).parents(DOM.PARAGRAPH_NODE_SELECTOR).last()[0];  // its a div.p inside paragraph

                                if (currentElement && !$(currentElement.parentNode).hasClass('pagecontent')) {
                                    currentElement = $(currentElement).parents(DOM.TABLE_NODE_SELECTOR).last()[0]; // its a text frame inside table(s)
                                }

                            } else {
                                // the height of the text frame will not change! It is not automatically resized.
                                fixedHeight = true;
                            }
                        }

                    } else if (target) {
                        if (!$(currentElement.parentNode).attr('data-container-id')) {
                            currentElement = $(currentElement).parents(DOM.TABLE_NODE_SELECTOR).last()[0]; //its a div.p inside table(s)
                        }
                        // TODO parent DrawingFrame
                    } else {
                        currentElement = $(currentElement).parents(DOM.TABLE_NODE_SELECTOR).last()[0]; // its a div.p inside table(s)
                    }
                }
            }

            // starting with preparing a text span for text insertion
            span = prepareTextSpanForInsertion(start, { isInsertText: true, useCache: useCache }, target);

            if (!span) { return false; }

            paragraphNode = span.parentNode;
            isSpellErrorSpan = DOM.isSpellerrorNode(span);

            // do not write text into an implicit paragraph, improvement for task 30906
            if (DOM.isImplicitParagraphNode(paragraphNode)) { return false; }

            // handling for paragraphs behind tables in tables, whose height is explicitely set to 0
            if ((undoRedoRunning || external) && $(paragraphNode).height() === 0) { $(paragraphNode).css('height', ''); }

            // finding the correct element for measuring height changes caused by this text insertion
            setValidHeightNode();

            prevHeight = currentElement ? currentElement.offsetHeight : 0;

            // Splitting text span is only required, if attrs are available
            if (attrs) {
                attrs = checkChangesMode(attrs); // expanding attrs to disable change tracking, if not explicitely set
                // split the text span again to get actual character formatting for
                // the span, and insert the text
                newSpan = DOM.splitTextSpan(span, span.firstChild.nodeValue.length, { append: true }).contents().remove().end().text(text);
                if (!text) { DOM.ensureExistingTextNode(newSpan); } // empty text should never happen

                // apply the passed text attributes
                if (_.isObject(attrs)) { characterStyles.setElementAttributes(newSpan, attrs); }

                // removing spell error attribute
                if (isSpellErrorSpan) { spellChecker.clearSpellErrorAttributes(newSpan); }

                // removing empty neighboring text spans (merging fails -> avoiding warning in console)
                newSpanNode = Utils.getDomNode(newSpan);
                if (DOM.isChangeTrackNode(newSpanNode) || (attrs.character && attrs.character.anchor)) {
                    if (DOM.isEmptySpan(newSpanNode.previousSibling)) { $(newSpanNode.previousSibling).remove(); }
                    if (DOM.isEmptySpan(newSpanNode.nextSibling)) { $(newSpanNode.nextSibling).remove(); }
                }

            } else {

                // Performance: Simply adding new text into existing node
                // -> But not for change tracking and not if this is an empty span
                if (isSpellErrorSpan || DOM.isChangeTrackNode(span) || DOM.isEmptySpan(span)) {
                    attrs = checkChangesMode(); // expanding attrs to disable change tracking, if not explicitely set
                    newSpan = DOM.splitTextSpan(span, span.firstChild.nodeValue.length, { append: true }).contents().remove().end().text(text);
                    characterStyles.setElementAttributes(newSpan, attrs);
                    if (isSpellErrorSpan) { spellChecker.clearSpellErrorAttributes(newSpan); }
                } else {
                    span.firstChild.nodeValue += text;
                    newSpan = $(span);
                }
            }

            // try to merge with preceding and following span
            Utils.mergeSiblingTextSpans(newSpan, true);
            Utils.mergeSiblingTextSpans(newSpan);

            // validate paragraph, store new cursor position
            // paste task 55767 -> synchronous formatting is required, if attributes are specified in this operation (to remove emtpy spans)
            if (guiTriggeredOperation || !self.isImportFinished() || (pasteInProgress && !_.isObject(attrs))) {
                implParagraphChanged(newSpan.parent());  // Performance and task 30587: Local client can defer attribute setting, Task 30603: deferring also during loading document
            } else {
                implParagraphChangedSync(newSpan.parent());  // Performance and task 30587: Remote client must set attributes synchronously
            }

            if (target) { self.setBlockOnInsertPageBreaks(true); }

            quitFromPageBreak = true; // quit from any currently running pagebreak calculus - performance optimization

            // checking changes of height for top level elements (for page breaks) and for direct paragraph (auto resize text frame)
            if (!fixedHeight) {
                if (directParagraph && prevDirectParaHeight !== directParagraph.offsetHeight) {
                    updateTextFrame = true;
                } else {
                    textFrameNode = null;
                }

                if ($(currentElement).data('lineBreaksData')) {
                    $(currentElement).removeData('lineBreaksData'); //we changed paragraph layout, cached line breaks data needs to be invalidated
                }

                currentHeight = (currentElement && currentElement.offsetHeight) || 0;
                if (prevHeight !== currentHeight) {
                    updatePageBreak = true;
                }
                if (DOM.isEmptySpan(span) && DOM.isHardBreakNode(newSpan.prev())) { // #39468 - also update pb when typing after ms page hardbreak
                    updatePageBreak = true;
                }

                if (updatePageBreak || updateTextFrame) {
                    insertPageBreaksDebounced(currentElement, textFrameNode);
                    if (!pbState) { app.getView().recalculateDocumentMargin(); }
                }
            }

            lastOperationEnd = Position.increaseLastIndex(start, text.length);

            // Performance: Saving paragraph info for following operations
            selection.setParagraphCache(newSpan.parent(), _.clone(_.initial(lastOperationEnd)), _.last(lastOperationEnd));

            return true;
        }

        /**
         * Inserts a text field component into the document DOM.
         *
         * @param {Number[]} start
         *  The logical start position for the new text field.
         *
         * @param {String} type
         *  A property describing the field type.
         *
         * @param {String} representation
         *  A fallback value, if the placeholder cannot be substituted with a
         *  reasonable value.
         *
         * @param {Object} [attrs]
         *  Attributes to be applied at the new text field, as map of attribute
         *  maps (name/value pairs), keyed by attribute family.
         *
         * @param {Boolean} [external]
         *  Will be set to true, if the invocation of this method originates
         *  from an external operation.
         *
         * @returns {Boolean}
         *  Whether the text field has been inserted successfully.
         */
        function implInsertField(start, type, representation, attrs, target/*, external*/) {
            return fieldManager.implInsertField(start, type, representation, attrs, target);
        }

        /**
         * Inserts a horizontal tabulator component into the document DOM.
         *
         * @param {Number[]} start
         *  The logical start position for the new tabulator.
         *
         * @param {Object} [attrs]
         *  Attributes to be applied at the new tabulator component, as map of
         *  attribute maps (name/value pairs), keyed by attribute family.
         *
         * @param {Object} [target]
         *  If exists, defines node, to which start position is related.
         *  Used primarily to address headers/footers.
         *
         * @param {Boolean} [external]
         *  Will be set to true, if the invocation of this method originates
         *  from an external operation.
         *
         * @returns {Boolean}
         *  Whether the tabulator has been inserted successfully.
         */
        function implInsertTab(start, attrs, target/*, external*/) {

            var // text span that will precede the field
                span = prepareTextSpanForInsertion(start, {}, target),
                // new text span for the tabulator node
                tabSpan = null,
                // new tabulator node
                newTabNode,
                // element from which we calculate pagebreaks
                currentElement = span ? span.parentNode : '';

            if (!span) { return false; }

            // tab must not be inserted into the slide (52416)
            if (self.useSlideMode() && start.length === 2) { return false; }

            // expanding attrs to disable change tracking, if not explicitely set
            attrs = checkChangesMode(attrs);

            // split the text span to get initial character formatting for the tab
            tabSpan = DOM.splitTextSpan(span, 0);
            if (_.browser.IE) {
                // Prevent resizing rectangle provided by IE for tab div even
                // set to contenteditable=false. Therefore we need to set IE
                // specific unselectable=on.
                tabSpan.attr('unselectable', 'on');
            } else if (_.browser.iOS && _.browser.Safari) {
                // Safari mobile allows to visit nodes which has contenteditable=false, therefore
                // we have to use a different solution. We use -webkit-user-select: none to force
                // Safari mobile to jump over the text span with the fill characters.
                tabSpan.css('-webkit-user-select', 'none');
            }

            newTabNode = DOM.createTabNode();

            // Bug #51172:
            //      cursor jumps one line down while inserting a tab-stop
            if (_.browser.Chrome) {
                var brElement = _.filter($(span).siblings('br'), function (ele) { return $(ele).data('dummy'); });
                if (brElement.length === 1) {
                    $(brElement).remove();
                }
            }

            // insert a tab container node before the addressed text node, move
            // the tab span element into the tab container node
            newTabNode.append(tabSpan).insertAfter(span);

            // apply the passed tab attributes
            if (_.isObject(attrs)) { characterStyles.setElementAttributes(tabSpan, attrs); }

            // validate paragraph, store new cursor position
            implParagraphChanged(span.parentNode);
            lastOperationEnd = Position.increaseLastIndex(start);

            // don't call explicitly page break rendering, if target for header/footer comes with operation
            if (target) { self.setBlockOnInsertPageBreaks(true); }

            // we changed paragraph layout, cached line breaks data needs to be invalidated
            if ($(currentElement).data('lineBreaksData')) { $(currentElement).removeData('lineBreaksData'); }
            // call for debounced render of pagebreaks
            // TODO: Check if paragraph height was modified
            quitFromPageBreak = true; // quit from any currently running pagebreak calculus - performance optimization

            insertPageBreaksDebounced(currentElement, DrawingFrame.getClosestTextFrameDrawingNode(span.parentNode));
            if (!pbState) { app.getView().recalculateDocumentMargin(); }

            // immediately updating element formatting to underline the tab correctly
            // -> This is not necessary with localStorage and fastLoad during loading
            if (self.isImportFinished() && changeTrack.isActiveChangeTracking()) {
                paragraphStyles.updateElementFormatting(span.parentNode);
            }

            return true;
        }

        /**
         * Inserts a comment into the document DOM.
         *
         * @param {Number[]} start
         *  The logical start position for the new comment.
         *
         * @param {String} id
         *  The unique id of the comment.
         *
         * @param {String} author
         *  The author of the comment.
         *
         * @param {String} date
         *  The date of the comment.
         *
         * @param {String} [target]
         *  The target corresponding to the specified logical start position.
         *
         * @returns {Boolean}
         *  Whether the comment has been inserted successfully.
         */
        function implInsertComment(start, id, author, uid, date, target) {
            return commentLayer.insertCommentHandler(start, id, author, uid, date, target);
        }

        /**
         * Inserts a complex field into the document DOM.
         *
         * @param {Number[]} start
         *  The logical start position for the new complex field.
         *
         * @param {String} instruction
         *  The instruction string of the complex field.
         *
         * @param {Object} attrs
         *  The attributes of the complex field.
         *
         * @param {String} [target]
         *  The target corresponding to the specified logical start position.
         *
         * @returns {Boolean}
         *  Whether the complex field has been inserted successfully.
         */
        function implInsertComplexField(start, instruction, attrs, target) {
            return fieldManager.insertComplexFieldHandler(start, instruction, attrs, target);
        }

        /**
         * Inserts a bookmark into the document DOM.
         *
         * @param {Number[]} start
         *  The logical start position for the new bookmark.
         *
         * @param {String} anchorName
         *  The anchor string of the bookmark.
         *
         * @param {String} position
         *  String marking start or end type of bookmark.
         *
         * @param {Object} attrs
         *  The attributes of the bookmark.
         *
         * @param {String} [target]
         *  The target corresponding to the specified logical start position.
         *
         * @returns {Boolean}
         *  Whether the bookmark has been inserted successfully.
         */
        function implInsertBookmark(start, id, anchorName, position, attrs, target) {
            var span = prepareTextSpanForInsertion(start, {}, target); // text span that will precede the bookmark
            var newBokmarkNode = $('<div class="bookmark inline">').attr({ anchor: anchorName, bmPos: position, bmId: id });

            // insert a bookmark container node before the addressed text node, move
            // the bookmark span element into the bookmark container node
            newBokmarkNode.insertAfter(span);

            // apply the passed tab attributes
            if (_.isObject(attrs)) { characterStyles.setElementAttributes(span, attrs); }

            // validate paragraph, store new cursor position
            implParagraphChanged(span.parentNode);
            lastOperationEnd = Position.increaseLastIndex(start);

            return true;
        }

        /**
         * Updates a complex field node with new instruction.
         *
         * @param {Number[]} start
         *  The logical start position for the new complex field.
         *
         * @param {String} instruction
         *  The instruction string of the complex field.
         *
         * @param {String} [target]
         *  The target corresponding to the specified logical start position.
         *
         * @returns {Boolean}
         *  Whether the complex field has been inserted successfully.
         */
        function implUpdateComplexField(start, instruction, attrs, target) {
            return fieldManager.updateComplexFieldHandler(start, instruction, attrs, target);
        }

        /**
         * Updates a simple field node with new properties.
         *
         * @param {Number[]} start
         *  The logical start position for the new simple field.
         *
         * @param {String} type
         *  Type of the simple field.
         *
         * @param {String} representation
         *  Content of the field that's updated.
         *
         * @param {String} [target]
         *  The target corresponding to the specified logical start position.
         *
         * @returns {Boolean}
         *  Whether the simple field has been inserted successfully.
         */
        function implUpdateField(start, type, representation, attrs, target) {
            return fieldManager.updateSimpleFieldHandler(start, type, representation, attrs, target);
        }

        /**
         * Inserts a range marker node into the document DOM. These ranges can be
         * used for several features like comments, complex fields, ...
         *
         * @param {Number[]} start
         *  The logical start position for the new range marker.
         *
         * @param {String} id
         *  The unique id of the range marker. This can be used to identify the
         *  corresponding node, that requires this range. This can be a comment
         *  node for example.
         *
         * @param {String} type
         *  The type of the range marker.
         *
         * @param {String} position
         *  The position of the range marker. Currently supported are 'start'
         *  and 'end'.
         *
         * @param {Object} [attrs]
         *  Attributes to be applied at the new range component, as map of
         *  attribute maps (name/value pairs), keyed by attribute family.
         *
         * @param {String} [target]
         *  The target corresponding to the specified logical start position.
         *
         * @returns {Boolean}
         *  Whether the range marker has been inserted successfully.
         */
        function implInsertRange(start, id, type, position, attrs, target) {
            return rangeMarker.insertRangeHandler(start, id, type, position, attrs, target);
        }

        /**
         * Inserts a drawing component into the document DOM.
         *
         * @param {String} type
         *  The type of the drawing. Supported values are 'shape', 'group',
         *  'image', 'diagram', 'chart', 'ole', 'horizontal_line', 'undefined'
         *
         * @param {Number[]} start
         *  The logical start position for the new tabulator.
         *
         * @param {Object} [attrs]
         *  Attributes to be applied at the new drawing component, as map of
         *  attribute maps (name/value pairs), keyed by attribute family.
         *
         * @param {Object} [target]
         *  If exists, defines node, to which start position is related.
         *  Used primary to address headers/footers.
         *
         * @returns {Boolean}
         *  Whether the drawing has been inserted successfully.
         */
        function implInsertDrawing(type, start, attrs, target) {

            var // text span that will precede the field
                span = null,
                // deep copy of attributes, because they are modified in webkit browsers
                attributes = _.copy(attrs, true),
                // new drawing node
                drawingNode = null,
                // root node container
                rootNode = self.getRootNode(target),
                // image aspect ratio
                currentElement = Position.getContentNodeElement(rootNode, start.slice(0, -1), { allowDrawingGroup: true }),
                // whether the drawing is inserted into a drawing group
                insertIntoDrawingGroup = false,
                // the function used to insert the new drawing frame
                insertFunction = 'insertAfter',
                // the number of children in the drawing group
                childrenCount = 0;

            // helper function to insert a drawing frame into an existing drawing group
            function addDrawingFrameIntoDrawingGroup() {

                childrenCount = DrawingFrame.getGroupDrawingCount(currentElement);

                if (_.isNumber(childrenCount)) {
                    if (childrenCount === 0) {
                        if (_.last(start) === 0) {
                            span = $(currentElement).children().first(); // using 'span' for the content element in the drawing group
                            insertFunction = 'appendTo';
                        }
                    } else {
                        if (_.last(start) === 0) {
                            // inserting before the first element
                            span = DrawingFrame.getGroupDrawingChildren(currentElement, 0);
                            insertFunction = 'insertBefore';
                        } else if (_.last(start) <= childrenCount) {
                            // inserting after the span element
                            span = DrawingFrame.getGroupDrawingChildren(currentElement, _.last(start) - 1);
                        }
                    }
                    insertIntoDrawingGroup = true;
                }
            }

            try {
                span = prepareTextSpanForInsertion(start, {}, target);
            } catch (ex) {
                // do nothing, try to repair missing text spans
            }

            // check, if the drawing is inserted into a drawing group
            if (!span && DrawingFrame.isGroupDrawingFrame(currentElement)) { addDrawingFrameIntoDrawingGroup(); }

            if (!span) { return false; }

            // expanding attrs to disable change tracking, if not explicitely set
            attrs = checkChangesMode(attrs);

            // insert the drawing with default settings between the two text nodes (store original URL for later use)
            drawingNode = DrawingFrame.createDrawingFrame(type)[insertFunction](span);

            if (type === 'horizontal_line') { drawingNode.addClass('horizontal-line'); }

            // apply the passed drawing attributes
            if (_.isObject(attributes)) { drawingStyles.setElementAttributes(drawingNode, attributes); }

            // validate paragraph, store new cursor position
            if (!insertIntoDrawingGroup) { implParagraphChanged(span.parentNode); }

            lastOperationEnd = Position.increaseLastIndex(start);

            // don't call explicitly page break rendering, if target for header/footer comes with operation
            if (target) { self.setBlockOnInsertPageBreaks(true); }
            if ($(currentElement).data('lineBreaksData')) {
                $(currentElement).removeData('lineBreaksData'); // we changed paragraph layout, cached line breaks data needs to be invalidated
            }
            insertPageBreaksDebounced(currentElement, insertIntoDrawingGroup ? null : DrawingFrame.getClosestTextFrameDrawingNode(span.parentNode));
            if (!pbState) { app.getView().recalculateDocumentMargin(); }

            // marking drawing as marginal for external operations (58739)
            if (!app.isEditable() && DOM.isMarginalNode(span.parentNode)) { drawingNode.addClass(DOM.MARGINAL_NODE_CLASSNAME); }

            return true;
        }

        /**
         * Internet Explorer helper function. Repairs empty text nodes, that are
         * removed by IE after inserting a node into the DOM. This function
         * adds this empty text nodes again.
         *
         * After updating to a new version of jQuery, this seems to be
         * nessecary in other browsers too.
         *
         * @param {HTMLElement|jQuery} element
         *  The DOM element whose empty text nodes shall be added again.
         *
         * @param {Object} [options]
         *  {Boolean} allNodes - optional, default value = false
         *      If set to true, iterator will go through all children nodes, not only first level children.
         *      Use only if really needed, because it costs performance.
         */
        function repairEmptyTextNodes(element, options) {
            var // option to iterate all children, not only first level (costs performance, use only if really needed)
                iterateAllNodes = Utils.getBooleanOption(options, 'allNodes', false),
                // when element is paragraph, selects ether all or only first level children spans
                selector = null;

            if (!_.browser.IE) { return; }  // only necessary for IE and Spartan

            if ($(element).is('span')) {
                DOM.ensureExistingTextNode(element);
            } else if (DOM.isParagraphNode(element)) {
                selector = iterateAllNodes ? 'span' : '> span';
                $(element).find(selector).each(function () {
                    DOM.ensureExistingTextNode(this);
                });
            } else if (iterateAllNodes) {
                _.each($(element).find(DOM.PARAGRAPH_NODE_SELECTOR + ' > span'), function (span) {
                    DOM.ensureExistingTextNode(span);
                });
            } else if (DOM.isTableCellNode(element)) {
                DOM.getCellContentNode(element).find('> ' + DOM.PARAGRAPH_NODE_SELECTOR + ' > span').each(function () {
                    DOM.ensureExistingTextNode(this);
                });
            } else if (DOM.isTableRowNode(element)) {
                $(element).find('> td').each(function () {
                    DOM.getCellContentNode(this).find('> ' + DOM.PARAGRAPH_NODE_SELECTOR + ' > span').each(function () {
                        DOM.ensureExistingTextNode(this);
                    });
                });
            } else if (DOM.isTableNode(element)) {
                $(element).find('> tbody > tr > td').each(function () {
                    DOM.getCellContentNode(this).find('> ' + DOM.PARAGRAPH_NODE_SELECTOR + ' > span').each(function () {
                        DOM.ensureExistingTextNode(this);
                    });
                });
            }
        }

        /**
         * Inserts the passed content node at the specified logical position.
         *
         * @param {Number[]} position
         *  The logical position of the new content node.
         *
         * @param {HTMLElement|jQuery} node
         *  The new content node. If this object is a jQuery collection, uses
         *  the first node it contains.
         *
         * @param {String} [target]
         *  If exists, this string determines class name of container node.
         *  Usually is used for headers/footers
         *
         * @returns {Boolean}
         *  Whether the content node has been inserted successfully.
         */
        function insertContentNode(position, node, target) {

            var origPosition = _.clone(position),
                index = _.last(position),
                contentNode = null,
                insertBefore = true,
                insertAtEnd = false,
                parentNode = null,
                rootNode = null;

            function getContainerNode(position) {

                var // logical position of the paragraph container node
                    parentPosition = position.slice(0, -1),
                    // the container node for the paragraph
                    containerInfo = Position.getDOMPosition(rootNode, parentPosition, true),
                    // the parent container node
                    containerNode = null;

                // resolve component node to the correct node that contains the
                // child content nodes (e.g. resolve table cell elements to the
                // embedded div.cellcontent elements, or drawing elements to the
                // embedded div.content elements)
                if (containerInfo && containerInfo.node) {
                    containerNode = DOM.getChildContainerNode(containerInfo.node)[0];

                    // preparing text frame node, if this is a placeholder node inside a drawing frame of type 'shape'
                    // if (DrawingFrame.isPlaceHolderNode(containerNode) && DrawingFrame.isShapeDrawingFrame(containerInfo.node)) {
                    if (DrawingFrame.isShapeDrawingFrame(containerInfo.node)) {

                        // the child node can be a placeholder or an empty content node (empty during loading)
                        if (DrawingFrame.isPlaceHolderNode(containerNode) || DrawingFrame.isEmptyDrawingContentNode(containerNode)) {
                            // preparing text frame structure in drawing
                            containerNode = DrawingFrame.prepareDrawingFrameForTextInsertion(containerInfo.node);
                            // update formatting, otherwise styles (border, background, ...) will be lost
                            // -> but waiting until the paragraph is inserted (executing delayed), so that also handling for empty
                            //    place holder drawings is done
                            if (self.isImportFinished()) {
                                self.executeDelayed(function () {
                                    drawingStyles.updateElementFormatting(DrawingFrame.getDrawingNode(containerNode));
                                }, 'Editor.insertContentNode.getContainerNode');
                            }
                            // #36327 - TF: Auto sizing for TF doesn't respect page size (not for OX Presentation)
                            if (!self.useSlideMode()) { containerNode.css('max-height', pageLayout.getDefPageActiveHeight({ convertToPixel: true }) - 15); } // 15 px is rounding for borders and inner space of textframe
                        }
                    }
                }

                return containerNode;
            }

            // setting the correct root node dependent from an optional target
            rootNode = self.getRootNode(target);

            // trying to find existing paragraphs, tables or the parent element
            if (index > -1) {

                try {
                    contentNode = Position.getContentNodeElement(rootNode, position);
                } catch (e) {
                    contentNode = null;  // trying to repair in the following code
                }

                // content node does not exist -> maybe the new paragraph/table shall be added to the end of the document
                if (!contentNode) {

                    if (index === 0) {
                        // trying to find parent node, because this is the first paragraph/table in document or table cell or text frame or comment
                        parentNode = getContainerNode(position);

                        if (!parentNode) {
                            Utils.warn('Editor.insertContentNode(): cannot find parent node at position ' + JSON.stringify(position));
                            return false;
                        }
                        insertBefore = false;
                        insertAtEnd = true;

                    } else {
                        // Further try to find an existing element, behind that the new paragraph/table shall be appended
                        position[position.length - 1]--;
                        contentNode = Position.getContentNodeElement(rootNode, position, { allowImplicitParagraphs: false });

                        if (contentNode) {
                            insertBefore = false;
                        } else {
                            Utils.warn('Editor.insertContentNode(): cannot find content node at position ' + JSON.stringify(origPosition));
                            return false;
                        }
                    }
                }
            } else if (index === -1) { // Definition: Adding paragraph/table to the end if index is -1
                if (target) {
                    parentNode = rootNode;
                } else {
                    parentNode = getContainerNode(position);
                }
                if (!parentNode) {
                    Utils.warn('Editor.insertContentNode(): cannot find parent node at position ' + JSON.stringify(position));
                    return false;
                }
                insertBefore = false;
                insertAtEnd = true;
            } else {
                Utils.warn('Editor.insertContentNode(): invalid logical position ' + JSON.stringify(position));
                return false;
            }

            // modify the DOM
            if (insertAtEnd) {
                $(node).first().appendTo(parentNode);
            } else if (insertBefore) {
                $(node).first().insertBefore(contentNode);
            } else {
                $(node).first().insertAfter(contentNode);
            }

            return true;
        }

        /**
         * Creates and inserts Header or footer with passes id and type
         *
         * @param {String} id
         * @param {String} type
         * @param {Object} attrs
         *
         * @returns {Boolean}
         */
        function implInsertHeaderFooter(id, type, attrs) {
            pageLayout.implCreateHeaderFooter(id, type, attrs, undoRedoRunning);
            return true;
        }

        /**
         * Deletes and removes from DOM node with passed id
         *
         * @param {String} id
         *
         * @returns {Boolean}
         */
        function implDeleteHeaderFooter(id) {
            return pageLayout.implDeleteHeaderFooter(id);
        }

        function implMove(_start, _end, _to, target) {

            var start = _.copy(_start, true),
                to = _.copy(_to, true),
                activeRootNode = self.getRootNode(target),
                sourcePos = Position.getDOMPosition(activeRootNode, start, true),
                destPos = Position.getDOMPosition(activeRootNode, to, true),
                insertBefore = true,
                splitNode = false,
                // a text span before the moved node before moving the node
                prevTextSpan = 0;

            // Fix for 28634 -> Moving a drawing to position with tab or hard break
            if (DOM.isHardBreakNode(destPos.node) || DOM.isTabNode(destPos.node) || DOM.isFieldNode(destPos.node)) {
                destPos.node = destPos.node.previousSibling;
            } else if (DrawingFrame.isDrawingFrame(destPos.node) && DOM.isDrawingLayerNode(destPos.node.parentNode)) {
                destPos.node = DOM.getDrawingPlaceHolderNode(destPos.node).previousSibling;
            }

            if (destPos.offset === 0) {
                insertBefore = true;
            } else if ((destPos.node.length) && (destPos.offset === (destPos.node.length - 1))) {
                insertBefore = false;
            } else if ((DrawingFrame.isDrawingFrame(destPos.node)) && (destPos.offset === 1)) {
                insertBefore = false;
            } else {
                splitNode = true;  // splitting node is required
                insertBefore = false;  // inserting after new created text node
            }

            if ((sourcePos) && (destPos)) {

                var sourceNode = sourcePos.node,
                    destNode = destPos.node,
                    useOffsetDiv = true,
                    offsetDiv = sourceNode.previousSibling,
                    doMove = true;

                if ((sourceNode) && (destNode)) {

                    if (!DrawingFrame.isDrawingFrame(sourceNode)) {
                        doMove = false; // supporting only drawings at the moment
                        Utils.warn('Editor.implMove(): moved  node is not a drawing: ' + Utils.getNodeName(sourceNode));
                    } else {
                        // also move the offset divs
                        if ((!offsetDiv) || (!DOM.isOffsetNode(offsetDiv))) { useOffsetDiv = false; }
                    }

                    if (doMove) {

                        if (splitNode) {
                            destNode = DOM.splitTextSpan(destNode, destPos.offset + 1)[0];
                        } else {
                            if (destNode.nodeType === 3) { destNode = destNode.parentNode; }
                        }

                        // using empty span as reference for inserting new components
                        if ((DrawingFrame.isDrawingFrame(destNode)) && (DOM.isOffsetNode(destNode.previousSibling))) {
                            destNode = destNode.previousSibling;  // switching temporary to offset
                        }

                        // there can be empty text spans before the destination node
                        if ((DOM.isTextSpan(destNode)) || (DrawingFrame.isDrawingFrame(destNode)) || (DOM.isOffsetNode(destNode))) {
                            while (DOM.isEmptySpan(destNode.previousSibling)) {
                                destNode = destNode.previousSibling;
                            }
                        }

                        if ((insertBefore) && (DOM.isTextSpan(destNode))) {
                            destNode = DOM.splitTextSpan(destNode, 0)[0]; // taking care of empty text span before drawing
                            insertBefore = false;  // insert drawing behind new empty text span
                        }

                        // removing empty text spans behind or after the source node
                        if ((sourceNode.previousSibling) && (sourceNode.nextSibling)) {
                            if ((DOM.isTextSpan(sourceNode.previousSibling)) && (DOM.isEmptySpan(sourceNode.nextSibling))) {
                                $(sourceNode.nextSibling).remove();
                            } else if ((DOM.isEmptySpan(sourceNode.previousSibling)) && (DOM.isTextSpan(sourceNode.nextSibling))) {
                                $(sourceNode.previousSibling).remove();
                            }
                        }

                        if ((sourceNode.previousSibling) && (sourceNode.previousSibling.previousSibling) && (sourceNode.nextSibling) && (DOM.isOffsetNode(sourceNode.previousSibling))) {
                            if ((DOM.isTextSpan(sourceNode.previousSibling.previousSibling)) && (DOM.isEmptySpan(sourceNode.nextSibling))) {
                                $(sourceNode.nextSibling).remove();
                            } else if ((DOM.isEmptySpan(sourceNode.previousSibling.previousSibling)) && (DOM.isTextSpan(sourceNode.nextSibling))) {
                                $(sourceNode.previousSibling.previousSibling).remove();
                            }
                        }

                        // saving the old previous text span for later merge
                        if (sourceNode.previousSibling && DOM.isTextSpan(sourceNode.previousSibling)) { prevTextSpan = sourceNode.previousSibling; }

                        // moving the drawing
                        if (insertBefore) {
                            $(sourceNode).insertBefore(destNode);
                        } else {
                            $(sourceNode).insertAfter(destNode);
                        }

                        // moving also the corresponding div before the moved drawing
                        if (useOffsetDiv) { $(offsetDiv).insertBefore(sourceNode); }

                        // merging text spans, if possible
                        if (prevTextSpan) { Utils.mergeSiblingTextSpans(prevTextSpan, true); }

                        implParagraphChanged(to);
                    }
                }
            }

            return true;
        }

        /**
         * Check, wheter a given paragraph node describes a paragraph that
         * is part of a list (bullet or numbering list).
         *
         * @param {HTMLElement|jQuery} node
         *  The paragraph whose attributes will be checked. If this object is a
         *  jQuery collection, uses the first DOM node it contains.
         *
         * @param {Object} [attributes]
         *  An optional map of attribute maps (name/value pairs), keyed by attribute.
         *  It this parameter is defined, the parameter 'paragraph' can be omitted.
         *
         * @returns {Boolean}
         *  Whether the content node is a list paragraph node.
         */
        function isListStyleParagraph(paragraph, attributes) {

            var // the attributes directly assigned to the paragraph
                paraAttrs = attributes || AttributeUtils.getExplicitAttributes(paragraph);

            // shortcut, not handling style attributes
            if (paraAttrs && paraAttrs.paragraph && paraAttrs.paragraph.listStyleId) { return true; }

            // checking paragraph style
            if (attributes && attributes.styleId && self.isParagraphStyleWithListStyle(attributes.styleId)) { return true; }

            return false;
        }

        /**
         * After reloading a document, some updates are necessary in the model.
         * For example refreshing the collector for artificial paragraphs (IE and
         * Firefox).
         */
        function documentReloadHandler() {

            var paragraphs = null;

            if (_.isEmpty(highlightedParagraphs)) { return; }  // nothing to do

            if ((_.browser.Firefox || _.browser.IE) && selection.hasRange()) {
                paragraphs = DOM.getPageContentNode(editdiv).find('div.selectionHighlighting');
                // converting to array
                highlightedParagraphs = $.makeArray(paragraphs);
            }
        }

        /**
         * The handler this is triggered after a change of the edit privileges.
         */
        function changeEditModeHandler(editMode) {

            // set CSS class for formatting dependent on edit mode (mouse pointers)
            editdiv.toggleClass('edit-mode', editMode);

            // Fix for 30992, setting valid text position after receiving edit rights
            if (editMode) {

                if (self.useSlideMode()) {
                    selection.restoreAllDrawingSelections();
                } else {

                    if (Position.isValidTextPosition(editdiv, selection.getStartPosition())) {
                        if (Position.isValidTextPosition(editdiv, selection.getEndPosition())) {
                            selection.setTextSelection(selection.getStartPosition(), selection.getEndPosition());
                        } else {
                            selection.setTextSelection(selection.getStartPosition());
                        }
                    } else if (selection.isDrawingSelection()) {
                        selection.restoreAllDrawingSelections();
                    } else if ((!_.isEmpty(selection.getStartPosition())) && (Position.isValidTextPosition(editdiv, Position.getFirstPositionInParagraph(editdiv, [selection.getStartPosition()[0]])))) {
                        selection.setTextSelection(Position.getFirstPositionInParagraph(editdiv, [selection.getStartPosition()[0]]));
                    } else if (lastOperationEnd && Position.isValidTextPosition(editdiv, lastOperationEnd)) {
                        selection.setTextSelection(lastOperationEnd);
                    } else {
                        selection.selectTopPosition();
                    }
                }
            }

            // in read-only mode change drawing selection to text selection
            if (!editMode) {

                if (self.useSlideMode()) {
                    selection.restoreAllDrawingSelections();
                } else {
                    if (selection.isDrawingSelection()) {
                        selection.restoreAllDrawingSelections();
                    } else {
                        selection.selectDrawingAsText();
                    }

                    if (self.isHeaderFooterEditState()) {
                        pageLayout.leaveHeaderFooterEditMode();
                        selection.setNewRootNode(editdiv);
                        selection.setTextSelection(selection.getFirstDocumentPosition()); // #43083
                    }
                }
            }

            // We are not able to disable the context menu in read-only mode on iPad
            // or other touch devices. Therefore we just set contenteditable dependent
            // on the current editMode.
            if (Utils.TOUCHDEVICE) {

                // prevent virtual keyboard on touch devices in read-only mode
                editdiv.attr('contenteditable', editMode);
            }

            // disable IE table manipulation handlers in edit mode
            // Utils.getDomNode(editdiv).onresizestart = function () { return false; };
            // The resizestart event does not appear to bubble in IE9+, so we use the selectionchange event to bind
            // the resizestart handler directly once the user selects an object (as this is when the handles appear).
            // The MS docs (http://msdn.microsoft.com/en-us/library/ie/ms536961%28v=vs.85%29.aspx) say it's not
            // cancelable, but it seems to work in practice.

            // The oncontrolselect event fires when the user is about to make a control selection of the object.
            // Non-standard event defined by Microsoft for use in Internet Explorer.
            // This event fires before the element is selected, so inspecting the selection object gives no information about the element to be selected.
            if (_.browser.IE) {
                Utils.getDomNode(editdiv).oncontrolselect = function (e) {
                    if (DOM.isImageNode(e.srcElement)) {
                        // if an image was selected, return browser selection to the image clipboard
                        selection.restoreBrowserSelection();
                    }
                    // return false to suppress IE default size grippers
                    return false;
                };
            }

            // focus back to editor
            app.getView().grabFocus();
        }

        /**
         * Listener to the 'change' event triggered by the selection.
         *
         * @param {jQuery.Event} event
         *  A jQuery 'change' event object.
         *
         * @param {Object} options
         *  Optional parameters:
         *  @param {Boolean} [options.insertOperation=false]
         *      Whether this function was triggered by an insert operation, for example 'insertText'.
         *  @param {Boolean} [options.splitOperation=false]
         *      Whether this function was triggered by an split paragraph operation.
         *  @param {Boolean} [options.keepChangeTrackPopup=false]
         *      Whether the change track pop up triggered this change of selection.
         *  @param {jQuery.Event} [options.event=null]
         *      The original browser event passed to this method
         */
        function selectionChangeHandler(event, options) {

            var // whether an implicit paragraph behind a table has to be increased
                increaseParagraph = false,
                domNode = null, paraNode = null, rowNode = null, tableNode = null,
                explicitAttributes = null,
                startPosition = null,
                nodeInfo = null,
                insertOperation = Utils.getBooleanOption(options, 'insertOperation', false),
                splitOperation = Utils.getBooleanOption(options, 'splitOperation', false),
                keepChangeTrackPopup = Utils.getBooleanOption(options, 'keepChangeTrackPopup', false),
                browserEvent = Utils.getObjectOption(options, 'event', null),
                readOnly = !app.isEditable(),
                // selection hightlighting: dom point of start and end position
                startNodeInfo = null, endNodeInfo = null,
                // selection hightlighting: paragraph dom nodes
                nextParagraph = null, startParagraph = null, endParagraph = null,
                // whether the iteration reached the end paragraph
                reachedEndParagraph = false,
                // helper nodes for explicit paragraphs inside text frames
                drawing = null, drawingBorder = null,
                // target for operation - if exists, it's for ex. header or footer
                target = self.getActiveTarget(),
                // current element container node
                thisRootNode = self.getCurrentRootNode(target);

            // handling height of implicit paragraphs, but not for insertText operations or splitting of paragraphs
            // and also not in readonly mode (28563)
            if (!insertOperation && !splitOperation && !readOnly) {

                startPosition = selection.getStartPosition();
                nodeInfo = Position.getDOMPosition(thisRootNode, startPosition);

                // if this is the last row of a table, a following implicit paragraph has to get height zero or auto
                if (Position.isPositionInTable(thisRootNode, startPosition)) {
                    domNode = nodeInfo.node;
                    paraNode = $(domNode).closest(DOM.PARAGRAPH_NODE_SELECTOR);
                    rowNode = $(paraNode).closest(DOM.TABLE_ROWNODE_SELECTOR);
                    tableNode = $(rowNode).closest(DOM.TABLE_NODE_SELECTOR);

                    if (paraNode !== null) {
                        if (paraNode.get(0) === increasedParagraphNode) {
                            increaseParagraph = true;  // avoid, that height is set to 0, if cursor is in this paragraph
                            if (paraNode.prev().length === 0) {  // maybe the previous table was just removed
                                paraNode.css('height', '');  // the current paragraph always has to be increased
                                increasedParagraphNode = null;
                            }
                        } else if (DOM.isIncreasableParagraph(paraNode)) {
                            if (increasedParagraphNode !== null) { $(increasedParagraphNode).css('height', 0); } // useful for tables in tables in tables in ...
                            paraNode.css('height', '');
                            increaseParagraph = true;   // cursor travelling from right/button directly into the explicit paragraph
                            increasedParagraphNode = paraNode.get(0);
                            // in Internet Explorer it is necessary to restore the browser selection, so that no frame is drawn around the paragraph (28132)
                            if (_.browser.IE) { selection.restoreBrowserSelection(); }
                        } else if ((DOM.isChildTableNode(tableNode)) && ($(rowNode).next().length === 0) && (DOM.isIncreasableParagraph($(tableNode).next()))) {
                            if (increasedParagraphNode !== null) { $(increasedParagraphNode).css('height', 0); } // useful for tables in tables in tables in ...
                            $(tableNode).next().css('height', '');
                            increaseParagraph = true;
                            increasedParagraphNode = $(tableNode).next().get(0);
                        } else if ((paraNode.prev().length === 0) && (paraNode.height() === 0)) {
                            // if the cursor was not in the table, when the inner table was deleted -> there was no increasedParagraphNode behind the table
                            paraNode.css('height', '');  // the current paragraph always has to be increased, maybe a previous table just was deleted (undo)
                        }
                    }
                } else {

                    // check if the currently increased paragraph behind a table is clicked (for example inside a text frame)
                    if (increasedParagraphNode) {
                        domNode = nodeInfo.node;
                        paraNode = domNode && $(domNode).closest(DOM.PARAGRAPH_NODE_SELECTOR);
                        if (paraNode && Utils.getDomNode(paraNode) === increasedParagraphNode) {
                            increaseParagraph = true;  // avoid, that height is set to 0, if cursor is in this paragraph
                        }
                    }

                    if (_.browser.IE) {
                        // in Internet Explorer it is necessary to restore the browser selection, so that no frame is drawn around the paragraph (28132)
                        paraNode = nodeInfo ? $(nodeInfo.node).closest(DOM.PARAGRAPH_NODE_SELECTOR) : null;
                        if (DOM.isIncreasableParagraph(paraNode)) {
                            paraNode.css('height', '');
                            selection.restoreBrowserSelection();
                        }
                    }
                }

                // is the increased paragraph node inside a text frame?
                drawing = $(increasedParagraphNode).closest(DrawingFrame.NODE_SELECTOR);

                // if the implicit paragraph no longer has to get an increase height
                if (!increaseParagraph && increasedParagraphNode) {
                    // can the increased paragraph node still be decreased (not if it is no longer increasable)
                    if (DOM.isIncreasableParagraph(increasedParagraphNode)) { $(increasedParagraphNode).css('height', 0); }
                    increasedParagraphNode = null;

                }

                // check, if the drawing border needs to be repainted (canvas)
                if (drawing.length > 0) {
                    drawingBorder = DrawingFrame.getCanvasNode(drawing);
                    if (drawingBorder.length > 0) { self.trigger('drawingHeight:update', drawing); }
                }

            }

            // in Internet Explorer it is necessary to restore the browser selection, so that no frame is drawn around the paragraph (28132),
            // this part for the read only mode
            if (readOnly && _.browser.IE && !Position.isPositionInTable(thisRootNode, selection.getStartPosition())) {
                nodeInfo = Position.getDOMPosition(thisRootNode, selection.getStartPosition());
                if (nodeInfo) {
                    paraNode = $(nodeInfo.node).closest(DOM.PARAGRAPH_NODE_SELECTOR);
                    if ((DOM.isImplicitParagraphNode(paraNode)) && (DOM.isFinalParagraphBehindTable(paraNode))) {
                        paraNode.css('height', '');
                        selection.restoreBrowserSelection();
                    }
                }
            }

            // clear preselected attributes when selection changes
            if (!keepPreselectedAttributes) { preselectedAttributes = null; }

            // Special behaviour for cursor positions directly behind fields.
            // -> the character attributes of the field have to be used
            // -> using preselectedAttributes
            if (!insertOperation && nodeInfo && (nodeInfo.offset === 0) && nodeInfo.node && nodeInfo.node.parentNode && DOM.isFieldNode(nodeInfo.node.parentNode.previousSibling)) {
                explicitAttributes = AttributeUtils.getExplicitAttributes(nodeInfo.node.parentNode.previousSibling.firstChild);
                self.addPreselectedAttributes(explicitAttributes);
            }

            // removing highlighting of selection of empty paragraphs in Firefox and IE
            if (!_.isEmpty(highlightedParagraphs)) { self.removeArtificalHighlighting(); }

            // highlighting selection of empty paragraphs in Firefox and IE
            // fix for Bug 47820
            if ((_.browser.Firefox || _.browser.IE) && selection.hasRange() && selection.getSelectionType() !== 'cell') {
                startNodeInfo = Position.getDOMPosition(thisRootNode, selection.getStartPosition());
                endNodeInfo = Position.getDOMPosition(thisRootNode, selection.getEndPosition());

                if (startNodeInfo && startNodeInfo.node && endNodeInfo && endNodeInfo.node) {
                    startParagraph = Utils.getDomNode($(startNodeInfo.node).closest(DOM.PARAGRAPH_NODE_SELECTOR));
                    endParagraph = Utils.getDomNode($(endNodeInfo.node).closest(DOM.PARAGRAPH_NODE_SELECTOR));

                    if (startParagraph && endParagraph && startParagraph !== endParagraph) {
                        nextParagraph = startParagraph;

                        while (nextParagraph && !reachedEndParagraph) {

                            if (nextParagraph) {

                                // only highlight empty paragraphs
                                if (Position.getParagraphNodeLength(nextParagraph) === 0) {
                                    // collecting all modified paragraphs in an array
                                    highlightedParagraphs.push(nextParagraph);
                                    // modify the background color of the empty paragraph to simulate selection
                                    // Task 35375: Using class, so that background-color is not included into copy/paste
                                    $(nextParagraph).addClass('selectionHighlighting');
                                }

                                if (nextParagraph === endParagraph) {
                                    reachedEndParagraph = true;
                                } else {
                                    nextParagraph = Utils.findNextNode(thisRootNode, nextParagraph, DOM.PARAGRAPH_NODE_SELECTOR);
                                }
                            }
                        }
                    }
                }
            }

            // forward selection change events to own listeners (view and controller)
            self.trigger('selection', selection, { insertOperation: insertOperation, browserEvent: browserEvent, keepChangeTrackPopup: keepChangeTrackPopup, updatingRemoteSelection: updatingRemoteSelection });
        }

        /**
         * Listener to the events 'undo:after' and 'redo:after'.
         *
         * @param {jQuery.Event} event
         *  A jQuery 'undo:after' and 'redo:after' event object.
         *
         * @param {Array<Object>} operations
         *  An array containing the undo or redo operations.
         *
         * @param {Object|Null} selectionState
         *  The selection state to be restored.
         *
         * @param {Object} [options]
         *  Additional options for undo/redo after
         *  Optional parameters:
         *  @param {Boolean} [options.preventSelectionChange=false]
         *      If set to true, the selection will not be changed after
         *      applying the undo/redo operations.
         */
        function undoRedoAfterHandler(event, operations, selectionState, options) {

            var para = null,
                newParagraph = null,
                pageContentNode = null,
                lastOperation = _.last(operations),
                // whether this is a redo operation
                isRedo = (event.type === 'redo:after'),
                // whether the saved selection state can be used (not in redo (53773))
                useSelectionState = !!(selectionState && selectionState.start && selectionState.end && !isRedo),
                // the target belonging to the logical position
                target = useSelectionState ? selectionState.target : (lastOperation && lastOperation.target), // handle empty lastOperation (41502)
                isCreateDeleteHeaderFooterOp = lastOperation ? (lastOperation.name === 'insertHeaderFooter' || lastOperation.name === 'deleteHeaderFooter') : false,
                $rootNode = self.getRootNode(target),
                // a temporary logical position
                tempLastOperationEnd = null,
                // holder for filtered set attributes operation for rotation of drawings
                rotationOps = null;

            var preventSelectionChange = Utils.getBooleanOption(options, 'preventSelectionChange', false);

            // unblock keyboard events again
            self.setBlockKeyboardEvent(false);

            // not blocking page break calculations after undo (40169)
            self.setBlockOnInsertPageBreaks(false);

            if (app.isTextApp()) {

                // group of undo operations, replace headers&footers after all operations are finnished
                if (_.findWhere(operations, { name: 'insertHeaderFooter' })) {
                    pageLayout.markAllUndoRedoNodesAsMarginal(_.where(operations, { name: 'insertHeaderFooter' }));
                    pageLayout.replaceAllTypesOfHeaderFooters();
                    pageLayout.updateStartHeaderFooterStyle();
                    self.setBlockOnInsertPageBreaks(false); // release block on insertPageBreaks to register debounced call
                    insertPageBreaksDebounced();
                }
                // odf needs also replacing of headers after delete (switching type)
                if (app.isODF() && _.findWhere(operations, { name: 'deleteHeaderFooter' })) {
                    pageLayout.replaceAllTypesOfHeaderFooters();
                }

                // deactivating an optionally active comment
                if (activeTarget && commentLayer.isCommentTarget(activeTarget) && target !== activeTarget) {
                    commentLayer.deActivateCommentNode(activeTarget, selection);
                }

                // return from edit mode of header/footer if undo/redo operation doesnt have target,
                // or is not create or delete headerFooter operation
                if (self.isHeaderFooterEditState() && !(target || isCreateDeleteHeaderFooterOp)) {
                    pageLayout.leaveHeaderFooterEditMode();
                    selection.setNewRootNode(editdiv); // restore original rootNode
                } else if (target) {
                    if (commentLayer.isCommentTarget(target) && target !== activeTarget) {
                        commentLayer.activateCommentNode(commentLayer.getCommentRootNode(target), selection, { deactivate: true });
                    } else if (pageLayout.isIdOfMarginalNode(target)) {
                        // jump into editing mode of header/footer with given target (first occurance in doc)
                        if (self.isHeaderFooterEditState()) {
                            // do nothing if we are already inside that node
                            if (self.getHeaderFooterRootNode()[0] !== $rootNode[0]) {
                                // first leave current node, and enter another
                                pageLayout.leaveHeaderFooterEditMode(self.getHeaderFooterRootNode());
                                pageLayout.updateEditingHeaderFooter(); // must be direct call

                                if (!$rootNode.parent().hasClass(DOM.HEADER_FOOTER_PLACEHOLDER_CLASSNAME)) {
                                    // only if h/f type is not in placeholder
                                    pageLayout.enterHeaderFooterEditMode($rootNode);
                                    selection.setNewRootNode($rootNode);
                                    app.getView().scrollToChildNode($rootNode);
                                }
                            }
                        } else if (!$rootNode.parent().hasClass(DOM.HEADER_FOOTER_PLACEHOLDER_CLASSNAME)) {
                            // only if h/f type is not in placeholder
                            pageLayout.enterHeaderFooterEditMode($rootNode);
                            selection.setNewRootNode($rootNode);
                            app.getView().scrollToChildNode($rootNode);
                        }
                    }
                }
            }

            // #31971, #32322
            pageContentNode = DOM.getPageContentNode(editdiv);
            if (pageContentNode.find(DOM.CONTENT_NODE_SELECTOR).length === 0 && pageContentNode.find(DOM.PAGE_BREAK_SELECTOR).length > 0) { //cleanup of dom
                pageContentNode.find(DOM.PAGE_BREAK_SELECTOR).remove();

                newParagraph = DOM.createImplicitParagraphNode();
                pageContentNode.append(newParagraph);
                validateParagraphNode(newParagraph);
                implParagraphChanged(newParagraph);
                // in Internet Explorer it is necessary to add new empty text nodes in paragraph again
                // newly, also in other browsers (new jQuery version?)
                repairEmptyTextNodes(newParagraph);
            }

            // setting text selection to specified 'selectionState' or 'lastOperationEnd' after undo/redo.
            // -> Spreadsheet can also use lastOperationEnd, until a better process is available.
            //
            // TODO: Because in the Spreadsheet app the function isSlideMode() returns 'true' like in the
            //       Presentation app, the existence of the Presentation specific function 'getActiveSlide'
            //       is used to exclude only the Presentation App in the following block. This requires a
            //       better separator for the applications in the future.
            //       The Spreadsheet App still requires the usage of 'lastOperationEnd' to avoid problems
            //       with undo, if text is inserted into a text shape.
            if (!self.getActiveSlide && !preventSelectionChange) {

                // trying to get a selection state for a redo operation (not relying on lastOperationEnd)
                if (!useSelectionState && isRedo && !_.isEmpty(operations)) {
                    if (_.every(operations, function (op) { return op.name === Operations.SET_ATTRIBUTES; })) { // only setAttributes operations
                        selectionState = { start: _.first(operations).start, end: Position.increaseLastIndex(_.last(operations).end) };
                        useSelectionState = true;
                    }
                }

                // using the predefined selection state, if it is specified
                if (useSelectionState) {
                    selection.setTextSelection(selectionState.start, selectionState.end);
                } else if (lastOperationEnd) {

                    if (_.browser.IE) {
                        // 29397: Not using implicit paragraph in IE
                        para = Position.getLastNodeFromPositionByNodeName($rootNode, lastOperationEnd, DOM.PARAGRAPH_NODE_SELECTOR);
                        if (DOM.isImplicitParagraphNode(para) && para.previousSibling) {
                            tempLastOperationEnd = Position.getLastPositionInParagraph($rootNode, Position.getOxoPosition($rootNode, para.previousSibling, 0));
                        }

                        if (tempLastOperationEnd) { lastOperationEnd = _.copy(tempLastOperationEnd); }
                    }

                    // undo might have inserted a table -> checking lastOperationEnd (Firefox)
                    if (!Position.getLastNodeFromPositionByNodeName($rootNode, lastOperationEnd, DOM.PARAGRAPH_NODE_SELECTOR)) {
                        lastOperationEnd = Position.getFirstPositionInParagraph($rootNode, lastOperationEnd);
                    }

                    // setting selection before calling 'implParagraphChangedSync' (35350)
                    if (lastOperationEnd) { selection.setTextSelection(lastOperationEnd); }

                    // Avoiding cursor jumping, if new paragraph is empty (32454), indicated by setting cursor to position ending with '0'
                    if (_.last(lastOperationEnd) === 0) {
                        para = para || Position.getLastNodeFromPositionByNodeName($rootNode, lastOperationEnd, DOM.PARAGRAPH_NODE_SELECTOR);
                        if (para && !DOM.isSlideNode(para)) { implParagraphChangedSync($(para)); }
                    }
                }
            }

            // check if it is undo/redo operation of drawing rotate, and update corespodning resize mouse pointers
            rotationOps = _.find(operations, function (a) { return a.attrs && a.attrs.drawing && !_.isUndefined(a.attrs.drawing.rotation); });
            if (rotationOps) {
                var selectedDrawings = selection.getAllDrawingsInSelection();
                if (selectedDrawings) {
                    _.each(selectedDrawings, function (drawingNode) {
                        var angle = DrawingFrame.getDrawingRotationAngle(self, drawingNode) || 0;
                        DrawingFrame.updateResizersMousePointers(drawingNode, angle);
                    });
                }
            }

            // this must be executed, never leave function before!
            undoRedoRunning = false;

            // after undo/redo, start formatting the paragraph colors and border
            self.startParagraphMarginColorAndBorderFormating({ filledAtUndoRedo: true });

            // Text only: Check for implicit paragraphs in shapes (53983)
            if (!app.isODF() && app.isTextApp()) { self.checkInsertedEmptyTextShapes(operations); }

            // hide old/invalid change track popups after undo/redo (not valid for all applications)
            if (_.isFunction(app.getView().getChangeTrackPopup)) { app.getView().getChangeTrackPopup().hide(); }
        }

        /**
         * Callback for the listener 'specialFieldsCT:before'.
         * Finds special fields among change-tracked nodes, and restores them to original state before applying operations.
         */
        function specialFieldsCTbeforeHandler(event, trackingNodes) {

            var complexFieldChildren = null;
            var nodeIndex = 0;

            _.each(trackingNodes, function (trackingNode, index) {
                if (DOM.isSpecialField(trackingNode)) {
                    var firstChild = $(trackingNode).children().first();
                    var insertChild = DOM.isChangeTrackNode(firstChild) && !$(trackingNodes).is(firstChild); // change tracked node, that is not inside the collection
                    if (insertChild) { trackingNodes.splice(index + 1, 0, Utils.getDomNode(firstChild)); }
                    if ($(trackingNode).children().length > 1) { complexFieldChildren = $(trackingNode).children(); } // saving children before restauration of complex field
                    fieldManager.restoreSpecialField(trackingNode);
                    fieldManager.addToTempSpecCollection(trackingNode);
                } else if (DOM.isSpecialField(trackingNode.parentNode)) {
                    var fieldNode = trackingNode.parentNode;
                    if ($(fieldNode).children().length > 1) { complexFieldChildren = $(fieldNode).children(); } // saving children before restauration of complex field
                    fieldManager.restoreSpecialField(fieldNode);
                    fieldManager.addToTempSpecCollection(fieldNode);
                }
            });

            // the child spans of the complex field need to be removed from the collection of tracking nodes
            // -> except the first one that was moved outside the complex field node
            if (complexFieldChildren) {
                _.each(complexFieldChildren, function (node, index) {
                    if (index > 0) { // not deleting the first child
                        nodeIndex = trackingNodes.index(node);
                        trackingNodes.splice(nodeIndex, 1);
                    }
                });
            }
        }

        /**
         * Callback for the listener 'specialFieldsCT:after'.
         * Finds special fields among change-tracked nodes, and returns them to special state after applying operations.
         */
        function specialFieldsCTafterHandler() {
            _.each(fieldManager.getSpecFieldsFromTempCollection(), function (trackingNode) {
                var instruction = DOM.getComplexFieldInstruction(trackingNode);
                var fieldType = (/NUMPAGES/i).test(instruction) ? 'NUMPAGES' : 'PAGE';
                var id = DOM.getComplexFieldId(trackingNode);
                var target = fieldManager.getMarginalComplexFieldsTarget(id);

                fieldManager.convertToSpecialField(id, target, fieldType);
            });
            fieldManager.emptyTempSpecCollection();
        }

        /**
         * Collecting data about event that triggered inserting page breaks
         *
         * @param {HTMLElement|jQuery} currentNode
         *  DOM node where the cursor position is - we process DOM from that element downward
         *
         * @param {jQuery} [textFrameNode]
         *  An optional text frame node.
         */
        function registerPageBreaksInsert(currentNode, textFrameNode) {

            // guarantee, that node is a direct child of page content node
            if (textFrameNode && !DOM.isChildOfPageContentNode(currentNode) && !DOM.isChildOfMarginalContentNode(currentNode)) {
                currentNode = $(currentNode).parentsUntil(DOM.PAGECONTENT_NODE_SELECTOR, DOM.CONTENT_NODE_SELECTOR, DOM.MARGINALCONTENT_NODE_SELECTOR).last();
            }

            currentProcessingNodeCollection.push(currentNode);
            if (textFrameNode && $(textFrameNode).length > 0) { currentProcessingTextFrameNode = textFrameNode; }
        }

        /**
         * Collecting nodes for update of all other header/footer nodes of same type
         *
         * @param {jQuery} targetNode
         *  header or footer node that is beeing updated
         *
         */
        function registerTargetNodesUpdate(targetNode) {
            if (targetNode) { targetNodesForUpdate = targetNodesForUpdate.add(targetNode); }
        }

        /**
         * Distribute collected target header-footer nodes for updating.
         */
        function sendTargetNodesForUpdate() {
            _.each(targetNodesForUpdate, function (targetNode) {
                pageLayout.updateEditingHeaderFooter({ givenNode: targetNode });
            });
            targetNodesForUpdate = $();
        }

        /**
         * Collecting nodes inside first row of table with repeating row property.
         * This unique collection is used in deffered callback to update connected cloned rows inside table.
         *
         * @param {jQuery} node
         *  header or footer node that is beeing updated
         *
         */
        function registerRepeatedNodesUpdate(node) {
            if (node) { tableRepeatNodes = targetNodesForUpdate.add(node); }
        }

        /**
         * Distribute collected nodes inside first row of table with repeating row property, to the cloned rows of that table
         */
        function sendRepeatedNodesForUpdate() {
            _.each(tableRepeatNodes, function (node) {
                pageLayout.updateRepeatedTableRows(node);
            });
            tableRepeatNodes = $();
        }

        /**
         * Inserts a collaborative overlay element.
         */
        function insertCollaborativeOverlay() {
            editdiv.append($('<div>').addClass('collaborative-overlay').attr('contenteditable', false));
        }

        /**
         * Inserts an overlay element for the drawing selection.
         */
        function insertDrawingSelectionOverlay() {
            editdiv.append($('<div>').addClass('drawingselection-overlay').attr('contenteditable', false));
        }

        /**
         * Shows a warning dialog with Yes/No buttons before deleting document
         * contents that cannot be restored.
         *
         * @param {String} title
         *  The title of the dialog box.
         *
         * @param {String} message
         *  The message text shown in the dialog box.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the Yes button has been pressed,
         *  or rejected if the No button has been pressed.
         */
        function showDeleteWarningDialog(title, message) {

            // disable modifications of div.page node (bug 36435)
            editdiv.attr('contenteditable', false);

            // show the query dialog
            var promise = app.getView().showQueryDialog(title, message, { width: 650 });

            // enable modifications of div.page node (bug 36435)
            return promise.always(function () {
                editdiv.attr('contenteditable', true);
            });
        }

        /**
         * Clean up after the busy mode of a local client with edit prileges. After the
         * asynchronous operations were executed, the busy mode can be left.
         */
        function leaveAsyncBusy() {
            if (!app.isInQuit()) {
                // closing the dialog
                app.getView().leaveBusy().grabFocus();
                // end of gui triggered operation
                self.setGUITriggeredOperation(false);
                // allowing keyboard events again
                self.setBlockKeyboardEvent(false);
                // always leaving clip board paste mode, also this is not always required
                setClipboardPasteInProgress(false);
                // restore the original selection (this is necessary for directly following key press)
                selection.restoreBrowserSelection();
            }
        }

        /**
         * Asynchronous execution of operations from the local client with edit privileges.
         * This is necessary for the operations that depend on a (huge) selection. This is
         * very important for setAttributes() and deleteSelected(). At least the applying
         * of operations needs to be done asynchronously.
         *
         * @param {OperationGenerator} generator
         *  The generated operations.
         *
         * @param {String} label
         *  The label that is shown while applying the operations.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.showProgress=true]
         *      Whether the progress bar shall be handled inside this function. This is not
         *      necessary, if there is already an outer function that handles the progress
         *      bar. The latter is for example the case for deleting the selection before
         *      pasting the content of the external clipboard.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after all actions have been applied
         *  successfully, or rejected immediately after applying an action has
         *  failed. See editModel.applyActions() for more information.
         */
        function applyTextOperationsAsync(generator, label, options) {

            var // the promise for the asychronous execution of operations
                operationsPromise = null,
                // whehter the visibility of the progress bar shall be handled by this function
                // -> this is typically not necessary during pasting, because there is already a visible progress bar
                showProgress = Utils.getBooleanOption(options, 'showProgress', true),
                // closing progress bar at the end, if not a user abort occurred
                leaveOnSuccess = Utils.getBooleanOption(options, 'leaveOnSuccess', false),
                // an optional start value for the progress bar
                progressStart = Utils.getNumberOption(options, 'progressStart', 0);

            // setting a default label, if not specified from caller
            if (!label) { label = gt('Sorry, applying changes will take some time.'); }

            // blocking keyboard input during applying of operations
            self.setBlockKeyboardEvent(true);

            // Wait for a second before showing the progress bar (entering busy mode).
            // Users applying minimal or 'normal' amount of change tracks will not see this progress bar at all.
            if (showProgress) {
                app.getView().enterBusy({
                    cancelHandler: function () {
                        if (operationsPromise) { operationsPromise.abort(); }
                    },
                    delay: 1000,
                    warningLabel: label
                }).grabFocus();
            }

            // fire apply actions asynchronously
            operationsPromise = self.applyOperations(generator, { async: true });

            // handle the result of change track operations
            operationsPromise.progress(function (progress) {
                // update the progress bar according to progress of the operations promise
                app.getView().updateBusyProgress(progressStart + progress * (1 - progressStart));
            });

            // handler for 'always', that is not triggered, if self is in destruction (document is closed, 42567)
            self.waitForAny(operationsPromise, function () {
                if (showProgress || leaveOnSuccess) { leaveAsyncBusy(); }
            });

            return operationsPromise;
        }

        // End of private functions -----------------------------------------------------

        // Initialization (registering the handler functions for the operations) --------

        this.registerContextOperationHandler(Operations.SET_DOCUMENT_ATTRIBUTES, function (context) {

            // the attribute set from the passed operation
            var attributes = _.clone(context.getObj('attrs'));

            //only undo for page attributes
            if (attributes.page) {
                var oldPageAttributes = pageStyles.getElementAttributes(editdiv).page;
                for (var key in oldPageAttributes) {
                    if (!(key in attributes.page)) {
                        delete oldPageAttributes[key];
                    }
                }
                var undo = { name: Operations.SET_DOCUMENT_ATTRIBUTES, attrs: { page: oldPageAttributes } };
                undoManager.addUndo(undo, context.operation);
            }

            this.applySetDocumentAttributesOperation(context);

            // update local pageAttributes
            self.setPageAttributes(pageStyles.getElementAttributes(editdiv));
        });

        this.registerOperationHandler(Operations.MOVE, function (operation) {
            if (undoManager.isUndoEnabled()) {
                // Todo: Ignoring 'end', only 'start' === 'end' is supported
                var undoOperation = { name: Operations.MOVE, start: operation.to, end: operation.to, to: operation.start };

                extendPropertiesWithTarget(undoOperation, operation.target);
                undoManager.addUndo(undoOperation, operation);
            }
            return implMove(operation.start, operation.end, operation.to, operation.target);
        });

        this.registerContextOperationHandler(Operations.TEXT_INSERT, function (context) {

            var start = context.getPos('start');
            var text = context.getStr('text');
            var attrs = context.getOptObj('attrs');
            var target = context.getOptStr('target');
            context.ensure(implInsertText(start, text, attrs, target, context.external), 'cannot insert text');

            if (undoManager.isUndoEnabled()) {
                var end = Position.increaseLastIndex(start, text.length - 1);
                var undoOperation = { name: Operations.DELETE, start: start, end: end };

                extendPropertiesWithTarget(undoOperation, target);
                undoManager.addUndo(undoOperation, context.operation);
            }
        });

        this.registerOperationHandler(Operations.FIELD_INSERT, function (operation, external) {

            if (!implInsertField(operation.start, operation.type, operation.representation, operation.attrs, operation.target, external)) {
                return false;
            }

            if (undoManager.isUndoEnabled()) {
                var undoOperation = { name: Operations.DELETE, start: operation.start };

                extendPropertiesWithTarget(undoOperation, operation.target);
                undoManager.addUndo(undoOperation, operation);
            }
            return true;
        });

        this.registerOperationHandler(Operations.FIELD_UPDATE, function (operation) {

            if (!implUpdateField(operation.start, operation.type, operation.representation, operation.attrs, operation.target)) {
                return false;
            }
            // undo is added in impl function
            return true;
        });

        this.registerOperationHandler(Operations.TAB_INSERT, function (operation, external) {

            if (!implInsertTab(operation.start, operation.attrs, operation.target, external)) {
                return false;
            }

            if (undoManager.isUndoEnabled()) {
                var undoOperation = { name: Operations.DELETE, start: operation.start };
                extendPropertiesWithTarget(undoOperation, operation.target);
                undoManager.addUndo(undoOperation, operation);
            }
            return true;
        });

        this.registerOperationHandler(Operations.INSERT_DRAWING, function (operation) {

            if (!implInsertDrawing(operation.type, operation.start, operation.attrs, operation.target)) {
                return false;
            }

            if (undoManager.isUndoEnabled()) {
                // not registering for undo if this is a drawing inside a drawing group (36150)
                var undoOperation = Position.isPositionInsideDrawingGroup(self.getRootNode(operation.target), operation.start) ? null : { name: Operations.DELETE, start: operation.start };

                if (undoOperation) {
                    extendPropertiesWithTarget(undoOperation, operation.target);
                }
                undoManager.addUndo(undoOperation, operation);
            }
            return true;
        });

        this.registerOperationHandler(Operations.INSERT_LIST, function (operation) {
            undoManager.addUndo({ name: Operations.DELETE_LIST, listStyleId: operation.listStyleId }, operation);
            listCollection.insertList(operation);
        });

        this.registerOperationHandler(Operations.DELETE_LIST, function (operation) {
            // no Undo, cannot be removed by UI
            listCollection.deleteList(operation.listStyleId);
        });

        this.registerOperationHandler(Operations.INSERT_HEADER_FOOTER, function (operation) {
            if (undoManager.isUndoEnabled()) {
                var undoOperation = { name: Operations.DELETE_HEADER_FOOTER, id: operation.id };

                undoManager.addUndo(undoOperation, operation);
            }
            return implInsertHeaderFooter(operation.id, operation.type, operation.attrs);
        });

        this.registerOperationHandler(Operations.COMMENT_INSERT, function (operation) {

            var // the undo operation
                undoOperation = null;

            if (!implInsertComment(operation.start, operation.id, operation.author, operation.uid, operation.date, operation.target)) {
                return false;
            }

            if (undoManager.isUndoEnabled()) {
                undoOperation = { name: Operations.DELETE, start: operation.start };
                extendPropertiesWithTarget(undoOperation, operation.target);
                undoManager.addUndo(undoOperation, operation);
            }
            return true;
        });

        this.registerOperationHandler(Operations.RANGE_INSERT, function (operation) {

            var // the undo operation
                undoOperation = null;

            if (!implInsertRange(operation.start, operation.id, operation.type, operation.position, operation.attrs, operation.target)) {
                return false;
            }

            if (undoManager.isUndoEnabled()) {
                undoOperation = { name: Operations.DELETE, start: operation.start };
                extendPropertiesWithTarget(undoOperation, operation.target);
                undoManager.addUndo(undoOperation, operation);
            }
            return true;
        });

        this.registerOperationHandler(Operations.COMPLEXFIELD_INSERT, function (operation) {

            var // the undo operation
                undoOperation = null;

            if (!implInsertComplexField(operation.start, operation.instruction, operation.attrs, operation.target)) {
                return false;
            }

            if (undoManager.isUndoEnabled()) {
                undoOperation = { name: Operations.DELETE, start: operation.start };
                extendPropertiesWithTarget(undoOperation, operation.target);
                undoManager.addUndo(undoOperation, operation);
            }
            return true;
        });

        this.registerOperationHandler(Operations.COMPLEXFIELD_UPDATE, function (operation) {

            if (!implUpdateComplexField(operation.start, operation.instruction, operation.attrs, operation.target)) {
                return false;
            }
            // undo is added in impl function

            return true;
        });

        this.registerOperationHandler(Operations.BOOKMARK_INSERT, function (operation) {

            var // the undo operation
                undoOperation = null;

            if (!implInsertBookmark(operation.start, operation.id, operation.anchorName, operation.position, operation.attrs, operation.target)) {
                return false;
            }

            if (undoManager.isUndoEnabled()) {
                undoOperation = { name: Operations.DELETE, start: operation.start };
                extendPropertiesWithTarget(undoOperation, operation.target);
                undoManager.addUndo(undoOperation, operation);
            }
            return true;
        });

        this.registerOperationHandler(Operations.DELETE_HEADER_FOOTER, function (operation) {

            var generator = undoManager.isUndoEnabled() ? this.createOperationGenerator() : null,
                containerNode = self.getRootNode(operation.id),
                // must be empty, so that generateContentOperations makes operations for all content inside node
                position = [],
                options = null,
                type = null,
                undoOperation = null;

            // undo/redo generation
            if (undoManager.isUndoEnabled()) {
                if (generator) {
                    options = { target: operation.id };

                    containerNode.children('.cover-overlay').remove();
                    // check if there are special fields for restoring before generating op
                    fieldManager.checkRestoringSpecialFieldsInContainer(containerNode);

                    generator.generateContentOperations(containerNode, position, options);
                    undoManager.addUndo(generator.getOperations());
                }

                type = pageLayout.getTypeFromId(operation.id);
                if (type) {
                    undoOperation = { name: Operations.INSERT_HEADER_FOOTER, id: operation.id, type: type };
                    undoManager.addUndo(undoOperation, operation);
                } else {
                    return;
                }

            }

            return implDeleteHeaderFooter(operation.id);
        });

        this.registerOperationHandler(Operations.TABLE_INSERT, self.insertTableHandler);
        this.registerOperationHandler(Operations.ROWS_INSERT, self.insertRowHandler);
        this.registerOperationHandler(Operations.COLUMN_INSERT, self.insertColumnHandler);
        this.registerOperationHandler(Operations.SET_ATTRIBUTES, self.setAttributesHandler);
        this.registerOperationHandler(Operations.DELETE, self.deleteHandler);
        this.registerOperationHandler(Operations.COLUMNS_DELETE, self.deleteColumnsHandler);
        this.registerOperationHandler(Operations.TABLE_SPLIT, self.tableSplitHandler);
        this.registerOperationHandler(Operations.TABLE_MERGE, self.tableMergeHandler);
        this.registerOperationHandler(Operations.CELL_MERGE, self.cellMergeHandler);
        this.registerOperationHandler(Operations.CELLS_INSERT, self.cellInsertHandler);
        this.registerOperationHandler(Operations.PARA_SPLIT, self.splitParagraphHandler);
        this.registerOperationHandler(Operations.PARA_MERGE, self.mergeParagraphHandler);
        this.registerContextOperationHandler(Operations.PARA_INSERT, self.insertParagraphHandler);
        this.registerOperationHandler(Operations.HARDBREAK_INSERT, self.insertHardBreakHandler);
        this.registerOperationHandler(Operations.GROUP, self.groupDrawingsHandler);
        this.registerOperationHandler(Operations.UNGROUP, self.ungroupDrawingsHandler);

        // initialization -----------------------------------------------------

        // setting the undo manager
        undoManager = this.getUndoManager();

        // creating several new objects
        selection = new Selection(this);
        remoteSelection = new RemoteSelection(this);
        changeTrack = new ChangeTrack(app, editdiv, self);
        drawingLayer = new DrawingLayer(app, editdiv);
        commentLayer = new CommentLayer(app, editdiv);
        rangeMarker = new RangeMarker(app, editdiv);
        pageLayout = new PageLayout(this, editdiv);
        searchHandler = new SearchHandler(app, this);
        spellChecker = new SpellChecker(this, initOptions);

        // initialize document contents
        app.onInit(initDocument);

        // Creating debounced function for updating drawing nodes.
        // Info: This function is important for performance reasons.
        // Example: After modifying the paragraphs inside a drawing, 'implParagraphChangedSync' is called for every paragraph.
        //          This function triggers an update of the surrounding drawing. But this should happen only once, not five
        //          times.
        updateDrawingsDebounced = (function () {

            var // all drawing nodes that need to be updated
                allDrawings = $();

            // Direct callback: Called every time when implParagraphChanged() has been called.
            // Parameter 'paragraph': HTMLElement|jQuery|Number[]
            // The paragraph element as DOM node or jQuery object, or the logical position of the paragraph or any of its child components.
            function registerDrawingNode(drawingNode) {
                // store the drawing in the collection (jQuery keeps the collection unique)
                if (drawingNode) {
                    allDrawings = allDrawings.add(drawingNode);
                }
            }

            // deferred callback: called once, after current script ends
            function updateAllDrawings() {

                var modifiedDrawings = app.isPresentationApp() ? [] : null;

                _.each(allDrawings, function (oneDrawing) {

                    if (!DOM.isCommentNode(oneDrawing)) {

                        if (DrawingFrame.isAutoResizeHeightDrawingFrame(oneDrawing)) {
                            updateDrawings(null, $(oneDrawing));
                            if (DrawingFrame.isGroupedDrawingFrame(oneDrawing)) {
                                DrawingFrame.updateDrawingGroupHeight(app, oneDrawing);
                            }
                            if (modifiedDrawings) { modifiedDrawings.push(oneDrawing); }
                        } else if (self.useSlideMode()) {
                            if (DrawingFrame.isAutoTextHeightDrawingFrame(oneDrawing)) {
                                // updating the font size dynamically inside the drawing node (only supported in slide mode)
                                if (self.updateDynFontSizeDebounced && !self.isClipboardPasteInProgress()) {
                                    self.updateDynFontSizeDebounced(oneDrawing);
                                    if (modifiedDrawings) { modifiedDrawings.push(oneDrawing); }
                                }
                            } else if (DrawingFrame.isGroupDrawingFrame(oneDrawing)) {
                                // generic formatting of group drawing frame after change of child drawing attributes
                                DrawingFrame.updateFormatting(app, oneDrawing, drawingStyles.getElementAttributes(oneDrawing));
                                if (modifiedDrawings) { modifiedDrawings.push(oneDrawing); }
                            }
                        } else if (app.isTextApp()) {
                            var attrs = AttributeUtils.getExplicitAttributes(oneDrawing);
                            var geometryAttrs = (attrs && attrs.geometry) || null;
                            DrawingFrame.handleTextframeOverflow(self, $(oneDrawing), app.isODF(), geometryAttrs);
                        }
                    }
                });

                // updating the side pane might be required in Presentation app
                if (modifiedDrawings && modifiedDrawings.length > 0) { self.updateAllAffectedSlidesInSlidePane(modifiedDrawings); }

                allDrawings = $(); // reset the collector
            }

            // create and return the debounced method
            return self.createDebouncedMethod('Editor.updateDrawingsDebounced', registerDrawingNode, updateAllDrawings, { delay: 100, maxDelay: 500 });
        }());

        // initialize selection after import
        this.waitForImport(function () {
            if (!self.useSlideMode()) { selection.selectDocumentLoadPosition(); } // in presentation a 'slide selection' is set
            app.getView().grabFocus({ afterImport: true });
        });

        // more initialization after successful import
        this.waitForImportSuccess(documentLoaded);

        // disable browser editing capabilities if import fails
        this.waitForImportFailure(function () {
            editdiv.attr('contenteditable', false);
        });

        // Generating 'implParagraphChanged' after successful document load.
        // Has to be called every time after a paragraph was changed.
        this.waitForImportSuccess(function () {

            var // all paragraph nodes that need to be updated
                paragraphs = $();

            // Direct callback: Called every time when implParagraphChanged() has been called.
            // Parameter 'paragraph': HTMLElement|jQuery|Number[]
            // The paragraph element as DOM node or jQuery object, or the logical position of the paragraph or any of its child components.
            function registerParagraph(paragraph) {
                if (_.isArray(paragraph)) {
                    paragraph = Position.getCurrentParagraph(self.getCurrentRootNode(), paragraph);
                }
                // store the new paragraph in the collection (jQuery keeps the collection unique)
                if (paragraph) {
                    // reset to 'not spelled'
                    spellChecker.resetDirectly(paragraph);
                    paragraphs = paragraphs.add(paragraph);
                }
            }

            // deferred callback: called once, after current script ends
            function updateParagraphs() {
                implParagraphChangedSync(paragraphs);
                paragraphs = $();
            }

            // create and return the deferred implParagraphChanged() method
            implParagraphChanged = self.createDebouncedMethod('Editor.implParagraphChanged', registerParagraph, updateParagraphs, { delay: 200, maxDelay: 1000 });
        });

        // Generating 'implTableChanged' after successful document load.
        // Has to be called every time after a table was changed. It recalculates the position
        // of each cell in the table and sets the corresponding attributes. This can be set for
        // the first or last column or row, or even only for the south east cell.
        this.waitForImportSuccess(function () {

            var // all table nodes that need to be updated
                tables = $();

            // direct callback: called every time when implTableChanged() has been called
            // @param {HTMLTableElement|jQuery} table
            function registerTable(table) {
                // store the new table in the collection (jQuery keeps the collection unique)
                tables = tables.add(table);
            }

            // deferred callback: called once, after current script ends
            function updateTables() {
                tables.each(function () {
                    var // the table node
                        table = this;
                    // the table may have been removed from the DOM in the meantime
                    // -> just checking, if the table is still somewhere below the page
                    if (Utils.containsNode(editdiv, table)) {
                        tableStyles.updateElementFormatting(table)
                            .done(function () { self.trigger('tableUpdate:after', table); });
                    }
                });
                tables = $();
            }

            // create and return the deferred implTableChanged() method
            implTableChanged = self.createDebouncedMethod('Editor.implTableChanged', registerTable, updateTables);
        });

        if (!self.useSlideMode()) {

            this.waitForImportSuccess(function () {

                // defining function for debounced setting of page breaks
                insertPageBreaksDebounced = self.createDebouncedMethod('Editor.insertPageBreaksDebounced', registerPageBreaksInsert, function () {
                    if (currentProcessingTextFrameNode) { self.trigger('drawingHeight:update', $(currentProcessingTextFrameNode)); }
                    currentProcessingTextFrameNode = null;
                    if (self.getBlockOnInsertPageBreaks()) { // if operation is inside header/footer, block page breaks. They will be triggered from other function if necessary.
                        self.setBlockOnInsertPageBreaks(false);
                        self.trigger('update:absoluteElements');  // updating comments and absolutely positioned drawings
                    } else {
                        pageLayout.insertPageBreaks();
                    }
                    // If you increase the delay you must increase the delay of the renderCollaborativeSelectionDebounced delay of the remoteselection
                }, { delay: 200 });

                // Debouncing distribution of updated data from editing node to all other headers&footers with same type
                updateEditingHeaderFooterDebounced = self.createDebouncedMethod('Editor.updateEditingHeaderFooterDebounced', registerTargetNodesUpdate, sendTargetNodesForUpdate, { delay: 200 });

                // Debouncing updating of repeated rows content inside table
                updateRepeatedTableRowsDebounced = self.createDebouncedMethod('Editor.updateRepeatedTableRowsDebounced', registerRepeatedNodesUpdate, sendRepeatedNodesForUpdate, { delay: 500 });

                // public handler function for updating comment ranges debounced. This is
                // required, if a comment range is visualized (with hover) and then it is
                // written inside this comment range. This happens after the event
                // 'paragraphUpdate:after' was triggered.
                updateHighligtedRangesDebounced = self.createDebouncedMethod('Editor.updateHighligtedRangesDebounced', null, rangeMarker.updateHighlightedRanges, { delay: 100 });
            });

        } else {

            this.waitForImportSuccess(function () {
                // defining function for debounced reacting of height changes of content inside text frames
                // -> in slide mode, it is not necessary to update page breaks, but to trigger the event 'drawingHeight:update'
                insertPageBreaksDebounced = self.createDebouncedMethod('Editor.insertPageBreaksDebounced', registerPageBreaksInsert, function () {
                    if (currentProcessingTextFrameNode) { self.trigger('drawingHeight:update', $(currentProcessingTextFrameNode)); }
                    currentProcessingTextFrameNode = null;
                }, { delay: 200 });
            });
        }

        // restore selection after undo/redo operations
        undoManager.on('undo:before redo:before', function () {
            undoRedoRunning = true;
            self.setBlockKeyboardEvent(true); // block keyboard events while processing undo operations
        });

        // restore selection after undo/redo operations
        undoManager.on('undo:after redo:after', undoRedoAfterHandler);

        // undogroup:open : checking if the operations in a group can be undone
        undoManager.on('undogroup:open', function () {
            deleteUndoStack = false;  // default: undo stack must not be deleted
            isInUndoGroup = true;
        });

        // undogroup:closed : delete undo stack, if required
        undoManager.on('undogroup:close', function () {
            if (deleteUndoStack) { undoManager.clearUndoActions(); }
            deleteUndoStack = false;
            isInUndoGroup = false;
        });

        // remove highlighting before changing the DOM which invalidates the positions in highlightRanges
        this.on('operations:before', function (event, operations, external) {

            // checking operations for change track attributes
            if (!changeTrack.isSideBarHandlerActive() && changeTrack.hasChangeTrackOperation(operations)) {
                self.trigger('changeTrack:stateInfo', { state: true });
            }

            // reset read-only paragraph
            if (roIOSParagraph) {
                if (app.isEditable()) { $(roIOSParagraph).removeAttr('contenteditable'); }
                roIOSParagraph = null;
            }

            // clear preselected attributes before applying any operations
            if (!keepPreselectedAttributes) { preselectedAttributes = null; }

            // Insert drawing operations with a document URL as imageUrl attribute also have the imageData attribute set.
            // Before the operation is applied we need to determine which attribute to use and which to remove.
            if (!external) {
                _(operations).each(function (operation) {
                    if (operation.name === Operations.INSERT_DRAWING && (operation.type === 'image' || operation.type === 'ole') && operation.attrs && operation.attrs.image) {
                        ImageUtils.postProcessOperationAttributes(operation.attrs.image, app.getFileDescriptor().id);
                    } else if (operation.name === Operations.INSERT_DRAWING && operation.type === 'shape' && operation.attrs && operation.attrs.fill && operation.attrs.fill.type === 'bitmap') {
                        ImageUtils.postProcessBackgroundImageOperationAttributes(operation.attrs.fill, app);
                    }
                });
            }
        });

        this.on('operations:success', function (event, operations) {
            // saving information about the type of operation
            useSelectionRangeInUndo = (_.last(operations).name === Operations.SET_ATTRIBUTES);

            // useSelectionRangeInUndo also needs to be true, if after setAttributes an operation like deleteListStyle comes (32005)
            // -> maybe there needs to be a list of operations that can be ignored like 'Operations.DELETE_LIST' following the setAttributes operation
            // -> leaving this simple now because of performance reasons
            if ((!useSelectionRangeInUndo) && (operations.length > 1) && (_.last(operations).name === Operations.DELETE_LIST) && (operations[operations.length - 2].name === Operations.SET_ATTRIBUTES)) {
                useSelectionRangeInUndo = true;
            }

            // update change track sidebar after successfully applied operation
            self.updateChangeTracksDebounced();

            // In read-only mode update the selection after an external operation (and after the document is loaded successfully)
            // -> maybe this need to be deferred
            if (!app.isEditable() && self.isImportFinished()) {
                updatingRemoteSelection = true;
                selection.updateRemoteSelection();
                updatingRemoteSelection = false;
            }
        });

        // Register the listener for the 'changeTrack:stateInfo' event. This event is
        // triggered with the option 'state: true' after loading the document and the
        // document contains change tracked elements or after receiving an operation
        // with change track attributes.
        // The event is triggered with 'state: false', if no change tracked element was
        // found inside the document.
        this.on('changeTrack:stateInfo', function (event, options) {

            var // whether the document contains change tracking content
                state = Utils.getBooleanOption(options, 'state', false);

            if (state) {
                if (!changeTrack.isSideBarHandlerActive() && !Utils.SMALL_DEVICE) {
                    changeTrack.setSideBarHandlerActive(true);
                    self.updateChangeTracksDebounced = self.createDebouncedMethod('Editor.updateChangeTracksDebounced', null, function () { changeTrack.updateSideBar({ invalidate: true }); }, { delay: 500, maxDelay: 1500 });
                    self.updateChangeTracksDebouncedScroll = self.createDebouncedMethod('Editor.updateChangeTracksDebouncedScroll', null, function () { changeTrack.updateSideBar({ invalidate: false }); }, { delay: 500, maxDelay: 1500 });
                }
            } else {
                if (changeTrack.isSideBarHandlerActive()) {
                    changeTrack.setSideBarHandlerActive(false);
                    self.updateChangeTracksDebounced = $.noop;
                    self.updateChangeTracksDebouncedScroll = $.noop;
                }
            }

        });

        // Register the listener for the events 'paragraphUpdate:after' and
        // 'tableUpdate:after'. These events are triggered, when the promise
        // updating of paragraphs or tables has finished. The listener receives
        // the formatted node. This is required, because of task 36495. The
        // parameter nodes contains one node or a list of nodes that can be handled
        // in this event handler (Node|Node[]|jQuery).
        this.on('paragraphUpdate:after tableUpdate:after drawingHeight:update', function (event, nodes, options) {

            _.each($(nodes), function (node) {

                var drawingNode = null;

                if (event.type === 'paragraphUpdate:after' && Utils.getBooleanOption(options, 'paragraphChanged', false) && DOM.isNodeInsideTextFrame(node)) {
                    drawingNode = DrawingFrame.getDrawingNode(node);
                    updateDrawingsDebounced(drawingNode); // updating the drawing debounced, not for every paragraph inside the drawing synchronously
                } else if (DOM.isImageDrawingNode(node) || DrawingFrame.isCanvasNode(node) || DOM.isNodeInsideTextFrame(node) || DrawingFrame.isShapeDrawingFrame(node)) {
                    drawingNode = DrawingFrame.getDrawingNode(node);
                    if (!DOM.isCommentNode(drawingNode)) {
                        updateDrawings(event, drawingNode);
                        if (DrawingFrame.isGroupedDrawingFrame(drawingNode) && DrawingFrame.isAutoResizeHeightDrawingFrame(drawingNode)) {
                            DrawingFrame.updateDrawingGroupHeight(app, drawingNode);
                        }

                        // updating the font size dynamically inside the drawing node (only supported in slide mode)
                        if (self.updateDynFontSizeDebounced && !self.isClipboardPasteInProgress() && !Utils.getBooleanOption(options, 'suppressFontSize', false)) {
                            self.updateDynFontSizeDebounced(drawingNode);
                        }
                    }
                }
                if (event.type !== 'drawingHeight:update' && pageLayout.isHeaderFooterActivated()) { // #37250 - Looping request when exit header edit mode
                    if (DOM.isMarginalNode(node)) {
                        updateEditingHeaderFooterDebounced(DOM.getMarginalTargetNode(editdiv, node));
                    } else if (DOM.isHeaderOrFooter(node)) {
                        updateEditingHeaderFooterDebounced(node);
                    } else if ($(node).find(DOM.TAB_NODE_SELECTOR).length && pageLayout.getHeaderFooterPlaceHolder().find(node).length && !self.isUndoRedoRunning()) {
                        // if updated paragraph is inside headerfooter, but in template node, and it has tabs inside,
                        // redistribute to all visible headerfooter nodes
                        pageLayout.replaceAllTypesOfHeaderFooters();
                    }
                }
                if (DOM.isCellContentNode($(node).parent()) && DOM.isFirstRowInRepeatingRowsTable($(node).closest(DOM.TABLE_REPEAT_ROWNODE_SELECTOR))) {
                    updateRepeatedTableRowsDebounced(node);
                }

                // updating an visualized (comment) range, if necessary
                updateHighligtedRangesDebounced();
            });
        });

        // Register the handler for document reload event. This happens, if the
        // user cancels a long running action.
        this.on('document:reloaded', documentReloadHandler);

        // Register the handler for the 'docs:editmode' event.
        this.listenTo(app, 'docs:editmode', changeEditModeHandler);

        // scroll to cursor, forward selection change events to own listeners
        selection.on('change', selectionChangeHandler);

        // Register selectionchange handler to get notification if user changes
        // selection via handles. No other way to detect these selection changes on
        // mobile Safari.
        if (Utils.IOS) {
            app.registerGlobalEventHandler(document, 'selectionchange', function (event) {
                var sel = window.getSelection();

                // special presentation behavior in ios
                if (self.getSlideTouchMode()) {

                    if (sel && sel.rangeCount) {
                        // Check that the selection is inside our editdiv node. We are listing to the whole document!
                        if (Utils.containsNode(editdiv, sel.getRangeAt(0).startContainer)) { selection.processBrowserEvent(event); }
                    }
                // normal behaviour
                } else {

                    if (sel && sel.rangeCount) { // bug 52999: TODO refactor conditions in this handler after release, it's the same now
                        // Check that the selection is inside our editdiv node. We are listing to the whole document!
                        if (Utils.containsNode(editdiv, sel.getRangeAt(0).startContainer)) { selection.processBrowserEvent(event); }
                    }

                    // behavoir... but there seems not to be a better way. Triggered when pressed on 'done' to close the softKeyboard in ios
                    if (sel.anchorNode === null) { self.trigger('selection:possibleKeyboardClose'); }
                }

            });
        }

        /**
         * Registering the handler for taking care of special fields BEFORE resolving changetracked nodes.
         */
        this.on('specialFieldsCT:before', specialFieldsCTbeforeHandler);

        /**
         * * Registering the handler for taking care of special fields AFTER resolving changetracked nodes.
         */
        this.on('specialFieldsCT:after', specialFieldsCTafterHandler);

        // registering keypress handler at document for Chrome to avoid problems with direct keypress to document
        // -> using app.registerGlobalEventHandler the handler is registered with 'show' and deregistered
        // with 'hide' event of the app automatically and does not influence other apps.
        if (_.browser.WebKit) { app.registerGlobalEventHandler(document, 'keypress', processKeypressOnDocument); }

        // hybrid edit mode ('mousedown touchstart' need to be registered specific for each application)
        editdiv.on(listenerList = {
            keydown: _.bind(self.getPageProcessKeyDownHandler(), self),
            keypress: _.bind(self.getPageProcessKeyPressHandler(), self),
            keyup: processKeyUp,
            compositionstart: processCompositionStart,
            compositionupdate: processCompositionUpdate,
            compositionend: processCompositionEnd,
            textInput: processTextInput,
            input: processInput,
            'mouseup touchend': processMouseUp,
            dragstart: _.bind(self.processDragStart, self),
            drop: _.bind(self.processDrop, self),
            dragover: _.bind(self.processDragOver, self),
            dragenter: _.bind(self.processDragEnter, self),
            dragleave: _.bind(self.processDragLeave, self),
            cut: _.bind(self.cut, self),
            copy: _.bind(self.copy, self),
            'paste beforepaste': _.bind(self.paste, self)     // for IE we need to handle 'beforepaste', on all other browsers 'paste'
        });

        if (Utils.SMALL_DEVICE) { editdiv.on('touchstart touchmove touchend', DOM.TABLE_NODE_SELECTOR, processTouchEventsForTableDraftMode); }

        // Fix for 29751: IE support double click for word selection
        if (_.browser.IE || _.browser.WebKit) { editdiv.on({ dblclick: processDoubleClick }); }

        // header/footer editing on doubleclick
        editdiv.on('dblclick', '.header.inactive-selection, .footer.inactive-selection, .cover-overlay', processHeaderFooterEdit);

        // user defined fields
        editdiv.on('mouseenter', '.field[class*=user-field]', pageLayout.createTooltipForField);
        editdiv.on('mouseleave', '.field[class*=user-field]', pageLayout.destroyTooltipForField);

        // mouseup events can be anywhere -> binding to $(document)
        this.listenTo($(document), 'mouseup touchend', processMouseUpOnDocument);

        // register handler for changes of page settings
        this.on('change:pageSettings', updatePageDocumentFormatting);

        // register handler for cleaning processing nodes collector for page breaks;
        // and after canceling snapshoot (#53275)
        this.on('empty:processingNodesCollector document:reloaded:after', function () {
            // empty the collection
            currentProcessingNodeCollection = [];
        });

        // if image crop mode is active on losing edit rights, exit
        self.listenTo(app, 'docs:editmode:leave', function () {
            if (selection.isCropMode()) { self.exitCropMode(); }
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            // remove page node from document
            editdiv.remove();
            // remove event handler from document
            $(document).off('keypress', processKeypressOnDocument);

            changeTrack.destroy();
            drawingLayer.destroy();
            commentLayer.destroy();
            fieldManager.destroy();
            rangeMarker.destroy();
            remoteSelection.destroy();
            selection.destroy();
            pageLayout.destroy();
            searchHandler.destroy();
            spellChecker.destroy();
            if (listCollection) { listCollection.destroy(); }

            self = changeTrack = drawingLayer = commentLayer = fieldManager = rangeMarker = selection = remoteSelection = null;
            pageLayout = searchHandler = spellChecker = listCollection = null;
        });

    } // class Editor

    // exports ================================================================

    // derive this class from class EditModel
    return EditModel.extend({ constructor: Editor });
});
