/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Mario Schroeder <mario.schroeder@open-xchange.com>
 */

define.async('io.ox/office/textframework/model/clipboardmixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/io',
    'io.ox/office/tk/render/colorutils',
    'io.ox/office/drawinglayer/utils/imageutils',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/dom',
    'static/3rd.party/unorm/unorm.js'           // Polyfill for String.prototype.normalize()
], function (Utils, IO, ColorUtils, ImageUtils, Operations, DOM) {

    'use strict';

    var OPEN_SYMBOL_REPLACEMENT = null;

    // mix-in class ClipboardMixin ============================================

    /**
     * Mix-in class for the class Editor that provides extensions for
     * dealing with clipboard functionality.
     *
     * @constructor
     *
     * @param {TextApplication} app
     *  The text application instance.
     */
    function ClipboardMixin(app) {

        var // self reference (the spreadsheet view)
            self = this,
            // regExp to parse list label text for decimal numbering
            decimalRegEx = new RegExp(/^\d+[.)]/),
            // regExp to parse list label text for roman numbering
            upperRomanRegEx = new RegExp(/^M{0,4}(CM|CD|D?C{0,3})(XC|XL|L?X{0,3})(IX|IV|V?I{0,3})[.)]/),
            lowerRomanRegEx = new RegExp(/^m{0,4}(cm|cd|d?C{0,3})(xc|xl|l?x{0,3})(ix|iv|v?i{0,3})[.)]/),
            // regExp to parse list label text for alpha numbering
            upperLetterRegEx = new RegExp(/^[A-Z]+[.)]/),
            lowerLetterRegEx = new RegExp(/^[a-z]+[.)]/);

        // private methods ----------------------------------------------------

        /**
         * Converts a list level String to a zero based number.
         *
         * @param {String} levelString
         *  The list level as one based String
         *
         * @returns {Number}
         *  The list level as zero based Number
         */
        function listLevelStringToZeroBasedLevel(levelString) {

            var level = parseInt(levelString, 10);
            level = (level > 0) ? (level - 1) : 0;

            return level;
        }

        /**
         * Removes ' and \ chars from given String.
         *
         * @param {String} text
         *  The String to clean.
         *
         * @returns {String|null}
         *  The cleaned String or null.
         */
        function cleanLevelText(text) {
            if (!_.isString(text)) { return null; }
            return text.replace(/["\\]/g, '');
        }

        function calcTableSize(child) {
            var rows        = $(child).find('tr'),
                rowCount    = rows.length,
                whileCount  = (rowCount > 50) ? 50 : rowCount,
                size        = 0,
                i           = 0;

            while (i < whileCount) {
                var row     = $(rows[i]),
                    cols    = row.find('>td,>th').length,
                    colSpan = $(row).attr('colspan') - 1;

                if (colSpan >= 1) { cols += colSpan; }

                if (cols > size) { size = cols; }

                i++;
            }

            return size;
        }

        // methods ------------------------------------------------------------

        // Text parser --------------------------------------------------------

        /**
         * Parses the pasted clipboard text.
         *
         * @param {String} text
         *
         * @returns {Array}
         *  The clipboard data array to create operations from.
         */
        this.parseClipboardText = function (text) {
            var ops = [];
            var paragraphs;

            if (_.isString(text) && (text.length > 0)) {
                // Unicode normalization
                text = text.normalize();

                paragraphs = text.split(/\r\n?|\n/);
                _.each(paragraphs, function (para) {
                    var splitted = para.match(/[^\t]+|\t/g);
                    // insert paragraph
                    ops.push({ operation: Operations.PARA_INSERT, listLevel: -1 });
                    // insert tab or text
                    _.each(splitted, function (part) {
                        if (part === '\t') {
                            ops.push({ operation: Operations.TAB_INSERT });
                        } else {
                            ops.push({ operation: Operations.TEXT_INSERT, data: part });
                        }
                    });
                });
            }

            return ops;
        };

        // HTML parser --------------------------------------------------------

        /**
         * Parses the clipboard div for pasted text content.
         *
         * @param {jQuery} clipboard
         *  The clipboard div.
         *
         * @returns {Array}
         *  The clipboard data array to create operations from.
         */
        this.parseClipboard = function (clipboard) {

            var result = [],
                fileId = app.getFileDescriptor().id,
                acceptTextNodes = true,
                skipTextNodes = false,
                // the definition off all MS list styles
                msListDefinitions = null;

            (function findTextNodes(current, depth, listLevel, tableLevel, type) {

                var // the length (of chars, tabs, drawings...) to be added to the oxo position
                    insertLength = 0,
                    // the current child node
                    child,
                    // text node content
                    text,
                    // text node content splitted by tab and text
                    splitted,
                    // additional data of the current child node
                    childData,
                    // ms list paragraph info on style id and list level
                    msParagraphLevelInfo,
                    // a single MS list style deinition
                    msListStyle,
                    // determins if the node gets parsed recursively or is skipped
                    nextLevel,
                    // contains the last operation
                    lastOp;

                for (var i = 0; i < current.childNodes.length; i++) {
                    child       = current.childNodes[i];
                    nextLevel   = true;
                    lastOp      = _.last(result);

                    if (child.nodeType === 3 && acceptTextNodes && !skipTextNodes) {
                        // handle non-whitespace characters and non-breaking spaces only
                        // except for our own text portion spans
                        if (DOM.isTextSpan(child.parentNode) || (/\S|\xa0/.test(child.nodeValue))) {
                            splitted = child.nodeValue.match(/[^\t]+|\t/g) || '';
                            for (var j = 0; j < splitted.length; j++) {
                                if (splitted[j] === '\t') {
                                    // --- tab ---
                                    result.push({ operation: Operations.TAB_INSERT, depth: depth });
                                    insertLength += 1;
                                } else {
                                    // --- text ---
                                    // Unicode normalization
                                    text = splitted[j].normalize();
                                    // replace '\r' and '\n' with space to fix pastes from aoo,
                                    // replace invisible control characters and reduce space sequences to a single spaces
                                    text = text.replace(/[\r\n]/g, ' ').replace(/[\x00-\x1f]/g, '').replace(/ +/g, ' ');
                                    insertLength += text.length;
                                    if (tableLevel >= 0) {
                                        result.push({ operation: 'insertCellData', data: text });
                                    } else {
                                        result.push({ operation: Operations.TEXT_INSERT, data: text, depth: depth });

                                        // ------------------------------------------------------
                                        //    SET CSS ATTRIBUTES
                                        // ------------------------------------------------------
                                        var charStyles = self.getCssStyle(child, 'character'),
                                            styles = {
                                                len: text.length,
                                                attrs: {
                                                    character: charStyles
                                                }
                                            };

                                        result.push({ operation: Operations.SET_ATTRIBUTES, data: styles, depth: depth });
                                        // ------------------------------------------------------
                                    }
                                }
                            }
                        }

                    } else if ((child.nodeType === 8) && child.nodeValue) {
                        // <!--EndFragment--> marks the end of the content to parse
                        // special handling for Win 8, do no longer accept text nodes after an <!--EndFragment-->
                        if (child.nodeValue.toLowerCase() === 'endfragment') {
                            acceptTextNodes = false;
                        }
                        // <!--[if !supportLists]--> marks the start of the list replacement content
                        if (child.nodeValue.toLowerCase().indexOf('!supportlists') > -1) {
                            skipTextNodes = true;
                        }
                        // <!--[endif]--> marks the end of the replacement content
                        if (child.nodeValue.toLowerCase().indexOf('endif') > -1) {
                            skipTextNodes = false;
                        }

                    } else {
                        // insert paragraph for <div>, heading tags and for
                        //      <br> tags not nested inside a <div>, <p> or <li>
                        //      <p> tags not nested inside a <li>

                        if ($(child.parentNode).is('a') && ($(child).is('div') || $(child).is('span') || $(child).is('br'))) {
                            insertLength += findTextNodes(child, depth + 1, listLevel, tableLevel, type);
                            // these children in links lead to empty following paragraphs, that receive setAttributes operations
                            nextLevel = false;

                        } else if (
                            $(child).is('div, h1, h2, h3, h4, h5, h6') ||
                            (
                                ($(child).is('br') && !$(child.parentNode).is('p, div, li')) ||
                                ($(child.parentNode).hasClass('clipboard') && !$(child).is('style, pre, ol, ul, table') && $(child.parentNode).children(':first').is(child))
                            ) ||
                            ($(child).is('p') && !$(child.parentNode).is('li'))
                        ) {
                            if (self.isMSListParagraph(child)) {
                                // read MS list definition styles if not yet done
                                if (!msListDefinitions) {
                                    msListDefinitions = self.getMSListDefinitions(clipboard) || self.getMSFallBackListDefinitions(clipboard);
                                }

                                // get infos for the current list level
                                msParagraphLevelInfo = self.getMSParagraphLevelInfo(child);
                                msListStyle = msListDefinitions && msParagraphLevelInfo && msParagraphLevelInfo.styleId && msListDefinitions[msParagraphLevelInfo.styleId];
                                if (msListStyle) {
                                    if (!msListStyle.inserted) {
                                        // insert MS list style definition
                                        result.push({ operation: 'insertMSList', depth: depth, listLevel: msParagraphLevelInfo.level, data: msListStyle });
                                        msListStyle.inserted = true;
                                    }

                                    // insert paragraph and list element for current list level
                                    result.push({ operation: Operations.PARA_INSERT, depth: depth, listLevel: msParagraphLevelInfo.level });
                                    result.push({ operation: 'insertListElement', depth: depth, listLevel: msParagraphLevelInfo.level, type: msListStyle });

                                } else {
                                    // special handling for pasting list paragraphs without bullet or numbering
                                    result.push({ operation: 'insertListParagraph', depth: depth, listLevel: 0 });
                                }

                            } else if (tableLevel === -1 && (!lastOp || (lastOp && lastOp.operation !== Operations.PARA_INSERT && lastOp.type !== 'leaveTable'))) {
                                result.push({ operation: Operations.PARA_INSERT, depth: depth, listLevel: listLevel });
                            }

                        } else if ($(child).is('p') && $(child.parentNode).is('li') && ($(child.parentNode).children('p').first().get(0) !== child)) {
                            // special handling for pasting lists from Word, the second <p> doesn't have a bullet or numbering
                            // <ol>
                            //   <li>
                            //     <p><span>foo</span></p>      =>   1. foo
                            //     <p><span>bar</span></p>              bar
                            //   </li>
                            // </ol>
                            result.push({ operation: 'insertListParagraph', depth: depth, listLevel: listLevel });

                        } else if ($(child).is('span')) {
                            // don't parse <span> elements with 'mso-list:Ignore' style
                            if (self.isMSListIgnore(child)) {
                                nextLevel = false;
                            }

                        } else if ($(child).is('img')) {

                            // we don't support images within tables
                            if (tableLevel === -1) {

                                // the img alt tag optionally stores the document media URL, the session id and the file id
                                try {
                                    childData = JSON.parse($(child).attr('alt'));
                                } catch (e) {
                                    childData = {};
                                }
                                if (ox.session === childData.sessionId && fileId === childData.fileId && childData.altsrc) {
                                    // The image URL points to a document and the copy&paste action
                                    // takes place inside the same editor instance, so we use relative document media URL.
                                    result.push({ operation: Operations.INSERT_DRAWING, type: 'image', data: childData.altsrc, depth: depth });

                                } else if (DOM.isDocumentImageNode(child)) {
                                    // The image URL points to a document, but copy&paste is done from one editor instance to another
                                    // and the image src has not been replaced with a base64 data URL (e.g. IE 9 does not support this).
                                    // So the base64 data URL can be generated from the image node as long as the session is valid.
                                    result.push({
                                        operation: Operations.INSERT_DRAWING,
                                        type: 'image',
                                        data: DOM.getBase64FromImageNode(child, ImageUtils.getMimeTypeFromImageUri(DOM.getUrlParamFromImageNode(child, 'get_filename'))),
                                        depth: depth
                                    });

                                } else {
                                    // The image src is an external web URL or already a base64 data URL
                                    result.push({ operation: Operations.INSERT_DRAWING, type: 'image', data: child.src, depth: depth });

                                }
                                insertLength += 1;
                            } else {
                                // placeholder for removed table
                                result.push({ operation: 'insertCellData', data: ' ' });
                            }

                        } else if ($(child).is('style, title')) {
                            // don't parse <style> and <title> elements
                            nextLevel = false;

                        }  else if ($(child).is('a')) {
                            if ($(child).children('img').length === 0) {
                                // before we can add the operation for inserting the hyperlink we need to add the insertText operation
                                // so we first parse the next recursion level
                                var len = findTextNodes(child, depth + 1, listLevel, tableLevel, type);
                                // and then add the hyperlink
                                result.push({ operation: 'insertHyperlink', data: child.href, length: len, depth: depth });
                            } else {
                                insertLength += findTextNodes(child, depth + 1, listLevel, tableLevel, type);
                            }

                            // we already traversed the tree, so don't do it again
                            nextLevel = false;

                        } else if ($(child).is('ol, ul')) {
                            // for list root level create a new array, otherwise add to current
                            if (listLevel === -1) { type = []; }
                            // add current list level type
                            type[listLevel + 1] = $(child).css('list-style-type') || $(child).attr('type') || ($(child).is('ol') ? 'decimal' : 'disc');
                            // insert list entry for current list level
                            result.push({ operation: 'insertList', depth: depth, listLevel: listLevel, type: type });
                            // look for deeper list levels
                            insertLength += findTextNodes(child, depth + 1, listLevel + 1, tableLevel, type);
                            // we already traversed the tree, so don't do it again
                            nextLevel = false;

                        } else if ($(child).is('li')) {
                            // insert paragraph and list element for current list level
                            result.push({ operation: Operations.PARA_INSERT, depth: depth, listLevel: listLevel });
                            result.push({ operation: 'insertListElement', depth: depth, listLevel: listLevel, type: type });

                        } else if ($(child).is('table')) {
                            if (tableLevel === -1) {
                                if ($(child).find('td').length > 1) {
                                    var size = calcTableSize(child);
                                    var caption = $(child).find('caption').text();

                                    result.push({ operation: 'insertTable', size: size, caption: caption });
                                    insertLength += findTextNodes(child, depth + 1, listLevel, tableLevel + 1, type);
                                    nextLevel = false;
                                    result.push({ operation: Operations.PARA_INSERT, type: 'leaveTable' });
                                }
                            } else {
                                // placeholder for removed table
                                result.push({ operation: 'insertCellData', data: ' ' });
                                // stop traversing nested tables!
                                nextLevel = false;
                            }

                        } else if ($(child).is('tr')) {
                            if (tableLevel !== -1) {
                                result.push({ operation: 'insertRow' });
                            } else {
                                //split para
                                result.push({ operation: Operations.PARA_INSERT, depth: depth });
                            }

                        } else if ($(child).is('td') || $(child).is('th')) {
                            if (tableLevel !== -1) {
                                result.push({ operation: 'prepareCell', depth: depth });
                                insertLength += findTextNodes(child, depth + 1, listLevel, tableLevel, type);
                                result.push({ operation: 'finalizeCell', depth: depth });
                                nextLevel = false;

                            } else {
                                //insert tab
                                if ($(child).is('td:not(:first-of-type)')) {
                                    if (lastOp && lastOp.operation === Operations.TEXT_INSERT) {
                                        lastOp.data = lastOp.data.replace(/^\s*/, '');
                                    }
                                    result.push({ operation: Operations.TAB_INSERT, depth: depth });
                                }
                            }
                        } else if ($(child).is('caption')) {
                            if (tableLevel !== -1) {
                                // stop traversing table captions, they are handled within insert table
                                nextLevel = false;
                            }
                        } else if ($(child).is('pre')) {
                            // parse preformatted text
                            var textOps = self.parseClipboardText($(child).text());
                            result = result.concat(textOps);
                            // all deeper levels already handled, don't traverse the tree
                            nextLevel = false;
                        }

                        if (nextLevel) {
                            insertLength += findTextNodes(child, depth + 1, listLevel, tableLevel, type);
                        }
                    }
                }

                return insertLength;

            }(/*node*/ clipboard.get(0), /*depth*/ 0, /*listLevel*/ -1, /*tableLevel*/ -1, /*type*/ undefined));

            // console.warn('paste result', result);
            return result;
        };

        // MS Word list pasting -----------------------------------------------

        /**
         * Returns true if the origin of the given HTML data is MS Word.
         *
         * @param {Node|jQuery} html
         *  The HTML data to check.
         *
         * @returns {Boolean}
         */
        this.isMSWordHtml = function (html) {
            return (($(html).find('meta[content*=Word]').length > 0) || ($(html).filter('meta[content*=Word]').length > 0));
        };

        /**
         * Returns true if the paragraph element is a MS Word list paragraph.
         *
         * @param {Node|jQuery} paragraph
         *  The HTML list paragraph.
         *
         * @returns {Boolean}
         */
        this.isMSListParagraph = function (paragraph) {
            return ($(paragraph).is('.MsoListParagraphCxSpFirst, .MsoListParagraphCxSpMiddle, .MsoListParagraphCxSpLast'));
        };

        /**
         * Returns true if the HTML element is a MS Office list replacement.
         * We look for 'mso-list:Ignore' style first, but since some browsers remove it,
         * we also check if the previous node is the <!--[if !supportLists]--> comment node.
         *
         * @param {Node|jQuery} element
         *  The HTML element.
         *
         * @returns {Boolean}
         */
        this.isMSListIgnore = function (element) {

            var node, style, prevNodeValue;

            node = (element instanceof $) ? element[0] : element;
            style = node && element.getAttribute('style');
            if (style && style.indexOf('mso-list:Ignore') > -1) {
                return true;
            }

            prevNodeValue = node && node.previousSibling && (node.previousSibling.nodeType === 8) && node.previousSibling.nodeValue;
            if (prevNodeValue && prevNodeValue.indexOf('!supportLists') > -1) {
                return true;
            }

            return false;
        };

        /**
         * Parses the given MS Word list paragraph for the list style id and the list level.
         *
         * @param {Node|jQuery} paragraph
         *  The MS Word list paragraph.
         *
         * @returns {Object|null}
         *              {String} styleId
         *                  The list style id
         *              {Number} level
         *                  The zero based list level
         *              {Number} lfo
         *                  The MS list formatting attribute
         */
        this.getMSParagraphLevelInfo = function (paragraph) {

            var originalRegEx = /mso-list:(l\d+)\s+level(\d+)\s+lfo(\d+)/i;
            var sanitizeRegEx = /content:\s*"(l\d+)-level(\d+)-lfo(\d+)"/i;
            var style = $(paragraph).attr('style');
            var parts;

            // first check for the sanitized mso-list / content attribute
            parts = sanitizeRegEx.exec(style);
            // then check for the original mso-list attribute
            if (!_.isArray(parts) || !parts[1] || !parts[2]) {
                parts = originalRegEx.exec(style);
            }
            if (!_.isArray(parts) || !parts[1] || !parts[2]) { return null; }

            return { styleId: parts[1].toLowerCase(), level: listLevelStringToZeroBasedLevel(parts[2]), lfo: parseInt(parts[3], 10) };
        };

        /**
         * Parses the given HTML for MS Word list style definitions
         * and returns them as object structure with the style id as the first
         * and the list level as the second object level.
         *
         * @param {Node|jQuery} html
         *  The HTML data to parse.
         *
         * @returns {Object}
         *              {Object} styleId
         *                  {Object} level
         *                      {String} numberFormat
         *                          The MS list level number format.
         *                      {String} levelText
         *                          The text for the list level.
         *                      {String} fontFamily
         *                          The level text font.
         *                      {Number} indentFirstLine
         *                          The first line text indent in 1/100 of millimeters.
         *                      {Number} indentLeft
         *                          The left indent in 1/100 of millimeters.
         *                      {String} numberPosition
         *                          The number position.
         */
        this.getMSListDefinitions = function (html) {

            var stylesText = $(html).find('style').text(),
                listStyles,
                result = {};

            // parse list styles out of all styles
            listStyles = stylesText.match(/@list\s+l\d+:level\d+\s*\{[^}]*\}/ig);
            if (!_.isArray(listStyles)) { return null; }

            _.each(listStyles, function (item) {

                var parts, styleId, listLevel, numberFormat, levelText, fontFamily;

                // parse style id and list level
                parts = item.match(/@list\s+(l\d+):level(\d+)/i);
                if (_.isArray(parts) && parts[1] && parts[2]) {

                    styleId = parts[1].toLowerCase();
                    listLevel = listLevelStringToZeroBasedLevel(parts[2]);

                    if (!result[styleId]) {
                        result[styleId] = {};
                    }
                    if (!result[styleId][listLevel]) {
                        result[styleId][listLevel] = {};
                    }

                    // parse number format, default to 'decimal'
                    parts = item.match(/mso-level-number-format:\s*(\S+);/i);
                    numberFormat = _.isArray(parts) && parts[1];

                    // parse level text, or generate it from number format and list level
                    parts = item.match(/mso-level-text:(\S+.*);/i);
                    levelText = _.isArray(parts) && parts[1];

                    // parse font family
                    parts = item.match(/font-family:\s*(\S+.*\S+);/i);
                    fontFamily = _.isArray(parts) && parts[1];

                    parts = self.createMSNumberFormatAndLevelText(numberFormat, listLevel, levelText, fontFamily);
                    result[styleId][listLevel].numberFormat = parts.numberFormat;
                    result[styleId][listLevel].levelText = parts.levelText;
                    if (parts.fontFamily) { result[styleId][listLevel].fontFamily = parts.fontFamily; }

                    // parse text indent and convert to 1/100 millimeters
                    parts = item.match(/text-indent:\s*(\S+);/i);
                    if (_.isArray(parts) && parts[1]) {
                        result[styleId][listLevel].indentFirstLine = Utils.convertCssLengthToHmm(parts[1]);
                    }

                    // parse margin left and convert to 1/100 millimeters
                    parts = item.match(/margin-left:\s*(\S+);/i);
                    if (_.isArray(parts) && parts[1]) {
                        result[styleId][listLevel].indentLeft = Utils.convertCssLengthToHmm(parts[1]);
                    }

                    // parse number position
                    parts = item.match(/mso-level-number-position:\s*(\S+);/i);
                    if (_.isArray(parts) && parts[1]) {
                        result[styleId][listLevel].numberPosition = parts[1];
                    }
                }

            });

            return result;
        };

        /**
         * Parses the given HTML for MS Word list paragraphs and generates
         * fall back style definitions out of the list replacements.
         * Returns them as object structure with the style id as the first
         * and the list level as the second object level.
         *
         * @param {Node|jQuery} html
         *  The HTML data to parse.
         *
         * @returns {Object}
         *              {Object} styleId
         *                  {Object} level
         *                      {String} numberFormat
         *                          The MS list level number format.
         *                      {String} levelText
         *                          The text for the list level.
         *                      {String} fontFamily
         *                          The level text font.
         *                      {Number} indentFirstLine
         *                          The first line text indent in 1/100 of millimeters.
         *                      {Number} indentLeft
         *                          The left indent in 1/100 of millimeters.
         */
        this.getMSFallBackListDefinitions = function (html) {

            var paragraphs = $(html).find('p'),
                result = {};

            function hasListDefinition(styles, styleId, listLevel) {
                if (!styles) { return false; }
                if (!styles[styleId]) { return false; }
                if (!styles[styleId][listLevel]) { return false; }
                if (!_.isString(styles[styleId][listLevel].levelText)) { return false; }
                if (!_.isString(styles[styleId][listLevel].numberFormat)) { return false; }
                return true;
            }

            _.each(paragraphs, function (p) {

                var msParagraphLevelInfo,
                    styleId,
                    listLevel,
                    text,
                    image,
                    numberFormat = null,
                    levelText,
                    fontFamily,
                    parts;

                if (self.isMSListParagraph(p)) {

                    msParagraphLevelInfo = self.getMSParagraphLevelInfo(p);

                    if (msParagraphLevelInfo && _.isString(msParagraphLevelInfo.styleId) && _.isNumber(msParagraphLevelInfo.level)) {

                        styleId = msParagraphLevelInfo.styleId;
                        listLevel = msParagraphLevelInfo.level;

                        if (!hasListDefinition(result, styleId, listLevel)) {

                            // get alt text from image bullet, or level text from bullet / numbering
                            image = $(p).find('img');
                            if (image.length > 0) {
                                text = image.attr('alt') || '\u2022';
                            } else {
                                text = $(p).text().match(/\s*(\S*)\s+(\S*)/);
                                text = (_.isArray(text) && _.isString(text[1])) ? text[1] : '';
                            }

                            numberFormat = self.getNumberFormatFromListLabelText(text);
                            fontFamily = $(p).children().first().css('font-family');

                            if (!result[styleId]) {
                                result[styleId] = {};
                            }
                            if (!result[styleId][listLevel]) {
                                result[styleId][listLevel] = {};
                            }

                            switch (numberFormat) {

                                case 'decimal':
                                case 'lowerRoman':
                                case 'upperRoman':
                                case 'upperLetter':
                                case 'lowerLetter':
                                    levelText = '%' + (listLevel + 1) + '.';
                                    break;

                                case 'bullet':
                                    levelText = text;
                                    break;

                                default:
                                    levelText = text;
                            }

                            parts = self.createMSNumberFormatAndLevelText(numberFormat, listLevel, levelText, fontFamily);
                            result[styleId][listLevel].numberFormat = parts.numberFormat;
                            result[styleId][listLevel].levelText = parts.levelText;
                            if (parts.fontFamily) {
                                result[styleId][listLevel].fontFamily = parts.fontFamily;
                            }
                        }
                    }
                }
            });

            return result;
        };

        /**
         * Creates the MS list level number format from the CSS number format and
         * creates a level text if none is given.
         *
         * @param {String} numberFormat
         *  The CSS number format String.
         *
         * @param {Number} listLevel
         *  The zero based list level.
         *
         * @param {String} [levelText]
         *  The level text to use instead of a generated one.
         *
         * @param {String} [fontFamily]
         *  The font family for mapping Wingdings to Unicode bullets.
         *
         * @returns {Object}
         *              {String} numberFormat
         *                  The MS list level number format String.
         *              {String} levelText
         *                  The level text that was given or that has been generated.
         */
        this.createMSNumberFormatAndLevelText = function (numberFormat, listLevel, levelText, fontFamily) {

            var result = {},
                wing;

            numberFormat =  _.isString(numberFormat) ? numberFormat.toLowerCase() : 'decimal';
            levelText = cleanLevelText(levelText);

            switch (numberFormat) {

                case 'decimal':
                case '1':
                    result.numberFormat = 'decimal';
                    result.levelText = _.isString(levelText) ? levelText : '%' + (listLevel + 1) + '.';
                    break;

                case 'decimal-leading-zero':
                    result.numberFormat = 'decimal';
                    result.levelText = _.isString(levelText) ? levelText : '0%' + (listLevel + 1) + '.';
                    break;

                case 'A':
                case 'upper-alpha':
                case 'alpha-upper':
                case 'upper-latin':
                case 'upperletter':
                    result.numberFormat = 'upperLetter';
                    result.levelText = _.isString(levelText) ? levelText : '%' + (listLevel + 1) + '.';
                    break;

                case 'a':
                case 'lower-alpha':
                case 'alpha-lower':
                case 'lower-latin':
                case 'lowerletter':
                    result.numberFormat = 'lowerLetter';
                    result.levelText = _.isString(levelText) ? levelText : '%' + (listLevel + 1) + '.';
                    break;

                case 'I':
                case 'upper-roman':
                case 'roman-upper':
                case 'upperroman':
                    result.numberFormat = 'upperRoman';
                    result.levelText = _.isString(levelText) ? levelText : '%' + (listLevel + 1) + '.';
                    break;

                case 'i':
                case 'lower-roman':
                case 'roman-lower':
                case 'lowerroman':
                    result.numberFormat = 'lowerRoman';
                    result.levelText = _.isString(levelText) ? levelText : '%' + (listLevel + 1) + '.';
                    break;

                case 'disc':
                case 'circle':
                case 'square':
                case 'bullet':
                    result.numberFormat = 'bullet';
                    if (_.isString(fontFamily) && ((fontFamily.indexOf('ingdings') > 0) || (fontFamily.indexOf('ymbol') > 0))) {
                        wing = self.getUnicodeReplacementForChar(levelText, fontFamily);
                        result.levelText = wing.levelText;
                        result.fontFamily = wing.fontFamily;

                    } else if (_.isString(levelText)) {
                        result.levelText = levelText;

                    } else {
                        switch (numberFormat) {
                            case 'disc':
                            case 'bullet':
                                result.levelText = '\uf0b7';
                                break;
                            case 'circle':
                                result.levelText = '\u25CB';
                                break;
                            case 'square':
                                result.levelText = '\u25A0';
                                break;

                            default:
                                result.levelText = '\uf0b7';
                        }
                    }
                    break;

                case 'none':
                    result.numberFormat = 'bullet';
                    result.levelText = _.isString(levelText) ? levelText : ' ';
                    break;

                case 'image':
                    result.numberFormat = 'bullet';
                    result.levelText = '\u2022';
                    break;

                default:
                    result.numberFormat = 'decimal';
                    result.levelText = _.isString(levelText) ? levelText : '%' + (listLevel + 1) + '.';
            }

            return result;
        };

        /**
         * Returns the number format for the given list label.
         *
         * @param {String} text
         *  The list label.
         *
         * @returns {String}
         *  The number format.
         */
        this.getNumberFormatFromListLabelText = function (text) {

            if (text.length > 0) {

                if (decimalRegEx.test(text)) {
                    return 'decimal';
                }
                if (upperRomanRegEx.test(text)) {
                    return 'upperRoman';
                }
                if (lowerRomanRegEx.test(text)) {
                    return 'lowerRoman';
                }
                if (upperLetterRegEx.test(text)) {
                    return 'upperLetter';
                }
                if (lowerLetterRegEx.test(text)) {
                    return 'lowerLetter';
                }
            }

            return 'bullet';
        };

        /**
         * Returns a Unicode replacement for the given char for font family Symbol or Wingdings.
         *
         * @param {String} char
         *  The Wingdings char.
         *
         * @param {String} fontFamily
         *  The font family.
         *
         * @returns {Object}
         *              {String} levelText
         *                  The Unicode char.
         *              {String} fontFamily
         *                  The font family of the char.
         */
        this.getUnicodeReplacementForChar = function (char, fontFamily) {
            // the font family in lower case
            var fontFamilyLowerCase = null;

            if (_.isString(fontFamily) && !_.isEmpty(fontFamily)) {
                fontFamilyLowerCase = fontFamily.toLowerCase();

                if (fontFamilyLowerCase.indexOf('wingdings') > -1) {
                    return (fontFamilyLowerCase.indexOf('2') > 0) ? this.getUnicodeForWingdings2Char(char) : this.getUnicodeForWingdings1Char(char);
                } else if ((fontFamilyLowerCase.indexOf('starsymbol') > -1) || (fontFamilyLowerCase.indexOf('opensymbol') > -1)) {
                    return this.getUnicodeForOpenSymbolChar(char);
                } else if (fontFamilyLowerCase.indexOf('symbol') > -1) {
                    return this.getUnicodeForSymbolChar(char);
                }
            }

            return char;
        };

        /**
         * Returns a Unicode replacement for a Wingdings 1 encoded char that can be displayed in common browser fonts.
         *
         * @param {String} char
         *  The Wingdings 1 char.
         *
         * @returns {Object}
         *              {String} levelText
         *                  The Unicode char.
         *              {String} fontFamily
         *                  The font family of the char.
         */
        this.getUnicodeForWingdings1Char = function (char) {

            if (!_.isString(char) || (char.length !== 1)) {
                return { levelText: '\u2022', fontFamily: 'Times New Roman' };
            }

            switch (char.charCodeAt(0)) {
                case 61472:
                case 32:
                    return { levelText: ' ', fontFamily: 'Times New Roman' };  // Space

                case 61473:
                case 33:
                    return { levelText: '\u270f', fontFamily: 'Arial' };  // Pencil

                case 61474:
                case 34:
                    return { levelText: '\u2702', fontFamily: 'Arial' };  // Black scissors

                case 61475:
                case 35:
                    return { levelText: '\u2701', fontFamily: 'Arial' };  // Upper blade scissors

                case 61480:
                case 40:
                    return { levelText: '\u260e', fontFamily: 'Arial' };  // Black telephone

                case 61481:
                case 41:
                    return { levelText: '\u2706', fontFamily: 'Arial' };  // Telephone location sign

                case 61482:
                case 42:
                    return { levelText: '\u2709', fontFamily: 'Arial' };  // Envelope

                case 61494:
                case 54:
                    return { levelText: '\u231b', fontFamily: 'Arial' };  // Hourglass

                case 61495:
                case 55:
                    return { levelText: '\u2328', fontFamily: 'Arial' };  // Keyboard

                case 61502:
                case 62:
                    return { levelText: '\u2707', fontFamily: 'Arial' };  // Tape drive

                case 61503:
                case 63:
                    return { levelText: '\u270d', fontFamily: 'Arial' };  // Writing hand

                case 61505:
                case 65:
                    return { levelText: '\u270c', fontFamily: 'Arial' };  // Victory hand

                case 61509:
                case 69:
                    return { levelText: '\u261c', fontFamily: 'Arial' };  // White left pointing index

                case 61510:
                case 70:
                    return { levelText: '\u261e', fontFamily: 'Arial' };  // White right pointing index

                case 61511:
                case 71:
                    return { levelText: '\u261d', fontFamily: 'Arial' };  // White up pointing index

                case 61512:
                case 72:
                    return { levelText: '\u261f', fontFamily: 'Arial' };  // White down pointing index

                case 61514:
                case 74:
                    return { levelText: '\u263a', fontFamily: 'Times New Roman' };  // White smiling face

                case 61516:
                case 76:
                    return { levelText: '\u2639', fontFamily: 'Arial' };  // White frowning face

                case 61518:
                case 78:
                    return { levelText: '\u2620', fontFamily: 'Arial' };  // Skull and crossbones

                case 61519:
                case 79:
                    return { levelText: '\u2690', fontFamily: 'Arial' };  // White flag

                case 61521:
                case 81:
                    return { levelText: '\u2708', fontFamily: 'Arial' };  // Airplane

                case 61522:
                case 82:
                    return { levelText: '\u263c', fontFamily: 'Times New Roman' };  // White sun with rays

                case 61524:
                case 84:
                    return { levelText: '\u2744', fontFamily: 'Arial' };  // Snowflake

                case 61526:
                case 86:
                    return { levelText: '\u271e', fontFamily: 'Arial' };  // Shadowed white Latin cross

                case 61528:
                case 88:
                    return { levelText: '\u2720', fontFamily: 'Arial' };  // Maltese cross

                case 61529:
                case 89:
                    return { levelText: '\u2721', fontFamily: 'Arial' };  // Star of David

                case 61530:
                case 90:
                    return { levelText: '\u262a', fontFamily: 'Arial' };  // Star and crescent

                case 61531:
                case 91:
                    return { levelText: '\u262f', fontFamily: 'Arial' };  // Yin Yang

                case 61532:
                case 92:
                    return { levelText: '\u0950', fontFamily: 'Arial' };  // Devanagari Om

                case 61533:
                case 93:
                    return { levelText: '\u2638', fontFamily: 'Arial' };  // Wheel of Dharma

                case 61534:
                case 94:
                    return { levelText: '\u2648', fontFamily: 'Arial' };  // Aries

                case 61535:
                case 95:
                    return { levelText: '\u2649', fontFamily: 'Arial' };  // Taurus

                case 61536:
                case 96:
                    return { levelText: '\u264a', fontFamily: 'Arial' };  // Gemini

                case 61537:
                case 97:
                    return { levelText: '\u264b', fontFamily: 'Arial' };  // Cancer

                case 61538:
                case 98:
                    return { levelText: '\u264c', fontFamily: 'Arial' };  // Leo

                case 61539:
                case 99:
                    return { levelText: '\u264d', fontFamily: 'Arial' };  // Virgo

                case 61540:
                case 100:
                    return { levelText: '\u264e', fontFamily: 'Arial' };  // Libra

                case 61541:
                case 101:
                    return { levelText: '\u264f', fontFamily: 'Arial' };  // Scorpio

                case 61542:
                case 102:
                    return { levelText: '\u2650', fontFamily: 'Arial' };  // Sagittarius

                case 61543:
                case 103:
                    return { levelText: '\u2651', fontFamily: 'Arial' };  // Capricorn

                case 61544:
                case 104:
                    return { levelText: '\u2652', fontFamily: 'Arial' };  // Aquarius

                case 61545:
                case 105:
                    return { levelText: '\u2653', fontFamily: 'Arial' };  // Pisces

                case 61546:
                case 106:
                    return { levelText: '&', fontFamily: 'Times New Roman' };  // Ampersand (italic, lower-case)

                case 61547:
                case 107:
                    return { levelText: '&', fontFamily: 'Times New Roman' };  // Ampersand (italic)

                case 61548:
                case 108:
                    return { levelText: '\u25cf', fontFamily: 'Times New Roman' };  // Black circle

                case 61549:
                case 109:
                    return { levelText: '\u274d', fontFamily: 'Arial' };  // Shadowed white circle

                case 61550:
                case 110:
                    return { levelText: '\u25a0', fontFamily: 'Times New Roman' };  // Black square

                case 61551:
                case 111:
                    return { levelText: '\u25a1', fontFamily: 'Times New Roman' };  // White square

                case 61553:
                case 113:
                    return { levelText: '\u2751', fontFamily: 'Arial' };  //  Lower right shadowed white square

                case 61554:
                case 114:
                    return { levelText: '\u2752', fontFamily: 'Arial' };  // Upper right shadowed white square

                case 61555:
                case 115:
                    return { levelText: '\u2b27', fontFamily: 'Arial' };  // Black medium lozenge

                case 61556:
                case 116:
                    return { levelText: '\u29eb', fontFamily: 'Arial' };  // Black lozenge

                case 61557:
                case 117:
                    return { levelText: '\u25c6', fontFamily: 'Arial' };  // Black diamond

                case 61558:
                case 118:
                    return { levelText: '\u2756', fontFamily: 'Arial' };  // Black diamond minus white X

                case 61560:
                case 120:
                    return { levelText: 'u2327', fontFamily: 'Arial' };  // X in a rectangle box

                case 61561:
                case 121:
                    return { levelText: '\u2353', fontFamily: 'Arial' };  // APL functional symbol quad up caret

                case 61562:
                case 122:
                    return { levelText: '\u2318', fontFamily: 'Arial' };  // Place of interest sign

                case 61563:
                case 123:
                    return { levelText: '\u2740', fontFamily: 'Arial' };  // White florette

                case 61564:
                case 124:
                    return { levelText: '\u273f', fontFamily: 'Arial' };  // Black florette

                case 61565:
                case 125:
                    return { levelText: '\u275d', fontFamily: 'Arial' };  // Heavy double turned comma quotation mark ornament

                case 61566:
                case 126:
                    return { levelText: '\u275e', fontFamily: 'Arial' };  // Heavy double comma quotation mark ornament

                case 61567:
                case 127:
                    return { levelText: '\u25af', fontFamily: 'Arial' };  // (White vertical rectangle)

                case 61568:
                case 128:
                    return { levelText: '\u24ea', fontFamily: 'Arial' };  // Circled digit zero

                case 61569:
                case 129:
                    return { levelText: '\u2460', fontFamily: 'Arial' };  // Circled digit one

                case 61570:
                case 130:
                    return { levelText: '\u2461', fontFamily: 'Arial' };  // Circled digit two

                case 61571:
                case 131:
                    return { levelText: '\u2462', fontFamily: 'Arial' };  // Circled digit three

                case 61572:
                case 132:
                    return { levelText: '\u2463', fontFamily: 'Arial' };  // Circled digit four

                case 61573:
                case 133:
                    return { levelText: '\u2464', fontFamily: 'Arial' };  // Circled digit five

                case 61574:
                case 134:
                    return { levelText: '\u2465', fontFamily: 'Arial' };  // Circled digit six

                case 61575:
                case 135:
                    return { levelText: '\u2466', fontFamily: 'Arial' };  // Circled digit seven

                case 61576:
                case 136:
                    return { levelText: '\u2467', fontFamily: 'Arial' };  // Circled digit eight

                case 61577:
                case 137:
                    return { levelText: '\u2468', fontFamily: 'Arial' };  // Circled digit nine

                case 61578:
                case 138:
                    return { levelText: '\u2469', fontFamily: 'Arial' };  // Circled number ten

                case 61579:
                case 139:
                    return { levelText: '\u24ff', fontFamily: 'Arial' };  // Negative circled digit zero

                case 61580:
                case 140:
                    return { levelText: '\u2776', fontFamily: 'Arial' };  // Dingbat negative circled digit one

                case 61581:
                case 141:
                    return { levelText: '\u2777', fontFamily: 'Arial' };  // Dingbat negative circled digit two

                case 61582:
                case 142:
                    return { levelText: '\u2778', fontFamily: 'Arial' };  // Dingbat negative circled digit three

                case 61583:
                case 143:
                    return { levelText: '\u2779', fontFamily: 'Arial' };  // Dingbat negative circled digit four

                case 61584:
                case 144:
                    return { levelText: '\u277a', fontFamily: 'Arial' };  // Dingbat negative circled digit five

                case 61585:
                case 145:
                    return { levelText: '\u277b', fontFamily: 'Arial' };  // Dingbat negative circled digit six

                case 61586:
                case 146:
                    return { levelText: '\u277c', fontFamily: 'Arial' };  // Dingbat negative circled digit seven

                case 61587:
                case 147:
                    return { levelText: '\u277d', fontFamily: 'Arial' };  // Dingbat negative circled digit eight

                case 61588:
                case 148:
                    return { levelText: '\u277e', fontFamily: 'Arial' };  // Dingbat negative circled digit nine

                case 61589:
                case 149:
                    return { levelText: '\u277f', fontFamily: 'Arial' };  // Dingbat negative circled number ten

                case 61598:
                case 158:
                    return { levelText: '\xb7', fontFamily: 'Times New Roman' };  // Middle dot

                case 61599:
                case 159:
                    return { levelText: '\u2022', fontFamily: 'Times New Roman' };  // Bullet

                case 61600:
                case 160:
                    return { levelText: '\u25aa', fontFamily: 'Times New Roman' };  // Black small square

                case 61601:
                case 161:
                    return { levelText: '\u25cb', fontFamily: 'Times New Roman' };  // White circle

                case 61604:
                case 164:
                    return { levelText: '\u25c9', fontFamily: 'Arial' };  // Fisheye

                case 61605:
                case 165:
                    return { levelText: '\u25ce', fontFamily: 'Arial' };  // Bullseye

                case 61606:
                case 166:
                    return { levelText: '\u274d', fontFamily: 'Arial' };  // Shadowed white circle

                case 61607:
                case 167:
                    return { levelText: '\u25aa', fontFamily: 'Times New Roman' };  // Black small square

                case 61608:
                case 168:
                    return { levelText: '\u25fb', fontFamily: 'Arial' };  // White medium square

                case 61610:
                case 170:
                    return { levelText: '\u2726', fontFamily: 'Arial' };  // Black four pointed star

                case 61611:
                case 171:
                    return { levelText: '\u2605', fontFamily: 'Arial' };  // Black star

                case 61612:
                case 172:
                    return { levelText: '\u2736', fontFamily: 'Arial' };  // Six pointed black star

                case 61613:
                case 173:
                    return { levelText: '\u2734', fontFamily: 'Arial' };  // Eight pointed black star

                case 61614:
                case 174:
                    return { levelText: '\u2739', fontFamily: 'Arial' };  // Twelve pointed black star

                case 61615:
                case 175:
                    return { levelText: '\u2735', fontFamily: 'Arial' };  // Eight pointed pinwheel star

                case 61617:
                case 177:
                    return { levelText: '\u2316', fontFamily: 'Arial' };  // Position indicator

                case 61619:
                case 179:
                    return { levelText: '\u2311', fontFamily: 'Arial' };  // Square lozenge

                case 61621:
                case 181:
                    return { levelText: '\u272a', fontFamily: 'Arial' };  // Circled white star

                case 61622:
                case 182:
                    return { levelText: '\u2730', fontFamily: 'Arial' };  // Shadowed white star

                case 61653:
                case 213:
                    return { levelText: '\u232b', fontFamily: 'Arial' };  // Erase to the left

                case 61654:
                case 214:
                    return { levelText: '\u2326', fontFamily: 'Arial' };  // Erase to the right

                case 61656:
                case 216:
                    return { levelText: '\u27a2', fontFamily: 'Arial' };  // Three-D top-lighted rightwards arrowhead

                case 61660:
                case 220:
                    return { levelText: '\u27b2', fontFamily: 'Arial' };  // Circled heavy white rightwards arrow

                case 61672:
                case 232:
                    return { levelText: '\u2794', fontFamily: 'Arial' };  // Heavy wide-headed rightwards arrow

                case 61679:
                case 239:
                    return { levelText: '\u21e6', fontFamily: 'Arial' };  // Leftwards white arrow

                case 61680:
                case 240:
                    return { levelText: '\u21e8', fontFamily: 'Arial' };  // Rightwards white arrow

                case 61681:
                case 241:
                    return { levelText: '\u21e7', fontFamily: 'Arial' };  // Upwards white arrow

                case 61682:
                case 242:
                    return { levelText: '\u21e9', fontFamily: 'Arial' };  // Downwards white arrow

                case 61683:
                case 243:
                    return { levelText: '\u2b04', fontFamily: 'Arial' };  // Left right white arrow

                case 61684:
                case 244:
                    return { levelText: '\u21f3', fontFamily: 'Arial' };  // Up down white arrow

                case 61685:
                case 245:
                    return { levelText: '\u2b01', fontFamily: 'Arial' };  // North west white arrow

                case 61686:
                case 246:
                    return { levelText: '\u2b00', fontFamily: 'Arial' };  // North east white arrow

                case 61687:
                case 247:
                    return { levelText: '\u2b03', fontFamily: 'Arial' };  // South west white arrow

                case 61688:
                case 248:
                    return { levelText: '\u2b02', fontFamily: 'Arial' };  // South east white arrow

                case 61689:
                case 249:
                    return { levelText: '\u25ad', fontFamily: 'Arial' };  // White rectangle

                case 61690:
                case 250:
                    return { levelText: '\u25ab', fontFamily: 'Times New Roman' };  // White small square

                case 61691:
                case 251:
                    return { levelText: '\u2717', fontFamily: 'Arial' };  // Ballot X

                case 61692:
                case 252:
                    return { levelText: '\u2713', fontFamily: 'Arial' };  // Check mark

                case 61693:
                case 253:
                    return { levelText: '\u2612', fontFamily: 'Arial' };  // Ballot box with X

                case 61694:
                case 254:
                    return { levelText: '\u2611', fontFamily: 'Arial' };  // Ballot box with check

                default:
                    return { levelText: '\u2022', fontFamily: 'Times New Roman' };
            }
        };

        /**
         * Returns a Unicode replacement for a Wingdings 2 encoded char that can be displayed in common browser fonts.
         *
         * @param {String} char
         *  The Wingdings 2 char.
         *
         * @returns {Object}
         *              {String} levelText
         *                  The Unicode char.
         *              {String} fontFamily
         *                  The font family of the char.
         */
        this.getUnicodeForWingdings2Char = function (char) {

            if (!_.isString(char) || (char.length !== 1)) {
                return { levelText: '\u2022', fontFamily: 'Times New Roman' };
            }

            switch (char.charCodeAt(0)) {
                case 61591:
                case 151:
                    return { levelText: '\u25cf', fontFamily: 'Times New Roman' };

                case 61598:
                case 158:
                    return { levelText: '\u29bf', fontFamily: 'Times New Roman' };

                default:
                    return { levelText: '\u2022', fontFamily: 'Times New Roman' };
            }
        };

        /**
         * Returns a Unicode replacement for a Symbol encoded char that can be displayed in common browser fonts.
         *
         * @param {String} char
         *  The Symbol char.
         *
         * @returns {Object}
         *              {String} levelText
         *                  The Unicode char.
         *              {String} fontFamily
         *                  The font family of the char.
         */
        this.getUnicodeForSymbolChar = function (char) {

            if (!_.isString(char) || (char.length !== 1)) {
                return { levelText: '\u2022', fontFamily: 'Times New Roman' };
            }

            switch (char.charCodeAt(0)) {
                case 32:
                case 61472:
                    return { levelText: ' ', fontFamily: 'Times New Roman' };  // Space

                case 33:
                case 61473:
                    return { levelText: '!', fontFamily: 'Times New Roman' };  // Exclamation mark

                case 34:
                case 61474:
                    return { levelText: '\u2200', fontFamily: 'Arial' };  // For all

                case 35:
                case 61475:
                    return { levelText: '#', fontFamily: 'Times New Roman' };  // Number sign

                case 36:
                case 61476:
                    return { levelText: '\u2203', fontFamily: 'Arial' };  // There exists

                case 37:
                case 61477:
                    return { levelText: '%', fontFamily: 'Times New Roman' };  // Percent sign

                case 38:
                case 61478:
                    return { levelText: '&', fontFamily: 'Times New Roman' };  // Ampersand

                case 39:
                case 61479:
                    return { levelText: '\u220b', fontFamily: 'Arial' };  // Contains as member

                case 40:
                case 61480:
                    return { levelText: '(', fontFamily: 'Times New Roman' };  // Left Parenthesis

                case 41:
                case 61481:
                    return { levelText: ')', fontFamily: 'Times New Roman' };  // Right Parenthesis

                case 42:
                case 61482:
                    return { levelText: '\u2217', fontFamily: 'Arial' };  // Asterisk

                case 43:
                case 61483:
                    return { levelText: '+', fontFamily: 'Times New Roman' };  // Plus sign

                case 44:
                case 61484:
                    return { levelText: ',', fontFamily: 'Times New Roman' };  // Comma

                case 45:
                case 61485:
                    return { levelText: '\u2212', fontFamily: 'Times New Roman' };  // Minus sign

                case 46:
                case 61486:
                    return { levelText: '.', fontFamily: 'Times New Roman' };  // Full stop

                case 47:
                case 61487:
                    return { levelText: '/', fontFamily: 'Times New Roman' };  // Solidus

                case 48:
                case 61488:
                    return { levelText: '0', fontFamily: 'Times New Roman' };  // Digit zero

                case 49:
                case 61489:
                    return { levelText: '1', fontFamily: 'Times New Roman' };  // Digit one

                case 50:
                case 61490:
                    return { levelText: '2', fontFamily: 'Times New Roman' };  // Digit two

                case 51:
                case 61491:
                    return { levelText: '3', fontFamily: 'Times New Roman' };  // Digit three

                case 52:
                case 61492:
                    return { levelText: '4', fontFamily: 'Times New Roman' };  // Digit four

                case 53:
                case 61493:
                    return { levelText: '5', fontFamily: 'Times New Roman' };  // Digit five

                case 54:
                case 61494:
                    return { levelText: '6', fontFamily: 'Times New Roman' };  // Digit six

                case 55:
                case 61495:
                    return { levelText: '7', fontFamily: 'Times New Roman' };  // Digit seven

                case 56:
                case 61496:
                    return { levelText: '8', fontFamily: 'Times New Roman' };  // Digit eight

                case 57:
                case 61497:
                    return { levelText: '9', fontFamily: 'Times New Roman' };  // Digit nine

                case 58:
                case 61498:
                    return { levelText: ':', fontFamily: 'Times New Roman' };  // Colon

                case 59:
                case 61499:
                    return { levelText: ';', fontFamily: 'Times New Roman' };  // Semicolon

                case 60:
                case 61500:
                    return { levelText: '<', fontFamily: 'Times New Roman' };  // Less-than sign

                case 61:
                case 61501:
                    return { levelText: '=', fontFamily: 'Times New Roman' };  // Equals sign

                case 62:
                case 61502:
                    return { levelText: '>', fontFamily: 'Times New Roman' };  // Greater-than sign

                case 63:
                case 61503:
                    return { levelText: '?', fontFamily: 'Times New Roman' };  // Question mark

                case 64:
                case 61504:
                    return { levelText: '\u2245', fontFamily: 'Arial' };  // Approximately equal to

                case 65:
                case 61505:
                    return { levelText: '\u0391', fontFamily: 'Times New Roman' };  // Greek capital letter alpha

                case 66:
                case 61506:
                    return { levelText: '\u0392', fontFamily: 'Times New Roman' };  // Greek capital letter beta

                case 67:
                case 61507:
                    return { levelText: '\u03a7', fontFamily: 'Times New Roman' };  // Greek capital letter chi

                case 68:
                case 61508:
                    return { levelText: '\u0394', fontFamily: 'Times New Roman' };  // Greek capital letter delta

                case 69:
                case 61509:
                    return { levelText: '\u0395', fontFamily: 'Times New Roman' };  // Greek capital letter epsilon

                case 70:
                case 61510:
                    return { levelText: '\u03a6', fontFamily: 'Times New Roman' };  // Greek capital letter phi

                case 71:
                case 61511:
                    return { levelText: '\u0393', fontFamily: 'Times New Roman' };  // Greek capital letter gamma

                case 72:
                case 61512:
                    return { levelText: '\u0397', fontFamily: 'Times New Roman' };  // Greek capital letter eta

                case 73:
                case 61513:
                    return { levelText: '\u0399', fontFamily: 'Times New Roman' };  // Greek capital letter iota

                case 74:
                case 61514:
                    return { levelText: '\u03d1', fontFamily: 'Arial' };  // Greek theta symbol

                case 75:
                case 61515:
                    return { levelText: '\u039a', fontFamily: 'Times New Roman' };  // Greek capital letter kappa

                case 76:
                case 61516:
                    return { levelText: '\u039b', fontFamily: 'Times New Roman' };  // Greek capital letter lambda

                case 77:
                case 61517:
                    return { levelText: '\u039c', fontFamily: 'Times New Roman' };  // Greek capital letter mu

                case 78:
                case 61518:
                    return { levelText: '\u039d', fontFamily: 'Times New Roman' };  // Greek capital letter nu

                case 79:
                case 61519:
                    return { levelText: '\u039f', fontFamily: 'Times New Roman' };  // Greek capital letter omicron

                case 80:
                case 61520:
                    return { levelText: '\u03a0', fontFamily: 'Times New Roman' };  // Greek capital letter pi

                case 81:
                case 61521:
                    return { levelText: '\u0398', fontFamily: 'Times New Roman' };  // Greek capital letter theta

                case 82:
                case 61522:
                    return { levelText: '\u03a1', fontFamily: 'Times New Roman' };  // Greek capital letter rho

                case 83:
                case 61523:
                    return { levelText: '\u03a3', fontFamily: 'Times New Roman' };  // Greek capital letter sigma

                case 84:
                case 61524:
                    return { levelText: '\u03a4', fontFamily: 'Times New Roman' };  // Greek capital letter tau

                case 85:
                case 61525:
                    return { levelText: '\u03a5', fontFamily: 'Times New Roman' };  // Greek capital letter upsilon

                case 86:
                case 61526:
                    return { levelText: '\u03c2', fontFamily: 'Times New Roman' };  // Greek small letter final sigma

                case 87:
                case 61527:
                    return { levelText: '\u03a9', fontFamily: 'Times New Roman' };  // Greek capital letter omega

                case 88:
                case 61528:
                    return { levelText: '\u039e', fontFamily: 'Times New Roman' };  // Greek capital letter xi

                case 89:
                case 61529:
                    return { levelText: '\u03a8', fontFamily: 'Times New Roman' };  // Greek capital letter psi

                case 90:
                case 61530:
                    return { levelText: '\u0396', fontFamily: 'Times New Roman' };  // Greek capital letter zeta

                case 91:
                case 61531:
                    return { levelText: '[', fontFamily: 'Times New Roman' };  // Left square bracket

                case 92:
                case 61532:
                    return { levelText: '\u2234', fontFamily: 'Arial' };  // Therefore

                case 93:
                case 61533:
                    return { levelText: ']', fontFamily: 'Times New Roman' };  // Right square bracket

                case 94:
                case 61534:
                    return { levelText: '\u22a5', fontFamily: 'Arial' };  // Up tack

                case 95:
                case 61535:
                    return { levelText: '_', fontFamily: 'Times New Roman' };  // Low line

                case 96:
                case 61536:
                    return { levelText: '\xaf', fontFamily: 'Times New Roman' };  // Macron

                case 97:
                case 61537:
                    return { levelText: '\u03b1', fontFamily: 'Times New Roman' };  // Greek small letter alpha

                case 98:
                case 61538:
                    return { levelText: '\u03b2', fontFamily: 'Times New Roman' };  // Greek small letter beta

                case 99:
                case 61539:
                    return { levelText: '\u03c7', fontFamily: 'Times New Roman' };  // Greek small letter chi

                case 100:
                case 61540:
                    return { levelText: '\u03b4', fontFamily: 'Times New Roman' };  // Greek small letter delta

                case 101:
                case 61541:
                    return { levelText: '\u03b5', fontFamily: 'Times New Roman' };  // Greek small letter epsilon

                case 102:
                case 61542:
                    return { levelText: '\u03d5', fontFamily: 'Arial' };  // Greek phi symbol

                case 103:
                case 61543:
                    return { levelText: '\u03b3', fontFamily: 'Times New Roman' };  // Greek small letter gamma

                case 104:
                case 61544:
                    return { levelText: '\u03b7', fontFamily: 'Times New Roman' };  // Greek small letter eta

                case 105:
                case 61545:
                    return { levelText: '\u03b9', fontFamily: 'Times New Roman' };  // Greek small letter iota

                case 106:
                case 61546:
                    return { levelText: '\u03c6', fontFamily: 'Times New Roman' };  // Greek small letter phi

                case 107:
                case 61547:
                    return { levelText: '\u03ba', fontFamily: 'Times New Roman' };  // Greek small letter kappa

                case 108:
                case 61548:
                    return { levelText: '\u03bb', fontFamily: 'Times New Roman' };  // Greek small letter lamda

                case 109:
                case 61549:
                    return { levelText: '\u03bc', fontFamily: 'Times New Roman' };  // Greek small letter mu

                case 110:
                case 61550:
                    return { levelText: '\u03bd', fontFamily: 'Times New Roman' };  // Greek small letter nu

                case 111:
                case 61551:
                    return { levelText: '\u03bf', fontFamily: 'Times New Roman' };  // Greek small letter omicron

                case 112:
                case 61552:
                    return { levelText: '\u03c0', fontFamily: 'Times New Roman' };  // Greek small letter pi

                case 113:
                case 61553:
                    return { levelText: '\u03b8', fontFamily: 'Times New Roman' };  // Greek small letter theta

                case 114:
                case 61554:
                    return { levelText: '\u03c1', fontFamily: 'Times New Roman' };  // Greek small letter rho

                case 115:
                case 61555:
                    return { levelText: '\u03c3', fontFamily: 'Times New Roman' };  // Greek small letter sigma

                case 116:
                case 61556:
                    return { levelText: '\u03c4', fontFamily: 'Times New Roman' };  // Greek small letter tau

                case 117:
                case 61557:
                    return { levelText: '\u03c5', fontFamily: 'Times New Roman' };  // Greek small letter upsilon

                case 118:
                case 61558:
                    return { levelText: '\u03d6', fontFamily: 'Arial' };  // Greek pi symbol

                case 119:
                case 61559:
                    return { levelText: '\u03c9', fontFamily: 'Times New Roman' };  // Greek small letter omega

                case 120:
                case 61560:
                    return { levelText: '\u03be', fontFamily: 'Times New Roman' };  // Greek small letter xi

                case 121:
                case 61561:
                    return { levelText: '\u03c8', fontFamily: 'Times New Roman' };  // Greek small letter psi

                case 122:
                case 61562:
                    return { levelText: '\u03b6', fontFamily: 'Times New Roman' };  // Greek small letter zeta

                case 123:
                case 61563:
                    return { levelText: '{', fontFamily: 'Times New Roman' };  // Left curly bracket

                case 124:
                case 61564:
                    return { levelText: '|', fontFamily: 'Times New Roman' };  // Vertical line

                case 125:
                case 61565:
                    return { levelText: '}', fontFamily: 'Times New Roman' };  // Right curly bracket

                case 126:
                case 61566:
                    return { levelText: '\u223c', fontFamily: 'Arial' };  // Tilde operator

                case 160:
                case 61600:
                    return { levelText: '\u20ac', fontFamily: 'Times New Roman' };  // Euro sign

                case 161:
                case 61601:
                    return { levelText: '\u03d2', fontFamily: 'Arial' };  // Greek upsilon with hook symbol

                case 162:
                case 61602:
                    return { levelText: '\u2032', fontFamily: 'Times New Roman' };  // Prime

                case 163:
                case 61603:
                    return { levelText: '\u2264', fontFamily: 'Times New Roman' };  // Less-than or equal to

                case 164:
                case 61604:
                    return { levelText: '\u2044', fontFamily: 'Times New Roman' };  // Fraction slash

                case 165:
                case 61605:
                    return { levelText: '\u221e', fontFamily: 'Times New Roman' };  // Infinity

                case 166:
                case 61606:
                    return { levelText: '\u0192', fontFamily: 'Times New Roman' };  // Latin small letter f with hook

                case 167:
                case 61607:
                    return { levelText: '\u2663', fontFamily: 'Times New Roman' };  // Black club suit

                case 168:
                case 61608:
                    return { levelText: '\u2666', fontFamily: 'Times New Roman' };  // Black diamond suit

                case 169:
                case 61609:
                    return { levelText: '\u2665', fontFamily: 'Times New Roman' };  // Black heart suit

                case 170:
                case 61610:
                    return { levelText: '\u2660', fontFamily: 'Times New Roman' };  // Black spade suit

                case 171:
                case 61611:
                    return { levelText: '\u2194', fontFamily: 'Times New Roman' };  // Left rigth arrow

                case 172:
                case 61612:
                    return { levelText: '\u2190', fontFamily: 'Times New Roman' };  // Leftwards arrow

                case 173:
                case 61613:
                    return { levelText: '\u2191', fontFamily: 'Times New Roman' };  // Upwards arrow

                case 174:
                case 61614:
                    return { levelText: '\u2192', fontFamily: 'Times New Roman' };  // Rigthwards arrow

                case 175:
                case 61615:
                    return { levelText: '\u2193', fontFamily: 'Times New Roman' };  // Downwards arrow

                case 176:
                case 61616:
                    return { levelText: '\xb0', fontFamily: 'Times New Roman' };  // Degree sign

                case 177:
                case 61617:
                    return { levelText: '\xb1', fontFamily: 'Times New Roman' };  // Plus-minus sign

                case 178:
                case 61618:
                    return { levelText: '\u2033', fontFamily: 'Times New Roman' };  // Double prime

                case 179:
                case 61619:
                    return { levelText: '\u2265', fontFamily: 'Times New Roman' };  // Greater-than or equal to

                case 180:
                case 61620:
                    return { levelText: '\xd7', fontFamily: 'Times New Roman' };  // Multiplication sign

                case 181:
                case 61621:
                    return { levelText: '\u221d', fontFamily: 'Arial' };  // Proportional to

                case 182:
                case 61622:
                    return { levelText: '\u2202', fontFamily: 'Times New Roman' };  // Partial differential

                case 183:
                case 61623:
                    return { levelText: '\u2022', fontFamily: 'Times New Roman' };  // Bullet

                case 184:
                case 61624:
                    return { levelText: '\xf7', fontFamily: 'Times New Roman' };  // Division sign

                case 185:
                case 61625:
                    return { levelText: '\u2260', fontFamily: 'Times New Roman' };  // Not equal to

                case 186:
                case 61626:
                    return { levelText: '\u2261', fontFamily: 'Times New Roman' };  // Identical to

                case 187:
                case 61627:
                    return { levelText: '\u2248', fontFamily: 'Times New Roman' };  // Almost equal sign

                case 188:
                case 61628:
                    return { levelText: Utils.ELLIPSIS_CHAR, fontFamily: 'Times New Roman' };  // Horizontal ellipsis

                case 189:
                case 61629:
                    return { levelText: '|', fontFamily: 'Times New Roman' };  // Pipe symbol

                case 190:
                case 61630:
                    return { levelText: '\u2014', fontFamily: 'Times New Roman' };  // Horizontal line

                case 191:
                case 61631:
                    return { levelText: '\u21b5', fontFamily: 'Arial' };  // Downward arrow with corner leftwards

                case 192:
                case 61632:
                    return { levelText: '\u2135', fontFamily: 'Arial' };  // Alef symbol

                case 193:
                case 61633:
                    return { levelText: '\u2111', fontFamily: 'Arial' };  // Black-letter capital I

                case 194:
                case 61634:
                    return { levelText: '\u211c', fontFamily: 'Arial' };  // Black-letter capital R

                case 195:
                case 61635:
                    return { levelText: '\u2118', fontFamily: 'Arial' };  // Script capital P

                case 196:
                case 61636:
                    return { levelText: '\u2297', fontFamily: 'Arial' };  // Circled times

                case 197:
                case 61637:
                    return { levelText: '\u2295', fontFamily: 'Arial' };  // Circled plus

                case 198:
                case 61638:
                    return { levelText: '\u2205', fontFamily: 'Arial' };  // Empty set

                case 199:
                case 61639:
                    return { levelText: '\u2229', fontFamily: 'Times New Roman' };  // Intersection

                case 200:
                case 61640:
                    return { levelText: '\u222a', fontFamily: 'Arial' };  // Union

                case 201:
                case 61641:
                    return { levelText: '\u2283', fontFamily: 'Arial' };  // Superset of

                case 202:
                case 61642:
                    return { levelText: '\u2287', fontFamily: 'Arial' };  // Superset of or equal to

                case 203:
                case 61643:
                    return { levelText: '\u2284', fontFamily: 'Arial' };  // Not a subset of

                case 204:
                case 61644:
                    return { levelText: '\u2282', fontFamily: 'Arial' };  // Subset of

                case 205:
                case 61645:
                    return { levelText: '\u2286', fontFamily: 'Arial' };  // Subset of or equal to

                case 206:
                case 61646:
                    return { levelText: '\u2208', fontFamily: 'Arial' };  // Element of

                case 207:
                case 61647:
                    return { levelText: '\u2209', fontFamily: 'Arial' };  // Not an element of

                case 208:
                case 61648:
                    return { levelText: '\u2220', fontFamily: 'Arial' };  // Angle

                case 209:
                case 61649:
                    return { levelText: '\u2207', fontFamily: 'Arial' };  // Nabla

                case 210:
                case 61650:
                    return { levelText: '\xae', fontFamily: 'Times New Roman' };  // Registered sign

                case 211:
                case 61651:
                    return { levelText: '\xa9', fontFamily: 'Times New Roman' };  // Copyright sign

                case 212:
                case 61652:
                    return { levelText: '\u2122', fontFamily: 'Times New Roman' };  // Trade mark sign

                case 213:
                case 61653:
                    return { levelText: '\u220f', fontFamily: 'Times New Roman' };  // N-ary product

                case 214:
                case 61654:
                    return { levelText: '\u221a', fontFamily: 'Times New Roman' };  // Square root

                case 215:
                case 61655:
                    return { levelText: '\u22c5', fontFamily: 'Arial' };  // Dot operator

                case 216:
                case 61656:
                    return { levelText: '\xac', fontFamily: 'Times New Roman' };  // Not sign

                case 217:
                case 61657:
                    return { levelText: '\u2227', fontFamily: 'Arial' };  // Logical and

                case 218:
                case 61658:
                    return { levelText: '\u2228', fontFamily: 'Arial' };  // Logical or

                case 219:
                case 61659:
                    return { levelText: '\u21d4', fontFamily: 'Arial' };  // Left right double arrow

                case 220:
                case 61660:
                    return { levelText: '\u21d0', fontFamily: 'Arial' };  // Leftwards double arrow

                case 221:
                case 61661:
                    return { levelText: '\u21d1', fontFamily: 'Arial' };  // Upwards double arrow

                case 222:
                case 61662:
                    return { levelText: '\u21d2', fontFamily: 'Arial' };  // Rightwards double arrow

                case 223:
                case 61663:
                    return { levelText: '\u21d3', fontFamily: 'Arial' };  // Downwards double arrow

                case 224:
                case 61664:
                    return { levelText: '\u25ca', fontFamily: 'Times New Roman' };  // Lozenge

                case 225:
                case 61665:
                    return { levelText: '\u2329', fontFamily: 'Arial' };  // Left-pointing angle bracket

                case 226:
                case 61666:
                    return { levelText: '\xae', fontFamily: 'Arial' };  // Registered sign

                case 227:
                case 61667:
                    return { levelText: '\xa9', fontFamily: 'Arial' };  // Copyright sign

                case 228:
                case 61668:
                    return { levelText: '\u2122', fontFamily: 'Arial' };  // Trade mark sign (sans serif)

                case 229:
                case 61669:
                    return { levelText: '\u2211', fontFamily: 'Times New Roman' };  // N-ary summation

                case 230:
                case 61670:
                    return { levelText: '\u239b', fontFamily: 'Arial' };  // Left parenthesis upper hook

                case 231:
                case 61671:
                    return { levelText: '\u239c', fontFamily: 'Arial' };  // Left parenthesis extension

                case 232:
                case 61672:
                    return { levelText: '\u239d', fontFamily: 'Arial' };  // Left parenthesis lower hook

                case 233:
                case 61673:
                    return { levelText: '\u23a1', fontFamily: 'Arial' };  // Left square bracket upper corner

                case 234:
                case 61674:
                    return { levelText: '\u23a2', fontFamily: 'Arial' };  // Left square bracket extension

                case 235:
                case 61675:
                    return { levelText: '\u23a3', fontFamily: 'Arial' };  // Left square bracket lower corner

                case 236:
                case 61676:
                    return { levelText: '\u23a7', fontFamily: 'Arial' };  // Left curly bracket upper hook

                case 237:
                case 61677:
                    return { levelText: '\u23a8', fontFamily: 'Arial' };  // Left curly bracket middle piece

                case 238:
                case 61678:
                    return { levelText: '\u23a9', fontFamily: 'Arial' };  // Left curly bracket lower hook

                case 239:
                case 61679:
                    return { levelText: '\u23aa', fontFamily: 'Arial' };  // Curly bracket extension

                case 241:
                case 61681:
                    return { levelText: '\u232a', fontFamily: 'Arial' };  // Right-pointing angle bracket

                case 242:
                case 61682:
                    return { levelText: '\u222b', fontFamily: 'Times New Roman' };  // Integral

                case 243:
                case 61683:
                    return { levelText: '\u2320', fontFamily: 'Times New Roman' };  // Top half integral

                case 244:
                case 61684:
                    return { levelText: '\u23ae', fontFamily: 'Arial' };  // Integral extension

                case 245:
                case 61685:
                    return { levelText: '\u2321', fontFamily: 'Times New Roman' };  // Bottom half integral

                case 246:
                case 61686:
                    return { levelText: '\u239e', fontFamily: 'Arial' };  // Right parenthesis upper hook

                case 247:
                case 61687:
                    return { levelText: '\u239f', fontFamily: 'Arial' };  // Right parenthesis extension

                case 248:
                case 61688:
                    return { levelText: '\u23a0', fontFamily: 'Arial' };  // Right parenthesis lower hook

                case 249:
                case 61689:
                    return { levelText: '\u23a4', fontFamily: 'Arial' };  // Right square bracket upper corner

                case 250:
                case 61690:
                    return { levelText: '\u23a5', fontFamily: 'Arial' };  // Right square bracket extension

                case 251:
                case 61691:
                    return { levelText: '\u23a6', fontFamily: 'Arial' };  // Right square bracket lower corner

                case 252:
                case 61692:
                    return { levelText: '\u23ab', fontFamily: 'Arial' };  // Right curly bracket upper hook

                case 253:
                case 61693:
                    return { levelText: '\u23ac', fontFamily: 'Arial' };  // Right curly bracket middle piece

                case 254:
                case 61694:
                    return { levelText: '\u23ad', fontFamily: 'Arial' };  // Right curly bracket lower hook

                default:
                    return { levelText: '\u2022', fontFamily: 'Times New Roman' };
            }
        };

        /**
         * Maps OpenSymbol characters of the Unicode private use area to a replacement
         * that can be displayed in common browser fonts.
         *
         * @param {String} char
         *  The Symbol char.
         *
         * @returns {Object}
         *              {String} levelText
         *                  The Unicode char.
         *              {String} fontFamily
         *                  The font family of the char.
         */
        this.getUnicodeForOpenSymbolChar = function (char) {

            if (!_.isString(char) || (char.length !== 1)) {
                return { levelText: '\u2022', fontFamily: 'Times New Roman' };
            }

            if (_.isObject(OPEN_SYMBOL_REPLACEMENT[char])) {
                // take private use area replacement
                return _.copy(OPEN_SYMBOL_REPLACEMENT[char]);
            }

            // take the character as it is
            return { levelText: char, fontFamily: 'Times New Roman' };
        };

        this.getCssStyle = function (child, family) {
            function colorResolver() {
                if (css.color) {
                    var colorDescriptor = ColorUtils.parseColor(css.color);
                    return (colorDescriptor) ? { type: 'rgb', value: colorDescriptor.hex } : null;

                } else {
                    return null;
                }
            }

            function backgroundcolorResolver() {
                if (css['background-color'] && css['background-color'] !== 'rgba(0, 0, 0, 0)' && css['background-color'] !== 'transparent') {
                    var colorDescriptor = ColorUtils.parseColor(css['background-color']);
                    return (colorDescriptor) ? { type: 'rgb', value: colorDescriptor.hex } : null;

                } else {
                    return null;
                }
            }

            function fontWeightResolver() {
                return (css['font-weight'] && (css['font-weight'] === 'bold' || parseInt(css['font-weight'], 10) > 400));
            }

            function fontSizeResolver() {
                return (css['font-size']) ? Utils.convertCssLength(css['font-size'], 'pt') : null;
            }

            function italicResolver() {
                return (css['font-style']) ? (css['font-style'] === 'italic') : null;
            }

            function underlineResolver() {
                return (css['text-decoration']) ? (css['text-decoration'].indexOf('underline') !== -1) : null;
            }

            function strikeResolver() {
                return (css['text-decoration']) ? (css['text-decoration'].indexOf('line-through') !== -1) ? 'single' : null : null;
            }

            var css         = Utils.getCss(child.parentNode, { attrs: ['color', 'background-color', 'font-weight', 'font-size', 'text-decoration', 'font-style'] }),
                attrs       = {},
                attributes  = {
                    character:      {
                        color:      colorResolver,
                        fillColor:  backgroundcolorResolver,
                        fontSize:   fontSizeResolver,
                        bold:       fontWeightResolver,
                        italic:     italicResolver,
                        strike:     strikeResolver,
                        underline:  underlineResolver
                    }
                };

            _.each(attributes[family], function (resolver, key) {
                attrs[key] = resolver();
            });

            return attrs;
        };

        // initialization -----------------------------------------------------

        // initialize class members

        // register event handlers for view events

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = self = null;
        });

    } // class ClipboardMixin

    // exports ================================================================

    var promise = IO.loadJSON('io.ox/office/textframework/resource/fontreplacements/opensymbol');

    promise.done(function (opensymbol) { OPEN_SYMBOL_REPLACEMENT = opensymbol; });

    return promise.then(_.constant(ClipboardMixin));
});
