/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/textframework/model/attributeoperationmixin', [
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/textframework/components/table/table',
    'io.ox/office/textframework/format/tablestyles',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/snapshot',
    'io.ox/office/textframework/utils/textutils',
    'gettext!io.ox/office/textframework/main'
], function (DrawingFrame, AttributeUtils, Table, TableStyles, DOM, Position, Operations, Snapshot, Utils, gt) {

    'use strict';

    // mix-in class AttributeOperationMixin ======================================

    var formatPainterWhitelist = {
        character: {
            character: [
                'anchor',
                'baseline',
                'bold',
                'caps',
                'color',
                'fillColor',
                'fontName',
                'fontNameComplex',
                'fontNameEastAsia',
                'fontNameSymbol',
                'fontSize',
                'italic',
                'strike',
                'underline',
                'vertAlign'
            ]
        },
        paragraph: {
            paragraph: [
                'alignment',
                'bullet',
                'bulletColor',
                'bulletFont',
                'bulletSize',
                'defaultTabSize',
                'fillColor',
                'indentFirstLine',
                'indentLeft',
                'indentRight',
                'level',
                'lineHeight',
                'listLabelHidden',
                'listLevel',
                'listStyleId',
                'listStartValue',
                'outlineLevel',
                'spacingAfter',
                'spacingBefore',
                'tabStops',
                'styleId'
            ]
        },
        image: {
            line: [
                'bitmap',
                'color',
                'color2',
                'gradient',
                'pattern',
                'style',
                'type',
                'width'
            ],
            fill: [
                'bitmap',
                'color',
                'color2',
                'gradient',
                'pattern',
                'type'
            ]
        },
        connector: {
            line: [
                'bitmap',
                'color',
                'color2',
                'gradient',
                'headEndLength',
                'headEndType',
                'headEndWidth',
                'pattern',
                'style',
                'tailEndLength',
                'tailEndType',
                'tailEndWidth',
                'type',
                'width'
            ]
        },
        shape: {
            character: [
                'anchor',
                'baseline',
                'bold',
                'caps',
                'color',
                'fillColor',
                'fontName',
                'fontNameComplex',
                'fontNameEastAsia',
                'fontNameSymbol',
                'fontSize',
                'italic',
                'strike',
                'underline',
                'vertAlign'
            ],
            paragraph: [
                'alignment',
                'bullet',
                'bulletColor',
                'bulletFont',
                'bulletSize',
                'defaultTabSize',
                'fillColor',
                'indentFirstLine',
                'indentLeft',
                'indentRight',
                'level',
                'lineHeight',
                'listLabelHidden',
                'listStartValue',
                'outlineLevel',
                'spacingAfter',
                'spacingBefore',
                'tabStops'
            ],
            shape: [
                'anchor',
                'anchorCentered',
                'autoResizeText',
                'fontScale',
                'horzOverflow',
                'lineReduction',
                'noAutoResize',
                'vert',
                'vertOverflow',
                'wordWrap'
            ],
            line: [
                'bitmap',
                'color',
                'color2',
                'gradient',
                'pattern',
                'style',
                'type',
                'width'
            ],
            fill: [
                'bitmap',
                'color',
                'color2',
                'gradient',
                'pattern',
                'type'
            ]
        },
        table: {
            table: [
                'borderBottom',
                'borderInsideHor',
                'borderInsideVert',
                'borderLeft',
                'borderRight',
                'borderTop',
                'exclude',
                'fillColor',
                'paddingBottom',
                'paddingLeft',
                'paddingRight',
                'paddingTop',
                'tableGrid',
                'width'
            ]
        }
    };

    /**
     * A mix-in class for the document model class providing the operation
     * handling for setting attributes used in a presentation and text document.
     *
     * @constructor
     *
     * @param {EditApplication} app
     *  The application instance.
     */
    function AttributeOperationMixin(app) {

        var // self reference for local functions
            self = this,
            // a list of drawing filters to filter multi drawing selection
            drawingFilterList = [],
            // FormatPainter
            formatPainterActive = false,
            formatPainterAttrs = {};

        // private methods ----------------------------------------------------

        /**
         * Registering filter functions to avoid that attributes are assigned to drawings
         * that they do not support. This can happen in the case of multi drawing selections.
         * Not in every case the button to set an attribute is disabled, if the current
         * selection contains more than one drawing. For example the 'fillColor' can be set,
         * if drawings of type 'connector' are part of the multi drawing selection. In this
         * case the drawing selection must be decreased, before the operations are generated.
         *
         * Addtionally these filter are used to reduce the specified attributes in the 'getter'
         * function. In a multi selection or a group selection it can happen, that a line and
         * a circle with yellow background are selected. In this case, the background color of
         * the merged attributes shall be yellow, because the line does not support this family.
         * In this case it is necessary to remove the fill-family from the attribute set.
         *
         * Info: This function is called during the initialization phase. Additional drawing
         *       filters can be specified within this function.
         *
         * Info: Every filter gets as parameter the jQuerified drawing node and the attributes
         *       object as it is used as parameter for the this.setAttributes function.
         *       If the filter function returns 'false', the drawing will NOT get the specified
         *       attributes. If it returns 'true', it is possible to generate an operation
         *       for the specified drawing and the specified attributes.
         *       Optionally the filter can get the 'deleteAttrs' flag. This is required for the
         *       getter function, that collects the attributes for all selected drawings. This can
         *       be used to remove the fill attributes from drawings, that do not support this.
         */
        function registerDrawingFilter() {

            // shapes, charts, images and groups support fill color
            var fillColorFilter = function (drawing, attributes, deleteAttrs) {
                if (attributes.fill && !AttributeOperationMixin.FILL_DRAWING_TYPES[DrawingFrame.getDrawingType(drawing)]) {
                    if (deleteAttrs) { attributes.fill = {}; } // empty object required for merging of attributes
                    return false;
                }
                return true;
            };

            // shapes, charts, images, connectors and groups support line color
            var lineColorFilter = function (drawing, attributes, deleteAttrs) {
                if (attributes.line && !AttributeOperationMixin.LINE_DRAWING_TYPES[DrawingFrame.getDrawingType(drawing)]) {
                    if (deleteAttrs) { attributes.line = {}; } // empty object required for merging of attributes
                    return false;
                }
                return true;
            };

            // further filters can be specified here ...

            // adding the filter to the global filter list
            drawingFilterList.push(fillColorFilter);
            drawingFilterList.push(lineColorFilter);
        }

        /**
         * Returns the family of the attributes supported by the passed DOM
         * element.
         *
         * @param {HTMLElement|jQuery} element
         *  The DOM element whose associated attribute family will be returned.
         *  If this object is a jQuery collection, returns its first node.
         */
        function resolveElementFamily(element) {

            var // the element, as jQuery object
                $element = $(element),
                // the resulting style family
                family = null;

            if (DrawingFrame.isDrawingFrame($element)) {
                family = 'drawing'; // selecting drawing first (necessary after 39312)
            } else if (DOM.isPartOfParagraph($element)) {
                family = 'character';
            } else if (DOM.isSlideNode($element)) {
                family = 'slide';
            } else if (DOM.isParagraphNode($element)) {
                family = 'paragraph';
            } else if (DOM.isTableNode($element)) {
                family = 'table';
            } else if ($element.is('tr')) {
                family = 'row';
            } else if ($element.is('td')) {
                family = 'cell';
            } else {
                Utils.warn('Editor.resolveElementFamily(): unsupported element');
            }

            return family;
        }

        /**
         * Returning the list style id from specified paragraph attributes. The list
         * style can be set directly or via a style id at the paragraph.
         *
         * @param {Object} [attributes]
         *  An optional map of attribute maps (name/value pairs), keyed by attribute.
         *  It this parameter is defined, the parameter 'paragraph' can be omitted.
         *
         * @returns {String|null}
         *  The list style id or null, if no list style defined in specified attributes.
         */
        function getListStyleIdFromParaAttrs(attributes) {

            // shortcut, not handling style attributes
            if (attributes && attributes.paragraph && attributes.paragraph.listStyleId) { return attributes.paragraph.listStyleId; }

            // checking paragraph style
            if (attributes && attributes.styleId && self.isParagraphStyleWithListStyle(attributes.styleId)) { return getListStyleInfoFromStyleId(attributes.styleId, 'listStyleId'); }

            return null;
        }

        /**
         * Returns the value of a specified paragraph attribute of a given
         * paragraph style.
         *
         * @param {String} styleId
         *  The paragraph style id, that will be checked checked.
         *
         * @param {String} paraAttr
         *  The attribute of the paragraph attributes of the style, whose value
         *  shall be returned.
         *
         * @returns {String}
         *  The value of the specified paragraph attribute of the specified style.
         */
        function getListStyleInfoFromStyleId(styleId, paraAttr) {

            var // the style attributes
                styleAttrs = self.getParagraphStyles().getStyleSheetAttributeMap(styleId);

            return (styleAttrs && styleAttrs.paragraph && styleAttrs.paragraph[paraAttr]);
        }

        /**
         * Updating the neighbor paragraphs, if the list level, border, color or margin of the specified paragraph
         * was modified.
         *
         * @param {Node|jQuery} paragraph
         *  The paragraph node that gets new attributes assigned.
         *
         * @param {Object} attributes
         *  The container with the modified attributes of the paragraph.
         */
        function handleNeighborUpdate(paragraph, attributes) {

            var // the previous and the next paragraph nodes
                prevParagraph = null, nextParagraph = null;

            // update neighbors when listLevel, a border, the margin or the fillColor is changed (note: null must also be checked, there are some cases where the attribute has changed and the value is null)
            if (attributes && attributes.paragraph && (_.isNumber(attributes.paragraph.listLevel) || attributes.paragraph.borderLeft || attributes.paragraph.borderRight || attributes.paragraph.borderTop || attributes.paragraph.borderBottom || attributes.paragraph.borderInside || _.isNumber(attributes.paragraph.marginTop) || _.isNumber(attributes.paragraph.marginBottom) || attributes.paragraph.fillColor || attributes.paragraph.fillColor === null || attributes.paragraph.borderLeft === null || attributes.paragraph.borderRight === null || attributes.paragraph.borderTop === null || attributes.paragraph.borderBottom === null || attributes.paragraph.borderInside === null)) {

                // get the neighbors
                prevParagraph = $(paragraph).prev();
                nextParagraph = $(paragraph).next();

                // also updating the neighboring paragraphs
                if (prevParagraph && prevParagraph.length > 0 && DOM.isParagraphNode(prevParagraph)) { self.implParagraphChanged(prevParagraph); }
                if (nextParagraph && nextParagraph.length > 0 && DOM.isParagraphNode(nextParagraph)) { self.implParagraphChanged(nextParagraph); }
            }

        }

        /**
         * Updating the table node inside a specified table drawing node.
         *
         * @param {Node|jQuery} drawing
         *  The drawing node whose table node gets new attributes assigned.
         *
         * @param {Object} attributes
         *  The container with the modified attributes of the table drawing node.
         */
        function setTableAttributesInTableDrawing(drawing, attributes) {

            var tableNode = $(drawing).find('table'),
                tableAttrs = null;

            if (attributes.table) { tableAttrs = { table:  attributes.table }; }

            if (attributes.styleId) {
                tableAttrs = tableAttrs || {};
                tableAttrs.styleId = attributes.styleId;
            }

            if (tableAttrs) { self.getTableStyles().setElementAttributes(tableNode, tableAttrs); }
        }

        /**
         * Reducing a set of specified drawings corresponding the the global drawing filter list.
         * This is necessary to avoid, that drawings get attributes assigned, that they do not
         * support.
         *
         * @param {jQuery[]} drawings
         *  A collector of drawing nodes.
         *
         * @param {Object} attributes
         *  A map of attribute value maps (name/value pairs), keyed by
         *  attribute families.
         *
         * @returns {undefined}
         *  A filtered collector of drawing nodes.
         */
        function filterDrawingNodes(drawings, attributes) {

            var // the local reference to the collector of drawing nodes.
                filteredDrawings = drawings;

            _.each(drawingFilterList, function (oneFilter) {
                filteredDrawings = _.filter(filteredDrawings, function (oneDrawing) { return oneFilter(oneDrawing, attributes); });
            });

            return filteredDrawings;
        }

        /**
         * Filtering the drawing attributes, so that not all drawing attributes are merged into the
         * set of merged attributes. For example the fill family attributes of a connector shall not
         * be used for the merged attributes. For example, if a cicle with yellow background is
         * selected together with a line (without any background), the mixture of all attributes
         * shall still be yellow. To achieve this, the fill-family has to be removed from the
         * attribute set of the line.
         *
         * @param {jQuery[]} drawings
         *  A drawing node.
         *
         * @param {Object} attributes
         *  A map of attribute value maps (name/value pairs), keyed by attribute families. This object
         *  will be modified in the filter function.
         */
        function filterDrawingAttrs(drawing, attributes) {
            _.each(drawingFilterList, function (oneFilter) {
                oneFilter(drawing, attributes, true); // third parameter is true -> deleting attributes in filter
            });
        }

        /**
         * Check, whether a specified text span is an empty span in an empty paragraph in a
         * selected empty drawing text frame.
         *
         * @param {Node|jQuery} span
         *  The text span.
         *
         * @returns {Boolean}
         *  Whether the specified span is empty and is located in an empty paragraph without
         *  neighbors inside a selected drawing.
         */
        function isEmptyTextFrameSelection(span) {

            var isEmptyTextFrameSelection = false;

            if (self.getSelection().isDrawingSelection() && span.firstChild.nodeValue.length === 0) {
                var paragraph = $(span).parent();
                if (DOM.isEmptyParagraph(paragraph) && DOM.isParagraphWithoutNeighbour(paragraph)) {
                    isEmptyTextFrameSelection = true;
                }
            }

            return isEmptyTextFrameSelection;
        }

        /**
         * Changes a specific formatting attribute of the specified element or
         * text range. The type of the attributes will be determined from the
         * specified range.
         *
         * @param {Number[]} start
         *  The logical start position of the element or text range to be
         *  formatted.
         *
         * @param {Number[]} [end]
         *  The logical end position of the element or text range to be
         *  formatted.
         *
         * @param {Object} attributes
         *  A map with formatting attribute values, mapped by the attribute
         *  names, and by attribute family names.
         *
         * @param {Boolean} external
         *
         * @param {Boolean} noUndo
         *  If set to true, no undo operation will be generated
         *
         */
        var implSetAttributes = (function () {

            // temporary array with only one entry, for use in Position functions
            var TMPPOSARRAY = [0];

            return function (start, end, attributes, target, external, noUndo) {

                var // node info for start/end position
                    startInfo = null, endInfo = null,
                    // the main attribute family of the target components
                    styleFamily = null,
                    // the style sheet container for the target components
                    styleSheets = null,
                    // a helper object to save the style sheets temporarely
                    savedStyleSheets = null,
                    // options for style collection method calls
                    options = null,
                    // the last text span visited by the character formatter
                    lastTextSpan = null,
                    // undo operations going into a single action
                    undoOperations = [],
                    // redo operation
                    redoOperation = null,
                    // Performance: Saving data for list updates
                    paraAttrs = null, listStyleId = null, listLevel = null, oldListStyleId = null,
                    // a helper attribute object with values set to null
                    nullAttributes = null,
                    // a helper iterator key
                    localkey = null,
                    //element which is passed to page breaks calculation
                    currentElement,
                    // the previous operation before this current operation
                    previousOperation = null,
                    // whether the paragraph cache can be reused (performance)
                    useCache = false,
                    // whether an update of formatting is required after assigning attributes (needed by spellcheck attribute)
                    forceUpdateFormatting = false,
                    // a data object, that is used for updating slides with master or layout target (presentation only)
                    forceLayoutFormattingData = null,
                    //flags for triggering page breaks
                    isCharOrDrawingOrRow,
                    isClearFormatting,
                    isAttributesInParagraph,
                    isPageBreakBeforeAttributeInParagraph,
                    isParSpacing,
                    isBorderTopBot,
                    isParaIndent,
                    // if target is present, root node is header or footer, otherwise editdiv
                    rootNode = self.getRootNode(target),
                    // the parent node
                    parentNode = null,
                    // the selection object
                    selection = self.getSelection(),
                    // a helper object for attribute modifications
                    modifiedAttributes = null,
                    // whether a special handling for character attributes at empty paragraphs was used
                    usedSpecialCharacterHandling = false,
                    // the unde manager object
                    undoManager = self.getUndoManager();

                // sets or clears the attributes using the current style sheet container
                // 'modifiedAttrs' is an optional parameter, that is only used, if the original attribute set was modified
                function setElementAttributes(element, modifiedAttrs) {
                    styleSheets.setElementAttributes(element, modifiedAttrs ? modifiedAttrs : attributes, options);
                }

                // change listener used to build the undo operations
                function changeListener(attributedNode, oldAttributes, newAttributes) {

                    if (noUndo) { return; }

                    var // the element node might be modified, if it is an empty text span (41250)
                        element = (!app.isODF() && DOM.isEmptySpan(attributedNode)) ? attributedNode.parentNode : attributedNode,
                        // selection object representing the passed element
                        range = Position.getPositionRangeForNode(rootNode, element),
                        // the attributes of the current family for the undo operation
                        undoAttributes = {},
                        // the operation used to undo the attribute changes
                        undoOperation = { name: Operations.SET_ATTRIBUTES, start: range.start, end: range.end, attrs: undoAttributes },
                        // last undo operation (used to merge character attributes of sibling text spans)
                        lastUndoOperation = ((undoOperations.length > 0) && (styleFamily === 'character')) ? _.last(undoOperations) : null,
                        // autoResizeHeight property for old and new attributes
                        oldAttrsAutoHeight, newAttrsAutoHeight;

                    function insertUndoAttribute(family, name, value) {
                        undoAttributes[family] = undoAttributes[family] || {};
                        undoAttributes[family][name] = value;
                    }

                    // using modified logical positions, if the attributed text span is a template text
                    // -> character attributes must be assigned to the paragraph
                    if (DOM.isTextFrameTemplateTextSpan(attributedNode)) {
                        undoOperation.start = _.initial(undoOperation.start);
                        undoOperation.end = _.clone(undoOperation.start);
                    }

                    // merging the place holder drawings into the old attributes object (presentation only)
                    if (forceLayoutFormattingData && forceLayoutFormattingData.oldAttrs) { oldAttributes = self.mergePlaceHolderAttributes(element, startInfo.family, oldAttributes, forceLayoutFormattingData.oldAttrs); }

                    // extend undoOperation if target is present
                    self.extendPropertiesWithTarget(undoOperation, target);

                    // exceptional case for setting line type to 'none', to save the old color
                    if (_.has(attributes, 'line') && _.has(attributes.line, 'type') && attributes.line.type === 'none') {
                        if (_.has(oldAttributes, 'line') && _.has(oldAttributes.line, 'color') && !_.isUndefined(oldAttributes.line.color)) {
                            undoAttributes.line = {};
                            undoAttributes.line.color = oldAttributes.line.color;
                        }
                    }

                    // exceptional case for setting auto-fit property in presentation, to preserve drawing height in undo, #46349
                    if (self.useSlideMode()) {
                        newAttrsAutoHeight = newAttributes && DrawingFrame.isAutoResizeHeightAttributes(newAttributes.shape);
                        oldAttrsAutoHeight = oldAttributes && DrawingFrame.isAutoResizeHeightAttributes(oldAttributes.shape);
                        if (newAttrsAutoHeight && !oldAttrsAutoHeight && oldAttributes && oldAttributes.drawing) {
                            undoAttributes.drawing = undoAttributes.drawing || {};
                            undoAttributes.drawing = _.extend(undoAttributes.drawing, oldAttributes.drawing);
                        }
                    }

                    // find all old attributes that have been changed or cleared
                    _(oldAttributes).each(function (attributeValues, family) {
                        if (family === 'styleId') {
                            // style sheet identifier is a string property
                            if (attributeValues !== newAttributes.styleId) {
                                undoAttributes.styleId = attributeValues;
                            }
                        } else {
                            // process attribute map of the current family
                            _(attributeValues).each(function (value, name) {
                                if (!(family in newAttributes) || !_.isEqual(value, newAttributes[family][name])) {
                                    insertUndoAttribute(family, name, value);
                                }
                            });
                        }
                    });

                    // find all newly added attributes
                    _(newAttributes).each(function (attributeValues, family) {
                        if (family === 'styleId') {
                            // style sheet identifier is a string property
                            if (!_.isString(oldAttributes.styleId)) {
                                undoAttributes.styleId = null;
                            }
                        } else {
                            // process attribute map of the current family
                            _(attributeValues).each(function (value, name) {
                                if (!(family in oldAttributes) || !(name in oldAttributes[family])) {
                                    insertUndoAttribute(family, name, null);
                                }
                            });
                        }
                    });

                    // try to merge 'character' undo operation with last array entry, otherwise add operation to array
                    if (lastUndoOperation && (_.last(lastUndoOperation.end) + 1 === _.last(undoOperation.start)) && _.isEqual(lastUndoOperation.attrs, undoOperation.attrs)) {
                        lastUndoOperation.end = undoOperation.end;
                    } else {
                        undoOperations.push(undoOperation);
                    }

                    // invalidate spell result if language attribute changes
                    self.getSpellChecker().resetClosest(element);
                }

                // fail if attributes is not of type object
                if (!_.isObject(attributes)) {
                    return false;
                }

                // do nothing if an empty attributes object has been passed
                if (_.isEmpty(attributes)) {
                    return;
                }

                // resolve start and end position
                if (!_.isArray(start)) {
                    Utils.warn('Editor.implSetAttributes(): missing start position');
                    return false;
                }

                // validating the logical positions (forced by 57048)
                if (!Position.isValidElementRange(rootNode, start, end)) {
                    Utils.warn('Editor.implSetAttributes(): invalid attribute range: ' + JSON.stringify(start) + ' ' + (end ? JSON.stringify(end) : ''));
                    return false;
                }

                // whether the cache can be reused (performance)
                previousOperation = self.getPreviousOperation();
                useCache = (!target && previousOperation && self.getParagraphCacheOperation()[previousOperation.name] &&
                            selection.getParagraphCache() && _.isEqual(_.initial(start), selection.getParagraphCache().pos) && _.isUndefined(previousOperation.target));

                if (useCache) {
                    TMPPOSARRAY[0] = _.last(start);
                    startInfo = Position.getDOMPosition(selection.getParagraphCache().node, TMPPOSARRAY, true);
                    // setting paragraph cache is not required, it can simply be reused in next operation
                } else {
                    startInfo = Position.getDOMPosition(rootNode, start, true);

                    if (!startInfo) {
                        Utils.warn('Editor.implSetAttributes(): invalid start node!');
                    }

                    if (startInfo.family === 'character') {
                        selection.setParagraphCache(startInfo.node.parentElement, _.clone(_.initial(start)), _.last(start));
                    } else {
                        selection.setParagraphCache(null); // invalidating the cache (previous operation might not be in PARAGRAPH_CACHE_OPERATIONS)
                    }
                }

                if (!startInfo || !startInfo.node) {
                    Utils.warn('Editor.implSetAttributes(): invalid start position: ' + JSON.stringify(start));
                    return false;
                }
                // get attribute family of start and end node
                startInfo.family = resolveElementFamily(startInfo.node);

                if (!startInfo.family) { return; }

                if (_.isArray(end)) {
                    if (startInfo.family === 'character') {
                        //characters are children of paragraph, end is in the same paragraph, so we dont have to search the complete document for it
                        TMPPOSARRAY[0] = _.last(end);
                        endInfo = Position.getDOMPosition(startInfo.node.parentElement, TMPPOSARRAY, true);
                    } else {
                        endInfo = Position.getDOMPosition(rootNode, end, true);
                    }

                    endInfo.family = resolveElementFamily(endInfo.node);

                    if (!endInfo.family) { return; }

                } else {
                    end = start;
                    endInfo = startInfo;
                }

                // options for the style collection method calls (build undo operations while formatting)
                options = undoManager.isUndoEnabled() ? { changeListener: changeListener } : null;

                // updating the slide model (the attributes are not only saved in the 'data' object, but also in the model itself)
                if (self.useSlideMode()) {
                    // updating the attributes of family 'slide'
                    if (startInfo.family === 'slide') {
                        self.updateSlideFamilyModel(start, attributes, target);
                        if (target && _.isFunction(self.isLayoutOrMasterId) && self.isLayoutOrMasterId(target) && attributes && attributes.fill) { forceLayoutFormattingData = { slideBackgroundUpdate: true, target: target, attrs: _.copy(attributes, true) }; } // only required for 48471
                    } else {
                        // updating the list style attributes at drawing nodes
                        if (startInfo.family === 'drawing' && self.getDrawingStyles().setDrawingListStyleAttributes) { self.getDrawingStyles().setDrawingListStyleAttributes(startInfo.node, attributes); }
                        // Special handling for presentation in master/layout slides -> updating styles (before calling setElementAttributes)
                        if (target && _.isFunction(self.isLayoutOrMasterId) && self.isLayoutOrMasterId(target)) { forceLayoutFormattingData = self.updateMasterLayoutStyles(styleFamily === 'character' ? startInfo.node.parentNode : startInfo.node, target, startInfo.family, attributes, { immediateUpdate: false, saveOldAttrs: true }); }
                    }
                }

                // characters (start or end may point to a drawing node, ignore that but format as
                // characters if the start object is different from the end object)
                if ((startInfo.family === 'character') || (endInfo.family === 'character') ||
                        ((startInfo.node !== endInfo.node) && DrawingFrame.isDrawingFrame(startInfo.node) && DrawingFrame.isDrawingFrame(endInfo.node))) {

                    // check that start and end are located in the same paragraph (and handling absolute positioned drawings correctly)
                    if (startInfo.node.parentNode !== endInfo.node.parentNode) {

                        // replacing the absolute positioned drawings by its place holder nodes
                        if (DOM.isDrawingLayerNode(startInfo.node.parentNode)) { startInfo.node = DOM.getDrawingPlaceHolderNode(startInfo.node); }
                        if (DOM.isDrawingLayerNode(endInfo.node.parentNode)) { endInfo.node = DOM.getDrawingPlaceHolderNode(endInfo.node); }

                        if (startInfo.node.parentNode !== endInfo.node.parentNode) {
                            Utils.warn('Editor.implSetAttributes(): end position in different paragraph');
                            return;
                        }
                    }

                    // visit all text span elements covered by the passed range
                    // (not only the direct children of the paragraph, but also
                    // text spans embedded in component nodes such as fields and tabs)
                    styleFamily = 'character';
                    styleSheets = self.getCharacterStyles();
                    Position.iterateParagraphChildNodes(startInfo.node.parentNode, function (node) {

                        // visiting the span inside a hard break node
                        // -> this is necessary for change tracking attributes
                        if (DOM.isHardBreakNode(node)) {
                            setElementAttributes(node.firstChild);
                        }
                        // if element is field, set fixed attribute (for automatic updating), also bug #43655
                        if (DOM.isComplexFieldNode(node) || DOM.isFieldNode(node) || DOM.isRangeMarkerNode(node)) {
                            setElementAttributes(node);
                            if (DOM.isFieldNode(node)) {
                                if (!self.useSlideMode()) {
                                    self.getFieldManager().addSimpleFieldToCollection(node, target);
                                }
                                $(node).css('font-size', ''); // see bug #47515
                            }
                        }

                        // DOM.iterateTextSpans() visits the node itself if it is a
                        // text span, otherwise it visits all descendant text spans
                        // contained in the node except for drawings which will be
                        // skipped (they may contain their own paragraphs).
                        DOM.iterateTextSpans(node, function (span) {
                            var convertBack = false;
                            // check for a spellchecked span (this needs to be checked, before attributes are applied)
                            if (!forceUpdateFormatting && self.isImportFinished() && DOM.isSpellerrorNode(span)) { forceUpdateFormatting = true; }

                            if (DOM.isSpecialField(node)) { // to properly apply attributes to special field span, they need to be unwrapped from parent node
                                self.getFieldManager().restoreSpecialField(node);
                                convertBack = true;
                            }

                            setElementAttributes(span);

                            if (convertBack) { // and later returned to previous state
                                var instruction = DOM.getComplexFieldInstruction(node);
                                var fieldType = (/NUMPAGES/i).test(instruction) ? 'NUMPAGES' : 'PAGE';
                                var id = DOM.getComplexFieldId(node);
                                self.getFieldManager().convertToSpecialField(id, target, fieldType);
                            }

                            // try to merge with the preceding text span
                            Utils.mergeSiblingTextSpans(span, false);
                            // remember span (last visited span will be merged with its next sibling)
                            lastTextSpan = span;
                        });

                    }, undefined, {
                        // options for Position.iterateParagraphChildNodes()
                        allNodes: true,
                        start: _(start).last(),
                        end: _(end).last(),
                        split: true
                    });

                    // updating the drawing node, if the change happened inside a text frame or shape (51964)
                    if (!forceUpdateFormatting && DrawingFrame.getDrawingNode(startInfo.node.parentNode)) { forceUpdateFormatting = true; }

                    // handling spell checking after modifying character attributes
                    if (forceUpdateFormatting) { self.implParagraphChanged(startInfo.node.parentNode); }

                    // try to merge last text span in the range with its next sibling
                    if (lastTextSpan) {
                        Utils.mergeSiblingTextSpans(lastTextSpan, true);
                    }

                    // BUG #48392 - Part 1
                    // check, whether we have to update the styles of the first (empty) span, or not
                    if ($(startInfo.node).prev().is('.drawing')) {
                        var firstSpan = $(startInfo.node).prevAll('span').first();
                        if (DOM.isEmptySpan(firstSpan) && DOM.isFirstTextSpanInParagraph(firstSpan)) {
                            if (attributes && attributes.character) {
                                // set new character-attrs to the leading empty span
                                self.getCharacterStyles().setElementAttributes(firstSpan, { character: attributes.character });
                            }
                        }
                    }

                // otherwise: only single components allowed at this time
                } else {

                    // check that start and end point to the same element
                    if (startInfo.node !== endInfo.node) {
                        Utils.warn('Editor.implSetAttributes(): no ranges supported for attribute family "' + startInfo.family + '"');
                        return;
                    }

                    // format the (single) element
                    styleFamily = startInfo.family;
                    styleSheets = self.getStyleCollection(styleFamily);

                    // Performance: Saving old list style id, before it is removed
                    if (self.isImportFinished() && (styleFamily === 'paragraph')) {
                        paraAttrs = AttributeUtils.getExplicitAttributes(startInfo.node);
                        if (paraAttrs) { oldListStyleId = getListStyleIdFromParaAttrs(paraAttrs); }
                    }

                    // assigning character attributes to text spans, not to paragraphs (41250)
                    if (!app.isODF() && styleFamily === 'paragraph' && 'character' in attributes) {

                        savedStyleSheets = styleSheets; // saving stylesSheet objects
                        styleSheets = self.getCharacterStyles(); // overwriting current styleSheet object to use valid styleSheet (53118)

                        _.each($(startInfo.node).children('span'), function (node) {
                            // setting the character attributes (only those!) to the text span(s) inside the paragraph
                            if (DOM.isEmptySpan(node) || DOM.isTextFrameTemplateTextSpan(node)) {
                                setElementAttributes(node, { character: _.copy(attributes.character, true) });
                                usedSpecialCharacterHandling = true;
                            }
                        });

                        styleSheets = savedStyleSheets; // restoring styleSheet objects

                        if (usedSpecialCharacterHandling) {

                            self.implParagraphChanged(startInfo.node);

                            // removing the character attributes, so that they are not assigned to the paragraph
                            modifiedAttributes = _.copy(attributes, true);  // not modifying the original object
                            delete modifiedAttributes.character;

                            if (!_.isEmpty(modifiedAttributes)) { setElementAttributes(startInfo.node, modifiedAttributes); }
                        }

                    }

                    if (!usedSpecialCharacterHandling) {

                        setElementAttributes(startInfo.node);

                        // update also the neighbors, if the list level, border, color or margin was modified
                        handleNeighborUpdate(startInfo.node, attributes);

                        // handling table attributes for drawing tables (in presentation app)
                        if (self.useSlideMode() && styleFamily === 'drawing' && DrawingFrame.isTableDrawingFrame(startInfo.node)) {
                            // TODO: Avoid duplicate table attributation (setElementAttributes for the drawing already formatted the table)
                            // -> in the following function the attributes shall only be set at the drawing node
                            setTableAttributesInTableDrawing(startInfo.node, attributes);
                        }

                        // updating the selection of a selected drawing node (and drawing table node)
                        if (self.useSlideMode()) {
                            if (styleFamily === 'drawing') {
                                DrawingFrame.updateSelectionDrawingSize(startInfo.node);
                            } else if (styleFamily === 'row') {
                                DrawingFrame.updateSelectionDrawingSize($(startInfo.node).closest(DrawingFrame.NODE_SELECTOR));
                            }
                        }

                        if (styleFamily === 'drawing') {

                            self.trigger('change:drawing');

                            // BUG #48392 - Part 2
                            // wait for "importSuccess" to be able to reach the sibbling-elements
                            self.waitForImportSuccess(function () {
                                var prevElement = $(startInfo.node).prev();
                                // If the previous element is a empty span (to be able to set the cursor before the drawing)
                                if (DOM.isEmptySpan(prevElement) && DOM.isFirstTextSpanInParagraph(prevElement)) {
                                    var nextElement = $(startInfo.node).nextAll('span').first();
                                    // and the next span-element ...
                                    if (nextElement) {
                                        var nextAttrs = AttributeUtils.getExplicitAttributes(nextElement);
                                        // ... has some explicit attributes
                                        if (nextAttrs && nextAttrs.character) {
                                            // set character-attrs from the following span to the leading empty span
                                            self.getCharacterStyles().setElementAttributes(prevElement, { character: nextAttrs.character });
                                        }
                                    }
                                }
                            });
                        }
                    }

                    if (styleFamily === 'paragraph') {
                        // updating the paragraph node, if the change happened inside (54768)
                        if (!forceUpdateFormatting && DrawingFrame.getDrawingNode(startInfo.node)) { forceUpdateFormatting = true; }
                        if (forceUpdateFormatting) { self.implParagraphChanged(startInfo.node); }
                    }
                }

                // create the undo action
                if (undoManager.isUndoEnabled() && !noUndo) {
                    redoOperation = { name: Operations.SET_ATTRIBUTES, start: start, end: end, attrs: attributes };
                    // extend redoOperation if target is present
                    self.extendPropertiesWithTarget(redoOperation, target);
                    undoManager.addUndo(undoOperations, redoOperation);
                }

                // update numberings and bullets (but updateListsDebounced can only be called after successful document import)
                if (self.isImportFinished() && (styleFamily === 'paragraph')) {
                    if (('styleId' in attributes) ||
                        (
                            _.isObject(attributes.paragraph) &&
                            (('listLevel' in attributes.paragraph) || ('listStyleId' in attributes.paragraph))
                        )
                    ) {

                        // determining listStyleId und listLevel
                        if ((_.isObject(attributes.paragraph)) && ('listStyleId' in attributes.paragraph) && (attributes.paragraph.listStyleId !== null)) {
                            listStyleId = attributes.paragraph.listStyleId;  // list style assigned to paragraph
                            listLevel = ('listLevel' in attributes.paragraph) ? attributes.paragraph.listLevel : 0;
                        } else if (attributes.styleId && self.isParagraphStyleWithListStyle(attributes.styleId)) {
                            listStyleId = getListStyleInfoFromStyleId(attributes.styleId, 'listStyleId');
                            listLevel = getListStyleInfoFromStyleId(attributes.styleId, 'listLevel') || 0;
                        } else {
                            // list level modified -> checking paragraph attributes for list styles to receive list style id
                            paraAttrs = AttributeUtils.getExplicitAttributes(startInfo.node);
                            // updating lists, if required
                            if (self.isListStyleParagraph(null, paraAttrs)) {
                                if (paraAttrs && paraAttrs.paragraph) {
                                    listStyleId = paraAttrs.paragraph.listStyleId;
                                    listLevel = paraAttrs.paragraph.listLevel;
                                }
                            }
                        }

                        // handling the case, that the list style id is removed
                        if (!listStyleId && oldListStyleId) { listStyleId = oldListStyleId; }

                        // defining list style Id or listLevel, that need to be updated
                        if (listStyleId || listLevel) {
                            if (!listStyleId) {
                                listStyleId = self.getParagraphStyles().getElementAttributes(startInfo.node).paragraph.listStyleId;
                                //fix for Bug 37594 styleId is not at the para-attrs and not in the style it self. but by merging we get the correct stlyeId
                            }
                            // mark this paragraph for later list update. This is necessary, because it does not yet contain
                            // the list label node (DOM.LIST_LABEL_NODE_SELECTOR) and is therefore ignored in updateLists.
                            $(startInfo.node).data('updateList', 'true');
                            // registering this list style for update
                            // special handling for removal of listLevel, for example after 'Backspace'
                            if (listStyleId && (listLevel === null || listLevel === -1)) {
                                $(startInfo.node).data('removeLabel', 'true');
                            }

                            // not registering listStyle for updating, if this is a bullet list -> updating only specific paragraphs
                            if (self.getListCollection() && self.getListCollection().isAllLevelsBulletsList(listStyleId)) { listStyleId = null; }

                            self.updateListsDebounced({ useSelectedListStyleIDs: true, paraInsert: true, listStyleId: listStyleId, listLevel: listLevel });
                        }
                    } else if (self.useSlideMode()) {

                        if (_.isObject(attributes.paragraph) && ('level' in attributes.paragraph || 'bullet' in attributes.paragraph)) {
                            self.updateListsDebounced($(startInfo.node));
                        }

                    } else if ('character' in attributes) {  // modified character attributes may lead to modified list labels

                        // checking paragraph attributes for list styles to receive list style id
                        paraAttrs = AttributeUtils.getExplicitAttributes(startInfo.node);
                        // updating lists, after character attributes were modified
                        if (self.isListStyleParagraph(null, paraAttrs)) {
                            if (paraAttrs && paraAttrs.paragraph) {
                                listStyleId = paraAttrs.paragraph.listStyleId;
                                listLevel = paraAttrs.paragraph.listLevel;
                                self.updateListsDebounced({ useSelectedListStyleIDs: true, paraInsert: true, listStyleId: listStyleId, listLevel: listLevel });
                            }
                        }
                    }
                }

                if (app.isODF() && (styleFamily === 'paragraph') && (attributes.character) && (Position.getParagraphNodeLength(startInfo.node) === 0)) {
                    // Task 28187: Setting character attributes at empty paragraphs
                    // -> removing character attributes from text span so that character styles at paragraph become visible (also 30927)
                    nullAttributes = _.copy(attributes, true);
                    for (localkey in nullAttributes.character) { nullAttributes.character[localkey] = null; }
                    self.getCharacterStyles().setElementAttributes(DOM.findFirstPortionSpan(startInfo.node), nullAttributes);
                }

                if (self.isImportFinished()) {
                    // adjust tabulators, if character or drawing attributes have been changed
                    // (changing paragraph or table attributes updates tabulators automatically)
                    if ((styleFamily === 'character') || (styleFamily === 'drawing')) {

                        parentNode = startInfo.node.parentNode;

                        if (!self.useSlideMode() || !DOM.isSlideNode(parentNode)) { // only update paragraphs, not slides
                            self.getParagraphStyles().updateTabStops(parentNode);
                            if (self.useSlideMode() || _.last(start) === 0) { // in Text app only if first character is modified (58026)
                                self.updateListsDebounced($(parentNode));
                            }
                            // Info: If only the current paragraph is updated, the item number will be wrong. TODO: Performance
                            // self.updateListsDebounced($(startInfo.node.parentNode), { singleParagraph: true }); // Updating list only the current paragraph
                        }
                    }

                    // if target comes with operation, take care of targeted header/footer, otherwise, run normal page break rendering
                    if (target) {
                        // trigger header/footer content update on other elements of same type, if change was made inside header/footer
                        if (DOM.isMarginalNode(startInfo.node)) {
                            self.updateEditingHeaderFooterDebounced(DOM.getMarginalTargetNode(self.getNode(), startInfo.node));
                        } else if (DOM.isMarginalNode(startInfo.node.parentNode)) {
                            self.updateEditingHeaderFooterDebounced(DOM.getMarginalTargetNode(self.getNode(), startInfo.node.parentNode));
                        }
                    }
                    self.setBlockOnInsertPageBreaks(!!target);

                    // after document is loaded, trigger pagebreaks reposition on certain events like font size change, height of drawing or row changed
                    isCharOrDrawingOrRow = 'character' in attributes || 'drawing' in attributes || 'row' in attributes || ('line' in attributes && 'width' in attributes.line);
                    isClearFormatting = 'styleId' in attributes && attributes.styleId === null;

                    if (!self.useSlideMode()) { // handling of page breaks, not required in presentation app

                        if (_.isObject(attributes.paragraph)) {
                            paraAttrs = attributes.paragraph;
                            // check level in Presentation & listLevel in Text
                            isAttributesInParagraph = ('lineHeight' in paraAttrs) || ('listLevel' in paraAttrs) || ('level' in paraAttrs) || ('listStyleId' in paraAttrs);
                            isPageBreakBeforeAttributeInParagraph = ('pageBreakBefore' in paraAttrs);
                            isParSpacing = 'marginBottom' in paraAttrs;
                            isBorderTopBot = 'borderTop' in paraAttrs || 'borderBottom' in paraAttrs;
                            isParaIndent = 'indentLeft' in paraAttrs || 'indentRight' in paraAttrs || 'indentFirstLine' in paraAttrs;
                        }

                        if (isCharOrDrawingOrRow || isClearFormatting || isAttributesInParagraph || isPageBreakBeforeAttributeInParagraph || isParSpacing || isBorderTopBot || isParaIndent) {
                            if (start.length > 1) {
                                // if its paragraph creation inside of table
                                currentElement = Position.getContentNodeElement(rootNode, start.slice(0, 1));
                            } else {
                                currentElement = Position.getContentNodeElement(rootNode, start);
                            }
                            if ($(currentElement).data('lineBreaksData')) {
                                $(currentElement).removeData('lineBreaksData'); //we changed paragraph layout, cached line breaks data needs to be invalidated
                            }
                            // attribute removed by user, needs to refresh page layout
                            if (isPageBreakBeforeAttributeInParagraph && !attributes.paragraph.pageBreakBefore) {
                                $(startInfo.node).removeClass('manual-page-break'); // remove class from div.p element
                                $(currentElement).removeClass('manual-page-break'); // remove class from parent table element also
                            }
                            self.insertPageBreaks(currentElement, DrawingFrame.getClosestTextFrameDrawingNode(startInfo.node));
                            if (!self.isPageBreakMode()) { app.getView().recalculateDocumentMargin(); }
                        }

                        // adding handling for ruler update after resize drawing or modifying cell width
                        if ((!isParaIndent) && ((_.isObject(attributes.drawing) && _.isNumber(attributes.drawing.width)) || (_.isObject(attributes.table) && _.isNumber(attributes.table.width)))) {
                            isParaIndent = true;
                        }
                    }
                }

                if (isParaIndent) { self.trigger('ruler:initialize'); }

                // Special handling for presentation in master/layout slides -> updating styles
                if (target && _.isFunction(self.isLayoutOrMasterId) && self.isLayoutOrMasterId(target) && forceLayoutFormattingData) { self.forceMasterLayoutUpdate(forceLayoutFormattingData); }

                if (attributes.character && attributes.character.field && attributes.character.field.formFieldType === 'checkBox') {
                    $(startInfo.node).text(attributes.character.field.checked ? '☒' : '☐');
                }

                return true;
            };
        }());

        // public methods -----------------------------------------------------

        /**
         * Changes multiple attributes of the specified attribute family in the
         * current selection.
         *
         * @param {String} family
         *  The name of the attribute family containing the passed attributes.
         *
         * @param {Object} attributes
         *  A map of attribute value maps (name/value pairs), keyed by
         *  attribute families.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.clear=false]
         *      If set to true, all existing explicit attributes will be
         *      removed from the selection while applying the new attributes.
         *  @param {Boolean} [options.drawingFilterRequired=false]
         *      If set to true, it might be necessary to filter an existing
         *      multi drawing selection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after all operations have been
         *  generated and applied successfully.
         */
        this.setAttributes = function (family, attributes, options) {

            // Create an undo group that collects all undo operations generated
            // in the local setAttributes() method (it calls itself recursively
            // with smaller parts of the current selection).
            return this.getUndoManager().enterUndoGroup(function () {

                var // table or drawing element contained by the selection
                    element = null,
                    // operations generator
                    generator = self.createOperationGenerator(),
                    // the style sheet container
                    styleSheets = self.getStyleCollection(family),
                    // logical position
                    localPosition = null,
                    // another logical position
                    localDestPosition = null,
                    // the length of the paragraph
                    paragraphLength = null,
                    // whether after assigning cell attributes, it is still necessary to assign table attributes
                    createTableOperation = true,
                    // paragraph helper position and helper node
                    paraPos = null, paraNode = null,
                    // the old attributes, required for change tracking
                    oldAttrs = null,
                    // whether the modification can be change tracked
                    isTrackableChange = true,
                    // currently active root node
                    activeRootNode = self.getCurrentRootNode(),
                    // if we apply new style, do not remove manual pagebreaks
                    pageBreakBeforeAttr = false,
                    pageBreakAfterAttr = false,
                    // the old attributes, required for pageBreak paragraph style
                    oldParAttr = null,
                    // the promise for the asychronous execution of operations
                    operationsPromise = null,
                    // the promise for generating the operations
                    operationGeneratorPromise = null,
                    // the complete promise for generation and applying of operations
                    setAttributesPromise = null,
                    // the ratio of operation generation to applying of operation
                    operationRatio = 0.3,
                    // an optional array with selection ranges for large selections
                    splittedSelectionRange = null,
                    // a snapshot object
                    snapshot = null,
                    // the selected drawings
                    drawings = null,
                    // the selection object
                    selection = self.getSelection(),
                    // whether this is a selection in a master/layout slide
                    isMasterLayoutSelection = false,
                    // whether the implicit paragraphs are handled like in ODF slide mode
                    isODFSlideMode = false;

                /**
                 * Helper function to generate operations. This is done synchronously. But for asynchronous usage,
                 * the selection range can be restricted. This is especially useful for large selections.
                 *
                 * @param {Number[][]} [selectionRange]
                 *  An array with two logical positions for the start and the end position of a selection range.
                 */
                function generateAllSetAttributeOperations(selectionRange) {

                    var // the logical start position, if specified
                        localStartPos = (selectionRange && selectionRange[0]) || null,
                        // the logical end position, if specified
                        localEndPos = (selectionRange && selectionRange[1]) || null,
                        // the change track object
                        changeTrack = self.getChangeTrack(),
                        // array containing start end end position of word in which cursor is placed
                        wordBoundaryPos = null,
                        // start postion of wrapped word
                        wordBoundaryStart = null,
                        // end position of wrapped word
                        wordBoundaryEnd = null,
                        // start position of changetracked span
                        changeTrackStartPos = null,
                        // end position of changetracked span
                        changeTrackEndPos = null,
                        // the selected drawing nodes
                        selectedDrawings = null,
                        // the affected drawing nodes of a multi drawing selection
                        affectedDrawings = null;

                    // generates a 'setAttributes' operation with the correct attributes
                    function generateSetAttributeOperation(startPosition, endPosition, oldAttrs) {

                        var // the options for the operation
                            operationOptions = null;

                        if (isMasterLayoutSelection && family === 'character') {
                            startPosition = _.initial(startPosition); // assigning character attributes to paragraph
                            endPosition = null;
                        }

                        operationOptions = { start: startPosition, attrs: _.clone(attributes) };

                        // adding the old attributes, author and date for change tracking
                        if (oldAttrs) {
                            oldAttrs = _.extend(oldAttrs, changeTrack.getChangeTrackInfo());
                            operationOptions.attrs.changes = { modified: oldAttrs };
                        }

                        // add end position if specified
                        if (_.isArray(endPosition)) {
                            operationOptions.end = endPosition;
                        }
                        // paragraph's style manual page break before
                        if (pageBreakBeforeAttr) {
                            operationOptions.attrs.paragraph = operationOptions.attrs.paragraph || {};
                            operationOptions.attrs.paragraph.pageBreakBefore = true;
                            pageBreakBeforeAttr = false;
                        }
                        // paragraph's style manual page break after
                        if (pageBreakAfterAttr) {
                            operationOptions.attrs.paragraph = operationOptions.attrs.paragraph || {};
                            operationOptions.attrs.paragraph.pageBreakAfter = true;
                            pageBreakAfterAttr = false;
                        }

                        // generate the 'setAttributes' operation
                        generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);
                    }

                    // setting a specified selection, if defined
                    // if (selectionRange) { selection.setTextSelection.apply(selection, selectionRange); }

                    // generate 'setAttribute' operations
                    switch (family) {

                        case 'character':
                            if (selection.hasRange()) {
                                selection.iterateContentNodes(function (paragraph, position, startOffset, endOffset) {

                                    if (isODFSlideMode && DOM.isImplicitParagraphNode(paragraph)) { return; } // no operation

                                    // validate start offset (iterator passes 'undefined' for fully covered paragraphs)
                                    if (!_.isNumber(startOffset)) {
                                        startOffset = 0;
                                    }
                                    // validate end offset (iterator passes 'undefined' for fully covered paragraphs)
                                    if (!_.isNumber(endOffset)) {
                                        endOffset = Position.getParagraphNodeLength(paragraph) - 1;
                                    }

                                    if (changeTrack.isActiveChangeTracking()) {

                                        // iterating over all children to send the old attributes in the operation
                                        // remove all empty text spans which have sibling text spans, and collect
                                        // sequences of sibling text spans (needed for white-space handling)
                                        Position.iterateParagraphChildNodes(paragraph, function (node, nodestart, nodelength, offsetstart, offsetlength) {

                                            // evaluate only text spans that are partly or completely covered by selection
                                            if (DOM.isTextSpan(node)) {
                                                // one operation for each span
                                                changeTrackStartPos = nodestart + offsetstart;
                                                changeTrackEndPos = nodestart + offsetstart + offsetlength - 1;
                                                // decrease end position as long as it is whitespace character at that position
                                                // but only if it is last span in paragraph
                                                if (endOffset === changeTrackEndPos) {
                                                    if (!Position.isWhitespaceCharInPosition(activeRootNode, position.concat([changeTrackEndPos + 1]))) {
                                                        while (changeTrackEndPos >= changeTrackStartPos && Position.isWhitespaceCharInPosition(activeRootNode, position.concat([changeTrackEndPos]))) {
                                                            changeTrackEndPos -= 1;
                                                        }
                                                    }
                                                }
                                                if (changeTrackEndPos >= changeTrackStartPos) {
                                                    generateSetAttributeOperation(position.concat([changeTrackStartPos]), position.concat([changeTrackEndPos]), changeTrack.getOldNodeAttributes(node));
                                                }
                                            }

                                        }, undefined, { start: startOffset, end: endOffset });

                                    } else {

                                        // set the attributes at the covered text range
                                        if (startOffset === 0 && endOffset <= 0 && DOM.isImplicitParagraphNode(paragraph)) {
                                            generator.generateOperation(Operations.PARA_INSERT, { start: position });  // generating paragraph
                                            generateSetAttributeOperation(position);  // assigning attribute to new paragraph
                                        } else if (startOffset === 0 && endOffset <= 0 && Position.getParagraphNodeLength(paragraph) === 0) {
                                            generateSetAttributeOperation(position);  // assigning attribute to existing empty paragraph
                                        } else if (startOffset <= endOffset) {
                                            // decrease end position as long as it is whitespace character at that position
                                            // we dont want ending whitespace to be attributed (underlined for example).
                                            // Exception is when end selection range is at the end of the paragraph (see #48059)
                                            if (!Position.isWhitespaceCharInPosition(activeRootNode, position.concat([endOffset + 1])) && (endOffset !== Position.getParagraphNodeLength(paragraph) - 1)) {
                                                while (endOffset > startOffset && Position.isWhitespaceCharInPosition(activeRootNode, position.concat([endOffset]))) {
                                                    endOffset -= 1;
                                                }
                                            }
                                            generateSetAttributeOperation(position.concat([startOffset]), position.concat([endOffset]));
                                        }

                                    }
                                    self.getSpellChecker().reset(attributes, paragraph);
                                }, null, { startPos: localStartPos, endPos: localEndPos });
                            } else {  // selection has no range
                                // expand set attributes operation to whole word if cursor is inside word
                                if (Position.shouldSetAttsExtendToWord(activeRootNode, selection.getStartPosition())) {
                                    wordBoundaryPos = Position.getWordBoundaries(activeRootNode, selection.getStartPosition());
                                    paraPos = _.clone(selection.getStartPosition());
                                    paraPos.pop();
                                    paraNode = Position.getParagraphElement(activeRootNode, paraPos);

                                    if (changeTrack.isActiveChangeTracking()) {
                                        // Creating one setAttribute operation for each span inside the paragraph, because of old attributes!
                                        oldAttrs = {};
                                        if (DOM.isParagraphNode(paraNode)) { oldAttrs = changeTrack.getOldNodeAttributes(paraNode); }
                                    }
                                    // set the attributes at the extended text range
                                    if (_.isArray(wordBoundaryPos) && wordBoundaryPos.length === 2) {
                                        wordBoundaryStart = _.isArray(wordBoundaryPos[0]) ? wordBoundaryPos[0] : null;
                                        wordBoundaryEnd = _.isArray(wordBoundaryPos[1]) ? Position.decreaseLastIndex(wordBoundaryPos[1]) : null;
                                        if (wordBoundaryStart && wordBoundaryEnd) {
                                            generateSetAttributeOperation(wordBoundaryStart, wordBoundaryEnd, oldAttrs);
                                            self.getSpellChecker().reset(attributes, paraNode);
                                        }
                                    }
                                } else if (Position.getParagraphLength(activeRootNode, _.clone(selection.getStartPosition())) === 0) {
                                    // Task 28187: Setting character attributes at empty paragraphs
                                    // In implSetAttributes it is additionally necessary to remove the character attributes
                                    // from text span so that character styles at paragraph become visible

                                    paraPos = _.clone(selection.getStartPosition());
                                    paraPos.pop();
                                    paraNode = Position.getParagraphElement(activeRootNode, paraPos);

                                    // Defining the oldAttrs, if change tracking is active
                                    if (changeTrack.isActiveChangeTracking()) {
                                        // Creating one setAttribute operation for each span inside the paragraph, because of old attributes!
                                        oldAttrs = {};
                                        if (DOM.isParagraphNode(paraNode)) { oldAttrs = changeTrack.getOldNodeAttributes(paraNode); }
                                    }

                                    if (DOM.isImplicitParagraphNode(paraNode)) { generator.generateOperation(Operations.PARA_INSERT, { start: paraPos }); }
                                    // generating setAttributes operation
                                    generateSetAttributeOperation(_.initial(_.clone(selection.getStartPosition())), undefined, oldAttrs);
                                } else {
                                    // using preselected attributes for non-empty paragraphs
                                    self.addPreselectedAttributes(attributes);
                                }
                            }
                            break;

                        case 'paragraph':
                            // deleted all changes for bug #26454#, because it did the opposite of word behavior, and that is not what the user expects! @see Bug 37174

                            selection.iterateContentNodes(function (paragraph, position) {

                                if (isODFSlideMode && DOM.isImplicitParagraphNode(paragraph)) { return; } // no operation

                                // Defining the oldAttrs, if change tracking is active
                                if (changeTrack.isActiveChangeTracking()) {
                                    // Expanding operation for change tracking with old explicit attributes
                                    oldAttrs = changeTrack.getOldNodeAttributes(paragraph);
                                }
                                // Preserve pageBreakBefore and/or pageBreakAfter attribute after setting new paragraph style
                                if (DOM.isManualPageBreakNode(paragraph)) {
                                    oldParAttr = AttributeUtils.getExplicitAttributes(paragraph);
                                    if (_.isObject(oldParAttr) && oldParAttr.paragraph) {
                                        if (oldParAttr.paragraph.pageBreakBefore === true) {
                                            pageBreakBeforeAttr = true;
                                        }
                                        if (oldParAttr.paragraph.pageBreakAfter === true) {
                                            pageBreakAfterAttr = true;
                                        }
                                    }

                                }

                                // generating a new paragraph, if it is implicit
                                if (DOM.isImplicitParagraphNode(paragraph)) { generator.generateOperation(Operations.PARA_INSERT, { start: position }); }
                                // generating setAttributes operation
                                generateSetAttributeOperation(position, undefined, oldAttrs);
                            }, null, { startPos: localStartPos, endPos: localEndPos });
                            break;

                        case 'cell':
                            selection.iterateTableCells(function (cell, position) {
                                // Defining the oldAttrs, if change tracking is active
                                if (changeTrack.isActiveChangeTracking()) {
                                    // Expanding operation for change tracking with old explicit attributes
                                    oldAttrs = changeTrack.getOldNodeAttributes(cell);
                                }
                                generateSetAttributeOperation(position, undefined, oldAttrs);
                            });
                            break;

                        case 'table':

                            element = selection.getEnclosingTable();

                            if (!element && self.useSlideMode() && selection.isDrawingSelection() && DrawingFrame.isTableDrawingFrame(selection.getSelectedDrawing())) {
                                element = selection.getSelectedDrawing();
                            }

                            if (element) {

                                localPosition = Position.getOxoPosition(activeRootNode, element, 0);  // the logical position of the table

                                if (Utils.getBooleanOption(options, 'clear', false)) {
                                    // removing hard attributes at tables and cells, so that the table style will be visible
                                    Table.removeTableAttributes(self, activeRootNode, element, localPosition, generator);
                                }

                                if (Utils.getBooleanOption(options, 'onlyVisibleBorders', false)) {
                                    // setting border width directly at cells -> overwriting values of table style
                                    Table.setBorderWidthToVisibleCells(element, attributes, self.getTableCellStyles(), activeRootNode, generator, self);
                                    createTableOperation = false;
                                }

                                if (Utils.getBooleanOption(options, 'cellSpecificTableAttribute', false)) {
                                    // setting border mode directly at cells -> overwriting values of table style
                                    Table.setBorderModeToCells(element, attributes, TableStyles.getBorderStyleFromAttributes(self.getAttributes('cell').cell || {}), activeRootNode, generator, self);
                                }

                                // setting attributes to the table element
                                // -> also setting table attributes, if attributes are already assigned to table cells, to
                                // keep the getter functions simple and performant
                                if (createTableOperation) {
                                    // Defining the oldAttrs, if change tracking is active
                                    if (changeTrack.isActiveChangeTracking()) {
                                        // Expanding operation for change tracking with old explicit attributes
                                        oldAttrs = changeTrack.getOldNodeAttributes(element);
                                    }
                                    generateSetAttributeOperation(localPosition, undefined, oldAttrs);
                                }
                            }
                            break;

                        case 'drawing':

                            selectedDrawings = selection.getSelectedDrawings();

                            if (!selectedDrawings || selectedDrawings.length === 0) { break; }

                            // a filter might be needed to generate only valid operations in a multi drawing selection
                            // -> example: No background for drawings of type 'connector'
                            affectedDrawings = (selection.isMultiSelection() && Utils.getBooleanOption(options, 'drawingFilterRequired', false)) ? filterDrawingNodes(selectedDrawings, attributes) : selectedDrawings;

                            _.each(affectedDrawings, function (element) {

                                // TODO: this fails if a drawing style sheet changes the inline/floating mode instead of explicit attributes
                                if (DrawingFrame.isDrawingFrame(element)) {

                                    localPosition = Position.getOxoPosition(activeRootNode, element, 0);

                                    if (_.isObject(attributes.drawing)) {

                                        // when switching from inline to floated, saving current position in the drawing, so that it can
                                        // be set correctly when switching back to inline.
                                        // This is only necessary, if the drawing was moved in that way, that implMove needed to be called.
                                        if ((attributes.drawing.inline === false) && (DOM.isInlineDrawingNode(element))) {
                                            $(element).data('inlinePosition', localPosition[localPosition.length - 1]);
                                            // Fixing vertical offset, if drawing is set from inline to floated (31722)
                                            if ((attributes.drawing.anchorVertBase) && (attributes.drawing.anchorVertBase === 'paragraph') &&
                                                (_.isNumber(attributes.drawing.anchorVertOffset)) && (attributes.drawing.anchorVertOffset === 0) &&
                                                (($(element).offset().top - $(element).closest(DOM.PARAGRAPH_NODE_SELECTOR).offset().top > 0))) {
                                                attributes.drawing.anchorVertOffset = Utils.convertLengthToHmm($(element).offset().top - $(element).closest(DOM.PARAGRAPH_NODE_SELECTOR).offset().top, 'px');
                                            }
                                        }

                                        // when switching from floated to inline, a move of the drawing might be necessary
                                        if ((attributes.drawing.inline === true) && (DOM.isFloatingDrawingNode(element)) && ($(element).data('inlinePosition'))) {

                                            localDestPosition = _.clone(localPosition);
                                            paragraphLength = Position.getParagraphLength(activeRootNode, localDestPosition);
                                            if ((paragraphLength - 1) < $(element).data('inlinePosition')) {  // -> is this position still valid?
                                                localDestPosition[localDestPosition.length - 1] = paragraphLength - 1;
                                            } else {
                                                localDestPosition[localDestPosition.length - 1] = $(element).data('inlinePosition');
                                            }
                                            if (!_.isEqual(localPosition, localDestPosition)) {

                                                var // the logical position of the paragraph
                                                    paraPos = _.clone(localDestPosition),
                                                    // the paragraph node
                                                    paraNode = null;

                                                paraPos.pop();
                                                paraNode = Position.getParagraphElement(activeRootNode, paraPos);

                                                if (DOM.isImplicitParagraphNode(paraNode)) { generator.generateOperation(Operations.PARA_INSERT, { start: paraPos }); }
                                                generator.generateOperation(Operations.MOVE, { start: localPosition, end: localPosition, to: localDestPosition });

                                                localPosition = _.clone(localDestPosition);
                                            }
                                        }

                                        // setting new position (inline, paragraph, ...) is not change tracked
                                        if (!changeTrack.ctSupportsDrawingPosition()) { isTrackableChange = false; }
                                    }

                                    // setting border and fillcolor of drawing is not change tracked
                                    if (_.isObject(attributes.fill) || _.isObject(attributes.line)) { isTrackableChange = false; }

                                    // Defining the oldAttrs, if change tracking is active
                                    // -> and if change track supports changing of drawing position
                                    if (changeTrack.isActiveChangeTracking() && isTrackableChange) {
                                        // Expanding operation for change tracking with old explicit attributes
                                        oldAttrs = changeTrack.getOldNodeAttributes(element);
                                    }

                                    // by setting fill- or line-attributes on a group
                                    if (DrawingFrame.isGroupDrawingFrame(element) && (_.isObject(attributes.fill) || _.isObject(attributes.line))) {
                                        // generate operations for all children -> but using filter to avoid setting attributes to drawings, that do not support them
                                        _.each(filterDrawingNodes(DrawingFrame.getAllGroupDrawingChildren(element), attributes), function (ele) {
                                            generateSetAttributeOperation(Position.getOxoPosition(activeRootNode, ele, 0));
                                        });
                                    // otherwise
                                    } else {
                                        // set attributes on the group
                                        generateSetAttributeOperation(localPosition, undefined, oldAttrs);
                                    }
                                    if (selection.isCropMode()) { self.exitCropMode(); }
                                    $(element).closest(DOM.PARAGRAPH_NODE_SELECTOR).removeData('lineBreaksData'); //we changed paragraph layout, cached line breaks data needs to be invalidated

                                }
                            });

                            break;

                        default:
                            Utils.error('Editor.setAttributes(): missing implementation for family "' + family + '"');
                    }
                }

                /**
                 * Helper function to apply the generated operations asynchronously.
                 *
                 * @returns {jQuery.Promise}
                 *  A promise that will be resolved when the operations have been applied.
                 */
                function doSetAttributesAsync() {

                    operationsPromise = self.applyTextOperationsAsync(generator, null, { showProgress: false, leaveOnSuccess: true, progressStart: operationRatio });

                    // restore the original selection after successful apply of operations
                    operationsPromise
                        .done(function () {
                            self.getSelection().restoreBrowserSelection();
                        });

                    return operationsPromise;
                }

                /**
                 * In ODP it is not allowed to convert an implicit paragraph into a real paragraph because of attributes.
                 * In this case an empty paragraph would be created, that make the surrounding drawing invisible after download.
                 * Instead all attributes must be collected in preselected attributes or at the drawings itself.
                 *
                 * @returns {Booelan}
                 *  Whether the required attribute handling was completely done inside this function. This is the case
                 *  for cursor setting inside implicit paragraphs (no operations required) or for selected drawing(s).
                 */
                function handleODFSlideMode() {

                    var // whether the drawing was already handled
                        allDrawingsHandled = false,
                        // the number of handled drawings
                        drawingCounter = 0;

                    isODFSlideMode = true; // implicit paragraphs are handled without any operation (ODP)

                    if (family === 'paragraph' || family === 'character') {

                        drawings = selection.getSelectedDrawings();

                        if (drawings && drawings.length > 0) {

                            // iterating over all selected drawings -> assigning attributes to drawings, not to paragraphs or spans
                            // -> avoiding inserting additional paragraphs
                            _.each(drawings, function (oneDrawing) {
                                if (DOM.isDrawingWithOnlyImplicitParagraph(oneDrawing)) {
                                    // setting the attributes as explicit attributes to implicit paragraphs
                                    self.getParagraphStyles().setElementAttributes($(oneDrawing).find('div.p'), attributes);
                                    drawingCounter++;
                                }
                            });

                            allDrawingsHandled = (drawings.length === drawingCounter);
                        }
                    }

                    return allDrawingsHandled;
                }

                // add all attributes to be cleared
                if (Utils.getBooleanOption(options, 'clear', false)) {
                    if (family !== 'table') {  // special behaviour for tables follows below
                        attributes = styleSheets.extendAttributeSet(styleSheets.buildNullAttributeSet(), attributes);
                    }
                }

                // nothig to do if no attributes will be changed
                if (_.isEmpty(attributes)) { return; }

                // avoid to insert empty paragraphs in empty drawings in ODP
                if (app.isODF() && self.useSlideMode() && handleODFSlideMode()) { return; }  // no further operations, if handled inside 'handleODFSlideMode'

                // register pending style sheet via 'insertStyleSheet' operation
                if (_.isString(attributes.styleId) && styleSheets.isDirty(attributes.styleId)) {
                    self.generateInsertStyleOp(generator, family, attributes.styleId);
                }

                // Special handling for presentation in master/layout slides -> selecting only complete paragraphs
                if (self.getActiveTarget() && _.isFunction(self.isLayoutOrMasterId) && self.isLayoutOrMasterId(self.getActiveTarget())) {
                    isMasterLayoutSelection = true;
                    self.setMasterLayoutSelection();
                }

                // checking selection size -> make array with splitted selection [0,100], [101, 200], ... (39061)
                splittedSelectionRange = Position.splitLargeSelection(self.getCurrentRootNode(), selection.getStartPosition(), selection.getEndPosition(), self.getMaxTopLevelNodes());

                if (splittedSelectionRange && splittedSelectionRange.length > 1) {

                    // make sure that only one asynchronous call is processed at the same time
                    if (self.checkSetClipboardPasteInProgress()) { return $.when(); }

                    // blocking keyboard input during generation and applying of operations
                    self.setBlockKeyboardEvent(true);

                    // creating a snapshot
                    snapshot = new Snapshot(app);

                    // show a message with cancel button
                    // -> immediately grabbing the focus, after calling enterBusy. This guarantees, that the
                    // keyboard blocker works. Otherwise the keyboard events will not be catched by the page.
                    app.getView().enterBusy({
                        cancelHandler: function () {
                            if (operationsPromise && operationsPromise.abort) { // order is important, the latter has to win
                                // restoring the old document state
                                snapshot.apply();
                                // calling abort function for operation promise
                                app.enterBlockOperationsMode(function () { operationsPromise.abort(); });
                            } else if (operationGeneratorPromise && operationGeneratorPromise.abort) {
                                operationGeneratorPromise.abort();  // no undo of changes required
                            }
                        },
                        immediate: true,
                        warningLabel: gt('Sorry, formatting content will take some time.')
                    }).grabFocus();

                    // generate operations asynchronously
                    operationGeneratorPromise = self.iterateArraySliced(splittedSelectionRange, function (oneRange) {
                        // calling function to generate operations synchronously with reduced selection range
                        generateAllSetAttributeOperations(oneRange);
                    }, 'AttributeOperationsMixin.generateAllSetAttributeOperations');

                    // add progress handling
                    operationGeneratorPromise.progress(function (progress) {
                        // update the progress bar according to progress of the operations promise
                        app.getView().updateBusyProgress(operationRatio * progress);
                    });

                    setAttributesPromise = operationGeneratorPromise.then(doSetAttributesAsync).always(function () {
                        if (snapshot) { snapshot.destroy(); }
                        self.leaveAsyncBusy();
                    });

                } else {
                    // synchronous handling for small selections

                    // calling function to generate operations synchronously
                    generateAllSetAttributeOperations();

                    // apply all collected operations
                    self.applyOperations(generator);

                    // applying operations synchronously
                    setAttributesPromise = $.when();
                }

                // in ODF format, the filter needs to be informed about changes of table heights
                if (app.isODF() && self.useSlideMode() && selection.isAdditionalTextframeSelection() && DrawingFrame.isTableDrawingFrame(selection.getSelectedTextFrameDrawing())) {
                    self.trigger('drawingHeight:update', selection.getSelectedTextFrameDrawing());
                }

                return setAttributesPromise;

            }, this); // enterUndoGroup();
        };

        /**
         * Changes a single attribute of the specified attribute family in the
         * current selection.
         *
         * @param {String} family
         *  The name of the attribute family containing the specified
         *  attribute.
         *
         * @param {String} name
         *  the key of the attribute in the assigned family
         *
         * @param {Object} value
         *  the new attributes
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.clear=false]
         *      If set to true, all existing explicit attributes will be
         *      removed from the selection, before applying the new attributes.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after all operations have been
         *  generated and applied successfully.
         */
        this.setAttribute = function (family, name, value, options) {
            return this.setAttributes(family, Utils.makeSimpleObject(family, Utils.makeSimpleObject(name, value)), options);
        };

        /**
         * Returns whether the format-painter is currently active or not
         *
         * @return {Boolean}
         *  True or false, depends on format-painter-state
         */
        this.isFormatPainterActive = function () {
            return formatPainterActive;
        };

        /**
         * Sets the state of the format-painter and handles some implicit
         * reset-work, when the format-painter will be disabled
         *
         * @param {Boolean} active
         *  Activate or deactivate the format-painter.
         */
        this.setFormatPainterState = function (active) {
            // set formatpainter-state
            formatPainterActive = active;

            // handle some implicit specifications
            if (!active) {
                // reset formatpainter-attributes
                formatPainterAttrs = {};

                // update the toolbar-icon
                app.getController().update();
            }

            // setting cursor with class at content root node (54851)
            app.getView().getContentRootNode().toggleClass('active-formatpainter', active);
        };

        /**
         * Handles the format-painting.
         *
         * @param  {Boolean} activate
         *  Activate or deactivate the format-painter.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.reset=false]
         *      If set to true, the active format-painter should simply
         *      turn off, without paste any style.
         */
        this.handleFormatPainter = function (activate, options) {

            function getWhitelistedAttrs(options) {

                function getFullAttributes() {
                    function hasStyleId() {
                        return (formatPainterAttrs.explicit.paragraph && !_.isEmpty(formatPainterAttrs.explicit.paragraph.styleId));
                    }

                    var merged      = formatPainterAttrs.merged[source],
                        explicit    = formatPainterAttrs.explicit[source],
                        attrs       = (hasStyleId()) ? explicit : merged;

                    return (Utils.getBooleanOption(options, 'forceMerged', false)) ? merged : attrs;
                }

                function getFilteredAttributes(whitelist, attrs) {
                    if (_.isUndefined(attrs) || _.isNull(attrs)) { return; }

                    var returnArr = {};

                    if (whitelist) {
                        _.each(whitelist, function (key) {
                            if (!_.isUndefined(attrs[key])) { returnArr[key] = attrs[key]; }
                        });
                    }

                    return returnArr;
                }

                var source          = Utils.getStringOption(options, 'source'),
                    target          = Utils.getStringOption(options, 'target', source),
                    whitelist       = formatPainterWhitelist[target],
                    attrs           = {};

                // console.log('    source: ', source);
                // console.log('    target: ', target);

                if (_.isEmpty(whitelist)) { return {}; }

                if (target === 'default') {
                    whitelist = formatPainterWhitelist[source];
                }

                _.each(getFullAttributes(), function (object, group) {
                    // step over, when character-/ or fill-attributes should copied for twoPointShapes
                    if (twoPointShape && (group === 'character' || group === 'fill')) { return; }

                    var listedAttrs = null;

                    if (group === 'styleId' && !_.isEmpty(object)) {
                        listedAttrs = { group: object };

                    } else {
                        listedAttrs = getFilteredAttributes(whitelist[group], object);
                    }

                    if (!_.isEmpty(listedAttrs)) {
                        attrs[group] = listedAttrs;
                    }
                });

                return _.copy(attrs, true); // using deep clone to avoid problem with shortened operations
            }

            function filterAttributes(whitelist, attrs, options) {
                if (_.isNull(attrs)) { return {}; }

                var returnArr = {},
                    type = Utils.getStringOption(options, 'type', 'drawing');

                _.each(whitelist, function (array, key) {
                    if (
                        // twoPoint-drawings (or -shapes) should not copy fill-styles
                        !(twoPointShape && key === 'fill')
                        &&
                        (
                            (type === 'drawing' && key !== 'character' && key !== 'paragraph')
                            ||
                            key === type
                        )
                    ) {
                        if (!_.isUndefined(attrs[key])) {
                            returnArr[key] = {};
                            _.each(attrs[key], function (v, k) {
                                if (_.contains(array, k)) {
                                    returnArr[key][k] = v;
                                }
                            });
                        }
                    }
                });

                return returnArr;
            }

            function getMergedDrawingAttributes(drawAttrs) {
                // remove character and paragraph-attributes from drawing-attributes
                delete drawAttrs.character;
                delete drawAttrs.paragraph;
                return drawAttrs;
            }

            function isNoTextElement() {
                return (drawingType === 'image' || twoPointShape || drawingType === 'table');
            }

            function extendChangeTrackingAttributes(attrs, node) {
                if (changeTrack.isActiveChangeTracking()) {
                    // Expanding operation for change tracking with old explicit attributes
                    oldAttrs      = changeTrack.getOldNodeAttributes(node);
                    oldAttrs      = _.extend(oldAttrs, changeTrack.getChangeTrackInfo());
                    attrs.changes = { modified: oldAttrs };
                }
                return attrs;
            }

            function generateAttributeOperation(generator, options) {
                var settings = {
                    start: Utils.getArrayOption(options, 'start'),
                    attrs: Utils.getObjectOption(options, 'attrs')
                };

                if (Utils.getArrayOption(options, 'end')) {
                    settings.end = Utils.getArrayOption(options, 'end');
                }

                generator.generateOperation(Operations.SET_ATTRIBUTES, settings);
            }

            var rootNode            = self.getCurrentRootNode(),

                selection           = self.getSelection(),
                hasRange            = selection.hasRange(),

                reset               = Utils.getBooleanOption(options, 'reset', false),
                changeTrack         = self.getChangeTrack(),
                oldAttrs            = null,

                drawingPos          = null,
                startParaPos        = null,
                startTextPos        = null,

                twoPointShape       = false,
                drawingNode         = null,
                drawingType         = null;

            // get all needed oxo-position
            if (selection.isDrawingSelection()) {
                drawingPos          = selection.getStartPosition();
                startParaPos        = _.clone(drawingPos);
                startParaPos.push(0);
                startTextPos        = _.clone(startParaPos);
                startTextPos.push(0);

                drawingNode         = Position.getDrawingElement(rootNode, drawingPos);
                drawingType         = DrawingFrame.getDrawingType(drawingNode);
                twoPointShape       = DrawingFrame.isTwoPointShape(drawingNode);

            } else {
                startTextPos        = selection.getStartPosition();
                startParaPos        = startTextPos.slice(0, startTextPos.length - 1);
                drawingPos          = startParaPos.slice(0, startParaPos.length - 1);
            }

            // get corresponding nodes to oxo-position
            var textSpanNode        = (!twoPointShape && drawingType !== 'image' && drawingType !== 'table' && drawingType !== 'group') ? Position.getSelectedElement(rootNode, startTextPos, 'span')   : null,
                paragraphNode       = (!twoPointShape && drawingType !== 'image' && drawingType !== 'table' && drawingType !== 'group') ? Position.getParagraphElement(rootNode, startParaPos)          : null;

            // formatpainter is active (ready to paste saved styles)
            if (formatPainterActive && !activate && !reset) {
                var generator       = self.createOperationGenerator(),
                    start           = null,
                    end             = null;

                self.getUndoManager().enterUndoGroup(function () {
                    var charOpAttrs = null,
                        paraOpAttrs = null,
                        styleDrawing = true;

                    if (!twoPointShape && drawingType !== 'image') {
                        self.doCheckImplicitParagraph(startTextPos); // 54885
                    }

                    if (hasRange) {
                        selection.iterateContentNodes(function (paragraphNode, position, startOffset, endOffset) {
                            charOpAttrs = getWhitelistedAttrs({ source: 'character', target: drawingType, forceMerged: true });
                            start       = _.clone(position);
                            end         = _.clone(position);

                            if (!DOM.isEmptyParagraph(paragraphNode) && !DOM.isEmptyListParagraph(paragraphNode) && !DOM.isTextFrameTemplateTextParagraph(paragraphNode)) {
                                start.push(startOffset || 0);
                                end.push(endOffset || (Position.getParagraphNodeLength(paragraphNode) - 1)); // length of paragraph must be reduced by 1 to be usable in operations
                            }

                            if (!_.isEmpty(charOpAttrs)) {
                                extendChangeTrackingAttributes(charOpAttrs, paragraphNode);

                                generateAttributeOperation(generator, {
                                    start:  _.clone(start),
                                    end:    _.clone(end),
                                    attrs:  charOpAttrs
                                });
                            }

                            paraOpAttrs = getWhitelistedAttrs({ source: 'paragraph', target: drawingType });

                            if (!_.isEmpty(paraOpAttrs)) {
                                extendChangeTrackingAttributes(paraOpAttrs, paragraphNode);
                                var formatListLevel, formatListStyleId;
                                paraOpAttrs.paragraph = paraOpAttrs.paragraph || {};

                                if (formatPainterAttrs.explicit.paragraph.paragraph) {
                                    formatListLevel = formatPainterAttrs.explicit.paragraph.paragraph.listLevel;
                                    formatListStyleId = formatPainterAttrs.explicit.paragraph.paragraph.listStyleId;
                                }
                                _.extend(paraOpAttrs, {
                                    styleId: formatPainterAttrs.explicit.paragraph.styleId || null,
                                    paragraph: {
                                        listStyleId: formatListStyleId || null,
                                        listStyleLevel: formatListLevel || -1
                                    }
                                });

                                generateAttributeOperation(generator, {
                                    start: _.clone(position),
                                    attrs: paraOpAttrs
                                });
                            }

                        }, null, { startTextPos: selection.getStartPosition(), endPos: selection.getEndPosition() });

                        // do not apply drawing-styles
                        if (selection.isAdditionalTextframeSelection()) { styleDrawing = false; }

                    // single click into a word
                    } else {
                        var wordOxoPos = (hasRange) ? null : Position.getWordBoundaries(rootNode, startTextPos);

                        // at this point wordOxoPos can be null or an array of numbers or an array with two logical positions (task 58492)

                        // equal, if there was no word clicked
                        if (_.isArray(wordOxoPos) && _.isArray(wordOxoPos[0]) && _.isArray(wordOxoPos[1]) && !_.isEqual(wordOxoPos[0], wordOxoPos[1])) {
                            var tmpStartPos = (hasRange) ? startTextPos               : wordOxoPos[0];
                            var tmpEndPos   = (hasRange) ? selection.getEndPosition() : wordOxoPos[1];

                            charOpAttrs     = getWhitelistedAttrs({ source: 'character', target: drawingType });
                            start           = tmpStartPos;
                            end             = Position.decreaseLastIndex(tmpEndPos);

                            if (!_.isEmpty(charOpAttrs)) {
                                extendChangeTrackingAttributes(charOpAttrs, textSpanNode);

                                generateAttributeOperation(generator, {
                                    start:  _.clone(start),
                                    end:    _.clone(end),
                                    attrs:  charOpAttrs
                                });
                            }
                        }

                        paraOpAttrs = getWhitelistedAttrs({ source: 'paragraph', target: drawingType });

                        if (!_.isEmpty(paraOpAttrs)) {
                            extendChangeTrackingAttributes(paraOpAttrs, paragraphNode);

                            if (formatPainterAttrs.explicit.paragraph.styleId) {
                                paraOpAttrs.styleId = formatPainterAttrs.explicit.paragraph.styleId;
                            }

                            generateAttributeOperation(generator, {
                                start:  _.clone(startParaPos),
                                attrs:  paraOpAttrs
                            });
                        }
                    }

                    var drawOpAttrs = getWhitelistedAttrs({ source: 'drawing', target: drawingType });

                    if (!_.isEmpty(drawOpAttrs) && styleDrawing) {
                        extendChangeTrackingAttributes(drawOpAttrs, drawingNode);

                        generateAttributeOperation(generator, {
                            start: drawingPos,
                            attrs: drawOpAttrs
                        });
                    }

                    // apply all collected operations (inside undo group)
                    self.applyOperations(generator);
                });

                this.setFormatPainterState(activate);

            // Activate formatpainter and save current styles!
            } else if (!formatPainterActive && activate) {
                var mergedChar   = (isNoTextElement())     ? null : self.getCharacterStyles().getElementAttributes(textSpanNode),
                    mergedPara   = (isNoTextElement())     ? null : self.getParagraphStyles().getElementAttributes(paragraphNode),
                    mergedDraw   = (_.isNull(drawingType)) ? null : getMergedDrawingAttributes(self.getDrawingStyles().getElementAttributes(drawingNode)),

                    explicitChar = (isNoTextElement())     ? null : AttributeUtils.getExplicitAttributes(textSpanNode),
                    explicitPara = (isNoTextElement())     ? null : AttributeUtils.getExplicitAttributes(paragraphNode),
                    explicitDraw = (_.isNull(drawingType)) ? null : AttributeUtils.getExplicitAttributes(drawingNode);

                // special case for ODF: inherit attributes
                if (app.isODF()) {
                    // merge character-attributes of all parent elements
                    if (explicitPara && explicitPara.character) {
                        explicitChar.character = _.extend(explicitPara.character, explicitChar.character);
                        delete explicitPara.character;
                    }
                    if (explicitDraw && explicitDraw.character) {
                        explicitChar.character = _.extend(explicitDraw.character, explicitChar.character);
                        delete explicitDraw.character;
                    }

                    // merge paragraph-attributes of all parent elements
                    if (explicitDraw && explicitDraw.paragraph) {
                        explicitChar.paragraph = _.extend(explicitDraw.paragraph, explicitChar.paragraph);
                        delete explicitDraw.paragraph;
                    }
                }

                formatPainterAttrs = {
                    explicit: {
                        character:  explicitChar,
                        paragraph:  explicitPara,
                        drawing:    explicitDraw
                    },
                    merged: {
                        character:  filterAttributes((drawingType) ? formatPainterWhitelist[drawingType] : formatPainterWhitelist.character, mergedChar, { type: 'character' }),
                        paragraph:  filterAttributes((drawingType) ? formatPainterWhitelist[drawingType] : formatPainterWhitelist.paragraph, mergedPara, { type: 'paragraph' }),
                        drawing:    filterAttributes(formatPainterWhitelist[drawingType], mergedDraw)
                    }
                };

                this.setFormatPainterState(activate);

                // close the change track popup, if format painter is active
                if (app.isTextApp() &&  app.getView().getChangeTrackPopup().isVisible()) {
                    app.getView().getChangeTrackPopup().hide();
                }

            } else if (reset) {
                this.setFormatPainterState(false);
            }
        };

        this.getDefaultMarginBottom = function () {
            return this.getAttributePool().getDefaultValue('paragraph', 'marginBottom') || 352;
        };

        /**
         * Returns a multiplier value for the spacing represented by the passed
         * paragraph attributes, as used in GUI controls.
         *
         * @param {Object} paraAttrs
         *  A map with paragraph attributes.
         *
         * @returns {Number|Null}
         *  A multiplier value for the spacing represented by the passed
         *  paragraph attributes; or null, if no explicit spacing could be
         *  determined.
         */
        this.getParagraphSpacing = function (paraAttrs) {
            var marginBottom = paraAttrs.marginBottom;
            if (typeof marginBottom !== 'number') { return null; }
            return Math.round(marginBottom / this.getDefaultMarginBottom());
        };

        /**
         * Changes the spacing between the selected paragraphs.
         *
         * @param {Number} multiplier
         *  A multiplier used in GUI controls that specifies the count of
         *  default bottom margins defined for paragraphs.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after all operations have been
         *  generated and applied successfully.
         */
        this.setParagraphSpacing = function (multiplier) {
            var marginBottom = multiplier * this.getDefaultMarginBottom();
            // bug 46249: margin top needs to be considered when 'Paragraph spacing' is set to none in our GUI
            if (self.useSlideMode()) {
                return this.setAttributes('paragraph', { paragraph: { spacingBefore: 0, spacingAfter: { value: marginBottom, type: 'fixed' } } });
            }
            return this.setAttributes('paragraph', { paragraph: { marginTop: 0, marginBottom: marginBottom } });
        };

        /**
         * Changes multiple attributes of the specified attribute family for
         * the passed table node. Different from setAttributes in a way that uses
         * passed node instead of selected node with selection.
         *
         * @param {Node|jQuery} tableNode
         *  Table node to which attributes are applied
         *
         * @param {String} family
         *  The name of the attribute family containing the passed attributes.
         *
         * @param {Object} attributes
         *  A map of attribute value maps (name/value pairs), keyed by
         *  attribute families.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.clear=false]
         *      If set to true, all existing explicit attributes will be
         *      removed from the selection while applying the new attributes.
         *  @param {Boolean} [options.onlyVisibleBorders=false]
         *      Setting border width directly at cells -> overwriting values of table style.
         *  @param {Boolean} [options.cellSpecificTableAttribute=false]
         *      Setting border mode directly at cells -> overwriting values of table style.
         *
         */
        this.setAttributesToPassedTableNode = function (tableNode, family, attributes, options) {

            var // the undo manager
                undoManager = self.getUndoManager();

            // Create an undo group that collects all undo operations generated
            // in the local setAttributes() method (it calls itself recursively
            // with smaller parts of the current selection).
            undoManager.enterUndoGroup(function () {

                var // logical position
                    localPosition = null,
                    // operations generator
                    generator = self.createOperationGenerator(),
                    // whether after assigning cell attributes, it is still necessary to assign table attributes
                    createTableOperation = true,
                    // the page node
                    editdiv = self.getNode(),
                    // the old attributes, required for change tracking
                    oldAttrs = null;

                // generates a 'setAttributes' operation with the correct attributes
                function generateSetAttributeOperation(startPosition, endPosition, oldAttrs) {

                    var // the options for the operation
                        operationOptions = { start: startPosition, attrs: _.clone(attributes) };

                    // adding the old attributes, author and date for change tracking
                    if (oldAttrs) {
                        oldAttrs = _.extend(oldAttrs, self.getChangeTrack().getChangeTrackInfo());
                        operationOptions.attrs.changes = { modified: oldAttrs };
                    }

                    // add end position if specified
                    if (_.isArray(endPosition)) {
                        operationOptions.end = endPosition;
                    }

                    // generate the 'setAttributes' operation
                    generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);
                }

                // nothig to do if no attributes will be changed
                if (_.isEmpty(attributes)) { return; }

                if (tableNode) {

                    localPosition = Position.getOxoPosition(editdiv, tableNode, 0);  // the logical position of the table

                    if (Utils.getBooleanOption(options, 'clear', false)) {
                        // removing hard attributes at tables and cells, so that the table style will be visible
                        Table.removeTableAttributes(self, editdiv, tableNode, localPosition, generator);
                    }

                    if (Utils.getBooleanOption(options, 'onlyVisibleBorders', false)) {
                        // setting border width directly at cells -> overwriting values of table style
                        Table.setBorderWidthToVisibleCells(tableNode, attributes, self.getTableCellStyles(), editdiv, generator, self);
                        createTableOperation = false;
                    }

                    if (Utils.getBooleanOption(options, 'cellSpecificTableAttribute', false)) {
                        // setting border mode directly at cells -> overwriting values of table style
                        Table.setBorderModeToCells(tableNode, attributes, TableStyles.getBorderStyleFromAttributes(self.getAttributes('cell').cell || {}), editdiv, generator, self);
                    }

                    // setting attributes to the table element
                    // -> also setting table attributes, if attributes are already assigned to table cells, to
                    // keep the getter functions simple and performant
                    if (createTableOperation) {
                        // Defining the oldAttrs, if change tracking is active
                        if (self.getChangeTrack().isActiveChangeTracking()) {
                            // Expanding operation for change tracking with old explicit attributes
                            oldAttrs = self.getChangeTrack().getOldNodeAttributes(tableNode);
                        }
                        generateSetAttributeOperation(localPosition, undefined, oldAttrs);
                    }
                }

                // apply all collected operations
                self.applyOperations(generator);

            }, this); // end of enterUndoGroup
        };

        /**
         * Returns the values of all formatting attributes of the elements in
         * the current selection associated to the specified attribute family.
         *
         * @param {String} family
         *  The name of the attribute family used to select specific elements
         *  in the current selection:
         *  - 'character': all text spans (text portions, text components),
         *  - 'paragraph': all paragraph nodes,
         *  - 'table': all table nodes,
         *  - 'drawing': all drawing object nodes.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.maxIterations]
         *      If set to true, all existing explicit attributes will be
         *      removed from the selection while applying the new attributes.
         *  @param {Boolean} [options.groupOnly=false]
         *      Whether only the attributes of the drawing group shall be
         *      collected, not of the children.
         *  @param {Number} [options.maxIterations=-1]
         *      An optional number for limiting the maximum iteration count for
         *      nodes to be investigated. If the value is set to -1, this limit
         *      is ignored.
         *
         * @returns {Object}
         *  A map of attribute value maps (name/value pairs), keyed by
         *  attribute family.
         */
        this.getAttributes = function (family, options) {

            var // the selection object
                selection = self.getSelection(),
                // whether the selection is a simple cursor
                isCursor = selection.isTextCursor(),
                // table or drawing element contained by the selection
                element = null,
                // resulting merged attributes
                mergedAttributes = null,
                // max iterations
                maxIterations = Utils.getOption(options, 'maxIterations'),
                // get only attributes of the group
                groupOnly = Utils.getBooleanOption(options, 'groupOnly', false);

            // merges the passed element attributes into the resulting attributes
            function mergeElementAttributes(elementAttributes) {

                var // whether any attribute is still unambiguous
                    hasNonNull = false;

                // merges the passed attribute value into the attributes map
                function mergeAttribute(attributes, name, value) {
                    if (!(name in attributes)) {
                        // initial iteration: store value
                        attributes[name] = value;
                    } else if (!_.isEqual(value, attributes[name])) {
                        // value differs from previous value: ambiguous state
                        attributes[name] = null;
                    }
                    hasNonNull = hasNonNull || !_.isNull(attributes[name]);
                }

                // initial iteration: store attributes and return
                if (!mergedAttributes) {
                    mergedAttributes = elementAttributes;
                    return;
                }

                // process all passed element attributes
                _(elementAttributes).each(function (attributeValues, subFamily) {
                    if (subFamily === 'styleId') {
                        mergeAttribute(mergedAttributes, 'styleId', attributeValues);
                    } else {
                        var mergedAttributeValues = mergedAttributes[subFamily];
                        _(attributeValues).each(function (value, name) {
                            mergeAttribute(mergedAttributeValues, name, value);
                        });
                    }
                });

                // stop iteration, if all attributes are ambiguous
                return hasNonNull ? undefined : Utils.BREAK;
            }

            switch (family) {

                case 'character':
                    selection.iterateNodes(function (node) {
                        return DOM.iterateTextSpans(node, function (span) {
                            // ignore empty text spans (they cannot be formatted via operations),
                            // but get formatting of an empty span selected by a text cursor
                            if (isCursor || (span.firstChild.nodeValue.length > 0) || isEmptyTextFrameSelection(span)) {
                                return mergeElementAttributes(self.getCharacterStyles().getElementAttributes(span));
                            }
                        });
                    }, null, { maxIterations: maxIterations });
                    if (isCursor && self.getPreselectedAttributes()) {
                        // add preselected attributes (text cursor selection cannot result in ambiguous attributes)
                        self.getAttributePool().extendAttributeSet(mergedAttributes, self.getPreselectedAttributes());
                    }
                    break;

                case 'paragraph':
                    selection.iterateContentNodes(function (paragraph) {
                        return mergeElementAttributes(self.getParagraphStyles().getElementAttributes(paragraph));
                    }, null, { maxIterations: maxIterations });
                    break;

                case 'cell':
                    selection.iterateTableCells(function (cell) {
                        return mergeElementAttributes(self.getTableCellStyles().getElementAttributes(cell));
                    });
                    break;

                case 'table':
                    element = selection.getEnclosingTable();

                    if (!element && self.useSlideMode() && selection.isDrawingSelection() && DrawingFrame.isTableDrawingFrame(selection.getSelectedDrawing())) {
                        element = selection.getSelectedDrawing();
                    }

                    if (element) {
                        mergeElementAttributes(self.getTableStyles().getElementAttributes(element));
                    }
                    break;

                case 'drawing':
                    _.each(selection.getSelectedDrawings(), function (oneDrawing) {
                        var drawingAttrs = null;
                        if (DrawingFrame.isGroupDrawingFrame(oneDrawing) && groupOnly !== true) {
                            _.each(DrawingFrame.getAllGroupDrawingChildren(oneDrawing), function (drawing) {
                                drawingAttrs = self.getDrawingStyles().getElementAttributes(drawing);
                                filterDrawingAttrs(drawing, drawingAttrs); // filter for the drawing attributes
                                mergeElementAttributes(drawingAttrs);
                            });
                        } else {
                            drawingAttrs = self.getDrawingStyles().getElementAttributes(oneDrawing);
                            if (selection.isMultiSelection()) { filterDrawingAttrs(oneDrawing, drawingAttrs); } // filter for the drawing attributes
                            mergeElementAttributes(drawingAttrs);
                        }
                    });

                    break;

                default:
                    Utils.error('Editor.getAttributes(): missing implementation for family "' + family + '"');
            }

            return mergedAttributes || {};
        };

        /**
         * Removes all hard-set attributes depending on the current selection.
         * A selection range clears only hard-set character attributes and without
         * a range the hard-set paragraph attributes are cleared.
         */
        this.resetAttributes = function () {

            var // the undo manager
                undoManager = self.getUndoManager(),
                // the selection object
                selection = self.getSelection();

            function clearParagraphAttributes() {
                var paraAttributes = self.getParagraphStyles().buildNullAttributeSet();
                delete paraAttributes.styleId;
                self.setAttributes('paragraph', paraAttributes);
            }

            function clearCharacterAttributes() {
                var charAttributes = self.getCharacterStyles().buildNullAttributeSet();
                // don't reset hyperlink attribute
                delete charAttributes.styleId;
                delete charAttributes.character.url;
                self.setAttributes('character', charAttributes);
            }

            if (!selection.hasRange()) {
                undoManager.enterUndoGroup(function () {

                    var start = selection.getStartPosition(),
                        end = selection.getEndPosition(),
                        activeRootNode = self.getCurrentRootNode(),
                        firstParagraph = Position.getLastNodeFromPositionByNodeName(activeRootNode, start, DOM.PARAGRAPH_NODE_SELECTOR),
                        lastParagraph = Position.getLastNodeFromPositionByNodeName(activeRootNode, end, DOM.PARAGRAPH_NODE_SELECTOR),
                        prevPara,
                        nextPara;

                    clearParagraphAttributes();
                    prevPara = Utils.findPreviousNode(activeRootNode, firstParagraph, DOM.PARAGRAPH_NODE_SELECTOR);
                    if (prevPara) {
                        self.getParagraphStyles().updateElementFormatting(prevPara);
                    }
                    nextPara = Utils.findNextNode(activeRootNode, lastParagraph, DOM.PARAGRAPH_NODE_SELECTOR);
                    if (nextPara) {
                        self.getParagraphStyles().updateElementFormatting(nextPara);
                    }
                    self.addPreselectedAttributes(self.getCharacterStyles().buildNullAttributeSet());
                }, this);
            } else {
                clearCharacterAttributes();
            }
        };

        /**
         * Getting the vertical text alignment of a currently selected text frame node.
         *
         * @returns {String}
         *  The vertical text alignment mode of the text inside a selected text frame node. If it cannot
         *  be determined, the default value 'top' is returned.
         */
        this.getVerticalAlignmentMode = function () {
            // the selection object
            var selection = self.getSelection();

            if (selection.isAnyTableDrawingSelection()) {
                var attr = self.getAttributes('cell').cell;
                return attr.alignVert === 'center' ? 'centered' : attr.alignVert;
            }
            // the selected text frame node (also finding text frames inside groups)
            var drawingFrame = selection.getAnyTextFrameDrawing({ forceTextFrame: true });
            // the text frame node inside the shape
            var textFrame = DrawingFrame.getTextFrameNode(drawingFrame);

            return (textFrame && textFrame.length > 0 && textFrame.attr(DrawingFrame.VERTICAL_ALIGNMENT_ATTRIBUTE)) || 'top';
        };

        /**
         * Setting the vertical text alignment mode inside a text frame.
         *
         * @param {String} mode
         *  The vertical text alignment mode of the text inside a selected text frame node.
         */
        this.setVerticalAlignmentMode = function (state) {
            // the selection object
            var selection = self.getSelection();
            // the operations generator
            var generator = self.createOperationGenerator();
            // the options for the setAttributes operation
            var operationOptions = {};
            // a container for the logical positions of all affected drawings
            var allDrawings = selection.getAllDrawingsInSelection(DrawingFrame.isTextFrameShapeDrawingFrame);

            if (!allDrawings.length > 0) { return; }

            if (selection.isAnyTableDrawingSelection()) {
                self.setAttribute('cell', 'alignVert', state === 'centered' ? 'center' : state);
            } else {
                // collecting the attributes for the operation
                operationOptions.attrs =  {};
                operationOptions.attrs.shape = { anchor: state };

                // iterating over all drawings
                _.each(allDrawings, function (oneDrawing) {
                    operationOptions.start = Position.getOxoPosition(self.getCurrentRootNode(), oneDrawing, 0);
                    // generate the 'setAttributes' operation
                    generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);
                });

                // apply all collected operations
                self.applyOperations(generator);
            }
        };

        // operation handler --------------------------------------------------

        /**
         * The handler for the setAttributes operation.
         *
         * Info:undo/redo generation is done inside 'implSetAttributes'.
         *
         * @param {Object} operation
         *  The operation object.
         *
         * @param {Boolean} external
         *  Whether the operation was triggered by an external client.
         *
         * @returns {Boolean}
         *  Whether the setting of attributes was successful.
         */
        this.setAttributesHandler = function (operation, external) {
            return implSetAttributes(operation.start, operation.end, operation.attrs, operation.target, external, operation.noUndo);
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = drawingFilterList = null;
        });

        // registering the filter for setting attributes to drawings. This is required
        // for multi drawing selections, that might make it possible to assign attributes
        // to drawings, that do not support these attributes.
        registerDrawingFilter();

    } // class AttributeOperationMixin

    // constants --------------------------------------------------------------

    /**
     * An object specifying those drawing types, at that attributes of the 'fill'
     * family can be assigned.
     * Info: This constant might move to DrawingLayer, especially if further
     *       constants of this type are required.
     */
    AttributeOperationMixin.FILL_DRAWING_TYPES = {
        shape: 1,
        group: 1,
        image: 1,
        chart: 1
    };

    /**
     * An object specifying those drawing types, at that attributes of the 'line'
     * family can be assigned.
     * Info: This constant might move to DrawingLayer, especially if further
     *       constants of this type are required.
     */
    AttributeOperationMixin.LINE_DRAWING_TYPES = {
        shape: 1,
        group: 1,
        image: 1,
        chart: 1,
        connector: 1
    };

    // exports ================================================================

    return AttributeOperationMixin;
});
