/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/text/view/view', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/utils/dateutils',
    'io.ox/office/baseframework/view/popup/compoundmenu',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/view/popup/attributestooltip',
    'io.ox/office/textframework/utils/config',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/view/view',
    'io.ox/office/textframework/view/labels',
    'io.ox/office/textframework/view/dialogs',
    'io.ox/office/textframework/view/popup/universalcontextmenu',
    'io.ox/office/text/view/controls',
    'io.ox/office/text/view/toolbars',
    'io.ox/office/text/view/rulerpane',
    'gettext!io.ox/office/text/main',
    'less!io.ox/office/text/view/style',
    'io.ox/core/tk/datepicker'
], function (Utils, Forms, DateUtils, CompoundMenu, AttributeUtils, AttributesToolTip, Config, Position, DOM, TextBaseView, Labels, Dialogs, UniversalContextMenu, Controls, ToolBars, RulerPane, gt) {

    'use strict';

    // convenience shortcuts
    var Button = Controls.Button;
    var CheckBox = Controls.CheckBox;
    var TextField = Controls.TextField;

    // predefined zoom factors
    var ZOOM_FACTORS = Utils.SMALL_DEVICE ? [100, 150, 200] : [35, 50, 75, 100, 150, 200, 300, 400, 600, 800];

    // class TextView =========================================================

    /**
     * The text editor view.
     *
     * Triggers the events supported by the base class EditView, and the
     * following additional events:
     * - 'change:zoom': When the zoom level for the displayed document was changed.
     * - 'change:pageWidth': When the width of the page or the inner margin of the
     *      page was changed. This happens for example on orientation changes on
     *      small devices.
     *
     * @constructor
     *
     * @extends TextBaseView
     *
     * @param {TextApplication} app
     *  The application containing this view instance.
     *
     * @param {TextModel} docModel
     *  The document model created by the passed application.
     */
    var TextView = TextBaseView.extend({ constructor: function (app, docModel) {

        var // self reference
            self = this,

            // the root node of the entire application pane
            appPaneNode = null,

            // the scrollable document content node
            contentRootNode = null,

            // the page node
            pageNode = null,

            //page content node
            pageContentNode = null,

            // toolbar pane for the horizontal ruler
            rulerPane = null,

            // scroll position of the application pane
            scrollPosition = { left: 0, top: 0 },

            // debug operations for replay
            replayOperations = null,

            // maximum osn for debug operations for replay
            replayOperationsOSN = null,

            // current zoom type (percentage or keyword)
            zoomType = (Utils.COMPACT_DEVICE && Utils.isPortrait()) ? 'width' : 100,

            // the current effective zoom factor, in percent
            zoomFactor = 100,

            // outer width of page node, used for zoom calculations
            pageNodeWidth = 0,

            // outer height of page node, used for zoom calculations
            pageNodeHeight = 0,

            // the generic content menu for all types of contents
            contextMenu = null,

            // change track pop-up and controls
            changeTrackPopup = null,
            changeTrackBadge = null,
            changeTrackAcceptButton = null,
            changeTrackRejectButton = null,
            changeTrackShowButton = null,

            // change track pop-up timeout promise
            changeTrackPopupTimeout = null,

            // instance of change field format popup
            fieldFormatPopup = null,
            fieldFormatList = null,
            descriptionNode = null,
            autoUpdateCheckbox = null,
            setToButton = null,
            toggleDatePickerBtn = null,
            datePickerNode = $('<div>').addClass('date-picker-node'),
            todayDatepickerWrapper = $('<div>').addClass('today-datepicker-wrapper'),
            currentCxNode = null,
            autoDate = null,
            currentFormatOfFieldPopup = null,

            // button in popup to remove field
            //removeFieldButton = new Button(self, { label: gt('Remove'), value: 'remove', tooltip: gt('Remove this field from document') }),

            // the top visible paragraph set during fast loading the document
            topVisibleParagraph = null,

            // storage for last know touch position
            // (to open context menu on the correct position)
            touchX = null,
            touchY = null;

        // TODO: Use the app-content node defined in baseview

        // base constructor ---------------------------------------------------

        TextBaseView.call(this, app, docModel, {
            initHandler: initHandler,
            initDebugHandler: initDebugHandler,
            initGuiHandler: initGuiHandler,
            initDebugGuiHandler: initDebugGuiHandler,
            grabFocusHandler: grabFocusHandler,
            contentScrollable: true,
            contentMargin: 30,
            enablePageSettings: true,
            showOwnCollaboratorColor: true
        });

        // private methods ----------------------------------------------------

        /**
         * Moves the browser focus focus to the editor root node.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Object} [options.changeEvent]
         *     TODO
         *  @param {Boolean} [options.afterImport=false]
         *     If grab focus handler is called directly after import finished.
         */
        function grabFocusHandler(options) {
            if (Utils.TOUCHDEVICE) {
                var changeEvent = Utils.getOption(options, 'changeEvent', { sourceEventType: 'click' });
                var softKeyboardOpen = Utils.isSoftKeyboardOpen();
                var sourceType = changeEvent.sourceEventType;

                if (Utils.SOFTKEYBORD_TOGGLE_MODE) {
                    if ((sourceType === 'tap' || sourceType === 'click') && (!softKeyboardOpen || docModel.isDrawingSelected()) && self.isSoftkeyboardEditModeOff()) {
                        Utils.focusHiddenNode();
                    } else {
                        Utils.trigger('change:focus', pageNode, pageNode, null);
                        Utils.setFocus(pageNode);

                        // when the user opens the shrunken tab burger menu and the focus is set back to the page the body can scroll
                        $('body').scrollTop(0);

                        docModel.trigger('selection:focusToEditPage');
                    }

                // old normal touch behavior
                } else {
                    if ((sourceType === 'tap' || sourceType === 'click') && (!softKeyboardOpen || docModel.isDrawingSelected())) {
                        Utils.focusHiddenNode();
                    } else {
                        Utils.trigger('change:focus', pageNode, pageNode, null);
                        Utils.setFocus(pageNode);
                    }
                }

            // desktop
            } else {
                var keepSelectionOnImport = Utils.getBooleanOption(options, 'afterImport', false);
                // don't set focus after import, if focus is already at the page node, to prevent browser from setting cursor to start position
                // See also #46562
                if (keepSelectionOnImport && Utils.getActiveElement() === pageNode[0]) {
                    return;
                }
                Utils.setFocus(pageNode);
            }
        }

        /**
         * Caches the current scroll position of the application pane.
         */
        function saveScrollPosition() {
            scrollPosition = { top: contentRootNode.scrollTop(), left: contentRootNode.scrollLeft() };
        }

        /**
         * Restores the scroll position of the application pane according to
         * the cached scroll position.
         */
        function restoreScrollPosition() {
            contentRootNode.scrollTop(scrollPosition.top).scrollLeft(scrollPosition.left);
        }

        /**
         * Scrolls the application pane to the focus position of the text
         * selection.
         */
        function scrollToSelection(options) {

            var // the boundaries of the text cursor
                boundRect = docModel.getSelection().getFocusPositionBoundRect();

            // make cursor position visible in application pane
            if (boundRect) {
                self.scrollToPageRectangle(boundRect, options);
            }
        }

        /**
         * Open the softkeyboard on touch devices.
         */
        function openSoftKeyboard() {

            app.getWindowNode().removeClass('soft-keyboard-off');
            Utils.setFocus(pageNode);

            $('body').scrollTop(0);
            scrollToSelection({ regardSoftkeyboard: true });

            docModel.trigger('selection:focusToEditPage');
        }

        /**
         * Close the softkeyboard on touch devices.
         */
        function closeSoftKeyboard() {
            app.getWindowNode().addClass('soft-keyboard-off');
            Utils.focusHiddenNode();
        }

        /**
         * Performance: Register in the selection, whether the current operation is an insertText
         * operation. In this case the debounced 'scrollToSelection' function can use the saved
         * DOM node instead of using Position.getDOMPosition in 'getPointForTextPosition'. This
         * makes the calculation of scrolling requirements faster.
         */
        function registerInsertText(options) {
            docModel.getSelection().registerInsertText(options);

            //fixing the problem with an upcoming soft-keyboard
            if (Utils.TOUCHDEVICE) { scrollToSelection(); }
        }

        /**
         * Gets the anchor (start) node of the given selection.
         *
         * @returns {Node | null}
         *  the anchor node of the given selection, otherwise null if its not found.
         */
        function getChangeTrackPopupAnchorNode() {
            if (!docModel || !docModel.getSelection()) { return null; }

            if (docModel.isDrawingSelected()) {
                return docModel.getSelection().getSelectedDrawing();

            } else {
                var selectionEndPoint = Position.getDOMPosition(docModel.getCurrentRootNode(), docModel.getSelection().getEndPosition());
                if (!selectionEndPoint) { return pageNode; }
                return selectionEndPoint.node.parentNode;
            }
        }
        /**
         * Returns objects required for creation of the change track popup, containing:
         * - relevant change track info containing type, author, and date of the change,
         * - plus additional info containing info/accept/reject button texts.
         *
         * @returns {Object | Null}
         *  returns an object containing required meta information for the popup, and null if current selection is not a change
         *  tracked node (changeTrackInfo is null)
         *
         */
        function getChangeTrackPopupMeta() {

            // get change tracking information from model
            var changeTrackInfo = _.clone(docModel.getChangeTrack().getRelevantTrackingInfo());
            if (!changeTrackInfo) { return null; }

            // create a GUI label for the action type
            changeTrackInfo.actionText = Labels.CHANGE_TRACK_LABELS[changeTrackInfo.type];
            if (!changeTrackInfo.actionText) {
                Utils.error('TextView.getChangeTrackPopupMeta(): missing label for change tracking type "' + changeTrackInfo.type + '"');
                return null;
            }
            changeTrackInfo.actionText += ': ' + Utils.capitalize(changeTrackInfo.nodeType);

            return changeTrackInfo;
        }

        /**
         * Custom Change track date formatter.
         *
         * @param {Number} timestamp
         *  timestamp in milliseconds
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.toLocal=true]
         *      timestamp are converted to local time zone per default.
         *
         * @returns {String}
         *  The formatted date string.
         */
        function formatChangeTrackDate(timestamp, options) {

            if (!_.isNumber(timestamp)) { return gt('Unknown'); }

            var toLocal = Utils.getBooleanOption(options, 'toLocal', true);
            var now = toLocal ? new Date() : DateUtils.getUTCDate();
            var timeStampDate = toLocal ? new Date(timestamp) : DateUtils.getUTCDate(timestamp);

            function getTimeString() { return DateUtils.format(timeStampDate, DateUtils.getTimeFormat()); }

            function getDateString() {
                // get locale country default date format
                var formatString = DateUtils.getDateFormat();
                // hide year if the document is modified in this year
                if (timeStampDate.getYear() === now.getYear()) { formatString = DateUtils.getDateFormat(); }
                // show full day name if timestamp is still in the previous 7 days
                if (timestamp > (_.now() - DateUtils.MSEC_PER_WEEK)) { formatString = DateUtils.getWeekDayFormat(); }
                return DateUtils.format(timeStampDate, formatString);
            }

            function isToday(date) {
                return date.getDate() === now.getDate() &&
                    date.getMonth() === now.getMonth() &&
                    date.getYear() === now.getYear();
            }

            function isYesterday(date) {
                return date.getDate() === (now.getDate() - 1) &&
                    date.getMonth() === now.getMonth() &&
                    date.getYear() === now.getYear();
            }

            // TODO: better use today & yesterday with moment.js !!!
            // show yesterday and only time if the timestamp is from today
            if (isYesterday(timeStampDate)) { return gt('Yesterday'); }
            if (isToday(timeStampDate)) { return getTimeString(); }

            // return date string for all other cases
            return getDateString();
        }

        /**
         * Initiates and shows change track popup where applicable.
         *
         * @param {Object} options
         *  @param {jQuery Event} options.browserEvent
         *   The original jQuery Event object from a user selection. This object
         *   is used to filter out selections that are triggered by the keyboard.
         */
        function handleChangeTrackPopup(options) {

            // do not show popup for odf files (not supported yet)
            if (app.isODF()) { return; }

            if (Utils.IOS) {
                docModel.getChangeTrack().setHighlightToCurrentChangeTrack();
            }

            if (Config.COMBINED_TOOL_PANES) {
                return;
            }

            // make sure old popup (if any) is hidden
            changeTrackPopup.hide();

            var // the event of the browser
                browserEvent = Utils.getObjectOption(options, 'browserEvent', null),
                // array of event types, that will trigger change track popup
                relevantEvents = ['mousedown', 'mouseup', 'touchstart', 'touchend'];

            // quit early if no listened events are found
            if (!browserEvent || !_.contains(relevantEvents, browserEvent.type) || (browserEvent.type === 'mousedown' && browserEvent.button === 2)) {
                if (changeTrackPopupTimeout) { changeTrackPopupTimeout.abort(); }
                return;
            }

            // clear any existing proprietary change track selection (e.q. table column selections)
            docModel.getChangeTrack().clearChangeTrackSelection();

            // abort existing old timeout promise
            if (changeTrackPopupTimeout) { changeTrackPopupTimeout.abort(); }

            changeTrackPopupTimeout = self.executeDelayed(function () {

                // hyperlink popup has higher priority. Don't show change track popup if its visible.
                if (pageNode.find('.inline-popup.hyperlink').is(':visible')) { return; }

                // don't show the change track popup, if format painter is active
                if (docModel.isFormatPainterActive()) { return; }

                // try show changetrack popup directly if check spelling is off
                // TODO: spell checker popups must have priority
                // -> check if spell checker is enabled or popup is not visible
                self.showChangeTrackPopup();

            }, 'TextView.changeTrackPopupTimeout', 1000);
        }

        /**
         * Initializes the change track popup.
         */
        function initChangeTrackPopup() {
            var ctRoot = self.getContentRootNode();

            // create and prepare controls of the popup
            changeTrackPopup = new CompoundMenu(self, {
                classes: 'change-track-popup f6-target',
                autoFocus: false,
                autoClose: false,
                autoLayout: false,
                rootContainerNode: ctRoot
            });
            changeTrackAcceptButton.setLabel(gt('Accept'));
            changeTrackRejectButton.setLabel(gt('Reject'));
            changeTrackShowButton.setLabel(gt('Show'));
            changeTrackPopup
                .addGroup(null, changeTrackBadge)
                .addSeparator()
                .addGroup('acceptSelectedChangeTracking', changeTrackAcceptButton, { inline: true })
                .addGroup('rejectSelectedChangeTracking', changeTrackRejectButton, { inline: true })
                .addGroup('changetrackPopupShow', changeTrackShowButton, { inline: true });

            // disable inherited contenteditable attribute from page node to prevent text selection
            changeTrackPopup.getNode().attr('contenteditable', false);

            // ARIA
            changeTrackPopup.getNode().attr({ tabindex: '-1', role: 'dialog', 'aria-label': gt('Change tracking') });

            // manual popup sizing and positioning
            changeTrackPopup.on('popup:beforelayout', function () {
                var anchorNode = $(getChangeTrackPopupAnchorNode()),
                    anchorPosition = Utils.getChildNodePositionInNode(ctRoot, anchorNode),
                    contentNode = this.getContentNode(),
                    contentSize = Utils.getExactNodeSize(contentNode),
                    zoom = self.getZoomFactor();
                // calculate positioning based on the anchor node position relative to the content root node
                //anchorPosition.left = anchorPosition.left;
                anchorPosition.top += anchorNode.height() * zoom;
                anchorPosition.width = contentSize.width;
                anchorPosition.height = contentSize.height;
                // apply positioning and dimensions
                this.getNode().css(anchorPosition);
            });
        }

        /**
         * Gets the anchor (start) node of the given field.
         *
         * @returns {jQuery | null}
         *  The anchor node of the given field, otherwise null if its not found.
         */
        function getFieldAnchorNode() {
            return docModel.getFieldManager().getSelectedFieldNode();
        }

        /**
         * Initialize field format change popup.
         */
        function initFieldFormatPopup() {
            var contentRootNode = self.getContentRootNode();
            var fieldFormatPopupNode = null;
            var setToButtonNode;
            var datePickerToggleBtnNode;
            var popupViewComponent;
            var autoUpdateCheckboxNode;

            fieldFormatList = new Controls.FieldFormatList(self);
            descriptionNode = $('<div>').addClass('field-format-description').text(gt('Edit field'));
            autoUpdateCheckbox = new CheckBox(self, { label: gt('Update automatically'), boxed: true, tooltip: gt('Update this date field automatically on document load, download and mail') });
            autoUpdateCheckboxNode = autoUpdateCheckbox.getNode();
            setToButton = new Button(self, { label: gt('Set to today'), tooltip: /*#. Set to today's date */ gt('Set to today\'s date') }, { inline: true });
            setToButtonNode = setToButton.getNode();
            setToButtonNode.addClass('set-today-btn');
            toggleDatePickerBtn = new Button(self, { icon: 'fa fa-calendar', tooltip: /*#. Pick date from calendar */ gt('Pick date') }, { inline: true });
            datePickerToggleBtnNode = toggleDatePickerBtn.getNode();
            datePickerToggleBtnNode.addClass('toggle-datepicker-btn');

            fieldFormatPopup = new CompoundMenu(self, {
                classes: 'field-format-popup f6-target',
                autoFocus: true,
                autoClose: false,
                autoLayout: false,
                rootContainerNode: contentRootNode
            });

            fieldFormatPopupNode = fieldFormatPopup.getNode();
            // ARIA and prevention of text selection
            fieldFormatPopupNode.attr({ contenteditable: false, tabindex: '-1', role: 'dialog', 'aria-label': gt('Field formatting') });
            popupViewComponent = fieldFormatPopupNode.find('.view-component');
            popupViewComponent.prepend(descriptionNode);

            fieldFormatPopup
                .addSectionLabel(gt('Select format'))
                .addGroup('document/formatfield', fieldFormatList)
                .addSectionLabel(gt('Change date'))
                .addGroup(null, autoUpdateCheckbox);
            //.addGroup('removeField', removeFieldButton, { inline: true });

            datePickerNode.datepicker({
                calendarWeeks: false,
                todayBtn: false
            }).hide();

            todayDatepickerWrapper.append(setToButtonNode, datePickerToggleBtnNode);
            popupViewComponent.append(todayDatepickerWrapper, datePickerNode);
            todayDatepickerWrapper.add(autoUpdateCheckboxNode).add(autoUpdateCheckboxNode.prev()).add(autoUpdateCheckboxNode.prev().prev()).add(setToButtonNode).addClass('dateSection');

            self.listenTo(datePickerToggleBtnNode, 'click touch', function () { self.toggleDatePicker(); });

            // manual popup sizing and positioning
            self.listenTo(fieldFormatPopup, 'popup:beforelayout', function () {
                var anchorNode = $(getFieldAnchorNode()),
                    anchorPosition,
                    contentNode = this.getContentNode(),
                    contentSize = Utils.getExactNodeSize(contentNode),
                    zoom = self.getZoomFactor();

                if (!anchorNode.length) {
                    docModel.getFieldManager().destroyFieldPopup();
                    return;
                }
                anchorPosition = Utils.getChildNodePositionInNode(contentRootNode, anchorNode);
                // calculate positioning based on the anchor node position relative to the content root node
                //anchorPosition.left = anchorPosition.left;
                anchorPosition.top += anchorNode.height() * zoom;
                anchorPosition.width = contentSize.width;
                anchorPosition.height = contentSize.height;
                // apply positioning and dimensions
                this.getNode().css(anchorPosition);
            });

            self.listenTo(fieldFormatPopup, 'popup:hide', function () {
                docModel.getSelection().restoreBrowserSelection();
            });
            self.listenTo(datePickerNode, 'changeDate', function () {
                var date = datePickerNode.data().datepicker.viewDate;
                var pickedDate = self.formatDateWithNumberFormatter(date, currentFormatOfFieldPopup);
                var standardizedDate = self.formatDateWithNumberFormatter(date, 'yyyy-mm-dd\\Thh:mm:ss.00');

                currentCxNode.data('datepickerValue', datePickerNode.data().datepicker.getDate());
                self.trigger('fielddatepopup:change', { value: pickedDate, standard: standardizedDate });
            });

            self.listenTo(autoUpdateCheckbox, 'group:change', function () {
                self.autoFieldCheckboxUpdate();
            });

            self.listenTo(setToButton, 'group:change', function () {
                self.setFieldToTodayDate();
            });
        }

        /**
         * Finding the first paragraph that is inside the visible area
         */
        function setFirstVisibleParagraph() {

            var // all paragraph elements in the document
                allParagraphs = pageNode.find(DOM.PARAGRAPH_NODE_SELECTOR),
                // top offset for the scroll node
                topOffset = Utils.round(contentRootNode.offset().top, 1);

            // iterating over all collected change tracked nodes
            Utils.iterateArray(allParagraphs, function (paragraph) {

                if (Utils.round($(paragraph).offset().top, 1) > topOffset) {
                    topVisibleParagraph = paragraph;
                    return Utils.BREAK;
                }

            });
        }

        /**
         * Returns whether the contextmenu should be displayed for the given event.
         */
        function canShowContextMenu(event) {
            // quit if there were no contextmenu event
            if (_.isUndefined(event)) { return false; }
            // quit if the contextmenu event was triggered on a node inside comment meta info
            if ($(event.target).hasClass('commentmetainfo') || ($('.commentmetainfo').has(event.target).length > 0)) { return false; }
            // quit if the contextmenu event was triggered on a node inside comment info
            if ($(event.target).hasClass('commentbottom') || ($('.commentbottom').has(event.target).length > 0)) { return false; }

            return true;
        }

        /**
         * Handles the 'contextmenu' event for the complete browser.
         */
        function globalContextMenuHandler(event) {
            // check whether to show the ox, native or no context menu
            if (!canShowContextMenu(event)) { return false; }

            // quit if the contextmenu event was triggered on an input-/textfield
            // workaround for android: use native context menu for textareas and input elements.
            // e.g. paste an image URL into the insert image dialog, bug #52920.
            if ($(event.target).is('input, textarea')) { return true; }

            var // was the event triggered by keyboard?
                isGlobalTrigger = Utils.isContextEventTriggeredByKeyboard(event),
                // node, on which the new event should be triggered
                triggerNode     = null,

                // search inside this node for the allowed targets
                windowNode      = app.getWindowNode(),
                // parent elements on which 'contextmenu' is allowed (inside the app node)
                winTargetAreas  = ['.commentlayer', '.page', '.textdrawinglayer', '.popup-content'],
                // global target areas
                globalTargetAreas  = ['.popup-container > .popup-content'],
                // was one of these targets clicked
                targetClicked   = false;

            // check if the event target is inside of one of the allowed containers
            _.each(winTargetAreas, function (selector) {
                var parentNode = windowNode.find(selector);

                if (parentNode.length > 0) {
                    if (Utils.containsNode(parentNode, event.target) === true || Utils.isElementNode(event.target, selector)) {
                        targetClicked = true;
                    }
                }
            });
            _.each(globalTargetAreas, function (selector) {
                var parentNode = $('body > ' + selector);

                if (parentNode.length > 0) {
                    if (Utils.containsNode(parentNode, event.target) === true || Utils.isElementNode(event.target, selector)) {
                        targetClicked = true;
                    }
                }
            });

            // get out here, if no allowed target was clicked, or the globla trigger
            // was fired (via keyboard)
            if (!targetClicked && !isGlobalTrigger) { return false; }

            // if a drawing is selected
            if (docModel.isDrawingSelected()) {
                if ($(event.target).is('.tracker')) { return false; }
                // get the (first) selected drawing as trigger node
                triggerNode = $(docModel.getSelection().getSelectedDrawing()[0]);

            } else {
                // get the current focus node to trigger on it
                _.each($('.popup-content'), function (popupContentNode) {
                    if (Utils.containsNode(popupContentNode, event.target)) { triggerNode = $(event.target); }
                });

                if (_.isNull(triggerNode)) {
                    triggerNode = $(event.target);
                }

                if (_.isNull(triggerNode)) {
                    Utils.warn('TextView.globalContextMenuHandler(): No "triggerNode" found. Contextmenu will not be open!');
                    return;
                }
            }

            // if we have a last known touch position, use it
            if (touchX && touchY) {
                event.pageX = touchX;
                event.pageY = touchY;

            // in case of global trigger (via keyboard), or x/y are both "0"
            // locate a meaningful position
            } else if (isGlobalTrigger || (event.pageX === 0 && event.pageY === 0) || (_.isUndefined(event.pageX) && _.isUndefined(event.pageY))) {
                // find the position of the trigger node
                var xy = Position.getPixelPositionToRootNodeOffset($('html'), triggerNode);
                // and set pageX/pageY manually
                event.pageX = (xy.x + (triggerNode.width() / 2));
                event.pageY = (xy.y + (triggerNode.height() / 2));
            }
            // close eventually open field popup, see #42478
            docModel.getFieldManager().destroyFieldPopup();

            //Bug 48159
            if (_.browser.Firefox && isGlobalTrigger) { event.target = triggerNode; }

            // trigger our own contextmennu event on the found trigger node
            triggerNode.trigger(new $.Event('documents:contextmenu', { sourceEvent: event }));

            return false;
        }

        /**
         * Initialization after construction.
         */
        function initHandler() {

            // deferred and debounced scroll handler, registering insertText operations immediately
            var scrollToSelectionDebounced = self.createDebouncedMethod('TextView.scrollToSelectionDebounced', registerInsertText, scrollToSelection, { delay: 50, maxDelay: 500 });

            // register the global event handler for 'contextmenu' event
            if (Utils.COMPACT_DEVICE) {
                app.registerGlobalEventHandler($('body'), 'taphold', globalContextMenuHandler);
                app.registerGlobalEventHandler($('body'), 'contextmenu', function (evt) {
                    if (evt && /^(INPUT|TEXTAREA)/.test(evt.target.tagName)) { return true; }
                    if (evt) { evt.preventDefault(); }
                    return false;
                });
            } else {
                app.registerGlobalEventHandler($('body'), 'contextmenu', globalContextMenuHandler);
            }

            // to save the position from 'touchstart', we have to register some global eventhandler
            // we need this position to open the contextmenu on the correct location
            if (Utils.IOS || _.browser.Android) {
                app.registerGlobalEventHandler($('body'), 'touchstart', function (event) {
                    if (_.isUndefined(event)) { return false; }
                    touchX = event.originalEvent.changedTouches[0].pageX;
                    touchY = event.originalEvent.changedTouches[0].pageY;
                });
                app.registerGlobalEventHandler($('body'), 'touchend', function (event) {
                    if (_.isUndefined(event)) { return false; }
                    touchX = touchY = null;
                });
            }

            // create the additional view panes
            if (!Utils.SMALL_DEVICE) {
                self.addPane(rulerPane = new RulerPane(self));
            }

            // initialize other instance fields
            appPaneNode = self.getAppPaneNode();
            contentRootNode = self.getContentRootNode();

            // the page root node
            pageNode = docModel.getNode();

            // insert the editor root node into the application pane
            self.insertContentNode(pageNode);

            //page content node, used for zooming on small devices
            pageContentNode = DOM.getPageContentNode(pageNode);

            // handle scroll position when hiding/showing the entire application
            self.listenTo(app.getWindow(), {
                beforehide: function () {
                    saveScrollPosition();
                    appPaneNode.css('opacity', 0);
                },
                show: function () {
                    restoreScrollPosition();
                    // restore visibility deferred, needed to restore IE scroll position first
                    self.executeDelayed(function () { appPaneNode.css('opacity', 1); }, 'TextView.initHandler');
                }
            });

            // disable drag&drop for the content root node to prevent dropping images which are
            // loaded by the browser
            self.listenTo(self.getContentRootNode(), 'drop dragstart dragenter dragexit dragover dragleave', false);

            // Registering scroll handler
            self.listenTo(self.getContentRootNode(), 'scroll', function () {
                // handler for updating change track markers in side bar after scrolling
                // -> the handler contains only functionality, if the document contains change tracks
                docModel.updateChangeTracksDebouncedScroll();
            });

            // handle selection change events
            self.listenTo(docModel, 'selection', function (event, selection, options) {
                // disable internal browser edit handles (browser may re-enable them at any time)
                self.disableBrowserEditControls();
                // scroll to text cursor (but not, if only a remote selection was updated (39904))
                if (!Utils.getBooleanOption(options, 'updatingRemoteSelection', false)) { scrollToSelectionDebounced(options); }

                var userData = {
                    selections: selection.getRemoteSelectionsObject(),
                    target: docModel.getActiveTarget() || undefined
                };
                if (docModel.isHeaderFooterEditState()) {
                    userData.targetIndex = docModel.getPageLayout().getTargetsIndexInDocument(docModel.getCurrentRootNode(), docModel.getActiveTarget());
                }
                // send user data to server
                app.updateUserData(userData);
                // manage change track popups, but do not update, if 'keepChangeTrackPopup' is set to true
                if (!Utils.getBooleanOption(options, 'keepChangeTrackPopup', false)) {
                    handleChangeTrackPopup(options);
                }
                // exit from crop image mode, if currently active
                if (selection.isCropMode()) { docModel.exitCropMode(); }
            });

            // process page break event triggered by the model
            self.listenTo(docModel, 'pageBreak:before', function () {
                // scrolled downwards -> restore scroll position after inserting page breaks
                if (contentRootNode.scrollTop() > 0) { setFirstVisibleParagraph(); }
            });

            // process page break event triggered by the model
            self.listenTo(docModel, 'pageBreak:after', function () {
                var selection = docModel.getSelection();
                if (!selection.isUndefinedSelection() && !selection.isTopPositionSelected()) {
                    // scrolling to an existing selection made by the user.
                    // 'topVisibleParagraph' should be used also in this case (user makes selection and then scrolls away
                    // without making further selection), but scrolling to 'topVisibleParagraph' does not work reliable,
                    // if there is a selection set.
                    scrollToSelection();
                } else if (topVisibleParagraph) {
                    // scrolling to the scroll position set by the user (if it was saved in 'pageBreak:before')
                    self.scrollToChildNode(topVisibleParagraph, { forceToTop: true });
                }
                topVisibleParagraph = null;
            });

            // store the values of page width and height with paddings, after loading is finished
            pageNodeWidth = pageNode.outerWidth();
            pageNodeHeight = pageNode.outerHeight();
            if (Utils.SMALL_DEVICE) {
                contentRootNode.addClass('draft-mode');
            }

            if (Utils.SOFTKEYBORD_TOGGLE_MODE) {
                // close keyboard to leave edit mode
                self.listenTo(docModel, 'selection:possibleKeyboardClose', function (e, options) {

                    var forceClose = Utils.getBooleanOption(options, 'forceClose', false);
                    //if a pop-up is currently open in the UI, ignore the context menu
                    var popupOpen = $('.io-ox-office-main.popup-container:not(.context-menu)').length > 0;

                    if ((!popupOpen && !self.isSoftkeyboardEditModeOff()) || forceClose) {
                        closeSoftKeyboard();
                    }
                });

                //add button to open the softKeyboard
                var openKeyboardButton = $('<div class = "soft-keyboard-button" > <i class="fa fa-angle-up" style = "display: block; color: white; height: 9px; margin-top: 13%;" aria-hidden="true"></i> <i class="fa fa-keyboard-o" style = " color: white; font-size:28px; " aria-hidden="true"></i> </div>').on('touchstart', function (e) { e.preventDefault(); openSoftKeyboard(); });
                app.getWindowNode().addClass('soft-keyboard-off');
                app.getWindowNode().append(openKeyboardButton);

                // fired when a the menu in web core is closed -> when the top blue toolbar burger menu is closed
                // important to restore state after the global toolbar menu (mail, portal...) was opened
                // important: it must not listen when the app is hidden
                app.registerGlobalEventHandler($(document), 'hide.bs.dropdown', function () {
                    closeSoftKeyboard();
                });
            }

            // initialize change track popup
            initChangeTrackPopup();
            // initialize field editing popup
            initFieldFormatPopup();
        }

        /**
         * Additional debug initialization after construction.
         */
        function initDebugHandler(operationsPane, clipboardPane) {

            var // pending operations
                pendingOperations = [],
                // delayed recording operations active
                delayedRecordingActive = false,
                // checking the operation length, so that operations removed from the
                // 'optimizeOperationHandler' will not collected twice in the recorder.
                oplCounter = 1;

            // collect operations
            function collectOperations(event, operations, external) {
                if (docModel.isRecordingOperations()) {
                    pendingOperations = pendingOperations.concat(_(operations).map(function (operation) {
                        return { operation: operation, external: external };
                    }));

                    if (!delayedRecordingActive) {
                        recordOperations();
                    }
                }
            }

            // record operations
            function recordOperations() {

                if (delayedRecordingActive) { return; }

                delayedRecordingActive = true;
                self.executeDelayed(function () {
                    var // full JSON string
                        fullJSONString = replayOperations.getFieldValue(),
                        // json part string to be appended
                        jsonPartString = '',
                        // current part to be processed
                        operations = pendingOperations.splice(0, 20),
                        // first entry
                        firstEntry = (fullJSONString.length === 0);

                    _(operations).each(function (entry) {

                        // skipping operations, that were 'optimized' in 'sendActions' in the
                        // optimizeOperationHandler. Otherwise 'merged operations' are still
                        // followed by the operations, that will not be sent to
                        // the server, because they were removed by the 'optimizeOperationHandler'.
                        var skipOperation = false,
                            opl = ('opl' in entry.operation) ? entry.operation.opl : '';

                        // checking if this operation was merged by the 'optimizeOperationHandler'. This can
                        // only happen for internal operations (task 29601)
                        if (!entry.external) {
                            if (_.isNumber(opl) && (opl > 1)) {
                                oplCounter = opl;
                            } else if (oplCounter > 1) {
                                oplCounter--;
                                skipOperation = true;
                            }
                        }

                        if (!skipOperation) {
                            if (!firstEntry) {
                                jsonPartString += ',';
                            }
                            jsonPartString += JSON.stringify(entry.operation);
                            firstEntry = false;
                        }
                    });

                    if (fullJSONString.length >= 2) {
                        if (fullJSONString.charAt(0) === '[') {
                            fullJSONString = fullJSONString.substr(1, fullJSONString.length - 2);
                        }
                    }
                    fullJSONString = '[' + fullJSONString + jsonPartString + ']';
                    replayOperations.setValue(fullJSONString);

                    delayedRecordingActive = false;
                    if (pendingOperations.length > 0) {
                        recordOperations();
                    }
                }, 'TextView.initDebugHandler.recordOperations', 50);
            }

            // create the output nodes in the operations pane
            operationsPane
                .addDebugInfoHeader('selection', 'Current Selection')
                .addDebugInfoNode('selection', 'type', 'Type of the selection')
                .addDebugInfoNode('selection', 'start', 'Start position')
                .addDebugInfoNode('selection', 'end', 'End position')
                .addDebugInfoNode('selection', 'target', 'Target position')
                .addDebugInfoNode('selection', 'dir', 'Direction')
                .addDebugInfoNode('selection', 'attrs1', 'Explicit attributes of cursor position')
                .addDebugInfoNode('selection', 'attrs2', 'Explicit attributes (parent level)')
                .addDebugInfoNode('selection', 'style', 'Paragraph style');

            // log all selection events, and new formatting after operations
            self.listenTo(docModel, 'selection operations:success', function () {

                function debugAttrs(position, ignoreDepth, target) {
                    if (!operationsPane.isVisible()) { return; }

                    var markup = '';

                    position = _.initial(position, ignoreDepth);
                    var element = (position && position.length) ? Position.getDOMPosition(docModel.getCurrentRootNode(), position, true) : null;
                    if (element && element.node) {
                        var attrs = AttributeUtils.getExplicitAttributes(element.node);
                        markup = '<span ' + AttributesToolTip.createAttributeMarkup(attrs) + '>' + Utils.escapeHTML(Utils.stringifyForDebug(attrs)) + '</span>';
                    }

                    operationsPane.setDebugInfoMarkup('selection', target, markup);
                }

                function debugStyle(family, target) {
                    if (!operationsPane.isVisible()) { return; }

                    var markup = '';

                    var styleId = docModel.getAttributes(family).styleId;
                    if (styleId) {
                        var attrs = docModel.getStyleCollection(family).getStyleAttributeSet(styleId);
                        markup = '<span>family=' + family + '</span> <span ' + AttributesToolTip.createAttributeMarkup(attrs) + '>id=' + Utils.escapeHTML(styleId) + '</span>';
                    }

                    operationsPane.setDebugInfoMarkup('selection', target, markup);
                }

                var selection = docModel.getSelection();
                operationsPane
                    .setDebugInfoText('selection', 'type', selection.getSelectionType())
                    .setDebugInfoText('selection', 'start', selection.getStartPosition().join(', '))
                    .setDebugInfoText('selection', 'end', selection.getEndPosition().join(', '))
                    .setDebugInfoText('selection', 'target', selection.getRootNodeTarget())
                    .setDebugInfoText('selection', 'dir', selection.getDirection());

                debugAttrs(selection.getStartPosition(), 0, 'attrs1');
                debugAttrs(selection.getStartPosition(), 1, 'attrs2');
                debugStyle('paragraph', 'style');
            });

            // operations to replay
            replayOperations = new TextField(self, { tooltip: _.noI18n('Paste operations here'), select: true });

            // operations to replay
            replayOperationsOSN = new TextField(self, { tooltip: _.noI18n('Define OSN for last executed operation (optional)'), width: 50 });

            // record operations after importing the document
            self.waitForImport(function () {
                self.listenTo(docModel, 'operations:after', collectOperations);
            });

            // process debug events triggered by the model
            self.listenTo(docModel, 'debug:clipboard', function (event, content) {
                clipboardPane.setClipboardContent(content);
            });
        }

        /**
         * Handler function for the 'refresh:layout' event.
         */
        function refreshLayoutHandler() {

            var // the app content node
                appContentRootNode = self.getContentRootNode(),
                // the page attributes contain the width of the page in hmm
                pageAttributeWidth = docModel.getPageLayout().getPageAttribute('width'),
                // whether the page padding need to be reduced
                reducePagePadding = (Utils.COMPACT_DEVICE && pageAttributeWidth > Utils.convertLengthToHmm(appContentRootNode.width(), 'px')),
                // the remote selection object
                remoteSelection = docModel.getRemoteSelection(),
                // temporary page node width value
                tempPageNodeWidth = pageNode.outerWidth(),
                // whether the applications width has changed
                // -> the value for appWidthChanged must be true, if the width of the page node was modified
                // or if the class 'small-device' was added or removed (41160)
                appWidthChanged = (pageNodeWidth !== tempPageNodeWidth) || (appContentRootNode.hasClass('small-device') !== reducePagePadding);

            // check, if there is sufficient space for left and right padding on the page
            appContentRootNode.toggleClass('small-device', reducePagePadding);

            // fix for #33829 - if padding is reduced, dependent variables have to be updated
            pageNodeWidth = tempPageNodeWidth;
            pageNodeHeight = pageNode.outerHeight();

            // rotating device screen on tablets needs margin recalculated, except for < 7" devices (there are no margins and page breaks)
            // also changing page properties from user dialog, needs page layout repaint
            if (appWidthChanged && self.isImportFinished()) {
                if (!Utils.SMALL_DEVICE) {
                    if (docModel.isHeaderFooterEditState()) {
                        docModel.getPageLayout().leaveHeaderFooterEditMode();
                        docModel.getSelection().setNewRootNode(pageNode); // restore original rootNode
                        docModel.getSelection().setTextSelection(docModel.getSelection().getFirstDocumentPosition());
                    }
                    // INFO: calling insertPageBreaks from model is absolutely necessary!
                    // It is debounced method in model, and direct method in pageLayout.
                    // Debounced method has little start delay, but direct model can be interupted by calls from other places,
                    // which can lead to unpredicted page sizes!
                    docModel.insertPageBreaks();
                    self.recalculateDocumentMargin({ keepScrollPosition: true });
                } else {
                    docModel.trigger('update:absoluteElements');   // updating comments and absolutely positioned drawings (41160)
                }
                if (Utils.COMPACT_DEVICE) {
                    if (reducePagePadding) {
                        self.executeDelayed(function () { self.setZoomType('width'); }, 'TextView.initGuiHandler');
                    } else {
                        self.setZoomType(100);
                    }
                }
            }

            // updating the change track side bar, if necessary
            docModel.updateChangeTracksDebounced();

            if (zoomType === 'width') {
                self.executeDelayed(function () { self.setZoomType('width'); }, 'TextView.initGuiHandler');
            }

            // clean and redraw remote selections
            remoteSelection.renderCollaborativeSelections();
        }

        /**
         * Initialization after importing the document. Creates all tool boxes
         * in the side pane and overlay pane. Needed to be executed after
         * import, to be able to hide specific GUI elements depending on the
         * file type.
         *
         * @param {String} tabId
         *  The id of the tool bar tab that is activated. Or the identifier for generating the tool bar
         *  tabs ('toolbartabs') or for the view menu in the top bar ('viewmenu').
         *
         * @param {CompoundButton} viewMenuGroup
         *  The 'View' drop-down menu group.
         */
        function initGuiHandler(tabId, viewMenuGroup) {

            var insertToolBar           = null,
                tableToolBar            = null,
                tableBorderToolBar      = null,
                drawingToolBar          = null,
                drawingAnchorToolBar    = null,
                drawingFillToolBar      = null,
                drawingCropToolBar      = null,
                drawingLineEndsToolBar  = null,
                drawingLineToolBar      = null,
                reviewToolBar           = null,
                changeTrackingToolBar   = null,
                commentToolBar          = null,

                btnInsertTextframe      = null,
                btnInsertComment        = null,

                pickTableSize           = null,
                pickLanguage            = null,

                btnInsertImage          = null,
                btnInsertHyperlink      = null,
                btnInsertRow            = null,
                btnInsertColumn         = null,
                btnDeleteRow            = null,
                btnDeleteColumn         = null,

                btnDeleteDrawing        = null,
                btnSpelling             = null,
                btnInsertCommentReview  = null,
                btnDeleteAllComments    = null,
                btnSelectPrevComment    = null,
                btnSelectNextComment    = null;

            // helper function to generate comment specific buttons (if required)
            function prepareCommentButtons() {
                btnInsertCommentReview = new Button(self, { icon: 'docs-comment-add', tooltip: /*#. insert a comment into the text */ gt('Insert a comment') });
                btnDeleteAllComments   = new Button(self, { icon: 'docs-comment-remove', label: gt('Delete all'), tooltip: /*#. delete all comments in the text */ gt('Delete all comments'), smallerVersion: { hideLabel: true } });
                btnSelectPrevComment   = new Button(self, { label: /*#. select the previous comment */ gt('Previous'), icon: 'docs-comment-back', tooltip: gt('Select the previous comment in the document'), smallerVersion: { hideLabel: true } });
                btnSelectNextComment   = new Button(self, { label: /*#. select the next comment */ gt('Next'), icon: 'docs-comment-next', tooltip: gt('Select the next comment in the document'), smallerVersion: { hideLabel: true } });
            }

            switch (tabId) {

                case 'toolbartabs':
                    // TABS: prepare all tabs (for normal or combined panes)
                    if (!Config.COMBINED_TOOL_PANES) {
                        self.createToolBarTab('format',  { label: Labels.FORMAT_HEADER_LABEL, visibleKey: 'document/editable/text',       priority: 2 });
                        self.createToolBarTab('insert',  { label: Labels.INSERT_HEADER_LABEL, visibleKey: 'document/editable/text',       priority: 2 });
                        self.createToolBarTab('table',   { label: Labels.TABLE_HEADER_LABEL,  visibleKey: 'document/editable/table',      priority: 1  });
                        self.createToolBarTab('drawing', { labelKey: 'drawing/type/label',    visibleKey: 'document/editable/anydrawing', priority: 1 });
                        self.createToolBarTab('review',  { label: Labels.REVIEW_HEADER_LABEL, visibleKey: 'app/valid',                    priority: 20, sticky: true });

                    } else {
                        self.createToolBarTab('format',    { label: Labels.FONT_HEADER_LABEL,   visibleKey: 'document/editable/text',       priority: 2 });
                        self.createToolBarTab('paragraph', { label: Labels.PARAGRAPH_LABEL,     visibleKey: 'document/editable/text',       priority: 2 });
                        self.createToolBarTab('insert',    { label: Labels.INSERT_HEADER_LABEL, visibleKey: 'document/editable/text',       priority: 2 });
                        self.createToolBarTab('table',     { label: Labels.TABLE_HEADER_LABEL,  visibleKey: 'document/editable/table',      priority: 1 });
                        self.createToolBarTab('drawing',   { labelKey: 'drawing/type/label',    visibleKey: 'document/editable/anydrawing', priority: 1 });
                        if (Config.SPELLING_ENABLED) {
                            self.createToolBarTab('review',    { label: Labels.REVIEW_HEADER_LABEL, visibleKey: 'document/editable',        priority: 20 });
                        }
                        if (!app.isODF()) {
                            self.createToolBarTab('changetracking', { label: Labels.CHANGETRACKING_HEADER_LABEL, priority: 20, sticky: true });
                        }
                        self.createToolBarTab('comment', { label: gt.pgettext('menu-title', 'Comments'), priority: 20 });
                    }
                    break;

                case 'viewmenu':
                    // the 'View' drop-down menu
                    viewMenuGroup
                        .addSectionLabel(Labels.ZOOM_LABEL)
                        .addGroup('view/zoom/dec',  new Button(self, Labels.ZOOMOUT_BUTTON_OPTIONS), { inline: true, sticky: true })
                        .addGroup('view/zoom/inc',  new Button(self, Labels.ZOOMIN_BUTTON_OPTIONS), { inline: true, sticky: true })
                        .addGroup('view/zoom',      new Controls.PercentageLabel(self), { inline: true })
                        .addGroup('view/zoom/type', new Button(self, { label: Labels.ZOOM_SCREEN_WIDTH_LABEL, value: 'width' }))
                        .addSectionLabel(Labels.OPTIONS_LABEL)
                        // note: we do not set aria role to 'menuitemcheckbox' or 'button' due to Safari just working correctly with 'checkox'. CheckBox constructor defaults aria role to 'checkbox'
                        .addGroup('view/toolbars/show',     new CheckBox(self, Labels.SHOW_TOOLBARS_CHECKBOX_OPTIONS))
                        .addGroup('view/rulerpane/show',    new CheckBox(self, { label: /*#. check box label: show/hide ruler */ gt('Show ruler'), tooltip: gt('Toggle ruler next to document') }))
                        .addGroup('document/users',         new CheckBox(self, Labels.SHOW_COLLABORATORS_CHECKBOX_OPTIONS));

                    // event handling
                    self.on('refresh:layout', refreshLayoutHandler);
                    break;

                case 'format':
                    self.addToolBar('format', new ToolBars.FontFamilyToolBar(self), { priority: 1 });
                    self.addToolBar('format', new ToolBars.FontStyleToolBar(self),  { priority: 2 });
                    self.addToolBar('format', new ToolBars.FontColorToolBar(self),  { priority: 3 });
                    self.addToolBar('format', new ToolBars.FormatPainterToolBar(self), { priority: 3 });

                    if (!Config.COMBINED_TOOL_PANES) {
                        self.addToolBar('format', new ToolBars.ParagraphAlignmentToolBar(self),  { priority: 4 });
                        self.addToolBar('format', new ToolBars.ParagraphFillBorderToolBar(self), { priority: 4, hideable: true });
                        self.addToolBar('format', new ToolBars.ParagraphStyleToolBar(self),      { priority: 5, hideable: true });
                        self.addToolBar('format', new ToolBars.ListStyleToolBar(self),           { priority: 5 });
                    }

                    break;

                case 'paragraph':
                    if (Config.COMBINED_TOOL_PANES) {
                        self.addToolBar('paragraph', new ToolBars.ParagraphAlignmentToolBar(self),  { priority: 1 });
                        self.addToolBar('paragraph', new ToolBars.ParagraphFillBorderToolBar(self), { priority: 2 });
                        self.addToolBar('paragraph', new ToolBars.ListStyleToolBar(self),           { priority: 3 });
                    }
                    break;

                case 'insert':
                    insertToolBar = self.createToolBar('insert');

                    // controls
                    pickTableSize = new Controls.TableSizePicker(self, { label: /*#. a table in a text document */ gt.pgettext('text-doc', 'Table'), tooltip: /*#. insert a table in a text document */ gt.pgettext('text-doc', 'Insert a table'), maxCols: Config.MAX_TABLE_COLUMNS, maxRows: Config.MAX_TABLE_ROWS, smallerVersion: { css: { width: 50 } } });
                    btnInsertImage = new Controls.ImagePicker(self);
                    btnInsertHyperlink = new Button(self, Labels.HYPERLINK_BUTTON_OPTIONS);

                    insertToolBar
                        .addGroup('table/insert', pickTableSize, { visibleKey: 'table/insert/available' })
                        .addGroup('image/insert/dialog', btnInsertImage);

                    if (!Config.COMBINED_TOOL_PANES) {
                        btnInsertTextframe = new Controls.InsertTextFrameButton(self, { toggle: true });

                        insertToolBar
                            .addGroup('textframe/insert', btnInsertTextframe);
                    }

                    insertToolBar
                        .addGroup('shape/insert', new Controls.ShapeTypePicker(self));

                    btnInsertComment = new Button(self, _.extend({ icon: 'docs-comment-add' }, Labels.INSERT_COMMENT_OPTIONS, { smallerVersion: { hideLabel: true } }));
                    insertToolBar
                        .addGroup('comment/insert', btnInsertComment)
                        .addSeparator();

                    insertToolBar.addGroup('character/hyperlink/dialog', btnInsertHyperlink);

                    if (!Config.COMBINED_TOOL_PANES) {
                        var btnInsertTab         = new Button(self, { icon: 'docs-insert-tab', label: gt('Tab stop'), tooltip: /*#. insert a horizontal tab stop into the text */ gt('Insert tab stop'), smallerVersion: { hideLabel: true } });
                        var btnInsertBreak       = new Button(self, { icon: 'docs-insert-linebreak', label: gt('Line break'),  tooltip: /*#. insert a manual line break into the text */ gt('Insert line break'), smallerVersion: { hideLabel: true } });
                        var btnInsertPagebreak   = new Button(self, { icon: 'docs-page-break', label: gt('Page break'),  tooltip: /*#. insert a manual page break into the text */ gt('Insert page break'), smallerVersion: { hideLabel: true } });
                        var insertField          = new Controls.InsertFieldPicker(self);
                        var insertToc            = new Controls.TocPicker(self);

                        insertToolBar
                            .addGroup('character/insert/tab',   btnInsertTab)
                            .addGroup('character/insert/break', btnInsertBreak)
                            .addGroup('character/insert/pagebreak', btnInsertPagebreak)
                            .addGroup('document/insert/headerfooter', new Controls.HeaderFooterPicker(self))
                            .addGroup('document/insertfield', insertField)
                            .addGroup('document/inserttoc', insertToc);
                    }

                    break;

                case 'table':
                    tableToolBar = self.createToolBar('table');
                    if (!Config.COMBINED_TOOL_PANES) {
                        tableBorderToolBar = self.createToolBar('table', { priority: 1, hideable: true, classes: 'noSeparator' });
                        self.addToolBar('table', new ToolBars.TableStyleToolBar(self), { priority: 2, hideable: true, visibleKey: 'document/ooxml' });
                    }

                    // controls
                    btnInsertRow    = new Button(self, { icon: 'docs-table-insert-row', tooltip: gt('Insert row') });
                    btnInsertColumn = new Button(self, { icon: 'docs-table-insert-column', tooltip: gt('Insert column') });
                    btnDeleteRow    = new Button(self, { icon: 'docs-table-delete-row', tooltip: gt('Delete selected rows') });
                    btnDeleteColumn = new Button(self, { icon: 'docs-table-delete-column', tooltip: gt('Delete selected columns') });

                    tableToolBar
                        .addGroup('table/insert/row', btnInsertRow)
                        .addGroup('table/delete/row', btnDeleteRow)
                        .addGap()
                        .addGroup('table/insert/column', btnInsertColumn)
                        .addGroup('table/delete/column', btnDeleteColumn);

                    if (!Config.COMBINED_TOOL_PANES) {

                        var btnTableSplit       = new Button(self, { icon: 'docs-table-split', tooltip: /*#. split current table */ gt('Split table') }),
                            pickFillColorTable  = new Controls.FillColorPicker(self, { icon: 'docs-table-fill-color', tooltip: gt('Cell fill color') }),
                            pickBorderModeTable = new Controls.BorderModePicker(self, { tooltip: Labels.CELL_BORDERS_LABEL, showInsideHor: true, showInsideVert: true }),
                            pickBorderWidth     = new Controls.BorderWidthPicker(self, { tooltip: gt('Cell border width') }),
                            alignmentGroup      = new Controls.TableAlignmentPicker(self);

                        tableToolBar
                            .addGap()
                            .addGroup('table/split', btnTableSplit)
                            .addSeparator()
                            .addGroup('table/alignment/menu', alignmentGroup)
                            .addSeparator()
                            .addGroup('table/fillcolor', pickFillColorTable)
                            .addGap()
                            .addGroup('table/cellborder', pickBorderModeTable);
                        tableBorderToolBar
                            .addGroup('table/borderwidth', pickBorderWidth);
                    }

                    break;

                case 'drawing':
                    drawingToolBar = self.createToolBar('drawing');

                    if (!Config.COMBINED_TOOL_PANES) {
                        drawingLineToolBar      = self.createToolBar('drawing', { visibleKey: 'drawing/shapes/support/line' });
                        drawingFillToolBar      = self.createToolBar('drawing', { visibleKey: 'drawing/shapes/support/fill', hideable: true });
                        drawingCropToolBar      = self.createToolBar('drawing', { visibleKey: 'drawing/singledrawingselection', hideable: true });
                        self.addToolBar('drawing', new ToolBars.FormatPainterToolBar(self));
                        drawingLineEndsToolBar  = self.createToolBar('drawing', { visibleKey: 'drawing/shapes/support/lineendings' });
                        drawingAnchorToolBar    = self.createToolBar('drawing', { visibleKey: 'document/editable/anydrawing' });
                    }

                    // controls
                    btnDeleteDrawing = new Button(self, Labels.DELETE_DRAWING_BUTTON_OPTIONS);

                    drawingToolBar.addGroup('shape/insert', new Controls.ShapeTypePicker(self), { visibleKey: 'drawing/type/shape' });
                    drawingToolBar.addGroup('drawing/delete', btnDeleteDrawing, { visibleKey: 'drawing/delete' });

                    if (!Config.COMBINED_TOOL_PANES) {

                        var pickAnchorage           = new Controls.AnchoragePicker(self),
                            drawingOrder            = new Controls.DrawingArrangementPicker(self),
                            pickDrawingPosition     = new Controls.DrawingPositionPicker(self),
                            cBoxToggleAutoFit       = new CheckBox(self, { label: gt('Autofit'), boxed: true, tooltip: gt('Turn automatic height resizing of text frames on or off') }),
                            pickArrowPresetStyle    = new Controls.ArrowPresetStylePicker(self),
                            pickBorderPresetStyle   = new Controls.BorderPresetStylePicker(self, { label: Labels.BORDER_STYLE_LABEL }),
                            pickLinePresetStyle     = new Controls.BorderPresetStylePicker(self, { label: Labels.LINE_STYLE_LABEL }),
                            pickBorderColor         = new Controls.BorderColorPicker(self, { label: gt('Border color'), smallerVersion: { hideLabel: true } }),
                            pickLineColor           = new Controls.BorderColorPicker(self, { label: gt('Line color'), smallerVersion: { hideLabel: true } }),
                            pickFillColorDrawing    = new Controls.FillColorPicker(self, { label: gt('Background color'),  icon: 'docs-drawing-fill-color', smallerVersion: { hideLabel: true } }),
                            vertAlignPicker         = new Controls.TextBoxAlignmentPicker(self, { label: Labels.TEXT_ALIGNMENT_IN_SHAPE_LABEL, tooltip: Labels.TEXT_ALIGNMENT_IN_SHAPE_TOOLTIP, smallerVersion: { hideLabel: true } });

                        drawingLineToolBar
                            .addGroup('drawing/border/style', pickBorderPresetStyle, { visibleKey: 'drawing/border/style/support' })
                            .addGroup('drawing/border/style', pickLinePresetStyle, { visibleKey: 'drawing/line/style/support' })
                            .addGroup('drawing/border/color', pickBorderColor, { visibleKey: 'drawing/border/style/support' })
                            .addGroup('drawing/border/color', pickLineColor, { visibleKey: 'drawing/line/style/support' });

                        drawingLineEndsToolBar
                            .addGroup('drawing/line/endings', pickArrowPresetStyle);

                        drawingFillToolBar
                            .addGroup('drawing/fill/color', pickFillColorDrawing, { visibleKey: 'drawing/fill/style/support' });

                        drawingAnchorToolBar
                            .addGap()
                            .addGroup('drawing/anchorTo', pickAnchorage)
                            .addGroup('drawing/position', pickDrawingPosition)
                            .addGroup('drawing/order', drawingOrder)
                            .addGroup('drawing/verticalalignment', vertAlignPicker)
                            .addGroup('drawing/textframeautofit', cBoxToggleAutoFit);

                        drawingCropToolBar
                            .addGroup('drawing/crop', new Controls.ImageCropPosition(self), { visibleKey: 'drawing/crop' });
                    }

                    break;

                case 'review':
                    if (!Config.COMBINED_TOOL_PANES) {
                        reviewToolBar = self.createToolBar('review', {
                            priority: 2,
                            shrinkTomenu: {
                                icon: 'docs-online-spelling',
                                splitKey: 'document/onlinespelling',
                                toggle: true,
                                tooltip: gt('Check spelling'),
                                //#. tooltip for a drop-down menu with functions for reviewing the document (spelling, change tracking)
                                caretTooltip: gt('More review actions')
                            }
                        });
                        changeTrackingToolBar = self.createToolBar('review');
                    } else {
                        reviewToolBar = self.createToolBar('review', { visibleKey: 'document/spelling/available' });
                    }
                    break;

                case 'changetracking':
                    if (Config.COMBINED_TOOL_PANES) {
                        changeTrackingToolBar = self.createToolBar('changetracking');
                    }
                    break;

                case 'comment':
                    if (Config.COMBINED_TOOL_PANES) {
                        commentToolBar = self.createToolBar('comment');
                    }
                    break;

                default:
            }

            // controls for the review toolbar

            if (Config.SPELLING_ENABLED) {
                if (reviewToolBar) {

                    pickLanguage = new Controls.LanguagePicker(self, { width: 200 });
                    btnSpelling = new Button(self, { icon: 'docs-online-spelling', tooltip: gt('Check spelling permanently'), toggle: true, dropDownVersion: { label: gt('Check spelling') } });

                    reviewToolBar
                        .addGroup('document/onlinespelling', btnSpelling)
                        .addSeparator()
                        .addGroup('character/language', pickLanguage);
                }
            }

            if (!app.isODF()) {

                if (changeTrackingToolBar) {

                    var cBoxToggleCT        = new CheckBox(self, _.extend({ boxed: true }, Labels.TRACK_CHANGES_OPTIONS)),
                        btnSelectPrevCT     = new Button(self, { label: /*#. change tracking: select the previous change */ gt('Previous'), icon: 'fa-chevron-left', tooltip: gt('Select the previous change in the document'), smallerVersion: { hideLabel: true } }),
                        btnSelectNextCT     = new Button(self, { label: /*#. change tracking: select the next change */ gt('Next'), icon: 'fa-chevron-right', tooltip: gt('Select the next change in the document'), smallerVersion: { hideLabel: true } });

                    if (!Config.COMBINED_TOOL_PANES) {

                        var btnAcceptMoveSelectCT   = new Button(self, Labels.ACCEPT_AND_NEXT_OPTIONS),
                            btnAcceptSelectCT       = new Button(self, Labels.ACCEPT_CURRENT_OPTIONS),
                            btnAcceptCT             = new Button(self, Labels.ACCEPT_ALL_OPTIONS),
                            btnRejectMoveSelectCT   = new Button(self, Labels.REJECT_AND_NEXT_OPTIONS),
                            btnRejectSelectCT       = new Button(self, Labels.REJECT_CURRENT_OPTIONS),
                            btnRejectCT             = new Button(self, Labels.REJECT_ALL_OPTIONS);

                        changeTrackingToolBar
                            .addGroup('toggleChangeTracking', cBoxToggleCT)
                            .addSeparator()
                            .addGroup('acceptMoveSelectedChangeTracking',       new Controls.CompoundSplitButton(self, { icon: 'fa-check', label: /*#. change tracking function: accept the current document change */ gt('Accept'), tooltip: gt('Accept the current change and select the next change'), smallerVersion: { hideLabel: true } })
                                .addGroup('acceptMoveSelectedChangeTracking',   btnAcceptMoveSelectCT)
                                .addGroup('acceptSelectedChangeTracking',       btnAcceptSelectCT)
                                .addGroup('acceptChangeTracking',               btnAcceptCT)
                            )
                            .addGroup('rejectMoveSelectedChangeTracking',       new Controls.CompoundSplitButton(self, { icon: 'fa-times', label: /*#. change tracking function: reject the current document change */ gt('Reject'), tooltip: gt('Reject the current change and select the next change'), smallerVersion: { hideLabel: true } })
                                .addGroup('rejectMoveSelectedChangeTracking',   btnRejectMoveSelectCT)
                                .addGroup('rejectSelectedChangeTracking',       btnRejectSelectCT)
                                .addGroup('rejectChangeTracking',               btnRejectCT)
                            )
                            .addGroup('selectPrevChangeTracking',               btnSelectPrevCT)
                            .addGroup('selectNextChangeTracking',               btnSelectNextCT);
                    } else {
                        changeTrackingToolBar
                            .addGroup('toggleChangeTracking',           cBoxToggleCT.clone({ label: /*#. change tracking: switch change tracking on/off */ gt('On') }))
                            .addSeparator()
                            .addGroup('acceptSelectedChangeTracking',   new Button(self, { icon: 'fa-check', label: gt('Accept'), tooltip: gt('Accept the current change'), smallerVersion: { hideLabel: true } }))
                            .addGroup('rejectSelectedChangeTracking',   new Button(self, { icon: 'fa-times', label: gt('Reject'), tooltip: gt('Reject the current change'), smallerVersion: { hideLabel: true } }))
                            .addGroup('selectPrevChangeTracking',       btnSelectPrevCT)
                            .addGroup('selectNextChangeTracking',       btnSelectNextCT);
                    }
                }
            }

            if (!Config.COMBINED_TOOL_PANES) {
                if (changeTrackingToolBar) {

                    prepareCommentButtons(); // preparing the buttons for comments

                    if (!app.isODF()) { changeTrackingToolBar.addSeparator(); } // one more separator, but not for ODF (56013)

                    changeTrackingToolBar
                        .addGroup('comment/insert', btnInsertCommentReview)
                        .addGap()
                        .addGroup('comment/displayModeParent', new Controls.CommentDisplayModePicker(self))
                        .addGap()
                        .addGroup('comment/prevComment', btnSelectPrevComment)
                        .addGroup('comment/nextComment', btnSelectNextComment)
                        .addGroup('comment/deleteAll', btnDeleteAllComments);
                }
            } else {
                if (commentToolBar) {

                    prepareCommentButtons(); // preparing the buttons for comments

                    commentToolBar
                        .addGroup('comment/insert', btnInsertCommentReview)
                        .addGap()
                        .addGroup('comment/displayModeParent', new Controls.CommentDisplayModePicker(self, { smallDevice: true }))
                        .addGap()
                        .addGroup('comment/prevComment', btnSelectPrevComment)
                        .addGroup('comment/nextComment', btnSelectNextComment)
                        .addGroup('comment/deleteAll', btnDeleteAllComments);
                }
            }

        }

        /**
         * Additional initialization of debug GUI after importing the document.
         *
         * @param {String} tabId
         *  The id of the tool bar tab that is activated. Or the identifier for generating the tool bar
         *  tabs ('toolbartabs') or for the view menu in the top bar ('viewmenu').
         *
         * @param {CompoundButton} viewMenuGroup
         *  The 'View' drop-down menu group.
         */
        function initDebugGuiHandler(tabId, viewMenuGroup) {

            switch (tabId) {
                case 'viewmenu':
                    viewMenuGroup
                        .addGroup('debug/pagebreaks/toggle', new CheckBox(self, { label: _.noI18n('Show page breaks') }))
                        .addGroup('debug/draftmode/toggle',  new CheckBox(self, { label: _.noI18n('View in draft mode') }))
                        .addGroup('debug/uselocalstorage',   new Button(self, { icon: 'fa-rocket', label: _.noI18n('Use local storage'), toggle: true }))
                        .addGroup('debug/useFastLoad',       new Button(self, { icon: 'fa-fighter-jet', label: _.noI18n('Use fast load'), toggle: true }));
                    break;
                case 'debug':
                    self.createToolBar('debug')
                        .addGroup('document/cut',            new Button(self, { icon: 'fa-cut',   tooltip: _.noI18n('Cut to clipboard') }))
                        .addGroup('document/copy',           new Button(self, { icon: 'fa-copy',  tooltip: _.noI18n('Copy to clipboard') }))
                        .addGroup('document/paste/internal', new Button(self, { icon: 'fa-paste', tooltip: _.noI18n('Paste from clipboard') }));

                    self.createToolBar('debug')
                        .addGroup('debug/recordoperations', new Button(self, { icon: 'fa-dot-circle-o', iconStyle: 'color:red;', toggle: true, tooltip: _.noI18n('Record operations') }))
                        .addGroup('debug/replayoperations', new Button(self, { icon: 'fa-play-circle-o', tooltip: _.noI18n('Replay operations') }))
                        .addGroup(null, replayOperations)
                        .addGroup(null, replayOperationsOSN);

                    self.createToolBar('debug')
                        .addGroup('debug/commentView', new Controls.CommentDisplayViewPicker(self))
                        //important for uitests!
                        .addGroup('paragraph/list/bullet', new Controls.BulletListStylePicker(self))
                        .addSeparator()
                        .addGroup('debug/deleteHeaderFooter', new Button(self, { icon: 'fa-eraser', tooltip: _.noI18n('Delete Header/Footer') }));
                    break;
                default:
            }
        }

        function calculateOptimalDocumentWidth() {
            var
                documentWidth = pageNode.outerWidth() + (pageNode.hasClass(DOM.COMMENTMARGIN_CLASS) ? pageNode.data(DOM.COMMENTMARGIN_CLASS) : 0),
                parentWidth = appPaneNode.width(),
                appContent = appPaneNode.find('.app-content'),
                outerDocumentMargin = 3 + Utils.getElementCssLength(appContent, 'margin-left') + Utils.getElementCssLength(appContent, 'margin-right'), // "3" because of the rounding issues
                optimalLevel = ((parentWidth * parentWidth) / (documentWidth * (parentWidth + Utils.SCROLLBAR_WIDTH + 4 + outerDocumentMargin))) * 100;

            return Math.floor(optimalLevel);
        }

        function setAdditiveMargin() {
            if (Utils.IOS) {
                //extra special code for softkeyboard behavior, we nee more space to scroll to the end of the textfile
                var dispHeight = Math.max(screen.height, screen.width);

                pageNode.css('margin-bottom', (Utils.getElementCssLength(pageNode, 'margin-bottom') + dispHeight / 3) + 'px');

                //workaround for clipped top of the first page (20)
                pageNode.css('margin-top', (Utils.getElementCssLength(pageNode, 'margin-top') + 20) + 'px');
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the position of the passed node in the entire scroll area of
         * the application pane.
         *
         * @param {HTMLElement|jQuery} node
         *  The DOM element whose dimensions will be calculated. Must be
         *  contained somewhere in the application pane. If this object is a
         *  jQuery collection, uses the first node it contains.
         *
         * @returns {Object}
         *  An object with numeric attributes representing the position and
         *  size of the passed node relative to the entire scroll area of the
         *  application pane. See method Utils.getChildNodePositionInNode() for
         *  details.
         */
        this.getChildNodePositionInNode = function (node) {
            return Utils.getChildNodePositionInNode(contentRootNode, node);
        };

        /**
         * Scrolls the application pane to make the passed node visible.
         *
         * @param {HTMLElement|jQuery} node
         *  The DOM element that will be made visible by scrolling the
         *  application pane. Must be contained somewhere in the application
         *  pane. If this object is a jQuery collection, uses the first node it
         *  contains.
         *
         * @param {Object} options
         *  @param {Boolean} [options.forceToTop=false]
         *   Whether the specified node shall be at the top border of the
         *   visible area of the scrolling node.
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.scrollToChildNode = function (node, options) {
            if (Utils.getBooleanOption(options, 'regardSoftkeyboard', false)) {
                //fix for Bug 39979 on Ipad text is sometimes under the keyboard
                if (!Utils.IOS || !Utils.isSoftKeyboardOpen()) { delete options.regardSoftkeyboard; }
            }
            Utils.scrollToChildNode(contentRootNode, node, _.extend(options || {}, { padding: 15 }));
            return this;
        };

        /**
         * Scrolls the application pane to make the passed document page
         * rectangle visible.
         *
         * @param {Object} pageRect
         *  The page rectangle that will be made visible by scrolling the
         *  application pane. Must provide the properties 'left', 'top',
         *  'width', and 'height' in pixels. The properties 'left' and 'top'
         *  are interpreted relatively to the entire document page.
         *
         *  @param {Object} options
         *   @param {Boolean} [options.regardSoftkeyboard=false]
         *   Whether the softkeyboard is considered in the calculation for the viewport.
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.scrollToPageRectangle = function (pageRect, options) {

            Utils.scrollToPageRectangle(contentRootNode, pageRect, { regardSoftkeyboard: Utils.getBooleanOption(options, 'regardSoftkeyboard', false), padding: 15 });
            return this;
        };

        /**
         * Provides the debug operations as string for the debugging feature
         * replay operations.
         *
         * @returns {String}
         *  The content of the debug operations field, or null if debug mode is
         *  not active.
         */
        this.getDebugOperationsContent = function () {
            var // the operations in the required json format
                json = null;

            if (replayOperations) {
                json = replayOperations.getFieldValue();

                try {
                    json = json.length ? JSON.parse(json) : null;
                    if (_.isObject(json) && json.operations) {
                        json = json.operations;
                    }
                    return json;
                } catch (e) {
                    // error handling outside this function
                }
            }
            return null;
        };

        /**
         * Setting the maximum OSN for the operations from the operation player, that will
         * be executed.
         *
         * @returns {Number|Null}
         *  The OSN of the last operation that will be executed from the operation player.
         */
        this.getDebugOperationsOSN = function () {

            // the operations in the required json format
            var number = null;

            if (replayOperationsOSN) {
                number = parseInt(replayOperationsOSN.getFieldValue(), 10);
                if (!Utils.isFiniteNumber(number)) {
                    number = null;
                    replayOperationsOSN.setValue('');
                }
            }

            return number;
        };

        /**
         * Returns the current zoom type.
         *
         * @returns {Number|String}
         *  The current zoom type, either as fixed percentage, or as one of the
         *  keywords 'width' (page width is aligned to width of the visible
         *  area), or 'page' (page size is aligned to visible area).
         */
        this.getZoomType = function () {
            return zoomType;
        };

        /**
         * Returns the current effective zoom factor in percent.
         *
         * @returns {Number}
         *  The current zoom factor in range [0, 1].
         */
        this.getZoomFactor = function () {
            return zoomFactor / 100;
        };

        /**
         * Returns the minimum zoom factor in percent.
         *
         * @returns {Number}
         *  The minimum zoom factor in percent.
         */
        this.getMinZoomFactor = function () {
            return ZOOM_FACTORS[0];
        };

        /**
         * Returns the maximum zoom factor in percent.
         *
         * @returns {Number}
         *  The maximum zoom factor in percent.
         */
        this.getMaxZoomFactor = function () {
            return _.last(ZOOM_FACTORS);
        };

        /**
         * Changes the current zoom settings.
         *
         * @param {Number|String} newZoomType
         *  The new zoom type. Either a fixed percentage, or one of the
         *  keywords 'width' (page width is aligned to width of the visible
         *  area), or 'page' (page size is aligned to visible area).
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.setZoomType = function (newZoomType) {
            var optimalLevel = calculateOptimalDocumentWidth();

            if (zoomType !== newZoomType && _.isNumber(newZoomType)) {
                if (zoomType < newZoomType) {
                    this.increaseZoomLevel(newZoomType);
                } else {
                    this.decreaseZoomLevel(newZoomType);
                }
            } else if (newZoomType === 'width') {
                if (Utils.SMALL_DEVICE || docModel.isDraftMode()) {
                    optimalLevel = 100;
                }
                if (Math.abs(zoomFactor - optimalLevel) > 1.1) {
                    this.increaseZoomLevel(optimalLevel);
                }
            }
            zoomType = newZoomType;
            return this;
        };

        /**
         * Switches to 'fixed' zoom mode, and decreases the current zoom level
         * by one according to the current effective zoom factor, and updates
         * the view.
         *
         * @param {Number} newZoomFactor
         *  The new zoom type. If passed as argument, use this zoom factor,
         *  otherwise use one of predefined
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.decreaseZoomLevel = function (newZoomFactor) {

            var // find last entry in ZOOM_FACTORS with a factor less than current zoom
                prevZoomFactor = Utils.findLast(ZOOM_FACTORS, function (factor) { return factor < zoomFactor; }),
                zoomInsertion = newZoomFactor || 0,
                // the parent node of the page
                appContentRoot = $(appPaneNode).find('.app-content');

            if (zoomInsertion === 0) {
                zoomInsertion = _.isNumber(prevZoomFactor) ? prevZoomFactor : this.getMinZoomFactor();
            }
            zoomFactor = zoomInsertion;

            if (Utils.SMALL_DEVICE || docModel.isDraftMode()) {
                Utils.setCssAttributeWithPrefixes(pageContentNode, 'transform', 'scale(' +  (zoomInsertion / 100) + ')');
                Utils.setCssAttributeWithPrefixes(pageContentNode, 'transform-origin', 'top left');
                if (zoomInsertion < 100 && docModel.isDraftMode()) {
                    pageContentNode.css({ width: '100%' });
                } else {
                    pageContentNode.css({ width: ((100 / zoomInsertion) * 100) + '%', backgroundColor: '#fff' });
                }
                zoomType = zoomFactor;
            } else {

                Utils.setCssAttributeWithPrefixes(pageNode, 'transform', 'scale(' +  (zoomInsertion / 100) + ')');
                Utils.setCssAttributeWithPrefixes(pageNode, 'transform-origin', 'top');

                if (_.browser.Firefox) {
                    pageNode.css({ opacity: (zoomInsertion > 100) ? 0.99 : 1 });
                }

                self.recalculateDocumentMargin();

                if (zoomInsertion <= 99) {
                    appContentRoot.css('overflow', 'hidden');
                } else {
                    appContentRoot.css({ overflow: '', paddingLeft: '', paddingRight: '' });
                }

                // Use zoom as soon as all browsers support it.
                // pageNode.css({
                //    'zoom': zoomInsertion / 100
                // });

                zoomType = zoomFactor;
                scrollToSelection();
            }
            self.trigger('change:zoom');
            return this;
        };

        /**
         * Switches to 'fixed' zoom mode, and increases the current zoom level
         * by one according to the current effective zoom factor, and updates
         * the view.
         *
         * @param {Number} newZoomFactor
         *  The new zoom type. If passed as argument, use this zoom factor,
         *  otherwise use one of predefined
         *
         * @param {Boolean} option.clean
         *  Switching between draft and normal mode requires zoom cleanup
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.increaseZoomLevel = function (newZoomFactor, clean) {

            var // find first entry in ZOOM_FACTORS with a factor greater than current zoom
                nextZoomFactor = _(ZOOM_FACTORS).find(function (factor) { return factor > zoomFactor; }),
                zoomInsertion = newZoomFactor || 0,
                appContentRoot = $(appPaneNode).find('.app-content');

            if (zoomInsertion === 0) {
                zoomInsertion = _.isNumber(nextZoomFactor) ? nextZoomFactor : this.getMaxZoomFactor();
            }
            zoomFactor = zoomInsertion;

            if ((Utils.SMALL_DEVICE || docModel.isDraftMode()) && !clean) {
                Utils.setCssAttributeWithPrefixes(pageContentNode, 'transform', 'scale(' + (zoomInsertion / 100) + ')');
                Utils.setCssAttributeWithPrefixes(pageContentNode, 'transform-origin', 'top left');
                pageContentNode.css({ width: ((100 / zoomInsertion) * 100) + '%', backgroundColor: '#fff' });
                zoomType = zoomFactor;
            } else {

                Utils.setCssAttributeWithPrefixes(pageNode, 'transform', 'scale(' +  (zoomInsertion / 100) + ')');
                Utils.setCssAttributeWithPrefixes(pageNode, 'transform-origin', 'top');

                if (_.browser.Firefox) {
                    pageNode.css({ opacity: (zoomInsertion > 100) ? 0.99 : 1 });
                }

                self.recalculateDocumentMargin();

                if (zoomInsertion <= 99) {
                    appContentRoot.css('overflow', 'hidden');
                } else {
                    appContentRoot.css({ overflow: '', paddingLeft: '', paddingRight: '' });
                }

                // Use zoom as soon as all browsers support it.
                //  pageNode.css({
                //      'zoom': zoomInsertion / 100
                //  });

                zoomType = zoomFactor;
                scrollToSelection();

            }
            self.trigger('change:zoom');
            return this;
        };

        /**
         * Creates necessary margin values when using transform scale css property
         *
         */
        this.recalculateDocumentMargin = function (options) {
            var
                marginAddition = zoomFactor < 99 ? 3 : 0,
                zoomFactorScaled = zoomFactor / 100,
                recalculatedMargin = (pageNodeWidth * (zoomFactorScaled - 1)) / 2 + marginAddition,
                // whether the (vertical) scroll position needs to be restored
                keepScrollPosition = Utils.getBooleanOption(options, 'keepScrollPosition', false),
                // an additional right margin used for comments that might be set at the page node
                commentMargin = pageNode.hasClass(DOM.COMMENTMARGIN_CLASS) ? pageNode.data(DOM.COMMENTMARGIN_CLASS) * zoomFactorScaled : 0;

            if (zoomFactor !== 100) {
                // #34735 page width and height values have to be updated
                pageNodeWidth = pageNode.outerWidth();
                pageNodeHeight = pageNode.outerHeight();
            }

            if (_.browser.Chrome && zoomFactor > 100) {
                pageNode.css({
                    margin:  marginAddition + 'px ' + (recalculatedMargin + commentMargin) + 'px ' + (30 / zoomFactorScaled) + 'px ' + recalculatedMargin + 'px'
                });
            } else {
                pageNode.css({
                    margin:  marginAddition + 'px ' + (recalculatedMargin + commentMargin) + 'px ' + (pageNodeHeight * (zoomFactorScaled - 1) + marginAddition) + 'px ' + recalculatedMargin + 'px'
                });
            }

            setAdditiveMargin();

            // in the end scroll to cursor position/selection (but not after event 'refresh:layout', that uses the option 'keepScrollPosition')
            if (!keepScrollPosition) { scrollToSelection(); }
        };

        /**
         * Rejects an attempt to edit the text document (e.g. due to reaching
         * the table size limits).
         *
         * @param {String} cause
         *  The cause of the rejection.
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.rejectEditTextAttempt = function (cause) {

            switch (cause) {
                case 'tablesizerow':
                    this.yell({ type: 'info', message: gt('The table reached its maximum size. You cannot insert further rows.') });
                    break;

                case 'tablesizecolumn':
                    this.yell({ type: 'info', message: gt('The table reached its maximum size. You cannot insert further columns.') });
                    break;
            }

            return this;
        };

        // returns the change track popup object
        this.getChangeTrackPopup = function () {
            return changeTrackPopup;
        };

        // show changes handler of the changetracking popup
        this.changeTrackPopupSelect = function () {
            docModel.getChangeTrack().showChangeTrackGroup();
            self.getChangeTrackPopup().show();
        };

        /**
         * Custom Change track date formatter. This function makes the private
         * date formatter 'formatChangeTrackDate' publicly available. Therefore
         * it supports the same parameter.
         *
         * @param {Number} timeInMilliSeconds
         *  Timestamp in milliseconds
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.toLocal=true]
         *      Timestamp is converted to local time zone per default.
         *
         * @returns {String}
         *  The formatted date string
         */
        this.getDisplayDateString = function (timeInMilliSeconds, options) {
            return formatChangeTrackDate(timeInMilliSeconds, options);
        };

        this.showPageSettingsDialog = function () {
            return new Dialogs.PageSettingsDialog(this).show();
        };

        /**
         * Shows the change track popup consisting updated change track data.
         */
        this.showChangeTrackPopup = function () {
            var popupMeta = getChangeTrackPopupMeta();
            // create show popup only on change track nodes
            if (popupMeta) {
                // update popup anchor node
                changeTrackPopup.setAnchor(getChangeTrackPopupAnchorNode());
                // update popup contents
                changeTrackBadge.update({
                    author: popupMeta.author || gt('Unknown'),
                    authorColorIndex: app.getAuthorColorIndex(popupMeta.author),
                    authorUserId: popupMeta.uid,
                    date: formatChangeTrackDate(Date.parse(popupMeta.date), { toLocal: false }),
                    action: popupMeta.actionText || gt('Unknown')
                });
                changeTrackPopup.show();
            }
        };

        /**
         * Returns the timer (promise) which opens the changeTrack-popup
         * after a specific time. It's abortable.
         *
         * @returns {promise}
         *  The changeTrack-popup-opening-timer
         */
        this.getChangeTrackPopupTimeout = function () {
            return changeTrackPopupTimeout;
        };

        /**
         * Removes the drawinganchor marker of all paragraphs.
         */
        this.clearVisibleDrawingAnchor = function () {
            docModel.getCurrentRootNode().find('.visibleAnchor').removeClass('visibleAnchor').find('.anchorIcon').remove();
        };

        /**
         * Sets the marker for the anchor
         *
         * @param {HTMLElement|jQuery} drawing
         *  The DOM element for that the anchor will be made
         *  visible.
         *
         * @param {Object} attributes
         *  A map of attribute value maps (name/value pairs), keyed by
         *  attribute families.
         */
        this.setVisibleDrawingAnchor = function (drawing, attributes) {
            var oxoPosition = null,
                paragraph = null;

            // get out here, no attributes given
            if (_.isEmpty(attributes)) { return; }

            // remove all visible anchors first
            self.clearVisibleDrawingAnchor();

            // inline drawing get no visible anchor
            if (!_.has(attributes, 'drawing') || !_.has(attributes.drawing, 'inline') || attributes.drawing.inline) { return; }

            oxoPosition = Position.getOxoPosition(docModel.getCurrentRootNode(), drawing, 0);
            paragraph = $(Position.getParagraphElement(docModel.getCurrentRootNode(), _.initial(oxoPosition)));

            $(paragraph).addClass('visibleAnchor').prepend('<i class="docs-position-anchor anchorIcon helper">');
        };

        /**
         * Public accesor to show the field format popup.
         *
         * @param {Object} fieldinstruction
         *  Instruction of the field for which the group with format codes will be updated.
         * @param {jQuery} node
         *  Field node.
         * @param {Object} options
         *  @param {Boolean} options.autoDate
         */
        this.showFieldFormatPopup = function (fieldinstruction, node, options) {
            var nodeHelper,
                datePickerVal = '',
                fieldText = '',
                datePickerDataValue;

            if (!this.isEditable() || docModel.getSelection().hasRange() || !fieldinstruction) { return; }
            fieldFormatPopup.setAnchor(getFieldAnchorNode());
            datePickerNode.hide(); // always hide datepicker

            if (fieldinstruction.formats && fieldinstruction.type && (/^DATE|^TIME/i).test(fieldinstruction.type)) {
                fieldFormatPopup.getNode().find('.dateSection').removeClass('not-available');
                currentFormatOfFieldPopup = fieldinstruction.instruction;

                autoDate = (options && options.autoDate) || false;
                autoUpdateCheckbox.setValue(autoDate);
                setToButton.getNode().toggleClass('disable', autoDate);
                toggleDatePickerBtn.getNode().toggleClass('disable', autoDate);

                if (node) {
                    currentCxNode = $(node);
                    datePickerDataValue = $(node).data('datepickerValue');
                    if (datePickerDataValue && !_.isEmpty(datePickerDataValue)) {
                        datePickerVal = datePickerDataValue;
                        datePickerNode.datepicker('update', datePickerVal); // reset previously picked date
                    } else {
                        nodeHelper = $(node).next();
                        fieldText = nodeHelper.text();
                        while (nodeHelper.length && nodeHelper.is('span')) {
                            nodeHelper = nodeHelper.next();
                            fieldText += nodeHelper.text();
                        }
                        datePickerVal = self.getDocModel().getNumberFormatter().parseLeadingDate(fieldText, { complete: true });
                        if (!datePickerVal || !moment(datePickerVal.date).isValid() || _.isUndefined(datePickerVal.Y) || _.isUndefined(datePickerVal.M) || _.isUndefined(datePickerVal.D)) {
                            datePickerNode.datepicker('update', ''); // reset previously picked date
                        } else {
                            datePickerNode.datepicker('update', new Date(datePickerVal.Y, datePickerVal.M, datePickerVal.D));
                        }
                    }
                }
            } else {
                fieldFormatPopup.getNode().find('.dateSection').addClass('not-available');
            }

            fieldFormatList.update(fieldinstruction);
            fieldFormatPopup.show();
        };

        /**
         * Public accesor to hide and destroy the field format popup.
         */
        this.hideFieldFormatPopup = function () {
            fieldFormatPopup.hide();
        };

        /**
         * Clicking on button from field format popup, field will get current date.
         */
        this.setFieldToTodayDate = function () {
            var date = new Date();
            var standardizedDate = self.formatDateWithNumberFormatter(date, 'yyyy-mm-dd\\Thh:mm:ss.00');
            currentCxNode.data('datepickerValue', date);
            self.trigger('fielddatepopup:change', { value: null, format: currentFormatOfFieldPopup, standard: standardizedDate }); // send default - which is current date
        };

        /**
         * Show or hide datepicker in field popup, and refresh popup.
         */
        this.toggleDatePicker = function () {
            datePickerNode.toggle();
            fieldFormatPopup.hide();
            fieldFormatPopup.show();
        };

        /**
         * Callback function for clicking checkbox in popup to update date field automatically.
         */
        this.autoFieldCheckboxUpdate = function () {
            datePickerNode.hide();

            autoDate = !autoDate;

            if (autoDate) {
                setToButton.getNode().addClass('disable');
                toggleDatePickerBtn.getNode().addClass('disable');
            } else {
                setToButton.getNode().removeClass('disable');
                toggleDatePickerBtn.getNode().removeClass('disable');
            }

            fieldFormatPopup.hide();
            fieldFormatPopup.show();

            autoUpdateCheckbox.setValue(autoDate);
            // trigger event for setting field to fixed
            self.trigger('fielddatepopup:autoupdate', autoDate);
        };

        /**
         * Formats default passed date to given format.
         *
         * @param {String} date
         * @param {String} formatCode
         * @returns {String}
         */
        this.formatDateWithNumberFormatter = function (date, formatCode) {
            var numberFormatter = docModel.getNumberFormatter();
            var parsedFormat = numberFormatter.getParsedFormat(formatCode);
            var serial = numberFormatter.convertDateToNumber(date);
            return numberFormatter.formatValue(parsedFormat, serial);
        };

        /**
         * Tries to find bookmark node for anchor set in current selection.
         * If found, sets cursor and jumps to that node.
         */
        this.goToSelectedAnchor = function () {
            var selection = docModel.getSelection();
            var rootNode = docModel.getCurrentRootNode();
            var start = selection.getStartPosition();
            var nodeInfo = Position.getDOMPosition(rootNode, start);
            var node, gotoNode, gotoNodePos, characterAttrs;

            if (nodeInfo && nodeInfo.node) {
                node = nodeInfo.node;
                if (node.nodeType === 3) { node = node.parentNode; }
                characterAttrs = AttributeUtils.getExplicitAttributes(node, { family: 'character', direct: true });
            }

            if (characterAttrs && characterAttrs.anchor) {
                gotoNode = pageNode.find('div[anchor="' + characterAttrs.anchor + '"]');
                if (gotoNode.length) {
                    gotoNodePos = Position.getOxoPosition(pageNode, gotoNode); // only headings in main document are considered for anchoring, => pageNode is used
                    if (gotoNodePos) {
                        selection.swichToValidRootNode(pageNode);
                        contentRootNode.animate({
                            scrollTop: gotoNode.parent().position().top
                        });
                        selection.setTextSelection(gotoNodePos);
                    }
                }
            }
        };

        /**
         * Returns the ruler pane containing the horizontal ruler.
         *
         * @returns {RulerPane}
         *  The ruler pane instance.
         */
        this.getRulerPane = function () {
            return rulerPane;
        };

        /**
         * Returns whether the edit mode for the softkeyboard is on or off.
         * When this mode is on, the softkeyboard should not open when the user
         * taps into the document in most cases.
         *
         * @returns {Boolean}
         */
        this.isSoftkeyboardEditModeOff = function () {
            // when this class exists the keyboard will not be opened in the page
            return app.getWindowNode().hasClass('soft-keyboard-off');
        };

        /**
         * Public accesor to scrollToSelection().
         */
        this.scrollToSelection = function (options) {
            scrollToSelection(options);
        };

        this.getActiveTableFlags = function () {
            var flags = {
                    firstRow:   true,
                    lastRow:    true,
                    firstCol:   true,
                    lastCol:    true,
                    bandsHor:   true,
                    bandsVert:  true
                },
                tableAttrs = AttributeUtils.getExplicitAttributes(docModel.getSelection().getEnclosingTable()).table || null;

            if (tableAttrs && tableAttrs.exclude && _.isArray(tableAttrs.exclude) && tableAttrs.exclude.length > 0) {
                tableAttrs.exclude.forEach(function (ex) {
                    flags[ex] = false;
                });
            }
            return flags;
        };

        // initialization -----------------------------------------------------

        // create the context menu
        contextMenu = new UniversalContextMenu(this, { delay: (_.browser.IE) ? 1 : 200 });

        // create other controls
        changeTrackBadge = new Controls.ChangeTrackBadge(this, { immediatedelete: true });
        changeTrackAcceptButton = new Button(this, { icon: 'fa-check', value: 'accept', tooltip: gt('Accept this change in the document') });
        changeTrackRejectButton = new Button(this, { icon: 'fa-times', value: 'reject', tooltip: gt('Reject this change in the document') });
        changeTrackShowButton = new Button(this, { icon: 'fa-eye', value: 'show', tooltip: gt('Show this change in the document') });

        // store the values of page width and height with paddings, after loading is finished
        this.waitForImportSuccess(function () {
            pageNodeWidth = pageNode.outerWidth();
            pageNodeHeight = pageNode.outerHeight();

            if (Utils.COMPACT_DEVICE && !Utils.SMALL_DEVICE) {
                self.executeDelayed(function () {
                    var optimalDocWidth = calculateOptimalDocumentWidth();
                    if (optimalDocWidth < 100) {
                        zoomType = 'width';
                    }
                }, 'TextView.waitForImportSuccess');
            }

            if (Utils.IOS) {
                var appContent = pageNode.parent();
                appContent.attr('contenteditable', true);

                var listenerList = docModel.getListenerList();
                pageNode.off(listenerList);
                appContent.on(listenerList);

                //we dont want the focus in the document on the beginning!
                Utils.focusHiddenNode();
            }

            // quick fix for bug 40053
            if (_.browser.Safari) { Forms.addDeviceMarkers(app.getWindowNode()); }
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            contextMenu.destroy();
            if (changeTrackPopup) { changeTrackPopup.destroy(); } // still null, if launching application fails
            if (fieldFormatPopup) { fieldFormatPopup.destroy(); }
            if (setToButton) { setToButton.destroy(); }
            if (toggleDatePickerBtn) { toggleDatePickerBtn.destroy(); }

            app = self = replayOperations = replayOperationsOSN = docModel = null;
            appPaneNode = contentRootNode = pageNode = pageContentNode = contextMenu = rulerPane = null;
            changeTrackPopup = changeTrackBadge = changeTrackAcceptButton = changeTrackRejectButton = changeTrackShowButton = null;
            fieldFormatPopup = fieldFormatList = datePickerNode = descriptionNode = autoUpdateCheckbox = setToButton = toggleDatePickerBtn = todayDatepickerWrapper = currentCxNode = null;
        });

    } }); // class TextView

    // exports ================================================================

    return TextView;

});
