/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/text/view/toolbars', [
    'io.ox/office/baseframework/view/toolbar',
    'io.ox/office/textframework/view/toolbars',
    'io.ox/office/text/view/controls',
    'gettext!io.ox/office/text/main'
], function (ToolBar, TextBaseToolBars, Controls, gt) {

    'use strict';

    // static class TextToolBars ==============================================

    /**
     * A static map with tool bar classes for presentation applications.
     *
     * @type Object
     * @constant
     */
    var TextToolBars = _.extend({}, TextBaseToolBars);

    // class ParagraphAlignmentToolBar ========================================

    /**
     * A tool bar with controls to change text text alignment of paragraphs.
     *
     * @constructor
     *
     * @extends ToolBar
     *
     * @param {TextView} docView
     *  The document view instance containing this tool bar.
     */
    TextToolBars.ParagraphAlignmentToolBar = ToolBar.extend({ constructor: function (docView) {

        // base constructor
        ToolBar.call(this, docView);

        // create the controls of this tool bar
        var alignPicker = new Controls.ParagraphAlignmentPicker(docView, { dropDownVersion: { visible: true } });
        var alignGroup = new Controls.ParagraphAlignmentGroup(docView, { smallerVersion: { pendant: alignPicker } });
        this.addGroup('paragraph/alignment', alignGroup);
        this.addGroup('paragraph/alignment', alignPicker);

        // destroy all class members on destruction
        this.registerDestructor(function () { docView = null; });

    } }); // class ParagraphAlignmentToolBar

    // class ParagraphFillBorderToolBar =======================================

    /**
     * A tool bar with controls to change the spcing, fill color, and border
     * style of paragraphs.
     *
     * @constructor
     *
     * @extends ToolBar
     *
     * @param {TextView} docView
     *  The document view instance containing this tool bar.
     */
    TextToolBars.ParagraphFillBorderToolBar = ToolBar.extend({ constructor: function (docView) {

        // base constructor
        ToolBar.call(this, docView, {
            shrinkToMenu: { icon: 'docs-border-h', tooltip: gt('Paragraph formatting') }
        });

        // create the controls of this tool bar
        this.addGroup('paragraph/spacing/menu', new Controls.ParagraphSpacingPicker(docView));
        this.addGap();
        this.addGroup('paragraph/fillcolor', new Controls.ParagraphFillColorPicker(docView));
        this.addGap();
        this.addGroup('paragraph/borders', new Controls.ParagraphBorderModePicker(docView));

        // destroy all class members on destruction
        this.registerDestructor(function () { docView = null; });

    } }); // class ParagraphFillBorderToolBar

    // class ParagraphStyleToolBar ============================================

    /**
     * A tool bar with a picker control for the paragraph style sheet.
     *
     * @constructor
     *
     * @extends ToolBar
     *
     * @param {TextView} docView
     *  The document view instance containing this tool bar.
     */
    TextToolBars.ParagraphStyleToolBar = ToolBar.extend({ constructor: function (docView) {

        // base constructor
        ToolBar.call(this, docView);

        // create the controls of this tool bar
        this.addGroup('paragraph/stylesheet', new Controls.ParagraphStylePicker(docView));

        // destroy all class members on destruction
        this.registerDestructor(function () { docView = null; });

    } }); // class ParagraphStyleToolBar

    // class TableStyleToolBar ================================================

    /**
     * A tool bar with a picker control for the table style sheet.
     *
     * @constructor
     *
     * @extends ToolBar
     *
     * @param {TextView} docView
     *  The document view instance containing this tool bar.
     */
    TextToolBars.TableStyleToolBar = ToolBar.extend({ constructor: function (docView) {

        // base constructor
        ToolBar.call(this, docView);

        // create the controls of this tool bar
        this.addGroup('table/stylesheet', new Controls.TableStylePicker(docView, docView.getActiveTableFlags.bind(docView)));

        // destroy all class members on destruction
        this.registerDestructor(function () { docView = null; });

    } }); // class TableStyleToolBar

    // exports ================================================================

    return TextToolBars;

});
