/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/text/model/pagehandlermixin', [
    'io.ox/office/textframework/utils/textutils',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/components/hyperlink/hyperlink',
    'io.ox/office/textframework/components/table/tableresize',
    'io.ox/office/drawinglayer/view/drawingframe'
], function (Utils, DOM, Position, Hyperlink, TableResize, DrawingFrame) {

    'use strict';

    // mix-in class PageHandlerMixin ======================================

    /**
     * A mix-in class for the document model class TextModel providing
     * the event handlers for the main slide page used in a text document.
     *
     * @constructor
     *
     * @param {Application} app
     *  The application containing this instance.
     */
    function PageHandlerMixin(app) {

        var // self reference for local functions
            self = this;

        // private methods ----------------------------------------------------

        /**
         * The handler for the 'mousedown' and the 'touchstart' event in the presentation
         * application.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object.
         */
        function pageProcessMouseDownHandler(event) {

            var // whether the user has privileges to edit the document
                readOnly = !app.isEditable(),
                // jQuerified target of the event
                $eventTarget = $(event.target),
                // mouse click on a table
                tableCell = $eventTarget.closest(DOM.TABLE_CELLNODE_SELECTOR),
                tableCellPoint = null,
                // drawing start and end position for selection
                startPosition = null, endPosition = null,
                // mouse click on a drawing node
                drawingNode = null,
                // mouse click into a comment node
                commentNode = null,
                // mouse click on a tabulator node
                tabNode = null,
                // move handler node for drawings
                moveNode = null,
                // mouse click on a table resize node
                resizerNode = null,
                // mouse click on field node
                field = null,
                // the id of an optionally activated comment node
                activatedCommentId = '',
                // change track popup
                changeTrackPopup = app.getView().getChangeTrackPopup(),
                // the currently activated node
                activeRootNode = self.getCurrentRootNode(),
                // current page number
                pageNum = null,
                // delay of opening the changeTrack-popup
                changeTrackPopupTimeout = app.getView().getChangeTrackPopupTimeout(),
                // the changeTrack-popup
                changeTrakkPopup = app.getView().getChangeTrackPopup(),
                // whether the right mouse-button was clicked, or not
                rightClick = (event.button === 2),
                // the selection object
                selection = self.getSelection(),
                // the page layout object
                pageLayout = self.getPageLayout(),
                // the comment layer object
                commentLayer = self.getCommentLayer(),
                // whether this event is of type 'touchstart'
                isTouchStartEvent = event.type === 'touchstart';

            // fix for not clearing anoying selection range in Chrome on click
            if (_.browser.Chrome && selection.hasRange() && !rightClick && !event.shiftKey) {
                window.getSelection().empty();
            }

            // expanding a waiting double click to a triple click
            if (self.getDoubleClickEventWaiting()) {
                self.setTripleClickActive(true);
                event.preventDefault();
                return;
            }

            // do not show the changeTrack-popup, if the right mousebutton was cliked
            // or hide it, if it's already open
            if (rightClick) {
                if (changeTrackPopupTimeout) { changeTrackPopupTimeout.abort(); }
                if (changeTrakkPopup.isVisible()) { changeTrakkPopup.hide(); }
            }

            // activation mouse down
            self.setActiveMouseDownEvent(true);

            // make sure that the clickable parts in the hyperlink popup are
            // processed by the browser
            if (Hyperlink.isClickableNode($eventTarget, readOnly)) {
                return;
            } else if (Hyperlink.isPopupOrChildNode($eventTarget)) {
                // don't call selection.processBrowserEvent as we don't have
                // a valid position in the document if we process mouse event
                // on the popup
                event.preventDefault();
                return;
            }

            // avoid in FF white background, if user double clicked on textframe node (54187)
            if (_.browser.Firefox && DrawingFrame.isTextFrameNode($eventTarget) && selection.isAdditionalTextframeSelection()) {
                // check, if this drawing is already selected
                if (Utils.getDomNode(selection.getSelectedTextFrameDrawing()) === Utils.getDomNode($eventTarget.closest('.drawing'))) {
                    $eventTarget.removeAttr('_moz_abspos'); // this attribute was already set by the browser
                    event.preventDefault();
                    return;
                }
            }

            // also dont do processBrowserEvent if the change track popup is clicked
            if (changeTrackPopup.getNode().find(event.target).length > 0) {
                event.preventDefault();
                return;
            }

            // not a link!!! for halo view fix for Bug 39056
            // clicking into a comment meta info node handler or in IE on the comment border (thread itself or text frame content node) (37672)
            if ((!isTouchStartEvent || !$eventTarget.is('a')) && (DOM.isNodeInsideCommentMetaInfo(event.target) || (_.browser.IE && (DOM.isCommentThreadNode(event.target) || (DrawingFrame.isTextFrameContentNode(event.target) && DOM.isCommentNode(event.target.parentNode)))))) {
                event.preventDefault();
                return;
            }

            // Fix for 29257 and 29380 and 31623
            if ((_.browser.IE === 11) && ($eventTarget.is('table') ||
                    DOM.isListLabelNode(event.target) || (event.target.parentNode && DOM.isListLabelNode(event.target.parentNode)) ||
                    DOM.isPlaceholderNode(event.target) || (event.target.parentNode && DOM.isPlaceholderNode(event.target.parentNode)) ||
                    (event.target.parentNode && event.target.parentNode.parentNode && DOM.isPlaceholderNode(event.target.parentNode.parentNode)))) {
                return;
            }

            // Fix for 29409: preventing default to avoid grabbers in table cells or inline components
            if ((_.browser.IE === 11) &&
                ($eventTarget.is('div.cell') ||
                DOM.isInlineComponentNode(event.target) ||
                (event.target.parentNode && DOM.isInlineComponentNode(event.target.parentNode)))) {
                event.preventDefault(); // -> not returning !
            }

            if (!Utils.SMALL_DEVICE) { // no page break handling on small devices

                if (self.isHeaderFooterEditState()) {
                    if ((_.browser.Firefox || _.browser.IE) && $eventTarget.is('.cover-overlay, .inactive-selection, .page-break')) {
                        event.preventDefault();
                        return; //#37151
                    }

                    if (DOM.isMarginalContextItem(event.target)) {
                        event.preventDefault();
                        if ($eventTarget.hasClass('marginal-menu-goto')) {
                            pageNum = pageLayout.getPageNumber(self.getCurrentRootNode());
                            pageLayout.leaveHeaderFooterAndSetCursor(self.getCurrentRootNode());
                            if ($eventTarget.parent().hasClass('above-node')) {
                                pageLayout.jumpToHeaderOnCurrentPage(pageNum);
                            } else {
                                pageLayout.jumpToFooterOnCurrentPage(pageNum);
                            }
                        } else {
                            pageLayout.leaveHeaderFooterAndSetCursor(self.getCurrentRootNode());
                            activeRootNode = self.getCurrentRootNode();
                        }
                        return;
                    } else if ($eventTarget.hasClass('marginal-menu-type') || $eventTarget.hasClass('marginal-label') || $eventTarget.hasClass('fa-caret-down')) {
                        event.preventDefault();
                        pageLayout.toggleMarginalMenuDropdown(event.target);
                        return;
                    } else if ($eventTarget.hasClass('marginal-none')) {
                        event.preventDefault();
                        pageLayout.setHeaderFooterTypeInDoc('none');
                        return;
                    } else if ($eventTarget.hasClass('marginal-same')) {
                        event.preventDefault();
                        pageLayout.setHeaderFooterTypeInDoc('default');
                        return;
                    } else if ($eventTarget.hasClass('marginal-first')) {
                        event.preventDefault();
                        pageLayout.setHeaderFooterTypeInDoc('first');
                        return;
                    } else if ($eventTarget.hasClass('marginal-even')) {
                        event.preventDefault();
                        pageLayout.setHeaderFooterTypeInDoc('evenodd');
                        return;
                    } else if ($eventTarget.hasClass('marginal-first-even')) {
                        event.preventDefault();
                        pageLayout.setHeaderFooterTypeInDoc('all');
                        return;
                    } else if ($eventTarget.hasClass('marginal-separator-line') || $eventTarget.hasClass('marginal-container')) {
                        event.preventDefault();
                        return;
                    } else if (selection.isAnyDrawingSelection() && $eventTarget.closest('.drawingselection-overlay').length > 0) {
                        // click inside selection overlay node -> simply not handle this event here
                    } else if (!DOM.isHeaderOrFooter(event.target) && DOM.getMarginalTargetNode(self.getNode(), event.target).length === 0) { // Leaving Header-footer edit state
                        pageLayout.leaveHeaderFooterAndSetCursor(self.getParentOfHeaderFooterRootNode().find('.active-selection'), self.getParentOfHeaderFooterRootNode());
                        activeRootNode = self.getCurrentRootNode(); // reset cached node after changing root node
                        self.updateEditingHeaderFooterDebounced(); // this call is important for repainting layout after leaving h/f edit mode
                    }
                } else {
                    if (_.browser.IE && !rightClick && $eventTarget.is('.inactive-selection, .cover-overlay')) { // #35678 prevent resize rectangles on headers in IE
                        event.preventDefault();
                        selection.updateIESelectionAfterMarginalSelection(activeRootNode, $eventTarget);
                        return;
                    }
                }

                if (_.browser.IE && $eventTarget.hasClass('inner-pb-line')) {
                    event.preventDefault();
                    return;
                }

                if (tableCell.length > 0) {
                    if (DOM.isTablePageBreakRowNode(tableCell.parent())) {
                        tableCell = tableCell.parent().next().children().eq(tableCell.index());
                        tableCell = tableCell.find('.p').first();
                        if (tableCell.length) { // if clicked on table cell in repeated row, try to find position in a same cell one row behind
                            startPosition = Position.getOxoPosition(selection.getRootNode(), tableCell, 0);
                            if (_.isArray(startPosition)) {
                                startPosition.push(0);
                                selection.setTextSelection(startPosition);
                                event.preventDefault();
                                return;
                            }
                        }
                    }
                    // saving initial cell as anchor cell for table cell selection
                    // -> this has to work also in read-only mode
                    tableCellPoint = DOM.Point.createPointForNode(tableCell);
                    selection.setAnchorCellRange(new DOM.Range(tableCellPoint, new DOM.Point(tableCellPoint.node, tableCellPoint.node + 1)));
                }
            }

            if (Utils.SMALL_DEVICE && self.getActiveTarget()) {
                commentLayer.handleCommentSelectionOnSmallDevice(event, self.getActiveTarget());
                selection.processBrowserEvent(event);
                return;
            }

            // tabulator handler
            tabNode = $eventTarget.closest(DOM.TAB_NODE_SELECTOR);
            if (tabNode.length > 0) {
                // handle in processMouseUp event
                return;
            }

            // checking for a selection on a drawing node. But if this is a text frame drawing node, this is not a drawing
            // selection, if the click happened on the text frame sub element.
            drawingNode = $eventTarget.closest(DrawingFrame.NODE_SELECTOR);

            if (drawingNode.length > 0) {
                var isInsideTextFrame = Utils.TOUCHDEVICE ? self.isNodeInsideTextframe($eventTarget) : ($eventTarget.closest(DrawingFrame.TEXTFRAME_NODE_SELECTOR).length > 0);
                if (isInsideTextFrame && DrawingFrame.isOnlyBorderMoveableDrawing(drawingNode)) {
                    drawingNode = $(); // for touch devices the event target must be inside a text frame, for other devices it can also be the text frame node.
                } else if (DOM.isNodeInsideComment(drawingNode.parent())) {
                    commentNode = drawingNode.closest(DOM.COMMENTNODE_SELECTOR);
                    activatedCommentId = commentLayer.activateCommentNode(commentNode, selection, { deactivate: true }); // happens not, if the node is already active
                    activeRootNode = commentNode; // selecting a drawing inside a comment
                } else if (DOM.isCommentNode(drawingNode)) {
                    // checking, if this was a click inside a comment
                    activatedCommentId = commentLayer.activateCommentNode(drawingNode, selection, { deactivate: true }); // happens not, if the node is already active
                    activeRootNode = drawingNode;
                    drawingNode = $();
                }
            }

            // deactivating comment node, but not if it was already deactivated in 'activateCommentNode' before
            if (self.getActiveTarget() !== '' && !activatedCommentId && !self.isHeaderFooterEditState()) {
                //  leaving comment, restore original rootNode (but not if this mouse down happens in the same comment)
                commentLayer.deActivateCommentNode(self.getActiveTarget(), selection);
                // reset cached active root node
                activeRootNode = selection.getRootNode(); // #43269
            }

            // checking for selection on a field
            field = Utils.SMALL_DEVICE ? [] : $eventTarget.closest(DOM.FIELD_NODE_SELECTOR);

            if (field.length > 0) {
                // prevent default click handling of the browser
                event.preventDefault();

                // set focus to the document container (may be located in GUI edit fields)
                app.getView().grabFocus();

                // select the field
                startPosition = Position.getOxoPosition(activeRootNode, field[0], 0);
                endPosition = startPosition;
                selection.setTextSelection(startPosition, endPosition, { event: event });
                return;
            }

            // in read only mode allow text selection only (and drawing selection, task 30041)
            if (readOnly && drawingNode.length === 0) {
                selection.processBrowserEvent(event);
                return;
            }

            // checking for a selection on a resize node
            resizerNode = Utils.SMALL_DEVICE ? [] : $eventTarget.closest(DOM.RESIZE_NODE_SELECTOR);

            if (drawingNode.length > 0) {

                // prevent default click handling of the browser
                // -> this avoids for example scrolling page on touch devices, if drawing shall be moved
                event.preventDefault();

                // converting selection overlay node to 'real' drawing node in the document (if required)
                if (drawingNode.hasClass('selectiondrawing')) {
                    drawingNode = self.getSelection().getDrawingNodeFromDrawingSelection(drawingNode);
                }

                // set focus to the document container (may be located in GUI edit fields);
                // This line needs to be executed after a selection drawing was switched to the 'real' drawingNode,
                // as it repaints selection, and (selection) drawingNode is not pointing to DOM node anymore
                app.getView().grabFocus();

                // do nothing if the drawing is already selected (prevent endless
                // recursion when re-triggering the event, to be able to start
                // moving the drawing immediately, see below)
                // Also create a drawing selection, if there is an additional text frame selection,
                // that also has a text selection. In this case the text selection is removed.
                if (!DrawingFrame.isSelected(drawingNode) || selection.isAdditionalTextframeSelection()) {

                    // select the drawing
                    startPosition = Position.getOxoPosition(activeRootNode, drawingNode[0], 0);

                    if (startPosition) {
                        endPosition = Position.increaseLastIndex(startPosition);
                        selection.setTextSelection(startPosition, endPosition, { event: event });
                    } else { // TODO: sometimes the position cannot be determined
                        Utils.warn('PageHandlerMixin.pageProcessMouseDownHandler(): Failed to determine position of drawing node!');
                    }

                    // The following triggering of the event is necessary to move a drawing without a further mousedown
                    // or touchstart event. Without this additional trigger the first mousedown/touchstart would only select
                    // the drawing. For moving it, a further mousedown/touchstart is requrired. By triggering this event
                    // again, it is possible to move a drawing already with the first mousedown/touchstart.
                    // For moving a drawing, a tracker node was created and registered for 'tracking:start' in the
                    // preceeding selection.drawDrawingSelection() call. This additional mousedown/touchstart
                    // generates a 'tracking:start' event, that will be handled by the tracker node.
                    // The event target for this event in 'processMouseDown' could be the 'img' or the 'div.drawing'. But
                    // it has to be sent to the tracker. Therefore it is necessary to adapt the event target, before
                    // triggering it again.
                    if (drawingNode.data('selection')) {
                        moveNode = drawingNode.data('selection').find('.tracker');
                    } else {
                        moveNode = drawingNode.find('.tracker');
                    }
                    if (moveNode.length > 0) {
                        event.target = moveNode[0];
                        moveNode.trigger(event);
                    }
                }

                return; // no browser handling for this event

            } else if (resizerNode.length > 0) {

                if (!DOM.isActiveResizeNode(resizerNode)) {
                    // Because the event is triggered again to avoid a second click, it is necessary to mark the resize node
                    // as 'active'. Otherwise the event would be triggered endlessly.

                    // prevent default click handling of the browser
                    event.preventDefault();
                    // set focus to the document container (may be located in GUI edit fields)
                    app.getView().grabFocus();
                    // draw the resizer
                    TableResize.drawTableCellResizeSelection(self, app.getView(), app.getWindowNode(), resizerNode);

                    // send initial mouse down event to the registered handlers
                    // The following triggering of the event is necessary to resize the table without a further mousedown
                    // or touchstart event. Without this additional trigger the first mousedown/touchstart would only select
                    // the resize node. For moving it, a further mousedown/touchstart is requrired. By triggering this event
                    // again, it is possible to move a resize node already with the first mousedown/touchstart.
                    resizerNode.trigger(event);
                }
            } else {

                // clicked somewhere else: calculate logical selection from browser selection,
                // after browser has processed the mouse event
                selection.processBrowserEvent(event);

            }
        }

        // public methods -----------------------------------------------------

        /**
         * Getting the handler function that is used for process mouse down and touchstart
         * events on the page node.
         *
         * @returns {Function}
         *  The handler function for the 'mousedown' and the 'touchstart' event.
         */
        this.getPageProcessMouseDownHandler = function () {
            return pageProcessMouseDownHandler;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = null;
        });

    } // class PageHandlerMixin

    // constants --------------------------------------------------------------

    // exports ================================================================

    return PageHandlerMixin;

});
