/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/toolbars', [
    'io.ox/office/baseframework/view/toolbar',
    'io.ox/office/textframework/view/toolbars',
    'io.ox/office/spreadsheet/utils/config',
    'io.ox/office/spreadsheet/view/labels',
    'io.ox/office/spreadsheet/view/controls',
    'gettext!io.ox/office/spreadsheet/main'
], function (ToolBar, TextBaseToolBars, Config, Labels, Controls, gt) {

    'use strict';

    // convenience shortcuts
    var Button = Controls.Button;

    // static class SpreadsheetToolBars =======================================

    /**
     * A static map with tool bar classes for spreadsheet applications.
     *
     * @type Object
     * @constant
     */
    var SpreadsheetToolBars = _.extend({}, TextBaseToolBars);

    // class CellColorToolBar =================================================

    /**
     * Extends the class FontColorToolBar with a color picker control for the
     * cell background.
     *
     * @constructor
     *
     * @extends FontColorToolBar
     *
     * @param {SpreadsheetView} docView
     *  The document view instance containing this tool bar.
     */
    SpreadsheetToolBars.CellColorToolBar = TextBaseToolBars.FontColorToolBar.extend({ constructor: function (docView) {

        // base constructor
        TextBaseToolBars.FontColorToolBar.call(this, docView);

        // create the controls of this tool bar
        this.addGroup('cell/fillcolor', new Controls.CellFillColorPicker(docView), { visibleKey: 'view/selection/cell' });

        // destroy all class members on destruction
        this.registerDestructor(function () { docView = null; });

    } }); // class CellColorToolBar

    // class CellStyleToolBar =================================================

    /**
     * A tool bar with a picker control for the cell style sheet.
     *
     * @constructor
     *
     * @extends ToolBar
     *
     * @param {SpreadsheetView} docView
     *  The document view instance containing this tool bar.
     */
    SpreadsheetToolBars.CellStyleToolBar = ToolBar.extend({ constructor: function (docView) {

        // base constructor
        ToolBar.call(this, docView, { visibleKey: 'view/selection/cell', hideable: true });

        // create the controls of this tool bar
        this.addGroup('cell/stylesheet', new Controls.CellStylePicker(docView));

        // destroy all class members on destruction
        this.registerDestructor(function () { docView = null; });

    } }); // class CellStyleToolBar

    // class CellAlignmentToolBar =============================================

    //#. checkbox: automatic word wrapping (multiple text lines) in spreadsheet cells
    var AUTO_WORD_WRAP_LABEL = gt('Automatic word wrap');

    /**
     * A tool bar with controls to change text text alignment of the cells.
     *
     * @constructor
     *
     * @extends ToolBar
     *
     * @param {SpreadsheetView} docView
     *  The document view instance containing this tool bar.
     */
    SpreadsheetToolBars.CellAlignmentToolBar = ToolBar.extend({ constructor: function (docView) {

        // base constructor
        ToolBar.call(this, docView, {
            visibleKey: 'view/selection/cell',
            shrinkToMenu: { icon: 'docs-cell-h-align-auto', tooltip: Labels.ALIGNMENT_HEADER_LABEL }
        });

        // create the controls of this tool bar
        this.addGroup('cell/alignhor',  new Controls.AlignmentPicker(docView, 'horizontal'));
        this.addGroup('cell/alignvert', new Controls.AlignmentPicker(docView, 'vertical'));
        this.addGroup('cell/linebreak', new Button(docView, { icon: 'docs-insert-linebreak', tooltip: AUTO_WORD_WRAP_LABEL, toggle: true, dropDownVersion: { label: AUTO_WORD_WRAP_LABEL } }));
        this.addGroup('cell/merge',     new Controls.MergeCellsPicker(docView));

        // destroy all class members on destruction
        this.registerDestructor(function () { docView = null; });

    } }); // class CellAlignmentToolBar

    // class NumberFormatToolBar ==============================================

    /**
     * A tool bar with controls used to change the cell number format.
     *
     * @constructor
     *
     * @extends ToolBar
     *
     * @param {SpreadsheetView} docView
     *  The document view instance containing this tool bar.
     */
    SpreadsheetToolBars.NumberFormatToolBar = ToolBar.extend({ constructor: function (docView) {

        // base constructor
        ToolBar.call(this, docView, {
            visibleKey: 'view/selection/cell',
            hideable: true,
            shrinkToMenu: { icon: 'docs-percent', tooltip: Labels.NUMBERFORMAT_HEADER_LABEL }
        });

        // create the controls of this tool bar
        this.addGroup('cell/numberformat/category', new Controls.FormatCategoryGroup(docView));
        this.addGroup('cell/numberformat/category', new Controls.FormatCategoryPicker(docView));
        this.addGroup('cell/numberformat/code',     new Controls.FormatCodePicker(docView));

        // destroy all class members on destruction
        this.registerDestructor(function () { docView = null; });

    } }); // class NumberFormatToolBar

    // class CellBorderToolBar ================================================

    /**
     * A tool bar with controls used to change the cell border style.
     *
     * @constructor
     *
     * @extends ToolBar
     *
     * @param {SpreadsheetView} docView
     *  The document view instance containing this tool bar.
     */
    SpreadsheetToolBars.CellBorderToolBar = ToolBar.extend({ constructor: function (docView) {

        // base constructor
        ToolBar.call(this, docView, {
            visibleKey: 'view/selection/cell',
            hideable: true,
            shrinkToMenu: { icon: 'docs-cell-style', tooltip: Labels.CELL_BORDER_LABEL }
        });

        // controller key for the cell border style (depends on file format)
        var borderStyleKey = docView.getApp().isOOXML() ? 'cell/border/style/preset' : 'cell/border/style';

        // create the controls of this tool bar
        this.addGroup('cell/border/mode',  new Controls.CellBorderModePicker(docView));
        this.addGroup(borderStyleKey,      new Controls.CellBorderStylePicker(docView));
        this.addGroup('cell/border/width', new Controls.BorderWidthPicker(docView), { visibleKey: 'document/odf' });
        this.addGroup('cell/border/color', new Controls.CellBorderColorPicker(docView));

        // destroy all class members on destruction
        this.registerDestructor(function () { docView = null; });

    } }); // class CellBorderToolBar

    // class FormatPainterToolBar =============================================

    /**
     * A tool bar with a controls to copy the formatting of the selected cell.
     *
     * @constructor
     *
     * @extends ToolBar
     *
     * @param {SpreadsheetView} docView
     *  The document view instance containing this tool bar.
     */
    SpreadsheetToolBars.FormatPainterToolBar = ToolBar.extend({ constructor: function (docView) {

        // base constructor
        ToolBar.call(this, docView, { visibleKey: 'view/selection/cell' });

        // create the controls of this tool bar
        this.addGroup('cell/painter', new Button(docView, { icon: 'docs-format-painter', tooltip: Labels.FORMAT_PAINTER_LABEL, toggle: true, dropDownVersion: { label: Labels.FORMAT_PAINTER_LABEL } }));

        // destroy all class members on destruction
        this.registerDestructor(function () { docView = null; });

    } }); // class FormatPainterToolBar

    // class ColRowToolBar ====================================================

    /**
     * A tool bar with controls to modify columns or rows in the sheet.
     *
     * @constructor
     *
     * @extends ToolBar
     *
     * @param {SpreadsheetView} docView
     *  The document view instance containing this tool bar.
     *
     * @param {Boolean} columns
     *  Whether to create controls for columns (true), or for rows (false).
     */
    SpreadsheetToolBars.ColRowToolBar = ToolBar.extend({ constructor: function (docView, columns) {

        // base constructor
        ToolBar.call(this, docView);

        // create the controls of this tool bar
        this.addGroup(columns ? 'column/insert' : 'row/insert', new Controls.ColRowOpIconButton(docView, true, columns));
        this.addGroup(columns ? 'column/delete' : 'row/delete', new Controls.ColRowOpIconButton(docView, false, columns));
        if (!Config.COMBINED_TOOL_PANES) {
            var sizeKey = columns ? 'column/width/active' : 'row/height/active';
            this.addGroup(sizeKey, new Controls.ColRowSizeLabel(docView, columns));
            this.addGroup(sizeKey, new Controls.ColRowSizeField(docView, columns));
        }

        // destroy all class members on destruction
        this.registerDestructor(function () { docView = null; });

    } }); // class ColRowToolBar

    // class InsertContentToolBar =============================================

    /**
     * A tool bar with controls to insert something into the selected cells.
     *
     * @constructor
     *
     * @extends ToolBar
     *
     * @param {SpreadsheetView} docView
     *  The document view instance containing this tool bar.
     */
    SpreadsheetToolBars.InsertContentToolBar = ToolBar.extend({ constructor: function (docView) {

        // base constructor
        ToolBar.call(this, docView, { visibleKey: 'view/selection/cell' });

        // create the controls of this tool bar
        this.addGroup('cell/autoformula', new Controls.InsertAutoSumButton(docView));
        this.addGroup('function/insert/dialog', new Controls.InsertFunctionButton(docView));
        this.addSeparator();
        this.addGroup('hyperlink/edit/dialog', new Button(docView, Labels.HYPERLINK_BUTTON_OPTIONS));

        // destroy all class members on destruction
        this.registerDestructor(function () { docView = null; });

    } }); // class InsertContentToolBar

    // class InsertDrawingToolBar =============================================

    /**
     * A tool bar with controls to insert drawing objects into the sheet.
     *
     * @constructor
     *
     * @extends ToolBar
     *
     * @param {SpreadsheetView} docView
     *  The document view instance containing this tool bar.
     */
    SpreadsheetToolBars.InsertDrawingToolBar = ToolBar.extend({ constructor: function (docView) {

        // base constructor
        ToolBar.call(this, docView);

        // create the controls of this tool bar
        this.addGroup('image/insert/dialog', new Controls.ImagePicker(docView));
        this.addGap();
        this.addGroup('shape/insert', new Controls.ShapeTypePicker(docView));
        this.addGap();
        this.addGroup('chart/insert', new Controls.ChartTypePicker(docView, Labels.INSERT_CHART_BUTTON_OPTIONS));

        // destroy all class members on destruction
        this.registerDestructor(function () { docView = null; });

    } }); // class InsertDrawingToolBar

    // class TableToolBar =====================================================

    /**
     * A tool bar with controls to manipulate table ranges.
     *
     * @constructor
     *
     * @extends ToolBar
     *
     * @param {SpreadsheetView} docView
     *  The document view instance containing this tool bar.
     */
    SpreadsheetToolBars.TableToolBar = ToolBar.extend({ constructor: function (docView) {

        // base constructor
        ToolBar.call(this, docView);

        // create the controls of this tool bar
        this.addGroup('table/stylesheet', new Controls.SheetTableStylePicker(docView), { visibleKey: 'document/ooxml' });

        // destroy all class members on destruction
        this.registerDestructor(function () { docView = null; });

    } }); // class TableToolBar

    // class CommentsToolBar ==================================================

    /**
     * A tool bar with controls to manipulate cell comments.
     *
     * @constructor
     *
     * @extends ToolBar
     *
     * @param {SpreadsheetView} docView
     *  The document view instance containing this tool bar.
     */
    SpreadsheetToolBars.CommentsToolBar = ToolBar.extend({ constructor: function (docView) {

        // base constructor
        ToolBar.call(this, docView);

        // create the controls of this tool bar
        this.addGroup('comment/toggle/all', new Button(docView, {
            icon: 'docs-comment-show',
            label: /*#. show/hide all comments in spreadsheet */ gt.pgettext('comment-actions', 'Show/Hide all'),
            tooltip: /*#. show/hide all comments in spreadsheet */ gt.pgettext('comment-actions', 'Show/Hide all comments'),
            smallerVersion: { hideLabel: true }
        }));
        this.addSeparator();
        this.addGroup('comment/toggle', new Button(docView, {
            icon: 'docs-comment-show',
            label: /*#. show/hide the selected comment in spreadsheet */ gt.pgettext('comment-actions', 'Show/Hide'),
            tooltip: /*#. show/hide the selected comment in spreadsheet */ gt.pgettext('comment-actions', 'Show/Hide selected comment'),
            smallerVersion: { hideLabel: true }
        }));
        this.addGroup('comment/show/prev', new Button(docView, {
            icon: 'docs-comment-back',
            label: /*#. go to preceding comment */ gt.pgettext('comment-actions', 'Previous'),
            tooltip: gt.pgettext('comment-actions', 'Go to previous comment'),
            smallerVersion: { hideLabel: true }
        }));
        this.addGroup('comment/show/next', new Button(docView, {
            icon: 'docs-comment-next',
            label: /*#. go to next comment */ gt.pgettext('comment-actions', 'Next'),
            tooltip: gt.pgettext('comment-actions', 'Go to next comment'),
            smallerVersion: { hideLabel: true }
        }));
        this.addGroup('comment/delete', new Button(docView, {
            icon: 'docs-comment-remove',
            label: /*#. delete the selected comment in spreadsheet */ gt.pgettext('comment-actions', 'Delete'),
            tooltip: /*#. delete the selected comment in spreadsheet */ gt.pgettext('comment-actions', 'Delete comment'),
            smallerVersion: { hideLabel: true }
        }));
        this.addSeparator();
        this.addGroup('comment/delete/all', new Button(docView, {
            icon: 'docs-comment-remove',
            label: /*#. delete all comments in spreadsheet */ gt.pgettext('comment-actions', 'Delete all'),
            tooltip: /*#. delete all comments in spreadsheet */ gt.pgettext('comment-actions', 'Delete all comments'),
            smallerVersion: { hideLabel: true }
        }));

        // destroy all class members on destruction
        this.registerDestructor(function () { docView = null; });

    } }); // class CommentsToolBar

    // exports ================================================================

    return SpreadsheetToolBars;

});
