/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/mixin/selectionmixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/utils/iterator',
    'io.ox/office/tk/utils/scheduler',
    'io.ox/office/tk/render/rectangle',
    'io.ox/office/drawinglayer/utils/drawingutils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/utils/sheetselection'
], function (Utils, KeyCodes, Iterator, Scheduler, Rectangle, DrawingUtils, SheetUtils, PaneUtils, SheetSelection) {

    'use strict';

    // convenience shortcuts
    var FilterIterator = Iterator.FilterIterator;
    var Interval = SheetUtils.Interval;
    var Address = SheetUtils.Address;
    var Range = SheetUtils.Range;
    var RangeArray = SheetUtils.RangeArray;

    // class CustomSelectDescriptor ===========================================

    /**
     * All settings for the active custom selection mode.
     *
     * @constructor
     */
    function CustomSelectDescriptor(docView, id) {

        this.id = id;
        // create a new deferred object representing the custom selection mode
        this.deferred = docView.createDeferred('CustomSelectDescriptor', { background: true });

    } // class CustomSelectDescriptor

    // class FrameSelectDescriptor ============================================

    /**
     * All settings for the active frame selection mode.
     *
     * @constructor
     */
    function FrameSelectDescriptor(docView, id, options) {

        this.id = id;
        // create a new deferred object representing the frame selection mode
        this.deferred = docView.createDeferred('FrameSelectDescriptor', { background: true });
        // renderer callback function for frame contents
        this.renderer = Utils.getFunctionOption(options, 'renderer', null);
        // options to be passed to DrawingUtils.getTrackingRectangle()
        this.trackingOptions = Utils.getObjectOption(options, 'trackingOptions', null);
        // the frame rectangle currently selected
        this.rectangle = null;

    } // class FrameSelectDescriptor

    // mix-in class SelectionMixin ============================================

    /**
     * Mix-in class for the class SpreadsheetView that provides methods to work
     * with the cell and drawing selection.
     *
     * @constructor
     */
    function SelectionMixin(rootNode) {

        // self reference
        var self = this;

        // the spreadsheet document model
        var docModel = this.getDocModel();
        // the text selection engine (from text framework)
        var textSelection = docModel.getSelection();

        // models of the active sheet
        var sheetModel = null;
        var colCollection = null;
        var rowCollection = null;
        var cellCollection = null;
        var mergeCollection = null;
        var drawingCollection = null;

        // callback handlers to override cell selection behavior
        var getCellSelectionHandler = null;
        var setCellSelectionHandler = null;

        // specifies whether selection tracking is currently active (between 'select:start' and 'select:end')
        var trackingActive = false;

        // settings for current custom selection mode
        var customSelectDesc = null;

        // settings for current frame selection mode
        var frameSelectDesc = null;

        // the unique identifier of highlighting mode for selected chart
        var chartHighlightUid = null;

        // private methods ----------------------------------------------------

        /**
         * Returns a reference to the original selection from the active sheet.
         */
        function getSheetSelectionAttribute() {
            // bug 57991: sheet model is not available after document import error
            return sheetModel ? sheetModel.getViewAttribute('selection') : new SheetSelection();
        }

        /**
         * Changes the selection of the active sheet.
         */
        function setSheetSelectionAttribute(selection) {
            sheetModel.setViewAttribute('selection', selection);
        }

        /**
         * Returns a reference to the active selection of the document.
         */
        function getActiveSelectionAttribute() {
            return sheetModel.getViewAttribute('activeSelection');
        }

        /**
         * Changes the active selection of the document.
         */
        function setActiveSelectionAttribute(selection) {
            sheetModel.setViewAttribute('activeSelection', selection);
            return true; // required for usage in registerCellSelectionHandlers()
        }

        /**
         * Triggers the 'active:selection' event with the effective selection.
         */
        function triggerActiveSelection() {
            var activeSelection = getActiveSelectionAttribute() || getSheetSelectionAttribute();
            self.trigger('active:selection', activeSelection, trackingActive);
        }

        /**
         * Changes the state of the tracking mode, and triggers an
         * 'active:selection' event if necessary.
         */
        function setTrackingActiveMode(newActive) {
            if (trackingActive !== newActive) {
                trackingActive = newActive;
                triggerActiveSelection();
            }
        }

        /**
         * Returns a selection object that is suitable for keyboard navigation.
         * Prefers the registered cell selection handler.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  - {Boolean} [options.allowEmpty=false]
         *      If set to true, and the registered cell selection handler
         *      returns a selection without any ranges (property 'ranges' set
         *      to an empty array), this selection object will be returned
         *      unmodified. Otherwise, an empty selection returned by that
         *      handler will be replaced by a selection consisting of the
         *      current active cell.
         *
         * @returns {SheetSelection}
         *  The active sheet selection.
         */
        function getEffectiveSelection(options) {

            // the custom selection provided by the registered cell selection handler
            var selection = getCellSelectionHandler ? getCellSelectionHandler.call(self) : null;

            if (!(selection instanceof SheetSelection)) {
                // no custom selection: use real selection of this sheet
                selection = self.getSelection();
            } else if (selection.ranges.empty() && !Utils.getBooleanOption(options, 'allowEmpty', false)) {
                // replace empty custom selection with active cell
                selection = self.getActiveCellAsSelection();
            } else {
                // bug 51024: do not modify original selection object in-place
                selection = selection.clone();
            }

            return selection;
        }

        /**
         * Changes the current selection of the active sheet. Prefers the
         * registered cell selection handler.
         *
         * @param {SheetSelection} selection
         *  The new sheet selection.
         */
        function setEffectiveSelection(selection) {

            // prefer the registered cell selection handler
            var result = setCellSelectionHandler && setCellSelectionHandler.call(self, selection);
            if (result === true) { return; }

            // leave text edit mode before changing the selection
            self.leaveTextEditMode().done(function () {
                setSheetSelectionAttribute(selection);
            });
        }

        /**
         * Leaves the custom selection mode, if it has been started with the
         * method SelectionMixin.enterCustomSelectionMode() before.
         *
         * @param {Boolean} apply
         *  If set to true, the promise representing the custom selection mode
         *  will be resolved with the current custom selection. Otherwise, the
         *  promise will be rejected (custom selection mode canceled).
         */
        function leaveCustomSelectionMode(apply) {

            // custom selection mode must be active
            if (!customSelectDesc) { return; }

            // the selection to be passed to the deferred object
            var selection = apply ? getActiveSelectionAttribute() : null;
            // local copy of the deferred object, to prevent overwriting recursively from handlers
            var selectionDef = customSelectDesc.deferred;

            // reset all settings for custom selection mode before resolving the deferred object
            customSelectDesc = null;
            self.unregisterCellSelectionHandlers();
            self.setStatusLabel(null);
            setActiveSelectionAttribute(null);

            // finally, resolve or reject the deferred object
            if (selection) {
                selectionDef.resolve(selection);
            } else {
                selectionDef.reject();
            }
        }

        /**
         * Switches the view to a special mode where selecting the next cell
         * range in the document will be displayed as 'active selection'. See
         * public method SelectionMixin.enterCustomSelectionMode() for details.
         */
        function enterCustomSelectionMode(id, options) {

            // store all important data for custom selection mode
            customSelectDesc = new CustomSelectDescriptor(self, id);
            var selectionDef = customSelectDesc.deferred;

            // set the initial selection
            setActiveSelectionAttribute(Utils.getObjectOption(options, 'selection', null));

            // register custom selection handler to draw the active selection while user selects
            self.registerCellSelectionHandlers(getActiveSelectionAttribute, setActiveSelectionAttribute);

            // set status label text
            self.setStatusLabel(Utils.getOption(options, 'statusLabel', null));

            // cancel custom selection mode when losing edit rights, or before applying any operations
            self.listenToWhile(selectionDef, self.getApp(), 'docs:editmode:leave', function () { leaveCustomSelectionMode(false); });
            self.listenToWhile(selectionDef, docModel, 'operations:before', function () { leaveCustomSelectionMode(false); });

            return selectionDef.promise();
        }

        /**
         * Leaves the frame selection mode, if it has been started with the
         * method SelectionMixin.enterFrameSelectionMode() before.
         *
         * @param {Boolean} apply
         *  If set to true, the promise representing the frame selection mode
         *  will be resolved with the current frame rectangle. Otherwise, the
         *  promise will be rejected (frame selection mode canceled).
         */
        function leaveFrameSelectionMode(apply) {

            // frame selection mode must be active
            if (!frameSelectDesc) { return; }

            // the selection to be passed to the deferred object
            var frameRect = apply ? frameSelectDesc.rectangle : null;
            // local copy of the deferred object, to prevent overwriting recursively from handlers
            var selectionDef = frameSelectDesc.deferred;

            // reset all settings for frame selection mode before resolving the deferred object
            frameSelectDesc = null;
            self.setStatusLabel(null);

            // notify renderer listeners
            self.trigger('change:frameselect', null);

            // finally, resolve or reject the deferred object
            if (frameRect) {
                selectionDef.resolve(frameRect);
            } else {
                selectionDef.reject();
            }
        }

        /**
         * Event handler for 'mousemove' events used to show the initial
         * cross-hair in frame selection mode, before frame selection tracking
         * actually starts.
         */
        function frameSelectMouseMoveHandler(event) {
            var gridPane = self.getGridPaneForEvent(event);
            var sheetOffset = gridPane ? gridPane.getSheetOffsetForEvent(event, { restricted: true }) : null;
            var frameRect = sheetOffset ? new Rectangle(sheetOffset.left, sheetOffset.top, 0, 0) : null;
            self.trigger('change:frameselect', frameRect);
        }

        /**
         * Shows a cross-hair in all grid panes at the current mouse position.
         */
        function startFrameSelectMouseMove() {
            self.listenTo($(document), 'mousemove', frameSelectMouseMoveHandler);
        }

        /**
         * Hides the cross-hair in all grid panes shown at the current mouse
         * position.
         */
        function stopFrameSelectMouseMove() {
            self.stopListeningTo($(document), 'mousemove', frameSelectMouseMoveHandler);
        }

        /**
         * Switches the view to a special seelction mode that allows to select
         * an arbitrary pixel-exact rectangle in the active sheet. See public
         * method SelectionMixin.enterFrameSelectionMode() for details.
         */
        function enterFrameSelectionMode(id, options) {

            // whether the frame selection mode is supported in read-only mode
            var readOnly = Utils.getBooleanOption(options, 'readOnly', false);
            if (!readOnly && !self.isEditable()) { return new $.Deferred().reject(); }

            // store all important data for frame selection mode
            frameSelectDesc = new FrameSelectDescriptor(self, id, options);
            var selectionDef = frameSelectDesc.deferred;

            // set status label text
            self.setStatusLabel(Utils.getOption(options, 'statusLabel', null));

            // show a cross-hair at the current mouse position
            startFrameSelectMouseMove();
            selectionDef.always(stopFrameSelectMouseMove);

            // cancel frame selection mode when document loses edit rights
            if (!readOnly) {
                self.listenToWhile(selectionDef, self.getApp(), 'docs:editmode:leave', function () {
                    leaveFrameSelectionMode(false);
                });
            }

            return selectionDef.promise();
        }

        /**
         * Handles all tracking events for frame selection mode.
         */
        var frameSelectTrackingHandler = (function () {

            // tracking start position (absolute position in the sheet, in pixels)
            var anchorX = 0, anchorY = 0;
            // sheet area as bounding rectangle
            var boundRect = null;
            // whether the mouse or touch point has really been moved (with threshold distance)
            var mouseMoved = false;

            // initializes the frame selection mode according to the passed tracking event
            function initializeTracking(event, gridPane) {

                // get the start position for the rectangle in the sheet
                var anchorOffset = gridPane.getSheetOffsetForEvent(event, { restricted: true });
                if (!anchorOffset || !frameSelectDesc) {
                    event.cancelTracking();
                    return;
                }

                // extract the anchor coordinates
                anchorX = anchorOffset.left;
                anchorY = anchorOffset.top;
                // restrict the frame rectangle to the sheet area
                boundRect = self.getSheetRectangle();
                // the initial frame rectangle with zero size
                frameSelectDesc.rectangle = new Rectangle(anchorOffset.left, anchorOffset.top, 0, 0);
                // start tracking after a specific initial move distance
                mouseMoved = false;

                // stop listening to mousemove events for the cross-hair
                stopFrameSelectMouseMove();
            }

            // updates the frame rectangle according to the passed tracking event
            function updateTracking(event, gridPane) {

                // current tracking position (absolute sheet position, in pixels)
                var currOffset = gridPane.getSheetOffsetForEvent(event);
                // options for the tracking helper
                var trackingOptions = _.clone(frameSelectDesc.trackingOptions);
                trackingOptions.modifierEvent = event;
                trackingOptions.boundRect = boundRect;
                // the current location of the selected frame rectangle
                var frameRect = DrawingUtils.getTrackingRectangle(anchorX, anchorY, currOffset.left, currOffset.top, trackingOptions);

                // start tracking after a threshold of 3 pixels
                mouseMoved = mouseMoved || (frameRect.origShift >= 3);
                if (mouseMoved) {
                    frameSelectDesc.rectangle = frameRect;
                    self.trigger('change:frameselect', frameRect);
                }
            }

            // return the actual frameSelectTrackingHandler() method
            return function (event, endPromise, gridPane) {

                switch (event.type) {
                    case 'tracking:start':
                        initializeTracking(event, gridPane);
                        break;
                    case 'tracking:move':
                        updateTracking(event, gridPane);
                        break;
                    case 'tracking:scroll':
                        gridPane.scrollRelative(event.scrollX, event.scrollY);
                        updateTracking(event, gridPane);
                        break;
                    case 'tracking:end':
                    case 'tracking:cancel':
                        endPromise.always(leaveFrameSelectionMode);
                        break;
                }
            };
        }()); // end of frameSelectTrackingHandler() local scope

        /**
         * Returns the address of the cell that will be selected with the HOME
         * key. In unfrozen views, this is always cell A1. In frozen views,
         * this is the first available cell in the unfrozen bottom-right grid
         * pane.
         *
         * @returns {Address}
         *  The address of the cell to be selected with the HOME key.
         */
        function getHomeCell() {
            var frozenSplit = sheetModel.hasFrozenSplit();
            var colInterval = frozenSplit ? sheetModel.getSplitColInterval() : null;
            var rowInterval = frozenSplit ? sheetModel.getSplitRowInterval() : null;
            return new Address(colInterval ? (colInterval.last + 1) : 0, rowInterval ? (rowInterval.last + 1) : 0);
        }

        /**
         * Returns the address of the cell that will be selected with the END
         * key (the bottom-right cell in the used area of the sheet).
         *
         * @returns {Address}
         *  The address of the cell to be selected with the END key.
         */
        function getEndCell() {
            var usedRange = sheetModel.getUsedRange();
            return usedRange ? usedRange.end : Address.A1;
        }

        /**
         * Expands the passed range address so that it completely includes all
         * merged ranges it currently overlaps. In case the passed range is a
         * full column or full row range, it will be expanded to full-column or
         * full-row merged ranges only.
         *
         * @param {Range} range
         *  A cell range address.
         *
         * @returns {Range}
         *  The address of the expanded cell range.
         */
        function expandRangeToMergedRanges(range) {

            // full column/row ranges: expand to full-size merged ranges only (bug 33313)
            if (docModel.isColRange(range) || docModel.isRowRange(range)) {
                return mergeCollection.getBoundingMergedRange(range) || range;
            }

            // expand other ranges to all the merged ranges it overlaps
            return mergeCollection.expandRangeToMergedRanges(range);
        }

        /**
         * Clears the current selection and selects the specified cell. If the
         * cell is hidden, tries to select the nearest visible cell.
         */
        function selectVisibleCell(address) {
            if ((address = sheetModel.getVisibleCell(address, 'prevNext'))) {
                self.selectCell(address);
                self.scrollToCell(address);
            }
        }

        /**
         * Expands the active range of the current selection to the passed cell
         * address.
         */
        function expandToVisibleCell(address, shrinkEnd) {

            // current selection in the active sheet
            var selection = getEffectiveSelection();
            // the active range in the selection
            var activeRange = selection.activeRange();

            if ((address = sheetModel.getVisibleCell(address, 'prevNext'))) {

                // start position of active range: extend to passed address,
                // if it is located before, but do not shrink the active range
                activeRange.start[0] = Math.min(address[0], activeRange.start[0]);
                activeRange.start[1] = Math.min(address[1], activeRange.start[1]);

                // end position of active range: extend to passed address, if it
                // is located after the range, but shrink to active cell if not
                activeRange.end[0] = Math.max(address[0], shrinkEnd ? selection.address[0] : activeRange.end[0]);
                activeRange.end[1] = Math.max(address[1], shrinkEnd ? selection.address[1] : activeRange.end[1]);

                // commit new selection, adjust scrolling position
                self.changeActiveRange(activeRange);
                self.scrollToCell(address);
            }
        }

        /**
         * Searches for the next/last filled cell in the current cell
         * collection.
         */
        function findNextUsedCell(address, direction) {

            // whether to move the active cell forwards (right or down)
            var forward = (direction === 'right') || (direction === 'down');
            // whether to move the active cell horizontally
            var columns = (direction === 'left') || (direction === 'right');
            // the resulting address
            var resultAddress = _.clone(address);

            // range address of the used area
            var usedRange = cellCollection.getUsedRange();
            var lastCell = usedRange ? usedRange.end : Address.A1;

            // maximum column/row index in the sheet (in move direction)
            var maxIndex = docModel.getMaxIndex(columns);
            // column/row index of the start cell
            var startIndex = address.get(columns);
            // column/row index of last visited cell
            var lastIndex = null;

            // forward mode: jump to end of sheet, if start cell is outside used area,
            // also if inside the last column/row of the used area, according to direction
            if (forward && ((address.get(!columns) > lastCell.get(!columns)) || (address.get(columns) >= lastCell.get(columns)))) {
                resultAddress.set(maxIndex, columns);
                return resultAddress;
            }

            // backward mode: jump to beginning of sheet, if start cell is outside
            // used area, and the new address will also be outside the used area
            if (!forward && (address.get(!columns) > lastCell.get(!columns))) {
                resultAddress.set(0, columns);
                return resultAddress;
            }

            // find next available content cell in the cell collection
            var addressIt = cellCollection.createLinearAddressIterator(address, direction, { type: 'value', visible: true });
            Iterator.forEach(addressIt, function (contentAddress) {

                // column/row index of the current cell
                var currIndex = contentAddress.get(columns);

                // started from an empty cell: stick to first content cell
                if (!_.isNumber(lastIndex) && (startIndex !== currIndex)) {
                    resultAddress.set(currIndex, columns);
                    return Utils.BREAK;
                }

                // found a content cell following an empty cell
                if (_.isNumber(lastIndex) && (Math.abs(currIndex - lastIndex) > 1)) {

                    // use the current cell, if iteration has been started at the cell
                    // before the gap; otherwise use the cell preceding the gap
                    resultAddress.set((startIndex === lastIndex) ? currIndex : lastIndex, columns);
                    return Utils.BREAK;
                }

                // remember address of current content cell for next iteration
                lastIndex = currIndex;
            });

            // jump to last found content cell if it is not the start cell;
            // or to sheet boundaries, if no valid content cell could be found
            if (address.equals(resultAddress)) {
                resultAddress.set(_.isNumber(lastIndex) && (lastIndex !== startIndex) ? lastIndex : forward ? maxIndex : 0, columns);
            }

            return resultAddress;
        }

        /**
         * Clears the current selection and moves the active cell into the
         * specified direction. If the target cell is hidden, moves the cursor
         * to the next available visible cell.
         *
         * @param {String} direction
         *  The direction the active cell will be moved to. Supported values
         *  are 'left', 'right', 'up', and 'down'.
         *
         * @param {String} mode
         *  The move mode. Supported values are:
         *  - 'cell':
         *      Moves the active cell to the next visible cell.
         *  - 'page':
         *      Moves the active cell by the width/height of the visible area,
         *      and adjusts the scroll position in a way that the new active
         *      cell remains on the same screen position if possible.
         *  - 'used':
         *      Moves the active cell to the next available non-empty cell; or,
         *      if none available, to the first or last cell in the sheet.
         */
        function selectNextCell(direction, mode) {

            // current selection in the active sheet
            var selection = getEffectiveSelection();
            // the cell to start movement from
            var activeCell = selection.origin || selection.address;
            // the merged range covering the active cell
            var mergedRange = mergeCollection.getMergedRange(activeCell);
            // whether to move the active cell forwards (right or down)
            var forward = (direction === 'right') || (direction === 'down');
            // whether to move the active cell horizontally
            var columns = (direction === 'left') || (direction === 'right');
            // the column/row collection in the move direction
            var collection = columns ? colCollection : rowCollection;

            switch (mode) {
                case 'cell':
                    (function () {

                        // start from leading/trailing border of a merged range
                        var index = !mergedRange ? activeCell.get(columns) : forward ? mergedRange.getEnd(columns) : mergedRange.getStart(columns);
                        index += (forward ? 1 : -1);

                        // insert new column/row entry (old selection will collapse to active cell, if no other cell has been found)
                        var entryData = collection.getVisibleEntry(index, forward ? 'next' : 'prev');
                        if (entryData) { activeCell.set(entryData.index, columns); }

                        // set selection to the new active cell, scroll to the cell
                        self.selectCell(activeCell, { storeOrigin: true });
                        self.scrollToCell(activeCell);
                    }());
                    break;

                case 'page':
                    (function () {

                        // the active (focused) grid pane
                        var gridPane = self.getActiveGridPane();
                        // the associated header pane in moving direction
                        var headerPane = gridPane.getHeaderPane(columns);

                        // bug 37023: correctly jump out of a frozen pane
                        if (forward && headerPane.isFrozen()) {
                            activeCell.set(headerPane.getAvailableInterval().last + 1, columns);
                            self.selectCell(activeCell, { storeOrigin: true });
                            self.scrollToCell(activeCell);
                            return;
                        }

                        // get position and size of the active cell
                        var cellRect = mergedRange ? self.getRangeRectangle(mergedRange) : self.getCellRectangle(activeCell);
                        var cellOffset = cellRect[columns ? 'left' : 'top'];
                        var cellSize = cellRect[columns ? 'width' : 'height'];
                        // get position and size of the visible area in this grid pane
                        var visibleRect = gridPane.getVisibleRectangle();
                        var visibleOffset = visibleRect[columns ? 'left' : 'top'];
                        var visibleSize = visibleRect[columns ? 'width' : 'height'];
                        // the new column/row entry to move to
                        var entryData = null;

                        // returns the descriptor of a new entry when moving in 'page' mode
                        function getPageTargetEntry(offset) {
                            return collection.getEntryByOffset(offset + (forward ? visibleSize : -visibleSize) * 0.98, { pixel: true });
                        }

                        // changes the column/row index of the active cell according to move direction without scrolling
                        function selectCellForPage(newIndex) {
                            var address = activeCell.clone();
                            address.set(newIndex, columns);
                            self.selectCell(address, { storeOrigin: true });
                            return self.grabFocus({ target: address });
                        }

                        // check if the active cell is located inside the visible area
                        if ((visibleOffset <= cellOffset) && (cellOffset + cellSize <= visibleOffset + visibleSize)) {
                            // find the column/row one page away
                            entryData = getPageTargetEntry(cellOffset + cellSize / 2);
                            selectCellForPage(entryData.index);
                            headerPane.scrollRelative(entryData.offset - cellOffset);
                        } else {
                            // otherwise, scroll one page and select the top visible entry
                            entryData = getPageTargetEntry(visibleOffset);
                            selectCellForPage(entryData.index);
                            headerPane.scrollTo(entryData.offset);
                        }
                    }());
                    break;

                case 'used':
                    (function () {

                        // search for a target cell
                        var targetCell = findNextUsedCell(activeCell, direction);

                        // fail-safety: if cell address could not be determined or did not change, make a single cell step
                        if (!targetCell || activeCell.equals(targetCell)) {
                            return selectNextCell(direction, 'cell');
                        }

                        // set selection to the new active cell
                        self.selectCell(targetCell, { storeOrigin: true });
                        self.scrollToCell(targetCell);
                    }());
                    break;
            }
        }

        /**
         * Extends the active range in the current selection into the specified
         * direction. If the target column/row is hidden, extends the active
         * range to the next available visible column/row.
         *
         * @param {String} direction
         *  The direction the active cell will be moved to. Supported values
         *  are 'left', 'right', 'up', and 'down'.
         *
         * @param {String} mode
         *  The move mode. Supported values are:
         *  - 'cell':
         *      Expands the active range to the next visible column/row.
         *  - 'page':
         *      Expands the active range by the width/height of the visible
         *      area, and adjusts the scroll position in a way that the range
         *      border remains on the same screen position if possible.
         *  - 'used':
         *      Expands the active range to the next available non-empty cell;
         *      or, if none available, to the first or last column/row in the
         *      sheet.
         */
        function expandToNextCell(direction, mode) {

            // current selection in the active sheet
            var selection = getEffectiveSelection();
            // the cell to start movement from
            var activeCell = (selection.origin || selection.address).clone();
            // the current active range address
            var activeRange = selection.activeRange();
            // the merged range covering the active cell, or the active cell as range object
            var mergedRange = mergeCollection.getMergedRange(activeCell) || Range.createFromAddresses(activeCell);
            // the new active range address
            var newRange = null;

            // whether to expand the active range forwards (right or down)
            var forward = (direction === 'right') || (direction === 'down');
            // whether to expand the active range horizontally
            var columns = (direction === 'left') || (direction === 'right');

            // the column/row collection in the move direction
            var collection = columns ? colCollection : rowCollection;
            // the active (focused) grid pane
            var gridPane = self.getActiveGridPane();

            // the new column/row entry to move to
            var entryData = null;

            // returns the descriptor of a new entry when moving in 'page' mode
            function getPageTargetEntry(index) {

                // the descriptor for the start entry
                var entryData = collection.getEntry(index);
                // the position in the middle of the entry
                var entryOffset = entryData.offset + entryData.size / 2;
                // the size of the visible area, in pixels
                var visibleSize = gridPane.getVisibleRectangle()[columns ? 'width' : 'height'];

                // return the target entry
                return collection.getEntryByOffset(entryOffset + (forward ? visibleSize : -visibleSize) * 0.98, { pixel: true });
            }

            // returns the descriptor of a new entry when moving in 'used' mode
            function getUsedTargetEntry(address) {
                var targetCell = findNextUsedCell(address, direction);
                return targetCell ? collection.getVisibleEntry(targetCell.get(columns), forward ? 'next' : 'prev') : null;
            }

            // first, try to shrink the range at the correct border according to move direction
            activeCell.set(forward ? activeRange.getStart(columns) : activeRange.getEnd(columns), columns);
            switch (mode) {
                case 'cell':
                    entryData = forward ?
                        collection.getVisibleEntry(activeCell.get(columns) + 1, 'next') :
                        collection.getVisibleEntry(activeCell.get(columns) - 1, 'prev');
                    break;

                case 'page':
                    entryData = getPageTargetEntry(activeCell.get(columns));
                    break;

                case 'used':
                    entryData = getUsedTargetEntry(activeCell);
                    break;
            }

            // build the new range address according to the collection entry
            if (entryData && (forward ? (entryData.index <= mergedRange.getStart(columns)) : (mergedRange.getEnd(columns) <= entryData.index))) {
                newRange = activeRange.clone();
                (forward ? newRange.start : newRange.end).set(entryData.index, columns);
                // shrink the range until it does not contain any merged ranges partly
                newRange = mergeCollection.shrinkRangeFromMergedRanges(newRange, columns);
            }

            // if shrinking the range is not possible or has failed, expand the range away from active cell
            if (!newRange) {
                activeCell.set(forward ? activeRange.getEnd(columns) : activeRange.getStart(columns), columns);

                switch (mode) {
                    case 'cell':
                        entryData = forward ?
                            collection.getVisibleEntry(activeCell.get(columns) + 1, 'next') :
                            collection.getVisibleEntry(activeCell.get(columns) - 1, 'prev');
                        break;

                    case 'page':
                        entryData = getPageTargetEntry(activeCell.get(columns));
                        break;

                    case 'used':
                        entryData = getUsedTargetEntry(activeCell);
                        break;
                }

                // build the new range address according to the collection entry
                if (entryData) {
                    newRange = activeRange.clone();
                    // expand in move direction, restrict to original active cell on the other border
                    (forward ? newRange.end : newRange.start).set(entryData.index, columns);
                    (forward ? newRange.start : newRange.end).set(selection.address.get(columns), columns);
                    // expand the range until it contains all merged ranges completely
                    newRange = mergeCollection.expandRangeToMergedRanges(newRange);
                }
            }

            // change the active range, if a new range has been found
            if (newRange && newRange.differs(activeRange)) {
                self.changeActiveRange(newRange);
            }

            // scroll to target entry (this allows e.g. to scroll inside a large merged range without changing the selection)
            if (entryData) {
                // activate correct grid pane in frozen split mode
                activeCell.set(entryData.index, columns);
                self.grabFocus({ target: activeCell });
                self.getActiveHeaderPane(columns).scrollToEntry(entryData.index);
            }
        }

        /**
         * Expands the selected ranges to entire columns or rows.
         *
         * @param {Boolean} columns
         *  If set to true, the current selection will be expanded to full
         *  columns. If set to false, the current selection will be expanded to
         *  full rows.
         */
        function expandSelectionToSheet(columns) {

            // current selection in the active sheet
            var selection = getEffectiveSelection();
            // the last available column/row index in the sheet
            var lastIndex = docModel.getMaxIndex(!columns);

            // expand all ranges to the sheet borders
            selection.ranges.forEach(function (range, index) {
                range.setStart(0, !columns);
                range.setEnd(lastIndex, !columns);
                selection.ranges[index] = mergeCollection.expandRangeToMergedRanges(range);
            });

            self.setCellSelection(selection, { unique: true });
        }

        /**
         * Selects the content range (the entire range containing consecutive
         * content cells) surrounding the current active cell (ignoring the
         * current selected ranges).
         */
        function selectContentRange() {

            // current active cell in the active sheet
            var activeCell = getEffectiveSelection().address;
            // the content range to be selected
            var contentRange = cellCollection.findContentRange(new Range(activeCell));

            // select the resulting expanded range
            self.selectRange(contentRange, { active: activeCell });
        }

        /**
         * Selects the specified drawing frame in the active sheet, and scrolls
         * the active grid pane to that drawing frame.
         */
        function selectAndScrollToDrawing(position) {
            self.selectDrawing(position);
            self.scrollToDrawingFrame(position);
        }

        /**
         * Creates an iterator for drawing objects that are not hidden by any
         * formatting attributes, nor are located in hidden columns/rows.
         */
        function createVisibleDrawingIterator(options) {

            // create an iterator for visible drawing objects
            var iterator = drawingCollection.createModelIterator(_.extend({ visible: true }, options));

            // filter drawing objects that are located in hidden columns/rows
            return new FilterIterator(iterator, function (drawingModel) {
                return drawingModel.getRectangle() !== null;
            });
        }

        /**
         * Selects the first visible drawing frame in the active sheet.
         */
        function selectFirstDrawing() {

            // find the first visible drawing model located in the sheet
            var iterResult = createVisibleDrawingIterator().next();

            // select the drawing object if available
            if (!iterResult.done) {
                selectAndScrollToDrawing([iterResult.index]);
            }
        }

        /**
         * Selects the previous or next visible drawing frame.
         */
        function selectNextDrawing(reverse) {

            // last drawing selected in the active sheet
            var position = _.last(self.getSelectedDrawings());
            // TODO: selection in embedded objects
            if (!position || (position.length > 1)) { return; }

            // index of the drawing object currently selected
            var index = _.last(position);

            // index of the next drawing object to be selected
            var begin = Utils.mod(index + (reverse ? -1 : 1), drawingCollection.getModelCount());
            var iterResult = createVisibleDrawingIterator({ reverse: reverse, begin: begin }).next();

            // no more drawings available: wrap to first drawing (reverse mode: wrap to last drawing)
            if (iterResult.done) {
                iterResult = createVisibleDrawingIterator({ reverse: reverse, end: index }).next();
            }

            // select the drawing object if available
            if (!iterResult.done) {
                selectAndScrollToDrawing([iterResult.index]);
            }
        }

        /**
         * Selects all visible drawing frames.
         */
        function selectAllDrawings() {

            // iterator that visits all visible drawing objects (TODO: selection in embedded objects)
            var iterator = createVisibleDrawingIterator();
            // the positions of all visible drawing objects
            var positions = Iterator.map(iterator, function (drawingModel) { return drawingModel.getPosition(); });

            // select the drawing objects if available
            if (positions.length > 0) {
                self.setDrawingSelection(positions);
            }
        }

        /**
         * Registers the selected drawing objects with in the selection engine
         * of the text framework.
         */
        function refreshTextSelection() {
            var positions = self.getSelectedDrawings();
            if (positions.length > 0) {
                // add sheet index to the positions, as expected by the text framework
                var sheet = self.getActiveSheet();
                positions.forEach(function (position) { position.unshift(sheet); });
                textSelection.setMultiDrawingSelectionByPosition(positions);
            } else {
                textSelection.clearMultiSelection();
                textSelection.setEmptySelection({ updateFocus: false });
            }
        }

        /**
         * Handles 'keydown' events from all grid panes to adjust the current
         * selection.
         */
        function keyDownHandler(event) {

            // nothing to do in chart sheets etc.
            if (!sheetModel.isCellType()) { return; }

            // special handling for drawing selection
            if (self.hasDrawingSelection()) {

                // traverse through drawing frames with TAB key
                if (KeyCodes.matchKeyCode(event, 'TAB', { shift: null })) {
                    selectNextDrawing(event.shiftKey);
                    return false;
                }

                // select all drawing objects
                if (KeyCodes.matchKeyCode(event, 'A', { ctrlOrMeta: true })) {
                    selectAllDrawings();
                    return false;
                }

                return;
            }

            // TAB/ENTER: move active cell to its neighbor according to the pressed key
            var moveDir = PaneUtils.getCellMoveDirection(event);
            if (moveDir) {
                self.moveActiveCell(moveDir);
                return false;
            }

            // the cell move mode for cursor keys.
            // if OX Viewer uses Spreadsheet, cursor key are handled by OX Viewer
            // and cursor up/down/left/right with CTRL/META are used for simple cell movement.
            var cellMoveMode = self.getApp().isViewerMode() ? 'cell' : 'used';

            // simple movements without modifier keys
            if (KeyCodes.matchModifierKeys(event, { shift: null })) {
                switch (event.keyCode) {
                    case KeyCodes.LEFT_ARROW:
                        // if OX Spreadsheet is plugged into OX Viewer, cursor left/right switches between view items
                        if (!self.getApp().isViewerMode()) {
                            (event.shiftKey ? expandToNextCell : selectNextCell)('left', 'cell');
                            return false;
                        }
                        return true;
                    case KeyCodes.RIGHT_ARROW:
                        // if OX Spreadsheet is plugged into OX Viewer, cursor left/right switches between view items
                        if (!self.getApp().isViewerMode()) {
                            (event.shiftKey ? expandToNextCell : selectNextCell)('right', 'cell');
                            return false;
                        }
                        return true;
                    case KeyCodes.UP_ARROW:
                        // if OX Spreadsheet is plugged into OX Viewer, cursor up/down have no function
                        if (!self.getApp().isViewerMode()) {
                            (event.shiftKey ? expandToNextCell : selectNextCell)('up', 'cell');
                            return false;
                        }
                        return true;
                    case KeyCodes.DOWN_ARROW:
                        // if OX Spreadsheet is plugged into OX Viewer, cursor up/down have no function
                        if (!self.getApp().isViewerMode()) {
                            (event.shiftKey ? expandToNextCell : selectNextCell)('down', 'cell');
                            return false;
                        }
                        return true;
                    case KeyCodes.PAGE_UP:
                        (event.shiftKey ? expandToNextCell : selectNextCell)('up', 'page');
                        return false;
                    case KeyCodes.PAGE_DOWN:
                        (event.shiftKey ? expandToNextCell : selectNextCell)('down', 'page');
                        return false;
                    case KeyCodes.HOME:
                        (event.shiftKey ? expandToVisibleCell : selectVisibleCell)(new Address(getHomeCell()[0], getEffectiveSelection().address[1]), false);
                        return false;
                    case KeyCodes.END:
                        (event.shiftKey ? expandToVisibleCell : selectVisibleCell)(new Address(getEndCell()[0], getEffectiveSelection().address[1]), true);
                        return false;
                }
            }

            // additional selection in page mode with ALT modifier key
            if (KeyCodes.matchModifierKeys(event, { shift: null, alt: true })) {
                switch (event.keyCode) {
                    case KeyCodes.PAGE_UP:
                        (event.shiftKey ? expandToNextCell : selectNextCell)('left', 'page');
                        return false;
                    case KeyCodes.PAGE_DOWN:
                        (event.shiftKey ? expandToNextCell : selectNextCell)('right', 'page');
                        return false;
                }
            }

            // additional selection with CTRL or META modifier key
            if (KeyCodes.matchModifierKeys(event, { shift: null, ctrlOrMeta: true })) {
                switch (event.keyCode) {
                    case KeyCodes.LEFT_ARROW:
                        (event.shiftKey ? expandToNextCell : selectNextCell)('left',  cellMoveMode);
                        return false;
                    case KeyCodes.RIGHT_ARROW:
                        (event.shiftKey ? expandToNextCell : selectNextCell)('right', cellMoveMode);
                        return false;
                    case KeyCodes.UP_ARROW:
                        (event.shiftKey ? expandToNextCell : selectNextCell)('up',    cellMoveMode);
                        return false;
                    case KeyCodes.DOWN_ARROW:
                        (event.shiftKey ? expandToNextCell : selectNextCell)('down',  cellMoveMode);
                        return false;
                    case KeyCodes.HOME:
                        (event.shiftKey ? expandToVisibleCell : selectVisibleCell)(getHomeCell(), false);
                        return false;
                    case KeyCodes.END:
                        (event.shiftKey ? expandToVisibleCell : selectVisibleCell)(getEndCell(), true);
                        return false;
                }
            }

            // CTRL+asterisk (numeric pad): select content range around active cell
            if (KeyCodes.matchKeyCode(event, 'NUM_MULTIPLY', { ctrlOrMeta: true })) {
                selectContentRange();
                return false;
            }

            // bug 31247: remaining shortcuts not in cell edit mode
            if (self.isTextEditMode('cell')) { return; }

            // convert selection to entire columns
            if (KeyCodes.matchKeyCode(event, 'SPACE', { ctrlOrMeta: true })) {
                expandSelectionToSheet(true);
                return false;
            }

            // convert selection to entire rows
            if (KeyCodes.matchKeyCode(event, 'SPACE', { shift: true })) {
                expandSelectionToSheet(false);
                return false;
            }

            // select entire sheet
            if (KeyCodes.matchKeyCode(event, 'SPACE', { shift: true, ctrlOrMeta: true }) || KeyCodes.matchKeyCode(event, 'A', { ctrlOrMeta: true })) {
                self.selectRange(docModel.getSheetRange(), { active: self.getActiveCell() });
                return false;
            }

            // SHIFT+F4: select the first visible drawing frame
            if (KeyCodes.matchKeyCode(event, 'F4', { shift: true })) {
                selectFirstDrawing();
                return false;
            }
        }

        /**
         * Registers a listener at the specified object. Execution of the event
         * handlers will be deferred until the browser focus is located inside
         * the application pane.
         *
         * @param {Object} source
         *  The event source object. Can be any object that provides a method
         *  'on()' to register an event listener.
         *
         * @param {String} type
         *  The type of the event to listen to. Can be a space-separated list
         *  of event type names.
         *
         * @param {Function} listener
         *  The event listener that will be invoked for the events triggered by
         *  the event source object.
         */
        var listenToWithFocus = (function () {

            // This method is a fix for bug 30513: when entering something into a text field
            // (e.g. a new font size), and directly clicking into a new cell without pressing
            // ENTER key, the following events have been occurred before:
            // 1) 'mousedown' in the grid pane will start to select the clicked cell,
            // 2) 'blur' in the text field will trigger the action attached to the text field.
            // This results in setting the font size to the wrong (now selected) cell.
            //
            // With this method, all selection events triggered by the grid panes, header
            // panes, and corner pane will be deferred until the browser focus returns to the
            // application pane (i.e. any grid pane therein), or to the text area (also in the
            // formula pane), if cell text edit mode is currently active.

            // returns a promise that will be resolved when the application pane has the focus,
            // or null, if the browser focus is already in the application pane
            function waitForViewFocus() {

                // the focus target node of the current text edit mode
                var textFocusNode = self.getTextEditFocusNode();
                if (textFocusNode && (Utils.getActiveElement() === textFocusNode)) { return null; }

                // check if the application pane is focused
                if (self.hasAppFocus()) { return null; }

                // the deferred object that will be resolved after the focus has returned into the application pane
                var deferred = Scheduler.createDeferred(self, 'SelectionMixin.listenToWithFocus.waitForViewFocus');
                // fail-safety (prevent deferring selection for too long): timeout after 200ms
                var promise = self.createAbortablePromise(deferred, null, 200);

                // let the focus event bubble up, then resolve the deferred object
                function focusHandler() {
                    _.defer(function () {
                        deferred.resolve();
                    });
                }

                // listen to the next 'focusin' event at the application pane
                var appPaneNode = self.getAppPaneNode();
                self.listenOnceToWhile(promise, appPaneNode, 'focusin', focusHandler);
                if (textFocusNode) {
                    self.listenOnceToWhile(promise, $(textFocusNode), 'focusin', focusHandler);
                }

                return promise;
            }

            // create a single synchronized method that checks the browser focus (and
            // waits for it if necessary), and invokes the passed event handler afterwards
            var synchronizedHandler = self.createSynchronizedMethod('SelectionMixin.listenToWithFocus.synchronizedHandler', function (handler, args) {

                // store calling context for usage in always() handler
                var context = this;
                // wait for the browser focus to return to the application pane
                var promise = waitForViewFocus();

                // nothing to wait for: immediately invoke the event handler
                if (!promise) {
                    handler.apply(context, args);
                    return;
                }

                // wait for focus, then invoke the event handler, return the promise to make the synchronized method working
                return promise.always(function () {
                    handler.apply(context, args);
                });
            });

            // the actual listenToWithFocus() method
            function listenToWithFocus(source, type, handler) {
                // listen to the event and invoke the synchronized method (pass the real event handler as argument)
                self.listenTo(source, type, function () {
                    synchronizedHandler.call(this, handler, arguments);
                });
            }

            return listenToWithFocus;
        }());

        // public methods -----------------------------------------------------

        /**
         * Registers a callback function that can change the behavior of the
         * application when selecting cell ranges with mouse or keyboard.
         *
         * @param {Function} getHandler
         *  A callback function that will be invoked to receive a different
         *  cell selection to be used for all selection code, especially
         *  changing the selection with keyboard shortcuts. Will be called in
         *  the context of this instance. Must return a selection object, or
         *  null. The latter indicates to use the actual cell selection of the
         *  active sheet.
         *
         * @param {Function} setHandler
         *  A callback function that will be invoked before actually changing
         *  the current cell selection in the active sheet. Will be called in
         *  the context of this instance. Receives the new selection intended
         *  to be set at the active sheet as first parameter. If this function
         *  has processed the selection by itself, it has to return true to
         *  prevent changing the cell selection in the active sheet.
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.registerCellSelectionHandlers = function (getHandler, setHandler) {
            getCellSelectionHandler = getHandler;
            setCellSelectionHandler = setHandler;
            return this;
        };

        /**
         * Unregisters the callback functions that have been registered with
         * the method SelectionMixin.registerCellSelectionHandlers().
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.unregisterCellSelectionHandlers = function () {
            getCellSelectionHandler = setCellSelectionHandler = null;
            return this;
        };

        /**
         * Switches the view to a special mode where selecting the next cell
         * range in the document will be displayed as 'active selection' (with
         * a marching ants effect), and that can be handled afterwards via the
         * promise returned by this method.
         *
         * @param {String} id
         *  An arbitrary name that identifies the custom selection mode that is
         *  currently active.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {SheetSelection} [options.selection]
         *      The initial selection shown when starting the custom selection
         *      mode.
         *  - {String|Object} [options.statusLabel]
         *      The status label to be shown in the status pane while custom
         *      selection mode is active. Can be a string, or a caption options
         *      object with label and icon settings.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved, if the custom selection mode has
         *  finished successfully; or that will be rejected, if the custom
         *  selection mode has been canceled. If the custom selection mode has
         *  been completed successfully, the promise's done handlers receive
         *  the selection object as first parameter.
         */
        this.enterCustomSelectionMode = function (id, options) {

            // cancel previous selection modes if active
            leaveCustomSelectionMode(false);
            leaveFrameSelectionMode(false);

            // initialize custom selection mode
            return enterCustomSelectionMode(id, options);
        };

        /**
         * Returns whether the specified custom selection mode is currently
         * active.
         *
         * @param {String} [id]
         *  An identifier for the custom selection mode, as passed to the
         *  method SelectionMixin.enterCustomSelectionMode(). If omitted, this
         *  method returns whether any custom selection mode is active.
         *
         * @returns {Boolean}
         *  Whether the specified custom selection mode is currently active.
         */
        this.isCustomSelectionMode = function (id) {
            return !!customSelectDesc && (!id || (customSelectDesc.id === id));
        };

        /**
         * Cancels the custom selection mode that has been started with the
         * method SelectionMixin.enterCustomSelectionMode() before.
         *
         * @returns {Boolean}
         *  Whether the custom selection mode was active and has been left.
         */
        this.cancelCustomSelectionMode = function () {
            if (customSelectDesc) {
                leaveCustomSelectionMode(false);
                return true;
            }
            return false;
        };

        /**
         * Switches the view to a special selection mode that allows to select
         * an arbitrary pixel-exact rectangle in the active sheet.
         *
         * @param {String} id
         *  An arbitrary name that identifies the frame selection mode that is
         *  currently active.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Function} [options.readOnly=false]
         *      If set to true, the frame selection mode can be activated while
         *      the document is in read-only mode; or remains active, when the
         *      document switches to read-only mode. By default, the frame
         *      selection mode will be canceled immediately when the read-only
         *      mode will be activated.
         *  - {String|Object} [options.statusLabel]
         *      The status label to be shown in the status pane while the frame
         *      selection mode is active. Can be a string, or a caption options
         *      object with label and icon settings.
         *  - {Number} [options.aspectRatio=1]
         *      The aspect ratio to be used for locked aspect mode (pressed
         *      SHIFT key), as quotient of width and height (a value greater
         *      than 1 will create a wide rectangle). MUST be a positive
         *      floating-point number.
         *  - {Function} [options.renderer]
         *      A callback function that can be used to render something into
         *      the selected frame rectangle while tracking mode is active.
         *      Receives the following parameters:
         *      (1) {jQuery} frameNode
         *          The DOM node representing the selected frame rectangle, as
         *          jQuery collection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved, if the frame selection mode has
         *  finished successfully; or that will be rejected, if the frame
         *  selection mode has been canceled. If the frame selection mode has
         *  been completed successfully, the promise's done handlers receive
         *  the selected sheet rectangle (instance of class Rectangle, in
         *  pixels) as first parameter, with the additional boolean properties
         *  'reverseX' and 'reverseY' according to the location of the tracking
         *  start point and tracking end point.
         */
        this.enterFrameSelectionMode = function (id, options) {

            // cancel previous selection modes if active
            leaveCustomSelectionMode(false);
            leaveFrameSelectionMode(false);

            // initialize frame selection mode
            return enterFrameSelectionMode(id, options);
        };

        /**
         * Returns whether the specified free-hand frame selection mode is
         * currently active.
         *
         * @param {String} [id]
         *  An identifier for the frame selection mode, as passed to the method
         *  SelectionMixin.enterFrameSelectionMode(). If omitted, this method
         *  returns whether any frame selection mode is active.
         *
         * @returns {Boolean}
         *  Whether the specified free-hand frame selection mode is currently
         *  active.
         */
        this.isFrameSelectionMode = function (id) {
            return !!frameSelectDesc && (!id || (frameSelectDesc.id === id));
        };

        /**
         * Cancels the free-hand frame selection mode that has been started
         * with the method SelectionMixin.enterFrameSelectionMode() before.
         *
         * @returns {Boolean}
         *  Whether the free-hand frame selection mode was active and has been
         *  left.
         */
        this.cancelFrameSelectionMode = function () {
            if (frameSelectDesc) {
                leaveFrameSelectionMode(false);
                return true;
            }
            return false;
        };

        /**
         * Returns the current selection in the active sheet. This method
         * returns a clone of the selection that can be freely manipulated.
         *
         * @returns {SheetSelection}
         *  The current selection in the active sheet.
         */
        this.getSelection = function () {
            return getSheetSelectionAttribute().clone();
        };

        /**
         * Returns the addresses of all selected cell ranges in the active
         * sheet. This method returns a clone of the selected ranges that can
         * be freely manipulated.
         *
         * @returns {RangeArray}
         *  The addresses of all selected cell ranges in the active sheet.
         */
        this.getSelectedRanges = function () {
            return getSheetSelectionAttribute().ranges.clone(true);
        };

        /**
         * Returns the address of the active cell range in the selection of the
         * active sheet. This method returns a clone of the active range that
         * can be freely manipulated.
         *
         * @returns {Range}
         *  The address of the active cell range in the selection of the active
         *  sheet.
         */
        this.getActiveRange = function () {
            return getSheetSelectionAttribute().activeRange().clone();
        };

        /**
         * Returns the address of the active cell in the selection of the
         * active sheet (the highlighted cell in the active range that will be
         * edited when switching to text edit mode). This method returns a
         * clone of the active cell address that can be freely manipulated.
         *
         * @returns {Address}
         *  The address of the active cell in the selection of the active
         *  sheet.
         */
        this.getActiveCell = function () {
            return getSheetSelectionAttribute().address.clone();
        };

        /**
         * Returns the active cell of the selection in the active sheet (the
         * highlighted cell in the active range that will be edited when
         * switching to text edit mode), as a complete selection object
         * containing a single range that consists of the active cell.
         *
         * @returns {SheetSelection}
         *  The position of the active cell, as a new sheet selection object.
         */
        this.getActiveCellAsSelection = function () {
            return SheetSelection.createFromAddress(this.getActiveCell());
        };

        /**
         * Returns whether the current selection consists of the active cell
         * only, either as single cell, or exactly covering a merged range.
         *
         * @returns {Boolean}
         *  Whether the current selection consists of the active cell only.
         */
        this.isSingleCellSelection = function () {
            var selection = this.getSelection();
            return (selection.ranges.length === 1) && sheetModel.isSingleCellInRange(selection.ranges[0]);
        };

        /**
         * Returns whether the current selection contains at least one range
         * spanning over multiple columns.
         *
         * @returns {Boolean}
         *  Whether the current selection contains at least one range spanning
         *  over multiple columns.
         */
        this.hasMultipleColumnsSelected = function () {
            if (this.isSingleCellSelection()) { return false; } // single cell checks also the merged range
            return this.getSelectedRanges().some(function (range) { return !range.singleCol(); });
        };

        /**
         * Returns whether the current selection contains at least one range
         * spanning over multiple columns.
         *
         * @returns {Boolean}
         *  Whether the current selection contains at least one range spanning
         *  over multiple columns.
         */
        this.hasMultipleRowsSelected = function () {
            if (this.isSingleCellSelection()) { return false; } // single cell checks also the merged range
            return this.getSelectedRanges().some(function (range) { return !range.singleRow(); });
        };

        /**
         * Returns whether the current selection contains at least one range
         * spanning over multiple cells.
         *
         * @returns {Boolean}
         *  Whether the current selection contains at least one range spanning
         *  over multiple cells.
         */
        this.hasAnyRangeSelected = function () {
            return this.getSelectedRanges().some(function (range) { return !range.single(); });
        };

        /**
         * Returns whether the current selection contains or partly overlaps at
         * least one merged cell range.
         *
         * @returns {Boolean}
         *  Whether the current selection contains or partly overlaps at least
         *  one merged cell range.
         */
        this.hasMergedRangeSelected = function () {
            return mergeCollection.rangesOverlapAnyMergedRange(this.getSelectedRanges());
        };

        /**
         * Returns, whether a drawing object is selected in the current active
         * sheet.
         *
         * @returns {Boolean}
         *  Whether a drawing object is selected in the active sheet.
         */
        this.hasDrawingSelection = function () {
            return getSheetSelectionAttribute().drawings.length > 0;
        };

        /**
         * Returns, whether exactly one drawing object is selected in
         * the current active sheet.
         *
         * @returns {Boolean}
         *  Whether exactly one drawing object is selected in the active sheet.
         */
        this.hasSingleDrawingSelection = function () {
            return getSheetSelectionAttribute().drawings.length === 1;
        };

        /**
         * Returns the positions of all selected drawing frames in the active
         * sheet.
         *
         * @returns {Array<Array<Number>>}
         *  The selected drawing frames, as array of drawing positions.
         */
        this.getSelectedDrawings = function () {
            return _.copy(getSheetSelectionAttribute().drawings, true);
        };

        /**
         * Returns whether the current selection contains only drawings of type
         * image and chart. Only for these types copy and paste is supported.
         *
         * @returns {Boolean}
         *  Whether the selection can be copy pasted.
         */
        this.canCopyPasteDrawingSelection = function () {
            if (!this.hasDrawingSelection()) { return false; }

            var drawingTypes = textSelection.getSelectedDrawingsTypes({ forceObject: true });
            return ((drawingTypes.image === 1) || (drawingTypes.chart === 1)) && _.chain(drawingTypes).keys().without('chart', 'image').isEmpty().value();
        };

        /**
         * Changes the cell selection in the active sheet, and deselects all
         * drawing frames.
         *
         * @param {SheetSelection} selection
         *  The new sheet selection. The array of cell range addresses must not
         *  be empty. The array of selected drawing objects must be empty.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  - {Boolean} [options.storeOrigin=false]
         *      If set to true, and if the active cell passed in the selection
         *      is hidden by a merged range, the address of the original active
         *      cell will be stored in the property 'origin' of the sheet
         *      selection.
         *  - {Boolean} [options.unique=false]
         *      If set to true, duplicate cell range addresses will be removed
         *      from the selection.
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.setCellSelection = function (selection, options) {

            // do not change the selection in chart sheets etc.
            if (!sheetModel.isCellType()) { return this; }

            // a merged range hiding the active cell
            var mergedRange = mergeCollection.getMergedRange(selection.address, 'hidden');
            // the new active range
            var activeRange = selection.activeRange();

            // deselect all drawing frames
            selection = selection.clone();
            selection.drawings = [];
            selection.origin = null;

            // adjust active cell if it is hidden by a merged range
            if (mergedRange) {
                if (Utils.getBooleanOption(options, 'storeOrigin', false)) {
                    selection.origin = selection.address;
                }
                selection.address = mergedRange.start.clone();
            }

            // restrict to single selection if multi-selection is not supported
            if (!SheetUtils.MULTI_SELECTION) {
                selection.ranges.clear().push(activeRange);
                selection.active = 0;
            }

            // remove duplicates
            if (Utils.getBooleanOption(options, 'unique', false) && (selection.ranges.length > 1)) {
                selection.ranges = selection.ranges.unify();
                selection.active = Utils.findFirstIndex(selection.ranges, function (range) { return range.equals(activeRange); });
            }

            // apply the new sheet selection
            setEffectiveSelection(selection);

            return this;
        };

        /**
         * Selects a single cell in the active sheet. The cell will become the
         * active cell of the sheet.
         *
         * @param {Address} address
         *  The address of the cell to be selected.
         *
         * @param {Object} options
         *  Additional optional parameters:
         *  - {Boolean} [options.append=false]
         *      If set to true, the new cell will be appended to the current
         *      selection. Otherwise, the old selection will be removed.
         *  - {Boolean} [options.storeOrigin=false]
         *      If set to true, and if the specified cell is hidden by a merged
         *      range, the address of the passed cell will be stored in the
         *      property 'origin' of the sheet selection.
         *  - {Boolean} [options.unique=false]
         *      If set to true, duplicate cell range addresses will be removed
         *      from the selection (only useful when appending a cell to the
         *      selection).
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.selectCell = function (address, options) {
            return this.selectRange(new Range(address), _.extend({}, options, { active: address }));
        };

        /**
         * Selects a single cell range in the active sheet.
         *
         * @param {Range} range
         *  The address of the cell range to be selected.
         *
         * @param {Object} options
         *  Additional optional parameters:
         *  - {Boolean} [options.append=false]
         *      If set to true, the new range will be appended to the current
         *      selection. Otherwise, the old selection will be removed.
         *  - {Address} [options.active]
         *      The address of the active cell in the selected range. If
         *      omitted, the top-left cell of the range will be activated. Must
         *      be located inside the passed range address.
         *  - {Boolean} [options.storeOrigin=false]
         *      If set to true, and if the new active cell is hidden by a
         *      merged range, the address of the active cell will be stored in
         *      the property 'origin' of the sheet selection.
         *  - {Boolean} [options.unique=false]
         *      If set to true, duplicate cell range addresses will be removed
         *      from the selection (only useful when appending a range to the
         *      selection).
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.selectRange = function (range, options) {

            // do not change the selection in chart sheets etc.
            if (!sheetModel.isCellType()) { return this; }

            // whether to append the range to the current selection
            var append = Utils.getBooleanOption(options, 'append', false);
            // the initial selection to append the new range to
            var selection = append ? getEffectiveSelection({ allowEmpty: true }) : new SheetSelection();
            // the new active cell
            var activeCell = Utils.getObjectOption(options, 'active');
            // next cell tested while validating the active cell
            var nextCell = activeCell ? activeCell.clone() : null;

            // returns a cell address object, if the specified cell position is inside the passed selection range
            function getValidCellAddress(col, row) {
                var address = new Address(col, row);
                return range.containsAddress(address) ? address : null;
            }

            // expand range to merged ranges
            range = expandRangeToMergedRanges(range);

            // bug 33313: move active cell if it is outside the active range due to a partly covered merged range
            var mergedRange = null;
            while (nextCell && (mergedRange = mergeCollection.getMergedRange(nextCell)) && !range.contains(mergedRange)) {
                nextCell = getValidCellAddress(nextCell[0], mergedRange.end[1] + 1) || getValidCellAddress(mergedRange.end[0] + 1, nextCell[1]);
            }
            if (nextCell) { activeCell = nextCell; }

            // initialize and set the new selection
            selection.ranges.push(range);
            selection.active = selection.ranges.length - 1;

            // set active cell
            if (activeCell && range.containsAddress(activeCell)) {
                selection.address = activeCell.clone();
            } else {
                selection.address = range.start.clone();
            }

            // commit new selection
            return this.setCellSelection(selection, options);
        };

        /**
         * Modifies the position of the specified selection range.
         *
         * @param {Number} index
         *  The array index of the range to be changed.
         *
         * @param {Range} range
         *  The new address of the selected range. If the specified range index
         *  refers to the active range, but the new range does not contain the
         *  current active cell, the active cell will be moved inside the new
         *  range.
         *
         * @param {Object} options
         *  Additional optional parameters:
         *  - {Boolean} [options.unique=false]
         *      If set to true, duplicate cell range addresses will be removed
         *      from the selection.
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.changeRange = function (index, range, options) {

            // the selection to be modified
            var selection = getEffectiveSelection();

            // expand range to merged ranges
            range = expandRangeToMergedRanges(range);

            // insert the new range into the selection object
            selection.ranges[index] = range;
            if ((index === selection.active) && !range.containsAddress(selection.address)) {
                selection.address[0] = Utils.minMax(selection.address[0], range.start[0], range.end[0]);
                selection.address[1] = Utils.minMax(selection.address[1], range.start[1], range.end[1]);
            }

            // commit new selection
            return this.setCellSelection(selection, options);
        };

        /**
         * Modifies the position of the current active selection range.
         *
         * @param {Range} range
         *  The new address of the active selection range. If the passed range
         *  does not contain the current active cell, the top-left cell of the
         *  range will be activated.
         *
         * @param {Object} options
         *  Additional optional parameters:
         *  - {Boolean} [options.unique=false]
         *      If set to true, duplicate cell range addresses will be removed
         *      from the selection.
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.changeActiveRange = function (range, options) {
            return this.changeRange(getEffectiveSelection().active, range, options);
        };

        /**
         * Modifies the position of the current active cell.
         *
         * @param {Number} index
         *  The array index of the range to be activated.
         *
         * @param {Address} address
         *  The new address of the active cell.
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.changeActiveCell = function (index, address) {

            // the selection to be modified
            var selection = getEffectiveSelection();

            // update index of active range, and active cell
            selection.active = Utils.minMax(index, 0, selection.ranges.length - 1);
            selection.address = address.clone();

            // commit new selection
            return this.setCellSelection(selection);
        };

        /**
         * If a single cell is selected (also if it is a merged range), the
         * selection will be expanded to the surrounding content range.
         * Otherwise (multiple cells in selection, or multi-selection), the
         * selection remains unmodified.
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.expandToContentRange = function () {

            // the current selection
            var selection = this.getSelection();
            // the expanded content range for a single selected cell
            var contentRange = (selection.ranges.length === 1) ? sheetModel.getContentRangeForCell(selection.ranges.first()) : null;

            // select content range, if it is different from the current range
            if (contentRange && contentRange.differs(selection.ranges.first())) {
                this.changeRange(0, contentRange);
            }

            return this;
        };

        /**
         * Moves the active cell in the current selection into the specified
         * direction. Cycles through all ranges contained in the selection. In
         * case the selection is a single cell, moves it in the entire sheet.
         *
         * @param {String} direction
         *  The direction the active cell will be moved to. Supported values
         *  are 'left', 'right', 'up', and 'down'.
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.moveActiveCell = function (direction) {

            // In the following, variable names and code comments will use the term 'line' for cells in the main
            // moving direction, i.e. for a row when using the TAB key (direction 'left' or 'right'), or for a
            // column when using the ENTER key (direction 'up' or 'down'). Thus, the active cell will always be
            // moved inside its current line (independently in what direction this results), and will then wrap
            // to the next or previous line. Similarly, the term 'step' will be used for the single cells in a
            // line (the TAB or ENTER key moves the cell by one step in the current line).
            //
            // Example: If moving horizontally using the TAB key, the 'line index' of a cell refers to its row
            // index, and the 'step index' refers to its column index (moving the cell by one step will increase
            // the column index, moving to the next line will increase the row index).

            // current selection in the active sheet
            var selection = getEffectiveSelection();
            // shortcut to the 'address' property in the selection
            var address = selection.address;
            // move in entire sheet, if the selection consists of a single cell only
            var singleCell = this.isSingleCellSelection();
            // whether to move the active cell forwards (right or down)
            var forward = (direction === 'right') || (direction === 'down');
            // whether to move the active cell horizontally
            var horizontal = (direction === 'left') || (direction === 'right');
            // readable flags for address index getters/setters (e.g. address.get(line) for the line index)
            var line = !horizontal;
            var step = horizontal;
            // the column/row collections for lines and steps
            var lineCollection = horizontal ? rowCollection : colCollection;
            var stepCollection = horizontal ? colCollection : rowCollection;
            // whether to jump to unlocked cells only
            var unlockedCells = horizontal && sheetModel.isLocked();

            // Increases the passed range array index, while cycling in the available selected ranges.
            function incRangeIndex(index) {
                return Utils.mod(index + 1, selection.ranges.length);
            }

            // Decreases the passed range array index, while cycling in the available selected ranges.
            function decRangeIndex(index) {
                return Utils.mod(index - 1, selection.ranges.length);
            }

            // Returns the index of the first available visible entry in the passed column/row collection
            // including the passed index, regardless whether that index is located inside the selected ranges.
            function getVisibleIndex(collection, index) {
                var entry = forward ? collection.getNextVisibleEntry(index) : collection.getPrevVisibleEntry(index);
                return entry ? entry.index : -1;
            }

            // Returns the index of the preceding or following visible entry (depending on the direction) in the
            // passed column/row collection, regardless whether that index is located inside the selected ranges.
            function getNextVisibleIndex(collection, index) {
                var entry = forward ? collection.getNextVisibleEntry(index + 1) : collection.getPrevVisibleEntry(index - 1);
                return entry ? entry.index : -1;
            }

            // Moves the active cell to the first or last visible cell of the previous or next selected range
            // (depending on the direction). The active cell may be located inside a merged range afterwards!
            function moveToNextRange() {
                var range = null;
                do {
                    selection.active = forward ? incRangeIndex(selection.active) : decRangeIndex(selection.active);
                    range = selection.activeRange();
                    address[0] = getVisibleIndex(colCollection, forward ? range.start[0] : range.end[0]);
                    address[1] = getVisibleIndex(rowCollection, forward ? range.start[1] : range.end[1]);
                } while (!range.containsAddress(address));
                return 'range';
            }

            // Moves the active cell to the preceding or next visible line of the current selected range (depending
            // on the direction). If there are no more visible lines in the current range, the first or last cell
            // of the next selected range will be activated. The active cell may be located inside a merged range
            // afterwards!
            function moveToNextLine() {
                var range = selection.activeRange();
                address.set(getNextVisibleIndex(lineCollection, address.get(line)), line);
                address.set(getVisibleIndex(stepCollection, (forward ? range.start : range.end).get(step)), step);
                return range.containsAddress(address) ? 'line' : moveToNextRange();
            }

            // Moves the active cell to the preceding or next visible step of the current line in the selected
            // range (depending on the direction). If there are no more visible steps in the line, the preceding
            // or next line will be activated. The active cell may be located inside a merged range afterwards!
            function moveToNextStep() {
                var range = selection.activeRange();
                address.set(getNextVisibleIndex(stepCollection, address.get(step)), step);
                return range.containsAddress(address) ? 'step' : moveToNextLine();
            }

            // Returns the address of a merged range that contains the current active cell. The merged range will
            // be shrunken to its visible area.
            function getVisibleMergedRange() {
                var mergedRange = mergeCollection.getMergedRange(address);
                return mergedRange ? sheetModel.shrinkRangeToVisible(mergedRange) : null;
            }

            // Single cell selected: Simple movement without wrapping at sheet borders. Happens always for
            // vertical movement, but only in unlocked sheet for horizontal movement (see below). The flag
            // 'unlockedCells' is only set to true for horizontal movement in locked sheets.
            if (singleCell && !unlockedCells) {
                selectNextCell(direction, 'cell');
                return;
            }

            // Moving horizontally through a locked sheet without selection wraps inside used area of sheet.
            if (singleCell && unlockedCells) {
                selection.ranges = new RangeArray(sheetModel.getUsedRange() || new Range(Address.A1.clone()));
                selection.active = 0;
            }

            // Do nothing, if the entire selection is hidden.
            if (sheetModel.areRangesHidden(selection.ranges)) { return; }

            // Ensure to start at the top-left cell of a merged range.
            var mergedRange = getVisibleMergedRange();
            if (mergedRange) { selection.address = address = mergedRange.start.clone(); }

            // Fake an endless while-loop via callback, to be able to define callback functions for iterator
            // methods which results in a build error when done directly inside a while-loop or for-loop. The
            // loop can be canceled as usual by returning Utils.BREAK.
            function invokeForever(callback) { while (callback() !== Utils.BREAK) {} }
            invokeForever(function () {

                // Find the next active cell. This may involve stepping into the next line, or even into the next
                // selected range in a multi-selection.
                var result = moveToNextStep();

                // Active cell did not land inside a merged range: Step was successful, exit the endless loop.
                if (!(mergedRange = getVisibleMergedRange())) { return Utils.BREAK; }

                // If the first visible line of the merged range has been entered, the range will actually be
                // selected as active cell. For backward movement, it is important to insert the leading step
                // index of the range into the active cell. Example: Stepping backwards into the merged range
                // B2:D4 activates cell D2 or B4, but must be corrected to cell B2.
                if (mergedRange.start.get(line) === address.get(line)) {
                    if ((forward ? mergedRange.start : mergedRange.end).get(step) === address.get(step)) {
                        address.set(mergedRange.start.get(step), step);
                        return Utils.BREAK;
                    }
                }

                // Move the active cell to the last or first step in the line, to prevent looping over all inner
                // cells of the merged range in the current line. Example: Stepping forwards from cell A3 into
                // the merged range B2:Z4 moves active cell to B3. The following correction to cell Z3 will
                // prevent to loop over all covered cells in row 3.
                address.set((forward ? mergedRange.end : mergedRange.start).get(step), step);

                // If a new line has been entered, check if the entire line consists of merged ranges only (e.g.
                // lots of vertical merged ranges with many rows when stepping horizontally). It would cause a
                // performance bottleneck when iterating unsuccessfully through all the single lines covered by
                // these merged ranges. Instead, the line index will be increased to the 'shortest' of all the
                // merged ranges in the line, to be able to reach the new cells above or below the merged ranges
                // faster.
                if (result === 'line') {

                    // Calculate the step intervals covered by the merged ranges, and by the current line.
                    var lineRange = selection.activeRange().clone().setBoth(address.get(line), line);
                    var mergedRanges = RangeArray.map(mergeCollection.getMergedRanges(lineRange), sheetModel.shrinkRangeToVisible.bind(sheetModel));
                    var mergedIntervals = mergedRanges.intervals(step).merge();
                    var visibleIntervals = stepCollection.getVisibleIntervals(lineRange.interval(step));

                    // Nothing to do, if there are cells in the line not covered by a merged range.
                    if (mergedIntervals.contains(visibleIntervals)) {

                        // Sort the merged ranges according to move direction.
                        mergedRanges.sortBy(function (range) {
                            return range.start.get(step) * (forward ? 1 : -1);
                        });

                        // Find the merged range with maximum start line index, and minimum end line index.
                        var maxStartRange = mergedRanges.first();
                        var minEndRange = mergedRanges.first();
                        mergedRanges.forEach(function (range) {
                            if (range.start.get(line) > maxStartRange.start.get(line)) { maxStartRange = range; }
                            if (range.end.get(line) < minEndRange.end.get(line)) { minEndRange = range; }
                        });

                        // In backward direction, always move to the range with the maximum start line index.
                        // In forward direction, if a merged range starts in the current line, select it.
                        if (!forward || (maxStartRange.start.get(line) === address.get(line))) {
                            selection.address = address = maxStartRange.start.clone();
                            return Utils.BREAK;
                        }

                        // Otherwise (in forward direction), move to the end of the line that contains the end
                        // of the shortest merged range. This is needed to be able to step into the next line,
                        // or the next selected range as needed (the merged range may end together with the
                        // selected range).
                        address.set(minEndRange.end.get(line), line);
                        address.set(lineRange.end.get(step), step);
                    }
                }
            });

            // When traversing unlocked cells in a locked sheet, and the new cell is locked, the next unlocked
            // cell in the selection must be searched now.
            if (unlockedCells && !cellCollection.getAttributeSet(address).cell.unlocked) {

                // Use the cell collection to search for the next cell with unlocked flag.
                var desc = cellCollection.findCellWithAttributes(selection.ranges, { cell: { unlocked: true } }, {
                    visible: true,
                    reverse: !forward,
                    startIndex: selection.active,
                    startAddress: selection.address,
                    skipStart: true,
                    wrap: true
                });

                // If there is no unlocked cell available, do not change the current selection.
                if (!address) { return this; }

                // Update the selection object according to the found cell. The cell descriptor returned  by the
                // cell collection contains the array index of the cell range address.
                selection.address = address = desc.address;
                selection.active = desc.index;
            }

            // In single cell mode (stepping through unlocked cells), select the new cell, otherwise set the entire
            // selection instance with the new active cell. Always scroll to the new active cell.
            if (singleCell) {
                this.selectCell(address, { storeOrigin: true });
            } else {
                this.setCellSelection(selection);
            }
            this.scrollToCell(address);

            return this;
        };

        /**
         * Selects the specified drawing frames in the active sheet.
         *
         * @param {Array<Array<Number>>} positions
         *  The positions of all drawing frames to be selected.
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.setDrawingSelection = function (positions) {

            // the current selection to be modified (keep cell selection as it is)
            var selection = this.getSelection();
            selection.drawings = positions;

            // commit new selection
            setSheetSelectionAttribute(selection);

            return this;
        };

        /**
         * Selects a single drawing frame in the active sheet.
         *
         * @param {Array<Number>} position
         *  The position of the drawing frame to be selected.
         *
         * @param {Object} options
         *  Additional optional parameters:
         *  - {Boolean} [options.toggle=false]
         *      If set to true, the selection state of the addressed drawing
         *      frame will be toggled. Otherwise, the old drawing selection
         *      will be removed, and the addressed drawing frame will be
         *      selected.
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.selectDrawing = function (position, options) {

            // do not change the selection in chart sheets etc.
            if (!sheetModel.isCellType()) { return this; }

            // the current selection to be modified (keep cell selection as it is)
            var selection = this.getSelection();

            // construct the new drawing selection
            if (!Utils.getBooleanOption(options, 'toggle', false)) {
                selection.drawings = [position];
            } else if (SheetUtils.MULTI_SELECTION && (Utils.spliceValue(selection.drawings, position) === 0)) {
                selection.drawings.push(position);
            }

            // commit new selection
            setSheetSelectionAttribute(selection);

            // leave text edit mode; back to cell selection if that fails (e.g. syntax error in formula)
            this.leaveTextEditMode().fail(function () {
                self.removeDrawingSelection();
            });

            return this;
        };

        /**
         * Removes selection from all selected drawing objects and restores the
         * cell selection.
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.removeDrawingSelection = function () {

            // the current selection to be modified (keep cell selection as it is)
            var selection = this.getSelection();

            // remove drawing selection
            selection.drawings.splice(0);

            // commit new selection
            setSheetSelectionAttribute(selection);
            return this;
        };

        /**
         * Returns the drawing model instances of all selected drawing objects.
         *
         * @returns {Array<DrawingModel>}
         *  An array containing the models of all selected drawing objects, in
         *  order of the current selection.
         */
        this.getSelectedDrawingModels = function () {
            // TODO: move to a drawing layer mix-in that provides application abstraction

            // pick all selected drawing models from the collection
            return this.getSelectedDrawings().reduce(function (drawingModels, position) {
                var drawingModel = drawingCollection.getModel(position);
                if (drawingModel) { drawingModels.push(drawingModel); }
                return drawingModels;
            }, []);
        };

        // initialization -----------------------------------------------------

        // initialize sheet-dependent class members according to the active sheet
        this.on('change:activesheet', function () {
            sheetModel = self.getSheetModel();
            colCollection = sheetModel.getColCollection();
            rowCollection = sheetModel.getRowCollection();
            cellCollection = sheetModel.getCellCollection();
            mergeCollection = self.getMergeCollection();
            drawingCollection = sheetModel.getDrawingCollection();
            // cancel the custom selection mode when the active sheet changes (TODO: allow on multiple sheets)
            leaveCustomSelectionMode(false);
        });

        // trigger the 'active:selection' event on changed active selection
        this.on('change:sheet:viewattributes', function (event, attributes) {
            if ('activeSelection' in attributes) { triggerActiveSelection(); }
        });

        // update all settings depending on the current sheet selection
        this.on('change:selection', function (event, selection) {

            // cancel custom selection mode when selecting drawings
            if (selection.drawings.length > 0) {
                leaveCustomSelectionMode(false);
            }

            // register the drawing objects in the selection engine of the text framework
            refreshTextSelection();

            // stop highlighting source ranges of a previously selected chart
            if (chartHighlightUid) {
                self.endRangeHighlighting(chartHighlightUid);
                chartHighlightUid = null;
            }

            // highlight the source ranges of the selected chart object
            if (selection.drawings.length === 1) {
                var chartModel = drawingCollection.getModel(selection.drawings[0], { type: 'chart' });
                if (chartModel) {
                    var tokenArray = chartModel.getHighlightTokenArray();
                    chartHighlightUid = self.startRangeHighlighting([tokenArray], { priority: true });
                }
            }

            // trigger the 'active:selection' event
            triggerActiveSelection();
        });

        // cancel custom selection mode when entering text edit mode
        this.on('textedit:enter', function () { leaveCustomSelectionMode(false); });

        // restore the correct text framework selection after leaving drawing edit mode
        this.on('textedit:leave:drawing', refreshTextSelection);

        // listen to events from corner pane to adjust the selection
        listenToWithFocus(this.getCornerPane(), 'select:all', function (event, mode) {

            // select mode: move active cell the top-left cell in the first visible grid pane
            // extend mode: do not change the current active cell
            var topLeftAddress = (mode === 'select') ? self.getVisibleGridPane('topLeft').getTopLeftAddress() : getEffectiveSelection().address;

            self.selectRange(docModel.getSheetRange(), { active: topLeftAddress });
            leaveCustomSelectionMode(true);
        });

        // listen to events from all header panes to adjust the selection
        this.iterateHeaderPanes(function (headerPane, paneSide) {

            // whether the header pane shows column headers
            var columns = PaneUtils.isColumnSide(paneSide);
            // the column/row index where selection tracking has started
            var startIndex = 0;

            // makes a cell address according to the orientation of the header pane
            function makeCell(index1, index2) {
                return columns ? new Address(index1, index2) : new Address(index2, index1);
            }

            // makes an adjusted range between start index and passed index
            function makeRange(index) {
                return docModel.makeFullRange(Interval.create(startIndex, index), columns);
            }

            listenToWithFocus(headerPane, 'select:start', function (event, index, mode) {

                var selection = getEffectiveSelection({ allowEmpty: true });
                var customMode = self.isCustomSelectionMode();

                setTrackingActiveMode(true);
                if (!customMode && (mode === 'extend') && !selection.ranges.empty()) {
                    // modify the active range (from active cell to tracking start index)
                    startIndex = selection.address.get(columns);
                    self.changeActiveRange(makeRange(index));
                } else {
                    // create/append a new selection range
                    startIndex = index;
                    // set active cell to current first visible cell in the column/row
                    self.selectRange(makeRange(index), {
                        append: !customMode && (mode === 'append'),
                        active: makeCell(startIndex, Math.max(0, self.getVisibleHeaderPane(!columns).getFirstVisibleIndex()))
                    });
                }
            });

            listenToWithFocus(headerPane, 'select:move', function (event, index) {
                self.changeActiveRange(makeRange(index));
            });

            listenToWithFocus(headerPane, 'select:end', function (event, index, success) {
                self.changeActiveRange(makeRange(index), { unique: true });
                leaveCustomSelectionMode(success);
                setTrackingActiveMode(false);
            });
        });

        // listen to events from all grid panes to adjust the selection
        this.iterateGridPanes(function (gridPane) {

            // the address of the cell where selection tracking has started
            var startAddress = null;

            // makes an adjusted range from the start address to the passed address
            function makeRange(address) {
                return Range.createFromAddresses(startAddress, address);
            }

            listenToWithFocus(gridPane, 'select:start', function (event, address, mode) {

                var selection = getEffectiveSelection({ allowEmpty: true });
                var customMode = self.isCustomSelectionMode();

                setTrackingActiveMode(true);
                if (!customMode && (mode === 'extend') && !selection.ranges.empty()) {
                    // modify the active range (from active cell to tracking start position)
                    startAddress = selection.address;
                    self.changeActiveRange(makeRange(address));
                } else {
                    // create/append a new selection range
                    startAddress = address;
                    self.selectCell(address, { append: !customMode && (mode === 'append') });
                }
            });

            listenToWithFocus(gridPane, 'select:move', function (event, address) {
                self.changeActiveRange(makeRange(address));
            });

            listenToWithFocus(gridPane, 'select:end', function (event, address, success) {
                self.changeActiveRange(makeRange(address), { unique: true });
                leaveCustomSelectionMode(success);
                setTrackingActiveMode(false);
            });

            listenToWithFocus(gridPane, 'select:range', function (event, index, range) {
                leaveCustomSelectionMode(false);
                self.changeRange(index, range);
            });

            listenToWithFocus(gridPane, 'select:active', function (event, index, address) {
                leaveCustomSelectionMode(false);
                self.changeActiveCell(index, address);
            });

            listenToWithFocus(gridPane, 'select:dblclick', function () {
                self.enterTextEditMode('cell');
            });

            // forward selection rendering events of the active grid pane
            self.listenTo(gridPane, 'render:cellselection render:drawingselection', function (event) {
                if (gridPane === self.getActiveGridPane()) {
                    self.trigger.apply(self, [event.type].concat(_.toArray(arguments).slice(1)));
                }
            });

            // register mouse/touch tracking handler for frame selection mode
            gridPane.registerTrackingHandler('frameselect', self.isFrameSelectionMode.bind(self, null), frameSelectTrackingHandler, { readOnly: true });

            // invoke callback handler to render something into the selected frame rectangle
            self.listenTo(gridPane, 'render:frameselection', function (event, frameNode, frameRect) {
                if (frameSelectDesc && frameSelectDesc.renderer) {
                    frameSelectDesc.renderer.call(self, frameNode, frameRect);
                }
            });
        });

        // listen to key events from all panes to adjust the selection
        rootNode.on('keydown', keyDownHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = rootNode = null;
            docModel = textSelection = null;
            sheetModel = colCollection = rowCollection = cellCollection = null;
            mergeCollection = drawingCollection = null;
            getCellSelectionHandler = setCellSelectionHandler = null;
            customSelectDesc = frameSelectDesc = null;
        });

    } // class SelectionMixin

    // exports ================================================================

    return SelectionMixin;

});
