/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/control/discretefiltergroup', [
    'io.ox/office/tk/control/checkgroup',
    'io.ox/office/spreadsheet/utils/scalarset',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'gettext!io.ox/office/spreadsheet/main'
], function (CheckGroup, ScalarSet, FormulaUtils, gt) {

    'use strict';

    // convenience shortcuts
    var Scalar = FormulaUtils.Scalar;

    //#. Represents a list entry in spreadsheet data filters that selects all blank (empty) cells in a column
    var BLANK_CELLS_LABEL = gt.pgettext('filter', 'Blank cells');

    // class DiscreteFilterGroup ==============================================

    /**
     * A check-list group control that shows all entries of a discrete filter
     * (all different values in a table column).
     *
     * @constructor
     *
     * @extends CheckGroup
     *
     * @param {SpreadsheetView} docView
     *  The spreadsheet view that contains this instance.
     */
    var DiscreteFilterGroup = CheckGroup.extend({ constructor: function (docView) {

        // self reference
        var self = this;

        // special behavior for ODF documents
        var odf = docView.getApp().isODF();

        // base constructor ---------------------------------------------------

        CheckGroup.call(this, docView, { matcher: itemMatcher, sorted: sortComparator, boxed: true, checkAll: true });

        // private methods ----------------------------------------------------

        /**
         * A case-insensitive list item matcher.
         */
        function itemMatcher(value, entry) {
            return (odf && (value === entry)) || (_.isString(value) && _.isString(entry) && (value.toLowerCase() === entry.toLowerCase()));
        }

        /**
         * Compares the passed sort order descriptors stored at the list
         * entries of this control.
         */
        function sortComparator(sortIndex1, sortIndex2) {
            var result = Scalar.compare(sortIndex1.value, sortIndex2.value, { nullMode: 'less' });
            return (result === 0) ? Scalar.compareStrings(sortIndex1.display, sortIndex2.display) : result;
        }

        // public methods -----------------------------------------------------

        /**
         * Collects and inserts the cell entries of a table column into this
         * control.
         *
         * @param {String} tableModel
         *  The model of the table range in the active sheet to be filtered.
         *
         * @param {Number} tableCol
         *  The zero-based column index, relative to the table range.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the control has been
         *  initialized.
         */
        this.initialize = function (tableModel, tableCol) {

            this.clearOptionButtons();

            // resolve the table column model
            var columnModel = tableModel.getColumnModel(tableCol);

            // unfiltered table column: show entries of all visible rows; filtered table column: show
            // entries for all data rows that are not filtered by other columns (but that may be hidden)
            var promise = columnModel.isFiltered() ?
                tableModel.queryFilterResult({ skipColumns: [tableCol] }) :
                this.createResolvedPromise(tableModel.getVisibleDataRows());

            // fill the menu with the values
            promise = promise.then(function (rowIntervals) {

                // fetch cell contents of the column from the document
                var cellArray = tableModel.getColumnContents(tableCol, rowIntervals);
                // the filter attributes of the table column
                var attributes = columnModel.getMergedAttributeSet(true).filter;
                // each display string will be inserted once only
                // OOXML: case-insensitive display string, always use first occurrence
                // ODF: case-insensitive, separated by value type
                var usedEntries = new ScalarSet();

                // insert the new entries into the list
                cellArray.forEach(function (cellData) {

                    // the label displayed in the list entry (preserve case)
                    // bug 36264: trim space characters but no other whitespace
                    var label = cellData.display.replace(/^ +| +$/g, '');
                    // the key used to detect duplicates
                    var key = odf ? cellData.value : label.toLowerCase();

                    // the case-insensitive key used to detect duplicates
                    if (usedEntries.has(key)) { return; }
                    usedEntries.add(key);

                    // create the list entry
                    var options = (label.length === 0) ?
                        { label: BLANK_CELLS_LABEL, labelStyle: 'font-style:italic;', sortIndex: { value: null, display: '' } } :
                        { label: _.noI18n(label), sortIndex: { value: cellData.value, display: label }, attributes: { role: 'checkbox' } };
                    self.createOptionButton(key, options);
                });

                // initialize list entries according to current filter rules
                self.setValue((attributes.type === 'discrete') ? attributes.entries : usedEntries.values());
            });

            return promise;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docView = null;
        });

    } }); // class DiscreteFilterGroup

    // exports ================================================================

    return DiscreteFilterGroup;

});
