/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 * @author York Richter <york.richter@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/chartcreator', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/render/rectangle',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/tokenarray',
    'io.ox/office/drawinglayer/view/chartstyleutil',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, Rectangle, SheetUtils, Operations, FormulaUtils, TokenArray, ChartStyleUtil, gt) {

    'use strict';

    // convenience shortcuts
    var AnchorMode = SheetUtils.AnchorMode;
    var Address = SheetUtils.Address;
    var Range = SheetUtils.Range;
    var Range3D = SheetUtils.Range3D;
    var Range3DArray = SheetUtils.Range3DArray;
    var FormulaType = FormulaUtils.FormulaType;

    // default width of new chart objects
    var CHART_DEFAULT_WIDTH = Utils.convertHmmToLength(15000, 'px', 1);

    // default height of new chart objects
    var CHART_DEFAULT_HEIGHT = Utils.convertHmmToLength(10000, 'px', 1);

    var DEFAULT_LINECOLOR = { transformations: [{ type: 'lumMod', value: 15000 }, { type: 'lumOff', value: 85000 }], type: 'scheme', value: 'text1' };

    var AXIS_LABEL_ENABLED =  { axis: { label: true },  line: getNoneShapeAttributes() };
    var AXIS_LABEL_LINE_ENABLED =  { axis: { label: true },  line: getStandardShapeAttributes() };
    var AXIS_DISABLED = { axis: { label: false }, line: getNoneShapeAttributes() };
    var GRID_ENABLED =  { line: getStandardShapeAttributes() };
    var GRID_DISABLED = { line: getNoneShapeAttributes() };

    function getStandardShapeAttributes() {
        return _.copy({ type: 'solid', color: DEFAULT_LINECOLOR, width: 50 }, true);
    }

    function getNoneShapeAttributes() {
        return _.copy({ type: 'none', color: null, width: null }, true);
    }

    function getHeadCharAttributes() {
        return _.copy({ fontSize: 16, bold: false }, true);
    }

    function getDefaultAttributes(docModel) {

        return _.copy({
            normal: {
                xAxis: AXIS_LABEL_LINE_ENABLED,
                xGrid: GRID_DISABLED,
                yAxis: AXIS_LABEL_ENABLED,
                yGrid: GRID_ENABLED
            },
            xy: {
                xAxis: AXIS_LABEL_LINE_ENABLED,
                xGrid: docModel.getApp().isOOXML() ? GRID_ENABLED : GRID_DISABLED,
                yAxis: AXIS_LABEL_LINE_ENABLED,
                yGrid: GRID_ENABLED
            },
            pie: {
                xAxis: AXIS_DISABLED,
                xGrid: GRID_DISABLED,
                yAxis: AXIS_DISABLED,
                yGrid: GRID_DISABLED
            }
        }, true);
    }

    function getFormula(docModel, sheet, chart, from, to) {
        var range = new Range(from, to);

        if (chart) {
            // workaround for Bug 52802

            var formula = null;

            _.times(chart.getSeriesCount(), function (i) {
                var seriesData = chart.getSeriesModel(i);
                seriesData.iterateTokenArrays(function (tokenArray, linkKey) {
                    if (formula) { return; }

                    var oldRangeList = seriesData.resolveRanges(linkKey);
                    if (oldRangeList.empty()) { return; }
                    var oldRange = oldRangeList.first();

                    if (oldRange.start.equals(range.start) && oldRange.end.equals(range.end)) {
                        formula = tokenArray.getFormula('op');
                    }
                });
            });

            if (formula) {
                return formula;
            }
        }

        var tokenArray = new TokenArray(docModel, FormulaType.LINK);
        tokenArray.appendRange(range, { sheet: sheet, abs: true });
        return tokenArray.getFormula('op');
    }

    function addDataSeries(docModel, generator, chartProperties, options) {

        var sourceSheet = options.sourceSheet;
        var position = options.position;
        var series = options.series;
        var namesFrom = options.namesFrom;
        var namesTo = options.namesTo;
        var title = options.title;
        var valueFrom = options.valueFrom;
        var valueTo = options.valueTo;
        var bubbleFrom = options.bubbleFrom;
        var bubbleTo = options.bubbleTo;
        var bubbleArray = options.bubbleArray;
        var lineAttrs = options.lineAttrs;
        var seriesData = options.seriesData;
        var colors = options.colors;
        var sourceChart = options.sourceChart;

        var insert = {};
        insert.values = getFormula(docModel, sourceSheet, sourceChart, valueFrom, valueTo);
        insert.type = seriesData.type;
        insert.dataLabel = seriesData.dataLabel;
        if (seriesData.curved) {
            insert.curved = seriesData.curved;
        }
        if (title) {
            insert.title = getFormula(docModel, sourceSheet, sourceChart, title);
        }
        if (namesFrom) {
            insert.names = getFormula(docModel, sourceSheet, sourceChart, namesFrom, namesTo);
        }
        if (bubbleFrom) {
            insert.bubbles = getFormula(docModel, sourceSheet, sourceChart, bubbleFrom, bubbleTo);
        } else if (bubbleArray) {
            insert.bubbles = bubbleArray;
        }

        var type = seriesData.type.replace('2d', '').replace('3d', '');
        if (type !== 'pie' && type !== 'donut') {
            if (type === 'bar') {
                insert.axisXIndex = 1;
                insert.axisYIndex = 0;
            } else {
                insert.axisXIndex = 0;
                insert.axisYIndex = 1;
            }
        }

        var properties = {
            series: series,
            attrs: { series: insert, chart: chartProperties, fill: { type: 'solid' } }
        };

        // TODO: _.clone( should not be needed, but it is, why?!
        if (lineAttrs) {
            properties.attrs.line = _.copy(lineAttrs, true);
        } else if (seriesData.markersOnly) {
            properties.attrs.line = getNoneShapeAttributes();
            properties.attrs.markerFill = getStandardShapeAttributes();
        } else {
            properties.attrs.line = getStandardShapeAttributes();
        }

        if (colors) {
            if (colors.length > 1) {
                var dataPoints = [];
                colors.forEach(function (color) {
                    var dataPoint = {
                        fill: { type: 'solid', color: color },
                        line: { type: seriesData.markersOnly ? 'none' : 'solid', color: color }
                    };
                    if (seriesData.markersOnly) {
                        dataPoint.markerFill = dataPoint.fill;
                    }
                    dataPoints.push(dataPoint);
                });
                properties.attrs.series.dataPoints = dataPoints;
            }
            properties.attrs.line.color = colors[0];
            properties.attrs.fill.color = colors[0];
            if (seriesData.markersOnly) {
                properties.attrs.markerFill = colors[0];
            }
        }
        generator.generateDrawingOperation(Operations.INSERT_CHART_DATASERIES, position, properties);
    }

    function generateSeriesOperations(options) {

        var docModel = options.docModel;
        var sourceRange = options.sourceRange;
        var sourceSheet = options.sourceSheet;
        var generator = options.generator;
        var position = options.position;
        var cells = options.cells;
        var axis = options.axis;
        var chart = options.chart;
        var seriesData = options.seriesData;
        var lineAttrs = options.lineAttrs;
        var sourceChart = options.sourceChart;

        // moved from createChart (old comment for this change:"odf support by Stefan 15 Feb 2017")
        // moved it to this position to use it for updateSeries type too
        if (!chart.chartStyleId) {
            chart.chartStyleId = 2;
        }

        var start = sourceRange.start;
        var end = sourceRange.end;

        var useAxis;
        // check if axis is set
        if (axis !== 0 && axis !== 1) {
            if (cells[0] === cells[1]) {
                var width = end[0] - start[0];
                var height = end[1] - start[1];
                if (width > height) {
                    useAxis = 1;
                } else {
                    useAxis = 0;
                }
            } else if (cells[0]) {
                useAxis = 0;
            } else {
                useAxis = 1;
            }
        } else {
            useAxis = axis;
        }

        var dataSeries;
        if (seriesData.type.indexOf('bubble') === 0) {
            dataSeries = generateBubbleSeriesOperations(docModel, sourceRange, sourceSheet, position, useAxis, seriesData, lineAttrs, sourceChart, options.switchRowColumn);
        } else {
            dataSeries = generateStandardSeriesOperations(docModel, sourceRange, sourceSheet, position, cells, useAxis, seriesData, lineAttrs, sourceChart, seriesData.type);
        }

        if (chart.varyColors !== false && dataSeries.length > 1) {
            chart.varyColors = true;
        } else {
            chart.varyColors = false;
        }

        generator.generateDrawingOperation(Operations.CHANGE_DRAWING, position, { attrs: { chart: chart } });

        dataSeries.forEach(function (dataSerie) {
            addDataSeries(docModel, generator, chart, dataSerie);
        });

        return dataSeries.length;
    }

    /**
     * makes an address array, needed for the code, which does not know if its row or column oriented
     */
    function makeAddress(activeAxis, otherAxis, activeValue, otherValue) {
        var address = Address.A1.clone();
        address[activeAxis] = activeValue;
        address[otherAxis] = otherValue;
        return address;
    }

    function prepareSeriesColors(docModel, sourceColors, seriesCount, dataPointsCount, colorSetIndex, seriesType, transparent) {

        var count = 0;
        var pieDonut = /^(pie|donut)/.test(seriesType);
        var single = seriesCount === 1 || pieDonut;

        var varyColors = true;
        if (single) {
            count = !pieDonut ? 1 : dataPointsCount;
            if (!pieDonut || (count > 1 && sourceColors.length > 1 && !ChartStyleUtil.isAutoColor(sourceColors[0]) && _.isEqual(sourceColors[0], sourceColors[1]))) {
                varyColors = false;
            }
        } else {
            count = seriesCount;
        }

        var colorSet = null;
        var newColors = null;
        if (Utils.isFiniteNumber(colorSetIndex)) {
            colorSet = ChartStyleUtil.getColorSet()[colorSetIndex];
            newColors = [];
        } else {
            colorSet = ChartStyleUtil.getColorSet()[1];
            newColors = sourceColors.slice(0);
        }

        for (var index = newColors.length; index < count; index++) {
            var color = null;
            if (varyColors) {
                color = ChartStyleUtil.getColorOfPattern('cycle', colorSet.type, index, colorSet.colors, count, docModel);
            } else {
                color = ChartStyleUtil.getColorOfPattern('cycle', colorSet.type, 0, colorSet.colors, 3, docModel);
            }
            if (transparent) {
                color.transformations = [{ type: 'alpha', value: 75000 }];
            }
            newColors.push(color);
        }

        var result = [];
        if (single) {
            for (var i = 0; i < seriesCount; i++) {
                result.push(newColors);
            }
        } else {
            newColors.forEach(function (color) { result.push([color]); });
        }
        return result;
    }

    /**
     * generate all series from start to end.
     * Depending if first row/column contains a String,
     * it is defined as Names (datapoint) and Titles (dataseries)
     *
     * @param {Array<Boolean>} isNumber
     *  first is a Boolean for the cell under the first cell
     *  second is a Boolean for the cell right of the first cell
     *
     * @param {Number} activeAxis
     *  [1 or 0] column or row for the direction of any series data
     */
    function generateStandardSeriesOperations(docModel, sourceRange, sourceSheet, position, cells, activeAxis, seriesData, lineAttrs, sourceChart, chartType) {

        var start = sourceRange.start;
        var end = sourceRange.end;

        var max = end[activeAxis] - start[activeAxis];
        var otherAxis = 1 - activeAxis;

        var namesFrom, namesTo;

        var noKeys = false;
        if (chartType.indexOf('scatter') === 0) {
            noKeys = max === 0 && !cells[activeAxis];
        } else {
            noKeys = !cells[activeAxis];
        }

        if (noKeys) {
            namesFrom = null;
            namesTo = null;
            start[activeAxis] -= 1;
            max += 1;
        } else {
            if (!cells[otherAxis]) {
                namesFrom = start;
            } else {
                namesFrom = makeAddress(activeAxis, otherAxis, start[activeAxis], start[otherAxis] + 1);
            }
            namesTo = makeAddress(activeAxis, otherAxis, start[activeAxis], end[otherAxis]);
        }

        var seriesColors = prepareSeriesColors(docModel, seriesData.colors, max, (end[otherAxis] - start[otherAxis]) + 1, seriesData.colorSet, seriesData.type);
        var dataSeries = [];
        for (var i = 0; i < max; i++) {
            var activeIndex = start[activeAxis] + 1 + i;

            var title;
            var valueFrom;
            if (!cells[otherAxis]) {
                title = null;
                valueFrom = makeAddress(activeAxis, otherAxis, activeIndex, start[otherAxis]);
            } else {
                title = makeAddress(activeAxis, otherAxis, activeIndex, start[otherAxis]);
                valueFrom = makeAddress(activeAxis, otherAxis, activeIndex, start[otherAxis] + 1);
            }
            var valueTo = makeAddress(activeAxis, otherAxis, activeIndex, end[otherAxis]);

            dataSeries.push({
                sourceSheet: sourceSheet,
                position: position,
                series: i,
                namesFrom: namesFrom,
                namesTo: namesTo,
                title: title,
                valueFrom: valueFrom,
                valueTo: valueTo,
                seriesData: seriesData,
                lineAttrs: lineAttrs,
                colors: seriesColors[i],
                sourceChart: sourceChart
            });
        }
        return dataSeries;
    }

    var NONE_TITLE = 0;
    var TOP_TITLE  = 1;
    var LEFT_TITLE = 2;
    var BOTH_TITLE = 3;

    function isStringCell(value) {
        return value !== '' && _.isString(value);
    }

    function isRangeContentValueAString(index, rangeContents) {
        return rangeContents.length !== 0 && isStringCell(rangeContents[index].value);
    }

    function getBubbleChartType(docModel, startRange, sourceSheet, colCount, rowCount) {
        var chartType = NONE_TITLE;
        var res = docModel.getRangeContents(Range3D.createFromAddress(startRange, sourceSheet), { blanks: true, attributes: true, display: true });
        var firstCellIsString = isRangeContentValueAString(0, res);

        var top = false;
        var left = false;
        if (colCount > 1) {
            res = docModel.getRangeContents(Range3D.createFromAddress(new Address(startRange[0] + 1, startRange[1]), sourceSheet), { blanks: true, attributes: true, display: true });
            if (isRangeContentValueAString(0, res)) {
                top = true;
            } else if (rowCount === 1 && firstCellIsString) {
                left = true;
            }
        }
        if (rowCount > 1) {
            res = docModel.getRangeContents(Range3D.createFromAddress(new Address(startRange[0], startRange[1] + 1), sourceSheet), { blanks: true, attributes: true, display: true });
            if (isRangeContentValueAString(0, res)) {
                left = true;
            } else if (colCount === 1 && firstCellIsString) {
                top = true;
            }
        }

        if (top && left) {
            chartType = BOTH_TITLE;
        } else if (top) {
            chartType = TOP_TITLE;
        } else if (left) {
            chartType = LEFT_TITLE;
        }

        return chartType;
    }

    function getActiveAxis(colCount, rowCount, chartType) {
        var activeAxis = 0;
        if (chartType === TOP_TITLE) {
            if (colCount === 1 || colCount === rowCount - 1) {
                activeAxis = 1;
            } else if (rowCount <= 2) {
                activeAxis = 0;
            } else if (isEven(colCount) && colCount >= rowCount) {
                activeAxis = 1;
            }  else if (isOdd(colCount) && isEven(rowCount) && rowCount < colCount) {
                activeAxis = 1;
            }
        } else if (chartType === LEFT_TITLE) {
            if (colCount === 1) {
                activeAxis = 1;
            } else if (rowCount === 1) {
                activeAxis = 0;
            } else if (rowCount === 2 || colCount === 2) {
                activeAxis = 1;
            } else if (rowCount - 1 === colCount) {
                activeAxis = 0;
            } else if (isEven(colCount)) {
                if (colCount > rowCount) {
                    activeAxis = 1;
                }
            } else if (colCount >= rowCount || isOdd(rowCount)) {
                activeAxis = 1;
            }
        } else if (chartType === BOTH_TITLE) {
            if (colCount === 2 || colCount === rowCount) {
                activeAxis = 1;
            } else if (rowCount === 2) {
                activeAxis = 0;
            } else if (rowCount === 3) {
                activeAxis = 1;
            } else if (isEven(colCount)) {
                if ((isOdd(rowCount) && rowCount >= colCount) || (colCount > rowCount)) {
                    activeAxis = 1;
                }
            } else if (isOdd(rowCount) && colCount > rowCount) {
                activeAxis = 1;
            }
        } else {
            var evenColumnCount = isEven(colCount);
            var evenRowCount = isEven(rowCount);

            if (colCount === 1) {
                activeAxis = 1;
            } else if (rowCount === 1) {
                activeAxis = 0;
            } else if (colCount === rowCount) {
                activeAxis = 1;
            } else if (evenColumnCount && evenRowCount) {
                if (colCount > rowCount) {
                    activeAxis = 1;
                }
            } else if ((evenColumnCount && rowCount > 1) || (!evenColumnCount && !evenRowCount && rowCount < colCount)) {
                activeAxis = 1;
            }
        }
        return activeAxis;
    }

    function isEven(number) {
        return !(number & 1);
    }

    function isOdd(number) {
        return number & 1;
    }

    function generateBubbleSeriesOperations(docModel, sourceRange, sourceSheet, position, activeAxis, seriesData, lineAttrs, chartModel, switchRowColumnByUser) {
        var size = 0;

        var start = sourceRange.start;
        var end = sourceRange.end;

        var colCount = end[0] - start[0] + 1;
        var rowCount = end[1] - start[1] + 1;

        var chartType = getBubbleChartType(docModel, start, sourceSheet, colCount, rowCount);

        if (!switchRowColumnByUser) {
            // activeAxis = 0 = Left To Right = Columns
            // activeAxis = 1 = Top to Bottom = Rows
            activeAxis = getActiveAxis(colCount, rowCount, chartType);
        }

        var activeStartIndex = null;
        var otherStartIndex = 0;
        var setXVal = false;
        var setSeriesName = false;
        if (chartType === TOP_TITLE) {
            if (activeAxis === 0) {

                if (rowCount === 1) {
                    activeStartIndex = isEven(colCount) ? 1 : 0;
                    setXVal = activeStartIndex && colCount > 2;
                } else {
                    if (isEven(rowCount) || isEven(colCount) || colCount === 1) {
                        otherStartIndex = 1;
                    }
                    if (rowCount === 2) {
                        activeStartIndex = colCount > 1 && isOdd(colCount) ? 1 : 0;
                    } else {
                        activeStartIndex = colCount !== 1 ? 1 : 0;
                    }
                    setXVal = activeStartIndex || (colCount === 1 && rowCount === 2);
                }
                setSeriesName = rowCount > 1 && colCount > 3 && (isEven(colCount) || isEven(rowCount));
            } else {
                activeStartIndex = (isEven(rowCount) && isOdd(colCount)) || (rowCount === 2 && isEven(colCount)) ? 1 : 0;
                setXVal = (rowCount > 2 && activeStartIndex) || (rowCount === 1 && colCount === 2) || (rowCount >= 2 && activeStartIndex && colCount > 1);
                otherStartIndex = rowCount === 1 || (colCount > 1 && isOdd(colCount) && !activeStartIndex) ? 1 : 0;
                setSeriesName = (colCount > 1 && isOdd(colCount)) && !activeStartIndex;
            }

        } else if (chartType === LEFT_TITLE) {
            if (activeAxis === 0) {
                if (isEven(colCount)) {
                    activeStartIndex = isEven(rowCount) && colCount > 2 ? 0 : 1;
                    setXVal = (rowCount > 1 || colCount > 2) && activeStartIndex;
                } else {
                    activeStartIndex = 0;
                    otherStartIndex = colCount === 1 ? 1 : 0;
                    setXVal = otherStartIndex === 1 && rowCount === 2;
                }
            } else {
                if (colCount <= 2) {
                    activeStartIndex = (colCount === 1 && isEven(rowCount)) || (colCount > 1 && isOdd(rowCount) && rowCount > 1) ? 1 : 0;
                    otherStartIndex = colCount === 2 ? 1 : 0;
                    setXVal = (colCount === 1 && rowCount > 2 && activeStartIndex) || (colCount > 1 && isOdd(rowCount));
                } else {
                    activeStartIndex = rowCount > 1 ? 1 : 0;
                    otherStartIndex = 1;
                    setXVal = activeStartIndex === 1;
                }
                setSeriesName = rowCount > 3 && colCount > 1;
            }
        } else if (chartType === BOTH_TITLE) {
            otherStartIndex = 1;
            if (activeAxis === 0) {
                activeStartIndex = (rowCount === 2 && isOdd(colCount)) || (isOdd(rowCount) && (colCount > 2 && isEven(colCount))) ? 0 : 1;
                setSeriesName = colCount > 3 || rowCount === 2;
            } else {
                activeStartIndex = (colCount > 3 && ((isEven(colCount) && isEven(rowCount)) || isOdd(rowCount) || rowCount === 2)) || (colCount === 2 && isEven(rowCount)) || (colCount === 3 && (isOdd(rowCount) || rowCount === 2)) ? 1 : 0;
                setSeriesName = (colCount === 2 && rowCount > 2) || (colCount > 2 && rowCount !== 3);
            }
            setXVal = activeStartIndex === 1;
        }
        if (activeStartIndex === null) {
            if (activeAxis === 0) {
                activeStartIndex = colCount === 1 || isEven(colCount) ? 0 : 1;
            } else {
                activeStartIndex = rowCount === 1 || isEven(rowCount) ? 0 : 1;
            }

            setXVal = activeStartIndex === 1;
        }

        var otherAxis = 1 - activeAxis;
        var activeAxisCount = end[activeAxis] - start[activeAxis] + 1;
        var xValFrom, xValTo;

        if (setXVal) {
            if (activeStartIndex) {
                xValFrom = makeAddress(activeAxis, otherAxis, start[activeAxis], start[otherAxis] + otherStartIndex);
                xValTo = makeAddress(activeAxis, otherAxis, xValFrom[activeAxis], end[otherAxis]);
            } else {
                xValFrom = makeAddress(activeAxis, otherAxis, start[activeAxis], start[otherAxis]);
                xValTo = xValFrom; //makeAddress(activeAxis, otherAxis, xValFrom[activeAxis], end[otherAxis] - otherStartIndex);
            }
        }

        var seriesColors = prepareSeriesColors(docModel, seriesData.colors, activeAxisCount / 2, (end[otherAxis] - start[otherAxis]) + 1, seriesData.colorSet, seriesData.type, true);
        var i = activeStartIndex;
        var dataSeries = [];
        for (; i < activeAxisCount; i += 2) {
            var activeIndex = start[activeAxis] + i;
            var otherIndex = start[otherAxis] + otherStartIndex;
            // Series name
            var seriesName = null;
            if (setSeriesName) {
                // Top-Left TB
                seriesName = makeAddress(activeAxis, otherAxis, activeIndex, otherIndex - otherStartIndex);
            } else if (size === 0 && i + 2 >= activeAxisCount) {
                // Test if the cell is not used for the x values
                if (otherStartIndex && !(setXVal && activeIndex === xValFrom[activeAxis] && start[otherAxis] === otherIndex - 1)) {
                    seriesName = makeAddress(activeAxis, otherAxis, activeIndex, otherIndex - 1);
                } else if (activeStartIndex && !setXVal) {
                    seriesName = makeAddress(activeAxis, otherAxis, start[activeAxis], otherIndex);
                }
            }

            var bubbleArray = null;

            var yValFrom = makeAddress(activeAxis, otherAxis, activeIndex, otherIndex);
            var yValTo = makeAddress(activeAxis, otherAxis, activeIndex, end[otherAxis]);
            // Bubble Size
            var bubbleFrom = null, bubbleTo = null;
            if (i + 1 < activeAxisCount) {
                bubbleFrom = makeAddress(activeAxis, otherAxis, yValFrom[activeAxis] + 1, yValFrom[otherAxis]);
                bubbleTo = makeAddress(activeAxis, otherAxis, yValTo[activeAxis] + 1, yValTo[otherAxis]);
            } else if (!docModel.getApp().isODF()) {
                bubbleArray = [];
                var bubbleCount = yValTo[otherAxis] - yValFrom[otherAxis] + 1;
                for (var j = 0; j < bubbleCount; j++) {
                    bubbleArray.push(1);
                }
            }

            dataSeries.push({
                sourceSheet: sourceSheet,
                position: position,
                series: size,
                namesFrom: xValFrom,
                namesTo: xValTo,
                title: seriesName,
                valueFrom: yValFrom,
                valueTo: yValTo,
                bubbleFrom: bubbleFrom,
                bubbleTo: bubbleTo,
                bubbleArray: bubbleArray,
                seriesData: seriesData,
                lineAttrs: lineAttrs,
                colors: seriesColors[size],
                sourceChart: chartModel,
                activeAxis: activeAxis // only needed for selenium tests
            });
            size++;
        }

        return dataSeries;
    }

    /**
     *
     * @param {type} docModel
     * @param {type} docView
     * @param {type} sourceRange
     * @param {type} sourceSheet
     * @param {Number} axis only set if it is a bubble chart
     * @param {type} forceTitle
     * @param {type} forceNames
     * @returns {Promise}
     */
    function getContentForRange(docModel, docView, sourceRange, sourceSheet, axis, forceTitle, forceNames) {

        var sheetModel = docModel.getSheetModel(sourceSheet);

        var start = sourceRange.start;
        var colSize = sourceRange.cols();
        var rowSize = sourceRange.rows();

        if (colSize > 50 || rowSize > 50) {
            docView.yell({ type: 'warning', message: gt('It is not possible to create a chart out of more than 50 input cells.') });
            return $.Deferred().reject();
        }

        var rightEntry = sheetModel.getColCollection().getNextVisibleEntry(start[0] + 1);
        var bottEntry = sheetModel.getRowCollection().getNextVisibleEntry(start[1] + 1);

        var rightAddress = new Address(rightEntry ? rightEntry.index : start[0], start[1]);
        var bottAddress = new Address(start[0], bottEntry ? bottEntry.index : start[1]);

        var ranges = new Range3DArray(
            Range3D.createFromAddress(bottAddress, sourceSheet),
            Range3D.createFromAddress(rightAddress, sourceSheet),
            Range3D.createFromAddress(start, sourceSheet)
        );

        // first array element of the result is an array with the contents of the three cells
        var res = docModel.getRangeContents(ranges, { blanks: true, attributes: true, display: true });
        var bottomCell = res[0];
        var rightCell = res[1];
        var startCell = res[2];

        function isLabelCell(cell) {
            if (!startCell) { return false; }
            if (!startCell.display) { return true; } // empty strings (blank cells) or null values (invalid number format)
            return cell.format.isAnyDateTime() || !_.isNumber(cell.value);
        }

        var bottom = false;
        var right = false;

        if (colSize === 1 || rowSize === 1) {
            // axis is not set
            if (axis !== 0 && axis !== 1) {
                if (colSize === 1) {
                    axis = 0;
                } else {
                    axis = 1;
                }
            }
            var firstString = !_.isNumber(startCell.value);
            if (axis === 1) {
                if (firstString) {
                    bottom = true;
                }
            } else {
                if (firstString) {
                    right = true;
                }
            }
        } else {
            bottom = isLabelCell(bottomCell);
            right = isLabelCell(rightCell);
        }

        if (!_.isUndefined(forceTitle)) {
            if (axis === 0) {
                right = forceTitle;
            } else if (axis === 1) {
                bottom = forceTitle;
            }
        }
        if (!_.isUndefined(forceNames)) {
            if (axis === 0) {
                bottom = forceNames;
            } else if (axis === 1) {
                right = forceNames;
            }
        }
        if (colSize === 1) {
            bottom = false;
        }
        if (rowSize === 1) {
            right = false;
        }

        return docModel.createResolvedPromise({ cells: [bottom, right], axis: axis });
    }

    function prepareFallbackValueAttrs(docModel, attrs) {
        if (!attrs) {
            return;
        }
        prepareFallbackValueShape(docModel, attrs.line);
        prepareFallbackValueShape(docModel, attrs.fill);

        return attrs;
    }

    function prepareFallbackValueShape(docModel, shape) {
        if (!shape) {
            return;
        }
        if (shape.color && shape.color.type === 'scheme') {
            shape.color.fallbackValue = docModel.parseAndResolveColor(shape.color, 'fill').hex;
        }
        return shape;
    }

    // static class ChartCreator ==============================================

    var ChartCreator = {};

    // static methods ---------------------------------------------------------

    /**
     * Creates and inserts a new chart into the active sheet.
     *
     * @returns {jQuery.Promise}
     *  A promise that will be resolved after the chart has been created and
     *  inserted successfully, or that will be rejected on any error.
     */
    ChartCreator.createChart = function (app, chartData) {

        var chartAttrs = {
            stacking: chartData.stacking,
            curved: chartData.curved
        };

        var seriesData = {
            type: chartData.series.type,
            markersOnly: chartData.series.markersOnly,
            colors: [],
            curved: chartData.curved
        };

        var docModel = app.getModel();
        var docView = app.getView();
        var gridPane = docView.getActiveGridPane();

        var drawingCollection = docView.getDrawingCollection();

        var range = docView.getActiveRange();

        var forceNames;
        var axis;
        if (seriesData.type.indexOf('bubble') !== -1 && Math.min(range.cols(), range.rows()) > 2) {
            forceNames = true;
            // TODO:YRI Is this needed, calculated in generateBubbleSeriesOperations(...) too
            if (range.cols() > range.rows()) {
                axis = 1;
            } else {
                axis = 0;
            }
        }

        var promise = getContentForRange(docModel, docView, range, docView.getActiveSheet(), axis, undefined, forceNames);

        promise = promise.then(function (data) {

            var visibleRect = gridPane.getVisibleRectangle();

            var chartRect = new Rectangle(
                Math.max(0, Math.round(visibleRect.left + (visibleRect.width - CHART_DEFAULT_WIDTH) / 2)),
                Math.max(0, Math.round(visibleRect.top + (visibleRect.height - CHART_DEFAULT_HEIGHT) / 2)),
                CHART_DEFAULT_WIDTH,
                CHART_DEFAULT_HEIGHT
            );

            var anchorAttrs = drawingCollection.getAnchorAttributesForRect(AnchorMode.TWO_CELL, chartRect);
            var drawingDesc = { type: 'chart', attrs: { drawing: anchorAttrs } };
            drawingDesc.attrs.fill = { type: 'solid', color: { type: 'scheme', value: 'light1' } };
            drawingDesc.attrs.line = { type: 'none' };

            return app.getController().execInsertDrawings([drawingDesc], function (generator, sheet, position) {

                var seriesCount = generateSeriesOperations({ docModel: docModel, sourceRange: range, sourceSheet: sheet, generator: generator, position: position, cells: data.cells, axis: data.axis, chart: chartAttrs, seriesData: seriesData, switchRowColumn: false });

                var defaults = getDefaultAttributes(docModel).normal;
                var createAxis = true;
                var type = seriesData.type.replace('2d', '').replace('3d', '');
                var x = 0;
                var y = 1;
                var allwaysShowLegend = false;
                switch (type) {
                    case 'bubble':
                    case 'scatter':
                        defaults = getDefaultAttributes(docModel).xy;
                        break;
                    case 'pie':
                    case 'donut':
                        createAxis = false;
                        allwaysShowLegend = true;
                        break;
                    case 'bar':
                        x = 1;
                        y = 0;
                        break;
                }

                if (allwaysShowLegend || seriesCount > 1) {
                    generator.generateDrawingOperation(Operations.SET_CHART_LEGEND_ATTRIBUTES, position, { attrs: { legend: { pos:  'bottom' } } });
                } else {
                    generator.generateDrawingOperation(Operations.SET_CHART_LEGEND_ATTRIBUTES, position, { attrs: { legend: { pos:  'off' } } });
                }

                if (createAxis) {
                    var xAxisDefault = defaults.xAxis;
                    var yAxisDefault = defaults.yAxis;
                    var xGridDefault = defaults.xGrid;
                    var yGridDefault = defaults.yGrid;

                    if (type === 'bar') {
                        xAxisDefault = defaults.yAxis;
                        yAxisDefault = defaults.xAxis;
                        xGridDefault = defaults.yGrid;
                        yGridDefault = defaults.xGrid;
                    }
                    generator.generateDrawingOperation(Operations.SET_CHART_AXIS_ATTRIBUTES, position, { axis: x, axPos: 'b', crossAx: y, attrs: prepareFallbackValueAttrs(docModel, xAxisDefault) });
                    generator.generateDrawingOperation(Operations.SET_CHART_AXIS_ATTRIBUTES, position, { axis: y, axPos: 'l', crossAx: x, attrs: prepareFallbackValueAttrs(docModel, yAxisDefault) });
                    generator.generateDrawingOperation(Operations.SET_CHART_GRIDLINE_ATTRIBUTES, position, { axis: x, attrs: prepareFallbackValueAttrs(docModel, xGridDefault) });
                    generator.generateDrawingOperation(Operations.SET_CHART_GRIDLINE_ATTRIBUTES, position, { axis: y, attrs: prepareFallbackValueAttrs(docModel, yGridDefault) });
                }
            });
        });

        return docView.yellOnFailure(promise);
    };

    /**
     *
     * @param {Object} options
     *  - {Range} [options.range]
     *      whole start to end area of the new series
     *
     *  - {Number} [options.sheet]
     *
     *  - {ChartModel} [options.chartModel]
     *      is used for the position and id in the document and as fallback for chart-type-checks
     *
     *  - {String} [options.axis]
     *      series direction
     *
     *  - {Object} [options.chartData]
     *      is used for changing other attributes of the chart and for the type info (bubble or not)
     *
     *  - {Boolean} [options.forceTitle]
     *      is set, it overwrites the internal logic for title data
     *
     *  - {Boolean} [options.forceNames]
     *      is set, it overwrites the internal logic for names data
     *
     * @returns {jQuery.Promise}
     *  A promise that will be resolved after the chart has been changed
     *  successfully, or that will be rejected on any error.
     */
    ChartCreator.updateSeries = function (options) {

        var app         = Utils.getOption(options, 'app');
        var sourceRange = Utils.getOption(options, 'sourceRange');
        var sourceSheet = Utils.getOption(options, 'sourceSheet');
        var chartModel  = Utils.getOption(options, 'chartModel');
        var axis        = Utils.getOption(options, 'axis');
        var chartData   = Utils.getOption(options, 'chartData');
        var forceTitle  = Utils.getOption(options, 'forceTitle');
        var forceNames  = Utils.getOption(options, 'forceNames');
        var switchRowColumn  = Utils.getOption(options, 'switchRowColumn', false);
        // var typeChanged = Utils.getOption(options, 'typeChanged');

        var chartAttrs = null;
        var seriesData = null;
        var mergedAttributes = chartModel.getMergedAttributeSet(true);

        if (chartData) {
            chartAttrs = {
                stacking: chartData.stacking,
                curved: chartData.curved
            };
            seriesData = chartData.series;
            seriesData.curved = chartData.curved;
        } else {
            //CURVED is a workaround, filter has not all the information it needs!
            chartAttrs = {
                stacking: mergedAttributes.chart.stacking,
                curved: mergedAttributes.chart.curved
            };
            seriesData = {
                type: chartModel.getChartType().split(' ')[0],
                markersOnly: chartModel.isMarkerOnly(),
                curved: mergedAttributes.chart.curved
            };
        }

        seriesData.colors = [];
        seriesData.dataLabel = chartModel.getDataLabel();

        chartAttrs.chartStyleId = mergedAttributes.chart.chartStyleId;
        chartAttrs.varyColors = chartModel.isVaryColor();
        if (chartAttrs.chartStyleId) {
            seriesData.colorSet = chartModel.getColorSet().replace('cs', '') | 0;
        } else {
            seriesData.colorSet = null;
        }

        var clonedData = chartModel.getCloneData();
        var undoAttrs = chartModel.getUndoAttributeSet({ chart: chartAttrs });

        var docModel = app.getModel();
        var docView = app.getView();

        var promise = getContentForRange(docModel, docView, sourceRange, sourceSheet, axis, forceTitle, forceNames);

        promise = promise.then(function (data) {

            //only overwritten if source range height is 1 oder width is 1
            if (data.axis === 0 || data.axis === 1) {
                axis = data.axis;
            }
            var position = chartModel.getPosition();
            var lineAttrs = null;

            return chartModel.getSheetModel().createAndApplyOperations(function (generator) {

                var oldCount = chartModel.getSeriesCount();
                var seriesCount = generateSeriesOperations({ docModel: docModel, sourceRange: sourceRange, sourceSheet: sourceSheet, generator: generator, position: position, cells: data.cells, axis: axis, chart: chartAttrs, seriesData: seriesData, lineAttrs: lineAttrs, sourceChart: chartModel, switchRowColumn: switchRowColumn });

                _.times(oldCount, function () {
                    generator.generateDrawingOperation(Operations.DELETE_CHART_DATASERIES, position, { series: seriesCount });
                });

                // UNDO ---
                generator.generateDrawingOperation(Operations.CHANGE_DRAWING, position, { attrs: undoAttrs }, { undo: true });

                var defaults = getDefaultAttributes(docModel).normal;
                var createAxis = true;
                var type = seriesData.type.replace('2d', '').replace('3d', '');
                var x = 0;
                var y = 1;
                switch (type) {
                    case 'bubble':
                    case 'scatter':
                        defaults = getDefaultAttributes(docModel).xy;
                        break;
                    case 'pie':
                    case 'donut':
                        createAxis = false;
                        break;
                    case 'bar':
                        x = 1;
                        y = 0;
                        break;
                }

                var xAxisIdOld = chartModel.getAxisIdForType('x');
                var yAxisIdOld = chartModel.getAxisIdForType('y');
                if (createAxis) {
                    generator.generateDrawingOperation(Operations.DELETE_CHART_AXIS, position, { axisId: String(x) }, { undo: true });
                    generator.generateDrawingOperation(Operations.DELETE_CHART_AXIS, position, { axisId: String(y) }, { undo: true });
                }

                _.each(clonedData.series, function (obj, index) {
                    generator.generateDrawingOperation(Operations.INSERT_CHART_DATASERIES, position, { series: index, attrs: obj }, { undo: true });
                });

                chartModel.removeAllAxis(generator, position);

                if (!_.isEmpty(clonedData.title)) {
                    generator.generateDrawingOperation(Operations.SET_CHART_TITLE_ATTRIBUTES, position, { attrs: clonedData.title });
                }
                if (!_.isEmpty(clonedData.legend)) {
                    generator.generateDrawingOperation(Operations.SET_CHART_LEGEND_ATTRIBUTES, position, { attrs: clonedData.legend });
                }

                if (createAxis) {
                    var xTitle;
                    var yTitle;
                    if (xAxisIdOld !== null) {
                        var xAxis = clonedData.axes[xAxisIdOld];
                        xTitle = xAxis.title;
                        defaults.xAxis = xAxis.axis;
                        defaults.xGrid = xAxis.grid;
                    }
                    if (yAxisIdOld !== null) {
                        var yAxis = clonedData.axes[yAxisIdOld];
                        yTitle = yAxis.title;
                        defaults.yAxis = yAxis.axis;
                        defaults.yGrid = yAxis.grid;
                    }

                    generator.generateDrawingOperation(Operations.SET_CHART_AXIS_ATTRIBUTES, position, { axis: x, axPos: 'b', crossAx: y, attrs: prepareFallbackValueAttrs(docModel, defaults.xAxis) });
                    generator.generateDrawingOperation(Operations.SET_CHART_AXIS_ATTRIBUTES, position, { axis: y, axPos: 'l', crossAx: x, attrs: prepareFallbackValueAttrs(docModel, defaults.yAxis) });
                    if (!_.isEmpty(xTitle)) {
                        generator.generateDrawingOperation(Operations.SET_CHART_TITLE_ATTRIBUTES, position, { axis: x, attrs: xTitle });
                    }
                    if (!_.isEmpty(yTitle)) {
                        generator.generateDrawingOperation(Operations.SET_CHART_TITLE_ATTRIBUTES, position, { axis: y, attrs: yTitle });
                    }
                    var xGridAttrs = prepareFallbackValueAttrs(docModel, defaults.xGrid);
                    var yGridAttrs = prepareFallbackValueAttrs(docModel, defaults.yGrid);

                    if (!_.isEmpty(xGridAttrs)) {
                        generator.generateDrawingOperation(Operations.SET_CHART_GRIDLINE_ATTRIBUTES, position, { axis: x, attrs: xGridAttrs });
                    }
                    if (!_.isEmpty(yGridAttrs)) {
                        generator.generateDrawingOperation(Operations.SET_CHART_GRIDLINE_ATTRIBUTES, position, { axis: y, attrs: yGridAttrs });
                    }
                }

                _.times(seriesCount, function () {
                    generator.generateDrawingOperation(Operations.DELETE_CHART_DATASERIES, position, { series: oldCount }, { undo: true });
                });

            }, { storeSelection: true });
        });

        return promise;
    };

    ChartCreator.generateOperationsFromModelData = function (generator, position, chartModelData) {

        var axes = chartModelData.axes;
        var series = chartModelData.series;
        var title = chartModelData.title;
        var legend = chartModelData.legend;

        _.each(series, function (serie, index) {
            var properties = {
                series: index,
                attrs: serie
            };
            generator.generateDrawingOperation(Operations.INSERT_CHART_DATASERIES, position, properties);
        });

        _.each(axes, function (axisData, axisId) {
            axisId = window.parseInt(axisId, 10);
            if (!_.isEmpty(axisData.axis)) {
                generator.generateDrawingOperation(Operations.SET_CHART_AXIS_ATTRIBUTES, position, { axis: axisId, axPos: axisData.axPos, crossAx: axisData.crossAx, attrs: axisData.axis });
            }
            if (!_.isEmpty(axisData.grid)) {
                generator.generateDrawingOperation(Operations.SET_CHART_GRIDLINE_ATTRIBUTES, position, { axis: axisId, attrs: axisData.grid });
            }
            if (!_.isEmpty(axisData.title)) {
                generator.generateDrawingOperation(Operations.SET_CHART_TITLE_ATTRIBUTES, position, { axis: axisId, attrs: axisData.title });
            }
        });

        if (!_.isEmpty(title)) {
            generator.generateDrawingOperation(Operations.SET_CHART_TITLE_ATTRIBUTES, position, { attrs: title });
        }

        if (!_.isEmpty(legend)) {
            generator.generateDrawingOperation(Operations.SET_CHART_LEGEND_ATTRIBUTES, position, { attrs: legend });
        }

    };

    ChartCreator.getStandardShape = function (docModel) {
        return prepareFallbackValueShape(docModel, getStandardShapeAttributes());
    };

    ChartCreator.getHeadChar = function (/*docModel*/) {
        return getHeadCharAttributes();
    };

    ChartCreator.getNoneShape = function () {
        return getNoneShapeAttributes();
    };

    /**
     * Only needed for JUnit test.
     */
    ChartCreator.getDataForBubbleSeriesOperations = function (docModel, sourceRange, sourceSheet, activeAxis, switchRowColumnByUser) {

        if (activeAxis === undefined) {
            activeAxis = null;
        }
        if (switchRowColumnByUser === undefined) {
            switchRowColumnByUser = false;
        }

        return generateBubbleSeriesOperations(docModel, sourceRange, sourceSheet, null, activeAxis, { colors: [], colorSet: [], type: 'bubble' }, null, null, switchRowColumnByUser);
    };

    // exports ================================================================

    return ChartCreator;

});
