/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/sheetutils', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/enum',
    'io.ox/office/tk/utils/logger',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/render/font',
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/spreadsheet/utils/address',
    'io.ox/office/spreadsheet/utils/interval',
    'io.ox/office/spreadsheet/utils/range',
    'io.ox/office/spreadsheet/utils/range3d',
    'io.ox/office/spreadsheet/utils/addressarray',
    'io.ox/office/spreadsheet/utils/intervalarray',
    'io.ox/office/spreadsheet/utils/rangearray',
    'io.ox/office/spreadsheet/utils/range3darray',
    'io.ox/office/spreadsheet/utils/addressset',
    'io.ox/office/spreadsheet/utils/intervalset',
    'io.ox/office/spreadsheet/utils/rangeset',
    'io.ox/office/spreadsheet/utils/changedescriptor',
    'io.ox/office/spreadsheet/utils/movedescriptor',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, Enum, Logger, LocaleData, Font, ErrorCode, Address, Interval, Range, Range3D, AddressArray, IntervalArray, RangeArray, Range3DArray, AddressSet, IntervalSet, RangeSet, ChangeDescriptor, MoveDescriptor, gt) {

    'use strict';

    // maps single-character border keys to the attribute names of cell borders
    var BORDER_ATTRIBUTE_NAMES = {
        l: 'borderLeft',
        r: 'borderRight',
        t: 'borderTop',
        b: 'borderBottom',
        d: 'borderDown',
        u: 'borderUp',
        v: 'borderInsideVert',
        h: 'borderInsideHor'
    };

    // maps horizontal alignment values used in operations to CSS text alignment values
    var CSS_TEXT_ALIGN_VALUES = {
        left: 'left',
        center: 'center',
        right: 'right',
        justify: 'justify',
        distribute: 'justify',
        fillAcross: 'left',
        centerAcross: 'center'
    };

    // private global functions ===============================================

    // converts the passed text direction to left/right alignment
    function getAlignFromDir(dir) {
        return (dir === 'rtl') ? 'right' : 'left';
    }

    // static class SheetUtils ================================================

    /**
     * The static class SheetUtils contains various helper methods for cell
     * addresses, cell range addresses, and column/row index intervals.
     * Additionally, the class is a console logger bound to the URL hash flag
     * 'spreadsheet:log-models', that logs detailed information about the
     * document model instances and collections.
     */
    var SheetUtils = {};

    // logger interface -------------------------------------------------------

    Logger.extend(SheetUtils, { enable: 'spreadsheet:log-models', prefix: 'MODEL' });

    // helper classes ---------------------------------------------------------

    // export address/interval/range helper classes for convenience
    SheetUtils.ErrorCode = ErrorCode;
    SheetUtils.Address = Address;
    SheetUtils.Interval = Interval;
    SheetUtils.Range = Range;
    SheetUtils.Range3D = Range3D;
    SheetUtils.AddressArray = AddressArray;
    SheetUtils.IntervalArray = IntervalArray;
    SheetUtils.RangeArray = RangeArray;
    SheetUtils.Range3DArray = Range3DArray;
    SheetUtils.AddressSet = AddressSet;
    SheetUtils.IntervalSet = IntervalSet;
    SheetUtils.RangeSet = RangeSet;
    SheetUtils.ChangeDescriptor = ChangeDescriptor;
    SheetUtils.MoveDescriptor = MoveDescriptor;

    // constants --------------------------------------------------------------

    /**
     * The identifier of the spreadsheet attribute pool for cell formatting
     * attributes.
     *
     * @constant
     * @type String
     */
    SheetUtils.CELL_POOL_ID = 'cell';

    /**
     * Maximum length of a defined name.
     *
     * @constant
     * @type Number
     */
    SheetUtils.MAX_NAME_LENGTH = 255;

    /**
     * The maximum number of cells to be filled with one range operation, or
     * auto-fill operation.
     *
     * @constant
     * @type Number
     */
    SheetUtils.MAX_FILL_CELL_COUNT = 20000;

    /**
     * The maximum number of entire columns/rows to be filled with one
     * auto-fill operation.
     *
     * @constant
     * @type Number
     */
    SheetUtils.MAX_AUTOFILL_COL_ROW_COUNT = 100;

    /**
     * Maximum number of merged ranges to be created with a single operation.
     *
     * @constant
     * @type Number
     */
    SheetUtils.MAX_MERGED_RANGES_COUNT = 1000;

    /**
     * The maximum number of cells to be filled when unmerging a range.
     *
     * @constant
     * @type Number
     */
    SheetUtils.MAX_UNMERGE_CELL_COUNT = 5000;

    /**
     * The maximum number of columns to be changed with one row operation.
     *
     * @constant
     * @type Number
     */
    SheetUtils.MAX_CHANGE_COLS_COUNT = 5000;

    /**
     * The maximum number of rows to be changed with one row operation.
     *
     * @constant
     * @type Number
     */
    SheetUtils.MAX_CHANGE_ROWS_COUNT = 5000;

    /**
     * The maximum number of columns/rows that can be sorted.
     *
     * @constant
     * @type Number
     */
    SheetUtils.MAX_SORT_LINES_COUNT = 5000;

    /**
     * Maximum number of characters for the absolute part of a number formatted
     * with the standard number format (in cells).
     *
     * @constant
     * @type Number
     */
    SheetUtils.MAX_LENGTH_STANDARD_CELL = 11;

    /**
     * Maximum number of characters for the absolute part of a number formatted
     * with the standard number format (in cell edit mode).
     *
     * @constant
     * @type Number
     */
    SheetUtils.MAX_LENGTH_STANDARD_EDIT = 21;

    /**
     * Maximum number of characters for the absolute part of a number formatted
     * with the standard number format (in formula interpretation, e.g. when
     * converting numbers to strings for functions expecting strings).
     *
     * @constant
     * @type Number
     */
    SheetUtils.MAX_LENGTH_STANDARD_FORMULA = 20;

    /**
     * Minimum effective size of a cell, in pixels. Also used as maximum width
     * of cell border lines, to prevent that the left/right borders or the top/
     * bottom borders of a cell overlay each other.
     *
     * @constant
     * @type Number
     */
    SheetUtils.MIN_CELL_SIZE = 5;

    /**
     * Whether multi-selection (multiple ranges at the same time) is supported.
     *
     * @constant
     * @type Boolean
     */
    SheetUtils.MULTI_SELECTION = !Utils.TOUCHDEVICE;

    // enumerations -----------------------------------------------------------

    /**
     * Enumeration for the types of sheets that can occur in a spreadsheet
     * document. The JSON string representation of the enumeration values
     * corresponds to the values used in the document operations 'insertSheet'.
     *
     * @constant
     * @type Enum
     *
     * @property {SheetType} WORKSHEET
     *  A regular sheet with value cells, formula cells, and drawing objects.
     *
     * @property {SheetType} CHARTSHEET
     *  A sheet consisting of a single big chart object.
     *
     * @property {SheetType} MACROSHEET
     *  A sheet consisting of formula cells with specific macro formulas. Used
     *  for compatibility with MS Excel 4.0 only.
     *
     * @property {SheetType} DIALOGSHEET
     *  A sheet consisting of a single dialog box. Used for compatibility with
     *  MS Excel 5.0 only.
     */
    SheetUtils.SheetType = Enum.create('worksheet chartsheet macrosheet dialogsheet');

    /**
     * Enumeration for the split modes for a sheet. The JSON string
     * representation of the enumeration values corresponds to the values used
     * in the sheet formatting attribute 'splitMode'.
     *
     * @constant
     * @type Enum
     *
     * @property {SplitMode} SPLIT
     *  Specifies a dynamic split, with resizable view panes.
     *
     * @property {SplitMode} FROZEN
     *  Specifies a frozen split, with fixed view panes.
     *
     * @property {SplitMode} FROZEN_SPLIT
     *  Specifies a frozen split similar to FROZEN, but thawing the frozen
     *  split will change split mode to SPLIT instead of removing the split
     *  completely.
     */
    SheetUtils.SplitMode = Enum.create('split frozen frozenSplit');

    /**
     * Enumeration for the move modes for cell ranges. The JSON string
     * representation of the enumeration values corresponds to the values used
     * in the document operations 'moveCells'.
     *
     * @constant
     * @type Enum
     *
     * @property {MoveMode} LEFT
     *  A cell range will be deleted, the following cells right of the cell
     *  range will be moved to the left.
     *
     * @property {MoveMode} RIGHT
     *  A cell range will be made blank by moving the existing cells to the
     *  right. Cells moved outside the sheet will be deleted.
     *
     * @property {MoveMode} UP
     *  A cell range will be deleted, the following cells below the cell range
     *  will be moved up.
     *
     * @property {MoveMode} DOWN
     *  A cell range will be made blank by moving the existing cells down.
     *  Cells moved outside the sheet will be deleted.
     */
    SheetUtils.MoveMode = Enum.create('left right up down');

    /**
     * Enumeration for the merge modes for cell ranges. The JSON string
     * representation of the enumeration values corresponds to the values used
     * in the document operations 'mergeCells'.
     *
     * @constant
     * @type Enum
     *
     * @property {MergeMode} MERGE
     *  Merges an entire cell range (multiple columns and rows).
     *
     * @property {MergeMode} HORIZONTAL
     *  Merges the individual rows in a cell range.
     *
     * @property {MergeMode} VERTICAL
     *  Merges the individual columns in a cell range.
     *
     * @property {MergeMode} UNMERGE
     *  Converts the affected merged ranges to single cells.
     */
    var MergeMode = SheetUtils.MergeMode = Enum.create('merge horizontal vertical unmerge');

    /**
     * Enumeration for different types of document operations that will cause
     * to update formula expressions.
     *
     * @constant
     * @type Enum
     *
     * @property {UpdateMode} DELETE_SHEET
     *  A sheet has been deleted in the document. All references in all formula
     *  expressions containing the deleted sheet become invalid and will be
     *  replaced with #REF! errors.
     *
     * @property {UpdateMode} RENAME_SHEET
     *  A sheet has been renamed or cloned in the document. All references in
     *  all formula expressions containing the renamed sheet will be updated.
     *
     * @property {UpdateMode} RELABEL_NAME
     *  The label of a defined name has been changed. All formula expressions
     *  will be updated to use the new label.
     *
     * @property {UpdateMode} MOVE_CELLS
     *  Cells will be moved in a sheet. This includes inserting or deleting
     *  some columns or rows. All affected references in all formula
     *  expressions will be updated.
     *
     * @property {UpdateMode} CUT_PASTE
     *  Cut cells have been pasted into a sheet. This includes recalculation of
     *  all dependent references.
     */
    SheetUtils.UpdateMode = Enum.create('deleteSheet renameSheet relabelName moveCells cutPaste');

    /**
     * Enumeration for different anchor types of drawing objects.
     *
     * @constant
     * @type Enum
     *
     * @property {AnchorMode} ABSOLUTE
     *  The top-left corner of the drawing object is positioned absolutely in
     *  the sheet, and its size is fixed. Column and row operations do not
     *  influence the (absolute) location of the drawing object.
     *
     * @property {AnchorMode} ONE_CELL
     *  The top-left corner of the drawing object is anchored to a spreadsheet
     *  cell, and its size is fixed. Column and row operations in front of the
     *  drawing object will cause to move it accordingly.
     *
     * @property {AnchorMode} TWO_CELL
     *  The top-left corner and the bottom-right corner of the drawing object
     *  are anchored to spreadsheet cells. Column and row operations in front
     *  of, or inside the drawing object will cause to move or resize it
     *  accordingly.
     */
    SheetUtils.AnchorMode = Enum.create('absolute oneCell twoCell');

    // public methods ---------------------------------------------------------

    /**
     * Convenience shortcut that returns a rejected promise with a specific
     * message code as object with 'cause' property.
     *
     * @param {String} msgCode
     *  The message code the returned promise will be rejected with.
     *
     * @returns {jQuery.Promise}
     *  A promise that has already been rejected with an object containing
     *  a string property 'cause' set to the passed message code.
     */
    SheetUtils.makeRejected = function (msgCode) {
        return new $.Deferred().reject({ cause: msgCode });
    };

    /**
     * Returns a unique map key for the passed label of a defined name.
     *
     * @param {String} label
     *  The original label of a defined name.
     *
     * @returns {String}
     *  A unique map key for the passed label of a defined name.
     */
    SheetUtils.getNameKey = function (label) {
        return label.toUpperCase();
    };

    /**
     * Returns a unique map key for the passed table name. Can also be used for
     * the names of table columns.
     *
     * @param {String} tableName
     *  The original name of a table range, or a table column.
     *
     * @returns {String}
     *  A unique map key for the passed table name, or table column name.
     */
    SheetUtils.getTableKey = function (tableName) {
        return tableName.toUpperCase();
    };

    /**
     * Returns the localized sheet name with the specified index, ready to be
     * used in the document model (without any I18N marker characters used for
     * UI debugging).
     *
     * @param {Number} index
     *  The one-based (!) sheet index to be inserted into the sheet name.
     *
     * @returns {String}
     *  The generated sheet name (e.g. 'Sheet1', 'Sheet2', etc.).
     */
    SheetUtils.getSheetName = function (index) {
        //#. Default sheet names in spreadsheet documents. Should be equal to the sheet
        //#. names used in existing spreadsheet applications (Excel, OpenOffice Calc, ...).
        //#. %1$d is the numeric index inside the sheet name
        //#. Resulting names e.g. "Sheet1", "Sheet2", etc.
        //#. No white-space between the "Sheet" label and sheet index!
        //#, c-format
        return _.noI18n.fix(gt('Sheet%1$d', _.noI18n(index)));
    };

    /**
     * Returns the translated word for 'Table' to be used for new table ranges,
     * ready to be used in the document model (without any I18N marker
     * characters used for UI debugging).
     *
     * @returns {String}
     *  The translated word for 'Table'.
     */
    SheetUtils.getTableName = function () {
        //#. Default name for table ranges (cell ranges in a sheet with filter settings) in spreadsheet
        //#. documents. Will be used to create serially numbered tables, e.g. "Table1", "Table2", etc.
        return _.noI18n.fix(gt.pgettext('filter', 'Table'));
    };

    /**
     * Returns the translated word for 'Column', ready to be used in the
     * document model (without any I18N marker characters used for UI
     * debugging).
     *
     * @returns {String}
     *  The translated word for 'Column'.
     */
    SheetUtils.getTableColName = function () {
        //#. Default name for columns in filtered ranges in spreadsheet documents. Will be
        //#. used to create serially numbered columns, e.g. "Column1", "Column2", etc.
        return _.noI18n.fix(gt.pgettext('filter', 'Column'));
    };

    /**
     * Returns whether the passed merged attribute set will force to split the
     * text contents of a cell into multiple lines, either by the cell
     * attribute 'wrapText' set to true, or by horizontally or vertically
     * justified alignment.
     *
     * @param {Object} attributeSet
     *  A merged (complete) cell attribute set.
     *
     * @returns {Boolean}
     *  Whether the attribute set will force to split the text contents of a
     *  cell into multiple lines.
     */
    SheetUtils.hasWrappingAttributes = function (attributeSet) {
        var cellAttrs = attributeSet.cell;
        return cellAttrs.wrapText || (cellAttrs.alignHor === 'justify') || (cellAttrs.alignVert === 'justify');
    };

    /**
     * Returns the effective text orientation settings for the passed cell
     * value, formatted display string, and horizontal alignment.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value
     *  A scalar cell value. Influences the behavior of the 'auto' alignment.
     *  The value null represents a blank cell.
     *
     * @param {String} align
     *  The horizontal alignment, as used in the cell attribute 'alignHor'. The
     *  value 'auto' will cause special behavior based on the passed cell value
     *  and display string.
     *
     * @param {String} [display]
     *  The formatted display string for the passed cell value. Can be omitted,
     *  or set to null, to return the text orientation based on the cell value
     *  only.
     *
     * @returns {Object}
     *  A descriptor containing various text orientation properties:
     *  - {String} textDir
     *      The effective writing direction in the single text lines of the
     *      display text, as value supported by the HTML element attribute
     *      'dir' (either 'ltr' or 'rtl').
     *  - {String} cssTextAlign
     *      The effective horizontal CSS text alignment, as value supported by
     *      the CSS property 'text-align'. For automatic alignment (parameter
     *      'align' is set to 'auto'), the resulting text alignment depends on
     *      the data type of the passed value. Numbers will be aligned to the
     *      right, boolean values and error codes will be centered, and strings
     *      will be aligned according to their own writing direction (NOT the
     *      passed display text which may differ from the string, e.g. by using
     *      a specific number format). Will be one of the following value:
     *      - 'left': The text lines are aligned to the left in their bounding
     *          box.
     *      - 'center': The text lines are centered in their bounding box.
     *      - 'right': The text lines are aligned to the right in their
     *          bounding box.
     *      - 'justify': The text lines will be expanded to the width of the
     *          bounding box. The alignment of a single text line, or of the
     *          the last text line in multi-line cells is dependent on the
     *          default writing direction of the text.
     *  - {String} baseBoxAlign
     *      The resolved horizontal alignment of the single text lines inside
     *      their bounding box. Will be equal to the property 'cssTextAlign'
     *      except for justified alignment (property cssTextAlign contains the
     *      value 'justify') which will be resolved to 'left' or 'right',
     *      depending on the default writing direction of the text value (NOT
     *      the display string).
     */
    SheetUtils.getTextOrientation = function (value, align, display) {

        // the effective writing direction of the display text
        var displayDir = (typeof display === 'string') ? Font.getTextDirection(display) : LocaleData.DIR;

        // the effective CSS text alignment
        var cssTextAlign = CSS_TEXT_ALIGN_VALUES[align];
        if (!cssTextAlign) {
            switch (typeof value) {
                // strings (depends on writing direction of the text result, not display string!)
                case 'string':
                    cssTextAlign = getAlignFromDir((value === display) ? displayDir : Font.getTextDirection(value));
                    break;
                // numbers (always right aligned, independent from display string)
                case 'number':
                    cssTextAlign = 'right';
                    break;
                default:
                    // blank cells (depending on system writing direction)
                    // booleans and error codes (always centered)
                    cssTextAlign = (value === null) ? getAlignFromDir(Font.DEFAULT_TEXT_DIRECTION) : 'center';
            }
        }

        // base box alignment in justified cells depends on writing direction of display string
        var baseBoxAlign = (cssTextAlign === 'justify') ? getAlignFromDir(displayDir) : cssTextAlign;

        // return the orientation descriptor object
        return { textDir: displayDir, cssTextAlign: cssTextAlign, baseBoxAlign: baseBoxAlign };
    };

    /**
     * Returns a single-character key for an outer cell border.
     *
     * @param {Boolean} columns
     *  Whether to return the key of a vertical border (true), or of a
     *  horizontal border (false).
     *
     * @param {Boolean} leading
     *  Whether to return the key of a leading border (true), or of a trailing
     *  border (false).
     *
     * @returns {String}
     *  One of the border keys 't', 'b', 'l', or 'r'.
     */
    SheetUtils.getOuterBorderKey = function (columns, leading) {
        return columns ? (leading ? 'l' : 'r') : (leading ? 't' : 'b');
    };

    /**
     * Returns a single-character key for an inner cell border.
     *
     * @param {Boolean} columns
     *  Whether to return the key of the vertical inner border (true), or of
     *  the horizontal inner border (false).
     *
     * @returns {String}
     *  One of the border keys 'h', or 'v'.
     */
    SheetUtils.getInnerBorderKey = function (columns) {
        return columns ? 'v' : 'h';
    };

    /**
     * Returns the attribute name of a cell border.
     *
     * @param {String} key
     *  The single-character key of a cell border. MUST be one of 't', 'b',
     *  'l', 'r', 'd', 'u', 'h', or 'v'.
     *
     * @returns {String}
     *  The name of the specified border attribute.
     */
    SheetUtils.getBorderName = function (key) {
        return BORDER_ATTRIBUTE_NAMES[key];
    };

    /**
     * Returns the attribute name of an outer cell border.
     *
     * @param {Boolean} columns
     *  Whether to return the name of a vertical border attribute (true), or of
     *  a horizontal border attribute (false).
     *
     * @param {Boolean} leading
     *  Whether to return the name of a leading border attribute (true), or of
     *  a trailing border attribute (false).
     *
     * @returns {String}
     *  One of the border attribute names 'borderTop', 'borderBottom',
     *  'borderLeft', or 'borderRight'.
     */
    SheetUtils.getOuterBorderName = function (columns, leading) {
        return BORDER_ATTRIBUTE_NAMES[SheetUtils.getOuterBorderKey(columns, leading)];
    };

    /**
     * Returns the attribute name of an inner cell border.
     *
     * @param {Boolean} columns
     *  Whether to return the name of the vertical inner border attribute
     *  (true), or of the horizontal inner border attribute (false).
     *
     * @returns {String}
     *  One of the border attribute names 'borderInsideHor', or
     *  'borderInsideVert'.
     */
    SheetUtils.getInnerBorderName = function (columns) {
        return BORDER_ATTRIBUTE_NAMES[SheetUtils.getInnerBorderKey(columns)];
    };

    /**
     * Returns the effective horizontal padding between cell grid lines and the
     * text contents of the cell, according to the width of the digits of a
     * specific font.
     *
     * @param {Number} digitWidth
     *  The width of the digits of a specific font, in pixels.
     *
     * @returns {Number}
     *  The effective horizontal cell content padding, in pixels.
     */
    SheetUtils.getTextPadding = function (digitWidth) {
        // text padding is 1/4 of the digit width
        return Math.ceil(digitWidth / 4);
    };

    /**
     * Returns the total size of all horizontal padding occupied in a cell that
     * cannot be used for the cell contents, according to the width of the
     * digits of a specific font. This value includes the text padding (twice,
     * for left and right border), and additional space needed for the trailing
     * grid line.
     *
     * @param {Number} digitWidth
     *  The width of the digits of a specific font, in pixels.
     *
     * @returns {Number}
     *  The total size of the horizontal cell content padding, in pixels.
     */
    SheetUtils.getTotalCellPadding = function (digitWidth) {
        // padding at left and right cell border, and 1 pixel for the grid line
        return 2 * SheetUtils.getTextPadding(digitWidth) + 1;
    };

    /**
     * Returns whether the specified address is an origin cell in the merged
     * range according to the passed merge mode.
     *
     * @param {Address} address
     *  The cell address to be checked.
     *
     * @param {Range} mergedRange
     *  The address of the cell range to be merged.
     *
     * @param {MergeMode} mergeMode
     *  The merge mode.
     *
     * @returns {Boolean}
     *  Whether the specified address is an origin cell in the merged range.
     */
    SheetUtils.isOriginCell = function (address, mergedRange, mergeMode) {
        switch (mergeMode) {
            case MergeMode.MERGE:      return mergedRange.startsAt(address);
            case MergeMode.HORIZONTAL: return mergedRange.start[0] === address[0];
            case MergeMode.VERTICAL:   return mergedRange.start[1] === address[1];
        }
        return false;
    };

    /**
     * Prepares the passed range addresses, intended to be used by various cell
     * iterators.
     *
     * @param {RangeArray|Range} ranges
     *  An array of range addresses, or a single cell range address.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  - {Boolean} [options.reverse=false]
     *      If set to true, the ranges will be prepared for a reverse iterator.
     *      If a start position inside the ranges has been specified, the parts
     *      before this start position will be included in the result. Without
     *      reverse mode, the parts after the start position will be returned.
     *  - {Address} [options.startIndex=0]
     *      If specified, the array index of the cell range in the passed
     *      range array that will be the first range in the result. MUST BE a
     *      valid array index. MUST remain zero, if a single range has been
     *      passed to this method.
     *  - {Address} [options.startAddress]
     *      If specified, the address of a cell inside the first cell range
     *      (see option "startIndex"). The cell address MUST BE located inside
     *      the start range. The start range will be split and shortened to the
     *      trailing parts (or leading parts in reverse mode) according to this
     *      start address.
     *  - {Boolean} [options.skipStart=false]
     *      If set to true, the specified start address will not be included in
     *      the result ranges. This option has no effect without the option
     *      "startAddress".
     *  - {Boolean} [options.wrap=false]
     *      If set to true, the leading parts of the ranges (in reverse mode:
     *      the trailing parts of the ranges), according to the start position,
     *      will be appended to the resulting cell ranges. This option has no
     *      effect, if no custom start range or start address has been set
     *      (options "startIndex" and "startAddress").
     *
     * @returns {RangeArray}
     *  The resulting array of cell range addresses. Each range object in this
     *  array contains the following additional properties:
     *  - {Range} orig
     *      The original cell range address. May be larger, if using the option
     *      "startAddress" which leads to split the start range into multiple
     *      parts.
     *  - {Number} index
     *      The array index of the original range in the range array passed to
     *      this method (or zero, if a single range has been passed).
     */
    SheetUtils.getIteratorRanges = function (ranges, options) {

        // whether to generate the ranges for a reverse iterator
        var reverse = Utils.getBooleanOption(options, 'reverse', false);
        // a specific start array index
        var startIndex = Utils.getIntegerOption(options, 'startIndex', 0);
        // a specific start address
        var startAddress = Utils.getOption(options, 'startAddress', null);
        // whether to skip the start address in the beginning
        var skipStart = Utils.getBooleanOption(options, 'skipStart', false);
        // whether to wrap at the end of the range array
        var wrap = Utils.getBooleanOption(options, 'wrap', false);

        // insert original range addresses, and array indexes, to the ranges
        var cloneRanges = RangeArray.map(ranges, function (range, index) {
            var cloneRange = range.clone();
            cloneRange.orig = range;
            cloneRange.index = index;
            return cloneRange;
        });

        // no further processing required without custom start position
        if (!startAddress && (startIndex === 0)) {
            return cloneRanges;
        }

        // the start range according to the passed options
        var startRange = cloneRanges[startIndex];
        // the effective ranges, in the correct order according to the passed options
        var resultRanges = new RangeArray();

        // appends a new range (part of the start range) to the array of effective iterator ranges
        function appendRange(col1, row1, col2, row2) {
            var range = Range.create(col1, row1, col2, row2);
            range.orig = startRange.orig;
            range.index = startIndex;
            resultRanges.push(range);
        }

        // add the parts of the range array following the start address (in forward mode, or in wrapping mode)
        if (!reverse || wrap) {
            // add the trailing parts of the start range
            if (startAddress) {
                var startCol = startAddress[0] + ((reverse !== skipStart) ? 1 : 0);
                if (startCol === startRange.start[0]) {
                    appendRange(startRange.start[0], startAddress[1], startRange.end[0], startRange.end[1]);
                } else {
                    if (startCol <= startRange.end[0]) {
                        appendRange(startCol, startAddress[1], startRange.end[0], startAddress[1]);
                    }
                    if (startAddress[1] < startRange.end[1]) {
                        appendRange(startRange.start[0], startAddress[1] + 1, startRange.end[0], startRange.end[1]);
                    }
                }
            } else if (!reverse) {
                // no start address: add the start range in forward mode (in backward mode, it must be the last array element)
                resultRanges.push(startRange);
            }
            // add the ranges following the start range
            resultRanges.append(cloneRanges.slice(startIndex + 1));
        }

        // add the parts of the range array preceding the start address (in backward mode, or in wrapping mode)
        if (reverse || wrap) {
            // add the ranges preceding the start range
            resultRanges.append(cloneRanges.slice(0, startIndex));
            // add the leading parts of the start range
            if (startAddress) {
                var endCol = startAddress[0] - ((reverse === skipStart) ? 1 : 0);
                if (endCol === startRange.end[0]) {
                    appendRange(startRange.start[0], startRange.start[1], startRange.end[0], startAddress[1]);
                } else {
                    if (startRange.start[1] < startAddress[1]) {
                        appendRange(startRange.start[0], startRange.start[1], startRange.end[0], startAddress[1] - 1);
                    }
                    if (startRange.start[0] <= endCol) {
                        appendRange(startRange.start[0], startAddress[1], endCol, startAddress[1]);
                    }
                }
            } else if (reverse) {
                // no start address: add the start range in backward mode (in forward mode, it must be the first array element)
                resultRanges.push(startRange);
            }
        }

        return resultRanges;
    };

    // exports ================================================================

    return SheetUtils;

});
