/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/paneutils', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/render/rectangle',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/spreadsheet/utils/range'
], function (Utils, KeyCodes, Rectangle, Border, Range) {

    'use strict';

    // configuration for pane sides (header panes)
    var PANE_SIDE_DATA = {
        left:   { nextSide: 'right',  panePos1: 'topLeft',    panePos2: 'bottomLeft',  columns: true,  leading: true,  anchorName: 'anchorLeft' },
        right:  { nextSide: 'left',   panePos1: 'topRight',   panePos2: 'bottomRight', columns: true,  leading: false, anchorName: 'anchorRight' },
        top:    { nextSide: 'bottom', panePos1: 'topLeft',    panePos2: 'topRight',    columns: false, leading: true,  anchorName: 'anchorTop' },
        bottom: { nextSide: 'top',    panePos1: 'bottomLeft', panePos2: 'bottomRight', columns: false, leading: false, anchorName: 'anchorBottom' }
    };

    // configuration for pane positions (grid panes)
    var PANE_POS_DATA = {
        topLeft:     { colSide: 'left',  rowSide: 'top',    nextColPos: 'topRight',    nextRowPos: 'bottomLeft' },
        topRight:    { colSide: 'right', rowSide: 'top',    nextColPos: 'topLeft',     nextRowPos: 'bottomRight' },
        bottomLeft:  { colSide: 'left',  rowSide: 'bottom', nextColPos: 'bottomRight', nextRowPos: 'topLeft' },
        bottomRight: { colSide: 'right', rowSide: 'bottom', nextColPos: 'bottomLeft',  nextRowPos: 'topRight' }
    };

    // used to map pane side identifiers to pane position identifiers
    var PANE_SIDE_TO_PANE_POS = {
        left: { top: 'topLeft', bottom: 'bottomLeft' },
        right: { top: 'topRight', bottom: 'bottomRight' }
    };

    // all predefined zoom factor steps
    var ZOOM_FACTORS = [0.5, 0.75, 1, 1.25, 1.5, 1.75, 2, 2.5, 3, 4, 5, 6, 8];

    // static class PaneUtils =================================================

    var PaneUtils = {};

    // constants --------------------------------------------------------------

    /**
     * All available pane side identifiers, as array of strings.
     *
     * @constant
     * @type Array<String>
     */
    PaneUtils.ALL_PANE_SIDES = _.keys(PANE_SIDE_DATA);

    /**
     * All available pane position identifiers, as array of strings.
     *
     * @constant
     * @type Array<String>
     */
    PaneUtils.ALL_PANE_POSITIONS = _.keys(PANE_POS_DATA);

    /**
     * The minimum size of grid/header panes, in pixels.
     *
     * @constant
     * @type Number
     */
    PaneUtils.MIN_PANE_SIZE = 10;

    /**
     * The ratio of the header/grid pane size to be rendered additionally
     * before and after the visible area.
     *
     * @constant
     * @type Number
     */
    PaneUtils.ADDITIONAL_SIZE_RATIO = Utils.RETINA ? 0.25 : Utils.COMPACT_DEVICE ? 0.3 : 0.5;

    /**
     * Minimum allowed zoom factor in the spreadsheet view.
     *
     * @constant
     * @type Number
     */
    PaneUtils.MIN_ZOOM = ZOOM_FACTORS[0];

    /**
     * Maximum allowed zoom factor in the spreadsheet view.
     *
     * @constant
     * @type Number
     */
    PaneUtils.MAX_ZOOM = _.last(ZOOM_FACTORS);

    // static methods ---------------------------------------------------------

    /**
     * Returns whether the passed pane side represents a column header pane.
     *
     * @param {String} paneSide
     *  The pane side identifier (one of 'left', 'right', 'top', or 'bottom').
     *
     * @returns {Boolean}
     *  True for the pane side identifiers 'left' and 'right'; false otherwise.
     */
    PaneUtils.isColumnSide = function (paneSide) {
        return PANE_SIDE_DATA[paneSide].columns;
    };

    /**
     * Returns whether the passed pane side represents a leading header pane.
     *
     * @param {String} paneSide
     *  The pane side identifier (one of 'left', 'right', 'top', or 'bottom').
     *
     * @returns {Boolean}
     *  True for the pane side identifiers 'left' and 'top'; false otherwise.
     */
    PaneUtils.isLeadingSide = function (paneSide) {
        return PANE_SIDE_DATA[paneSide].leading;
    };

    /**
     * Returns the name of the scroll anchor view property for the passed pane
     * side.
     *
     * @param {String} paneSide
     *  The pane side identifier (one of 'left', 'right', 'top', or 'bottom').
     *
     * @returns {String}
     *  The name of the scroll anchor sheet view property.
     */
    PaneUtils.getScrollAnchorName = function (paneSide) {
        return PANE_SIDE_DATA[paneSide].anchorName;
    };

    /**
     * Returns the column pane side identifier for the passed grid pane
     * position.
     *
     * @param {String} panePos
     *  The pane position (one of 'topLeft', 'topRight', 'bottomLeft', or
     *  'bottomRight').
     *
     * @returns {String}
     *  The pane side 'left' for the pane positions 'topLeft' and 'bottomLeft',
     *  or 'right' for the pane positions 'topRight' and 'bottomRight'.
     */
    PaneUtils.getColPaneSide = function (panePos) {
        return PANE_POS_DATA[panePos].colSide;
    };

    /**
     * Returns the row pane side identifier for the passed grid pane position.
     *
     * @param {String} panePos
     *  The pane position (one of 'topLeft', 'topRight', 'bottomLeft', or
     *  'bottomRight').
     *
     * @returns {String}
     *  The pane side 'top' for the pane positions 'topLeft' and 'topRight', or
     *  'bottom' for the pane positions 'bottomLeft' and 'bottomRight'.
     */
    PaneUtils.getRowPaneSide = function (panePos) {
        return PANE_POS_DATA[panePos].rowSide;
    };

    /**
     * Returns the pane position of the horizontal neighbor of the passed grid
     * pane position.
     *
     * @param {String} panePos
     *  The pane position (one of 'topLeft', 'topRight', 'bottomLeft', or
     *  'bottomRight').
     *
     * @returns {String}
     *  The pane position of the horizontal neighbor (for example, returns
     *  'bottomLeft' for the pane position 'bottomRight').
     */
    PaneUtils.getNextColPanePos = function (panePos) {
        return PANE_POS_DATA[panePos].nextColPos;
    };

    /**
     * Returns the pane position of the vertical neighbor of the passed grid
     * pane position.
     *
     * @param {String} panePos
     *  The pane position (one of 'topLeft', 'topRight', 'bottomLeft', or
     *  'bottomRight').
     *
     * @returns {String}
     *  The pane position of the vertical neighbor (for example, returns
     *  'topRight' for the pane position 'bottomRight').
     */
    PaneUtils.getNextRowPanePos = function (panePos) {
        return PANE_POS_DATA[panePos].nextRowPos;
    };

    /**
     * Returns the pane position identifier according to the passed pane side
     * identifiers.
     *
     * @param {String} colPaneSide
     *  The column pane side ('left' or 'right').
     *
     * @param {String} rowPaneSide
     *  The row pane side ('top' or 'bottom').
     *
     * @returns {String}
     *  The pane position for the passed pane side identifiers.
     */
    PaneUtils.getPanePos = function (colPaneSide, rowPaneSide) {
        return PANE_SIDE_TO_PANE_POS[colPaneSide][rowPaneSide];
    };

    /**
     * Returns the pane position nearest to the specified source pane position,
     * but matching the specified target pane side.
     *
     * @param {String} panePos
     *  The source pane position (one of 'topLeft', 'topRight', 'bottomLeft',
     *  or 'bottomRight').
     *
     * @param {String} paneSide
     *  The target pane side (one of 'left', 'right', 'top', or 'bottom').
     *
     * @returns {String}
     *  The grid pane position of the nearest neighbor, matching the specified
     *  pane side (for example, returns 'topRight' for the source pane position
     *  'bottomRight' and target pane side 'top').
     */
    PaneUtils.getNextPanePos = function (panePos, paneSide) {
        var panePosData = PANE_POS_DATA[panePos];
        var paneSideData = PANE_SIDE_DATA[paneSide];
        return ((paneSide === panePosData.colSide) || (paneSide === panePosData.rowSide)) ? panePos :
            (paneSideData.columns ? panePosData.nextColPos : panePosData.nextRowPos);
    };

    // zoom helpers -----------------------------------------------------------

    PaneUtils.getValidZoom = function (zoom) {
        return Utils.minMax(zoom, PaneUtils.MIN_ZOOM, PaneUtils.MAX_ZOOM);
    };

    PaneUtils.getPrevPresetZoom = function (zoom) {
        // find last entry in ZOOM_FACTORS with a factor less than current zoom
        return Utils.findLast(ZOOM_FACTORS, function (presetZoom) { return presetZoom < zoom; }, { sorted: true }) || null;
    };

    PaneUtils.getNextPresetZoom = function (zoom) {
        // find first entry in ZOOM_FACTORS with a factor greater than current zoom
        return Utils.findFirst(ZOOM_FACTORS, function (presetZoom) { return zoom < presetZoom; }, { sorted: true }) || null;
    };

    // style helpers ----------------------------------------------------------

    /**
     * Returns the CSS position and size attributes for the passed rectangle in
     * pixels, as HTML mark-up value for the 'style' element attribute.
     *
     * @param {Object} rectangle
     *  The rectangle to be converted to CSS position and size attributes, in
     *  the properties 'left', 'top', 'width', and 'height'.
     *
     * @returns {String}
     *  The value for the 'style' element attribute in HTML mark-up text.
     */
    PaneUtils.getRectangleStyleMarkup = function (rectangle) {
        return 'left:' + rectangle.left + 'px;top:' + rectangle.top + 'px;width:' + rectangle.width + 'px;height:' + rectangle.height + 'px;';
    };

    /**
     * Returns the identifier of a predefined border style for the passed mixed
     * border object.
     *
     * @param {Object} mixedBorder
     *  The mixed border to be converted to a predefined border style.
     *
     * @returns {String|Null}
     *  The identifier of a predefined border style, either the string 'none',
     *  or two string tokens separated by a colon character. The first token
     *  represents the line style as used in all border attribute values
     *  (except 'none', see above), the second token represents the line width
     *  (one of 'hair', 'thin', 'medium', or 'thick'). If the passed mixed
     *  border could not be resolved to a border style, null will be returned
     *  instead.
     */
    PaneUtils.getPresetStyleForBorder = function (mixedBorder) {

        // special identifier 'none' regardless of line width (even not existing)
        if (mixedBorder.style === 'none') {
            return 'none';
        }

        // ambiguous line style or width: return null
        if ((typeof mixedBorder.style !== 'string') || (typeof mixedBorder.width !== 'number') || (mixedBorder.width === 0)) {
            return null;
        }

        // concatenate the line style and the line width into a single identifier
        return mixedBorder.style + ':' + Border.getPresetForWidth(mixedBorder.width);
    };

    /**
     * Returns an incomplete border attribute value for the passed predefined
     * border style.
     *
     * @param {String} presetStyle
     *  The identifier of a predefined border style, as returned by the method
     *  PaneUtils.getPresetStyleForBorder().
     *
     * @returns {Object}
     *  An incomplete border attribute value, containing the properties 'style'
     *  (line style), and 'width' (line width in 1/100 of millimeters). The
     *  property 'color' will not be inserted into the border value.
     */
    PaneUtils.getBorderForPresetStyle = function (presetStyle) {

        // the tokens in the passed preset style
        var tokens = presetStyle.split(':');
        // resulting border value
        var border = { style: 'none', width: 0 };

        if ((tokens.length === 2) && (tokens[0].length > 0) && (tokens[1].length > 0)) {
            border.style = tokens[0];
            border.width = Border.getWidthForPreset(tokens[1]);
        } else if (presetStyle !== 'none') {
            Utils.warn('PaneUtils.getBorderForPresetStyle(): invalid preset style "' + presetStyle + '"');
        }

        return border;
    };

    // events -----------------------------------------------------------------

    /**
     * Returns the GUI selection mode according to the modifier keys in the
     * passed GUI event.
     *
     * @param {jQuery.Event} event
     *  The event object received by any jQuery event handlers (keyboard,
     *  mouse, tracking).
     *
     * @returns {String}
     *  The selection mode according to the keyboard modifier keys:
     *  - 'select': Standard selection without modifier keys.
     *  - 'append': Append new range to current selection (CTRL/META key).
     *  - 'extend': Extend current active range (SHIFT key).
     */
    PaneUtils.getSelectionMode = function (event) {
        if (!event.shiftKey && (event.ctrlKey || event.metaKey)) { return 'append'; }
        if (event.shiftKey && !event.ctrlKey && !event.metaKey) { return 'extend'; }
        return 'select';
    };

    /**
     * Returns the move direction for the active cell according to the passed
     * GUI keyboard event.
     *
     * @param {jQuery.Event} event
     *  The 'keydown' event object received by any jQuery event handlers.
     *
     * @returns {String|Null}
     *  The value null, if the passed event would not move the active cell;
     *  otherwise, the move direction for the active cell:
     *  - 'left' for SHIFT+TAB,
     *  - 'right' for TAB key,
     *  - 'up' for SHIFT+ENTER,
     *  - 'down' for ENTER key.
     */
    PaneUtils.getCellMoveDirection = function (event) {
        if (KeyCodes.matchKeyCode(event, 'ENTER', { shift: null })) {
            return event.shiftKey ? 'up' : 'down';
        }
        if (KeyCodes.matchKeyCode(event, 'TAB', { shift: null })) {
            return event.shiftKey ? 'left' : 'right';
        }
        return null;
    };

    // class HeaderBoundary ===================================================

    /**
     * Represents a bounding column/row interval in a header pane, together
     * with the exact pixel position of the bounding area. This pixel position
     * may not be aligned exactly to the start position of the first
     * column/row, or the end position of the last column/row in the interval.
     *
     * @constructor
     *
     * @property {Interval|Null} interval
     *  The column/row indexes of the bounding interval; or null, if this
     *  instance represents an invalid (hidden) header boundary.
     *
     * @property {Object|Null} position
     *  The exact position and size of the bounding area, in pixels (in the
     *  properties 'offset' and 'size'). May result in a start position before
     *  or inside the first column/row of the interval, or in an end position
     *  after or inside the last column/row.
     */
    function HeaderBoundary(interval, position) {

        if (interval && position) {
            this.interval = interval.clone();
            this.position = _.clone(position);
        } else {
            this.interval = this.position = null;
        }

    } // class HeaderBoundary

    // public methods ---------------------------------------------------------

    /**
     * Returns a deep clone of this instance.
     *
     * @returns {HeaderBoundary}
     *  A deep clone of this instance.
     */
    HeaderBoundary.prototype.clone = function () {
        return new HeaderBoundary(this.interval, this.position);
    };

    /**
     * Returns whether this instance contains a valid interval and position
     * (neither of the properties is null).
     *
     * @returns {Boolean}
     *  Whether this instance contains a valid interval and position.
     */
    HeaderBoundary.prototype.isValid = function () {
        return (this.interval !== null) && (this.position !== null);
    };

    // class RangeBoundary ====================================================

    /**
     * Represents a bounding range in a grid pane, together with the exact
     * pixel position of the bounding rectangle. This pixel position may not be
     * aligned exactly to the outer borders of the bounding range.
     *
     * @constructor
     *
     * @property {Range|Null} range
     *  The address of the bounding range; or null, if this instance represents
     *  an invalid (hidden) range boundary.
     *
     * @property {Rectangle|Null} rectangle
     *  The exact pixel position of the bounding rectangle, in pixels. May
     *  result in a start position before or inside the start cell of the
     *  range, or in an end position after or inside the last cell.
     *
     * @param {HeaderBoundary|Null} colBoundary
     *  The column interval and position represented by this instance.
     *
     * @param {HeaderBoundary|Null} rowBoundary
     *  The row interval and position represented by this instance.
     */
    function RangeBoundary(colBoundary, rowBoundary) {

        if (colBoundary && colBoundary.isValid() && rowBoundary && rowBoundary.isValid()) {
            this.range = Range.createFromIntervals(colBoundary.interval, rowBoundary.interval);
            this.rectangle = new Rectangle(colBoundary.position.offset, rowBoundary.position.offset, colBoundary.position.size, rowBoundary.position.size);
        } else {
            this.range = this.rectangle = null;
        }

    } // class RangeBoundary

    // public methods ---------------------------------------------------------

    /**
     * Returns a (deep) clone of this instance.
     *
     * @returns {RangeBoundary}
     *  A (deep) clone of this instance.
     */
    RangeBoundary.prototype.clone = function () {
        var clone = new RangeBoundary();
        clone.range = this.range ? this.range.clone() : null;
        clone.rectangle = this.rectangle ? this.rectangle.clone() : null;
        return clone;
    };

    /**
     * Returns whether this instance contains a valid range address and pixel
     * position (neither of the properties is null).
     *
     * @returns {Boolean}
     *  Whether this instance contains a valid range address and rectangle.
     */
    RangeBoundary.prototype.isValid = function () {
        return (this.range !== null) && (this.rectangle !== null);
    };

    // exports ================================================================

    // export helper classes
    PaneUtils.HeaderBoundary = HeaderBoundary;
    PaneUtils.RangeBoundary = RangeBoundary;

    return PaneUtils;

});
