/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/intervalset', [
    'io.ox/office/tk/container/valueset',
    'io.ox/office/tk/container/sortedarray',
    'io.ox/office/spreadsheet/utils/interval',
    'io.ox/office/spreadsheet/utils/intervalarray'
], function (ValueSet, SortedArray, Interval, IntervalArray) {

    'use strict';

    // convenience shortcuts
    var min = Math.min;
    var max = Math.max;
    var floor = Math.floor;
    var ceil = Math.ceil;
    var log = Math.log;
    var SetProto = ValueSet.prototype;

    var ZEROS = '000000000';

    // private global functions ===============================================

    /**
     * Returns the key of the passed index interval.
     *
     * @param {Interval} interval
     *  An index interval.
     *
     * @returns {String}
     *  The key of the passed index interval.
     */
    function getKey(interval) {
        return interval.key();
    }

    /**
     * Converts the passed number to a 10-digit string with leading zeros.
     *
     * @param {Number} number
     *  The number to be converted to a string.
     *
     * @returns {String}
     *  The passed number, as string with 10 digits and leading zeros.
     */
    function fillZeros(number) {
        return (ZEROS + number).slice(-10);
    }

    /**
     * Sorter callback for a SortedArray instance that stores index intervals
     * sorted by their first index.
     *
     * @param {Interval} interval
     *  The index interval to calculate the sort index for.
     *
     * @returns {String}
     *  A unique key for the passed interval that causes to sort intervals by
     *  their start position.
     */
    function firstSorter(interval) {
        return fillZeros(interval.first) + fillZeros(interval.last);
    }

    /**
     * Sorter callback for a SortedArray instance that stores index intervals
     * sorted by their last index.
     *
     * @param {Interval} interval
     *  The index interval to calculate the sort index for.
     *
     * @returns {String}
     *  A unique key for the passed interval that causes to sort intervals by
     *  their end position.
     */
    function lastSorter(interval) {
        return fillZeros(interval.last) + fillZeros(interval.first);
    }

    /**
     * Returns the dual logarithm of the passed number.
     *
     * @param {Number} number
     *  The number whose dual logarithm shall be returned.
     *
     * @returns {Number}
     *  The dual logarithm of the passed number.
     */
    var log2 = Math.log2 || function (number) { return log(number) / Math.LN2; };

    // class IntervalTree =====================================================

    /**
     * Internal implementation helper for the class IntervalSet that provides
     * optimized methods for insertion, deletion, and access of intervals.
     *
     * This class implements a centered interval tree. Each instance contains
     * four data structures:
     * - A subtree (another instance of this class) for all intervals that end
     *      before the center index of the bounding interval.
     * - A subtree (another instance of this class) for all intervals that start
     *      after the center index of the bounding interval.
     * - An array for all intervals that cover the center point, sorted by the
     *      start position of the intervals.
     * - An array for all intervals that cover the center point, sorted by the
     *      end position of the intervals.
     *
     * When inserting a new interval, it will either be inserted into one of
     * the subtrees, or into both sorted arrays.
     *
     * When removing an interval, it will either be removed from one of the
     * subtrees, or from both sorted arrays.
     *
     * When looking up intervals for a target index, one of the subtrees will
     * be asked (according to the position of the index relative to the center
     * point), and one of the sorted arrays will be evaluated.
     *
     * Effectively, all operations have complexity O(log N), with N being the
     * number of intervals contained in this tree.
     *
     * @constructor
     *
     * @param {Number} first
     *  The first index of the bounding interval (the 'universe' of this tree).
     *  The intervals inserted in this tree MUST NOT exceed this limit to the
     *  left.
     *
     * @param {Number} last
     *  The last index of the bounding interval (the 'universe' of this tree).
     *  The intervals inserted in this tree MUST NOT exceed this limit to the
     *  right.
     *
     * @param {Number} center
     *  The effective center index of the bounding interval. MUST be inside the
     *  interval specified by the parameters 'first' and 'last'.
     *
     * @param {IntervalTree|Null}
     *  An existing left subtree for the new tree.
     *
     * @param {IntervalTree|Null}
     *  An existing right subtree for the new tree.
     */
    function IntervalTree(first, last, center, tree1, tree2) {

        // the bounding interval of this tree
        this._first = first;
        this._last = last;
        this._center = center;

        // subtree (instance of this class) containing all intervals ending before the center index
        this._tree1 = tree1;
        // subtree (instance of this class) containing all intervals starting after the center index
        this._tree2 = tree2;

        // an array of all intervals covering the center point, sorted by start index
        this._sorted1 = new SortedArray(firstSorter);
        // an array of all intervals covering the center point, sorted by end index
        this._sorted2 = new SortedArray(lastSorter);

    } // class IntervalTree

    // public methods ---------------------------------------------------------

    /**
     * Returns whether this tree is empty.
     *
     * @returns {Boolean}
     *  Whether this tree is empty.
     */
    IntervalTree.prototype.empty = function () {
        return !this._tree1 && !this._tree2 && this._sorted1.empty();
    };

    /**
     * Returns a shallow or deep clone of this interval tree.
     *
     * @param {Boolean|Function} [deep=false]
     *  If set to true or to a function, a deep clone will be created (all
     *  intervals in this set will be cloned); otherwise a shallow clone will
     *  be created (the intervals in this set will be inserted by reference
     *  into the clone). If a function has been passed, it will be invoked for
     *  each cloned interval, receiving the new cloned interval inserted into
     *  the cloned set as first parameter, and the old interval of this set as
     *  second parameter.
     *
     * @returns {IntervalTree}
     *  The cloned interval tree.
     */
    IntervalTree.prototype.clone = function (deep) {

        // create an empty interval tree, and clone the subtrees
        var tree1 = this._tree1 ? this._tree1.clone(deep) : null;
        var tree2 = this._tree2 ? this._tree2.clone(deep) : null;
        var clone = new IntervalTree(this._first, this._last, this._center, tree1, tree2);

        // clone the first array (deeply if necessary)
        clone._sorted1 = this._sorted1.clone(!!deep);

        // insert the cloned elements into the second array (do NOT clone deeply again!)
        if (deep) {
            clone._sorted1.forEach(function (interval) {
                clone._sorted2.insert(interval, true);
            });
        } else {
            clone._sorted2 = this._sorted2.clone();
        }

        // invoke the callback function ONCE for each interval in the sorted arrays
        if (_.isFunction(deep)) {
            var newIntervals = clone._sorted1.values();
            var oldIntervals = this._sorted1.values();
            newIntervals.forEach(function (newInterval, index) {
                deep(newInterval, oldIntervals[index]);
            });
        }

        return clone;
    };

    /**
     * Returns whether this tree contains an interval that overlaps with the
     * passed index interval.
     *
     * @param {Interval} interval
     *  The index interval used to look for an interval in this set.
     *
     * @returns {Boolean}
     *  Whether this tree contains an interval overlapping with the passed
     *  index interval.
     */
    IntervalTree.prototype.overlaps = function (interval) {

        if (interval.last < this._center) {
            var interval1 = this._sorted1.first();
            return (!!interval1 && (interval1.first <= interval.last)) || (!!this._tree1 && this._tree1.overlaps(interval));
        }

        if (this._center < interval.first) {
            var interval2 = this._sorted2.last();
            return (!!interval2 && (interval.first <= interval2.last)) || (!!this._tree2 && this._tree2.overlaps(interval));
        }

        // interval covers the center point of this tree: non-empty sorted array contains a matching
        // interval, otherwise search in both trees, if the interval exceeds the center point
        return !this._sorted1.empty() ||
            ((interval.first < this._center) && this._tree1 && this._tree1.overlaps(interval)) ||
            ((this._center < interval.last) && this._tree2 && this._tree2.overlaps(interval));
    };

    /**
     * Returns all intervals in this tree that overlap with the passed index
     * interval.
     *
     * @param {Interval} interval
     *  The index interval used to look for the intervals in this tree.
     *
     * @returns {IntervalArray}
     *  An array with all intervals in this tree overlapping with the passed
     *  index interval, in no specific order.
     */
    IntervalTree.prototype.findIntervals = function (interval) {

        var intervals = new IntervalArray();

        // passed interval covers the center point of this tree: return all elements of the sorted array
        var containsCenter = interval.containsIndex(this._center);
        if (containsCenter) { intervals.append(this._sorted1.values()); }

        // collect all intervals from the left side of the center point
        if (interval.first < this._center) {
            // collect all intervals of the left subtree
            if (this._tree1) {
                var last1 = min(interval.last, this._center - 1);
                intervals.append(this._tree1.findIntervals(new Interval(interval.first, last1)));
            }
            // collect all intervals from the first-sorted array
            if (!containsCenter) {
                var desc1 = this._sorted1.find(fillZeros(interval.last + 1));
                var array1 = this._sorted1.values();
                intervals.append(desc1 ? array1.slice(0, desc1._ai) : array1);
            }
        }

        // collect all intervals from the right side of the center point
        if (this._center < interval.last) {
            // collect all intervals of the right subtree
            if (this._tree2) {
                var first2 = max(interval.first, this._center + 1);
                intervals.append(this._tree2.findIntervals(new Interval(first2, interval.last)));
            }
            // collect all intervals from the last-sorted array
            if (!containsCenter) {
                var desc2 = this._sorted2.find(fillZeros(interval.first));
                if (desc2) { intervals.append(this._sorted2.values().slice(desc2._ai)); }
            }
        }

        return intervals;
    };

    /**
     * Inserts a new index interval into this tree.
     *
     * @param {Interval} interval
     *  The index interval to be inserted into this tree.
     *
     * @param {Boolean} [replace=false]
     *  If set to true, an existing interval with equal start and end position
     *  as the passed interval will be replaced with the new interval. By
     *  default, the existing interval will be kept, and the passed interval
     *  will be ignored. May be important in situations where the intervals in
     *  the tree contain additional user settings in arbitrary properties.
     *
     * @returns {Interval}
     *  A reference to the interval contained in this tree. This may not be the
     *  same as the passed interval, e.g. if an equal interval already exists
     *  in this tree, and the flag 'replace' has not been set.
     */
    IntervalTree.prototype.insert = function (interval, replace) {

        var subtree = null;

        if (interval.last < this._center) {
            subtree = this._tree1 || (this._tree1 = new IntervalTree(this._first, this._center - 1, ceil((this._first + this._center - 1) / 2), null, null));
            interval = subtree.insert(interval, replace);
        } else if (this._center < interval.first) {
            subtree = this._tree2 || (this._tree2 = new IntervalTree(this._center + 1, this._last, ceil((this._center + this._last + 1) / 2), null, null));
            interval = subtree.insert(interval, replace);
        } else {
            this._sorted1.insert(interval, replace);
            interval = this._sorted2.insert(interval, replace);
        }

        return interval;
    };

    /**
     * Removes the specified index interval from this tree.
     *
     * @param {Interval} interval
     *  The index interval to be removed from this tree.
     *
     * @returns {Interval|Null}
     *  A reference to the removed interval; or null, if this tree does not
     *  contain an interval that is equal to the passed interval.
     */
    IntervalTree.prototype.remove = function (interval) {

        if (interval.last < this._center) {
            interval = this._tree1 ? this._tree1.remove(interval) : null;
            if (interval && this._tree1.empty()) { this._tree1 = null; }
        } else if (this._center < interval.first) {
            interval = this._tree2 ? this._tree2.remove(interval) : null;
            if (interval && this._tree2.empty()) { this._tree2 = null; }
        } else {
            this._sorted1.remove(interval);
            interval = this._sorted2.remove(interval);
        }

        return interval;
    };

    // class IntervalSet ======================================================

    /**
     * A unique set of index intervals. This class provides optimized methods
     * for insertion, deletion, and access of index intervals, while keeping
     * the set unique (two different interval objects in this set will never
     * have the same start end end index).
     *
     * @constructor
     *
     * @extends ValueSet
     */
    var IntervalSet = ValueSet.extend(function () {

        // base constructor
        ValueSet.call(this, getKey);

        // the centered interval tree storing all intervals
        this._tree = null;

    }); // class IntervalSet

    // public methods ---------------------------------------------------------

    /**
     * Returns whether this set is empty.
     *
     * @returns {Boolean}
     *  Whether this set is empty.
     */
    IntervalSet.prototype.empty = function () {
        return !this._tree;
    };

    /**
     * Removes all elements from this set.
     *
     * @returns {IntervalSet}
     *  A reference to this instance.
     */
    IntervalSet.prototype.clear = function () {
        SetProto.clear.call(this);
        this._tree = null;
        return this;
    };

    /**
     * Returns all index intervals contained in this set as an array.
     *
     * @returns {IntervalArray}
     *  All index intervals contained in this set as an array, in no specific
     *  order.
     */
    IntervalSet.prototype.values = function () {
        var intervals = new IntervalArray();
        this.forEach(function (interval) { intervals.push(interval); });
        return intervals;
    };

    /**
     * Returns a shallow or deep clone of this interval set.
     *
     * @param {Boolean|Function} [deep=false]
     *  If set to true or to a function, a deep clone will be created (all
     *  intervals in this set will be cloned); otherwise a shallow clone will
     *  be created (the intervals in this set will be inserted by reference
     *  into the clone). If a function has been passed, it will be invoked for
     *  each cloned interval, receiving the new cloned interval inserted into
     *  the cloned set as first parameter, and the old interval of this set as
     *  second parameter.
     *
     * @param {Object} [context]
     *  The calling context used if the 'deep' parameter is a callback.
     *
     * @returns {IntervalSet}
     *  The cloned interval set.
     */
    IntervalSet.prototype.clone = function (deep, context) {

        // start with a new empty interval set
        var clone = new IntervalSet();

        // create a shallow or deep clone of the interval tree, and the own map
        if (this._tree) {
            if (deep) {
                // deep mode: collect the cloned intervals in the own set while cloning the tree
                // (cloning the tree is faster than building it with distinct element insertions)
                var callback = (typeof deep === 'function') ? deep : _.noop;
                clone._tree = this._tree.clone(function (newInterval, oldInterval) {
                    callback.call(context, newInterval, oldInterval);
                    clone._insert(newInterval.key(), newInterval);
                });
            } else {
                // shallow mode: simple clones of the tree and the set
                this.forEachKey(clone._insert, clone);
                clone._tree = this._tree.clone();
            }
        }

        return clone;
    };

    /**
     * Returns the interval object from this set that has the exact position of
     * the passed index interval.
     *
     * @param {Interval} interval
     *  An interval to search for.
     *
     * @returns {Interval|Null}
     *  An interval from this collection with the same start and end position
     *  as the passed interval; otherwise null.
     */
    IntervalSet.prototype.get = function (interval) {
        return this._get(interval.key()) || null;
    };

    /**
     * Returns whether this set contains an interval covering the passed index.
     *
     * @param {Number} index
     *  The index used to look for an interval in this set.
     *
     * @returns {Boolean}
     *  Whether this set contains an interval covering the passed index.
     */
    IntervalSet.prototype.containsIndex = function (index) {
        return !!this._tree && this._tree.overlaps(new Interval(index));
    };

    /**
     * Returns all intervals in this set that cover the passed index.
     *
     * @param {Number} index
     *  The index used to look for the intervals in this set.
     *
     * @returns {IntervalArray}
     *  An array with all intervals in this set that cover the passed index, in
     *  no specific order.
     */
    IntervalSet.prototype.findByIndex = function (index) {
        return this._tree ? this._tree.findIntervals(new Interval(index)) : new IntervalArray();
    };

    /**
     * Returns whether this set contains an interval that overlaps with the
     * passed index interval.
     *
     * @param {Interval} interval
     *  The index interval used to look for an interval in this set.
     *
     * @returns {Boolean}
     *  Whether this set contains an interval overlapping with the passed index
     *  interval.
     */
    IntervalSet.prototype.overlaps = function (interval) {
        // quick check for an exact match
        return !!this._tree && (this.has(interval) || this._tree.overlaps(interval));
    };

    /**
     * Returns all intervals in this set that overlap with the passed index
     * interval.
     *
     * @param {Interval} interval
     *  The index interval used to look for the intervals in this set.
     *
     * @returns {IntervalArray}
     *  An array with all intervals in this set overlapping with the passed
     *  index interval, in no specific order.
     */
    IntervalSet.prototype.findIntervals = function (interval) {
        return this._tree ? this._tree.findIntervals(interval) : new IntervalArray();
    };

    /**
     * Inserts a new index interval into this set.
     *
     * @param {Interval} interval
     *  The index interval to be inserted into this set.
     *
     * @param {Boolean} [replace=false]
     *  If set to true, an existing interval with equal start and end position
     *  as the passed interval will be replaced with the new interval. By
     *  default, the existing interval will be kept, and the passed interval
     *  will be ignored. May be important in situations where the intervals in
     *  the set contain additional user settings in arbitrary properties.
     *
     * @returns {Interval}
     *  A reference to the interval contained in this set. This may not be the
     *  same as the passed interval, e.g. if an equal interval already exists
     *  in this set, and the flag 'replace' has not been set.
     */
    IntervalSet.prototype.insert = function (interval, replace) {

        // the start, center, and end point for new interval trees created on demand
        var treeFirst = 0, treeCenter = 0, treeLast = 0;

        // create the smallest possible tree for the very first interval inserted into the set
        if (!this._tree) {

            // nothing to do for interval 0:0 (following bit arithmetic will fail anyway)
            if (interval.last > 0) {

                // Find the optimal center point for the interval. This is a multiple of the largest power of
                // two contained in the passed interval.
                //
                // In mathematical terms: find the largest possible n>=0 so that the number i*2^n (i being an
                // integer) is contained in the interval.
                //
                // Example: The interval 21:26 contains the number 24 (3*2^3 = 3*8) but does not conain a
                // multiple of 2^4 (16). Therefore, an interval tree 17...31 (a tree between the parent center
                // points 16 and 32) with own center point 24 will be used.
                //
                // First, find the highest bit that is different in the first and last index in the passed
                // interval. Reduce first index by one to handle special case that an interval starts at the
                // resulting center point. After that, reduce the end point of the passed interval to the
                // leading bits according to the first different bit which becomes the center point of the tree.
                //
                // Example: The interval 21:26, as binary 00010101:00011010, with reduced starting point
                // 00010100:00011010, differs in most significant bit 3 (XOR results in 00001110). The end point
                // of the interval will be reduced to the leading bits up to that bit 3 (AND with the bit mask
                // 11111000), which results in 00011010 & 11111000 = 00011000 = 24.
                //
                var first = max(interval.first - 1, 0);
                var msb = floor(log2(first ^ interval.last));
                var mask = (1 << msb) - 1;
                treeCenter = interval.last & ~mask;
                treeFirst = (mask + 1 === treeCenter) ? 0 : (treeCenter - mask);
                treeLast = treeCenter + mask;
            }

            // create the initial interval tree
            this._tree = new IntervalTree(treeFirst, treeLast, treeCenter, null, null);

        } else {

            // expand the tree to the left, as long as the passed interval exceeds its start point
            while (interval.first < this._tree._first) {
                treeCenter = this._tree._first - 1;
                treeLast = this._tree._last;
                treeFirst = 2 * treeCenter - treeLast;
                if (treeFirst === 1) { treeFirst = 0; }
                this._tree = new IntervalTree(treeFirst, treeLast, treeCenter, null, this._tree);
            }

            // expand the tree to the right, as long as the passed interval exceeds its end point
            while (this._tree._last < interval.last) {
                treeCenter = this._tree._last + 1;
                treeFirst = this._tree._first;
                treeLast = 2 * treeCenter - treeFirst;
                this._tree = new IntervalTree(treeFirst, treeLast, treeCenter, this._tree, null);
            }
        }

        // add the interval to the tree and the set
        interval = this._tree.insert(interval, replace);
        return this._insert(interval.key(), interval);
    };

    /**
     * Removes the specified index interval from this set.
     *
     * @param {Interval} interval
     *  The index interval to be removed from this set.
     *
     * @returns {Interval|Null}
     *  A reference to the removed interval; or null, if this set does not
     *  contain an interval that is equal to the passed interval.
     */
    IntervalSet.prototype.remove = function (interval) {

        // early exit, if the interval does not exist
        var key = interval.key();
        interval = this._get(key);
        if (!interval) { return null; }

        // remove the interval from the set
        this._remove(key);

        // remove the interval from the tree, and shrink the tree, if one subtree and the sorted arrays are empty
        this._tree.remove(interval);
        while (this._tree && this._tree._sorted1.empty() && !(this._tree._tree1 && this._tree._tree2)) {
            this._tree = this._tree._tree1 || this._tree._tree2;
        }

        return interval;
    };

    /**
     * Inserts all index intervals of the passed set into this set.
     *
     * @param {Object} list
     *  A generic data source with index intervals as elements. See static
     *  method Container.forEach() for details.
     *
     * @param {Boolean} [replace=false]
     *  If set to true, intervals in the passed list will replace equal
     *  intervals in this set. By default, the existing intervals of this set
     *  will be preferred.
     *
     * @returns {IntervalSet}
     *  A reference to this instance.
     */
    IntervalSet.prototype.merge = function (list, replace) {
        ValueSet.forEach(list, function (interval) { this.insert(interval, replace); }, this);
        return this;
    };

    // exports ================================================================

    return IntervalSet;

});
