/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Mario Schroeder <mario.schroeder@open-xchange.com>
 */
define('io.ox/office/spreadsheet/utils/clipboard', [
    'io.ox/office/tk/io',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/iterator',
    'io.ox/office/tk/render/rectangle',
    'io.ox/office/tk/locale/parser',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/hyperlinkutils',
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (IO, Utils, Iterator, Rectangle, Parser, Color, HyperlinkUtils, SheetUtils) {

    'use strict';

    var // convenience shortcuts
        Address = SheetUtils.Address,
        Range = SheetUtils.Range,
        RangeArray = SheetUtils.RangeArray,

        // XML name spaces for Office and Excel custom tags
        EXCEL_NAMESPACES = 'xmlns:o="urn:schemas-microsoft-com:office:office" xmlns:x="urn:schemas-microsoft-com:office:excel" xmlns="http://www.w3.org/TR/REC-html40"',

        // the HTML header
        HTML_HEADER = '<head><meta charset="utf-8"></head>',

        // the localized names of the standard number format
        STANDARD_NUMBER_FORMATS = [
            'Standard',
            'General',
            'Estandarra',
            'Skoueriek',
            'Standaard',
            'Yleinen',
            'Bendras',
            '\u041E\u043F\u0448\u0442\u043E',
            '\u0639\u0627\u062F\u06CC',
            'Geral',
            'Padr\xe3o',
            'Estandar'
        ],

        // regex to check for invalid XML characters; for UTF-16 the high surrogate and the low surrogate need to match.
        INVALID_XML_CHARS_PATTERN = // UTF-8:
                                    '[\\x00-\\x08\\x0B-\\x0C\\x0E-\\x1F\\x7F-\\x84\\x86-\\x9F\\uFDD0-\\uFDDF]|' +
                                    // UTF-16:
                                    // [#x1FFFE-#x1FFFF], [#x2FFFE-#x2FFFF], [#x3FFFE-#x3FFFF],
                                    '\\uD83F[\\uDFFE-\\uDFFF]|\\uD87F[\\uDFFE-\\uDFFF]|\\uD8BF[\\uDFFE-\\uDFFF]|' +
                                    // [#x4FFFE-#x4FFFF], [#x5FFFE-#x5FFFF], [#x6FFFE-#x6FFFF],
                                    '\\uD8FF[\\uDFFE-\\uDFFF]|\\uD93F[\\uDFFE-\\uDFFF]|\\uD97F[\\uDFFE-\\uDFFF]|' +
                                    // [#x7FFFE-#x7FFFF], [#x8FFFE-#x8FFFF], [#x9FFFE-#x9FFFF],
                                    '\\uD9BF[\\uDFFE-\\uDFFF]|\\uD9FF[\\uDFFE-\\uDFFF]|\\uDA3F[\\uDFFE-\\uDFFF]|' +
                                    // [#xAFFFE-#xAFFFF], [#xBFFFE-#xBFFFF], [#xCFFFE-#xCFFFF],
                                    '\\uDA7F[\\uDFFE-\\uDFFF]|\\uDABF[\\uDFFE-\\uDFFF]|\\uDAFF[\\uDFFE-\\uDFFF]|' +
                                    // [#xDFFFE-#xDFFFF], [#xEFFFE-#xEFFFF], [#xFFFFE-#xFFFFF],
                                    '\\uDB3F[\\uDFFE-\\uDFFF]|\\uDB7F[\\uDFFE-\\uDFFF]|\\uDBBF[\\uDFFE-\\uDFFF]|' +
                                    // [#x10FFFE-#x10FFFF]
                                    '\\uDBFF[\\uDFFE-\\uDFFF]',

        INVALID_XML_CHARS_REGEX = new RegExp(INVALID_XML_CHARS_PATTERN, 'g');

    // private static functions ===============================================

    /**
     * Returns all descendants of the passed root node matching the passed CSS
     * selector. Additionally, the root node itself will be included in the
     * result set, if it matches the selector.
     *
     * @param {HTMLElement|jQuery|String} rootNode
     *  The root node to be searched for matching elements, or HTML mark-up
     *  text to be searched on-the-fly.
     *
     * @returns {jQuery}
     *  A jQuery collection containing all matching elements.
     */
    function collectMatchingNodes(rootNode, selector) {
        rootNode = $(rootNode);
        return rootNode.filter(selector).add(rootNode.find(selector));
    }

    /**
     * Converts a number format code to be used with the mso-number-format style of Excel.
     *
     * @param {String} format
     *  The generic number format code.
     *
     * @returns {String}
     *  The number format code prepared for Excel.
     */
    function convertNumberFormatToExcel(format) {

        // TODO: the token 'standard' may be part of the format code, e.g.: Standard;-Standard;0
        if (format.toLowerCase() === 'standard') {
            return Parser.GENERAL;
        }

        format = Utils.cleanString(format);

        // replace the ampersand with the text &amp; (must be done first!)
        format = format.replace(/&/g, '&amp;');

        if (_.browser.IE) {
            // use ISO Latin-1 code
            format = format
                // replace space or backslash space with the text &#32;
                .replace(/(\\\s|\s)/g, '&#32;')
                // replace backslash double quote with the text &#92;&#34;
                .replace(/\\"/g, '&#92;&#34;')
                // replace backslash apostrophe with the text &#92;&#39;
                .replace(/\\'/g, '&#92;&#39;')
                // replace backslash percent with the text &#92;&#37;
                .replace(/\\%/g, '&#92;&#37;')
                // replace backslash exclamation mark with the text &#92;&#33;
                .replace(/\\!/g, '&#92;&#33;')
                // replace double backslash the text &#92;&#92;
                .replace(/\\\\/g, '&#92;&#92;')
                // replace double quote, Euro symbol, double quote with the text &#92;&#8364; as work around for IE problems with the double quote
                .replace(/"\u20ac"/g, '&#92;&#8364;');

        } else {
            // use UTF-8 representation
            format = format
                // replace space or backslash space with the text \\0020;
                .replace(/(\\\s|\s)/g, '\\0020')
                // replace backslash double quote with the text \\005c&#34;
                .replace(/\\"/g, '\\005c&#34;')
                // replace backslash percent with the text \\005c&#37;
                .replace(/\\%/g, '\\005c&#37;')
                // replace backslash exclamation mark with the text \\005c&#33;
                .replace(/\\!/g, '\\005c&#33;')
                // replace double backslash the text \\005c;&#92;
                .replace(/\\\\/g, '\\005c\\005c');
        }

        return format
            // replace the left angle bracket with the text &lt;
            .replace(/</g, '&lt;')
            // replace the right angle bracket with the text &gt;
            .replace(/>/g, '&gt;')
            // replace the double quote character with the text &#34;
            .replace(/"/g, '&#34;')
            // replace the apostrophe with the text &#39; (&apos; is not an HTML entity!)
            .replace(/'/g, '&#39;');
    }

    /**
     * Converts a number format code to be used with the sdnum attribute of Calc.
     *
     * @param {String} format
     *  The generic number format code.
     *
     * @returns {String}
     *  The number format code prepared for Calc.
     */
    function convertNumberFormatToCalc(format) {
        // TODO: the token 'standard' may be part of the format code, e.g.: Standard;-Standard;0
        return (format.toLowerCase() === 'standard') ? Parser.GENERAL : Utils.escapeHTML(format);
    }

    /**
     * Returns true for the standard number format.
     *
     * @param {String} format
     *  The number format code as String.
     *
     * @returns {Boolean}
     *  Returns true for the standard number format.
     */
    function isStandardNumberFormat(format) {
        return _.any(STANDARD_NUMBER_FORMATS, function (item) {
            return (_.isString(this) && this.indexOf(item.toLowerCase()) > -1);
        }, format.toLowerCase());
    }

    /**
     * Creates a number format cell style object from a Calc number format
     * string.
     *
     * @param {String|Null} formatCode
     *  The number format code as String.
     *
     * @returns {Object|Null}
     *  Result object, with the properties 'format' and 'lcid'.
     */
    function parseCalcNumberFormat(formatCode) {

        if (!_.isString(formatCode)) { return null; }

        if (isStandardNumberFormat(formatCode)) {
            return { format: 0, lcid: -1 };
        }

        var matches = /(?:(\d{4,5}|0);){0,1}(?:(\d{4,5}|0);){0,1}(.*)/.exec(formatCode);
        var lcid = -1;

        if (matches && matches[3]) {
            matches[2] = parseInt(matches[2], 10);
            if (matches[2] > 0) {
                lcid = matches[2];
            } else {
                matches[1] = parseInt(matches[1], 10);
                lcid = (matches[1] > 0) ? matches[1] : 1033;
            }

            return { format: matches[3], lcid: lcid };
        }

        return null;
    }

    /**
     * Returns the direct child element that matches the given jQuery selector.
     * Is limited to work inside HTML tables, e.g. find a 'strong' as child of a 'td' element.
     *
     * @param {jQuery} element
     *  The jQuery root element to start looking for the child.
     *
     * @returns {jQuery}
     *  The matching child element.
     */
    function findMatchingChild(element, selector) {
        return element.find(selector).filter(function () {
            var parent = $(this).parents('span,a,th,td,tr,tfoot,thead,tbody,colgroup,table').first();
            return parent[0] === element[0];
        });
    }

    /**
     * Returns true if the given jQuery selector matches a direct child element.
     * Is limited to work inside HTML tables, e.g. find a 'strong' as child of a 'td' element.
     *
     * @param {jQuery} element
     *  The jQuery root element to start looking for the child.
     *
     * @returns {Boolean}
     *  Returns true if a child of the given 'element' matches the given 'selector'.
     */
    function hasMatchingChild(element, selector) {
        return findMatchingChild(element, selector).length > 0;
    }

    /**
     * Converts a CSS font size with measurement unit into a HTML font size.
     *
     * @param {String} cssSize
     *  The CSS font size value with its measurement unit to be converted, as String.
     *
     * @returns {Number}
     *  The converted CSS size.
     */
    function convertToFontSize(cssSize) {
        var size = Utils.convertCssLength(cssSize, 'pt');
        if (size > 30) { return 7; }    // 7 -> 36pt
        if (size > 21) { return 6; }    // 6 -> 24pt
        if (size > 16) { return 5; }    // 5 -> 18pt
        if (size > 13) { return 4; }    // 4 -> 14pt
        if (size > 11) { return 3; }    // 3 -> 12pt
        if (size > 9) { return 2; }     // 2 -> 10pt
        if (size > 0) { return 1; }     // 1 -> 8pt
        return 0;
    }

    /**
     * Converts a HTML font size String (used as size attribute inside a font element)
     * into a size in point.
     *
     * @param {String} size
     *  The font size to be converted, as String.
     *
     * @returns {Number}
     *  The converted size, in point.
     */
    function convertFontSizeToPoint(size) {
        switch (size) {
            case '1':
                return 8;
            case '2':
                return 10;
            case '3':
                return 12;
            case '4':
                return 14;
            case '5':
                return 18;
            case '6':
                return 24;
            case '7':
                return 36;
            default:
                return 0;
        }
    }

    /**
     * Returns the character styles as CSS style String.
     *
     * @param {SpreadsheetModel} docModel
     *  The document model.
     *
     * @param {Object|Null} attributeSet
     *  The formatting attributes of a cell, or null for undefined cells.
     *
     * @returns {String}
     *  The CSS style String.
     */
    function getCssCharacterStyles(docModel, attributeSet) {

        if (!attributeSet) { return ''; }

        // attribute map for the 'character' attribute family
        var charAttrs = attributeSet.character;
        // the CSS text decoration
        var textDecoration = 'none';
        // the character style attribute data
        var characterStyle = '';

        characterStyle += 'font-family:' + docModel.getCssFontFamily(charAttrs.fontName) + ';';
        characterStyle += 'font-size:' + charAttrs.fontSize + 'pt;';

        characterStyle += 'font-weight:' + (charAttrs.bold ? 'bold' : 'normal') + ';';
        characterStyle += 'font-style:' + (charAttrs.italic ? 'italic' : 'normal') + ';';

        if (charAttrs.underline) { textDecoration = Utils.addToken(textDecoration, 'underline', 'none'); }
        if (charAttrs.strike !== 'none') { textDecoration = Utils.addToken(textDecoration, 'line-through', 'none'); }
        characterStyle += 'text-decoration:' + textDecoration + ';';

        characterStyle += 'color:' + docModel.getCssTextColor(charAttrs.color, [attributeSet.cell.fillColor]) + ';';
        return characterStyle;
    }

    /**
     * Returns the cell styles as CSS style String.
     *
     * @param {SpreadsheetModel} docModel
     *  The document model.
     *
     * @param {Object|Null} attributeSet
     *  The formatting attributes of a cell, or null for undefined cells.
     *
     * @returns {String}
     *  The CSS style String.
     */
    function getCssCellStyles(docModel, attributeSet) {

        if (!attributeSet) { return ''; }

        // attribute map for the 'cell' attribute family
        var cellAttrs = attributeSet.cell;
        // the cell style attribute data
        var cellStyle = '';

        if (/^(left|center|right|justify)$/.test(cellAttrs.alignHor)) { cellStyle += 'text-align:' + cellAttrs.alignHor + ';'; }
        if (/^(top|middle|bottom|justify)$/.test(cellAttrs.alignVert)) { cellStyle += 'vertical-align:' + cellAttrs.alignVert + ';'; }

        cellStyle += 'white-space:' + (cellAttrs.wrapText ? 'normal' : 'nowrap') + ';';
        cellStyle += 'background-color:' + docModel.getCssColor(cellAttrs.fillColor, 'fill') + ';';

        ['Top', 'Bottom', 'Left', 'Right'].forEach(function (attrPart) {

            var // the cell style attribute name
                oxAttrName = 'border' + attrPart,
                // the CSS attribute name
                cssAttrName = 'border-' + attrPart.toLowerCase() + ':',
                // the border cell style attribute value
                border = cellAttrs[oxAttrName],
                // the effective CSS attributes
                cssAttrs = docModel.getCssBorderAttributes(border);

            cellStyle += cssAttrName;
            cellStyle += ((cssAttrs && cssAttrs.width > 0) ? cssAttrs.style + ' ' + cssAttrs.width + 'px ' + cssAttrs.color : 'none');
            cellStyle += ';';
        });

        return cellStyle;
    }

    function getCssMergedCellBorders(docModel, range, currentStyles, cellCollection, autoStyles) {
        var cellStyle = '';

        ['Top', 'Right', 'Bottom', 'Left'].forEach(function (side) {
            // cleanup current styles from old border-attributes
            currentStyles = currentStyles.split('border-' + side.toLowerCase() + ':none;').join('');

            var adr     = (side === 'Top' || side === 'Left') ? range.start : range.end,
                styleId = cellCollection.getStyleId(adr);

            if (autoStyles.isDefaultStyleId(styleId)) {
                return currentStyles;
            }

            var attributeSet    = autoStyles.getMergedAttributeSet(styleId),
                cellAttrs       = attributeSet.cell;

            var oxAttrName = 'border' + side,
                // the CSS attribute name
                cssAttrName = 'border-' + side.toLowerCase() + ':',
                // the border cell style attribute value
                border = cellAttrs[oxAttrName],
                // the effective CSS attributes
                cssAttrs = docModel.getCssBorderAttributes(border);

            cellStyle += cssAttrName;
            cellStyle += (cssAttrs && cssAttrs.width > 0) ? cssAttrs.style + ' ' + cssAttrs.width + 'px ' + cssAttrs.color : 'none';
            cellStyle += ';';
        });

        return currentStyles + cellStyle;
    }

    /**
     * Converts the passed formatted cell display text to HTML mark-up.
     *
     * @param {String|Null} display
     *  The formatted cell display text. The value null represents a cell that
     *  cannot be formatted to a valid display string with its current number
     *  format.
     *
     * @returns {String}
     *  The encoded display text ready to be inserted into an HTML mark-up
     *  string. The null value will be converted to an empty string.
     */
    function escapeDisplayString(display) {
        // nothing to do for empty strings or null values
        return display ? Utils.escapeHTML(display) : '';
    }

    /**
     * Returns the character styles as a String of the HTML elements tree.
     *
     * @param {SpreadsheetModel} docModel
     *  The document model.
     *
     * @param {Object|Null} display
     *  The cell display string.
     *
     * @param {Object|Null} attributeSet
     *  The formatting attributes of a cell, or null for undefined cells.
     *
     * @param {String} [url]
     *  The URL of the hyperlink attached to the cell.
     *
     * @returns {String}
     *  The HTML elements String.
     */
    function getHtmlCharacterStyles(docModel, display, attributeSet, url) {

        // the character style HTML elements
        var characterStyle = '';
        // attribute map for the 'character' attribute family
        var charAttrs = attributeSet ? attributeSet.character : null;

        if (url && (url = HyperlinkUtils.checkForHyperlink(url))) {
            characterStyle += '<a href="' + url + '">';
        }

        if (attributeSet) {

            // create opening tags
            if (charAttrs.bold) { characterStyle += '<b>'; }
            if (charAttrs.italic) { characterStyle += '<i>'; }
            if (charAttrs.underline) { characterStyle += '<u>'; }
            if (charAttrs.strike !== 'none') { characterStyle += '<s>'; }

            characterStyle += '<font ';
            // To make it work in Calc we need to use just the font name instead of the entire CSS font family
            characterStyle += 'face="' + Utils.escapeHTML(charAttrs.fontName) + '" ';
            // We need to use rgb colors, like #RRGGBB
            characterStyle += 'color="' + docModel.getCssTextColor(charAttrs.color, [attributeSet.cell.fillColor]) + '" ';
            // Add the font-size CSS also here to support precise font sizes in Excel
            characterStyle += 'size="' + convertToFontSize(charAttrs.fontSize) + '" style="font-size:' + charAttrs.fontSize + 'pt;"';
            characterStyle += '>';
        }

        // add cell value (nothing to do for empty strings or null
        if (display) { characterStyle += escapeDisplayString(display); }

        if (attributeSet) {
            // create closing tags
            characterStyle += '</font>';

            if (charAttrs.strike !== 'none') { characterStyle += '</s>'; }
            if (charAttrs.underline) { characterStyle += '</u>'; }
            if (charAttrs.italic) { characterStyle += '</i>'; }
            if (charAttrs.bold) { characterStyle += '</b>'; }
        }

        if (url) { characterStyle += '</a>'; }

        return characterStyle;
    }

    /**
     * Returns the cell styles as HTML attributes String.
     *
     * @param {SpreadsheetModel} docModel
     *  The document model.
     *
     * @param {Object|Null} attributeSet
     *  The formatting attributes of a cell, or null for undefined cells.
     *
     * @returns {String}
     *  The HTML attributes String.
     */
    function getHtmlCellStyles(docModel, attributeSet) {

        if (!attributeSet) { return ''; }

        // attribute map for the 'cell' attribute family
        var cellAttrs = attributeSet.cell;
        // the cell style attribute data
        var cellStyle = '';

        if (/^(left|center|right|justify)$/.test(cellAttrs.alignHor)) { cellStyle += ' align="' + cellAttrs.alignHor + '"'; }
        if (/^(top|middle|bottom)$/.test(cellAttrs.alignVert)) { cellStyle += ' valign="' + cellAttrs.alignVert + '"'; }
        if (!Color.parseJSON(cellAttrs.fillColor).isAuto()) { cellStyle += ' bgcolor="' + docModel.getCssColor(cellAttrs.fillColor, 'fill') + '"'; }

        return cellStyle;
    }

    /**
     * Returns the attribute map for the 'character' attribute family of the
     * HTML element.
     *
     * @param {jQuery} element
     *  The HTML element.
     *
     * @returns {Object}
     *  The character attribute map.
     */
    function getCharAttributes(element) {

        // attribute map for the 'character' attribute family
        var charAttrs = {};
        // the <font> element, if HTML elements are used instead of CSS
        var fontElement = findMatchingChild(element, 'font');

        var fontName = fontElement.attr('face') || element.css('font-family');
        if (_.isString(fontName) && fontName.length > 0) {
            fontName = (fontName.split(',')[0]).replace(/["']/g, '');
            charAttrs.fontName = fontName;
        }

        var fontSize = convertFontSizeToPoint(fontElement.attr('size')) || Utils.convertCssLength(element.css('font-size'), 'pt');
        if (fontSize > 0) {
            charAttrs.fontSize = fontSize;
        }

        if (hasMatchingChild(element, 'em,i')) {
            charAttrs.italic = true;
        } else {
            var fontStyle = element.css('font-style');
            if (_.isString(fontStyle) && fontStyle.length > 0) {
                charAttrs.italic = fontStyle === 'italic';
            }
        }

        if (hasMatchingChild(element, 'strong,b')) {
            charAttrs.bold = true;
        } else {
            var fontWeight = element.css('font-weight');
            if (_.isString(fontWeight) && fontWeight.length > 0) {
                charAttrs.bold = /^(bold|bolder)$/.test(fontWeight) || (parseInt(fontWeight, 10) >= 600);
            }
        }

        var textDecoration = element.css('text-decoration');
        if (hasMatchingChild(element, 'u')) {
            charAttrs.underline = true;
        } else if (_.isString(textDecoration) && textDecoration.length > 0) {
            charAttrs.underline = textDecoration.indexOf('underline') >= 0;
        }
        if (hasMatchingChild(element, 's')) {
            charAttrs.strike = 'single';
        } else if (_.isString(textDecoration) && textDecoration.length > 0) {
            charAttrs.strike = (textDecoration.indexOf('line-through') >= 0) ? 'single' : 'none';
        }

        var color = Color.parseCSS(fontElement.attr('color') || element.css('color'));
        if (color) {
            charAttrs.color = color.toJSON();
        }

        return charAttrs;
    }

    /**
     * Returns the attribute map for the 'cell' attribute family of the HTML
     * element.
     *
     * @param {jQuery} element
     *  The HTML element.
     *
     * @returns {Object}
     *  The cell attribute map.
     */
    function getCellAttributes(element) {

        // attribute map for the 'cell' attribute family
        var cellAttrs = {};

        // map CSS border style to operation border line style
        function mapCssBorderStyle(style) {
            switch (style) {
                case 'solid':
                case 'groove':
                case 'ridge':
                case 'inset':
                case 'outset':
                    return 'single';
                case 'double':
                    return 'double';
                case 'dotted':
                    return 'dotted';
                case 'dashed':
                    return 'dashed';
                default:
                    return 'none';
            }
        }

        // check value and filter prefix (webkit-, moz-, ms-)
        var alignHor = element.attr('align') || element.css('text-align');
        alignHor = _.isString(alignHor) ? /(?:^|-)(left|center|right|justify)$/i.exec(alignHor) : null;
        if (alignHor) { cellAttrs.alignHor = alignHor[1].toLowerCase(); }

        // check value and filter prefix (webkit-, moz-, ms-)
        var alignVert = element.attr('valign') || element.css('vertical-align');
        alignVert = _.isString(alignVert) ? /(?:^|-)(top|middle|bottom|justify)$/i.exec(alignVert) : null;
        if (alignVert) { cellAttrs.alignVert = alignVert[1].toLowerCase(); }

        var wrapText = element.css('white-space');
        if (_.isString(wrapText) && wrapText.length > 0) {
            cellAttrs.wrapText = wrapText !== 'nowrap';
        }

        var fillColor = Color.parseCSS(element.attr('bgcolor') || element.css('background-color'), true);
        if (fillColor) {
            cellAttrs.fillColor = fillColor.toJSON();
        }

        ['Top', 'Bottom', 'Left', 'Right'].forEach(function (attrPart) {

            var // the element style attribute name
                oxAttrName = 'border' + attrPart,
                // the CSS attribute name
                cssAttrName = 'border-' + attrPart.toLowerCase(),
                // the border style
                borderStyle = mapCssBorderStyle(element.css(cssAttrName + '-style')),
                // the border width
                borderWidth,
                // the border color
                borderColor;

            if (borderStyle !== 'none') {

                if (_.browser.Firefox && element[0]) {
                    // On FF element.css(cssAttrName + '-width') is always 0px, bug #35125
                    borderWidth = element[0].style[oxAttrName + 'Width'] || element.css(cssAttrName + '-width');
                } else {
                    borderWidth = element.css(cssAttrName + '-width');
                }
                borderColor = Color.parseCSS(element.css(cssAttrName + '-color'));

                cellAttrs[oxAttrName] = {
                    style: borderStyle,
                    width: Utils.convertCssLengthToHmm(borderWidth),
                    color: borderColor ? borderColor.toJSON() : Color.AUTO
                };
            }

        });

        return cellAttrs;
    }

    function extractElementAttributes(docModel, attrData, element) {

        docModel.getCellAttributePool().extendAttributeSet(attrData.attrs, {
            character: getCharAttributes(element),
            cell: getCellAttributes(element)
        });

        // add format code information
        return _.extend(attrData, parseCalcNumberFormat(element.attr('sdnum')));
    }

    /**
     * Creates a DOM node for a drawing object, and adds data attributes
     * describing the position of the drawing object relative to the bounding
     * rectangle containing all drawing objects copied to the clipboard.
     */
    function createDataNodeFor(drawingModel, rectangle, boundRect, modelData) {

        // the data node to be returned
        var dataNode = $('<div data-type="' + drawingModel.getType() + '">');

        // adjust rectangle relative to bounding rectangle, and the explicit attributes
        dataNode.attr({
            'data-left': rectangle.left - boundRect.left,
            'data-top': rectangle.top - boundRect.top,
            'data-width': rectangle.width,
            'data-height': rectangle.height,
            'data-attrs': JSON.stringify(drawingModel.getExplicitAttributeSet(true))
        });

        // add passed custom JSON data
        if (modelData) { dataNode.attr('data-json', JSON.stringify(modelData)); }

        return dataNode;
    }

    /**
     * Returns the BASE64 encoded image data URL, whose data URL is for the
     * given (image) drawing attributes.
     *
     * @param {Object} mergedAttr
     *  The merged image drawing attributes, containing the image width, height
     *  and source data or URL.
     *
     * @returns {String}
     *  The image data URL, containing the image data.
     */
    function getImageDataUrl(app, mergedAttr) {

        var imageAttr = mergedAttr.image,
            imageUrl = null;

        if (imageAttr.imageData) {
            imageUrl = imageAttr.imageData;
        } else if (imageAttr.imageUrl) {

            if (/^(https?:\/\/|s?ftps?:)/.test(imageAttr.imageUrl)) {
                imageUrl = imageAttr.imageUrl;
            } else {
                var canvas = document.createElement('canvas'),
                    jqImageNode =  $('<img>', { src: app.getServerModuleUrl(IO.FILTER_MODULE_NAME, {
                        action: 'getfile',
                        get_filename: imageAttr.imageUrl
                    }, { uid: false }) });

                canvas.width = jqImageNode[0].naturalWidth || jqImageNode.width() || Math.floor(mergedAttr.drawing.width * 96 / 2540.0);
                canvas.height = jqImageNode[0].naturalHeight || jqImageNode.height() || Math.floor(mergedAttr.drawing.height * 96 / 2540.0);

                // retrieve image data as image/png to avoid
                // quality losses with lossy, compressed formats
                try {
                    var canvasCtx = canvas.getContext('2d');
                    canvasCtx.drawImage(jqImageNode[0], 0, 0);
                    imageUrl = canvas.toDataURL('image/png');
                } catch (ex) {
                    Utils.warn('cannot convert image to data URL');
                }
            }
        }

        return imageUrl || 'data:,';
    }

    /**
     * Returns the CSS style Object for the line/border
     * attributes of the image.
     *
     * @param {Object} mergedAttr
     *  The merged image attributes, containing the line attrs
     *
     * @returns {Object}
     *  The CSS style object, containing line/border attributes.
     */
    function getImageLineStyle(app, mergedAttr) {
        var lineAttr = mergedAttr.line;

        return (lineAttr.type === 'solid' ? {
            borderStyle: lineAttr.style,
            borderWidth: (lineAttr.width * 0.001).toString() + 'cm',
            borderColor: app.getModel().getCssColor(lineAttr.color, 'line')
        } : null);
    }

    /**
     * Creates a HTML node, containing the data, necessary to
     * reconstruct an image object later in time (e.g. at paste)
     */
    function createDataNodeForImage(app, imageModel, rectangle, boundRect) {

        // create the DOM node representing the image object
        var dataNode = createDataNodeFor(imageModel, rectangle, boundRect),
            imageNode = $('<img>'),
            mergedAttr = imageModel.getMergedAttributeSet(true),
            lineStyle = getImageLineStyle(app, mergedAttr);

        // add image attributes
        imageNode.attr({
            width: rectangle.width,
            height: rectangle.height,
            src: getImageDataUrl(app, mergedAttr)
        });

        // add image border style, if set
        if (lineStyle) {
            imageNode.css(lineStyle);
        }

        return dataNode.append(imageNode);
    }

    /**
     *
     */
    function createDataNodeForChart(chartModel, rectangle, boundRect) {

        if (!chartModel.isRestorable()) { return null; }

        // the JSON data to be transported in the DOM node
        var chartData = {
            modelData: chartModel.getCloneData()
        };

        // create the DOM node representing the chart object
        var dataNode = createDataNodeFor(chartModel, rectangle, boundRect, chartData);

        // try to create a replacement image that will be used when pasting into other documents
        dataNode.append(chartModel.getReplacementNode({ mimeType: 'image/png' }));

        return dataNode;
    }

    // static class Clipboard =================================================

    var Clipboard = {};

    // static methods ---------------------------------------------------------

    /**
     * Creates a HTML text from the selected drawings.
     *
     * @param {Array<DrawingModel>} drawingModels
     *  The array of selected drawing models
     *
     * @returns {String|Null}
     *  The HTML mark-up of a DOM container node with data nodes for the passed
     *  drawing objects. If none of the drawing objects could be converted to
     *  DOM nodes, null will be returned instead.
     */
    Clipboard.convertDrawingModels = function (app, drawingModels) {

        // the positions of all drawing objects
        var rectangles = drawingModels.map(function (drawingModel) { return drawingModel.getRectangle(); });
        // the bounding rectangle of all drawings for the top-left position
        var boundRect = rectangles.reduce(function (boundRect, rect) { return boundRect ? boundRect.boundary(rect) : rect; }, null);
        // the resulting container up for all drawing obejcts
        var containerNode = $('<div>');

        // process all drawing models
        drawingModels.forEach(function (drawingModel, index) {

            // the location of the drawing object (already calculated above)
            var rectangle = rectangles[index];
            // the resulting data node
            var dataNode = null;

            // generate the DOM representation of the drawing model
            switch (drawingModel.getType()) {
                case 'image':
                    dataNode = createDataNodeForImage(app, drawingModel, rectangle, boundRect);
                    break;
                case 'chart':
                    dataNode = createDataNodeForChart(drawingModel, rectangle, boundRect);
                    break;
            }

            // insert the data node into the container
            if (dataNode) { containerNode.append(dataNode); }
        });

        // nothing valid to copy
        if (containerNode.children().length === 0) { return ''; }

        // initialize the container node
        containerNode.attr({
            id: 'ox-clipboard-data',
            'data-app-guid': app.getGlobalUid(),
            'data-ox-drawings': true
        });

        return '<html ' + EXCEL_NAMESPACES + '>' + HTML_HEADER + '<body>' + containerNode[0].outerHTML + '</body></html>';
    };

    /**
     * Creates a HTML table from the active range of the current selection.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing the cells.
     *
     * @param {Range} range
     *  The range address.
     *
     * @param {String} clientClipboardId
     *  The client clipboard id to identify the copy data when pasting.
     *
     * @returns {String}
     *  The resulting HTML mark-up of the <table> element.
     */
    Clipboard.getHtmlMarkupFromRange = function (sheetModel, range, clientClipboardId, options) {

        function getOxCellData(adr) {
            var // cell model
                cellModel = cellCollection.getCellModel(adr),
                // style id
                styleId = cellCollection.getStyleId(adr),
                // the merged attribute set of the cell
                attributeSet = autoStyles.isDefaultStyleId(styleId) ? null : autoStyles.getMergedAttributeSet(styleId),
                // data
                oxCellData = {};

            // handle formula
            var opFormula = cellCollection.getFormula(adr, 'op:ooxml');
            if (opFormula) {
                oxCellData.formula = opFormula;
                if (cellModel.mr) {
                    oxCellData.mr = cellModel.mr.toString();
                }
            }

            oxCellData.attributes = attributeSet ? {
                apply: attributeSet.apply,
                cell: attributeSet.cell,
                character: attributeSet.character
                //FIXME: no styleId, there is a bug in calc!!!
                //styleId: attributeSet.styleId
            } : {};
            oxCellData.styleId = styleId;
            oxCellData.explicit = autoStyles.getExplicitAttributeSet(styleId);
            oxCellData.value = cellModel ? cellModel.v : null;

            return oxCellData;
        }

        var // the document model
            docModel = sheetModel.getDocModel(),
            // the collection of auto-styles
            autoStyles = docModel.getCellAutoStyles(),
            // formula grammar for native formulas in operations
            formulaGrammar = docModel.getFormulaGrammar('op'),
            // the cell collection of the passed sheet
            cellCollection = sheetModel.getCellCollection(),
            // the merged ranges covering the passed range
            mergedRanges = sheetModel.getMergeCollection().getMergedRanges(range),
            // the formula parser
            tokenArray = sheetModel.createCellTokenArray(),
            // the col, the range starts with
            startCol = range.start[0],
            // the row, the range starts with
            startRow = range.start[1],
            // the current table row
            row = range.start[1],
            // the tableMarkup string
            tableMarkup = '',
            // whether it is a cut event (or a copy)
            cut = Utils.getBooleanOption(options, 'cut', false),
            // the collection of cell-styles
            cellStyles = docModel.getCellStyles();

        tableMarkup += '<html ' + EXCEL_NAMESPACES + '>' + HTML_HEADER;
        tableMarkup += '<body><table id="ox-clipboard-data"';

        if (clientClipboardId) {
            tableMarkup += ' data-ox-clipboard-id="' + clientClipboardId + '"';
        }

        tableMarkup += ' data-ox-clipboard-type="' + (cut ? 'cut' : 'copy') + '"';
        tableMarkup += ' data-app-guid="' + docModel.getApp().getGlobalUid() + '"';
        tableMarkup += ' data-range="' + range.toString() + '"';
        tableMarkup += ' data-sheet="' + docModel.getActiveSheet() + '"';

        // the conditional formattings covering the passed range
        var ruleModels = sheetModel.getCondFormatCollection().findRules(range);
        var condFormats = ruleModels.reduce(function (resultArray, ruleModel) {

            // calculate the intersection ranges
            var targetRanges = ruleModel.getTargetRanges().intersect(range);
            if (!targetRanges.empty()) {

                // relocate the target ranges to cell A1
                targetRanges.forEach(function (targetRange) {
                    targetRange.moveBoth(-startCol, true);
                    targetRange.moveBoth(-startRow, false);
                });
                var attrs = ruleModel.getExplicitAttributeSet().rule;
                if (attrs.attrs && attrs.attrs.styleId) {
                    attrs.attrs = cellStyles.getStyleAttributeSet(attrs.attrs.styleId);
                }
                // add an entry for the rule to the result array
                resultArray.push({
                    id: ruleModel.getRuleId(),
                    ranges: targetRanges,
                    attrs: attrs
                });
            }

            return resultArray;
        }, []);
        if (condFormats.length > 0) {
            tableMarkup += ' data-ox-cond-formats="' + Utils.escapeHTML(JSON.stringify(condFormats)) + '"';
        }

        tableMarkup += '>';
        tableMarkup += '<tbody><tr>';

        var iterator = cellCollection.createAddressIterator(range, { visible: false, covered: true });
        Iterator.forEach(iterator, function (address) {

            // merged range covering the current cell
            var mergedRange = mergedRanges.findByAddress(address);

            var colspan,
                rowspan,
                style,
                excelNumberFormat,
                calcNumberFormat;

            // check if the cell is the reference cell of a merged collection
            var isReferenceCell = mergedRange && mergedRange.startsAt(address);

            // check if we need to add a new row to the tableMarkup
            if (row !== address[1]) {
                tableMarkup += '</tr><tr>';
                row = address[1];
            }

            // render cell if it's the reference cell or if it's not a merged cell
            if (isReferenceCell || !mergedRange) {

                // the complete cell model (value, formula, style), may be null
                var cellModel = cellCollection.getCellModel(address);
                // the auto-style identifier
                var styleId = cellCollection.getStyleId(address);
                // the merged attribute set of the cell
                var attributeSet = autoStyles.isDefaultStyleId(styleId) ? null : autoStyles.getMergedAttributeSet(styleId);
                // the merged attribute set of existing cells with custom formatting
                var parsedFormat = autoStyles.isDefaultStyleId(styleId) ? null : autoStyles.getParsedFormat(styleId);

                var cellStyles = getCssCellStyles(docModel, attributeSet);

                tableMarkup += '<td ';

                if (isReferenceCell) {
                    // add cell with colspan and rowspan
                    colspan = mergedRange.end[0] - mergedRange.start[0] + 1;
                    rowspan = mergedRange.end[1] - mergedRange.start[1] + 1;

                    tableMarkup += (colspan > 1) ? 'colspan="' + colspan + '" ' : '';
                    tableMarkup += (rowspan > 1) ? 'rowspan="' + rowspan + '" ' : '';

                    cellStyles = getCssMergedCellBorders(docModel, mergedRange, cellStyles, cellCollection, autoStyles);
                }

                // add cell and character style attributes
                style = Utils.escapeHTML(getCssCharacterStyles(docModel, attributeSet) + cellStyles);

                // handle Excel number format
                excelNumberFormat = parsedFormat ? convertNumberFormatToExcel(parsedFormat.formatCode) : null;
                if (excelNumberFormat) {
                    style += 'mso-number-format:\'' + excelNumberFormat + '\';';
                }

                if (style.length > 0) {
                    tableMarkup += ' style="' + style + '"';
                }

                // set the cell value type
                if (cellModel) {
                    if (cellModel.isNumber()) {
                        tableMarkup += ' x:num="' + cellModel.v + '"';
                        tableMarkup += ' sdval="' + cellModel.v + '"';
                    } else if (cellModel.isBoolean()) {
                        tableMarkup += ' x:bool="' + cellModel.v + '"';
                    } else if (cellModel.isError()) {
                        tableMarkup += ' x:err="' + escapeDisplayString(formulaGrammar.getErrorName(cellModel.v)) + '"';
                    } else if (cellModel.isText()) {
                        tableMarkup += ' x:str="' + escapeDisplayString(cellModel.v) + '"';
                    }
                }

                // handle Calc number format
                calcNumberFormat = parsedFormat ? convertNumberFormatToCalc(parsedFormat.formatCode) : null;
                if (calcNumberFormat) {
                    tableMarkup += ' sdnum="1033;1033;' + calcNumberFormat + '"';
                }

                var oxCellData = getOxCellData(address);

                if (oxCellData.formula) {
                    tokenArray.parseFormula('op', oxCellData.formula, { refAddress: range.start });

                    var enFormula = tokenArray.getFormula('en');

                    Utils.log('external clipboard - formula - range:', range, ', original: ', oxCellData.formula, ', relocated: ', enFormula);

                    tableMarkup += ' x:fmla="' + Utils.escapeHTML(enFormula) + '"';
                    tableMarkup += ' formula="' + Utils.escapeHTML(enFormula) + '"';
                }

                if (isReferenceCell) {
                    oxCellData.mergedRangeCells = {};
                    oxCellData.mergedRange = false;

                    Iterator.forEach(cellCollection.createAddressIterator(mergedRange, { type: 'defined', covered: true }), function (adr) {
                        var m_col = (adr[0] - startCol),
                            m_row = (adr[1] - startRow);

                        oxCellData.mergedRangeCells['col' + m_col.toString() + '_row' + m_row.toString()] = getOxCellData(adr);
                        oxCellData.mergedRange = true;
                    });
                }

                tableMarkup += ' data-ox-cell-data="' + Utils.escapeHTML(JSON.stringify(oxCellData)) + '"';

                // add HTML attributes
                tableMarkup += getHtmlCellStyles(docModel, attributeSet);

                tableMarkup += '>';

                // add HTML elements and cell content (also URLs returned from HYPERLINK function in cell formulas)
                var url = cellCollection.getEffectiveURL(address);
                tableMarkup += getHtmlCharacterStyles(docModel, cellModel ? cellModel.d : '', attributeSet, url);

                tableMarkup += '</td>';
            }
        });

        tableMarkup += '</tr></tbody></table></body></html>';

        return tableMarkup;
    };

    /**
     * Creates a plain text representation from the active range of the current
     * selection.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing the cells.
     *
     * @param {Range} range
     *  The range address.
     *
     * @returns {String}
     *  The resulting plain text string.
     */
    Clipboard.getPlainTextFromRange = function (sheetModel, range) {

        // the resulting TAB-separated display strings
        var resultText = '';
        // the cell collection of the active sheet
        var cellCollection = sheetModel.getCellCollection();

        var iterator = cellCollection.createAddressIterator(range, { visible: true, covered: true });
        Iterator.forEach(iterator, function (address) {
            if (address[0] > range.start[0]) {
                resultText += '\t';
            } else if (address[1] > range.start[1]) {
                resultText += '\r\n';
            }
            resultText += cellCollection.getDisplayString(address) || '';
        });

        return resultText;
    };

    /**
     * Retrieves the client clipboard ID from the HTML data.
     *
     * @param {HTMLElement|jQuery|String} rootNode
     *  The root node to be searched for the clipboard identifier, or HTML
     *  mark-up text to be searched on-the-fly.
     *
     * @returns {String|Null}
     *  The client clipboard ID or undefined.
     */
    Clipboard.getClientClipboardId = function (rootNode) {
        var clipboardId = collectMatchingNodes(rootNode, '#ox-clipboard-data').attr('data-ox-clipboard-id');
        return clipboardId ? clipboardId : null;
    };

    /**
     * Retrieves the GUID of the source application from the passed HTML data.
     *
     * @param {HTMLElement|jQuery|String} rootNode
     *  The root node to be searched for the application identifier, or HTML
     *  mark-up text to be searched on-the-fly.
     *
     * @returns {String|Null}
     *  The GUID of the source application from the HTML data; or null, if no
     *  identifier has been found.
     */
    Clipboard.getApplicationGuid = function (rootNode) {
        var appGuid = collectMatchingNodes(rootNode, '#ox-clipboard-data').attr('data-app-guid');
        return appGuid ? appGuid : null;
    };

    /**
     * Returns true if the HTML contains exactly one element of the given type.
     *
     * @param {String|jQuery} html
     *  The HTML data to check.
     *
     * @param {String} query
     *  The element to search the hml node for,
     *  but only one arbitrary level deep
     *
     * @returns {jQuery|Null}
     *  Returns the matched jQuery object if such an object is found.
     *  or null otherwise
     */
    Clipboard.getContainedHtmlElements = function (html, query) {
        var jqResult = html.filter(query);
        if (!jqResult.length) {
            jqResult = html.find(query);
        }
        return (jqResult.length > 0) ? jqResult : null;
    };

    /**
     * Removes characters that are invalid in XML from the given string
     * and returns a new valid string.
     *
     * @param {String} inputStr
     *  The string to clean up.
     *
     * @returns {String}
     *  The result string containing only valid XML characters.
     */
    Clipboard.removeInvalidXmlChars = function (inputStr) {
        if (!_.isString(inputStr) || inputStr.length === 0) {
            return '';
        }
        return inputStr.replace(INVALID_XML_CHARS_REGEX, '');
    };

    /**
     * Parses the HTML from the clipboard data for a table and
     * creates the cell contents and merged cell collection.
     *
     * @param {SpreadsheetView} docView
     *  The document view that processes the paste event.
     *
     * @param {String|jQuery} html
     *  The HTML input to parse.
     *
     * @returns {Object}
     *  An object with the following attributes:
     *  - {Array} contents
     *      A two-dimensional array containing the values and attribute sets
     *      for the cells.
     *  - {RangeArray} mergedRanges
     *      An array of merged cell ranges.
     *  - {Object} urlMap
     *      A map with URLs as map keys, and range arrays (instances of the
     *      class RangeArray) as values.
     *
     */
    Clipboard.parseHTMLData = function (docView, html) {

        // convert HTML mark-up string to DOM tree
        html = $(html);

        var // the document model
            docModel = docView.getDocModel(),
            // the attribute pool to be used for cell attributes
            attributePool = docModel.getCellAttributePool(),
            // a two dimensional array containing the cell contents
            contents = [],
            // an array of ranges of merged cells
            mergedRanges = new RangeArray(),
            // the row nodes
            rows,
            // the number of columns
            colCount,
            // start address of the clipboard range
            sourceStart = null,
            sourceRange = null,
            sourceSheet = null,
            // copy from the same document
            sameDoc = false,
            // copy from (other) ox-spreadsheet
            intern = false,
            // stores all found merged ranges
            foundMergedRanges = {},
            // whether it is a cut-event (or a copy)
            cut = false;

        // Bug 35126: Excel sends CSS rules containing fractional border widths, e.g.
        // 'border-top: .5pt solid ...' which will be translated to 0px borders in Chrome.
        // Only solution is to manually parse and modify the <style> element contained in
        // the pasted HTML data.
        html.find('>style').each(function () {

            // the first and only child node of the <style> element must be a text node
            if ((this.childNodes.length !== 1) || (this.firstChild.nodeType !== 3)) { return; }

            // the text contents of the <style> node
            var cssText = this.firstChild.nodeValue;

            // text must be a (multi-line) HTML comment
            if (!/^\s*<!--(.|[\r\n])+-->\s*$/.test(cssText)) { return; }

            // replace fractional line width in all border specifications with 1px
            cssText = cssText.replace(/([\r\n]\s*\{?(?:border(?:-top|-bottom|-left|-right)?|mso-diagonal-(?:down|up)):\s*)\.\d+pt /g, '$11px ');

            // replace non-standard line style 'hairline' with solid lines
            cssText = cssText.replace(/([\r\n]\s*\{?(?:border(?:-top|-bottom|-left|-right)?|mso-diagonal-(?:down|up)):.*?) hairline /g, '$1 solid ');

            // write the modified CSS rules back to the <style> element
            this.firstChild.nodeValue = cssText;
        });

        //TODO: use of Clipboard.getContainedHtmlElements(html, query)
        var table = html.find('table');
        if (table.length === 0) {
            table = html.filter('table');
        }

        // the default character and cell attributes for the table
        var tableAttrData = { attrs: attributePool.getDefaultValueSet(['character', 'cell']), format: null, lcid: null };
        delete tableAttrData.attrs.cell.formatId;
        delete tableAttrData.attrs.cell.formatCode;

        // merge in the character and cell attributes of the HTML table
        tableAttrData = extractElementAttributes(docModel, tableAttrData, table);

        // copy from same document
        sameDoc = (table.attr('data-app-guid') === docModel.getApp().getGlobalUid());

        // copy from ox-spreadsheet
        if (table.attr('data-range')) {
            sourceRange = Range.parse(table.attr('data-range'));
            sourceStart = sourceRange.start;
        }
        if (table.attr('data-sheet')) {
            sourceSheet = parseInt(table.attr('data-sheet'), 10);
        }
        if (table.attr('data-ox-clipboard-type') === 'cut') {   cut         = true;                           }
        if (table.attr('data-app-guid')) {                      intern      = true;                           }

        if (!sourceStart) {
            sourceStart = new Address(0, 0);
        }

        // make sure that table header, body and footer are in the correct order
        rows = _.union(table.find('thead tr').get(), table.find('tbody tr').get(), table.find('tfoot tr').get());

        colCount = _.max(_.map(rows, function (row) {
            var sum = _.reduce($(row).find('td,th'), function (memo, cell) {
                return memo + (($(cell).attr('colspan')) ? (parseInt($(cell).attr('colspan'), 10) - 1) : 0);
            }, 0);
            return ($(row).find('td,th').length + sum);
        }));

        _.each(rows, function (row, rowId) {

            var colId,
                rowData = { c: [] },
                rowAttrData = null,
                cells,
                cell,
                cellId = 0,
                cellValue,
                colspan,
                rowspan,
                isMergedCell,
                linkChild,
                lastAttributes;

            row = $(row);

            cells = row.find('td,th');

            for (colId = 0; colId < colCount; colId++) {

                var address = new Address(colId, rowId);
                var cellData = {};
                var cellAttrData = null;

                isMergedCell = mergedRanges.containsAddress(address);

                cellData.v = cellData.f = cellData.mr = null;

                // nothing to do, if the cell is part of a merged collection, but not the reference cell
                if (!isMergedCell || foundMergedRanges['col' + colId + '_row' + rowId]) {

                    // the cell is not merged or it is the reference cell
                    cell = $(cells.get(cellId));
                    colspan = cell.attr('colspan');
                    rowspan = cell.attr('rowspan');

                    if (colspan || rowspan) {
                        // the cell is merged and it's a reference cell
                        colspan = (colspan && colspan.length > 0) ? parseInt(colspan, 10) : 1;
                        rowspan = (rowspan && rowspan.length > 0) ? parseInt(rowspan, 10) : 1;

                        mergedRanges.push(Range.create(colId, rowId, colId + colspan - 1, rowId + rowspan - 1));
                    }

                    if (!intern) {
                        // look for Calc cell value
                        cellValue = cell.attr('sdval');
                        if (_.isString(cellValue) && cellValue.length > 0) {
                            cellValue = parseFloat(cellValue);
                        }
                        if (!_.isNumber(cellValue) || _.isNaN(cellValue)) {
                            cellValue = cell.text();
                            // if the cell value contains '\n' we need to make sure that we keep only the line feeds
                            // the user wants and not those Excel added by surprise.
                            if ((cellValue.indexOf('\n') > -1) || (cell.find('br').length > 0)) {
                                // remove line feeds from HTML and leave only the <br> elements
                                cell.html(cell.html().replace(/<br\s*\/?>([^\n]*)\n +/gi, '<br>$1').replace(/\n /g, ''));
                                // make '\n' out of the <br> elements
                                cell.find('br').replaceWith('\n');
                                // finally the cell value contains only the user line feeds
                                cellValue = cell.text();
                            }
                        }

                        cellData.v = _.isString(cellValue) ? Clipboard.removeInvalidXmlChars(cellValue) : cellValue;
                    }

                    // insert hyperlinks
                    var url = findMatchingChild(cell, 'a').attr('href');
                    if (_.isString(url) && url.length > 0) {
                        if ((url = HyperlinkUtils.checkForHyperlink(url))) {
                            cellData.url = url;
                        }
                    }

                    var oxCellData = isMergedCell ? foundMergedRanges['col' + colId + '_row' + rowId] : cell.attr('data-ox-cell-data');
                    if (oxCellData) {
                        oxCellData = isMergedCell ? oxCellData : JSON.parse(oxCellData);

                        if (oxCellData.formula) {
                            cellData.f = oxCellData.formula;
                            if (oxCellData.mr) {
                                cellData.mr = Range.parse(oxCellData.mr);
                            }
                        }

                        cellData.v = oxCellData ? oxCellData.value : null;

                        // styles
                        if (sameDoc) {
                            cellData.s = oxCellData.styleId;
                            cellData.a = null;

                        } else if (intern) {
                            cellData.a = oxCellData.explicit;
                            cellData.s = '';

                        } else {
                            cellAttrData = _.copy(tableAttrData, true);
                            cellData.a = cellAttrData.attrs ? cellAttrData.attrs : {};
                            attributePool.extendAttributeSet(cellData.a, oxCellData.attributes);
                            cellData.s = ''; //set to empty string to overwrite old styleId with <Default>-StyleId
                        }

                        // if this is the referece cell of a merged range, get all defined cells of the merged range
                        //  (collected in "copyHandler")
                        if (oxCellData.mergedRange) {
                            foundMergedRanges = _.extend(foundMergedRanges, oxCellData.mergedRangeCells);
                            // if the colCount is less than the colspan, use the colspan to iterate (to get all covered cells)
                            //  otherwise use the normal colCount
                            colCount = (colCount > colspan) ? colCount : colspan;
                        }

                    } else {

                        if (!rowAttrData) {
                            var rowParent = row.parents('thead, tbody, tfoot, table').first();
                            rowAttrData = extractElementAttributes(docModel, _.copy(tableAttrData, true), rowParent);
                            rowAttrData = extractElementAttributes(docModel, rowAttrData, row);
                        }

                        // merge in the character and cell attributes of the HTML row and cell
                        cellAttrData = extractElementAttributes(docModel, _.copy(rowAttrData, true), cell);
                    }

                    // merge in the hyperlink's character attributes that are attached to it's child span
                    linkChild = findMatchingChild(cell, 'a').children('span').first();
                    if (linkChild.length > 0) {
                        attributePool.extendAttributeSet(cellAttrData.attrs, { character: getCharAttributes(linkChild) });
                    }

                    if (!isMergedCell) { cellId++; }
                }

                // add the cell information to the cell contents object
                if (cellAttrData) {
                    if (cellAttrData.attrs) {
                        cellData.a = cellAttrData.attrs;
                        lastAttributes = cellAttrData.attrs; // save "last" cell-attributes for following merged cells
                    }
                    if ('format' in cellAttrData) { cellData.format = cellAttrData.format; }
                    if ('lcid' in cellAttrData) { cellData.lcid = cellAttrData.lcid; }
                }

                // set styles for merged cells
                if (isMergedCell && lastAttributes) {   cellData.a = lastAttributes;    }

                rowData.c.push(cellData);
            }

            contents.push(rowData);
        });

        var condFormats = (function () {
            var condFormatsStr = intern ? table.attr('data-ox-cond-formats') : null;
            if ((typeof condFormatsStr === 'string') && condFormatsStr) {
                try {
                    var condFormats = JSON.parse(condFormatsStr);
                    return _.isArray(condFormats) ? condFormats : [];
                } catch (ex) {
                    Utils.exception(ex);
                }
            }
            return [];
        }());

        condFormats.forEach(function (condFormat) {
            // parse the JSON data to a range array
            if (_.isObject(condFormat)) {
                if (condFormat.attrs && condFormat.attrs.attrs && condFormat.attrs.attrs.styleId) {
                    if (docModel.getApp().isOOXML() || !sameDoc) {
                        delete condFormat.attrs.attrs.styleId;
                    } else {
                        // set the style Id if it is the copy document and if it is not OOXML
                        condFormat.attrs.attrs = { styleId: condFormat.attrs.attrs.styleId };
                    }
                }
            }
        });

        return {
            contents:       contents,
            mergedRanges:   mergedRanges,
            condFormats:    (condFormats.length > 0) ? condFormats : null,
            intern:         intern,
            sameDoc:        sameDoc,
            cut:            cut,
            sourceRange:    sourceRange,
            sourceSheet:    sourceSheet
        };
    };

    /**
     * Parses the plain text from the clipboard data and creates the cell contents.
     * Rows should be separated by line endings ('\r\n', '\r' or '\n')
     * and cells should be separated by tab or semicolon.
     *
     * @param {String} text
     *  The text input to parse.
     *
     * @returns {Object}
     *  An object with the following attributes:
     *  - {Array} contents
     *      A two-dimensional array containing the values and attribute sets
     *      for the cells.
     */
    Clipboard.parseTextData = function (text) {

        var // a two dimensional array containing the cell contents
            contents = [],
            // the row nodes
            rows,
            // the cell separator
            sep;

        if (!_.isString(text)) { return { contents: [] }; }

        // remove \r,\n, \t and ; to remove empty rows at the end of the text
        text = text.replace(/[;\s]+$/, '');

        // parse rows according to the line ending type
        rows = (text.indexOf('\r\n') > -1) ? text.split(/\r\n/) : text.split(/[\r\n]/);
        // look for the cell separator
        sep = (text.indexOf('\t') > -1) ? '\t' : ';';

        _.each(rows, function (row) {
            var rowData = { c: [] },
                cells;

            cells = row.split(sep);
            _.each(cells, function (cell) {
                rowData.c.push({ v: Utils.cleanString(cell) });
            });
            contents.push(rowData);
        });

        return { contents: contents };
    };

    /**
     * Returns the container node with DOM data nodes for drawing objects that
     * have been copied into the clipboard by an OX editor application.
     */
    Clipboard.getDrawingContainerNode = function (clipboardNodes) {
        var containerNode = collectMatchingNodes(clipboardNodes, '#ox-clipboard-data[data-ox-drawings="true"]');
        return (containerNode.length > 0) ? containerNode.first() : null;
    };

    /**
     * Returns the position and size stored in the passed data node. This
     * method is tolerant against missing data attributes, and falls back to
     * reasonable default values.
     *
     * @param {jQuery} dataNode
     *  The DOM node containing a rectangle specification.
     *
     * @returns {Rectangle}
     *  The rectangle representing the location of the object.
     */
    Clipboard.getRectangleFromNode = function (dataNode, offsetX, offsetY) {
        return new Rectangle(
            offsetX + Utils.getElementAttributeAsInteger(dataNode, 'data-left', 0),
            offsetY + Utils.getElementAttributeAsInteger(dataNode, 'data-top', 0),
            Utils.getElementAttributeAsInteger(dataNode, 'data-width', 0) || 256,
            Utils.getElementAttributeAsInteger(dataNode, 'data-height', 0) || 192
        );
    };

    /**
     * Returns the explicit attributes stored in the passed data node..
     *
     * @param {jQuery} dataNode
     *  The DOM node containing a rectangle specification.
     *
     * @returns {Object|Null}
     *  The explicit attributes of the object.
     */
    Clipboard.getAttributesFromNode = function (dataNode) {

        // get the stringified JSON data from the 'data-attrs' element attribute
        var jsonStr = dataNode.attr('data-attrs');
        if (!jsonStr) { return null; }

        try {
            var attrs = JSON.parse(jsonStr);
            return _.isObject(attrs) ? attrs : null;
        } catch (ex) {
            Utils.warn('Clipboard.getAttributesFromNode(): invalid attributes');
        }
    };

    /**
     * Extracts the JSON data from the passed DOM node representing some
     * document contents in the clipboard.
     *
     * @param {jQuery} dataNode
     *  The DOM node containing JSON data.
     *
     * @returns {Any}
     *  The reconstructed JSON data; or null on error.
     */
    Clipboard.getJSONDataFromNode = function (dataNode) {

        // get the stringified JSON data from the 'data-json' element attribute
        var jsonStr = dataNode.attr('data-json');
        if (!jsonStr) { return null; }

        // parsing JSON may throw
        try {
            return JSON.parse(jsonStr);
        } catch (ex) {
            Utils.warn('Clipboard.getJSONDataFromNode(): invalid JSON data');
            return null;
        }
    };

    // exports ================================================================

    return Clipboard;

});
