/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/addressset', [
    'io.ox/office/tk/utils/iterator',
    'io.ox/office/tk/container/valueset',
    'io.ox/office/tk/container/sortedarray',
    'io.ox/office/spreadsheet/utils/interval',
    'io.ox/office/spreadsheet/utils/range',
    'io.ox/office/spreadsheet/utils/addressarray'
], function (Iterator, ValueSet, SortedArray, Interval, Range, AddressArray) {

    'use strict';

    // convenience shortcuts
    var NestedIterator = Iterator.NestedIterator;
    var SetProto = ValueSet.prototype;

    // private global functions ===============================================

    /**
     * Returns the key of the passed cell address. Intended to be used as keyer
     * callback function for object sets.
     *
     * @param {Address} address
     *  A cell address.
     *
     * @returns {String}
     *  The key of the passed cell address.
     */
    function getKey(address) {
        return address.key();
    }

    function colVectorSorter(vector) {
        return vector.first()[0];
    }

    function rowVectorSorter(vector) {
        return vector.first()[1];
    }

    // class AddressVector ====================================================

    /**
     * An instance of this class represents a sorted sparse array of cell
     * addresses in a single column or row of a sheet.
     *
     * @constructor
     *
     * @extends SortedArray
     *
     * @param {Boolean} columns
     *  Whether this cell address vector is a column vector (true; addresses
     *  will be sorted by their row index), or a row vector (false; addresses
     *  will be sorted by their column index).
     */
    var AddressVector = SortedArray.extend(function (columns) {

        // base constructor: sorter callback sorts the addresses by row or column index
        SortedArray.call(this, _.property(columns ? 1 : 0));

    }); // class AddressVector

    // class AddressMatrix ====================================================

    /**
     * An instance of this class represents a sorted sparse matrix of cell
     * addresses for a specific orientation (addresses in column vectors, or in
     * row vectors). It consists of a sorted array of vectors (which are sorted
     * arrays too) of cell addresses. The double-sorted data structure allows
     * fast binary lookup for addresses contained in a specific cell range, and
     * fast iterator implementations for existing cells in large cell ranges.
     *
     * Example: An instance of this class is used as row matrix. In this case,
     * it will contain several row vectors (instances of the class SortedArray)
     * that contain the cell addresses in the respective rows.
     *
     * @constructor
     *
     * @extends SortedArray
     *
     * @param {Boolean} columns
     *  Whether this instance contains column vectors (true; the vectors will
     *  be sorted by column index), or row vectors (false; the vectors will be
     *  sorted by row index).
     */
    var AddressMatrix = SortedArray.extend(function (columns) {

        // base constructor
        SortedArray.call(this, columns ? colVectorSorter : rowVectorSorter);

        // whether the matrix contain column vectors as elements
        this._columns = columns;

    }); // class AddressMatrix

    // public methods ---------------------------------------------------------

    /**
     * Returns the index interval covered by the main direction of this matrix.
     *
     * @returns {Interval|Null}
     *  The index interval covered by the main direction of this matrix; or
     *  null, if the matrix is empty.
     */
    AddressMatrix.prototype.boundary = function () {
        return this.empty() ? null : new Interval(this.index(this.first()), this.index(this.last()));
    };

    /**
     * Inserts a new cell address into this matrix.
     *
     * @param {Address} address
     *  The new cell model to be inserted into this matrix. This matrix
     *  DOES NOT take ownership of the cell models; and it DOES NOT check
     *  whether it already contains a cell model with the same address as
     *  the passed cell model (it is up to the caller to ensure uniqueness
     *  of the cell models).
     *
     * @param {Boolean} [replace=false]
     *  If set to true, an existing cell address that is equal to the passed
     *  cell address will be replaced with the new cell address. By default,
     *  the existing cell address will be kept, and the passed cell address
     *  will be ignored. May be important in situations where the addresses in
     *  the set contain additional user settings in arbitrary properties.
     *
     * @returns {Address}
     *  A reference to the cell address contained in this matrix. This may not
     *  be the same as the passed cell address, e.g. if an equal cell address
     *  already exists in this matrix, and the flag 'replace' has not been set.
     */
    AddressMatrix.prototype.insert = function (address, replace) {
        var index = address.get(this._columns);
        var vector = this.getOrConstruct(index, AddressVector, this._columns);
        return vector.insert(address, replace);
    };

    /**
     * Removes a cell address from this matrix.
     *
     * @param {Address} address
     *  The cell address to be removed from this matrix.
     *
     * @returns {Address}
     *  A reference to the removed cell address; or null, if this matrix does
     *  not contain a cell address that is equal to the passed cell address.
     */
    AddressMatrix.prototype.remove = function (address) {
        var index = address.get(this._columns);
        var desc = this.find(index, 'exact');
        if (!desc) { return null; }
        var vector = desc.value;
        address = vector.remove(address);
        if (vector.empty()) { desc.remove(); }
        return address;
    };

    AddressMatrix.prototype.rangeIterator = function (outerFirst, outerLast, innerFirst, innerLast, reverse) {
        var iterator = this.intervalIterator(outerFirst, outerLast, reverse);
        return new NestedIterator(iterator, function (vector) {
            return vector.intervalIterator(innerFirst, innerLast, reverse);
        });
    };

    // class AddressSet =======================================================

    /**
     * A unique set of cell addresses. This class provides optimized methods
     * for insertion, deletion, and access of cell addresses, while keeping the
     * set unique (two different addresses in this set will never have the same
     * column and row index).
     *
     * @constructor
     *
     * @extends ValueSet
     */
    var AddressSet = ValueSet.extend(function () {

        // base constructor
        ValueSet.call(this, getKey);

        // the address matrix containing column vectors
        this._colMatrix = new AddressMatrix(true);
        // the address matrix containing row vectors
        this._rowMatrix = new AddressMatrix(false);

    }); // class AddressSet

    // public methods ---------------------------------------------------------

    /**
     * Removes all elements from this set.
     *
     * @returns {AddressSet}
     *  A reference to this instance.
     */
    AddressSet.prototype.clear = function () {
        SetProto.clear.call(this);
        this._colMatrix.clear();
        this._rowMatrix.clear();
        return this;
    };

    /**
     * Returns all cell addresses contained in this set as an array.
     *
     * @returns {AddressArray}
     *  All cell addresses contained in this set as an array, in no specific
     *  order.
     */
    AddressSet.prototype.values = function () {
        var addresses = new AddressArray();
        this.forEach(function (address) { addresses.push(address); });
        return addresses;
    };

    /**
     * Returns a shallow or deep clone of this address set.
     *
     * @param {Boolean|Function} [deep=false]
     *  If set to true or to a function, a deep clone will be created (the cell
     *  addresses in this set will be cloned); otherwise a shallow clone will
     *  be created (the cell addresses in this set will be copied by reference
     *  into the clone). If a function has been passed, it will be invoked for
     *  each cloned cell address, receiving the new cloned cell address that
     *  will be inserted into the cloned set as first parameter, and the old
     *  cell address from this set as second parameter.
     *
     * @param {Object} [context]
     *  The calling context used if the 'deep' parameter is a callback.
     *
     * @returns {AddressSet}
     *  The cloned address set.
     */
    AddressSet.prototype.clone = function (deep, context) {

        // start with a new empty address set
        var clone = new AddressSet();

        // create a shallow or deep clone of the set
        if (deep) {
            var callback = (typeof deep === 'function') ? deep : _.noop;
            this.forEach(function (oldAddress) {
                var newAddress = oldAddress.clone();
                callback.call(context, newAddress, oldAddress);
                clone.insert(newAddress, true);
            });
        } else {
            this.forEach(function (address) {
                clone.insert(address, true);
            });
        }

        return clone;
    };

    /**
     * Returns the cell address from this set that has the exact position of
     * the passed address.
     *
     * @param {Address} address
     *  A cell address to search for.
     *
     * @returns {Address|Null}
     *  A cell address from this set at the same position as the passed cell
     *  address; otherwise null.
     */
    AddressSet.prototype.get = function (address) {
        return this._get(address.key()) || null;
    };

    /**
     * Returns the smallest address in this set, using horizontal sorting
     * order.
     *
     * @returns {Address|Null}
     *  The smallest address in this set, using horizontal sorting order; or
     *  null, if this set is empty.
     */
    AddressSet.prototype.first = function () {
        var vector = this._rowMatrix.first();
        return vector ? vector.first() : null;
    };

    /**
     * Returns the largest address in this set, using horizontal sorting order.
     *
     * @returns {Address|Null}
     *  The largest address in this set, using horizontal sorting order; or
     *  null, if this set is empty.
     */
    AddressSet.prototype.last = function () {
        var vector = this._rowMatrix.last();
        return vector ? vector.last() : null;
    };

    /**
     * Returns the smallest address in this set, using vertical sorting order.
     *
     * @returns {Address|Null}
     *  The smallest address in this set, using vertical sorting order; or
     *  null, if this set is empty.
     */
    AddressSet.prototype.firstVert = function () {
        var vector = this._colMatrix.first();
        return vector ? vector.first() : null;
    };

    /**
     * Returns the largest address in this set, using vertical sorting order.
     *
     * @returns {Address|Null}
     *  The largest address in this set, using vertical sorting order; or null,
     *  if this set is empty.
     */
    AddressSet.prototype.lastVert = function () {
        var vector = this._colMatrix.last();
        return vector ? vector.last() : null;
    };

    /**
     * Returns the address of the bounding range (the smallest cell range that
     * contains all addresses) of this address set.
     *
     * @returns {Range|Null}
     *  The address of the bounding range of this address set; or null, if this
     *  set is empty.
     */
    AddressSet.prototype.boundary = function () {
        var colInterval = this._colMatrix.boundary();
        var rowInterval = this._rowMatrix.boundary();
        return (colInterval && rowInterval) ? Range.createFromIntervals(colInterval, rowInterval) : null;
    };

    /**
     * Creates an iterator that visits all existing cell addresses in this set
     * that are covered by the specified cell range. The cell addresses will be
     * visited in row-by-row order, or in column-by-column order (for optimal
     * performance), unless the passed options require to use a specific
     * direction.
     *
     * @param {Range} range
     *  The address of the cell range whose addresses will be visited.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  - {String} [options.direction='auto']
     *      Can be set to one of the values 'horizontal' or 'vertical' to force
     *      to create an iterator into the specified direction. The value
     *      'horizontal' will cause to visit the addresses from first to last
     *      row, and in each row from left to right. The value 'vertical' will
     *      cause to visit the addresses from first to last column, and in
     *      each row from top to bottom. If omitted or set to another value,
     *      the direction will be picked automatically according to the
     *      contents of this address set for optimal performance.
     *  - {Boolean} [options.reverse=false]
     *      If set to true, the order of the visited addresses will be reversed
     *      (e.g. from bottom row to top row, and in each row from right to
     *      left).
     *
     * @returns {Iterator}
     *  The new iterator. The result objects will contain the current cell
     *  address (class Address) as value.
     */
    AddressSet.prototype.rangeIterator = function (range, options) {

        // number of column vectors covered by the passed range (early exit, if no vector exists)
        var col1 = range.start[0];
        var col2 = range.end[0];
        var colVectors = this._colMatrix.intervalSize(col1, col2);
        if (colVectors === 0) { return Iterator.EMPTY; }

        // number of row vectors covered by the passed range (early exit, if no vector exists)
        var row1 = range.start[1];
        var row2 = range.end[1];
        var rowVectors = this._rowMatrix.intervalSize(row1, row2);
        if (rowVectors === 0) { return Iterator.EMPTY; }

        // whether to reverse the direction of the iterator
        var reverse = options && options.reverse;
        // create an iterator for a specific direction
        var direction = options && options.direction;

        // pick the orientation with less vectors (for performance to minimize vector lookups)
        return ((direction === 'horizontal') || ((direction !== 'vertical') && (rowVectors <= colVectors))) ?
            this._rowMatrix.rangeIterator(row1, row2, col1, col2, reverse) :
            this._colMatrix.rangeIterator(col1, col2, row1, row2, reverse);
    };

    /**
     * Returns all cell addresses in this set that are located inside the
     * passed cell range address.
     *
     * @param {Range} range
     *  The cell range address used to look for the addresses in this set.
     *
     * @returns {AddressArray}
     *  An array with all cell addresses in this set located inside the passed
     *  cell range address, in no specific order.
     */
    AddressSet.prototype.findAddresses = function (range) {
        var addresses = new AddressArray();
        Iterator.forEach(this.rangeIterator(range), function (address) { addresses.push(address); });
        return addresses;
    };

    /**
     * Inserts a new cell address into this set.
     *
     * @param {Address} address
     *  The cell address to be inserted into this set.
     *
     * @param {Boolean} [replace=false]
     *  If set to true, an existing cell address that is equal to the passed
     *  cell address will be replaced with the new cell address. By default,
     *  the existing cell address will be kept, and the passed cell address
     *  will be ignored. May be important in situations where the addresses in
     *  the set contain additional user settings in arbitrary properties.
     *
     * @returns {Address}
     *  A reference to the cell address contained in this set. This may not be
     *  the same as the passed cell address, e.g. if an equal cell address
     *  already exists in this set, and the flag 'replace' has not been set.
     */
    AddressSet.prototype.insert = function (address, replace) {
        this._colMatrix.insert(address, replace);
        address = this._rowMatrix.insert(address, replace);
        return this._insert(address.key(), address);
    };

    /**
     * Removes the specified cell address from this set.
     *
     * @param {Address} address
     *  The cell address to be removed from this set.
     *
     * @returns {Address|Null}
     *  A reference to the removed cell address; or null, if this set does not
     *  contain a cell address that is equal to the passed cell address.
     */
    AddressSet.prototype.remove = function (address) {
        var key = address.key();
        address = this._get(key);
        if (!address) { return null; }
        this._remove(key);
        this._colMatrix.remove(address);
        this._rowMatrix.remove(address);
        return address;
    };

    /**
     * Inserts all cell addresses of the passed set into this set.
     *
     * @param {Object} list
     *  A generic data source with cell addresses as elements. See static
     *  method Container.forEach() for details.
     *
     * @param {Boolean} [replace=false]
     *  If set to true, cell addresses in the passed list will replace equal
     *  cell addresses in this set. By default, the existing cell addresses of
     *  this set will be preferred.
     *
     * @returns {AddressSet}
     *  A reference to this instance.
     */
    AddressSet.prototype.merge = function (list, replace) {
        ValueSet.forEach(list, function (address) { this.insert(address, replace); }, this);
        return this;
    };

    // exports ================================================================

    return AddressSet;

});
