/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author York Richter <york.richter@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/tablestylecollection', [
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/model/tablestylecollection',
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (Border, TableStyleCollection, SheetUtils) {

    'use strict';

    // class Ph1Color ==========================================================

    function PhColor(shading, index) {
        this.shading = shading;
        this.index = index || 0;
    }

    // class SheetTableStyleCollection ========================================

    /**
     * Contains the style sheets for table formatting attributes. The CSS
     * formatting will be written to table elements and their rows and cells.
     *
     * @constructor
     *
     * @extends TableStyleCollection
     *
     * @param {SpreadsheetModel} docModel
     *  The document model containing this instance.
     */
    var SheetTableStyleCollection = TableStyleCollection.extend({ constructor: function (docModel) {

        // self reference
        var self = this;

        // base constructor ---------------------------------------------------

        TableStyleCollection.call(this, docModel, {
            poolId: SheetUtils.CELL_POOL_ID,
            families: 'cell paragraph character',
            rowBandsOverColBands: true, // row band attributes will be rendered over column band attributes
            expandColBandsToOuterCols: true, // merged column band attributes over active first/last columns
            restrictColBandsToInnerRows: true // column bands will not be drawn into header/footer rows
        });

        // private methods ----------------------------------------------------

        function createStyleSheet(category, index, uiPriority, attrs) {
            var styleId = 'TableStyle' + category + index;
            var styleName = 'Table Style ' + category + ' ' + index;
            // check by style sheet name (they are always loaded in English from Excel, while the style IDs are localized)
            if (self.containsStyleSheetByName(styleName)) {
                self.setStyleOptions(styleId, { category: category, priority: uiPriority });
            } else {
                self.createDirtyStyleSheet(styleId, styleName, attrs, { priority: uiPriority, category: category });
            }
        }

        function resolvePlaceHolderAttrs(placeholderAttrs, scheme1Index, scheme2Index) {
            var colorIds = ['accent' + scheme1Index, 'accent' + scheme2Index];
            return (function processValue(value) {
                if (value instanceof PhColor) { return docModel.createSchemeColor(colorIds[value.index], value.shading); }
                if (_.isArray(value)) { return value.map(processValue); }
                if (_.isObject(value)) { return _.mapObject(value, processValue); }
                return value;
            }(placeholderAttrs));
        }

        function createSchemeStyleSheets(category, startIndex, uiPriority, leadingAttrs, placeholderAttrs) {
            createStyleSheet(category, startIndex, uiPriority, leadingAttrs);
            startIndex += 1;
            for (var schemeIndex = 1; schemeIndex <= 6; schemeIndex += 1, startIndex += 1) {
                createStyleSheet(category, startIndex, uiPriority, resolvePlaceHolderAttrs(placeholderAttrs, schemeIndex));
            }
        }

        function createSchemeStyleSheets2(category, startIndex, uiPriority, leadingAttrs, placeholderAttrs) {
            createStyleSheet(category, startIndex, uiPriority, leadingAttrs);
            startIndex += 1;
            for (var schemeIndex = 1; schemeIndex <= 6; schemeIndex += 2, startIndex += 1) {
                createStyleSheet(category, startIndex, uiPriority, resolvePlaceHolderAttrs(placeholderAttrs, schemeIndex, schemeIndex + 1));
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Silently inserts all predefiend table style sheets that have not
         * been imported from the document file.
         *
         * @returns {SheetTableStyleCollection}
         *  A reference to this instance.
         */
        this.insertMissingTableStyles = function () {

            // JSON color helper functions
            var schemeColor = docModel.createSchemeColor.bind(docModel);

            // predefined colors
            var DARK1 = schemeColor('dark1', 0);
            var DARK1_15 = schemeColor('dark1', 15);
            var DARK1_25 = schemeColor('dark1', 25);
            var DARK1_45 = schemeColor('dark1', 45);
            var LIGHT1 = schemeColor('light1', 0);
            var LIGHT1_15 = schemeColor('light1', -15);
            var LIGHT1_35 = schemeColor('light1', -35);
            var SCHEME = new PhColor(0);
            var SCHEME_DARK_25 = new PhColor(-25);
            var SCHEME_DARK_50 = new PhColor(-50);
            var SCHEME_LIGHT_40 = new PhColor(40);
            var SCHEME_LIGHT_60 = new PhColor(60);
            var SCHEME_LIGHT_80 = new PhColor(80);
            var SCHEME2 = new PhColor(0, 1);

            // JSON border helper functions
            function singleBorder(width, color) { return { style: 'single', width: width, color: color }; }
            function thinBorder(color) { return singleBorder(Border.THIN_WIDTH_HMM, color); }
            function mediumBorder(color) { return singleBorder(Border.MEDIUM_WIDTH_HMM, color); }
            function thickBorder(color) { return singleBorder(Border.THICK_WIDTH_HMM, color); }
            function doubleBorder(color) { return { style: 'double', width: Border.THICK_WIDTH_HMM, color: color }; }

            // predefined border values
            var THIN_LIGHT = thinBorder(LIGHT1);
            var THIN_DARK = thinBorder(DARK1);
            var THIN_SCHEME = thinBorder(SCHEME);
            var THIN_SCHEME_40 = thinBorder(SCHEME_LIGHT_40);
            var MEDIUM_LIGHT = mediumBorder(LIGHT1);
            var MEDIUM_DARK = mediumBorder(DARK1);
            var MEDIUM_SCHEME = mediumBorder(SCHEME);
            var THICK_LIGHT = thickBorder(LIGHT1);
            var DOUBLE_DARK = doubleBorder(DARK1);
            var DOUBLE_SCHEME = doubleBorder(SCHEME);

            // predefined character attributes
            var CHARACTER_LIGHT = { color: LIGHT1 };
            var CHARACTER_DARK = { color: DARK1 };
            var CHARACTER_BOLD_LIGHT = { color: LIGHT1, bold: true };
            var CHARACTER_BOLD_DARK = { color: DARK1, bold: true };
            var CHARACTER_BOLD_SCHEME_25 = { color: SCHEME_DARK_25, bold: true };

            // border attribute helpers
            function outerBorderAttrs(border) { return { borderLeft: border, borderRight: border, borderTop: border, borderBottom: border }; }
            function outerAndHorBorderAttrs(border) { return _.extend(outerBorderAttrs(border), { borderInsideHor: border }); }
            function allBorderAttrs(border) { return _.extend(outerAndHorBorderAttrs(border), { borderInsideVert: border }); }

            // Light 1...7: without borders
            createSchemeStyleSheets('Light', 1, 72, {
                wholeTable: { character: CHARACTER_DARK, table: { borderTop: THIN_DARK, borderBottom: THIN_DARK } },
                firstRow: { character: CHARACTER_BOLD_DARK, cell: { borderBottom: THIN_DARK } },
                lastRow: { character: CHARACTER_BOLD_DARK, cell: { borderTop: THIN_DARK } },
                firstCol: { character: CHARACTER_BOLD_DARK },
                lastCol: { character: CHARACTER_BOLD_DARK },
                band1Vert: { cell: { fillColor: LIGHT1_15 } },
                band1Hor: { cell: { fillColor: LIGHT1_15 } }
            }, {
                wholeTable: { character: { color: SCHEME_DARK_25 }, table: { borderTop: THIN_SCHEME, borderBottom: THIN_SCHEME } },
                firstRow: { character: CHARACTER_BOLD_SCHEME_25, cell: { borderBottom: THIN_SCHEME } },
                lastRow: { character: CHARACTER_BOLD_SCHEME_25, cell: { borderTop: THIN_SCHEME } },
                firstCol: { character: CHARACTER_BOLD_SCHEME_25 },
                lastCol: { character: CHARACTER_BOLD_SCHEME_25 },
                band1Vert: { cell: { fillColor: SCHEME_LIGHT_80 } },
                band1Hor: { cell: { fillColor: SCHEME_LIGHT_80 } }
            });

            // Light 8...14: with borders in bands
            createSchemeStyleSheets('Light', 8, 72, {
                wholeTable: { character: CHARACTER_DARK, table: outerBorderAttrs(THIN_DARK) },
                firstRow: { character: CHARACTER_BOLD_LIGHT, cell: { fillColor: DARK1 } },
                lastRow: { character: CHARACTER_BOLD_DARK, cell: { borderTop: DOUBLE_DARK } },
                firstCol: { character: CHARACTER_BOLD_DARK },
                lastCol: { character: CHARACTER_BOLD_DARK },
                band1Vert: { cell: { borderLeft: THIN_DARK, borderRight: THIN_DARK } },
                band1Hor: { cell: { borderTop: THIN_DARK, borderBottom: THIN_DARK } }
            }, {
                wholeTable: { character: CHARACTER_DARK, table: outerBorderAttrs(THIN_SCHEME) },
                firstRow: { character: CHARACTER_BOLD_LIGHT, cell: { fillColor: SCHEME } },
                lastRow: { character: CHARACTER_BOLD_DARK, cell: { borderTop: DOUBLE_SCHEME } },
                firstCol: { character: CHARACTER_BOLD_DARK },
                lastCol: { character: CHARACTER_BOLD_DARK },
                band1Vert: { cell: { borderLeft: THIN_SCHEME, borderRight: THIN_SCHEME } },
                band1Hor: { cell: { borderTop: THIN_SCHEME, borderBottom: THIN_SCHEME } }
            });

            // Light 15...21: with all borders
            createSchemeStyleSheets('Light', 15, 72, {
                wholeTable: { character: CHARACTER_DARK, table: allBorderAttrs(THIN_DARK) },
                firstRow: { character: CHARACTER_BOLD_DARK, cell: { borderBottom: MEDIUM_DARK } },
                lastRow: { character: CHARACTER_BOLD_DARK, cell: { borderTop: DOUBLE_DARK } },
                firstCol: { character: CHARACTER_BOLD_DARK },
                lastCol: { character: CHARACTER_BOLD_DARK },
                band1Vert: { cell: { fillColor: LIGHT1_15 } },
                band1Hor: { cell: { fillColor: LIGHT1_15 } }
            }, {
                wholeTable: { character: CHARACTER_DARK, table: allBorderAttrs(THIN_SCHEME) },
                firstRow: { character: CHARACTER_BOLD_DARK, cell: { borderBottom: MEDIUM_SCHEME } },
                lastRow: { character: CHARACTER_BOLD_DARK, cell: { borderTop: DOUBLE_SCHEME } },
                firstCol: { character: CHARACTER_BOLD_DARK },
                lastCol: { character: CHARACTER_BOLD_DARK },
                band1Vert: { cell: { fillColor: SCHEME_LIGHT_80 } },
                band1Hor: { cell: { fillColor: SCHEME_LIGHT_80 } }
            });

            // Medium 1...7: with outer borders
            createSchemeStyleSheets('Medium', 1, 71, {
                wholeTable: { character: CHARACTER_DARK, table: outerAndHorBorderAttrs(THIN_DARK) },
                firstRow: { character: CHARACTER_BOLD_LIGHT, cell: { fillColor: DARK1 } },
                lastRow: { character: CHARACTER_BOLD_DARK, cell: { borderTop: DOUBLE_DARK } },
                firstCol: { character: CHARACTER_BOLD_DARK },
                lastCol: { character: CHARACTER_BOLD_DARK },
                band1Vert: { cell: { fillColor: LIGHT1_15 } },
                band1Hor: { cell: { fillColor: LIGHT1_15 } }
            }, {
                wholeTable: { character: CHARACTER_DARK, table: outerAndHorBorderAttrs(THIN_SCHEME_40) },
                firstRow: { character: CHARACTER_BOLD_LIGHT, cell: { fillColor: SCHEME } },
                lastRow: { character: CHARACTER_BOLD_DARK, cell: { borderTop: DOUBLE_SCHEME } },
                firstCol: { character: CHARACTER_BOLD_DARK },
                lastCol: { character: CHARACTER_BOLD_DARK },
                band1Vert: { cell: { fillColor: SCHEME_LIGHT_80 } },
                band1Hor: { cell: { fillColor: SCHEME_LIGHT_80 } }
            });

            // Medium 8...14: with white borders
            createSchemeStyleSheets('Medium', 8, 71, {
                wholeTable: { character: CHARACTER_DARK, cell: { fillColor: LIGHT1_15 }, table: { borderInsideHor: THIN_LIGHT, borderInsideVert: THIN_LIGHT } },
                firstRow: { character: CHARACTER_BOLD_LIGHT, cell: { fillColor: DARK1, borderBottom: THICK_LIGHT } },
                lastRow: { character: CHARACTER_BOLD_LIGHT, cell: { fillColor: DARK1, borderTop: THICK_LIGHT } },
                firstCol: { character: CHARACTER_BOLD_LIGHT, cell: { fillColor: DARK1 } },
                lastCol: { character: CHARACTER_BOLD_LIGHT, cell: { fillColor: DARK1 } },
                band1Vert: { cell: { fillColor: LIGHT1_35 } },
                band1Hor: { cell: { fillColor: LIGHT1_35 } }
            }, {
                wholeTable: { character: CHARACTER_DARK, cell: { fillColor: SCHEME_LIGHT_80 }, table: { borderInsideHor: THIN_LIGHT, borderInsideVert: THIN_LIGHT } },
                firstRow: { character: CHARACTER_BOLD_LIGHT, cell: { fillColor: SCHEME, borderBottom: THICK_LIGHT } },
                lastRow: { character: CHARACTER_BOLD_LIGHT, cell: { fillColor: SCHEME, borderTop: THICK_LIGHT } },
                firstCol: { character: CHARACTER_BOLD_LIGHT, cell: { fillColor: SCHEME } },
                lastCol: { character: CHARACTER_BOLD_LIGHT, cell: { fillColor: SCHEME } },
                band1Vert: { cell: { fillColor: SCHEME_LIGHT_60 } },
                band1Hor: { cell: { fillColor: SCHEME_LIGHT_60 } }
            });

            // Medium 15...21: without borders
            createSchemeStyleSheets('Medium', 15, 71, {
                wholeTable: { character: CHARACTER_DARK, table: { borderLeft: THIN_DARK, borderRight: THIN_DARK, borderTop: MEDIUM_DARK, borderBottom: MEDIUM_DARK, borderInsideHor: THIN_DARK, borderInsideVert: THIN_DARK } },
                firstRow: { character: CHARACTER_BOLD_LIGHT, cell: { fillColor: DARK1, borderBottom: THIN_DARK } },
                lastRow: { cell: { borderTop: DOUBLE_DARK } },
                firstCol: { character: CHARACTER_BOLD_LIGHT, cell: { fillColor: DARK1 } },
                lastCol: { character: CHARACTER_BOLD_LIGHT, cell: { fillColor: DARK1 } },
                band1Vert: { cell: { fillColor: LIGHT1_15 } },
                band1Hor: { cell: { fillColor: LIGHT1_15 } }
            }, {
                wholeTable: { character: CHARACTER_DARK, table: { borderTop: MEDIUM_DARK, borderBottom: MEDIUM_DARK } },
                firstRow: { character: CHARACTER_BOLD_LIGHT, cell: { fillColor: SCHEME, borderBottom: MEDIUM_DARK } },
                lastRow: { cell: { borderTop: DOUBLE_DARK } },
                firstCol: { character: CHARACTER_BOLD_LIGHT, cell: { fillColor: SCHEME } },
                lastCol: { character: CHARACTER_BOLD_LIGHT, cell: { fillColor: SCHEME } },
                band1Vert: { cell: { fillColor: LIGHT1_15 } },
                band1Hor: { cell: { fillColor: LIGHT1_15 } }
            });

            // Medium 22...28: with colored borders
            createSchemeStyleSheets('Medium', 22, 71, {
                wholeTable: { character: CHARACTER_DARK, table: allBorderAttrs(THIN_DARK), cell: { fillColor: LIGHT1_15 } },
                firstRow: { character: CHARACTER_BOLD_DARK },
                lastRow: { character: CHARACTER_BOLD_DARK, cell: { borderTop: MEDIUM_DARK } },
                firstCol: { character: CHARACTER_BOLD_DARK },
                lastCol: { character: CHARACTER_BOLD_DARK },
                band1Vert: { cell: { fillColor: LIGHT1_35 } },
                band1Hor: { cell: { fillColor: LIGHT1_35 } }
            }, {
                wholeTable: { character: CHARACTER_DARK, table: allBorderAttrs(THIN_SCHEME_40), cell: { fillColor: SCHEME_LIGHT_80 } },
                firstRow: { character: CHARACTER_BOLD_DARK },
                lastRow: { character: CHARACTER_BOLD_DARK, cell: { borderTop: MEDIUM_SCHEME } },
                firstCol: { character: CHARACTER_BOLD_DARK },
                lastCol: { character: CHARACTER_BOLD_DARK },
                band1Vert: { cell: { fillColor: SCHEME_LIGHT_60 } },
                band1Hor: { cell: { fillColor: SCHEME_LIGHT_60 } }
            });

            // Dark 1...7: without borders
            createSchemeStyleSheets('Dark', 1, 73, {
                wholeTable: { character: CHARACTER_LIGHT, cell: { fillColor: DARK1_45 } },
                firstRow: { character: CHARACTER_BOLD_LIGHT, cell: { fillColor: DARK1, borderBottom: MEDIUM_LIGHT } },
                lastRow: { character: CHARACTER_BOLD_LIGHT, cell: { fillColor: DARK1_15, borderTop: MEDIUM_LIGHT } },
                firstCol: { character: CHARACTER_BOLD_LIGHT, cell: { fillColor: DARK1_25, borderRight: MEDIUM_LIGHT } },
                lastCol: { character: CHARACTER_BOLD_LIGHT, cell: { fillColor: DARK1_25, borderLeft: MEDIUM_LIGHT } },
                band1Vert: { cell: { fillColor: DARK1_25 } },
                band1Hor: { cell: { fillColor: DARK1_25 } }
            }, {
                wholeTable: { character: CHARACTER_LIGHT, cell: { fillColor: SCHEME } },
                firstRow: { character: CHARACTER_BOLD_LIGHT, cell: { fillColor: DARK1, borderBottom: MEDIUM_LIGHT } },
                lastRow: { character: CHARACTER_BOLD_LIGHT, cell: { fillColor: SCHEME_DARK_50, borderTop: MEDIUM_LIGHT } },
                firstCol: { character: CHARACTER_BOLD_LIGHT, cell: { fillColor: SCHEME_DARK_25, borderRight: MEDIUM_LIGHT } },
                lastCol: { character: CHARACTER_BOLD_LIGHT, cell: { fillColor: SCHEME_DARK_25, borderLeft: MEDIUM_LIGHT } },
                band1Vert: { cell: { fillColor: SCHEME_DARK_25 } },
                band1Hor: { cell: { fillColor: SCHEME_DARK_25 } }
            });

            // Dark 8...11: with colored header
            createSchemeStyleSheets2('Dark', 8, 73, {
                wholeTable: { cell: { fillColor: LIGHT1_15 } },
                firstRow: { character: CHARACTER_LIGHT, cell: { fillColor: DARK1 } },
                lastRow: { character: CHARACTER_BOLD_DARK, cell: { borderTop: DOUBLE_DARK } },
                firstCol: { character: CHARACTER_BOLD_DARK },
                lastCol: { character: CHARACTER_BOLD_DARK },
                band1Vert: { cell: { fillColor: LIGHT1_35 } },
                band1Hor: { cell: { fillColor: LIGHT1_35 } }
            }, {
                wholeTable: { cell: { fillColor: SCHEME_LIGHT_80 } },
                firstRow: { character: CHARACTER_LIGHT, cell: { fillColor: SCHEME2 } },
                lastRow: { character: CHARACTER_BOLD_DARK, cell: { borderTop: DOUBLE_DARK } },
                firstCol: { character: CHARACTER_BOLD_DARK },
                lastCol: { character: CHARACTER_BOLD_DARK },
                band1Vert: { cell: { fillColor: SCHEME_LIGHT_60 } },
                band1Hor: { cell: { fillColor: SCHEME_LIGHT_60 } }
            });

            return this;
        };

        /**
         * Return the attributes for a specific table cell, depending on its
         * position in the table.
         *
         * @param {Object|String} rawStyleAttrSet
         *  The raw style attribute set of a table style sheet; or the
         *  identifier of a table style sheet.
         *
         * @param {TableModel} tableModel
         *  The model of a table range.
         *
         * @param {Address} address
         *  The address of the cell to resolve the table attributes for. This
         *  address MUST be located inside the range of the passed table model.
         *
         * @returns {Object}
         *  The resulting resolved attribute set for the specified cell.
         */
        this.resolveCellAttributeSetInTable = function (rawStyleAttrSet, tableModel, address) {

            // resolve raw style attribute map from table style identifier
            if (typeof rawStyleAttrSet === 'string') {
                rawStyleAttrSet = this.getStyleSheetAttributeMap(rawStyleAttrSet, true);
            }

            // convert passed cell address to relative column/row indexes in the table range
            var tableRange = tableModel.getRange();
            var colIdx = address[0] - tableRange.start[0];
            var rowIdx = address[1] - tableRange.start[1];

            // resolve the attribute set of the cell
            var attrSet = this.resolveCellAttributeSet(rawStyleAttrSet, colIdx, rowIdx, tableRange.cols(), tableRange.rows(), tableModel.getStyleFlags());
            delete attrSet.table;
            return attrSet;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = self = null;
        });

    } }); // class SheetTableStyleCollection

    // exports ================================================================

    return SheetTableStyleCollection;

});
