/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/paragraphstylecollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/editframework/utils/lineheight',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/format/paragraphstyles'
], function (Utils, DrawingFrame, LineHeight, DOM, ParagraphStyles) {

    'use strict';

    // definitions for paragraph attributes
    var DEFINITIONS = {

        // list level
        level: { def: 0 },

        lineHeight: { def: LineHeight._115 },
        // workaround for bug 47960 & bug 45951

        listLabelHidden: { def: false },

        listStartValue: { def: -1 },

        outlineLevel: { def: 9 },

        bullet: { def: {} },

        bulletFont: { def: { followText: true } },

        bulletColor: { def: { followText: true } },

        bulletSize: { def: { type: 'followText', size: 0 } },

        // default size for tabulators (per paragraph in spreadsheets)
        defaultTabSize: { def: 2540 },

        spacingBefore: { def: 0 },

        spacingAfter: { def: 0 }
    };

    // class ParagraphStyleCollection =========================================

    /**
     * Implementation helper for paragraph formatting in drawing objects. The
     * CSS formatting will be read from and written to DOM paragraph elements.
     *
     * @constructor
     *
     * @extends ParagraphStyles
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model containing this instance.
     */
    var ParagraphStyleCollection = ParagraphStyles.extend({ constructor: function (docModel) {

        // self reference
        var self = this;

        // the collection of character style sheets
        var charStyles = null;

        // base constructor ---------------------------------------------------

        ParagraphStyles.call(this, docModel, {
            families: 'character',
            parentResolvers: { drawing: DrawingFrame.getDrawingNode }
        });

        // private methods ----------------------------------------------------

        /**
         * Will be called for every paragraph whose character attributes have
         * been changed.
         *
         * @param {jQuery} paragraph
         *  The paragraph node whose attributes have been changed, as jQuery
         *  object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute
         *  family, containing the effective attribute values merged from style
         *  sheets and explicit attributes.
         */
        function updateParagraphFormatting(paragraph, mergedAttributes) {

            var // the paragraph attributes of the passed attribute map
                paragraphAttributes = mergedAttributes.paragraph,

                leftMargin = 0,
                rightMargin = 0,

                nextParagraph = paragraph.next(),
                collectedAttrs =  {},
                textIndent = 0;

            function calcSpacingHmm(lineHeight) {
                // fix for Bug 48442
                if (!lineHeight.value) { return 0; }

                var height = 0;
                var fontHeight = Utils.convertLength(mergedAttributes.character.fontSize, 'pt', 'px');

                if (lineHeight.type === 'fixed') {
                    height = Utils.convertHmmToLength(lineHeight.value, 'px');
                } else if (lineHeight.type === 'percent') {
                    height = fontHeight * lineHeight.value / 100;
                } else {
                    Utils.warn('ParagraphStyles.calcSpacingHmm(): invalid line height type', lineHeight.type);
                    return 0;
                }

                // use zoom level as workaround for bug 47960 & Bug 48476
                var fontDesc = docModel.getRenderFont(mergedAttributes.character, null, 10);
                var normalLineHeight = fontDesc.getNormalLineHeight() / 10;
                height -= (normalLineHeight - fontHeight);
                height = Math.max(0, height);
                return Utils.convertLengthToHmm(height, 'px');
            }

            // Bug 49446, Bug 52808 & Bug 52475
            var drawingNode = paragraph.closest(DrawingFrame.NODE_SELECTOR);
            if (docModel.getApp().isOOXML() && DrawingFrame.isNoWordWrapDrawingFrame(drawingNode) && !DrawingFrame.isAutoResizeHeightDrawingFrame(drawingNode)) {
                collectedAttrs.alignSelf = (function () {
                    switch (paragraphAttributes.alignment) {
                        case 'center':  return 'center';
                        case 'left':    return 'flex-start';
                        case 'right':   return 'flex-end';
                        case 'justify': return 'stretch';
                        default:        return '';
                    }
                }());
            }

            // Always update character formatting of all child nodes which may
            // depend on paragraph settings, e.g. automatic text color which
            // depends on the paragraph fill color. Also visit all helper nodes
            // containing text spans, e.g. numbering labels.
            Utils.iterateDescendantNodes(paragraph, function (node) {

                // visiting the span inside a hard break node
                // -> this is necessary for change tracking attributes
                if (DOM.isHardBreakNode(node)) {
                    charStyles.updateElementFormatting(node.firstChild);
                }

                DOM.iterateTextSpans(node, function (span) {
                    charStyles.updateElementFormatting(span, { baseAttributes: mergedAttributes });
                });
            }, undefined, { children: true });

            // update borders
            var leftPadding = 0;
            var topMargin = 0;
            var bottomMargin = 0;

            //calculate list indents
            //            var listLabel = $(paragraph).children(DOM.LIST_LABEL_NODE_SELECTOR);
            //            var listStyleId = paragraphAttributes.listStyleId;
            //            if (listStyleId.length) {
            //                var listLevel = paragraphAttributes.listLevel,
            //                    lists = docModel.getListCollection();
            //                if (listLevel < 0) {
            //                    // is a numbering level assigned to the current paragraph style?
            //                    listLevel = lists.findIlvl(listStyleId, mergedAttributes.styleId);
            //                }
            //                if (listLevel !== -1 && listLevel < 10) {
            //                    var listItemCounter = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0];
            //                    var listObject = lists.formatNumber(paragraphAttributes.listStyleId, listLevel, listItemCounter, 1);
            //
            //                    if (listObject.indent > 0) {
            //                        leftPadding += listObject.indent - listObject.firstLine;
            //                        leftMargin += listObject.firstLine;
            //                    }
            //                    if (listLabel.length) {
            //                        var listSpan = listLabel.children('span');
            //                        if (listObject.fontSize) {
            //                            listSpan.css('font-size', listObject.fontSize + 'pt');
            //                        }
            //                        if (listObject.color) {
            //                            listSpan.css('color', docModel.getCssTextColor(listObject.color, [paragraphAttributes.fillColor, listObject.fillColor]));
            //                        }
            //                    }
            //                } else {
            //                    //fix for Bug 37594 some list-helper dont disappear
            //                    listLabel.detach();
            //                }
            //            } else {
            //                //fix for Bug 37594 some list-helper dont disappear
            //                listLabel.detach();
            //            }

            // paragraph margin attributes - also applying to paragraphs in a list, if they are defined as explicit attribute
            // -> handle both cases correctly: 40792 and 41118
            // if (listStyleId === '') {
            leftMargin += paragraphAttributes.indentLeft ? paragraphAttributes.indentLeft : 0;
            rightMargin += paragraphAttributes.indentRight ? paragraphAttributes.indentRight : 0;
            textIndent = paragraphAttributes.indentFirstLine ? paragraphAttributes.indentFirstLine : 0;
            collectedAttrs.textIndent = textIndent / 100 + 'mm';
            //            } else {
            //                // but explicit attributes need to be handled (40792)
            //                explicitParaAttributes = AttributeUtils.getExplicitAttributes(paragraph);
            //                explicitParaAttributes = (explicitParaAttributes && explicitParaAttributes.paragraph) || {};
            //                if (explicitParaAttributes.indentLeft) { leftMargin += explicitParaAttributes.indentLeft; }
            //                if (explicitParaAttributes.indentRight) { rightMargin += explicitParaAttributes.indentRight; }
            //                if (explicitParaAttributes.indentFirstLine) {
            //                    textIndent += explicitParaAttributes.indentRight;
            //                    collectedAttrs.textIndent = textIndent / 100 + 'mm';
            //                }
            //            }

            if (textIndent < 0) {
                leftPadding -= textIndent;
                leftMargin += textIndent;
            }

            _.extend(collectedAttrs, {
                paddingLeft: (leftPadding / 100) + 'mm',
                marginLeft: (leftMargin / 100) + 'mm',
                marginRight: (rightMargin / 100) + 'mm',
                textIndent: (textIndent / 100) + 'mm'
            });

            // Overwrite of margin left for lists in draft mode (conversion from fixed unit mm to %)
            if (Utils.SMALL_DEVICE && paragraph.data('draftRatio')) {
                collectedAttrs.marginLeft = (parseInt($(paragraph).css('margin-left'), 10) * paragraph.data('draftRatio')) + '%';
            }

            if (paragraphAttributes.spacingBefore) {
                topMargin += calcSpacingHmm(paragraphAttributes.spacingBefore);
            }
            if (paragraphAttributes.spacingAfter && nextParagraph.length) {
                bottomMargin += calcSpacingHmm(paragraphAttributes.spacingAfter);
            }

            _.extend(collectedAttrs, {
                marginTop: (topMargin / 100) + 'mm',
                marginBottom: (bottomMargin / 100) + 'mm'
            });

            if (!_.browser.IE) {
                collectedAttrs.width = null;
            }

            // apply collected attributes at the end
            paragraph.css(collectedAttrs);

            // update the size of all tab stops in this paragraph
            self.implUpdateTabStops(paragraph, mergedAttributes);
        }

        // initialization -----------------------------------------------------

        // deferred initialization
        docModel.getApp().onInit(function () {
            charStyles = docModel.getCharacterStyles();
        }, this);

        // register the attribute definitions for the style family
        docModel.registerAttributes('paragraph', DEFINITIONS);

        // register the formatting handler for DOM elements
        this.registerFormatHandler(updateParagraphFormatting);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docModel = charStyles = null;
        });

    } }); // class ParagraphStyleCollection

    // exports ================================================================

    return ParagraphStyleCollection;

});
