/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/operationgenerator', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/model/operationgenerator',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (Utils, ValueMap, Border, OperationGenerator, Operations, SheetUtils) {

    'use strict';

    // class SheetOperationGenerator ==========================================

    /**
     * An operations generator specifically for spreadsheet documents. Provides
     * additional methods to generate document operations, and undo operations,
     * for a specific sheet model. However, the generator is not restricted to
     * a single sheet. The sheet index used to generate sheet operations can be
     * changed at any time.
     *
     * @constructor
     *
     * @extends OperationGenerator
     *
     * @param {SpreadsheetModel} docModel
     *  The document model this operations generator will be created for.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the base class, and the
     *  following additional options:
     *  - {Number} [initOptions.sheet=0]
     *      The initial zero-based sheet index to be inserted into all sheet
     *      operations.
     */
    var SheetOperationGenerator = OperationGenerator.extend(function (docModel, initOptions) {

        // base constructor
        OperationGenerator.call(this, docModel, initOptions);

        // the collection of cell auto-styles of the document (private property)
        this._autoStyles = docModel.getCellAutoStyles();

        // cache for calculated auto-style identifiers (private property)
        this._styleIdCache = new ValueMap();

        // number of cache hits during lifetime of this instance (private property)
        this._cacheHits = 0;

        // number of cache misses during lifetime of this instance (private property)
        this._cacheMisses = 0;

        // the sheet index to be inserted into sheet operations (private property)
        this._sheet = Utils.getIntegerOption(initOptions, 'sheet', 0);

    }); // class SheetOperationGenerator

    // private methods --------------------------------------------------------

    /**
     * Tries to resolve a cached auto-style identifier.
     *
     * @param {String} oldStyleId
     *  The identifier of the old auto-style.
     *
     * @param {String} [cacheKey]
     *  The unique key used to look-up a cached auto-style identifier for the
     *  specified auto-style. This parameter may be missing, in this case, no
     *  cache lookup will be performed.
     *
     * @returns {String|Null}
     *  The cached identifier of a auto-style, if available; otherwide null.
     */
    SheetOperationGenerator.prototype._getCachedStyleId = function (oldStyleId, cacheKey) {

        var styleIdMap = cacheKey ? this._styleIdCache.get(cacheKey) : null;
        if (!styleIdMap) { return null; }

        var newStyleId = styleIdMap.get(oldStyleId, null);
        if (newStyleId !== null) { this._cacheHits += 1; }
        return newStyleId;
    };

    /**
     * Inserts an auto-style identifier into the internal cache.
     *
     * @param {String} oldStyleId
     *  The identifier of the old auto-style for which the new auto-style
     *  identifier will be cached.
     *
     * @param {String} newStyleId
     *  The identifier of the new auto-style to be inserted into the internal
     *  cache.
     *
     * @param {String} [cacheKey]
     *  The unique key used to cache the new auto-style identifier. This
     *  parameter may be missing, in this case, the cache will not be modified.
     */
    SheetOperationGenerator.prototype._setCachedStyleId = function (oldStyleId, newStyleId, cacheKey) {
        if (cacheKey) {
            this._styleIdCache.getOrConstruct(cacheKey, ValueMap).insert(oldStyleId, newStyleId);
            this._cacheMisses += 1;
        }
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns the sheet index this generator inserts into all sheet
     * operations.
     *
     * @returns {Number}
     *  The sheet index this generator will insert into all sheet operations.
     */
    SheetOperationGenerator.prototype.getSheetIndex = function () {
        return this._sheet;
    };

    /**
     * Changes the sheet index this generator will insert in the next sheet
     * operations.
     *
     * @param {Number} sheet
     *  The new sheet index this generator will insert in the next sheet
     *  operations. MUST be a valid sheet index in the spreadsheet document.
     *
     * @returns {SheetOperationGenerator}
     *  A reference to this instance.
     */
    SheetOperationGenerator.prototype.setSheetIndex = function (sheet) {
        this._sheet = sheet;
        return this;
    };

    /**
     * Creates and appends a new operation to the operations array, which will
     * contain a property 'sheet' set to the current sheet index of this
     * generator.
     *
     * @param {String} name
     *  The name of the operation.
     *
     * @param {Object} [properties]
     *  Additional properties that will be stored in the operation.
     *
     * @param {Object} [options]
     *  Optional parameters. See method OperationGenerator.generateOperation()
     *  for details.
     *
     * @returns {Object}
     *  The created operation object.
     */
    SheetOperationGenerator.prototype.generateSheetOperation = function (name, properties, options) {
        properties = _.extend({ sheet: this._sheet }, properties);
        return this.generateOperation(name, properties, options);
    };

    /**
     * Creates and appends a new operation with 'start' and (optional) 'end'
     * properties to the operations array addressing a column or row interval.
     * If first and last index of the interval are equal, the 'end' property
     * will be omitted in the operation.
     *
     * @param {String} name
     *  The name of the operation.
     *
     * @param {Interval} interval
     *  The column/row interval to be inserted into the operation.
     *
     * @param {Object} [properties]
     *  Additional properties that will be stored in the operation.
     *
     * @param {Object} [options]
     *  Optional parameters. See method OperationGenerator.generateOperation()
     *  for details. Additionally, the following options are supported:
     *  - {Boolean} [options.complete=false]
     *      If set to true, the property 'end' will be written, if it is equal
     *      to the property 'start'. By default, the property will be omitted
     *      in that case.
     *
     * @returns {Object}
     *  The created operation object.
     */
    SheetOperationGenerator.prototype.generateIntervalOperation = function (name, interval, properties, options) {
        // add 'start' and 'end' properties to the operation
        properties = _.extend({ start: interval.first }, properties);
        if ((interval.first < interval.last) || (options && options.complete)) {
            properties.end = interval.last;
        }
        return this.generateSheetOperation(name, properties, options);
    };

    /**
     * Creates and appends a new operation with a 'start' property to the
     * operations array addressing a single cell.
     *
     * @param {String} name
     *  The name of the operation.
     *
     * @param {Address} address
     *  The address of the cell targeted by the operation.
     *
     * @param {Object} [properties]
     *  Additional properties that will be stored in the operation.
     *
     * @param {Object} [options]
     *  Optional parameters. See method OperationGenerator.generateOperation()
     *  for details.
     *
     * @returns {Object}
     *  The created operation object.
     */
    SheetOperationGenerator.prototype.generateCellOperation = function (name, address, properties, options) {
        properties = _.extend({ start: address.toJSON() }, properties);
        return this.generateSheetOperation(name, properties, options);
    };

    /**
     * Creates and appends a new operation with 'start' and (optional) 'end'
     * properties to the operations array addressing a cell range. If the range
     * addresses a single cell, the 'end' property will be omitted in the
     * operation.
     *
     * @param {String} name
     *  The name of the operation.
     *
     * @param {Range} range
     *  The address of the cell range addressed by the operation.
     *
     * @param {Object} [properties]
     *  Additional properties that will be stored in the operation.
     *
     * @param {Object} [options]
     *  Optional parameters. See method OperationGenerator.generateOperation()
     *  for details. Additionally, the following options are supported:
     *  - {Boolean} [options.complete=false]
     *      If set to true, the property 'end' will be written, if it is equal
     *      to the property 'start'. By default, the property will be omitted
     *      in that case.
     *
     * @returns {Object}
     *  The created operation object.
     */
    SheetOperationGenerator.prototype.generateRangeOperation = function (name, range, properties, options) {
        // add 'start' and 'end' properties to the operation
        properties = _.extend({ start: range.start.toJSON() }, properties);
        if (!range.single() || (options && options.complete)) {
            properties.end = range.end.toJSON();
        }
        return this.generateSheetOperation(name, properties, options);
    };

    /**
     * Creates and appends a new operation with a 'ranges' property to the
     * operations array addressing multiple cell ranges. If a cell range in the
     * array addresses a single cell, the 'end' property of that range will be
     * omitted in the operation.
     *
     * @param {String} name
     *  The name of the operation.
     *
     * @param {RangeArray} ranges
     *  The addresses of the cell ranges addressed by the operation.
     *
     * @param {Object} [properties]
     *  Additional properties that will be stored in the operation.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  Optional parameters. See method OperationGenerator.generateOperation()
     *  for details. Additionally, the following options are supported:
     *  - {Boolean} [options.complete=false]
     *      If set to true, the property 'end' of all cell ranges will be
     *      written, also if it is equal to the property 'start'. By default,
     *      the property will be omitted in that case.
     *
     * @returns {Object}
     *  The created operation object.
     */
    SheetOperationGenerator.prototype.generateRangesOperation = function (name, ranges, properties, options) {
        var complete = options && options.complete;
        ranges = ranges.map(function (range) {
            var json = { start: range.start.toJSON() };
            if (complete || !range.single()) { json.end = range.end.toJSON(); }
            return json;
        });
        properties = _.extend({ ranges: ranges.toJSON() }, properties);
        return this.generateSheetOperation(name, properties, options);
    };

    /**
     * Creates and appends a new operation with a 'table' property to the
     * operations array addressing a table range.
     *
     * @param {String} name
     *  The name of the operation.
     *
     * @param {String} tableName
     *  The name of a table range. The empty string addresses the anonymous
     *  table range used to store filter settings for the standard auto filter
     *  of the sheet (the 'table' property will not be inserted into the
     *  generated operation).
     *
     * @param {Object} [properties]
     *  Additional properties that will be stored in the operation.
     *
     * @param {Object} [options]
     *  Optional parameters. See method OperationGenerator.generateOperation()
     *  for details.
     *
     * @returns {Object}
     *  The created operation object.
     */
    SheetOperationGenerator.prototype.generateTableOperation = function (name, tableName, properties, options) {
        if (tableName.length > 0) { properties = _.extend({ table: tableName }, properties); }
        return this.generateSheetOperation(name, properties, options);
    };

    /**
     * Creates and appends a new operation with a 'start' property to the
     * operations array addressing a drawing object.
     *
     * @param {String} name
     *  The name of the operation.
     *
     * @param {Array<Number>} position
     *  The position of the drawing object in the sheet.
     *
     * @param {Object} [properties]
     *  Additional properties that will be stored in the operation.
     *
     * @param {Object} [options]
     *  Optional parameters. See method OperationGenerator.generateOperation()
     *  for details.
     *
     * @returns {Object}
     *  The created operation object.
     */
    SheetOperationGenerator.prototype.generateDrawingOperation = function (name, position, properties, options) {
        properties = _.extend({ start: [this._sheet].concat(position) }, properties);
        return this.generateOperation(name, properties, options);
    };

    // explicit operations ----------------------------------------------------

    /**
     * Creates and appends a new 'moveCells' operation.
     *
     * @param {Range} range
     *  The address of the cell range addressed by the operation.
     *
     * @param {MoveMode} moveMode
     *  The move mode to be inserted into the 'moveCells' operation.
     *
     * @param {Object} [options]
     *  Optional parameters. See description of the public method
     *  SheetOperationGenerator.generateRangeOperation() for details.
     *
     * @returns {Object}
     *  The created operation object.
     */
    SheetOperationGenerator.prototype.generateMoveCellsOperation = function (range, moveMode, options) {
        return this.generateRangeOperation(Operations.MOVE_CELLS, range, { dir: moveMode.toJSON() }, options);
    };

    /**
     * Creates and appends a new 'mergeCells' operation.
     *
     * @param {Range} range
     *  The address of the cell range addressed by the operation.
     *
     * @param {MergeMode} mergeMode
     *  The merge mode to be inserted into the 'mergeCells' operation.
     *
     * @param {Object} [options]
     *  Optional parameters. See description of the public method
     *  SheetOperationGenerator.generateRangeOperation() for details.
     *
     * @returns {Object}
     *  The created operation object.
     */
    SheetOperationGenerator.prototype.generateMergeCellsOperation = function (range, mergeMode, options) {
        return this.generateRangeOperation(Operations.MERGE_CELLS, range, { type: mergeMode.toJSON() }, options);
    };

    /**
     * Creates and appends a new 'moveDrawing' operation.
     *
     * @param {Array<Number>} fromPos
     *  The current position of the drawing object in the sheet.
     *
     * @param {Array<Number>} toPos
     *  The new position for the drawing object in the sheet.
     *
     * @param {Object} [options]
     *  Optional parameters. See method OperationGenerator.generateOperation()
     *  for details.
     *
     * @returns {Object}
     *  The created operation object.
     */
    SheetOperationGenerator.prototype.generateMoveDrawingOperation = function (fromPos, toPos, options) {
        return this.generateDrawingOperation(Operations.MOVE_DRAWING, fromPos, { to: [this._sheet].concat(toPos) }, options);
    };

    // auto-style handling ----------------------------------------------------

    /**
     * Returns whether the passed auto-style identifier is considered to be the
     * default auto-style. Convenience shortcut for the same method of the
     * auto-style collection.
     *
     * @param {String} styleId
     *  The identifier of an auto-style.
     *
     * @returns {Boolean}
     *  Whether the passed auto-style identifier is the default auto-style.
     */
    SheetOperationGenerator.prototype.isDefaultStyleId = function (styleId) {
        return this._autoStyles.isDefaultStyleId(styleId);
    };

    /**
     * Returns whether the passed auto-style identifiers are considered to be
     * equal. Convenience shortcut for the same method of the auto-style
     * collection.
     *
     * @param {String} styleId1
     *  The identifier of the first auto-style.
     *
     * @param {String} styleId2
     *  The identifier of the second auto-style.
     *
     * @returns {Boolean}
     *  Whether the passed auto-style identifiers are equal.
     */
    SheetOperationGenerator.prototype.areEqualStyleIds = function (styleId1, styleId2) {
        return this._autoStyles.areEqualStyleIds(styleId1, styleId2);
    };

    /**
     * Returns the identifier of the auto-style containing the merged cell
     * formatting attributes of the passed auto-style, and the attribute set
     * contained in the contents object. Tries to resolve a cached auto-style
     * identifier that has been generated in a previous invocation.
     *
     * @param {String} oldStyleId
     *  The identifier of the original auto-style to be merged with new cell
     *  formatting attributes.
     *
     * @param {Object} contents
     *  A contents descriptor used in various operation generators for columns,
     *  rows, and cells. This method uses the optional properties "s" (a new
     *  explicit auto-style), "a" (an incomplete attribute set), "format" (the
     *  identifier of a number format, or an explicit format code), "table" (an
     *  attribute set resolved form a table style sheet, to be merged "under"
     *  the explicit cell attributes and cell style), "cacheKey" (a hash key
     *  for the attributes in the passed contents object, used to improve
     *  performance when processing the same original auto-styles multiple
     *  times), and "processAttributes" (callback function for additional
     *  processing of the resulting merged attribute set).
     *
     * @returns {String}
     *  The identifier of the resulting auto-style containing the merged cell
     *  formatting attributes of the original auto-style, and the formatting
     *  settings in the contents object.
     */
    SheetOperationGenerator.prototype.generateAutoStyle = function (oldStyleId, contents) {

        // nothing to do without any style settings in the contents object
        if (!('s' in contents) && !('a' in contents) && !('format' in contents) && !('table' in contents)) {
            return oldStyleId;
        }

        // try to resolve an existing cached auto-style identifier
        var newStyleId = this._getCachedStyleId(oldStyleId, contents.cacheKey);
        if (newStyleId !== null) { return newStyleId; }

        // start with an explicit auto-style identifier that replaces the current auto-style
        newStyleId = (typeof contents.s === 'string') ? contents.s : oldStyleId;

        // merge explicit formatting attributes to the resulting auto-style
        if (_.isObject(contents.a) || ('format' in contents) || _.isObject(contents.table)) {
            var processAttributesFunc = Utils.getFunctionOption(contents, 'processAttributes');
            newStyleId = this._autoStyles.generateAutoStyleOperation(this, newStyleId, contents, processAttributesFunc);
        }

        // put the new auto-style identifier into the cache
        this._setCachedStyleId(oldStyleId, newStyleId, contents.cacheKey);

        return newStyleId;
    };

    /**
     * Returns the identifier of an auto-style containing the formatting that
     * results when inserting content between two objects with the specified
     * auto-styles (e.g. when inserting columns or rows, or shifting cells in
     * the sheet to the right or down).
     *
     * @param {String} styleId1
     *  The auto-style identifier of the leading object.
     *
     * @param {String} styleId2
     *  The auto-style identifier of the trailing object.
     *
     * @param {Boolean} horizontal
     *  Whether the leading object containing the first auto-style is located
     *  left of the trailing object with the second auto-style (true), or on
     *  top of the trailing object (false).
     *
     * @returns {String}
     *  The identifier of the auto-style to be set to the objects inserted
     *  between the existing objects with the passed auto-styles.
     */
    SheetOperationGenerator.prototype.generateInsertedAutoStyle = function (styleId1, styleId2, horizontal) {

        // nothing to do if styles are equal
        if (this.areEqualStyleIds(styleId1, styleId2)) { return styleId1; }

        // try the cached auto-styles first
        var cacheKey = styleId1 + '\n' + styleId2;
        var newStyleId = this._getCachedStyleId(styleId1, cacheKey);
        if (newStyleId !== null) { return newStyleId; }

        // early exit if none of the auto-styles contains any visible borders
        var borderMap1 = this._autoStyles.getBorderMap(styleId1);
        var borderMap2 = this._autoStyles.getBorderMap(styleId2);
        if (_.isEmpty(borderMap1) && _.isEmpty(borderMap2)) { return styleId1; }

        // returns whether the passed cell attribute map contains two visible and equal border attributes
        function hasEqualBorders(borderMap, key1, key2) {
            var border1 = borderMap[key1], border2 = borderMap[key2];
            return border1 && border2 && Border.isEqual(border1, border2);
        }

        // returns the border attribute, if it is visible and equal in both border maps, otherwise null
        function getMergedBorder(key1, key2) {
            var border1 = borderMap1[key1], border2 = borderMap2[key2];
            return (border1 && border2 && Border.isEqual(border1, border2)) ? border1 : null;
        }

        // the new border attributes to be applied over the leading auto-style
        var borderAttrs = {};

        // copy leading borders along move direction if equal in both objects (e.g. top borders of two adjacent cells when inserting columns)
        var alongKey1 = SheetUtils.getOuterBorderKey(!horizontal, true);
        borderAttrs[SheetUtils.getBorderName(alongKey1)] = getMergedBorder(alongKey1, alongKey1);

        // copy trailing borders along move direction if equal in both objects (e.g. bottom borders of two adjacent cells when inserting columns)
        var alongKey2 = SheetUtils.getOuterBorderKey(!horizontal, false);
        borderAttrs[SheetUtils.getBorderName(alongKey2)] = getMergedBorder(alongKey2, alongKey2);

        // copy the inner overlapping border of the objects, but only if *both* (leading and trailing) borders in either of the objects are equal
        var acrossKey1 = SheetUtils.getOuterBorderKey(horizontal, true);
        var acrossKey2 = SheetUtils.getOuterBorderKey(horizontal, false);
        var innerBorder = getMergedBorder(acrossKey2, acrossKey1);
        var equalBorders = innerBorder && (hasEqualBorders(borderMap1, acrossKey1, acrossKey2) || hasEqualBorders(borderMap2, acrossKey1, acrossKey2));
        borderAttrs[SheetUtils.getBorderName(acrossKey1)] = borderAttrs[SheetUtils.getBorderName(acrossKey2)] = equalBorders ? innerBorder : null;

        // create and cache the new auto-style
        return this.generateAutoStyle(styleId1, { a: { cell: borderAttrs }, cacheKey: cacheKey });
    };

    /**
     * Helper callback function for the column, row, and cell collections, used
     * to create the contents objects for cell ranges or column/row intervals
     * expected by the content generators, while adding or removing border
     * lines inside a cell selection.
     *
     * @param {Array<Object>} fillDataArray
     *  An array of fill data objects created by the operation generators for
     *  cell formatting, or column/row formatting. Each object must contain the
     *  property 'key', and may contain the properties 'border', 'keepBorder',
     *  and 'cacheType'.
     *
     * @param {String} cacheKey
     *  The root cache key used to optimize creation of new auto-styles,
     *  intended to be reused in different generators (columns, rows, and
     *  cells) while creating the operations for the same border settings.
     *
     * @returns {Object}
     *  The contents object with the converted border settings.
     */
    SheetOperationGenerator.prototype.createBorderContents = (function () {

        // cache key for an outer, inner, and deleted border line (to be combined with CACHE_KEY_MULTIPLIERS)
        var BORDER_CACHE_FLAGS = { outer: 1, inner: 2, clear: 3 };
        // factors used to build unique cache keys for generating border auto-styles
        var CACHE_KEY_MULTIPLIERS = { t: 1, b: 4, l: 16, r: 64 };

        // helper callback function used to finalize the attribute set for a new auto-style,
        // intended to be added into a contents object for an interval or cell range
        function processBorderAttributes(newAttributeSet, mergedAttributeSet, keepBorderAttrs) {

            // shortcut to the current cell attributes of the auto-style
            var oldBorderAttrs = mergedAttributeSet.cell;
            // shortcut to the new border attributes
            var newBorderAttrs = newAttributeSet.cell;

            // update the existing borders of the auto-style, delete other border attributes from the attribute set
            _.each('tblrdu', function (borderKey) {

                // nothing to do without an explicit new border attribute
                var borderName = SheetUtils.getBorderName(borderKey);
                if (!(borderName in newBorderAttrs)) { return; }

                // the old and new border attribute
                var oldBorder = oldBorderAttrs[borderName];
                var newBorder = newBorderAttrs[borderName];

                // do not apply invisible borders (do not set the 'border' apply flag to the auto-style), if the old
                // border is already invisible, or if the old border is equal to the border attribute passed in the
                // parameter 'keepBorderAttrs' (do not change existing borders of adjacent cells outside the ranges)
                if (!Border.isVisibleBorder(newBorder)) {
                    if (!Border.isVisibleBorder(oldBorder) || Border.isEqual(oldBorder, keepBorderAttrs[borderName])) {
                        delete newBorderAttrs[borderName];
                    }
                }
            });
        }

        // the actual implementation returned from local scope
        function createBorderContents(fillDataArray, cacheKey) {

            // fill data objects, divided by border keys
            var borderDataMap = {};
            // the resulting border attributes to be inserted into the contents object
            var borderAttrs = {};
            // border attributes to be kept while deleting adjacent outer borders
            var keepBorderAttrs = {};
            // a bitmask for the auto-style cache, according to which borders will be changed
            var cacheFlags = 0;

            // collect the fill data objects separated by border key
            fillDataArray.forEach(function (fillData) {
                var key = fillData.key;
                var borderData = borderDataMap[key] || (borderDataMap[key] = {});
                _.extend(borderData, fillData);
            });

            // collect the resulting border attributes for each border key
            _.each(borderDataMap, function (borderData, key) {
                var borderName = SheetUtils.getBorderName(key);
                var multiplier = CACHE_KEY_MULTIPLIERS[key];
                if (borderData.border) {
                    borderAttrs[borderName] = borderData.border;
                    cacheFlags |= multiplier * BORDER_CACHE_FLAGS[borderData.cacheType];
                } else if (borderData.keepBorder) {
                    borderAttrs[borderName] = Border.NONE;
                    keepBorderAttrs[borderName] = borderData.keepBorder;
                    cacheFlags |= multiplier * BORDER_CACHE_FLAGS.clear;
                }
            });

            // callback for the auto-style generator, to be inserted into the contents object
            function processAttributes(newAttributeSet, mergedAttributeSet) {
                return processBorderAttributes(newAttributeSet, mergedAttributeSet, keepBorderAttrs);
            }

            // return the contents object with border properties, auto-style cache key, and attribute set processor
            return { a: { cell: borderAttrs }, cacheKey: cacheKey + ':' + cacheFlags, processAttributes: processAttributes };
        }

        return createBorderContents;
    }());

    /**
     * Helper callback function for the column, row, and cell collections, used
     * to create the contents objects for cell ranges or column/row intervals
     * expected by the content generators, while modifying the properties of
     * existing borders inside a cell selection.
     *
     * @param {Array<Object>} fillDataArray
     *  An array of fill data objects created by the operation generators for
     *  cell formatting, or column/row formatting. Each object must contain a
     *  string property 'keys' specifying with the cell borders to be changed
     *  with the passed border properties.
     *
     * @param {Object} border
     *  The border properties to be changed with the generated operations. All
     *  properties contained in this object (color, line style, line width)
     *  will be set at existing visible borders. Omitted border properties will
     *  not be changed.
     *
     * @param {String} cacheKey
     *  The root cache key used to optimize creation of new auto-styles,
     *  intended to be reused in different generators (columns, rows, and
     *  cells) while creating the operations for the same border settings.
     *
     * @returns {Object}
     *  The contents object with the converted border settings.
     */
    SheetOperationGenerator.prototype.createVisibleBorderContents = (function () {

        // bit flags used to build unique cache keys for generating border auto-styles
        var BORDER_CACHE_FLAGS = { t: 1, b: 2, l: 4, r: 8, d: 16, u: 32 };

        // helper callback function used to finalize the attribute set for a new auto-style,
        // intended to be added into a contents object for an interval or cell range
        function processVisibleBorderAttributes(newAttributeSet, mergedAttributeSet) {

            // shortcut to the current cell attributes of the auto-style
            var oldBorderAttrs = mergedAttributeSet.cell;
            // shortcut to the new border attributes
            var newBorderAttrs = newAttributeSet.cell;

            // update the existing borders of the auto-style, delete other border attributes from the attribute set
            _.each('tblrdu', function (borderKey) {

                // nothing to do without an explicit new border attribute
                var borderName = SheetUtils.getBorderName(borderKey);
                if (!(borderName in newBorderAttrs)) { return; }

                // the old and new border attribute
                var oldBorder = oldBorderAttrs[borderName];
                var newBorder = newBorderAttrs[borderName];

                // extend visible borders of the auto-style only, ignore other borders (by removing them from the attribute set)
                if (Border.isVisibleBorder(oldBorder)) {
                    newBorder = _.extend({}, oldBorder, newBorder);
                    newBorderAttrs[borderName] = Border.isVisibleBorder(newBorder) ? newBorder : Border.NONE;
                } else {
                    delete newBorderAttrs[borderName];
                }
            });
        }

        // the actual implementation returned from local scope
        function createVisibleBorderContents(fillDataArray, border, cacheKey) {

            // the border attributes for all cell borders to be changed
            var borderAttrs = {};
            // a bitmask for the auto-style cache, according to which borders will be changed
            var cacheFlags = 0;

            // add all borders that will be changed accoding to the different fill data objects
            fillDataArray.forEach(function (fillData) {
                _.each(fillData.keys, function (key) {
                    borderAttrs[SheetUtils.getBorderName(key)] = border;
                    cacheFlags |= BORDER_CACHE_FLAGS[key];
                });
            });

            // return the contents object with border properties, auto-style cache key, and attribute set processor
            return { a: { cell: borderAttrs }, cacheKey: cacheKey + ':' + cacheFlags, processAttributes: processVisibleBorderAttributes };
        }

        return createVisibleBorderContents;
    }());

    /**
     * Prints debugging information to the browser console.
     */
    SheetOperationGenerator.prototype.logCacheUsage = SheetUtils.isLoggingActive() ? function () {
        var cacheAccesses = this._cacheHits + this._cacheMisses;
        if (cacheAccesses > 0) {
            SheetUtils.log('style cache hits: ' + this._cacheHits + ' (' + Math.round(this._cacheHits / cacheAccesses * 100) + '%)');
            SheetUtils.log('style cache miss: ' + this._cacheMisses + ' (' + Math.round(this._cacheMisses / cacheAccesses * 100) + '%)');
        }
    } : _.noop;

    // exports ================================================================

    return SheetOperationGenerator;

});
