/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/parser/rangelistparser', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/class',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils'
], function (Utils, Class, SheetUtils, FormulaUtils) {

    'use strict';

    // convenience shortcuts
    var Address = SheetUtils.Address;
    var Range = SheetUtils.Range;
    var RangeArray = SheetUtils.RangeArray;
    var SheetRef = FormulaUtils.SheetRef;

    // class RangeListParserBase ==============================================

    /**
     * An instance of this class parses range list expressions according to a
     * specific formula grammar. This is the base class of several class
     * specializations for the different grammars and reference syntaxes.
     *
     * @constructor
     */
    var RangeListParserBase = Class.extendable(function (docModel, formulaGrammar) {

        // the document model
        this.docModel = docModel;

        // maximum column/row index in a sheet
        this.MAXCOL = docModel.getMaxCol();
        this.MAXROW = docModel.getMaxRow();

        // formula grammar settings
        this.grammar = formulaGrammar;

        // reference grammar configuration
        this.refGrammar = formulaGrammar.refGrammar;
        this.addrConfig = this.refGrammar.addrConfig;
        this.RE = this.refGrammar.RE;

    }); // class RangeListParserBase

    // private methods --------------------------------------------------------

    /**
     * Initializes the members of this parser instance according to the passed
     * options.
     *
     * @param {Object} [options]
     *  Optional parameters.
     */
    RangeListParserBase.prototype._initOptions = function (options) {

        // position of the reference sheet
        this.refSheet = Utils.getIntegerOption(options, 'refSheet', -1);
        this.refSheetModel = this.docModel.getSheetModel(this.refSheet);

        // whether to accept additional separators
        this.extSep = Utils.getBooleanOption(options, 'extSep', false);
        // whether to accept (and ignore) sheet names in 2D range lists
        this.skipSheets = Utils.getBooleanOption(options, 'skipSheets', false);
    };

    /**
     * Parses a range list separator from the passed subexpression.
     *
     * @param {String} expr
     *  The formula subexpression to be parsed.
     *
     * @returns {Object|Null}
     *  A descriptor for the parsed separator, with the properties 'sep' and
     *  'text', or null on error.
     */
    RangeListParserBase.prototype._parseSeparator = function (expr) {

        // the collection of regular expressions
        var RE = this.RE;
        // the matches of the regular expressions
        var matches = null;

        // accept sequences of various white-space characters, use a SPACE as abstract separator
        if ((matches = RE.RL_SEP_WS.exec(expr))) {
            return { sep: ' ', text: matches[0] };
        }

        // accept other separators in extended mode only (only single characters, no sequences)
        if (this.extSep && (matches = RE.RL_SEP_OTHER.exec(expr))) {
            return { sep: matches[0], text: matches[0] };
        }

        return null;
    };

    // protected methods ------------------------------------------------------

    /**
     * Extracts a cell address from the passed matches of a regular expression.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index
     *  The start array index of the cell address in the passed matches. The
     *  matches are expected to contain two elements for the column name, and
     *  the row name, in this order.
     *
     * @returns {Address|Null}
     *  A cell address for the passed matches, if the column and row indexes
     *  are valid; otherwise null.
     */
    RangeListParserBase.prototype.getAddress = function (matches, index) {
        return this.addrConfig.getAddress(matches, index, this.MAXCOL, this.MAXROW);
    };

    /**
     * Extracts a cell range address from the passed matches of a regular
     * expression.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index1
     *  The start array index of the first cell address in the passed matches.
     *
     * @param {Number} [index2]
     *  The start array index of the second cell address in the passed matches.
     *  If omitted, the range will be created from a single cell address.
     *
     * @returns {Range|Null}
     *  A cell range address for the passed matches, if the column and row
     *  indexes are valid; otherwise null.
     */
    RangeListParserBase.prototype.getRange = function (matches, index1, index2) {
        var address1 = this.getAddress(matches, index1);
        var address2 = (typeof index2 === 'number') ? this.getAddress(matches, index2) : address1;
        return (address1 && address2) ? Range.createFromAddresses(address1, address2) : null;
    };

    /**
     * Extracts a cell range address for a column interval from the passed
     * matches of a regular expression.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index1
     *  The start array index of the first column index in the passed matches.
     *
     * @param {Number} [index2]
     *  The start array index of the second column index in the passed matches.
     *  If omitted, the range will be created from a single column index.
     *
     * @returns {Range|Null}
     *  A cell range address for the passed matches, if the column indexes are
     *  valid; otherwise null.
     */
    RangeListParserBase.prototype.getColRange = function (matches, index1, index2) {
        var col1 = this.addrConfig.getCol(matches, index1, this.MAXCOL);
        var col2 = (typeof index2 === 'number') ? this.addrConfig.getCol(matches, index2, this.MAXCOL) : col1;
        return ((col1 >= 0) && (col2 >= 0)) ? Range.create(col1, 0, col2, this.MAXROW) : null;
    };

    /**
     * Extracts a cell range address for a row interval from the passed matches
     * of a regular expression.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index1
     *  The start array index of the first row index in the passed matches.
     *
     * @param {Number} [index2]
     *  The start array index of the row column index in the passed matches. If
     *  omitted, the range will be created from a single row index.
     *
     * @returns {Range|Null}
     *  A cell range address for the passed matches, if the row indexes are
     *  valid; otherwise null.
     */
    RangeListParserBase.prototype.getRowRange = function (matches, index1, index2) {
        var row1 = this.addrConfig.getRow(matches, index1, this.MAXROW);
        var row2 = (typeof index2 === 'number') ? this.addrConfig.getRow(matches, index2, this.MAXROW) : row1;
        return ((row1 >= 0) && (row2 >= 0)) ? Range.create(0, row1, this.MAXCOL, row2) : null;
    };

    RangeListParserBase.prototype.createRangeDesc = function (sheetRefs, range, text) {

        // if the range is missing, the parsed expression is invalid (e.g. row index overflow)
        if (!range) { return null; }

        // check the sheet references (both must refer to the reference sheet)
        if (sheetRefs) {
            if (sheetRefs.r1.sheet !== this.refSheet) { return null; }
            if (sheetRefs.r2 && (sheetRefs.r2.sheet !== this.refSheet)) { return null; }
        }

        return { range: range, text: text };
    };

    // virtual methods --------------------------------------------------------

    /**
     * Sub classes MUST provide an implementation to parse a single cell range
     * from the passed range list subexpression.
     */
    RangeListParserBase.prototype.implParseRange = function (/*expr*/) {
        Utils.error('RangeListParserBase.implParseRange(): missing implementation');
        return null;
    };

    /**
     * Sub classes MUST provide an implementation to generate and return the
     * text representation of a single cell range address.
     */
    RangeListParserBase.prototype.implFormatRange = function (/*range, sheetName*/) {
        Utils.error('RangeListParserBase.implFormatRange(): missing implementation');
        return null;
    };

    // public methods ---------------------------------------------------------

    /**
     * Parses the passed textual range to a cell range address. The range is
     * expected to be in A1 notation.
     *
     * @param {String} formula
     *  The text to be parsed to a cell range address.
     *
     * @param {Object} [options]
     *  Optional parameters. See method FormulaParser.parseRange() for more
     *  details.
     *
     * @returns {Range|Null}
     *  The cell range address parsed from the specified range; or null on any
     *  parser error.
     */
    RangeListParserBase.prototype.parseRange = function (formula, options) {

        // initialize all instance properties
        this._initOptions(options);

        // try to parse the next range address from the formula expression
        var rangeDesc = this.implParseRange(formula);

        // return the range, if the formula does not contain any other garbage
        return (rangeDesc && (formula.length === rangeDesc.text.length)) ? rangeDesc.range : null;
    };

    /**
     * Converts the passed cell range addresss to its text representation.
     *
     * @param {Range} range
     *  The cell range address to be converted.
     *
     * @param {Object} [options]
     *  Optional parameters. See method FormulaParser.formatRange() for more
     *  details.
     *
     * @returns {String}
     *  The text representation of the passed cell range address.
     */
    RangeListParserBase.prototype.formatRange = function (range, options) {

        // the sheet name to be inserted for the cell range address
        var sheetName = Utils.getStringOption(options, 'sheetName', null);

        // convert the range address
        return this.implFormatRange(range, sheetName);
    };

    /**
     * Parses the passed textual range list to an array of cell range
     * addresses. The range list is expected to contain range addresses in A1
     * notation, separated by white-space characters.
     *
     * @param {String} formula
     *  The text to be parsed to an array of cell range addresses.
     *
     * @param {Object} [options]
     *  Optional parameters. See method FormulaParser.parseRangeList() for more
     *  details.
     *
     * @returns {RangeArray|Null}
     *  The array of cell range addresses parsed from the specified range
     *  list; or null on any parser error.
     */
    RangeListParserBase.prototype.parseRangeList = function (formula, options) {

        // initialize all instance properties
        this._initOptions(options);

        // current state of the table reference processor
        var state = 'start';
        // the resulting range array
        var ranges = new RangeArray();
        // the separator character (do not allow mixed separators in a single range list)
        var currSep = null;

        // parse until the end of the expression has been reached, or an error occurs
        while (state !== 'end') {

            // try to parse the next particle of the table reference
            switch (state) {

                case 'start':
                    state = 'range';
                    break;

                case 'range':
                    // try to parse the next range address from the formula expression
                    var rangeDesc = this.implParseRange(formula);
                    // immediately return with value null if the parser fails
                    if (!rangeDesc) { return null; }
                    ranges.push(rangeDesc.range);
                    formula = formula.substr(rangeDesc.text.length);
                    state = 'sep';
                    break;

                case 'sep':
                    // end of formula expression: exit the state machine successfully
                    if (formula.length === 0) {
                        state = 'end';
                        break;
                    }
                    // try to parse a separator, immediately return with value null if the parser fails
                    // (also check the consistency of multiple separators)
                    var sepDesc = this._parseSeparator(formula);
                    if (!sepDesc || ((currSep !== null) && (currSep !== sepDesc.sep))) { return null; }
                    currSep = sepDesc.sep;
                    formula = formula.substr(sepDesc.text.length);
                    state = 'range';
                    break;
            }
        }

        return ranges;
    };

    /**
     * Converts the passed cell range addressses to their text representation.
     *
     * @param {RangeArray|Range} ranges
     *  An array of cell range addresses, or a single cell range address.
     *
     * @param {Object} [options]
     *  Optional parameters. See method FormulaParser.formatRangeList() for
     *  more details.
     *
     * @returns {String}
     *  The text representation of the passed cell range addresses.
     */
    RangeListParserBase.prototype.formatRangeList = function (ranges, options) {

        // the separator character
        var sep = Utils.getStringOption(options, 'sep', ' ');
        // the sheet name to be inserted for each cell range address
        var sheetName = Utils.getStringOption(options, 'sheetName', null);

        // convert the single range addresses in the passed arrray
        return RangeArray.get(ranges).map(function (range) {
            return this.implFormatRange(range, sheetName);
        }, this).join(sep);
    };

    /**
     * Releases the references to other document objects.
     */
    RangeListParserBase.prototype.destroy = function () {
        this.docModel = this.grammar = this.RE = this.refSheetModel = null;
    };

    // class RangeListParserOOX ===============================================

    /**
     * A formula expression parser for the OOXML reference syntax.
     *
     * @constructor
     *
     * @extends RangeListParserBase
     */
    var RangeListParserOOX = RangeListParserBase.extend(function (docModel, formulaGrammar) {

        // base constructor
        RangeListParserBase.call(this, docModel, formulaGrammar);

    }); // class RangeListParserOOX

    // virtual methods --------------------------------------------------------

    /**
     * Extracts a single cell range from the passed range list subexpression.
     */
    RangeListParserOOX.prototype.implParseRange = function (expr) {

        // the collection of regular expressions
        var RE = this.RE;
        // the matches of the regular expressions
        var matches = null;

        // the parser may allow to skip sheet names in the expression
        if (this.skipSheets) {

            // range address with sheet name (before cell addresses!), e.g. Sheet1!A1:C3
            if ((matches = RE.RL_RANGE_3D_REF.exec(expr))) {
                return this.createRangeDesc(null, this.getRange(matches, 3, 5), matches[0]);
            }

            // cell address with sheet name, e.g. Sheet1!A1
            if ((matches = RE.RL_CELL_3D_REF.exec(expr))) {
                return this.createRangeDesc(null, this.getRange(matches, 3), matches[0]);
            }

            // column interval with sheet name, e.g. Sheet1!A:C
            if ((matches = RE.RL_COLS_3D_REF.exec(expr))) {
                return this.createRangeDesc(null, this.getColRange(matches, 3, 4), matches[0]);
            }

            // single column with sheet name (R1C1 only), e.g. Sheet1!C1
            if ((matches = RE.RL_COL_3D_REF.exec(expr))) {
                return this.createRangeDesc(null, this.getColRange(matches, 3), matches[0]);
            }

            // row interval with sheet name, e.g. Sheet1!1:3
            if ((matches = RE.RL_ROWS_3D_REF.exec(expr))) {
                return this.createRangeDesc(null, this.getRowRange(matches, 3, 4), matches[0]);
            }

            // single row with sheet name (R1C1 only), e.g. Sheet1!R1
            if ((matches = RE.RL_ROW_3D_REF.exec(expr))) {
                return this.createRangeDesc(null, this.getRowRange(matches, 3), matches[0]);
            }
        }

        // local range address (before cell addresses!), e.g. A1:C3
        if ((matches = RE.RL_RANGE_REF.exec(expr))) {
            return this.createRangeDesc(null, this.getRange(matches, 1, 3), matches[0]);
        }

        // local cell address, e.g. A1
        if ((matches = RE.RL_CELL_REF.exec(expr))) {
            return this.createRangeDesc(null, this.getRange(matches, 1), matches[0]);
        }

        // local column interval, e.g. A:C
        if ((matches = RE.RL_COLS_REF.exec(expr))) {
            return this.createRangeDesc(null, this.getColRange(matches, 1, 2), matches[0]);
        }

        // local column (R1C1 only), e.g. C1
        if ((matches = RE.RL_COL_REF.exec(expr))) {
            return this.createRangeDesc(null, this.getColRange(matches, 1), matches[0]);
        }

        // local row interval, e.g. 1:3
        if ((matches = RE.RL_ROWS_REF.exec(expr))) {
            return this.createRangeDesc(null, this.getRowRange(matches, 1, 2), matches[0]);
        }

        // local row (R1C1 only), e.g. R1
        if ((matches = RE.RL_ROW_REF.exec(expr))) {
            return this.createRangeDesc(null, this.getRowRange(matches, 1), matches[0]);
        }
    };

    /**
     * Returns the text representation of a single cell range address.
     */
    RangeListParserOOX.prototype.implFormatRange = function (range, sheetName) {

        // the text representation of the start and end address
        var startStr = null, endStr = null;
        if (range.single()) {
            // single cell address (e.g. B3)
            startStr = range.start.toString();
        } else if (this.docModel.isRowRange(range)) {
            // row interval (e.g. 3:5), preferred over column interval for entire sheet range
            startStr = Address.stringifyRow(range.start[1]);
            endStr = Address.stringifyRow(range.end[1]);
        } else if (this.docModel.isColRange(range)) {
            // column interval (e.g. B:D)
            startStr = Address.stringifyCol(range.start[0]);
            endStr = Address.stringifyCol(range.end[0]);
        } else {
            // complete range address (e.g. B3:D5)
            startStr = range.start.toString();
            endStr = range.end.toString();
        }

        // the sheet reference structure
        var sheetRef = sheetName ? new SheetRef(sheetName, false) : null;

        // generate the text representation with sheet names
        return this.grammar.formatGenericRange(this.docModel, sheetRef, null, startStr, endStr);
    };

    // class RangeListParserOF ================================================

    /**
     * A formula expression parser for the OpenFormula reference syntax.
     *
     * @constructor
     *
     * @extends RangeListParserBase
     */
    var RangeListParserOF = RangeListParserBase.extend(function (docModel, formulaGrammar) {

        // base constructor
        RangeListParserBase.call(this, docModel, formulaGrammar);

    }); // class RangeListParserOF

    // virtual methods --------------------------------------------------------

    /**
     * Extracts a single cell range from the passed range list subexpression.
     */
    RangeListParserOF.prototype.implParseRange = function (expr) {

        // the collection of regular expressions
        var RE = this.RE;
        // the matches of the regular expressions
        var matches = null;

        // the parser may allow to skip sheet names in the expression
        if (this.skipSheets) {

            // range address with sheet range (before 3D ranges!), e.g. Sheet1.A1:Sheet2.C3
            if ((matches = RE.RL_RANGE_CUBE_REF.exec(expr))) {
                return this.createRangeDesc(null, this.getRange(matches, 3, 7), matches[0]);
            }

            // range address with sheet name (before cell addresses!), e.g. Sheet1.A1:C3
            if ((matches = RE.RL_RANGE_3D_REF.exec(expr))) {
                return this.createRangeDesc(null, this.getRange(matches, 3, 5), matches[0]);
            }

            // cell address with sheet name, e.g. Sheet1.A1
            if ((matches = RE.RL_CELL_3D_REF.exec(expr))) {
                return this.createRangeDesc(null, this.getRange(matches, 3), matches[0]);
            }
        }

        // local range address (before cell addresses!), e.g. A1:C3
        if ((matches = RE.RL_RANGE_REF.exec(expr))) {
            return this.createRangeDesc(null, this.getRange(matches, 1, 3), matches[0]);
        }

        // local cell address, e.g. A1
        if ((matches = RE.RL_CELL_REF.exec(expr))) {
            return this.createRangeDesc(null, this.getRange(matches, 1), matches[0]);
        }
    };

    /**
     * Returns the text representation of a single cell range address.
     */
    RangeListParserOF.prototype.implFormatRange = function (range, sheetName) {

        // the text representation of the start address
        var startStr = range.start.toString();
        // the text representation of the end address
        var endStr = range.single() ? null : range.end.toString();

        // the sheet reference structure for the start address
        var sheet1Ref = sheetName ? new SheetRef(sheetName, false) : null;
        // the sheet reference structure for the end address
        var sheet2Ref = (sheetName && endStr) ? sheet1Ref : null;

        // generate the text representation with sheet names
        return this.grammar.formatGenericRange(this.docModel, sheet1Ref, sheet2Ref, startStr, endStr);
    };

    // static class RangeListParser ===========================================

    var RangeListParser = {};

    // static methods ---------------------------------------------------------

    /**
     * Creates a new range list parser instance for the specified spreadsheet
     * document and formula grammar.
     *
     * @param {SpreadsheetModel} docModel
     *  The document model to be associated with the new range list parser.
     *
     * @param {String} grammarId
     *  The identifier of the formula grammar to be used to parse range list
     *  expressions. See SpreadsheetDocument.getFormulaGrammar() for more
     *  details.
     *
     * @returns {RangeListParserBase}
     *  A new range list parser for the specified document and formula grammar.
     */
    RangeListParser.create = function (docModel, grammarId) {
        var formulaGrammar = docModel.getFormulaGrammar(grammarId);
        switch (formulaGrammar.REF_SYNTAX) {
            case 'ooxml':
            case 'odfui':
                return new RangeListParserOOX(docModel, formulaGrammar);
            case 'of':
                return new RangeListParserOF(docModel, formulaGrammar);
        }
        Utils.error('RangeListParser.create(): invalid reference syntax: "' + formulaGrammar.REF_SYNTAX + '"');
    };

    // exports ================================================================

    return RangeListParser;

});
