/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/funcs/informationfuncs', [
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/spreadsheet/model/formula/utils/cellref'
], function (ErrorCode, CellRef) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all spreadsheet functions providing some kind of
     * information about the contents of sheet cells.
     *
     * See the README file in this directory for a detailed documentation about
     * the format of function descriptor objects.
     *
     *************************************************************************/

    // shortcuts to mathematical functions
    var floor = Math.floor;
    var abs = Math.abs;

    // exports ================================================================

    return {

        CELL: {
            category: 'information',
            minParams: 1,
            maxParams: 2,
            type: 'val:any',
            recalc: 'always',
            signature: 'val:str ref:single|deps:skip',
            resolve: (function () {

                function formatAddress(context, address) {
                    var cellRef = CellRef.create(address, true, true);
                    context.checkCellRef(cellRef);
                    return context.formulaGrammar.formatReference(context.docModel, context.getTargetAddress(), null, null, cellRef, null);
                }

                function getSheetModel(context, sheet) {
                    var sheetModel = context.docModel.getSheetModel(sheet);
                    if (!sheetModel) { throw ErrorCode.VALUE; }
                    return sheetModel;
                }

                function getCellCollection(context, sheet) {
                    return getSheetModel(context, sheet).getCellCollection();
                }

                function getAttributeSet(context, sheet, address) {
                    return getCellCollection(context, sheet).getAttributeSet(address);
                }

                function getColorFormat(context, sheet, address) {
                    var numberSections = getCellCollection(context, sheet).getParsedFormat(address).numberSections;
                    return ((numberSections.length > 1) && numberSections[1].colorName) ? 1 : 0;
                }

                function getFormatCode(context, sheet, address) {
                    var parsedFormat = getCellCollection(context, sheet).getParsedFormat(address);
                    var formats = ['G', 'F0', ',0', 'F2', ',2', 'C0', 'C0-', 'C2', 'C2-', 'P0', 'P2', 'S2', 'G', 'D4', 'D1', 'D2', 'D3', 'D5', 'D7', 'D6', 'D9', 'D8'];
                    var formatId = context.numberFormatter.resolveFormatId(parsedFormat.formatCode);
                    return ((formatId !== null) && (formatId <= formats.length)) ? formats[formatId] : 'G';
                }

                function getPrefix(context, sheet, address) {
                    if (getValueType(context, sheet, address) !== 'l') { return ''; }
                    switch (getAttributeSet(context, sheet, address).cell.alignHor) {
                        case 'right':
                            return '"';
                        case 'center':
                        case 'centerAcross':
                            return '^';
                        case 'fill':
                            return '\\';
                        default:
                            return '\'';
                    }
                }

                function getValueType(context, sheet, address) {
                    var value = context.getCellValue(sheet, address);
                    return (value === null) ? 'b' : (typeof value === 'string') ? 'l' : 'v';
                }

                function getFileName(context, sheet) {
                    var filePath = context.app.getFilePath();
                    var fileName = context.app.getFullFileName();
                    var sheetName = context.docModel.getSheetName(sheet);
                    return ((filePath.length > 0) ? ('/' + filePath.join('/') + '/') : '') + '[' + fileName + ']' + sheetName;
                }

                return function (infoType, range) {

                    // resolve translated or English info type (prefer translations over English names)
                    var paramCollection = this.formulaResource.getCellParamCollection();
                    var paramKey = paramCollection.getKey(infoType, true) || paramCollection.getKey(infoType, false);

                    // Get the address of the target cell. If the parameter is omitted, then:
                    // - OOXML: active cell in active sheet of the document during last modification (TODO),
                    // - ODF: the formula reference address (formula cell itself).
                    var sheet = range ? range.sheet1 : this.getRefSheet();
                    var address = range ? range.start : this.getTargetAddress();

                    switch (paramKey) {
                        case 'ADDRESS':
                            return formatAddress(this, address);
                        case 'COL':
                            return address[0] + 1;
                        case 'COLOR':
                            return getColorFormat(this, sheet, address);
                        case 'CONTENTS':
                            // bug 47975: prevent circular error when CELL references itself without second parameter
                            return range ? this.getCellValue(sheet, address) : 0;
                        case 'FILENAME':
                            return getFileName(this, sheet);
                        case 'FORMAT':
                            return getFormatCode(this, sheet, address);
                        case 'PARENTHESES':
                            throw ErrorCode.NA; // TODO
                        case 'PREFIX':
                            return getPrefix(this, sheet, address);
                        case 'PROTECT':
                            return getAttributeSet(this, sheet, address).cell.unlocked ? 0 : 1;
                        case 'ROW':
                            return address[1] + 1;
                        case 'TYPE':
                            // bug 47975: prevent circular error when CELL references itself without second parameter
                            return range ? getValueType(this, sheet, address) : 'l';
                        case 'WIDTH':
                            throw ErrorCode.NA; // TODO
                    }
                    throw ErrorCode.VALUE;
                };
            }()),
            // result depends on reference address (behaves like a relative reference), if second parameter is missing
            relColRef: function (count) { return count < 2; },
            relRowRef: function (count) { return count < 2; }
        },

        CURRENT: {
            category: 'information',
            name: { ooxml: null, odf: 'ORG.OPENOFFICE.CURRENT' },
            hidden: true,
            minParams: 0,
            maxParams: 0,
            type: 'any'
        },

        'ERROR.TYPE': {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:any',
            resolve: function (value) {
                if (value instanceof ErrorCode) { return value.num; }
                throw ErrorCode.NA;
            }
        },

        'ERRORTYPE.ODF': {
            category: 'information',
            name: { ooxml: null, odf: 'ORG.OPENOFFICE.ERRORTYPE' },
            minParams: 1,
            maxParams: 1,
            type: 'val:num'
        },

        INFO: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val:str',
            recalc: 'always'
        },

        ISBLANK: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val:bool',
            signature: 'val:any',
            resolve: _.isNull // error codes result in FALSE
        },

        ISERR: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val:bool',
            signature: 'val:any',
            resolve: function (value) {
                return (value instanceof ErrorCode) && (value !== ErrorCode.NA);
            }
        },

        ISERROR: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val:bool',
            signature: 'val:any',
            resolve: function (value) { return value instanceof ErrorCode; }
        },

        ISEVEN: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val:bool',
            signature: 'val:num',
            resolve: function (number) {
                return (floor(abs(number)) % 2) === 0;
            }
        },

        ISFORMULA: {
            category: 'information',
            name: { ooxml: '_xlfn.ISFORMULA' },
            minParams: 1,
            maxParams: 1,
            type: 'val:bool',
            recalc: 'always',
            signature: 'ref:single|deps:skip',
            resolve: function (range) {

                // always use the first cell in the range (TODO: matrix context)
                var address = range.start;

                // accept reference to own cell (no circular reference error)
                if (this.isRefSheet(range.sheet1) && this.isTargetAddress(address)) {
                    return true; // do not check the cell, a new formula has not been inserted yet
                }

                // getCellFormula() returns null for value cells
                return this.getCellFormula(range.sheet1, address) !== null;
            }
        },

        ISLOGICAL: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val:bool',
            signature: 'val:any',
            resolve: _.isBoolean // error codes and empty cells result in FALSE
        },

        ISNA: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val:bool',
            signature: 'val:any',
            resolve: function (value) { return value === ErrorCode.NA; }
        },

        ISNONTEXT: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val:bool',
            signature: 'val:any',
            resolve: function (value) {
                return !_.isString(value); // error codes and empty cells result in TRUE
            }
        },

        ISNUMBER: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val:bool',
            signature: 'val:any',
            resolve: _.isNumber // error codes and empty cells result in FALSE
        },

        ISODD: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val:bool',
            signature: 'val:num',
            resolve: function (number) {
                return (floor(abs(number)) % 2) !== 0;
            }
        },

        ISREF: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val:bool',
            signature: 'any|deps:skip',
            resolve: function (operator) { return operator.isReference(); } // error codes result in FALSE
        },

        ISTEXT: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val:bool',
            signature: 'val:any',
            resolve: _.isString // error codes and empty cells result in FALSE
        },

        NA: {
            category: 'information',
            minParams: 0,
            maxParams: 0,
            type: 'val:err',
            resolve: _.constant(ErrorCode.NA)
        },

        STYLE: {
            category: 'information',
            name: { ooxml: null, odf: 'ORG.OPENOFFICE.STYLE' },
            hidden: true,
            minParams: 1,
            maxParams: 3,
            type: 'any'
        },

        TYPE: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'val:any', // resolve cell references to values, accept error codes
            resolve: function (value) {
                // array literals always result in 64 regardless of their contents
                if (this.getOperand(0).isMatrix()) { return 64; }
                // determine type of value parameters and cell contents
                if (_.isNumber(value)) { return 1; }
                if (_.isString(value)) { return 2; }
                if (_.isBoolean(value)) { return 4; }
                if (value instanceof ErrorCode) { return 16; }
                // default (reference to empty cell, or cell not fetched from server)
                return 1;
            }
        }
    };

});
