/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/drawingcollectionmixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/iterator',
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/tk/render/rectangle',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/drawinglayer/utils/drawingutils',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/drawing/framenodemanager',
    'io.ox/office/spreadsheet/model/drawing/text/textframeutils'
], function (Utils, Iterator, ValueMap, Rectangle, AttributeUtils, DrawingUtils, DrawingFrame, SheetUtils, FrameNodeManager, TextFrameUtils) {

    'use strict';

    // convenience shortcuts
    var AnchorMode = SheetUtils.AnchorMode;
    var Interval = SheetUtils.Interval;
    var Address = SheetUtils.Address;

    // horizontal drawing anchor attribute names
    var COL_ATTRIBUTE_NAMES = {
        offset: 'left',
        size: 'width',
        startIndex: 'startCol',
        startOffset: 'startColOffset',
        endIndex: 'endCol',
        endOffset: 'endColOffset'
    };

    // vertical drawing anchor attribute names
    var ROW_ATTRIBUTE_NAMES = {
        offset: 'top',
        size: 'height',
        startIndex: 'startRow',
        startOffset: 'startRowOffset',
        endIndex: 'endRow',
        endOffset: 'endRowOffset'
    };

    // the names of all cell anchor attributes in the 'drawing' attribute family
    var CELL_ANCHOR_ATTR_NAMES = ['anchorType'].concat(_.values(COL_ATTRIBUTE_NAMES)).concat(_.values(ROW_ATTRIBUTE_NAMES));

    // class CellAnchor =======================================================

    /**
     * Represents the position of a single corner of a drawing object that is
     * anchored to a specific spreadsheet cell.
     *
     * @constructor
     *
     * @property {Address} address
     *  The address of the spreadsheet cell containing the corner of the
     *  drawing object.
     *
     * @property {Number} colOffset
     *  The relative offset inside the column, in 1/100 mm.
     *
     * @property {Number} rowOffset
     *  The relative offset inside the row, in 1/100 mm.
     */
    function CellAnchor(address, colOffset, rowOffset) {

        this.address = address.clone();
        this.colOffset = colOffset;
        this.rowOffset = rowOffset;

    } // class CellAnchor

    // private global functions ===============================================

    /**
     * Returns the effective anchor mode for the passed anchor attributes.
     *
     * @param {Object} anchorAttrs
     *  The anchor attributes of a drawing object. Expects existing attribute
     *  'anchorType'.
     *
     * @param {Boolean} displayAnchor
     *  If set to true, uses the runtime behavior defined by the 'anchorType'
     *  formatting attribute as anchor mode, instead of the internal anchor
     *  mode normally used.
     *
     * @returns {AnchorMode}
     *  The effective anchor mode for the passed anchor attributes.
     */
    var getAnchorMode = (function () {

        // maps 'anchorType' attribute values to the internal anchor mode used for anchor attributes
        var INTERNAL_ANCHOR_MODES = {
            twoCell: AnchorMode.TWO_CELL,
            oneCell: AnchorMode.ONE_CELL,
            absolute: AnchorMode.ABSOLUTE,
            twoCellAsOneCell: AnchorMode.TWO_CELL, // uses "twoCell" attributes
            twoCellAsAbsolute: AnchorMode.TWO_CELL // uses "twoCell" attributes
        };

        // maps 'anchorType' attribute values to display anchor mode defining runtime behavior
        var DISPLAY_ANCHOR_MODES = {
            twoCell: AnchorMode.TWO_CELL,
            oneCell: AnchorMode.ONE_CELL,
            absolute: AnchorMode.ABSOLUTE,
            twoCellAsOneCell: AnchorMode.ONE_CELL, // behaves as "oneCell" anchor
            twoCellAsAbsolute: AnchorMode.ABSOLUTE // behaves as "absolute" anchor
        };

        return function (drawingAttrs, displayAnchor) {
            var enumMap = displayAnchor ? DISPLAY_ANCHOR_MODES : INTERNAL_ANCHOR_MODES;
            return enumMap[drawingAttrs.anchorType] || AnchorMode.TWO_CELL;
        };
    }());

    // class SheetDrawingCollectionMixin ======================================

    /**
     * A mix-in class that extends the public API of a drawing collection in a
     * spreadsheet document.
     *
     * @constructor
     *
     * @internal
     *  This is a mix-in class supposed to extend an existing instance of the
     *  class DrawingCollection or any of its sub classes used in spreadsheet
     *  documents. Expects the symbol 'this' to be bound to an instance of
     *  DrawingCollection.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model containing the drawing collection with this drawing
     *  model object.
     *
     * @param {Function} cloneConstructor
     *  A callback function invoked from the public method clone() which has to
     *  create and return a new instance cloned from this drawing model.
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  - {Function} [initOptions.generateRestoreOperations]
     *      A callback function that will be invoked from the own public method
     *      DrawingModelMixin.generaterestoreOperations() while generating
     *      document operations that will delete this drawing object. An
     *      implementation must generate more undo operations to restore the
     *      complete drawing object.
     *  - {Function} [initOptions.generateUpdateFormulaOperations]
     *      A callback function that will be invoked from the own public method
     *      DrawingModelMixin.generateUpdateFormulaOperations() while
     *      generating document operations that involve to update the formula
     *      expressions in this drawing object. Receives all parameters passed
     *      to the public method mentioned before. May return an attribute set
     *      with additional formatting attributes to be applied at this drawing
     *      object, or null.
     */
    function SheetDrawingCollectionMixin(sheetModel) {

        // self reference
        var self = this;

        // the document model
        var docModel = sheetModel.getDocModel();

        // the application instance
        var app = docModel.getApp();

        // the manager for the DOM drawing frames of the drawing models in this collection
        var frameNodeManager = new FrameNodeManager(docModel);

        // cached settings for drawing model frames, mapped by drawing model UIDs
        var modelFrameCache = new ValueMap();

        // the column and row collections (size and formatting of all columns/rows)
        var colCollection = sheetModel.getColCollection();
        var rowCollection = sheetModel.getRowCollection();

        // special behavior in OOXML documents
        var ooxml = app.isOOXML();

        // private methods ----------------------------------------------------

        /**
         * Returns the anchor type attribute for the specified display anchor
         * mode.
         *
         * @param {AnchorMode}
         *  The display anchor mode to be converted to the attribute value.
         *
         * @returns {String}
         *  The resulting value for the attribute "anchorType".
         */
        function getAnchorType(anchorMode) {
            // prefer internal two-cell anchor for OOXML
            switch (anchorMode) {
                case AnchorMode.ABSOLUTE:
                    return ooxml ? 'twoCellAsAbsolute' : 'absolute';
                case AnchorMode.ONE_CELL:
                    return ooxml ? 'twoCellAsOneCell' : 'oneCell';
                case AnchorMode.TWO_CELL:
                    return 'twoCell';
            }
        }

        /**
         * Returns all available information about the position of a drawing
         * object, according to the passed drawing attributes.
         *
         * @param {Object} drawingAttrs
         *  The values of the drawing anchor attributes, and the anchor type.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.displayAnchor=false]
         *      If set to true, uses the runtime behavior defined by the
         *      'anchorType' formatting attribute as anchor mode, instead of
         *      the internal anchor mode normally used.
         *
         * @returns {Object}
         *  The result object with position information for the passed drawing
         *  attribute set, in the following properties:
         *  - {AnchorMode} anchorMode
         *      The effective anchor mode, according to the anchor type
         *      attribute, and the passed options.
         *  - {Rectangle} rectHmm
         *      The absolute location of the drawing object in the sheet, in
         *      1/100 of millimeters, independent from the current sheet zoom
         *      factor.
         *  - {Rectangle} rectPx
         *      The absolute location of the drawing object in the sheet, in
         *      pixels, according to the current sheet zoom factor.
         *  - {CellAnchor} startAnchor
         *      The location of the top-left corner of the drawing object, as
         *      cell anchor position.
         *  - {CellAnchor} endAnchor
         *      The location of the bottom-right corner of the drawing object,
         *      as cell anchor position.
         *  - {Boolean} hiddenCols
         *      Whether the drawing object is located entirely in a range of
         *      hidden columns.
         *  - {Boolean} hiddenRows
         *      Whether the drawing object is located entirely in a range of
         *      hidden rows.
         */
        function getAnchorDescriptor(drawingAttrs, options) {

            // whether to use the display anchor instead of the anchor type
            var displayAnchor = Utils.getBooleanOption(options, 'displayAnchor', false);
            // the actual anchor mode used to find inactive anchor attributes
            var anchorMode = getAnchorMode(drawingAttrs, displayAnchor);
            // the location of the drawing object, in 1/100 mm
            var rectHmm = new Rectangle(0, 0, 0, 0);
            // the location of the drawing object, in pixels
            var rectPx = new Rectangle(0, 0, 0, 0);
            // the location of the top-left corner, as cell anchor
            var startAnchor = new CellAnchor(Address.A1, 0, 0);
            // the location of the bottom-right corner, as cell anchor
            var endAnchor = new CellAnchor(Address.A1, 0, 0);
            // the resulting cell anchor descripor
            var anchorDesc = { anchorMode: anchorMode, rectHmm: rectHmm, rectPx: rectPx, startAnchor: startAnchor, endAnchor: endAnchor };
            // collection entry of the column/row collection
            var entryData = null;

            // get start position of the drawing object
            switch (anchorMode) {
                case AnchorMode.TWO_CELL:
                case AnchorMode.ONE_CELL:
                    // left position in 1/100 mm and pixels, start column index and offset
                    rectHmm.left = colCollection.getEntryOffsetHmm(drawingAttrs.startCol, drawingAttrs.startColOffset);
                    rectPx.left = colCollection.getEntryOffset(drawingAttrs.startCol, drawingAttrs.startColOffset);
                    startAnchor.address[0] = drawingAttrs.startCol;
                    startAnchor.colOffset = drawingAttrs.startColOffset;

                    // top position in 1/100 mm and pixels, start row index and offset
                    rectHmm.top = rowCollection.getEntryOffsetHmm(drawingAttrs.startRow, drawingAttrs.startRowOffset);
                    rectPx.top = rowCollection.getEntryOffset(drawingAttrs.startRow, drawingAttrs.startRowOffset);
                    startAnchor.address[1] = drawingAttrs.startRow;
                    startAnchor.rowOffset = drawingAttrs.startRowOffset;
                    break;

                case AnchorMode.ABSOLUTE:
                    // left position in 1/100 mm and pixels, start column index and offset
                    entryData = colCollection.getEntryByOffset(drawingAttrs.left);
                    rectHmm.left = entryData.offsetHmm + entryData.relOffsetHmm;
                    rectPx.left = entryData.offset + entryData.relOffset;
                    startAnchor.address[0] = entryData.index;
                    startAnchor.colOffset = entryData.relOffsetHmm;

                    // top position in 1/100 mm and pixels, start row index and offset
                    entryData = rowCollection.getEntryByOffset(drawingAttrs.top);
                    rectHmm.top = entryData.offsetHmm + entryData.relOffsetHmm;
                    rectPx.top = entryData.offset + entryData.relOffset;
                    startAnchor.address[1] = entryData.index;
                    startAnchor.rowOffset = entryData.relOffsetHmm;
                    break;
            }

            // get effective size of the drawing object
            switch (anchorMode) {
                case AnchorMode.TWO_CELL:
                    // width in 1/100 mm and pixels; end column index and offset
                    rectHmm.width = colCollection.getEntryOffsetHmm(drawingAttrs.endCol, drawingAttrs.endColOffset) - rectHmm.left;
                    rectPx.width = colCollection.getEntryOffset(drawingAttrs.endCol, drawingAttrs.endColOffset) - rectPx.left;
                    endAnchor.address[0] = drawingAttrs.endCol;
                    endAnchor.colOffset = drawingAttrs.endColOffset;

                    // height in 1/100 mm and pixels, end row index and offset
                    rectHmm.height = rowCollection.getEntryOffsetHmm(drawingAttrs.endRow, drawingAttrs.endRowOffset) - rectHmm.top;
                    rectPx.height = rowCollection.getEntryOffset(drawingAttrs.endRow, drawingAttrs.endRowOffset) - rectPx.top;
                    endAnchor.address[1] = drawingAttrs.endRow;
                    endAnchor.rowOffset = drawingAttrs.endRowOffset;
                    break;

                case AnchorMode.ONE_CELL:
                case AnchorMode.ABSOLUTE:
                    // width in 1/100 mm and pixels; end column index and offset
                    entryData = colCollection.getEntryByOffset(rectHmm.left + drawingAttrs.width);
                    rectHmm.width = entryData.offsetHmm + entryData.relOffsetHmm - rectHmm.left;
                    rectPx.width = entryData.offset + entryData.relOffset - rectPx.left;
                    endAnchor.address[0] = entryData.index;
                    endAnchor.colOffset = entryData.relOffsetHmm;

                    // height in 1/100 mm and pixels, end row index and offset
                    entryData = rowCollection.getEntryByOffset(rectHmm.top + drawingAttrs.height);
                    rectHmm.height = entryData.offsetHmm + entryData.relOffsetHmm - rectHmm.top;
                    rectPx.height = entryData.offset + entryData.relOffset - rectPx.top;
                    endAnchor.address[1] = entryData.index;
                    endAnchor.rowOffset = entryData.relOffsetHmm;
                    break;
            }

            // ooxml has special behavior to swap width/height values of the drawing if certan rotation/flip conditions are met.
            // For proper display, those values should be reverted back.
            if (ooxml && DrawingUtils.hasSwappedDimensions(drawingAttrs)) {
                anchorDesc.rectHmm.rotateSelf();
                anchorDesc.rectPx.rotateSelf();
            }

            // whether the drawing is located entirely in hidden columns or rows
            anchorDesc.hiddenCols = colCollection.isIntervalHidden(new Interval(startAnchor.address[0], endAnchor.address[0]));
            anchorDesc.hiddenRows = rowCollection.isIntervalHidden(new Interval(startAnchor.address[1], endAnchor.address[1]));

            return anchorDesc;
        }

        /**
         * Returns the complete anchor attributes needed to set the position
         * and size of a drawing object to the passed absolute rectangle in the
         * sheet area.
         *
         * @param {Rectangle} rectangle
         *  The position of a drawing object in the sheet in pixels, according
         *  to the current sheet zoom factor.
         *
         * @returns {Object}
         *  The anchor attributes for the location of a drawing object.
         */
        function calculateAnchorForRectangle(rectangle) {

            // collection entry of the column/row collection
            var entryData = null;
            // the new anchor attributes
            var anchorAttrs = {};

            // horizontal start position
            entryData = colCollection.getEntryByOffset(rectangle.left, { pixel: true });
            anchorAttrs.left = entryData.offsetHmm + entryData.relOffsetHmm;
            anchorAttrs.startCol = entryData.index;
            anchorAttrs.startColOffset = entryData.relOffsetHmm;

            // object width and horizontal end position
            entryData = colCollection.getEntryByOffset(rectangle.right(), { pixel: true });
            anchorAttrs.width = entryData.offsetHmm + entryData.relOffsetHmm - anchorAttrs.left;
            anchorAttrs.endCol = entryData.index;
            anchorAttrs.endColOffset = entryData.relOffsetHmm;

            // vertical start position
            entryData = rowCollection.getEntryByOffset(rectangle.top, { pixel: true });
            anchorAttrs.top = entryData.offsetHmm + entryData.relOffsetHmm;
            anchorAttrs.startRow = entryData.index;
            anchorAttrs.startRowOffset = entryData.relOffsetHmm;

            // object height and vertical end position
            entryData = rowCollection.getEntryByOffset(rectangle.bottom(), { pixel: true });
            anchorAttrs.height = entryData.offsetHmm + entryData.relOffsetHmm - anchorAttrs.top;
            anchorAttrs.endRow = entryData.index;
            anchorAttrs.endRowOffset = entryData.relOffsetHmm;

            return anchorAttrs;
        }

        /**
         * Recalculates the inactive cell anchor attributes for the passed
         * anchor attributes, and returns an updated anchor attributes to be
         * applied at the drawing model.
         *
         * @param {Object} anchorAttrs
         *  The cell anchor attributes to be finalized.
         *
         * @returns {Object|Null}
         *  The map of anchor attributes to be applied at the passed drawing
         *  model to update its complete anchor location; or null, if the
         *  drawing model does not need to be updated.
         */
        function finalizeAnchorAttributes(drawingModel, anchorAttrs) {

            // recalculate the inactive anchor attributes according to the current display anchor mode
            anchorAttrs = _.extend({}, anchorAttrs, self.getInactiveAnchorAttributes(anchorAttrs, { displayAnchor: true }));

            // return null, if no anchor attribute does actually change (bug 57983: but do not return reduced attribute set!)
            return _.isEmpty(drawingModel.getReducedAttributeSet({ drawing: anchorAttrs })) ? null : anchorAttrs;
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the DOM container node for the DOM models of all drawing
         * objects in this collection.
         *
         * @returns {HTMLElement}
         *  The DOM container node for the DOM models of all drawing objects in
         *  this collection.
         */
        this.getContainerNode = function () {
            return frameNodeManager.getRootNode();
        };

        /**
         * Returns the DOM drawing frame that represents the drawing model at
         * the specified document position. The DOM drawing frame is used to
         * store the text contents of the drawing objects.
         *
         * @param {Array<Number>} position
         *  The document position of a drawing model to return the DOM drawing
         *  frame for.
         *
         * @returns {jQuery|Null}
         *  The DOM drawing frame for the specified drawing object; or null, if
         *  no drawing frame could be found.
         */
        this.getDrawingFrame = function (position) {
            var drawingModel = this.getModel(position);
            return drawingModel ? frameNodeManager.getDrawingFrame(drawingModel) : null;
        };

        /**
         * Returns the DOM drawing frame that represents the passed drawing
         * model. The DOM drawing frame is used to store the text contents of
         * the drawing objects.
         *
         * @param {DrawingModel} drawingModel
         *  The drawing model instance to return the DOM drawing frame for.
         *
         * @returns {jQuery|Null}
         *  The DOM drawing frame for the passed drawing model; or null, if no
         *  drawing frame could be found.
         */
        this.getDrawingFrameForModel = function (drawingModel) {
            return frameNodeManager.getDrawingFrame(drawingModel);
        };

        /**
         * Refreshes the settings of the DOM drawing frame associated with the
         * passed drawing model, and returns the cached rendering settings for
         * the drawing model.
         *
         * @param {DrawingModel} drawingModel
         *  The drawing model whose DOM drawing frame will be refreshed.
         *
         * @returns {Object|Null}
         *  A descriptor with rendering data for the passed drawing model; or
         *  null, if no data is available for the drawing model. The descriptor
         *  will contain the following properties:
         *  - {jQuery} drawingFrame
         *      The DOM drawing frame associated with the drawing model. This
         *      drawing frame serves as model container for the text contents
         *      of shape objects.
         *  - {Object} explicitAttrSet
         *      A deep copy of the explicit attribute set of the drawing model.
         *  - {Object|Null} textPosition
         *      The position of the text frame inside the drawing frame at a
         *      fixed zoom level of 100%, as object with the properties 'left',
         *      'top', 'right'm, 'bottom', 'width', and 'height', in pixels. If
         *      the drawing model does not contain a text frame, this property
         *      will be null.
         */
        this.refreshDrawingFrameForModel = function (drawingModel) {

            // directly return existing cached settings for the passed drawing model
            var renderSettings = modelFrameCache.get(drawingModel.getUid(), null);
            if (renderSettings) { return renderSettings; }

            // resolve the DOM drawing frame, do nothing if the frame does not exist yet
            var drawingFrame = drawingModel.getModelFrame();
            if (!drawingFrame) { return null; }

            // create a new entry in the drawing model cache
            renderSettings = modelFrameCache.insert(drawingModel.getUid(), { drawingFrame: drawingFrame, textPosition: null });

            // copy explicit formatting attributes (default character formatting)
            renderSettings.explicitAttrSet = drawingModel.getExplicitAttributeSet();
            AttributeUtils.setExplicitAttributes(drawingFrame, renderSettings.explicitAttrSet);

            // set the pixel size of the drawing frame (always for 100% zoom factor to get the original padding!)
            var rectangle = drawingModel.getRectangleHmm();
            if (ooxml && drawingModel.hasSwappedDimensions()) { rectangle.rotateSelf(); }
            var width = Utils.convertHmmToLength(rectangle.width, 'px', 1);
            var height = Utils.convertHmmToLength(rectangle.height, 'px', 1);
            drawingFrame.css({ width: width, height: height });

            // additional processing for the embedded text frame
            TextFrameUtils.withTextFrame(drawingFrame, function (textFrame) {

                // update the text frame position, but not the entire formatting (especially, do not create canvas elements!)
                var mergedAttrSet = drawingModel.getMergedAttributeSet(true);
                DrawingFrame.updateTextFramePosition(app, drawingFrame, mergedAttrSet);

                // reset CSS transformations before measuring DOM positions
                var drawingTransform = drawingFrame.css('transform');
                var textTransform = textFrame.css('transform');
                drawingFrame.css('transform', '');
                textFrame.css('transform', '');

                // store the padding of the text frame to its enclosing drawing frame (to be used by renderers)
                renderSettings.textPosition = Utils.getChildNodePositionInNode(drawingFrame, textFrame);

                // restore the CSS transformations
                drawingFrame.css('transform', drawingTransform);
                textFrame.css('transform', textTransform);
            });

            return renderSettings;
        };

        /**
         * Updates the CSS text formatting of the text contents in the drawing
         * frame associated to the passed drawing model.
         *
         * @param {DrawingModel} drawingModel
         *  The model of the drawing objects whose text formatting will be
         *  updated.
         *
         * @returns {SheetDrawingCollection}
         *  A reference to this instance.
         */
        this.updateTextFormatting = function (drawingModel) {
            frameNodeManager.updateTextFormatting(drawingModel);
            return this;
        };

        /**
         * Returns whether the passed ranges can be deleted, i.e. if the ranges
         * do not contain drawing objects that cannot be restored with undo
         * operations.
         *
         * @param {RangeArray} ranges
         *  The addresses of the cell ranges to be deleted.
         *
         * @returns {Boolean}
         *  Whether the cell ranges can be deleted safely.
         */
        this.canRestoreDeletedRanges = function (ranges) {
            return Iterator.every(this.createModelIterator(), function (drawingModel) {

                // absolutely positioned or sized drawings will not be deleted
                var anchorMode = getAnchorMode(drawingModel.getMergedAttributeSet(true).drawing, true);
                if (anchorMode !== AnchorMode.TWO_CELL) { return true; }

                // Check that the drawing object can be restored deeply (all children of groups).
                // Return false if it will be deleted and cannot be restored.
                return drawingModel.isDeepRestorable() || !ranges.contains(drawingModel.getRange());
            });
        };

        /**
         * Extracts the anchor attributes from the passed drawing object.
         *
         * @param {DrawingModel} drawingModel
         *  The model of a drawing object.
         *
         * @returns {Object}
         *  The anchor attributes of the passed drawing object.
         */
        this.getAnchorAttributesForModel = function (drawingModel) {
            return _.pick(drawingModel.getMergedAttributeSet(true).drawing, CELL_ANCHOR_ATTR_NAMES);
        };

        /**
         * Returns the inactive cell anchor attributes of the passed attribute
         * set, according to the anchor mode and anchor attributes.
         *
         * @param {Object} drawingAttrs
         *  The values of the drawing anchor attributes, and the anchor type.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.displayAnchor=false]
         *      If set to true, uses the runtime behavior as defined by the
         *      formatting attribute "anchorType" as anchor mode, instead of
         *      the internal anchor mode normally used.
         *
         * @returns {Object}
         *  The current values of the inactive anchor attributes, according to
         *  the anchor mode.
         */
        this.getInactiveAnchorAttributes = function (drawingAttrs, options) {

            // current position of the drawing object
            var anchorDesc = getAnchorDescriptor(drawingAttrs, options);
            // the location of the drawing object, in 1/100 mm
            var rectHmm = anchorDesc.rectHmm;
            // the resulting inactive anchor attributes
            var anchorAttrs = {};

            // add absolute position, if anchor starts in a cell; otherwise add the start cell address
            switch (anchorDesc.anchorMode) {
                case AnchorMode.TWO_CELL:
                case AnchorMode.ONE_CELL:
                    anchorAttrs.left = rectHmm.left;
                    anchorAttrs.top = rectHmm.top;
                    break;
                case AnchorMode.ABSOLUTE:
                    var startAnchor = anchorDesc.startAnchor;
                    anchorAttrs.startCol = startAnchor.address[0];
                    anchorAttrs.startColOffset = startAnchor.colOffset;
                    anchorAttrs.startRow = startAnchor.address[1];
                    anchorAttrs.startRowOffset = startAnchor.rowOffset;
                    break;
            }

            // add absolute size, if anchor is two-cell; otherwise add the end cell address
            switch (anchorDesc.anchorMode) {
                case AnchorMode.TWO_CELL:
                    anchorAttrs.width = rectHmm.width;
                    anchorAttrs.height = rectHmm.height;
                    break;
                case AnchorMode.ONE_CELL:
                case AnchorMode.ABSOLUTE:
                    var endAnchor = anchorDesc.endAnchor;
                    anchorAttrs.endCol = endAnchor.address[0];
                    anchorAttrs.endColOffset = endAnchor.colOffset;
                    anchorAttrs.endRow = endAnchor.address[1];
                    anchorAttrs.endRowOffset = endAnchor.rowOffset;
                    break;
            }

            return anchorAttrs;
        };

        /**
         * Returns the sheet rectangle in pixels covered by the passed drawing
         * object.
         *
         * @param {DrawingModel} drawingModel
         *  The drawing object to return the rectangle for.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.absolute=false]
         *      If set to true, this method will return the absolute sheet
         *      position also for drawing objects embedded into other drawing
         *      objects. By default, this method returns the location of the
         *      specified drawing object relative to its parent object.
         *
         * @returns {Rectangle|Null}
         *  The location of the specified drawing object in pixels according to
         *  the current sheet zoom factor, either relative to its parent object
         *  (sheet area for top-level drawing objects), or always absolute in
         *  the sheet area (according to the passed options); or null, if the
         *  drawing object is located entirely in hidden columns or rows and
         *  thus has no meaningful location.
         */
        this.getRectangleForModel = function (drawingModel, options) {

            // the top-level drawing model containing the passed drawing model
            var rootModel = drawingModel.getRootModel();
            // the complete anchor descriptor of the top-level drawing
            var anchorDesc = getAnchorDescriptor(rootModel.getMergedAttributeSet(true).drawing);
            // return null for all drawing objects (top-level and embedded) in hidden columns/rows
            if (anchorDesc.hiddenCols || anchorDesc.hiddenRows) { return null; }

            // the resulting rectangle in pixels
            var rectangle = null;
            // the direct parent of the passed drawing model (null for top-level drawings)
            var parentModel = drawingModel.getParentModel();
            // whether to convert to effective absolute coordinates
            var absolute = Utils.getBooleanOption(options, 'absolute', false);

            if (parentModel) {

                // the exact location of the drawing model in 1/100 mm
                var childRectHmm = absolute ? drawingModel.getEffectiveRectangleHmm() : drawingModel.getRectangleHmm();
                // the exact location of the parant drawing model in 1/100 mm
                var parentRectHmm = absolute ? anchorDesc.rectHmm.rotatedBoundingBox(rootModel.getRotationRad()) : parentModel.getRectangleHmm();
                // the exact location of the parant drawing model in pixels
                var parentRectPx = absolute ? anchorDesc.rectPx.rotatedBoundingBox(rootModel.getRotationRad()) : this.getRectangleForModel(parentModel);

                // conversion factor for left position and width
                var widthRatio = (parentRectHmm.width > 0) ? (parentRectPx.width / parentRectHmm.width) : 0;
                // conversion factor for top position and height
                var heightRatio = (parentRectHmm.height > 0) ? (parentRectPx.height / parentRectHmm.height) : 0;

                // the location of the drawing object, relative to its parent object
                rectangle = childRectHmm.clone().scaleSelf(widthRatio, heightRatio).roundSelf();
            } else {
                // position of top-level drawings is contained in the anchor descriptor
                rectangle = absolute ? anchorDesc.rectPx.rotatedBoundingBox(rootModel.getRotationRad()) : anchorDesc.rectPx;
            }

            // keep minimum size of 5 pixels for shapes, or 1 pixel for connectors
            var minSize = drawingModel.getMinSize();
            if (rectangle.width < minSize) {
                rectangle.left = Math.max(0, rectangle.left - Math.round((minSize - rectangle.width) / 2));
                rectangle.width = minSize;
            }
            if (rectangle.height < minSize) {
                rectangle.top = Math.max(0, rectangle.top - Math.round((minSize - rectangle.height) / 2));
                rectangle.height = minSize;
            }

            return rectangle;
        };

        // anchor attribute generators ----------------------------------------

        /**
         * Returns the anchor attributes needed to set the position and size of
         * a drawing object to the passed sheet rectangle.
         *
         * @param {AnchorMode} anchorMode
         *  The display anchor mode to be converted to an appropriate value of
         *  the "anchorType" attribute.
         *
         * @param {Rectangle} rectangle
         *  The location in the sheet in pixels, according to the current sheet
         *  zoom factor.
         *
         * @returns {Object}
         *  The anchor attributes for the passed anchor mode and rectangle.
         */
        this.getAnchorAttributesForRect = function (anchorMode, rectangle) {
            var anchorAttrs = calculateAnchorForRectangle(rectangle);
            anchorAttrs.anchorType = getAnchorType(anchorMode);
            return anchorAttrs;
        };

        /**
         * Calculates the current effective cell anchor attributes of the
         * specified top-level drawing object.
         *
         * @param {DrawingModel} drawingModel
         *  The model of a drawing object.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {AnchorMode} [options.anchorMode]
         *      If specified, this method will generate the anchor attributes
         *      needed to change the display anchor mode to the passed value.
         *      By default, the returned attributes will contain the current
         *      anchor type attribute of the passed drawing model.
         *  - {Rectangle} [options.rectangle]
         *      If specified, this method will generate the anchor attributes
         *      needed to move the drawing object to the specified absolute
         *      location in the sheet area, in pixels. By default, the anchor
         *      attributes of the current location of the drawing model will be
         *      recalculated.
         *
         * @returns {Object|Null}
         *  The anchor attributes that need to be applied at the drawing object
         *  to update its location; or null, if the drawing object does not
         *  need to be updated.
         */
        this.generateAnchorAttributes = function (drawingModel, options) {

            // the complete anchor attributes of the drawing object (either for target rectangle, or current location)
            var rectangle = options && options.rectangle;
            var anchorAttrs = rectangle ? calculateAnchorForRectangle(rectangle) : drawingModel.getAnchorAttributes();

            // change the anchor mode if specified
            var anchorMode = options && options.anchorMode;
            if (anchorMode) { anchorAttrs.anchorType = getAnchorType(anchorMode); }

            // add current anchor type if still missing
            if (!anchorAttrs.anchorType) {
                anchorAttrs.anchorType = drawingModel.getMergedAttributeSet(true).drawing.anchorType;
            }

            // reduce to the anchor attributes really needed to update the anchor
            return finalizeAnchorAttributes(drawingModel, anchorAttrs);
        };

        /**
         * Calculates the cell anchor attributes needed to update the location
         * of the passed top-level drawing object, after cells have been moved
         * in the document.
         *
         * @param {DrawingModel} drawingModel
         *  The model of the drawing object to be updated.
         *
         * @param {Array<MoveDescriptor>} moveDescs
         *  An array of move descriptors that specify how to transform the
         *  location of the drawing object.
         *
         * @returns {String|Object|Null}
         *  The string value "delete" to indicate that the drawing object will
         *  be deleted from the sheet by the passed move descriptors; ot the
         *  anchor attributes that need to be applied at the drawing object to
         *  update its location; or null, if the drawing object does not need
         *  to be updated.
         */
        this.generateAnchorAttributesForMove = function (drawingModel, moveDescs) {

            // the cell range covered by the drawing object
            var drawingRange = drawingModel.getRange();
            // copy of the current cell anchor attributes, will be recalculated below
            var anchorAttrs = drawingModel.getAnchorAttributes();
            // the effective anchor mode that influences how the drawing object will move
            var anchorMode = getAnchorMode(anchorAttrs, true);
            // whether the position of the drawing object will be changed (one-cell and two-cell mode)
            var moveAnchor = anchorMode !== AnchorMode.ABSOLUTE;
             // whether the size of the drawing object will be changed (two-cell mode only)
            var resizeAnchor = anchorMode === AnchorMode.TWO_CELL;

            // process all move descriptors, returning true indicates to delete the drawing
            // (absolutely positioned drawing objects will not change at all, but inactive
            // anchor attributes still need to be updated at the end of this method)
            var deleteDrawing = moveAnchor && moveDescs.some(function (moveDesc) {

                // nothing to do, if the move descriptor does not cover the drawing object completely
                if (!moveDesc.containsRange(drawingRange)) { return false; }

                // whether to modify the column or row indexes of the range
                var columns = moveDesc.columns;
                // the attribute names according to the passed direction
                var ATTR_NAMES = columns ? COL_ATTRIBUTE_NAMES : ROW_ATTRIBUTE_NAMES;
                // the maximum column/row
                var maxIndex = docModel.getMaxIndex(columns);

                // calculate new column/row indexes; if the drawing object has one-cell or absolute anchor mode,
                // these attributes will be overwritten when recalculating the inactive anchor attributes
                if (moveDesc.insert) {

                    // insertion mode: process the target intervals from first to last
                    moveDesc.targetIntervals.every(function (interval) {

                        // the number of columns/rows to be inserted
                        var moveSize = interval.size();
                        // the column/row interval covered by the drawing model
                        var startIndex = drawingRange.getStart(columns);
                        var endIndex = drawingRange.getEnd(columns);
                        // the maximum number of columns/rows the drawing can be moved
                        var maxMoveSize = Math.min(moveSize, maxIndex - endIndex);

                        // do nothing, if drawing cannot be moved anymore
                        if (maxMoveSize === 0) { return false; }

                        // move end position, if the columns/rows are inserted inside the drawing object
                        if (interval.first <= endIndex) {
                            anchorAttrs[ATTR_NAMES.endIndex] += maxMoveSize;
                            drawingRange.end.move(maxMoveSize, columns);
                            // move completely, if the columns/rows are inserted before the drawing object
                            if (interval.first <= startIndex) {
                                anchorAttrs[ATTR_NAMES.startIndex] += maxMoveSize;
                                drawingRange.start.move(maxMoveSize, columns);
                            }
                        }

                        // continue to process the target intervals
                        return true;
                    });

                    // do not delete drawings when inserting cells
                    return false;
                }

                // deletion mode: process the target intervals in reversed order
                return moveDesc.targetIntervals.someReverse(function (interval) {

                    // the number of columns/rows to be inserted
                    var moveSize = interval.size();
                    // the column/row interval covered by the drawing model
                    var startIndex = drawingRange.getStart(columns);
                    var endIndex = drawingRange.getEnd(columns);

                    // adjust end position of drawings overlapping or following the deleted interval
                    if (resizeAnchor) {

                        // delete drawing object, if it is contained in the deleted column/row interval,
                        // or if it ends at offset zero in the first column/row after the interval
                        if ((interval.first <= startIndex) && (endIndex <= interval.last)) {
                            return true;
                        }

                        if (interval.last < endIndex) {
                            // move end position, if the columns/rows are deleted before the end
                            anchorAttrs[ATTR_NAMES.endIndex] -= moveSize;
                            drawingRange.end.move(-moveSize, columns);
                        } else if (interval.first <= endIndex) {
                            // delete drawing if it shrinks towards the left border of the sheet
                            if (interval.first === 0) { return true; }
                            // cut end of drawing object at the deleted interval
                            anchorAttrs[ATTR_NAMES.endIndex] = interval.first;
                            anchorAttrs[ATTR_NAMES.endOffset] = 0;
                            drawingRange.setEnd(interval.first - 1, columns);
                        }
                    }

                    // adjust start position of drawings overlapping or following the deleted interval
                    if (interval.last < startIndex) {
                        // move start position, if the columns/rows are deleted before the start
                        anchorAttrs[ATTR_NAMES.startIndex] -= moveSize;
                        drawingRange.start.move(-moveSize, columns);
                    } else if (interval.first <= startIndex) {
                        // cut start of drawing object at the deleted interval
                        anchorAttrs[ATTR_NAMES.startIndex] = interval.first;
                        anchorAttrs[ATTR_NAMES.startOffset] = 0;
                        drawingRange.setStart(interval.first, columns);
                    }
                });
            });

            // return a string to indicate to delete the drawing object completely, or insert the inactive
            // anchor attributes according to anchor type (this may return to the old cell anchor)
            return deleteDrawing ? 'delete' : finalizeAnchorAttributes(drawingModel, anchorAttrs);
        };

        // operation generators -----------------------------------------------

        /**
         * Generates the operations and undo operations to update and restore
         * the inactive anchor attributes of the top-level drawing objects,
         * after the size or visibility of the columns or rows in the sheet has
         * been changed.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateRefreshAnchorOperations = SheetUtils.profileAsyncMethod('SheetDrawingCollectionMixin.generateRefreshAnchorOperations()', function (generator) {
            return this.iterateSliced(this.createModelIterator(), function (drawingModel) {
                drawingModel.generateRefreshAnchorOperations(generator);
            }, 'SheetDrawingCollectionMixin.generateRefreshAnchorOperations');
        });

        /**
         * Generates the undo operations to restore drawing objects that will
         * be deleted implicitly by moving cells (e.g. deleting columns or
         * rows) in the sheet.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {MoveDescriptor} moveDescs
         *  An array of move descriptors that specify how the cells in the
         *  sheet will be moved.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateBeforeMoveCellsOperations = SheetUtils.profileAsyncMethod('SheetDrawingCollectionMixin.generateBeforeMoveCellsOperations()', function (generator, moveDescs) {

            // iterate in reversed order to have stable drawing positions when deleting drawings
            var iterator = this.createModelIterator({ reverse: true });
            return this.iterateSliced(iterator, function (drawingModel) {

                // a local generator to be able to prepend all undo operations at once (for deleted drawings)
                var generator2 = sheetModel.createOperationGenerator();

                // generate the operations, and insert them into the passed generator
                drawingModel.generateBeforeMoveCellsOperations(generator2, moveDescs);
                generator.appendOperations(generator2);
                generator.prependOperations(generator2, { undo: true });

            }, 'SheetDrawingCollectionMixin.generateBeforeMoveCellsOperations');
        });

        /**
         * Generates the operations and undo operations to update and restore
         * the formula expressions of the source links in all drawing objects
         * that refer to some cells in this document.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Object} updateDesc
         *  The properties describing the document change. The properties that
         *  are expected in this descriptor depend on its 'type' property. See
         *  method TokenArray.resolveOperation() for more details.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateUpdateFormulaOperations = SheetUtils.profileAsyncMethod('SheetDrawingCollectionMixin.generateUpdateFormulaOperations()', function (generator, updateDesc) {
            // bug 58153: do not iterate deeply into the drawing objects, they will update their children by themselves
            return this.iterateSliced(this.createModelIterator(), function (drawingModel) {
                drawingModel.generateUpdateFormulaOperations(generator, updateDesc);
            }, 'SheetDrawingCollectionMixin.generateUpdateFormulaOperations');
        });

        // initialization -----------------------------------------------------

        // create the DOM drawing frame for a new drawing model
        this.on('insert:drawing', function (event, drawingModel) {
            frameNodeManager.createDrawingFrame(drawingModel, { internal: true });
        });

        // remove the DOM drawing frame of a deleted drawing model
        this.on('delete:drawing', function (event, drawingModel) {
            frameNodeManager.removeDrawingFrame(drawingModel);
            modelFrameCache.remove(drawingModel.getUid());
        });

        // move the DOM drawing frame of a drawing model to its new position (Z order)
        this.on('move:drawing', function (event, drawingModel) {
            frameNodeManager.moveDrawingFrame(drawingModel);
        });

        // remove cached model frame settings if formatting has changed
        this.on('change:drawing', function (event, drawingModel, changeType) {
            if (changeType === 'attributes') {
                modelFrameCache.remove(drawingModel.getUid());
            }
        });

        // remove cached model frame settings if text contents have changed
        this.on('change:drawing:text', function (event, drawingModel) {
            modelFrameCache.remove(drawingModel.getUid());
        });

        // destroy all class members
        this.registerDestructor(function () {
            frameNodeManager.destroy();
            modelFrameCache.clear();
            self = app = docModel = frameNodeManager = modelFrameCache = null;
            sheetModel = colCollection = rowCollection = null;
        });

    } // mix-in class SheetDrawingCollectionMixin

    // exports ================================================================

    return SheetDrawingCollectionMixin;

});
