/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/chart/chartformatter', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/tk/locale/parser',
    'io.ox/office/spreadsheet/view/labels'
], function (Utils, BaseObject, Parser, Labels) {

    'use strict';

    //'io.ox/office/spreadsheet/model/numberformatter'
    // null date (corresponding to cell value zero) TODO: use null date from document settings

    // class ChartFormatter ===================================================

    function ChartFormatter(docModel, chartModel) {

        var fileFormat = docModel.getApp().getFileFormat();

        // base constructor ---------------------------------------------------

        BaseObject.call(this, docModel);

        // private methods ----------------------------------------------------

        function isAnyDateTime(format, value) {
            if (format && format.isAnyDateTime() && Utils.isFiniteNumber(value)) {
                return true;
            }
        }

        function newInfoHolder(type, label) {
            return {
                max: Number.NEGATIVE_INFINITY,
                date: false,
                type: type,
                label: label,
                unclean: false,
                exp: 0
            };
        }

        function handleData(info, source, dataPoint, format, nullAsZero) {
            var realNr = source.value;
            var display = source.display;

            if (!format && source.format) {
                format = source.format;
                if (isAnyDateTime(format, realNr)) {
                    info.date = format;
                }
                // Store the first format for the axis values
                if (!info.format) {
                    info.format = format;
                }
            }

            if (!Utils.isFiniteNumber(realNr) && info.type === 'y' && nullAsZero) {
                realNr = 0;
            }

            var label = null;
            var res = null;
            if (info.date) {
                res = realNr;
                info.max = Math.max(info.max, Math.abs(realNr));
                label = display;
            } else if (!Utils.isFiniteNumber(realNr)) {
                info.unclean = true;
                res = nullAsZero ? 0 : undefined;
                label = display;
            } else if (format) {
                // Do not use info.format
                info.max = Math.max(info.max, Math.abs(realNr));
                res = realNr;
                label = docModel.getNumberFormatter().formatValue(format, realNr);
            } else {
                info.max = Math.max(info.max, Math.abs(realNr));
                res = realNr;
                label = docModel.getNumberFormatter().formatStandardNumber(realNr, 12);
            }

            dataPoint[info.label] = label;
            dataPoint[info.type] = res;
        }

        function calcMaxExp(info) {
            if (Utils.isFiniteNumber(info.max) && !Utils.isFiniteNumber(info.exp)) {
                var norm = Utils.normalizeNumber(info.max);
                if (Math.abs(norm.exp) < 6) {
                    info.exp = 0;
                } else {
                    info.exp = norm.exp;
                }
            } else {
                info.exp = 0;
            }
        }

        function handleMantissa(info, dataPoint) {
            if (info.exp) {
                dataPoint[info.type] = Utils.mant(dataPoint[info.type], info.exp);
            }
        }

        function handleLabelLength(info, dataPoint) {
            var label = dataPoint[info.label];
            if (label && label.length > 30) {
                dataPoint[info.label] = label.substring(0, 27) + '...';
            }
        }

        /**
         * Set the dataseries axis attributes for the canvas data object, if the
         * axes contains the axis for the DataSeries.
         * @param {Array} axes
         * @param {String} axisType the type of the axis 'X' or 'Y'
         * @param {Object} dataSeries the DataSeries to the the axis attributes
         * @returns {Boolean} If the Attributes are set for the DataSeries.
         */
        function setDataSeriesAxis(axes, axisType, dataSeries) {
            var isSet = false;
            axisType = axisType.toUpperCase();
            var axPos = axisType === 'X' ? 'b' : 'l';
            var axis;
            for (var i = 0; i < axes.length; i++) {
                axis = axes[i];
                if (axis.axisId === dataSeries['axis' + axisType + 'Id']) {
                    if (i > 0) {
                        dataSeries['axis' + axisType + 'Index'] = i;
                    }
                    dataSeries['axis' + axisType + 'Type'] = axis.axPos === axPos ? 'primary' : 'secondary';
                    break;
                }
            }

            return isSet;
        }

        // public methods -----------------------------------------------------

        this.format = function () {
            var data = chartModel.getModelData();

            var form = null;
            var xyChart = false;
            var bubbleChart = false;
            var nullAsZero = false;
            var attrs = chartModel.getMergedAttributeSet(true);
            var stacking = attrs.chart.stacking;
            data.linearX = false;

            if (data.series.length) {
                form = data.series[0].model.getExplicitAttributeSet(true).series.format;
                if (form) {
                    form = this.parseFormatCode(form);
                }
                bubbleChart = chartModel.isBubbleChart();
                xyChart = chartModel.isXYType();
                nullAsZero = bubbleChart || chartModel.isPieOrDonut() || /^(column|bar)/.test(chartModel.getChartType());
            }

            _.each(data.series, function (dataSeries, seriesIndex) {
                dataSeries.seriesIndex = seriesIndex;

                var before = data.series[seriesIndex - 1];

                if (before && before.axisXType === dataSeries.axisXType) {
                    dataSeries.xInfo = before.xInfo;
                } else {
                    dataSeries.xInfo = newInfoHolder('x', 'label');
                }
                if (before && before.axisYType === dataSeries.axisYType) {
                    dataSeries.yInfo = before.yInfo;
                } else {
                    dataSeries.yInfo = newInfoHolder('y', 'name');
                }

                var x = dataSeries.xInfo;
                var y = dataSeries.yInfo;

                if (dataSeries.nameHolder) {
                    dataSeries.name = dataSeries.nameHolder.display;
                } else if (docModel.getApp().isODF()) {
                    var fallback = true;
                    try {
                        var valueRanges = dataSeries.model.resolveRanges('series.bubbles');
                        if (!valueRanges || !valueRanges.length) {
                            valueRanges = dataSeries.model.resolveRanges('series.values');
                        }
                        if (valueRanges && valueRanges.length) {
                            var valueRange = valueRanges[0];
                            if (Math.abs(valueRange.start[0] - valueRange.end[0]) > 0) {
                                dataSeries.name = Labels.getRowLabel(valueRange.start[1]);
                                fallback = false;
                            } else if (Math.abs(valueRange.start[1] - valueRange.end[1]) > 0) {
                                dataSeries.name = Labels.getColLabel(valueRange.start[0]);
                                fallback = false;
                            } else if (dataSeries.dps.length) {
                                dataSeries.name = dataSeries.dps[0].name;
                                fallback = false;
                            }
                        }
                    } catch (e) {
                        Utils.warn('chart error while formatting', e);
                    }

                    if (fallback) {
                        dataSeries.name = Labels.getSeriesLabel(seriesIndex);
                    }
                } else {
                    dataSeries.name = Labels.getSeriesLabel(seriesIndex);
                }

                var dataPoints = dataSeries.dps;

                var onlyNumberNamedSource = false;

                if (xyChart) {
                    onlyNumberNamedSource = !(_.find(dataPoints, function (dataPoint) {
                        return !dataPoint.nameSource || dataPoint.nameSource.value === null || !Utils.isFiniteNumber(dataPoint.nameSource.value);
                    }));
                }

                var n = 0;
                _.each(dataPoints, function (dataPoint) {
                    dataPoint.x = undefined;
                    dataPoint.y = undefined;
                    dataPoint.z = undefined;

                    if (dataPoint.sizeSource) {
                        dataPoint.z = Math.abs(dataPoint.sizeSource.value);

                        //fix for funny bug in canvasjs
                        //Don't know what this funny bug is or when it occur. But it overrides the markerBorderColor of the bubbles in the chartModel
//                        dataPoint.markerBorderColor = 'transparent';
//                        dataPoint.markerBorderThickness = 1;

                    } else {
                        dataPoint.z = bubbleChart ? 1 : undefined;
                    }

                    if (dataPoint.visible && (!bubbleChart || !isNaN(dataPoint.z))) {
                        if (dataPoint.valueSource === null) {
                            // fix for charts without values e.g. bubblechart with one column
                            dataPoint[y.type] = n + 1;
                        } else {
                            handleData(y, dataPoint.valueSource, dataPoint, form, nullAsZero);
                        }

                        if (xyChart && dataPoint.nameSource && onlyNumberNamedSource) {
                            handleData(x, dataPoint.nameSource, dataPoint, null, nullAsZero);
                        } else {
                            dataPoint[x.type] = n + 1;
                            data.linearX = true;
                        }

                        //ugly part against double formatting!!!
                        if (dataPoint.nameSource) {
                            dataPoint[x.label] = dataPoint.nameSource.display;
                        } else {
                            dataPoint[x.label] = String(dataPoint[x.type]);
                        }

                        if (dataPoint.label) { dataPoint.legendLabel = dataPoint.label; }

                        handleLabelLength(x, dataPoint);
                        handleLabelLength(y, dataPoint);

                        dataPoint.legendText = dataPoint.legendLabel;

                        if (dataPoint.x === null) {
                            dataPoint.x = undefined;
                        }
                        if (dataPoint.y === null || isNaN(dataPoint.y)) {
                            dataPoint.y = undefined;
                        }
                        if (dataPoint.z === null) {
                            dataPoint.z = undefined;
                        }
                        n++;
                    }
                });

                if (xyChart && x.unclean) {
                    _.each(dataPoints, function (dataPoint, n) {
                        dataPoint.x = n + 1;
                    });
                }

                if (xyChart) {
                    calcMaxExp(x);
                }
                calcMaxExp(y);
                var xAxisType = chartModel.getAxisTypeForDrawing('x');
                var yAxisType = chartModel.getAxisTypeForDrawing('y');
                if (!setDataSeriesAxis(data.axisX, xAxisType, dataSeries)) {
                    setDataSeriesAxis(data.axisX2, xAxisType, dataSeries);
                }
                if (!setDataSeriesAxis(data.axisY, yAxisType, dataSeries)) {
                    setDataSeriesAxis(data.axisY2, yAxisType, dataSeries);
                }

                var xAxisModel = chartModel.getAxisModel(dataSeries.axisXId);
                if (xAxisModel) {
                    xAxisModel.setFormat(xAxisType === 'x' ? x : y);
                }
                var yAxisModel = chartModel.getAxisModel(dataSeries.axisYId);
                if (yAxisModel) {
                    yAxisModel.setFormat(yAxisType === 'y' ? y : x);
                }
            });

            var chartType = chartModel.getChartType();
            var mustReverseDataPoints = false;
            if (docModel.getApp().isODF()) {
                mustReverseDataPoints = /^(pie|donut)/.test(chartType);
            }

            _.each(data.series, function (dataSeries) {
                var x = dataSeries.xInfo;
                var y = dataSeries.yInfo;
                _.each(dataSeries.dps, function (dataPoint) {
                    if (!xyChart && (stacking === 'clustered' || stacking === 'standard')) {
                        var thresholdY = 0.005;
                        //workaround that too small values stay visible
                        var rel = dataPoint.y / y.max;
                        if (Math.abs(rel) < thresholdY) {
                            if (rel < 0) {
                                dataPoint.y = -thresholdY * y.max;
                            } else {
                                dataPoint.y = thresholdY * y.max;
                            }
                        }
                    }

                    handleMantissa(y, dataPoint);
                    handleMantissa(x, dataPoint);
                });

                dataSeries.dataPoints = dataSeries.dps.slice(0);
                if (dataSeries.type === 'bubble') {
                    _.each(dataSeries.dataPoints, function (dataPoint) {
                        if (isNaN(dataPoint.z)) {
                            delete dataPoint.x;
                            delete dataPoint.y;
                            delete dataPoint.z;
                        }
                    });
                }
                if (mustReverseDataPoints) {
                    dataSeries.dataPoints.reverse();
                }
            });

            data.data = data.series.slice(0);
            var mustReverse = false;
            chartModel.getModelData().legend.reversed = false;

            if (stacking === 'clustered' || stacking === 'standard') {
                if (docModel.getApp().isOOXML()) {
                    mustReverse = /^(bar)/.test(chartType);
                } else {
                    chartModel.getModelData().legend.reversed = mustReverseDataPoints || /^(bar)/.test(chartModel.getChartType());
                    mustReverse = /^(bar|area)/.test(chartType);
                }
            }
            if (mustReverse) {
                data.data.reverse();
            }
        };

        this.update = function (seriesIndex, values, titles, names, bubbles) {
            var data = chartModel.getModelData();

            var dataSeries = data.series[seriesIndex];
            var isBubble = dataSeries.type.indexOf('bubble') >= 0;
            if (!dataSeries) {
                //workaround for Bug 46928
                Utils.warn('no dataSeries found for ' + seriesIndex + ' title: ' + titles + ' names: ' + names);
                return;
            }

            // build series title
            dataSeries.nameHolder = titles ? titles[0] : null;

            function isValueNotNull(source) {
                return source && source.value !== null;
            }

            function hasValues(sources) {
                var count = isValueNotNull(dataSeries.nameHolder) ? 1 : 0;
                sources.forEach(function (source) {
                    if (isValueNotNull(source)) {
                        count++;
                    }
                });
                return count > 1;
            }

            dataSeries.dps = [];
            _.each(values, function (valueCell, n) {
                var dataPoint = {};
                var nameSource = names[n];
                var sizeSource = bubbles[n];
                dataSeries.dps.push(dataPoint);
                if (!isBubble || (data.containsSeriesValues && isValueNotNull(valueCell) && (isValueNotNull(sizeSource) || sizeSource === undefined)) || hasValues([valueCell, nameSource, sizeSource])) {

                    dataPoint.nameSource = nameSource;    //x
                    dataPoint.valueSource = valueCell;  //y
                    dataPoint.sizeSource = sizeSource;  //z
                    dataPoint.color = null;
                    dataPoint.markerColor = null;
                    data.containsSeriesValues = true;
                    dataPoint.visible = true;
                } else {
                    dataPoint.nameSource = null;    //x
                    dataPoint.valueSource = null;  //y
                    dataPoint.sizeSource = null;  //z
                    dataPoint.color = null;
                    dataPoint.markerColor = null;
                    dataPoint.visible = false;
                }
            });
        };

        this.parseFormatCode = function (formatCode) {
            return Parser.parseFormatCode(fileFormat, 'op', formatCode);
        };

        // initialization -----------------------------------------------------

        this.registerDestructor(function () {
            docModel = chartModel = null;
        });

    } // class ChartFormatter

    // exports ================================================================

    return BaseObject.extend({ constructor: ChartFormatter });

});
