/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/chart/axismodel', [
    'io.ox/office/tk/utils',
    'io.ox/office/drawinglayer/view/chartstyleutil',
    'io.ox/office/editframework/model/attributedmodel',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/model/drawing/chart/titlemodel',
    'io.ox/office/spreadsheet/model/drawing/chart/gridlinemodel'
], function (Utils, ChartStyleUtil, AttributedModel, Operations, TitleModel, GridlineModel) {

    'use strict';

    // class AxisModel ========================================================

    /**
     * Representation of a simgle axis in a chart drawing object.
     *
     * @constructor
     *
     * @extends AttributedModel
     */
    var AxisModel = AttributedModel.extend({ constructor: function (chartModel, axisType, axisId, axPos, crossAx) {

        var self = this;
        var titleModel = null;
        var gridModel = null;
        var format = null;

        var indexName = 'axis' + chartModel.getAxisTypeForDrawing(axisType[0]).toUpperCase() + 'Id';
        var dataAxis;
        if (axisType === 'z') {
            dataAxis = { axisId: axisId, axPos: axPos, crossAx: crossAx };
        } else {
            dataAxis = { labelAutoFit: true, labelAngle: 0, axisId: axisId, axPos: axPos, crossAx: crossAx };
        }

        chartModel.getModelData()['axis' + axisType.toUpperCase()].push(dataAxis);

        if (dataAxis) {
            titleModel = new TitleModel(chartModel, axisId, dataAxis, 'title', 'title');
            gridModel = new GridlineModel(chartModel, axisId, {}, dataAxis);

            dataAxis.labelFormatter = function (info) {
                if (chartModel.isXYType()) {
                    return formatNumber(info.value);
                } else {
                    return info.label || formatNumber(info.value);
                }
            };
        }

        // base constructor ---------------------------------------------------

        AttributedModel.call(this, chartModel.getDocModel(), null, { families: 'axis line character' });

        // private methods ----------------------------------------------------

        function formatNumber(value) {
            if (!chartModel) {
                Utils.warn('no chartmodel in AxisModel.formatNumber()');
                return;
            }
            var formatter = chartModel.getDocModel().getNumberFormatter();
            var realValue = value * (format && format.exp ? Math.pow(10, format.exp) : 1);
            var formattedNumber;
            if (format && format.format) {
                formattedNumber = formatter.formatValue(format.format, realValue);
            } else {
                formattedNumber = formatter.formatStandardNumber(realValue, 8);
            }
            return formattedNumber === null ? '' : formattedNumber;
        }

        function isHorizontal() {
            return chartModel.getAxisTypeForDrawing(axisType[0]) === 'x';
        }

        function refreshMinMax() {
            var data = chartModel.getModelData();
            if (!data.series.length) {
                return;
            }

            if (axisType[0] === 'x') {
                refreshAxisX(data);
            } else if (axisType[0] === 'y') {
                refreshAxisY(data);
            }
        }

        function makeMin(min, max) {
            if ((!format || !format.date) && (max === min || (min / max < 5 / 6))) {
                return 0;
            }
            return min - ((max - min) / 2);
        }

        function makeMax(min, max) {
            var res = max + 0.2 * (max - min);
            if (res < max) {
                res = max;
            }
            return res;
        }

        function distance(a, b) {
            return Math.abs(a - b);
        }

        function updateInterval(number) {
            var normalized = Utils.normalizeNumber(number);

            var distances = [];
            distances.push({ value: 1, dist: distance(normalized.mant, 1) });
            distances.push({ value: 2, dist: distance(normalized.mant, 2) });
            distances.push({ value: 5, dist: distance(normalized.mant, 5) });
            distances.push({ value: 10, dist: distance(normalized.mant, 10) });

            var newMant = null;
            var minDist = 999999;

            distances.forEach(function (d) {
                if (d.dist < minDist) {
                    minDist = d.dist;
                    newMant = d.value;
                }
            });

            return newMant * Math.pow(10, normalized.exp);
        }

        function updateMinMax(minValue, maxValue, precision, intervalCount) {
            if (minValue < maxValue) {
                var interval = (maxValue - minValue) / intervalCount;
                interval = round(interval, precision - 1);
                interval = updateInterval(interval);

                if (interval === 0) {
                    interval = null;
                    dataAxis.minimum = minValue;
                    dataAxis.maximum = maxValue;
                    dataAxis.interval = null;
                }
                // removed for Bug 52880
                setMinMax(minValue, maxValue, interval);
            }
        }

        function setMinMax(min, max, interval) {
            if (!interval || !format || !isFinite(min) || !isFinite(max)) {
                delete dataAxis.minimum;
                delete dataAxis.maximum;
                delete dataAxis.interval;
                return;
            }

            if (min < 0 && format && format.format && format.format.category === 'date') { min = 0; }

            dataAxis.minimum = min;
            dataAxis.maximum = max;
            dataAxis.interval = interval;

            if (chartModel.containsCurvedSeries()) {
                dataAxis.minimum -= dataAxis.interval * 2;
            }
            if (!isHorizontal()) {
                // workaround for Bug 49637
                dataAxis.minimum -= dataAxis.interval * 0.001;
                dataAxis.maximum += dataAxis.interval * 0.001;
            }
            if (chartModel.isXYType()) {
                // workaround for Bug 53190
                dataAxis.minimum -= dataAxis.interval * 0.1;
                dataAxis.maximum += dataAxis.interval * 0.1;
            }

        }

        function isBubbleChart() {
            return chartModel.getChartType().indexOf('bubble') === 0;
        }

        function isAreaChart() {
            return chartModel.getChartType().indexOf('area') === 0;
        }

        function refreshAxisX(data) {

            if (!data.linearX && (chartModel.isXYType() || isAreaChart())) {
                var minValue = Number.POSITIVE_INFINITY;
                var maxValue = Number.NEGATIVE_INFINITY;

                _.each(data.series, function (dd) {
                    if (dd[indexName] !== axisId) { return; }

                    _.each(dd.dps, function (dataPoint) {
                        var value = dataPoint[axisType[0]];
                        minValue = Math.min(minValue, value);
                        maxValue = Math.max(maxValue, value);
                    });
                });

                if (isBubbleChart()) {
                    var diff = maxValue - minValue;
                    maxValue +=  diff / 8;
                    minValue -=  diff / 8;
                }

                if (!isFinite(minValue) || !isFinite(maxValue)) {
                    setMinMax();
                    return;
                }
                if (minValue === maxValue) {
                    maxValue = minValue + 0.5;
                    minValue -= 0.5;
                }

                updateMinMax(minValue, maxValue, 2, 4);
            } else {
                setMinMax(null, null, null);
            }
        }

        function round(value, precision) {
            if ((precision > 0) && (precision < 20)) {
                return parseFloat(value.toPrecision(precision));
            }
            return value;
        }

        function refreshAxisY(data) {
            var chartAttrs = chartModel.getMergedAttributeSet(true).chart;
            var stacking = chartAttrs.stacking;

            var givenMin = null;
            var givenMax = null;

            var axisInfo = self.getMergedAttributeSet(true);
            if (axisInfo) {
                if (axisInfo.min !== 'auto') {
                    givenMin = axisInfo.min;
                }
                if (axisInfo.max !== 'auto') {
                    givenMax = axisInfo.max;
                }
            }

            var minValue = Number.POSITIVE_INFINITY;
            var maxValue = Number.NEGATIVE_INFINITY;

            if (stacking === 'percentStacked' || (stacking === 'stacked' && data.series.length > 1)) {
                var first = data.series[0];
                _.times(first.dps.length, function (j) {
                    var posValue = 0;
                    var negValue = 0;
                    _.each(data.series, function (dd) {
                        if (dd[indexName] !== axisId) { return; }

                        var jValue = dd.dps[j];
                        if (!jValue) { return; }
                        var useValue = jValue[axisType[0]];
                        if (useValue > 0) {
                            posValue += useValue;
                        } else if (useValue < 0) {
                            negValue += useValue;
                        }
                    });

                    // fix for Bug 52765
                    if (format && format.date) { negValue = posValue; }

                    minValue = Math.min(minValue, negValue);
                    maxValue = Math.max(maxValue, posValue);
                });
            } else {
                _.each(data.series, function (dd) {
                    if (dd['axis' + axisType.toUpperCase() + 'Id'] !== axisId) { return; }

                    _.each(dd.dps, function (dataPoint) {
                        var value = dataPoint[axisType[0]];
                        minValue = Math.min(minValue, value);
                        maxValue = Math.max(maxValue, value);
                    });
                });
            }

            var precision = 2;
            if (format && format.date) { precision = 0; }

            if (chartModel.isXYType()) {
                precision = 2;
                var diff = (maxValue - minValue) / 4;
                if (isBubbleChart()) { diff *= 1.5; }
                maxValue += diff * 2;
                minValue -= diff;
            } else if (stacking === 'percentStacked') {
                format = {
                    exp: -2,
                    format: chartModel.getDocModel().getNumberFormatter().getParsedFormat('0%', { grammarId: 'ui' })
                };

                if (maxValue >= 0 && minValue >= 0) {
                    // scenario 1
                    minValue = givenMin || 0;
                    maxValue = givenMax || 100;
                } else if (maxValue <= 0 && minValue <= 0) {
                    // scenario 2
                    maxValue = givenMax || 0;
                    minValue = givenMin || -100;
                } else {
                    // scenario 3
                    var maxSize = Math.max(maxValue, -minValue);
                    var tmpMaxValue = 100 * (maxValue / maxSize);
                    var tmpMinValue = -100 * ((-minValue) / maxSize);

                    tmpMinValue = Math.max(-100, Utils.round(tmpMinValue - 5, 10));
                    tmpMaxValue = Math.min(100, Utils.round(tmpMaxValue + 5, 10));

                    minValue = givenMin || tmpMinValue;
                    maxValue = givenMax || tmpMaxValue;
                }
            } else {
                if (maxValue >= 0 && minValue >= 0) {
                    // scenario 1
                    minValue = givenMin || makeMin(minValue, maxValue);
                    maxValue = givenMax || makeMax(minValue, maxValue);
                } else if (maxValue <= 0 && minValue <= 0) {
                    // scenario 2
                    maxValue = givenMax || -makeMin(-maxValue, -minValue);
                    minValue = givenMin || -makeMax(-maxValue, -minValue);
                } else {
                    // scenario 3
                    var ma = maxValue;
                    var mi = minValue;
                    maxValue = givenMax || makeMax(mi, ma);
                    minValue = givenMin || -makeMax(-ma, -mi);
                }
            }

            if (!isFinite(minValue) || !isFinite(maxValue)) {
                setMinMax();
                return;
            }

            if (minValue === maxValue) {
                maxValue = minValue + 0.5;
                minValue -= 0.5;
            }

            if (Math.abs(maxValue - minValue) > 0.00000000001) {
                minValue = Math.round(minValue * 100) / 100;
                maxValue = Math.round(maxValue * 100) / 100;
            }

            updateMinMax(minValue, maxValue, precision, 8);
        }

        // public methods -----------------------------------------------------

        this.refreshInfo = function () {
            refreshMinMax();

            if (!dataAxis) { return; }

            var attrs = this.getMergedAttributeSet(true);

            ChartStyleUtil.handleLineProps(chartModel, attrs.line, dataAxis, 'line');
            ChartStyleUtil.handleLineProps(chartModel, attrs.line, dataAxis, 'tick');

            if (attrs.axis.label) {
                ChartStyleUtil.handleCharacterProps(chartModel, attrs.character, dataAxis, 'label');
            } else {
                dataAxis.labelFontColor = 'transparent';
                dataAxis.labelFontSize = 1;
            }
            dataAxis.labelPlacement = 'outside';

            gridModel.refreshInfo();
            titleModel.refreshInfo();
        };

        this.setFormat = function (newFormat) {
            format = newFormat;
        };

        this.getGrid = function () {
            return gridModel;
        };

        this.getTitle = function () {
            return titleModel;
        };

        this.getAxisType = function () {
            return axisType;
        };

        this.getAxisPos = function () {
            return axPos;
        };

        this.getCrossAxis = function () {
            return crossAx;
        };

        // operation generators -----------------------------------------------

        /**
         * Generates the undo operations needed to restore this chart axis.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the undo operations.
         *
         * @param {Array<Number>} position
         *  The position of the parent chart object in the sheet, as expected
         *  by the method SheetOperationGenerator.generateDrawingOperation().
         *
         * @returns {AxisModel}
         *  A reference to this instance.
         */
        this.generateRestoreOperations = function (generator, position) {

            // restore this axis
            if (this.hasExplicitAttributes()) {
                var properties = { axis: axisId, axPos: axPos, crossAx: crossAx, attrs: this.getExplicitAttributeSet(false) };
                if (axisType === 'z') {
                    properties.zAxis = true;
                }
                generator.generateDrawingOperation(Operations.SET_CHART_AXIS_ATTRIBUTES, position, properties, { undo: true });
            }

            // restore the axis title, and the grid line settings
            if (titleModel) { titleModel.generateRestoreOperations(generator, position); }
            if (gridModel) { gridModel.generateRestoreOperations(generator, position); }

            return this;
        };

        /**
         * Generates the operations and undo operations to update or restore
         * the formula expressions of the source link of the axis title object.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Array<Number>} position
         *  The position of the parent chart object in the sheet, as expected
         *  by the method SheetOperationGenerator.generateDrawingOperation().
         *
         * @param {Object} updateDesc
         *  The properties describing the document change. The properties that
         *  are expected in this descriptor depend on its 'type' property. See
         *  method TokenArray.resolveOperation() for more details.
         *
         * @returns {AxisModel}
         *  A reference to this instance.
         */
        this.generateUpdateFormulaOperations = function (generator, position, updateDesc) {
            if (titleModel) { titleModel.generateUpdateFormulaOperations(generator, position, updateDesc); }
            return this;
        };

        // initialization -----------------------------------------------------

        this.registerDestructor(function () {
            if (dataAxis) { delete dataAxis.labelFormatter; }
            if (gridModel) { gridModel.destroy(); }
            if (titleModel) { titleModel.destroy(); }
            dataAxis = chartModel = gridModel = titleModel = null;
        });

    } }); // class AxisModel

    // exports ================================================================

    return AxisModel;

});
