/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/controller/sheetmixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/view/dialogs',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, SheetUtils, Dialogs, gt) {

    'use strict';

    // convenience shortcuts
    var SheetType = SheetUtils.SheetType;

    // mix-in class SheetMixin ================================================

    /**
     * Implementations of all controller items for manipulating the collection
     * of sheets in the document, intended to be mixed into a document
     * controller instance.
     *
     * @constructor
     *
     * @param {SpreadsheetView} docView
     *  The document view providing view settings such as the active sheet.
     */
    function SheetMixin(docView) {

        // self reference
        var self = this;

        // the document model
        var docModel = docView.getDocModel();
        var dependencyManager = docModel.getDependencyManager();

        // the model and index of the active sheet
        var sheetModel = null;
        var activeSheet = -1;

        // private methods ----------------------------------------------------

        /**
         * Activates the nearest visible sheet next to the specified sheet in
         * the document view.
         *
         * @param {Number} sheet
         *  The zero-based index of the sheet to be activated.
         *
         * @param {String} [method='nearest']
         *  The method how to find a visible sheet:
         *  - 'next': If the specified sheet is hidden, searches for the
         *      nearest following visible sheet. All sheets preceding the
         *      specified sheet will be ignored.
         *  - 'prev': If the specified sheet is hidden, searches for the
         *      nearest preceding visible sheet. All sheets following the
         *      specified sheet will be ignored.
         *  - 'nearest' (default value): If the specified sheet is hidden,
         *      first searches with the method 'next', and if this fails, with
         *      the method 'prev'.
         */
        function activateSheet(sheet, method) {

            // Set the new active sheet at the document model. The event handlers
            // of the 'change:doc:viewattributes' event with do all further
            // processing and initialization for the new active sheet.
            var newSheet = docModel.findVisibleSheet(sheet, method);
            if (newSheet >= 0) { docModel.setActiveSheet(newSheet); }
        }

        /**
         * Inserts a new sheet in the spreadsheet document, and activates it in
         * the view. In case of an error, shows a notification message.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved, if the new sheet has been inserted
         *  successfully, or that will be rejected otherwise.
         */
        function insertSheet() {

            // leave text edit mode before inserting the sheet
            var promise = docView.leaveTextEditMode().then(function () {

                // server-side configuration contains the maximum number of sheets per document
                if (!docModel.canInsertSheet()) {
                    return $.Deferred().reject();
                }

                // insert the new sheet directly after the active sheet
                var newSheet = activeSheet + 1;
                // create a new unused sheet name
                var sheetName = docModel.generateUnusedSheetName();

                // attribute set of the new sheet (bug 57798: resolve and insert default row height)
                var defCharAttrs = docModel.getCellAutoStyles().getDefaultAttributeSet().character;
                var rowHeight = docModel.convertRowHeightToUnit(docModel.getRowHeightHmm(defCharAttrs));
                var newAttrSet = { row: { height: rowHeight } };

                // bug 57798: resolve and insert default column width according to file format
                switch (self.getApp().getFileFormat()) {
                    case 'ooxml':
                        newAttrSet.sheet = { baseColWidth: sheetModel.getMergedAttributeSet(true).sheet.baseColWidth };
                        break;
                    case 'odf':
                        newAttrSet.column = { width: 2258 };
                        break;
                }

                // generate and apply the 'insertSheet' operation
                return docModel.createAndApplySheetOperations(function (generator) {
                    return docModel.generateInsertSheetOperations(generator, newSheet, SheetType.WORKSHEET, sheetName, newAttrSet);
                }, { storeSelection: function () { docModel.setActiveSheet(newSheet); } });
            });

            // show warning alert if necessary
            return docView.yellOnFailure(promise);
        }

        /**
         * Deletes the active sheet from the spreadsheet document, and
         * activates another sheet in the view. Does nothing, if the document
         * contains a single sheet only or if the user doesn't really want
         * to delete the sheet without undo after being asked for.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved, if the active sheet has been
         *  deleted, or that will be rejected otherwise.
         */
        function deleteSheet() {

            // bug 31772: cancel edit mode before deleting the sheet (do not commit something into the deleted sheet)
            docView.cancelTextEditMode();

            // do not delete the last visible sheet
            if (sheetModel.isVisible() && (docModel.getSheetCount({ supported: true, visible: true }) <= 1)) {
                return $.Deferred().reject();
            }

            // ask to proceed (deleting sheets cannot be undone)
            var promise = docView.showQueryDialog(
                gt('Delete Sheet'),
                gt('Deleting a sheet cannot be undone. Proceed operation?')
            );

            // generate and apply the 'deleteSheet' operation after confirmation
            promise = promise.then(function () {
                return docModel.createAndApplySheetOperations(function (generator) {
                    return docModel.generateDeleteSheetOperations(generator, activeSheet);
                }, { storeSelection: true, undoMode: 'clear' });
            });

            // show warning alert if necessary
            return docView.yellOnFailure(promise);
        }

        /**
         * Copies the active sheet and inserts the copy with the provided sheet
         * name behind the active sheet. In case of an error, shows a warning
         * message.
         *
         * @param {String} sheetName
         *  The name of the new sheet.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved, if the sheet has been copied, or
         *  that will be rejected otherwise.
         */
        function copySheet(sheetName) {

            // leave text edit mode before copying the sheet
            var promise = docView.leaveTextEditMode().then(function () {

                // remove all NPCs from passed sheet name
                sheetName = Utils.cleanString(sheetName);
                // insert the new sheet directly after the active sheet
                var newSheet = activeSheet + 1;

                // generate and apply the 'copySheet' operation
                return docModel.createAndApplySheetOperations(function (generator) {
                    return docModel.generateCopySheetOperations(generator, activeSheet, newSheet, sheetName);
                }, { storeSelection: function () { docModel.setActiveSheet(newSheet); } });
            });

            // show warning alert if necessary
            return docView.yellOnFailure(promise);
        }

        /**
         * Shows a dialog that allows to enter a name of a new sheet copied
         * from the active sheet. The dialog will be kept open until a valid
         * sheet name has been entered.
         *
         * @returns {jQuery.Promise}
         *  A promise representing the dialog. Will be resolved with the new
         *  sheet name, after the sheet has been copied successfully; or will
         *  be rejected, if the dialog has been canceled, or if the document
         *  has switched to read-only mode.
         */
        function showCopySheetDialog() {

            // leave text edit mode before copying the sheet
            return docView.leaveTextEditMode().then(function () {

                // create the dialog, initialize with an unused sheet name
                var dialog = new Dialogs.SheetNameDialog(docView, copySheet, {
                    title: gt('Copy Sheet'),
                    value: docModel.generateUnusedSheetName(),
                    okLabel: gt('Copy')
                });

                // show the dialog, copying the sheet will be done in the callback function
                return dialog.show();
            });
        }

        /**
         * Moves the specified sheet in the document to a new position. In
         * difference to most other methods of this class, this method does NOT
         * work on the active sheet, but on an arbitrary sheet in the document.
         *
         * @param {Number} fromSheet
         *  The zero-based index of the sheet to be moved.
         *
         * @param {Number} toSheet
         *  The new zero-based index of the sheet.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved, if the sheet has been moved, or
         *  that will be rejected otherwise.
         */
        function moveSheet(fromSheet, toSheet) {

            // nothing to do, if the sheet does not move
            if (fromSheet === toSheet) { return self.createResolvedPromise(); }

            // leave text edit mode before moving the sheet
            var promise = docView.leaveTextEditMode().then(function () {
                // generate and apply the 'moveSheet' operation
                return docModel.createAndApplySheetOperations(function (generator) {
                    return docModel.generateMoveSheetOperations(generator, fromSheet, toSheet);
                });
            });

            // show warning alert if necessary
            return docView.yellOnFailure(promise);
        }

        /**
         * Shows the 'Reorder Sheets' dialog and creates and applies the
         * respective 'moveSheet' operations.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the dialog has been closed,
         *  and the sheets have been reordered.
         */
        function showMoveSheetsDialog() {

            // leave text tedit mode before moving the sheets
            var promise = docView.leaveTextEditMode();

            // create and show the 'Reorder Sheets' dialog
            promise = promise.then(function () {
                return new Dialogs.SheetOrderDialog(docView).show();
            });

            // generate all 'moveSheet' operations
            promise = promise.then(function (moveActions) {
                return docModel.createAndApplySheetOperations(function (generator) {

                    // generate all move operations, and the undo operations
                    // TODO: optimize the move actions
                    var promise2 = docModel.iterateArraySliced(moveActions, function (moveData) {
                        return docModel.generateMoveSheetOperations(generator, moveData.from, moveData.to);
                    }, 'SheetMixin.showMoveSheetsDialog');

                    // reverse the undo operations to correctly undo multiple moves
                    return promise2.done(function () { generator.reverseOperations({ undo: true }); });
                });
            });

            // show warning alert if necessary
            return docView.yellOnFailure(promise);
        }

        /**
         * Shows a dialog that allows to make some or all hidden sheets visible
         * again.
         *
         * @returns {jQuery.Promise}
         *  A promise representing the dialog. Will be resolved with the sheets
         *  to unhide, after the sheets has been unhide successfully; or will
         *  rejected, if the dialog has been canceled, or if the document has
         *  switched to read-only mode.
         */
        function showShowSheetsDialog() {

            // leave text edit mode before displaying the dialog
            var promise = docView.leaveTextEditMode();

            // create and show the 'Show Sheets' dialog
            promise = promise.then(function () {
                return new Dialogs.UnhideSheetsDialog(docView).show();
            });

            // generate all sheet operations
            promise = promise.then(function (sheets) {

                // the sheet attribute set used to make a sheet visible
                var attrSet = { sheet: { visible: true } };

                // generate the 'changeSheet' operations for all hidden sheets
                return docModel.createAndApplySheetOperations(function (generator) {
                    return docModel.generateOperationsForSheets(generator, sheets, function (sheetModel) {
                        if (!sheetModel.isVisible()) {
                            return sheetModel.generateChangeSheetOperations(generator, attrSet);
                        }
                    });
                });
            });

            // show warning alert if necessary
            return docView.yellOnFailure(promise);
        }

        /**
         * Renames the active sheet in the spreadsheet document, if possible.
         * In case of an error, shows a notification message.
         *
         * @param {String} sheetName
         *  The new name of the active sheet.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved, if the sheet has been renamed, or
         *  that will be rejected otherwise.
         */
        function renameSheet(sheetName) {

            // leave text edit mode before renaming the sheet (sheet name may be used in formulas)
            var promise = docView.leaveTextEditMode().then(function () {

                // remove all NPCs from passed sheet name
                sheetName = Utils.cleanString(sheetName);

                // generate and apply the 'renameSheet' operation
                return docModel.createAndApplySheetOperations(function (generator) {
                    return docModel.generateRenameSheetOperations(generator, activeSheet, sheetName);
                });
            });

            // show warning alert if necessary
            return docView.yellOnFailure(promise);
        }

        /**
         * Shows a dialog that allows to enter a new name for the active sheet.
         * The dialog will be kept open until a valid sheet name has been
         * entered.
         *
         * @returns {jQuery.Promise}
         *  A promise representing the dialog. Will be resolved with the new
         *  sheet name, after the sheet has been renamed successfully; or will
         *  rejected, if the dialog has been canceled, or if the document has
         *  switched to read-only mode.
         */
        function showRenameSheetDialog() {

            // leave text edit mode before renaming the sheet (sheet name may be used in formulas)
            return docView.leaveTextEditMode().then(function () {

                // create the dialog, initialize with the name of the active sheet
                var dialog = new Dialogs.SheetNameDialog(docView, renameSheet, {
                    title: gt('Rename Sheet'),
                    value: docModel.getSheetName(activeSheet),
                    okLabel: gt('Rename')
                });

                // show the dialog, renaming the sheet will be done in the callback function
                return dialog.show();
            });
        }

        /**
         * Changes the formatting attributes of the active sheet.
         *
         * @param {Object} attributeSet
         *  An (incomplete) attribute set with the new formatting attributes
         *  for the active sheet.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved, if the sheet has been changed, or
         *  that will be rejected otherwise.
         */
        function changeSheet(attributeSet) {

            // leave text edit mode before manipulating the active sheet
            var promise = docView.leaveTextEditMode().then(function () {
                // generate and apply the operations
                return sheetModel.createAndApplyOperations(function (generator) {
                    return sheetModel.generateChangeSheetOperations(generator, attributeSet);
                });
            });

            // show warning alert if necessary
            return docView.yellOnFailure(promise);
        }

        // protected methods --------------------------------------------------

        /**
         * Activates a sheet for the import preview, while the document import
         * is still running.
         *
         * @internal
         *  Called from the preview handler of the application. MUST NOT be
         *  called directly.
         *
         * @param {Number} index
         *  The zero-based index of the initial active sheet.
         *
         * @returns {Boolean}
         *  Whether a sheet has been activated successfully.
         */
        this.activatePreviewSheet = function (sheet) {

            // the sheet model (no preview possible, if the sheet does not exist or is not visible)
            var sheetModel = docModel.getSheetModel(sheet);
            if (!sheetModel || !sheetModel.isVisible() || !sheetModel.isSupportedType()) { return false; }

            // initialize view settings (scroll position, selection, zoom, split)
            sheetModel.initializeViewAttributes();
            // activate the preview sheet in the view
            docModel.setActiveSheet(sheet);

            // return whether the sheet has been activated successfully (i.e. the event
            // handler for the 'change:activesheet' event has updated 'activeSheet')
            return activeSheet >= 0;
        };

        // item registration --------------------------------------------------

        // register all controller items
        this.registerDefinitions({

            // view settings --------------------------------------------------

            // model index of the active sheet
            'view/sheet/active': {
                parent: 'app/valid',
                get: function () { return activeSheet; },
                set: activateSheet
            },

            'view/sheet/prev': {
                parent: 'app/valid',
                enable: function () { return this.getValue() >= 0; },
                get: function () { return docModel.findVisibleSheet(activeSheet - 1, 'prev'); },
                set: function () { activateSheet(activeSheet - 1, 'prev'); },
                shortcut: { keyCode: 'PAGE_UP', ctrlOrMeta: true, alt: true }
            },

            'view/sheet/next': {
                parent: 'app/valid',
                enable: function () { return this.getValue() >= 0; },
                get: function () { return docModel.findVisibleSheet(activeSheet + 1, 'next'); },
                set: function () { activateSheet(activeSheet + 1, 'next'); },
                shortcut: { keyCode: 'PAGE_DOWN', ctrlOrMeta: true, alt: true }
            },

            // document operations --------------------------------------------

            // enabled if a worksheet is currently active (disabled for chart sheets, dialog sheets, etc.)
            'document/worksheet': {
                parent: 'app/valid',
                enable: function () { return sheetModel.isWorksheet(); }
            },

            // enabled if the document is editable, and a worksheet is currently active
            'document/editable/worksheet': {
                parent: ['document/editable', 'document/worksheet']
            },

            // enabled if the document is in read-only mode, and a worksheet is currently active
            'document/readonly/worksheet': {
                parent: ['document/readonly', 'document/worksheet']
            },

            // parent item that is enabled if the document contains more than one visible sheet
            'document/editable/multisheets': {
                parent: 'document/editable',
                enable: function () { return docModel.getSheetCount({ supported: true, visible: true }) > 1; }
            },

            'document/insertsheet': {
                parent: 'document/editable',
                enable: function () { return docModel.canInsertSheet(); },
                set: insertSheet
            },

            'document/deletesheet': {
                parent: 'document/editable/multisheets',
                set: deleteSheet
            },

            'document/copysheet/dialog': {
                parent: ['document/insertsheet', 'document/worksheet', 'document/ooxml'], // copy chart sheet not implemented in filter yet!
                set: showCopySheetDialog
            },

            'document/movesheet': {
                parent: 'document/editable',
                set: function (data) { return moveSheet(data.from, data.to); }
            },

            'document/movesheets/dialog': {
                parent: 'document/editable/multisheets',
                set: showMoveSheetsDialog
            },

            'document/showsheets/dialog': {
                parent: 'document/editable',
                enable: function () { return docModel.getSheetCount({ supported: true, visible: false }) > 0; },
                set: showShowSheetsDialog
            },

            // Parent item for controller items used for formula recalculation, and other formula dependency functionality.
            // The item is enabled, if the dependency manager is still active. It may become inactive if the document is too
            // complex (too many formulas).
            'document/formula/dependencies': {
                parent: 'document/editable',
                enable: function () { return dependencyManager.isConnected(); }
            },

            // recalculate all formulas contained in the document (also all conditional formattings, chart source ranges, etc.)
            'document/formula/recalc/all': {
                parent: 'document/formula/dependencies',
                set: function () { return dependencyManager.recalcFormulas({ all: true }); }
            },

            // single-sheet operations ----------------------------------------

            // returns/changes the name of the active sheet
            'sheet/name': {
                parent: 'document/editable',
                get: function () { return docModel.getSheetName(activeSheet); },
                set: renameSheet
            },

            'sheet/rename/dialog': {
                parent: 'document/editable',
                set: showRenameSheetDialog
            },

            // item value is the map of attributes of family 'sheet' of the active sheet
            'sheet/attributes': {
                parent: 'document/editable',
                // bug 57991: sheet model is not available after document import error
                get: function () { return sheetModel ? sheetModel.getMergedAttributeSet(true).sheet : {}; }
            },

            // represents the visibility state of the active sheet
            'sheet/visible': {
                parent: ['sheet/attributes', 'document/editable/multisheets'],
                get: function (attributes) { return attributes.visible; },
                set: function (visible) { return changeSheet({ sheet: { visible: visible } }); }
            },

            // represents the locked state of the active sheet
            'sheet/locked': {
                parent: ['sheet/attributes', 'document/editable/worksheet'],
                get: function (attributes) { return attributes.locked; },
                set: function (locked) { return changeSheet({ sheet: { locked: locked } }); }
            }

        });

        // initialization -----------------------------------------------------

        // no formula recalculation actions if the document is too complex
        this._updateOnEvent(dependencyManager, 'recalc:overflow');

        // initialize sheet-dependent class members according to the active sheet
        this.listenTo(docView, 'change:activesheet', function () {
            sheetModel = docView.getSheetModel();
            activeSheet = sheetModel.getIndex();
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docView = docModel = dependencyManager = sheetModel = null;
        });

    } // class SheetMixin

    // exports ================================================================

    return SheetMixin;

});
