/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/controller/chartmixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/drawinglayer/view/drawinglabels',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/view/chartcreator',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, DrawingLabels, Operations, ChartCreator, gt) {

    'use strict';

    // class ChartMixin =======================================================

    /**
     * Implementations of all controller items for manipulating chart objects
     * in the active sheet, intended to be mixed into a document controller
     * instance.
     *
     * @constructor
     *
     * @param {SpreadsheetView} docView
     *  The document view providing view settings such as the current drawing
     *  selection.
     */
    function ChartMixin(docView) {

        // self reference
        var self = this;

        // the application instance containing this controller
        var app = this.getApp();

        // the document model
        var docModel = docView.getDocModel();

        var sourceSelector = false;

        // private methods ----------------------------------------------------

        /**
         * Shows a warning alert box for the 'exchange direction' function for
         * chart series.
         */
        function yellChartDirection(type) {

            switch (type) {
                case 'sheets':
                    docView.yell({
                        type: 'warning',
                        message:
                            //#. Warning text: A chart object in a spreadsheet contains complex data source (cells from multiple sheets).
                            gt('Data references are located on different sheets.')
                    });
                    break;

                case 'directions':
                    docView.yell({
                        type: 'warning',
                        message:
                            //#. Warning text: A chart object in a spreadsheet contains complex data source (not a simple cell range).
                            gt('Data references are too complex for this operation.')
                    });
                    break;

                default:
                    Utils.warn('DrawingController.yellChartDirection(): unknown warning type: "' + type + '"');
            }

            return $.Deferred().reject();
        }

        /**
         * Returns the position of the selected drawing object, if exactly one
         * drawing object is currently selected.
         *
         * @returns {Array<Number>|Null}
         *  The position of the selected drawing object in the active sheet
         *  (without leading sheet index), if available; otherwise null.
         */
        function getDrawingPosition() {
            var drawingPositions = docView.getSelectedDrawings();
            return (drawingPositions.length === 1) ? drawingPositions[0] : null;
        }

        /**
         * @returns {DrawingModel|Null}
         *  The drawing model currently selected.
         */
        function getDrawingModel(type) {
            var drawingPos = getDrawingPosition();
            if (drawingPos) {
                return docView.getDrawingCollection().getModel(drawingPos, { type: type });
            }
        }

        /**
         * Generates and applies a document operation for a chart object with correct undo operation.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the drawing operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        function createAndApplyChartOperation(chartModel, opName, properties, callback) {
            return docView.getSheetModel().createAndApplyOperations(function (generator) {

                var position = getDrawingPosition();
                var promise = self.convertToPromise(callback ? callback(generator, position) : null);

                return promise.done(function () {
                    generateChartOperation(generator, chartModel, opName, position, properties);
                });
            }, { storeSelection: true });
        }

        /**
         * Generates a document operation for a chart object with correct undo operation.
         */
        function generateChartOperation(generator, chartModel, opName, position, properties) {
            var undoProperties = null;
            switch (opName) {
                case Operations.SET_CHART_LEGEND_ATTRIBUTES:
                    var legendModel = chartModel.getLegendModel(properties.axis);
                    undoProperties = { attrs: legendModel.getUndoAttributeSet(properties.attrs) };
                    break;

                case Operations.SET_CHART_TITLE_ATTRIBUTES:
                    var axisId = properties.axis === null || properties.axis === undefined ? null : properties.axis;
                    var titleModel = chartModel.getTitleModel(axisId);
                    undoProperties = { attrs: titleModel.getUndoAttributeSet(properties.attrs) };
                    if (axisId !== null) {
                        undoProperties.axis = axisId;
                    }
                    break;

                case Operations.SET_CHART_AXIS_ATTRIBUTES:
                    var axisModel = chartModel.getAxisModel(properties.axis);
                    undoProperties = { axis: properties.axis, axPos: properties.axPos, crossAx: properties.crossAx, attrs: axisModel.getUndoAttributeSet(properties.attrs) };
                    break;

                case Operations.SET_CHART_GRIDLINE_ATTRIBUTES:
                    var gridModel = chartModel.getAxisModel(properties.axis).getGrid();
                    undoProperties = { axis: properties.axis, attrs: gridModel.getUndoAttributeSet(properties.attrs) };
                    break;

                case Operations.CHANGE_DRAWING:
                    undoProperties = { attrs: chartModel.getUndoAttributeSet(properties.attrs) };
                    break;

                default:
                    Utils.error('generateChartOperation(): unknown operation "' + opName + '"');
                    return;
            }

            generator.generateDrawingOperation(opName, position, undoProperties, { undo: true });
            generator.generateDrawingOperation(opName, position, properties);
        }

        /**
         * Returns the title text from the passed attribute set.
         *
         * @param {Object} [attributes]
         *  The merged attribute set for a chart title. If omitted, returns an
         *  empty string.
         *
         * @returns {String}
         *  The title text from the passed attribute set.
         */
        function getChartTitle(attributes) {
            if (_.isObject(attributes) && _.isArray(attributes.text.link)) { return (attributes.text.link[0] || '').trim(); }
            return '';
        }

        /**
         * Changes the attributes of the specified title in the selected chart
         * object.
         *
         * @param {ChartModel} chartModel
         *  The chart model to be manipulated.
         *
         * @param {Number} axisId
         *  The identifier of the title object axis. Must be one of null for the MainTitle or
         *  the Id for the respective axis title.
         *
         * @param {Object} attrs
         *  The attribute set to be set for the title.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the operation has been
         *  applied successfully, or that will be rejected on any error.
         */
        function setChartTitleAttributes(chartModel, axisId, attrs) {
            var properties = { attrs: attrs };
            if (axisId !== null) {
                properties.axis = axisId;
            }
            return createAndApplyChartOperation(chartModel, Operations.SET_CHART_TITLE_ATTRIBUTES, properties);
        }

        /**
         * Changes the text of the specified title in the selected chart
         * object.
         *
         * @param {ChartModel} chartModel
         *  The chart model to be manipulated.
         *
         * @param {String} titleId
         *  The identifier of the title object. Must be one of null, 'x',
         *  'y', or 'z'.
         *
         * @param {String} title
         *  The new text contents of the title. An empty string will remove the
         *  title from the chart.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the operation has been
         *  applied successfully, or that will be rejected on any error.
         */
        function setChartTitle(chartModel, titleId, title) {
            return setChartTitleAttributes(chartModel, titleId, { text: { link: [title] }, character: ChartCreator.getHeadChar(docModel) });
        }

        function selectNewChartSource() {

            // enter custom selection mode
            var promise = docView.enterCustomSelectionMode('chartsource', {
                //#. change source data for a chart object in a spreadsheet
                statusLabel: gt('Select source data')
            });

            // set selected range as source data
            promise.done(function (selection) {

                var range = selection.ranges[0];
                var sheet = docView.getActiveSheet();

                var chart = getDrawingModel('chart');

                var possSources = chart.getDataSourceInfo();
                if (possSources.warn) {
                    return ChartCreator.updateSeries({ app: app, sourceRange: range, sourceSheet: sheet, chartModel: chart });
                }
                var forceNames;
                if (chart.isXYType()) { forceNames = chart.isNamesLabel(); }

                return ChartCreator.updateSeries({ app: app, sourceRange: range, sourceSheet: sheet, chartModel: chart, axis: possSources.axis, forceNames: forceNames });
            });
        }

        /**
         * changeChartType maps the assigned id to chart-data
         * all ids are mapped in DrawingLabels.CHART_TYPE_STYLES
         *
         * There is a special behavior for bubble-chart, change to bubble or from bubble.
         * all series-data will be removed an initialized complete new by the ChartCreator
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the operation has been
         *  applied successfully, or that will be rejected on any error.
         */
        function changeChartType(id) {

            var chartModel = getDrawingModel('chart');
            var data = DrawingLabels.CHART_TYPE_STYLES[id];

            // fix for Bug 46917 & Bug 49738
            var possSources = chartModel.getDataSourceInfo();
            if (possSources.warn) {
                return yellChartDirection(possSources.warn);
            }
            var forceNames;
            if (chartModel.isXYType()) { forceNames = chartModel.isNamesLabel(); }
            return ChartCreator.updateSeries({ app: app, sourceRange: possSources.range, sourceSheet: possSources.sheet, chartModel: chartModel, axis: possSources.axis, chartData: data, forceNames: forceNames/*, typeChanged: true*/ });
        }

        function isFirstRowColEnabled(chartModel, rowCol) {
            if (!chartModel) { return false; }

            var possSources = chartModel.getDataSourceInfo();
            if (possSources.warn) { return false; }
            if (possSources.axis) { rowCol = Math.abs(rowCol - 1); }
            if (rowCol === 1) {
                return isTitleLabelEnabled(chartModel);
            } else {
                return chartModel.isNamesLabel() || (chartModel.getSeriesCount() > 1);
            }
        }

        function isFirstRowCol(chartModel, rowCol) {
            if (!chartModel) { return false; }

            var possSources = chartModel.getDataSourceInfo();
            if (possSources.warn) { return false; }
            if (possSources.axis) { rowCol = Math.abs(rowCol - 1); }
            if (rowCol === 1) {
                return chartModel.isTitleLabel();
            } else {
                return chartModel.isNamesLabel();
            }
        }

        function setFirstRowCol(chartModel, rowCol, value) {
            if (!chartModel) { return null; }
            var possSources = chartModel.getDataSourceInfo();
            if (possSources.warn) { return null; }

            if (possSources.axis) { rowCol = Math.abs(rowCol - 1); }

            var forceTitle;
            var forceNames;
            if (rowCol === 1) {
                forceNames = chartModel.isNamesLabel();
                forceTitle = value;
            } else {
                forceNames = value;
                forceTitle = chartModel.isTitleLabel();
            }
            return ChartCreator.updateSeries({ app: app, sourceRange: possSources.range, sourceSheet: possSources.sheet, chartModel: chartModel, axis: possSources.axis, forceTitle: forceTitle, forceNames: forceNames });
        }

        function isTitleLabelEnabled(chartModel) {
            if (chartModel.isTitleLabel()) {
                return true;
            }
            var min = 1;
            if (chartModel.isAreaOrLine()) { min = 2; }
            return chartModel.getFirstPointsCount() > min;
        }

        function exchangeEnabled(chartModel) {
            var min = 0;
            if (chartModel.isAreaOrLine()) { min = 1; }
            return chartModel.getSeriesCount() > min;
        }

        function switchRowAndColumn(chartModel) {
            var possSources = chartModel.getDataSourceInfo();
            if (possSources.warn) {
                return yellChartDirection(possSources.warn);
            }
            var forceTitle = chartModel.isNamesLabel();
            var forceNames = chartModel.isTitleLabel();
            return ChartCreator.updateSeries({ app: app, sourceRange: possSources.range, sourceSheet: possSources.sheet, chartModel: chartModel, axis: 1 - possSources.axis, forceTitle: forceTitle, forceNames: forceNames, switchRowColumn: true });
        }

        // item registration --------------------------------------------------

        // register all controller items
        this.registerDefinitions({

            'chart/insert': {
                parent: 'drawing/insert/enable',
                enable: function () {
                    var ranges = docView.getSelectedRanges();
                    return (ranges.length === 1) && !docView.getSheetModel().isSingleCellInRange(ranges[0]);
                },
                set: function (id) { return ChartCreator.createChart(app, DrawingLabels.CHART_TYPE_STYLES[id]); }
            },

            'drawing/chart': {
                parent: 'drawing/operation',
                enable: function () { return _.isObject(this.getValue()); },
                get: function () { return getDrawingModel('chart'); }
            },

            'drawing/chart/valid': {
                parent: 'drawing/chart',
                enable: function (chartModel) { return chartModel.isRestorable(); }
            },

            'drawing/chartlabels': {
                parent: 'drawing/chart/valid',
                get: function () { return docView.getChartLabelsMenu().isVisible(); },
                set: function (state) { docView.getChartLabelsMenu().toggle(state); }
            },

            'drawing/chartexchange': {
                parent: 'drawing/chart/valid',
                enable: exchangeEnabled,
                set: function () { return switchRowAndColumn(this.getParentValue()); }
            },

            'drawing/chartfirstcol': {
                parent: 'drawing/chart/valid',
                enable: function (chartModel) { return isFirstRowColEnabled(chartModel, 0); },
                get: function (chartModel) { return isFirstRowCol(chartModel, 0); },
                set: function (state) { return setFirstRowCol(this.getParentValue(), 0, state); }
            },

            'drawing/chartfirstrow': {
                parent: 'drawing/chart/valid',
                enable: function (chartModel) { return isFirstRowColEnabled(chartModel, 1); },
                get: function (chartModel) { return isFirstRowCol(chartModel, 1); },
                set: function (state) { return setFirstRowCol(this.getParentValue(), 1, state); }
            },

            'drawing/charttype': {
                parent: 'drawing/chart/valid',
                get: function (chartModel) { return chartModel ? chartModel.getChartTypeForGui() : null; },
                set: changeChartType
            },

            // parent item providing access to the attributes of a chart model
            'drawing/chart/attributes': {
                parent: 'drawing/chart/valid',
                get: function (chartModel) { return chartModel ? chartModel.getMergedAttributeSet(true) : null; }
            },

            'drawing/chartvarycolor': {
                parent: ['drawing/chart/valid', 'drawing/chart/attributes'],
                enable: function (chartModel) { return chartModel.isVaryColorEnabled(); },
                get: function (chartModel) { return chartModel ? chartModel.isVaryColor() : null; },
                set: function (state) { return this.getParentValue().changeVaryColors(state); }
            },

            'drawing/chartdatalabel': {
                parent: 'drawing/chart/valid',
                get: function (chartModel) { return chartModel ? (!_.isEmpty(chartModel.getDataLabel())) : null; },
                set: function (state) { return this.getParentValue().setDataLabel(state); }
            },

            'drawing/chartcolorset': {
                parent: 'drawing/chart/valid',
                get: function (chartModel) { return chartModel ? chartModel.getColorSet() : null; },
                set: function (colorset) { return this.getParentValue().changeColorSet(colorset); }
            },

            'drawing/chartstyleset': {
                parent: 'drawing/chart/valid',
                get: function (chartModel) { return chartModel ? chartModel.getStyleSet() : null; },
                set: function (colorset) { return this.getParentValue().changeStyleSet(colorset); }
            },

            'drawing/chartdatasource': {
                parent: 'drawing/chart/valid'
            },

            'drawing/chartsource': {
                parent: 'drawing/chart/valid',
                get: function () { return sourceSelector; },
                set: selectNewChartSource,
                enable: function () { return !Utils.TOUCHDEVICE; } // disable for touch devices. remove this if the story DOCS-1036 user can edit chart data reference on touch devices is ready
            },

            'drawing/chartlegend/pos': {
                parent: 'drawing/chart/valid',
                get: function (chartModel) { return chartModel ? chartModel.getLegendModel().getMergedAttributeSet(true).legend.pos : null; },
                set: function (pos) { return createAndApplyChartOperation(this.getParentValue(), Operations.SET_CHART_LEGEND_ATTRIBUTES, { attrs: { legend: { pos: pos } } }); }
            },

            'drawing/chart/axes/enabled': {
                parent: 'drawing/chart/valid',
                enable: function (chartModel) { return chartModel.isAxesEnabled(); }
            },

            // parent item providing access to the main title model of a chart
            'drawing/chart/title/model': {
                parent: 'drawing/chart/valid',
                get: function (chartModel) { return chartModel ? chartModel.getTitleModel(null) : null; }
            },

            // parent item providing access to the attributes of the main title model of a chart
            'drawing/chart/title/attributes': {
                parent: ['drawing/chart/title/model', 'drawing/chart/valid'],
                get: function (titleModel) { return titleModel ? titleModel.getMergedAttributeSet(true) : null; },
                set: function (attributes) { return setChartTitleAttributes(this.getParentValue(1), null, attributes); }
            },

            // return or modify the text contents of the main title of a chart
            'drawing/chart/title/text': {
                parent: ['drawing/chart/title/attributes', 'drawing/chart/valid'],
                get: function (attributes) { return getChartTitle(attributes); },
                set: function (title) { return setChartTitle(this.getParentValue(1), null, title); }
            },

            // remove the main title from a chart (disabled if title does not exist)
            'drawing/chart/title/delete': {
                parent: ['drawing/chart/title/text', 'drawing/chart/valid'],
                enable: function () { return this.getValue().length > 0; },
                set: function () { return setChartTitle(this.getParentValue(1), null, ''); }
            }
        });

        ['x', 'y'].forEach(function (axisId) {

            // the base path of axis items
            var keyPath = 'drawing/chart/axis/' + axisId + '/';
            // item definitions map with dynamic keys
            var definitions = {};

            function getCorrectAxis(chartModel) {
                return chartModel ? chartModel.getAxisForType(axisId) : null;
            }

            function getCorrectAxisId(chartModel) {
                return chartModel ? chartModel.getAxisIdForType(axisId) : null;
            }

            function getStandardLineAttributes(visible) {
                return { line: visible ? ChartCreator.getStandardShape(docModel) : ChartCreator.getNoneShape() };
            }

            function setAxisAttributes(chartModel, attrs) {
                var axis = getCorrectAxis(chartModel);
                return createAndApplyChartOperation(chartModel, Operations.SET_CHART_AXIS_ATTRIBUTES, { axis: axis.axisId, axPos: axis.axPos, crossAx: axis.crossAx, attrs: attrs });
            }

            function setGridLineAttributes(chartModel, attrs) {
                return createAndApplyChartOperation(chartModel, Operations.SET_CHART_GRIDLINE_ATTRIBUTES, { axis: getCorrectAxisId(chartModel), attrs: attrs });
            }

            // *** axis items ***

            // parent item providing access to a chart axis model
            definitions[keyPath + 'model'] = {
                parent: ['drawing/chart/valid', 'drawing/chart/axes/enabled'],
                get: function (chartModel) { return chartModel ? chartModel.getAxisModel(getCorrectAxisId(chartModel)) : null; }
            };

            // parent item providing access to the attributes of a chart axis model
            definitions[keyPath + 'attributes'] = {
                parent: [keyPath + 'model', 'drawing/chart/valid'],
                get: function (axisModel) { return axisModel ? axisModel.getMergedAttributeSet(true) : null; },
                set: function (attributes) { return setAxisAttributes(this.getParentValue(1), attributes); }
            };

            // return or modify the visibility of the axis caption labels
            definitions[keyPath + 'labels/visible'] = {
                parent: [keyPath + 'attributes', 'drawing/chart/valid'],
                get: function (attributes) { return _.isObject(attributes) && (attributes.axis.label === true) && this.areParentsEnabled(); },
                set: function (visible) { return setAxisAttributes(this.getParentValue(1), { axis: { label: visible } }); }
            };

            // return or modify the visibility of the axis line
            definitions[keyPath + 'line/visible'] = {
                parent: [keyPath + 'attributes', 'drawing/chart/valid'],
                get: function (attributes) { return _.isObject(attributes) && (attributes.line.type !== 'none'); },
                set: function (visible) { return setAxisAttributes(this.getParentValue(1), getStandardLineAttributes(visible)); }
            };

            // *** grid line items ***

            // parent item providing access to a chart axis grid model
            definitions[keyPath + 'grid/model'] = {
                parent: keyPath + 'model',
                get: function (axisModel) { return axisModel ? axisModel.getGrid() : null; }
            };

            // parent item providing access to the attributes of a chart axis grid model
            definitions[keyPath + 'grid/attributes'] = {
                parent: [keyPath + 'grid/model', 'drawing/chart/valid'],
                get: function (gridModel) { return gridModel ? gridModel.getMergedAttributeSet(true) : null; },
                set: function (attributes) { return setGridLineAttributes(this.getParentValue(1), attributes); }
            };

            // return or modify the visibility of the axis grid lines
            definitions[keyPath + 'grid/visible'] = {
                parent: [keyPath + 'grid/attributes', 'drawing/chart/valid'],
                get: function (attributes) { return _.isObject(attributes) && (attributes.line.type !== 'none'); },
                set: function (visible) { return setGridLineAttributes(this.getParentValue(1), getStandardLineAttributes(visible)); }
            };

            // *** title items ***

            // parent item providing access to a chart axis title model
            definitions[keyPath + 'title/model'] = {
                parent: keyPath + 'model',
                get: function (axisModel) { return axisModel ? axisModel.getTitle() : null; }
            };

            // parent item providing access to the attributes of a chart axis title model
            definitions[keyPath + 'title/attributes'] = {
                parent: [keyPath + 'title/model', 'drawing/chart/valid'],
                get: function (titleModel) { return titleModel ? titleModel.getMergedAttributeSet(true) : null; },
                set: function (attributes) {
                    var chartModel = this.getParentValue(1);
                    return setChartTitleAttributes(chartModel, getCorrectAxisId(chartModel), attributes);
                }
            };

            // return or modify the text contents of a chart axis title
            definitions[keyPath + 'title/text'] = {
                parent: [keyPath + 'title/attributes', 'drawing/chart/valid'],
                get: function (attributes) { return getChartTitle(attributes); },
                set: function (title) {
                    var chartModel = this.getParentValue(1);
                    return setChartTitle(chartModel, getCorrectAxisId(chartModel), title);
                }
            };

            // remove the main title from a chart (disabled if title does not exist)
            definitions[keyPath + 'title/delete'] = {
                parent: [keyPath + 'title/text', 'drawing/chart/valid'],
                enable: function () { return this.getValue().length > 0; },
                set: function () {
                    var chartModel = this.getParentValue(1);
                    return setChartTitle(chartModel, getCorrectAxisId(chartModel), '');
                }
            };

            this.registerDefinitions(definitions);
        }, this);

        // initialization -----------------------------------------------------

        this.registerDestructor(function () {
            self = app = docModel = docView = null;
        });

    } // class ChartMixin

    // exports ================================================================

    return ChartMixin;

});
