/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/app/application', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/baseframework/utils/errorcode',
    'io.ox/office/baseframework/utils/errorcontext',
    'io.ox/office/editframework/app/editapplication',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/model',
    'io.ox/office/spreadsheet/model/presetformattable',
    'io.ox/office/spreadsheet/view/view',
    'io.ox/office/spreadsheet/controller/controller',
    'io.ox/office/spreadsheet/services/commandmixin'
], function (Utils, LocaleData, ErrorCode, ErrorContext, EditApplication, SheetUtils, SpreadsheetModel, PresetFormatTable, SpreadsheetView, SpreadsheetController, CommandMixin) {

    'use strict';

    // all parameters sent to server for new empty documents
    var NEW_DOCUMENT_PARAMS = {
        initial_sheetname: SheetUtils.getSheetName(1),
        initial_formats: JSON.stringify(PresetFormatTable.getInitialFormats(LocaleData.LOCALE))
    };

    // class SpreadsheetApplication ===========================================

    /**
     * The OX Spreadsheet application.
     *
     * In additions of the events triggered by the base classes, the following
     * events will be triggered:
     *  - 'docs:preview:activesheet'
     *      Triggered during the import process when the active sheet has been
     *      loaded, and import of all other sheets starts.
     *
     * @constructor
     *
     * @extends EditApplication
     *
     * @param {Object} launchOptions
     *  All options passed to the core launcher (the ox.launch() method).
     */
    var SpreadsheetApplication = EditApplication.extend(function (launchOptions) {

        // self reference
        var self = this;

        // the spreadsheet document model and view
        var docModel = null;
        var docView = null;

        // base constructor ---------------------------------------------------

        EditApplication.call(this, SpreadsheetModel, SpreadsheetView, SpreadsheetController, launchOptions, {
            newDocumentParams: NEW_DOCUMENT_PARAMS,
            postProcessHandler: postProcessDocument,
            previewHandler: previewHandler,
            importFailedHandler: importFailedHandler,
            prepareFlushHandler: prepareFlushDocument,
            prepareLoseEditRightsHandler: prepareLoseEditRights,
            prepareRenameHandler: prepareRenameHandler,
            realTimeDelay: 10,
            supportsOnlineSync: true,
            isSnapshotApp: true
        });

        // private methods ----------------------------------------------------

        /**
         * Post-processing of the document, after all import operations have
         * been applied successfully.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the document has been
         *  post-processed successfully; or rejected when the document is
         *  invalid, or any other error has occurred.
         */
        function postProcessDocument() {
            return docModel.postProcessImport();
        }

        /**
         * Shows an early preview of the document. Activates the sheet that is
         * specified in the passed preview data, and starts querying cell
         * contents for the active sheet.
         *
         * @param {Object} previewData
         *  The preview data containing the index of the active sheet.
         *
         * @returns {Boolean}
         *  Whether a sheet has been activated successfully.
         */
        function previewHandler(previewData) {
            var activeSheet = Utils.getIntegerOption(previewData, 'activeSheet', 0);
            var success = self.getController().activatePreviewSheet(activeSheet);
            // notify view components, e.g. repaint the sheet tabs once during import
            if (success) { self.trigger('docs:preview:activesheet'); }
            return success;
        }

        /**
         * Handler will be called by base class if importing the document
         * failed. The functions handles Spreadsheet specific errors.
         */
        function importFailedHandler(response) {

            // specific error code sent by the server
            var error = new ErrorCode(response);

            switch (error.getCodeAsConstant()) {
                case 'LOADDOCUMENT_COMPLEXITY_TOO_HIGH_ERROR':
                    self.setInternalError(error, ErrorContext.SPREADSHEET, null, { showMessage: false });
                    break;
                case 'LOADDOCUMENT_COMPLEXITY_TOO_HIGH_SHEET_COUNT_ERROR':
                    self.setInternalError(error, ErrorContext.SPREADSHEET, null, { showMessage: false });
                    break;
            }
        }

        /**
         * Generic handler that finalizes the document in preparation of any
         * global actions, such as downloading or closing the document, or
         * losing the edit rights.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Number} [options.timeout]
         *      The maximum time allowed to recalculate all pending cell
         *      formulas, in milliseconds. If the time elapses before all
         *      formulas have been calculated, the recalculation cycle will be
         *      aborted, and the 'calcOnLoad' document flag will be set. If
         *      omitted, no timeout will be set.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all pending update tasks have
         *  been finished (e.g. recalculating all dirty cell formulas).
         */
        function finalizePendingDocumentChanges(options) {

            // commit current edit text (without validation)
            var promise = docView.leaveTextEditMode({ commitMode: 'force' });

            // wait for the dependency manager recalculating all dirty formulas
            promise = promise.then(function () {
                options = _.extend({}, options, { volatile: false });
                return docModel.getDependencyManager().recalcFormulas(options);
            });

            // do not forward the rejected state of the dependency manager after a timeout
            promise = promise.catch(function (cause) {
                if (cause === 'destroy') { throw cause; }
            });

            // send all changed view settings (after recalculating the formulas which may
            // initially set the 'document modified' flag)
            return promise.then(function () {
                return self.getController().sendChangedViewAttributes();
            });
        }

        /**
         * Preprocessing before the document will be flushed for downloading,
         * printing, sending as mail, or closing.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all pending update tasks have
         *  been finished (e.g. recalculating all dirty cell formulas).
         */
        function prepareFlushDocument(cause) {
            var timeout = (cause === 'quit') ? 5000 : 15000;
            return finalizePendingDocumentChanges({ timeout: timeout });
        }

        /**
         * Preparations before the edit mode will be switched off.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all pending update tasks have
         *  been finished (e.g. recalculating all dirty cell formulas).
         */
        function prepareLoseEditRights() {
            return finalizePendingDocumentChanges();
        }

        /**
         * Preparations before the request to rename the edited document will
         * be sent to the backend.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all pending update tasks have
         *  been finished (e.g. recalculating all dirty cell formulas).
         */
        function prepareRenameHandler() {
            return finalizePendingDocumentChanges({ timeout: 15000 });
        }

        // initialization -----------------------------------------------------

        // add mix-in classes (need a complete application instance)
        CommandMixin.call(this);

        // seleniumWrapper (editFramework) needs the SheetUtils ... especially the class "Address"
        this.getSheetUtils = function () {
            return SheetUtils;
        };

        // initialization of class members
        this.onInit(function () {
            docModel = self.getModel();
            docView = self.getView();
        });

        // destroy all class members
        this.registerDestructor(function () {
            launchOptions = self = docModel = docView = null;
        });

    }); // class SpreadsheetApplication

    // exports ================================================================

    return SpreadsheetApplication;

});
