/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Mario Schroeder <mario.schroeder@open-xchange.com>
 * @author Edy Haryono <edy.haryono@open-xchange.com>
 */

define('io.ox/office/presenter/main', [
    'io.ox/files/api',
    'io.ox/core/page-controller',
    'io.ox/office/presenter/rtconnection',
    'io.ox/office/presenter/rtmodel',
    'io.ox/office/presenter/localmodel',
    'io.ox/office/presenter/views/mainview',
    'io.ox/office/presenter/views/notification',
    'io.ox/office/tk/utils',
    'io.ox/core/tk/sessionrestore',
    'less!io.ox/office/presenter/style'
], function (FilesAPI, PageController, RTConnection, RTModel, LocalModel, MainView, Notification, Utils, SessionRestore) {

    'use strict';

    var NAME = 'io.ox/office/presenter';

    ox.ui.App.mediator(NAME, {

        'pages-desktop': function (app) {

            // add page controller
            app.pages = new PageController(app);

            app.pages.addPage({
                name: 'presentationView',
                container: app.getWindow().nodes.main
            });

            app.pages.setCurrentPage('presentationView');
        },

        'show-error-notification': function (app) {
            app.showErrorNotification = function (error, options) {
                var page = app.pages.getPage('presentationView');
                var notificationNode = Notification.createErrorNode(error, options);
                page.empty().append(notificationNode);
            };
        },

        'connect-realtime': function (app) {
            app.connectRealtime = function () {

                var def = null;

                // RT connect success handler
                function rtConnectSuccess(response) {
                    return app.rtModel.set(app.rtModel.parse(response));
                }

                // RT connect error handler
                function rtConnectError(response) {
                    Utils.warn('RT Connect Error', response);
                    app.showErrorNotification(response, { category: 'rt' });
                }

                // RT connection update event handler
                function rtUpdateHandler(event, data) {
                    app.rtModel.set(app.rtModel.parse(data));
                }

                // RT online handler
                function rtOnlineHandler() {
                    if (app.offlineHandlerTriggered) {
                        app.offlineHandlerTriggered = false;
                        Notification.notifyRealtimeOnline();
                    }
                }

                // RT offline handler
                function rtOfflineHandler() {
                    app.offlineHandlerTriggered = true;
                    Notification.notifyRealtimeOffline();
                }

                // RT error handler
                function rtErrorHandler(event) {
                    Notification.notifyRealtimeError(event && event.type);
                }

                // init RT connection
                app.rtModel      = new RTModel();
                app.rtConnection = new RTConnection(app.fileModel.toJSON());

                if (app.rtConnection.isInitialized()) {
                    def = app.rtConnection.connect().then(rtConnectSuccess, rtConnectError);
                    app.rtConnection.on({
                        update: rtUpdateHandler,
                        online: rtOnlineHandler,
                        offline: rtOfflineHandler,
                        'timeout reset error:notMember error:stanzaProcessingFailed error:joinFailed error:disposed': rtErrorHandler
                    });
                } else {
                    app.showErrorNotification({ error: '', code: 'RT_STANZA-1015' }, { category: 'rt' });
                    def = $.Deferred().reject();
                }

                return def;
            };
        },

        'start-presentation': function (app) {

            app.startPresenter = function (file) {

                var title = file.get('filename') || file.get('title');

                app.setTitle(title);
                app.fileModel = file;
                app.offlineHandlerTriggered = false;

                // init local model
                app.localModel = new LocalModel();

                // init and connect RT
                app.connectRealtime().then(function () {
                    var page      = app.pages.getPage('presentationView');
                    var lastState = SessionRestore.state('presenter~' + app.fileModel.get('id'));

                    app.mainView = new MainView({ model: app.fileModel, app: app });
                    page.append(app.mainView.render().$el);

                    // restore state before the browser reload
                    if (lastState && lastState.isPresenter) {
                        app.rtConnection.startPresentation({ activeSlide: lastState.slideId || 0 });
                    }
                    // join a runnig presentation if Presenter was started from a deep link
                    if (app.deepLink && app.rtModel.canJoin()) {
                        Notification.notifyPresentationJoin(app.rtModel, app.rtConnection);
                        app.mainView.joinPresentation();
                    }
                });
            };
        },

        'dispose-rt-connection': function (app) {
            // dispose RT connection instance
            app.disposeRTConnection = function () {
                if (app.rtConnection) {
                    if (app.rtConnection.isInitialized()) {
                        app.rtConnection.close();
                        app.rtConnection.dispose();
                    }

                    app.rtConnection.off();
                    app.rtConnection = null;
                }
            };
        },

        'on-app-window-show': function (app) {
            app.getWindow().on('show', function () {
                var id        = app.fileModel && app.fileModel.get('id');
                var folder_id = app.fileModel && app.fileModel.get('folder_id');

                if (id && folder_id) {
                    app.setState({ id: id, folder: folder_id });
                }
            });
        }

    });

    // multi instance pattern
    function createInstance() {

        // application object
        var app = ox.ui.createApp({
            closable: true,
            name: NAME,
            title: ''
        });

        function beforeUnloadHandler() {
            var id    = 'presenter~' + app.fileModel.get('id');
            var state = SessionRestore.state(id);

            if (app.rtConnection.isInitialized() && state) {
                state.slideId = app.mainView.getActiveSlideIndex();
                SessionRestore.state(id, state);
            }

            app.disposeRTConnection();
        }

        function quitHandler() {
            ox.off('beforeunload', beforeUnloadHandler);

            if (app.rtConnection.isInitialized()) {
                SessionRestore.state('presenter~' + app.fileModel.get('id'), null);
            }
            app.disposeRTConnection();

            // clear file authentication from file model
            if (app.fileModel.has('file_options')) {
                app.fileModel.set('file_options', {});
            }
        }

        function getModelFromFileDescriptor(file) {

            function showFileError(error) {
                Utils.warn('File Error', error);
                app.showErrorNotification(error, { category: 'drive' });
            }

            return FilesAPI.get(file).then(function (data) {
                return FilesAPI.pool.get('detail').get(_.cid(data));

            }, showFileError);
        }

        // launcher
        return app.setLauncher(function (options) {

            var state = null;
            var model = Utils.getObjectOption(options, 'model', null);

            var win = ox.ui.createWindow({
                chromeless: true,
                name: NAME,
                toolbar: false
            });

            app.setWindow(win);
            app.mediate();

            ox.on('beforeunload', beforeUnloadHandler);
            app.on('quit', quitHandler);

            win.show();

            if (model instanceof FilesAPI.Model) {
                // launched with a file, mail attachment or PIM attachment
                app.deepLink = false;

                if (model.isFile()) {
                    app.setState({ folder: model.get('folder_id'), id: model.get('id') });
                }

                app.startPresenter(model);

            } else {
                // launched with a deep-link
                app.deepLink = true;
                state = app.getState();

                // get file model and start
                getModelFromFileDescriptor(state).then(app.startPresenter);
            }
        });
    }

    return {
        getApp: createInstance
    };

});
