/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define('io.ox/office/presentation/components/field/slidefield', [
    'io.ox/office/tk/utils/dateutils',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/baseframework/app/appobjectmixin',
    'io.ox/office/textframework/utils/textutils',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/utils/snapshot',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/textframework/components/field/simplefield',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/presentation/utils/presentationutils',
    'io.ox/office/presentation/view/dialog/insertfooterdialog',
    'io.ox/office/presentation/view/dialog/insertdatedialog',
    'gettext!io.ox/office/presentation/main'
], function (DateUtils, LocaleData, TriggerObject, AppObjectMixin, Utils, Operations, DOM, Position, Snapshot, DrawingFrame, SimpleField, AttributeUtils, PresentationUtils, InsertFooterDialog, InsertDateFieldDialog, gt) {

    'use strict';

    // class SlideField =======================================================

    /**
     * An instance of this class represents the handler for all fields in the edited Presentation document.
     *
     * @constructor
     *
     * @param {PresentationApplication} app
     *  The application instance.
     */
    function SlideField(app) {

        var // self reference
            self = this,
            // simple field instance
            simpleField = new SimpleField(app),
            // counter for simple field Id
            simpleFieldID = 0,
            // number formatter instance
            numberFormatter = null,
            // the text model object
            model = null,
            // the selection object
            selection = null,
            // flag to determine if cursor position is inside field
            isCursorHighlighted = false,
            // saved id of currently highlighted complex field during cursor traversal or click
            currentlyHighlightedId,
            // for simple fields we store node instance of highlighted field
            currentlyHighlightedNode,
            // field format popup promise
            fieldFormatPopupTimeout = null,
            // holds value of type of highlighted field
            highlightedFieldType = null,
            // holds value of format instruction of highlighted field
            highlightedFieldInstruction = null,
            localFormatList = (app.isODF()) ? {
                date: [],
                time: [],
                creator: [],
                'page-number': [
                    { option: 'default', value: /*#. dropdown list option: original string value, no formatting applied */ gt('(no format)') },
                    { option: 'i', value: 'i, ii, iii, ...' },
                    { option: 'I', value: 'I, II, III, ...' },
                    { option: 'a', value: 'a, b, c, ...' },
                    { option: 'A', value: 'A, B, C, ...' }
                ]
            } : {
                date: [],
                time: []
            },
            // locale date formats
            categoryCodesDate = [],
            // collection of all fields in document
            allFields = [],
            // collection of all fields in normal slide view
            allNormalSlideFields = [],
            // collection of all date fields in document
            allDateFields = [],
            // collection of all footer fields in document
            allFooterFields = [],
            // collection of all page number fields in document
            allSlideNumFields = [],
            // collection of all page number fields only on master and layout slides in non-placeholder drawings
            allMasterNumFields = [],
            // collection of all date/time fields only on master and layout slides in non-placeholder drawings
            allMasterDateFields = [],
            // collection of all footer fields only on master slides (ODP only)
            allMasterFooterFields = [],
            // temporary collection of special complex field nodes, used durring changetracking
            tempSpecFieldCollection = [];

        // base constructors --------------------------------------------------

        TriggerObject.call(this, app);
        AppObjectMixin.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Private function called after importing document.
         * It loops all datetime fields in collection and:
         *  - updates content to current date WITHOUT sending the operations to server.
         */
        function updateDateTimeFields() {

            var date = numberFormatter.convertDateToNumber(DateUtils.makeUTCNow());
            var data = []; // collection of modified field contents, to send to slidepane
            var isODF = app.isODF();
            if (allDateFields.length) {
                _.each(allDateFields, function (dateFieldObj) {
                    if (!dateFieldObj.node.length) { return; }

                    var node = dateFieldObj.node;
                    var dateType = node.data('type');
                    var fieldHolder = null;
                    var formattedDate = isODF ? createDateFromFormat(node) : createDateFromType(dateType, date); // ooxml uses format such as "datetime2", odf classical "DD.MM.YYYY" variations
                    var isNotFixedDate = isODF ? !DOM.isFixedSimpleField(node, isODF) : true; // ooxml has only auto-update fields, odf can have fixed and auto-update

                    if (formattedDate && isNotFixedDate) { // if date field is not fixed, it's content is updated with new date string
                        fieldHolder = node.children();
                        if (fieldHolder.text() !== formattedDate) {
                            fieldHolder.text(formattedDate);
                            data.push({ slideID: dateFieldObj.slideID, fieldID: node.attr('data-fid'), value: formattedDate });
                        }
                    }
                });
                if (data.length) { // inform slidepane about new field values
                    model.trigger('slideDateFields:update', data);
                }
            }
        }

        /**
         * Private helper function to create localized current date from given field type (ex.: 'datetime2')
         *
         * @param {String} fieldType
         *
         * @param {Date} [date]
         *  Optional parameter, if not sent, date object will be created.
         *
         * @returns {String} created localized current date
         */
        function createDateFromType(fieldType, date) {

            var formatIndexStr = (fieldType && fieldType.match(/\d+/) && fieldType.match(/\d+/)[0]) || null;
            var formatIndex = (formatIndexStr && parseInt(formatIndexStr, 10)) || null;
            var localFormat = formatIndex  && localFormatList.date[formatIndex - 1] && localFormatList.date[formatIndex - 1].option;
            var formatCode = localFormat ? localFormat : LocaleData.SHORT_DATE;
            var parsedFormat = numberFormatter.getParsedFormat(formatCode);

            if (!date) {
                date = numberFormatter.convertDateToNumber(DateUtils.makeUTCNow());
            }

            return numberFormatter.formatValue(parsedFormat, date);
        }

        /**
         * Convinience function that fetches date format from node,
         * and uses it to convert and format current date to string.
         *
         * @param {jQuery} node
         *
         * @returns {String} created localized current date
         */
        function createDateFromFormat(node) {
            var nodeAttrs = node.data('attributes');
            var dateFormat = (nodeAttrs && nodeAttrs.character && nodeAttrs.character.field && nodeAttrs.character.field.dateFormat) || LocaleData.SHORT_DATE;

            return getDateTimeRepresentation(dateFormat);
        }

        /**
         * Handler for highlighting of fields, after selection has changed.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object.
         *
         * @param {Function} options
         *  Passed selection options.
         *
         */
        function checkHighlighting(event, options) {
            var isSimpleTextSelection = Utils.getBooleanOption(options, 'simpleTextSelection', false),
                isSlideSelection = Utils.getBooleanOption(options, 'slideSelection', false),
                domPos = null,
                node = null,
                atEndOfSpan = null,
                relevantEvents = ['mousedown', 'mouseup', 'touchstart', 'touchend'],
                browserEvent = (options && options.event) ? options.event : null,
                mouseTouchEvent = _.isObject(browserEvent) && _.contains(relevantEvents, browserEvent.type),
                rightClick = _.isObject(browserEvent) && (browserEvent.type === 'mousedown') && (browserEvent.button === 2);

            if (self.isEmpty() || (isSimpleTextSelection && !isSlideSelection) || selection.hasRange()) {

            } else {
                domPos = Position.getDOMPosition(selection.getRootNode(), selection.getStartPosition());
                if (domPos && domPos.node) {
                    node = domPos.node;
                    if (node.nodeType === 3) {
                        atEndOfSpan = domPos.offset === node.textContent.length;
                        node = node.parentNode;
                    }

                    if (atEndOfSpan && DOM.isFieldNode(node.nextSibling)) {
                        self.createEnterHighlight(node.nextSibling, { simpleField: true, mouseTouchEvent: mouseTouchEvent, rightClick: rightClick });
                    } else if (self.isHighlightState()) {
                        self.removeEnterHighlight();
                    }
                }
            }
        }

        /**
         * Create current date(time) value string with passed localized format code.
         *
         * @param {String} formatCode
         * @return {String}
         */
        function getDateTimeRepresentation(formatCode) {
            var parsedFormat = numberFormatter.getParsedFormat(formatCode);
            var serial = numberFormatter.convertDateToNumber(DateUtils.makeUTCNow());
            return numberFormatter.formatValue(parsedFormat, serial);
        }

        /**
         * Callback function for triggering event fielddatepopup:change, when value of datepicker of input field is changed.
         * @param {Event} event
         * @param {Object} options
         *   @param {String} options.value
         *       Changed value for the field.
         *   @param {String} options.format
         *       Format for the field.
         *   @param {String} options.standard
         *       Odf needs this standard date for fixed fields
         */
        function updateDateFromPopup(event, options) {

            var node,
                domPos,
                atEndOfSpan,
                fieldValue = (options && options.value) || null,
                fieldFormat = (options && options.format) || null,
                standardizedDate = (options && options.standard) || null;

            if (!fieldValue) {
                if (!fieldFormat) { return; } // no data to update, exit
                fieldValue = getDateTimeRepresentation(fieldFormat);
            }

            if (currentlyHighlightedNode) { // simple field
                node = currentlyHighlightedNode;
                simpleField.updateDateFromPopupValue(node, fieldValue, standardizedDate);

                // remove highlight from deleted node
                currentlyHighlightedNode = null;
                self.setCursorHighlightState(false);

                // add highlight to newly created node
                domPos = Position.getDOMPosition(selection.getRootNode(), selection.getStartPosition());
                if (domPos && domPos.node) {
                    node = domPos.node;
                    if (node.nodeType === 3) {
                        atEndOfSpan = domPos.offset === node.textContent.length;
                        node = node.parentNode;
                    }
                    if (atEndOfSpan && DOM.isFieldNode(node.nextSibling)) {
                        self.createEnterHighlight(node.nextSibling, { simpleField: true });
                    }
                }
            }
        }

        /**
         * Sets state of date field to be updated automatically or to be fixed value.
         * It is callback from field popup checkbox toggle.
         *
         * @param {Event} event
         *  jQuery Event that is triggered
         * @param {Boolean} state
         *  State of the field, fixed or updated automatically
         */
        function setDateFieldToAutoUpdate(event, state) {
            var node,
                pos,
                generator = model.createOperationGenerator(),
                operation = {},
                // target for operation - if exists, it's for ex. header or footer
                target = model.getActiveTarget(),
                representation,
                attrs = {},
                type;

            if (currentlyHighlightedNode) { // simple field
                node = currentlyHighlightedNode;
                representation = $(node).text();
            }
            if (!node) {
                Utils.warn('SlideField.setDateFieldToAutoUpdate(): node not fetched!');
                return;
            }
            pos = Position.getOxoPosition(selection.getRootNode(), node);

            if (pos) {
                return model.getUndoManager().enterUndoGroup(function () {
                    if (app.isODF()) {
                        type = 'date';
                        attrs = { field: { fixed: JSON.stringify(!state), dateFormat: DOM.getFieldDateTimeFormat(node) } };
                    } else {
                        type = DOM.getFieldInstruction(node);
                        attrs = { character: { autoDateField: JSON.stringify(state) } };
                    }
                    operation = { start: pos, type: type, representation: representation, attrs: attrs };
                    model.extendPropertiesWithTarget(operation, target);
                    generator.generateOperation(Operations.FIELD_UPDATE, operation);

                    model.applyOperations(generator.getOperations());
                });
            } else {
                Utils.warn('SlideField.setDateFieldToAutoUpdate(): invalid position for field node!');

            }
        }

        /**
         * Helper function that returns the required values for the footer dialog in ODF format.
         *
         * @returns {Array}
         *  An array, that contains objects for the footer place holder drawings of type 'dt', 'ftr'
         *  or 'sldNum'. One object for every footer place holder drawing, that is displayed on the
         *  affected document slide. If the drawing is not displayed, the array does not contain an
         *  object for this place holder type.
         */
        function getFieldOptionsOnSlideODF() {

            var // the ID of that slide, whose content will be displayed in the footer dialog
                slideId = null,
                // the attributes of 'slide' family
                slideFamilyAttrs = null,
                // a collector cointaining the footer place holder attributes of a document slide
                options = [];

            // getting the affected slide, whose content will be shown in the dialog:
            // - the active slide if this is not the master view
            // - the last active document slide, if this is the master view
            // - the first document slide, if no other slide can be determined (should never occur)
            if (model.isStandardSlideId(model.getActiveSlideId())) {
                slideId = model.getActiveSlideId();  // active slide in document view
            } else if (model.getLastActiveStandardSlideId()) {
                slideId = model.getLastActiveStandardSlideId(); // in master view, the last active slide is used
            } else {
                slideId = model.getStandardSlideOrder()[0]; // using the first standard slide (should never happen)
            }

            // getting the attributes of family 'slide' for the specfied slide
            slideFamilyAttrs = model.getSlideAttributesByFamily(slideId, 'slide') || {};

            // collecting the information in the 'options' collector
            if (slideFamilyAttrs.isSlideNum) { options.push({ type: 'sldNum' }); }

            if (slideFamilyAttrs.isFooter) { options.push({ type: 'ftr', ftrText: (slideFamilyAttrs.footerText ? slideFamilyAttrs.footerText : '') }); }

            if (slideFamilyAttrs.isDate) {
                var dateObject = { type: 'dt' };

                if (slideFamilyAttrs.dateField) {
                    dateObject.automatic = true;
                    dateObject.dtType = slideFamilyAttrs.dateField;
                    dateObject.reprensentation = slideFamilyAttrs.dateField;
                } else {
                    dateObject.automatic = false;
                    dateObject.dtText = slideFamilyAttrs.dateText;
                }

                options.push(dateObject);
            }

            return options;
        }

        /**
         * Helper function that returns which fields are present on slide with given id, or active slide.
         *
         * @param {String} [slideId]
         *  If ommited, active slide id is used.
         * @returns {Array}
         */
        function getFieldOptionsOnSlide(slideId) {

            var options = [];
            var tempObj = {};
            var drawingStyles = model.getDrawingStyles();
            var activeSlideId = slideId || model.getActiveSlideId();
            var ftr, sldNum, dt;
            var dtFields;
            var slideAttrs;

            if (model.isLayoutOrMasterId(activeSlideId)) {
                slideAttrs = model.getAllFamilySlideAttributes(activeSlideId);
                slideAttrs = slideAttrs && slideAttrs.slide;
                if (slideAttrs) {
                    if (slideAttrs.isFooter) {
                        ftr = drawingStyles.getAllPlaceHolderDrawingsOnSlideBySelector(activeSlideId, 'ftr');
                    }
                    if (slideAttrs.isSlideNum) {
                        sldNum = drawingStyles.getAllPlaceHolderDrawingsOnSlideBySelector(activeSlideId, 'sldNum');
                    }
                    if (slideAttrs.isDate) {
                        dt = drawingStyles.getAllPlaceHolderDrawingsOnSlideBySelector(activeSlideId, 'dt');
                    }
                }
            } else {
                ftr = drawingStyles.getAllPlaceHolderDrawingsOnSlideBySelector(activeSlideId, 'ftr');
                sldNum = drawingStyles.getAllPlaceHolderDrawingsOnSlideBySelector(activeSlideId, 'sldNum');
                dt = drawingStyles.getAllPlaceHolderDrawingsOnSlideBySelector(activeSlideId, 'dt');
            }

            if (sldNum && sldNum.length) { options.push({ type: 'sldNum' }); }
            if (ftr && ftr.length) {
                tempObj.type = 'ftr';
                tempObj.ftrText = ftr.text();
                options.push(tempObj);
                tempObj = {};
            }
            if (dt && dt.length) {
                tempObj.type = 'dt';
                dtFields = dt.find('.field');
                if (dtFields.length) {
                    _.each(dtFields, function (field) {
                        var type = DOM.getFieldInstruction(field);
                        if (type && type.indexOf('date') > -1) {
                            tempObj.dtType = type;
                            tempObj.automatic = true;
                            tempObj.representation = $(field).text();
                        }
                    });
                } else {
                    tempObj.dtText = dt.text();
                    tempObj.automatic = false;
                }
                options.push(tempObj);
            }

            return options;
        }

        /**
         * Helper to generate delete operation from passed options. Used in insertFooter and insertAllFooters
         *
         * @param {Object} drawingStyles
         *  instance of the DrawingStyles class.
         * @param {Array} deleteOptions
         *  list of delete options to be transformed to operations
         * @param {String} currentSlideId
         *  Id of the slide.
         * @returns {Array}
         */
        function getDelOpsFromOptions(drawingStyles, deleteOptions, currentSlideId) {
            var deleteOperations = [];
            var operation = null;
            _.each(deleteOptions, function (option) {
                var slideDrawing = drawingStyles.getAllPlaceHolderDrawingsOnSlideBySelector(currentSlideId, option.type);

                if (slideDrawing.length) {
                    operation = { start: Position.getOxoPosition(model.getNode(), slideDrawing) };
                    model.extendPropertiesWithTarget(operation, model.isLayoutOrMasterId(currentSlideId) ? currentSlideId : null);
                    deleteOperations.push(operation);
                }
            });
            if (deleteOperations.length) {
                deleteOperations = _.sortBy(deleteOperations, 'start');
                deleteOperations.reverse();
            }
            return deleteOperations;
        }

        // methods ------------------------------------------------------------

        /**
         * Inserts a simple text field component into the document DOM.
         *
         * @param {Number[]} start
         *  The logical start position for the new text field.
         *
         * @param {String} type
         *  A property describing the field type.
         *
         * @param {String} representation
         *  A fallback value, if the placeholder cannot be substituted with a
         *  reasonable value.
         *
         * @param {Object} [attrs]
         *  Attributes to be applied at the new text field, as map of attribute
         *  maps (name/value pairs), keyed by attribute family.
         *
         * @param {Boolean} [external]
         *  Will be set to true, if the invocation of this method originates
         *  from an external operation.
         *
         * @returns {Boolean}
         *  Whether the text field has been inserted successfully.
         */
        this.implInsertField = function (start, type, representation, attrs, target) {

            var // text span that will precede the field
                span = model.prepareTextSpanForInsertion(start, {}, target),
                // whether this is an ODF application
                isODF = app.isODF(),
                // new text span for the field node
                fieldSpan = null,
                // the field node
                fieldNode = null,
                // format of the field
                format,
                // the ID of the slide
                slideID = target || model.getSlideIdByPosition(start),
                // the field type for slide number
                slideNum = isODF ? 'page-number' : 'slidenum',
                // whether the page number field needs to be updated immediately
                updatePageNumber = true;

            if (!span) { return false; }

            // expanding attrs to disable change tracking, if not explicitely set
            attrs = model.checkChangesMode(attrs);

            if (!target && type === slideNum && representation) { // don't update slidenum field on slides with target (layout&master slides)
                representation = model.getSlideNumByID(slideID); // ensure to insert up-to-date slide number
            }

            // on ODF master slides, the fields representation must be visible immediately in footer fields
            if (isODF && target && !representation && _.contains(SlideField.ODF_FOOTER_FIELD_TYPES, type)) {
                representation = PresentationUtils.getODFFooterRepresentation(SlideField.ODF_FIELD_PLACEHOLDER_CONVERTER[type]);
            }

            // split the text span again to get initial character formatting
            // for the field, and insert the field representation text

            // Fix for 29265: Removing empty text node in span with '.contents().remove().end()'.
            // Otherwise there are two text nodes in span after '.text(representation)' in IE.
            fieldSpan = DOM.splitTextSpan(span, 0).contents().remove().end().text(representation);
            if (!representation) { DOM.ensureExistingTextNode(fieldSpan); }

            // insert a new text field before the addressed text node, move
            // the field span element into the field node
            fieldNode = DOM.createFieldNode();
            fieldNode.append(fieldSpan).insertAfter(span);
            fieldNode.data('type', type);

            // odf - field types needed to be updated in frontend
            if (type === 'page-number') {
                fieldNode.addClass('field-' + type);
                if (attrs && attrs.field && attrs.field.pageNumFormat) {
                    format = attrs.field.pageNumFormat;
                }
                updatePageNumber = !(model.useSlideMode() && model.isMasterView()); // not updating field on odp master slides
                if (updatePageNumber) { model.getPageLayout().updatePageNumberField(fieldNode, format); }
            }

            if (!representation) { fieldNode.addClass('empty-field'); }

            self.addSimpleFieldToCollection(fieldNode, slideID);

            // validate paragraph, store new cursor position
            model.implParagraphChanged(span.parentNode);
            model.setLastOperationEnd(Position.increaseLastIndex(start));
            return true;
        };

        /**
         * Updating field with given instruction.
         * Depending of passed instruction, different type of field is updated.
         *
         * @param {jQuery|Node} node
         *  Complex field node
         *
         * @param {String} instruction
         *  Complex field instruction, containing type,
         *  formating and optionally style, separated by \ .
         *
         */
        this.updateByInstruction = function (node, instruction) {

            if (instruction) {
                simpleField.updateByInstruction(node, instruction);
            } else {
                Utils.warn('SlideField.updateByInstruction(): missing instruction!');
            }
        };

        /**
         * Gets simple field from collection by passed id.
         *
         * @param {String} id
         *  Id of queried field.
         *
         * @returns {jQuery|null}
         *  The simple field with the specified id, or null, if no such start marker exists.
         *
         */
        this.getSimpleField = function (id) {
            var fieldObj = _.findWhere(allFields, { fid: id });
            return (fieldObj && fieldObj.node) || null;
        };

        /**
         * Check if there are simple fields in document.
         *
         * @returns {Boolean}
         */
        this.isSimpleFieldEmpty = function () {
            return self.isEmpty();
        };

        /**
         * Check if there are any type of fields in document.
         *
         * @returns {Boolean}
         */
        this.fieldsAreInDocument = function () {
            return !self.isEmpty();
        };

        /**
         * Warning: This accessor is used only for unit testing! Please use DOM's public method!
         *
         * @param {Node|jQuery|Null} node
         *  The DOM node to be checked. If this object is a jQuery collection, uses
         *  the first DOM node it contains. If missing or null, returns false.
         *
         * @param {Boolean} isODF
         *  If document format is odf, or ooxml.
         *
         * @returns {Boolean}
         *  If the field has fixed date property or not
         */
        this.isFixedSimpleField = function (field, isODF) {
            return DOM.isFixedSimpleField(field, isODF);
        };

        /**
         * Getting the collection with all date fields on master/layout slides, optionally filtered by a master or layout slide ID.
         *
         * @param {String} [filterSlideID=undefined]
         *  An ID of a master or layout slide to get only the date fields located on this master or layout slide.
         *
         * @returns {Object[]}
         *  The collection of all date/time fields on master and layout slides.
         */
        this.getAllMasterDateFields = function (filterSlideID) {
            return filterSlideID ? _.filter(allMasterDateFields, function (dateField) { return dateField.slideID === filterSlideID; }) : allMasterDateFields;
        };

        /**
         * Getting the collection with all slide number fields on master/layout slides, optionally filtered by a master or layout slide ID.
         *
         * @param {String} [filterSlideID=undefined]
         *  An ID of a master or layout slide to get only the slide number fields located on this master or layout slide.
         *
         * @returns {Object[]}
         *  The collection of all slide number fields on master and layout slides.
         */
        this.getAllMasterNumFields = function (filterSlideID) {
            return filterSlideID ? _.filter(allMasterNumFields, function (dateField) { return dateField.slideID === filterSlideID; }) : allMasterNumFields;
        };

        /**
         * Getting the collection with all footer fields on master/layout slides, optionally filtered by a master or layout slide ID.
         * This footer fields exist only in ODF applications.
         *
         * @param {String} [filterSlideID=undefined]
         *  An ID of a master or layout slide to get only the footer fields located on this master or layout slide.
         *
         * @returns {Object[]}
         *  The collection of all footer fields on master and layout slides.
         */
        this.getAllMasterFooterFields = function (filterSlideID) {
            return filterSlideID ? _.filter(allMasterFooterFields, function (dateField) { return dateField.slideID === filterSlideID; }) : allMasterFooterFields;
        };

        /**
         * Create current date(time) value string with passed format type.
         *
         * @param {String} formatCode
         *  The format type.
         *
         * @return {String}
         *  The current date(time) value string.
         */
        this.getDateTimeRepresentation = function (formatCode) {
            return getDateTimeRepresentation(formatCode);
        };

        /**
         * Creates highlight range on enter cursor in complex field,
         * with keyboard key pressed or mouse click.
         *
         * @param {jQuery} field
         *  Field to be highlighted.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.simpleField=false]
         *      If it's simple field or not.
         *  @param {Boolean} [options.mouseTouchEvent=false]
         *      If event is triggered by mouse or touch event.
         *  @param {Boolean} [options.rightClick=false]
         *      If its right click or not.
         */
        this.createEnterHighlight = function (field, options) {
            var
                isSimpleField = Utils.getBooleanOption(options, 'simpleField', false),
                isMouseTouchEvent = Utils.getBooleanOption(options, 'mouseTouchEvent', false),
                isRightClick = Utils.getBooleanOption(options, 'rightClick', false),
                fieldinstruction = null,
                node = null,
                fieldFormats,
                isDateTimeField,
                isSupportedFormat,
                isNotChangeTracked;

            options = {};

            if (isSimpleField) {
                if (currentlyHighlightedId || currentlyHighlightedNode) {
                    self.removeEnterHighlight();
                }
                currentlyHighlightedNode = $(field);
                currentlyHighlightedNode.addClass('sf-highlight');

                options.autoDate = !DOM.isFixedSimpleField(field, app.isODF());
                fieldinstruction = { type: DOM.getFieldInstruction(field) };

                if (fieldinstruction.type && app.isODF() && (simpleField.isCurrentDate(fieldinstruction.type) || simpleField.isCurrentTime(fieldinstruction.type))) {
                    fieldinstruction.instruction = DOM.getFieldDateTimeFormat(field);
                } else if (fieldinstruction.type && app.isODF() && simpleField.isPageNumber(fieldinstruction.type)) {
                    fieldinstruction.instruction = DOM.getFieldPageFormat(field);
                } else {
                    fieldinstruction = simpleField.cleanUpAndExtractType(fieldinstruction.type);
                }
                if (!fieldinstruction.instruction) {
                    fieldinstruction.instruction = 'default';
                }
            }

            if (fieldinstruction && fieldinstruction.type) {
                fieldinstruction.formats = localFormatList[fieldinstruction.type.toLowerCase()];
            }
            highlightedFieldType = fieldinstruction.type || null;
            highlightedFieldInstruction = fieldinstruction.instruction || null;
            fieldFormats = fieldinstruction.formats || null;

            self.setCursorHighlightState(true);
            node = field;

            self.destroyFieldPopup();
            isDateTimeField = (highlightedFieldType && (/^DATE|^TIME/i).test(highlightedFieldType)) || false;
            isSupportedFormat = (fieldFormats && fieldFormats.length) || isDateTimeField;
            isNotChangeTracked = !DOM.isChangeTrackNode(field) && !DOM.isChangeTrackNode($(field).parent());

            if (highlightedFieldType && isSupportedFormat && isMouseTouchEvent && !isRightClick && isNotChangeTracked) {
                fieldFormatPopupTimeout = self.executeDelayed(function () {
                    app.getView().showFieldFormatPopup(fieldinstruction, node, options);
                }, 'SlideField.fieldFormatPopupTimeout', 500);
            }

            // for debugging ->
//            if (!highlightedFieldType) {
//                Utils.error('no highlightedFieldType');
//            }
        };

        /**
         * Removes highlight range on enter cursor in complex field, or in simple field,
         * with keyboard key pressed or mouse click.
         */
        this.removeEnterHighlight = function () {
            if (currentlyHighlightedNode) {
                // simple field
                currentlyHighlightedNode.removeClass('sf-highlight');
                currentlyHighlightedNode = null;
            }
            self.setCursorHighlightState(false);
            highlightedFieldType = null;
            highlightedFieldInstruction = null;

            self.destroyFieldPopup();
        };

        /**
         * Set highlighting state of field during cursor traversal.
         *
         * @param {Boolean} state
         *  If cursor is inside field or not.
         */
        this.setCursorHighlightState = function (state) {
            isCursorHighlighted = state === true;
        };

        /**
         * If field highlight state is active, or not.
         *
         * @returns {Boolean}
         *  Whether the field highlight state is active, or not.
         */
        this.isHighlightState = function () {
            return isCursorHighlighted;
        };

        /**
         * Forces update of highlighted simple/complex field content.
         */
        this.updateHighlightedField = function () {
            var node,
                instruction,
                atEndOfSpan,
                domPos;

            if (currentlyHighlightedNode) { // simple field
                node = currentlyHighlightedNode;
                instruction = DOM.getFieldInstruction(node);
                self.updateByInstruction(node, instruction);

                // remove highlight from deleted node
                currentlyHighlightedNode = null;
                self.setCursorHighlightState(false);

                // add highlight to newly created node
                domPos = Position.getDOMPosition(selection.getRootNode(), selection.getStartPosition());
                if (domPos && domPos.node) {
                    node = domPos.node;
                    if (node.nodeType === 3) {
                        atEndOfSpan = domPos.offset === node.textContent.length;
                        node = node.parentNode;
                    }
                    if (atEndOfSpan && DOM.isFieldNode(node.nextSibling)) {
                        self.createEnterHighlight(node.nextSibling, { simpleField: true });
                    }
                }
            }
        };

        /**
         * Forces update of complex fields inside selection range.
         */
        this.updateHighlightedFieldSelection = function () {
            var instruction = null;
            if (self.isEmpty()) { return; }

            selection.iterateNodes(function (node) {
                if (DOM.isFieldNode(node)) {
                    instruction = DOM.getFieldInstruction(node);
                    self.updateByInstruction(node, instruction);
                }
            });
        };

        /**
         * Update all fields in document.
         *
         */
        this.updateAllFields = function () {
            var // the generate operations deferred
                generateDef = null;

            // helper iterator function for handling field update
            function handleFieldUpdate(entry, fieldId) {
                var field,
                    instruction;

                if (fieldId && fieldId.indexOf('s') > -1) { // simple field
                    field = self.getSimpleField(fieldId);
                    if (!DOM.isMarginalNode(field) || !DOM.isInsideHeaderFooterTemplateNode(model.getNode(), field)) {
                        instruction = DOM.getFieldInstruction(entry);
                        if (!DOM.isSpecialField(field)) {
                            self.updateByInstruction(field, instruction);
                        }
                    }
                }
            }

            if (self.fieldsAreInDocument()) {
                if (model.isHeaderFooterEditState()) { // #42671
                    model.getPageLayout().leaveHeaderFooterEditMode();
                    selection.setNewRootNode(model.getNode());
                }
                // show a message with cancel button
                app.getView().enterBusy({
                    cancelHandler: function () {
                        if (generateDef && generateDef.abort) {
                            generateDef.abort();
                        }
                    },
                    warningLabel: gt('Sorry, updating of all fields in document will take some time.')
                });

                // iterate objects
                generateDef = self.iterateObjectSliced(self.getAllFields(), handleFieldUpdate, 'SlideField.updateAllFields')
                .always(function () {
                    // leave busy state
                    app.getView().leaveBusy().grabFocus();
                });

                // add progress handling
                generateDef.progress(function (partialProgress) {
                    var progress = 0.2 + (partialProgress * 0.8);
                    app.getView().updateBusyProgress(progress);
                });
            }
        };

        /**
         * This method dispatch parent node to simple field, for removing it.
         *
         * @param {jQuery|Node} parentNode
         *  Element that's being searched for simple fields.
         */
        this.removeAllFieldsInNode = function (parentNode) {

            // checking, if the specified node contains additional simple field nodes, that need to be removed, too.
            //simpleField.removeSimpleFieldsInNode(parentNode);
            var // all simple field nodes inside the parentNode
                slideFieldNodes = $(parentNode).find(DOM.FIELD_NODE_SELECTOR);

            // update  the marker nodes in the collection objects
            _.each(slideFieldNodes, function (field) {
                self.removeSimpleFieldFromCollection(field);
            });
        };

        /**
         * Dispatch function to simpleField class's method removeSimpleFieldFromCollection.
         *
         * @param {HTMLElement|jQuery} field
         *  One simple field node.
         */
        this.removeSimpleFieldFromCollection = function (node) {
            var id = DOM.getSlideFieldId(node);

            function removeFieldInCollection(collection) { return _.filter(collection, function (obj) { return obj && obj.fid !== id; }); }

            allFields               = removeFieldInCollection(allFields);
            allNormalSlideFields    = removeFieldInCollection(allNormalSlideFields);
            allSlideNumFields       = removeFieldInCollection(allSlideNumFields);
            allDateFields           = removeFieldInCollection(allDateFields);
            allFooterFields         = removeFieldInCollection(allFooterFields);
            allMasterNumFields      = removeFieldInCollection(allMasterNumFields);
            allMasterDateFields     = removeFieldInCollection(allMasterDateFields);
            allMasterFooterFields   = removeFieldInCollection(allMasterFooterFields);
        };

        /**
         * Filling collections with presentation fields.
         *
         * @param {HTMLElement|jQuery} field
         *  One simple field node.
         *
         * @param {String} slideID
         *  ID of the slide, where the simple field is located.
         */
        this.addSimpleFieldToCollection = function (fieldNode, slideID) {
            var $fieldNode = $(fieldNode);
            var type = $fieldNode.data().type;
            var id = 'sf' + simpleFieldID;
            var isODF = app.isODF();
            var fieldObj = { slideID: slideID, node: $fieldNode, fid: id };
            var slideNum = isODF ? 'page-number' : 'slidenum';
            var dateField = isODF ? 'date' : 'datetime';
            var isDateFieldType = type && type.indexOf(dateField) > -1;

            if (!slideID) {
                Utils.error('slidefield.addSimpleFieldToCollection(): no slide id provided!');
                return;
            }

            allFields.push(fieldObj);

            if (model.isStandardSlideId(slideID)) {
                allNormalSlideFields.push(fieldObj);
                if (type === slideNum) {
                    allSlideNumFields.push(fieldObj);
                }
            } else if (type === slideNum && (isODF || !PresentationUtils.isPlaceHolderDrawing(DrawingFrame.getDrawingNode(fieldNode)))) {
                allMasterNumFields.push(fieldObj);
            } else if (isDateFieldType && (isODF || !PresentationUtils.isPlaceHolderDrawing(DrawingFrame.getDrawingNode(fieldNode)))) {
                allMasterDateFields.push(fieldObj);
            } else if (type === 'footer' && isODF) {
                allMasterFooterFields.push(fieldObj); // ODP only
            }

            if (isDateFieldType) {
                if (isODF) { fieldObj.node.removeClass('empty-field'); } // no need to handle 'empty-field' in ODP files
                allDateFields.push(fieldObj);
            } else if (type && type === 'footer') {
                fieldObj.node.removeClass('empty-field'); // no need to handle 'empty-field' in ODP files
                allFooterFields.push(fieldObj);
            }

            $fieldNode.attr('data-fid', id);
            simpleFieldID += 1;
        };

        /**
         * Refresh collection of fields after fastload and local storage.
         */
        this.refreshSlideFields = function () {
            var // page node
                pageNode = model.getNode(),
                // the page content node's children slides
                pageContentNodeSlides = DOM.getPageContentNode(pageNode).children(DOM.SLIDE_SELECTOR),
                // master slide layer node's children slide containers
                masterLayerSlideContainers = pageNode.children(DOM.MASTERSLIDELAYER_SELECTOR).children(DOM.SLIDECONTAINER_SELECTOR),
                // layout slide layer node's children slide containers
                layoutLayerSlideContainers = pageNode.children(DOM.LAYOUTSLIDELAYER_SELECTOR).children(DOM.SLIDECONTAINER_SELECTOR);

            // reset collections
            allFields = [];
            allNormalSlideFields = [];
            allDateFields = [];
            allSlideNumFields = [];
            allFooterFields = [];
            allMasterNumFields = [];
            allMasterDateFields = [];
            allMasterFooterFields = [];

            _.each(pageContentNodeSlides.add(masterLayerSlideContainers).add(layoutLayerSlideContainers), function (slide) {
                var slideID = DOM.getTargetContainerId(slide);
                _.each($(slide).find(DOM.FIELD_NODE_SELECTOR), function (fieldNode) {
                    self.addSimpleFieldToCollection(fieldNode, slideID);
                });
            });
        };

        /**
         * After splitting a paragraph, it is necessary, that all complex field nodes in the cloned
         * 'new' paragraph are updated in the collectors.
         * This method dispatches to complex field class.
         *
         * @param {Node|jQuery} paragraph
         *  The paragraph node.
         */
        this.updateSimpleFieldCollector = function () {
            //simpleField.updateSimpleFieldCollector(paragraph);
        };

        /**
         * Dispatches to handler for updateField operations.
         *
         * @param {Number[]} start
         *  The logical start position for the new simple field.
         *
         * @param {String} type
         *  Type of the simple field.
         *
         * @param {String} representation
         *  Content of the field that's updated.
         *
         * @param {String} [target]
         *  The target corresponding to the specified logical start position.
         *
         * @returns {Boolean}
         *  Whether the simple field has been inserted successfully.
         */
        this.updateSimpleFieldHandler = function (start, type, representation, attrs, target) {
            return simpleField.updateSimpleFieldHandler(start, type, representation, attrs, target);
        };

        /**
         * Dispatch update of date time fields on download to field classes.
         */
        this.prepareDateTimeFieldsDownload = function () {

            // helper iterator function for handling field update
            function handleFieldUpdate(entry, fieldId) {
                var field,
                    instruction;

                self.destroyFieldPopup();

                if (fieldId && fieldId.indexOf('s') > -1) { // simple field
                    field = self.getSimpleField(fieldId);
                    if (field) {
                        instruction = DOM.getFieldInstruction(entry);
                        simpleField.updateDateTimeField(field, instruction);
                    }
                }
            }

            _.each(self.getAllFields(), handleFieldUpdate);
        };

        /**
         * Method to update all slide number fields, triggered after slides are moved, inserted, or deleted.
         */
        this.updateSlideNumFields = function () {

            var isODF         = app.isODF();
            var isMasterView  = model.isMasterView();

            var data = [];
            var targetChain = model.getTargetChain(model.getActiveSlideId());

            _.each(allSlideNumFields, function (slideNumObj) {
                var node = slideNumObj.node,
                    fieldHolder = null,
                    representation = model.getSlideNumByID(slideNumObj.slideID);

                if (node.length && representation) {
                    fieldHolder = node.children();
                    if (fieldHolder.text() !== representation + '') {
                        fieldHolder.text(representation);
                        data.push({ slideID: slideNumObj.slideID, fieldID: node.attr('data-fid'), value: representation });
                    }
                }
            });

            _.each(allMasterNumFields, function (slideNumObj) {
                var node = slideNumObj.node;
                var representation = isMasterView ? '<#>' : model.getSlideNumByID(model.getActiveSlideId());
                var fieldHolder;

                if (node.length && representation) {
                    if (_.contains(targetChain, slideNumObj.slideID)) {
                        fieldHolder = node.children();
                        if (fieldHolder.text() !== representation + '') {
                            fieldHolder.text(representation);
                        }
                    }
                }

                _.each(model.getAllSlideIDsWithSpecificTargetAncestor(slideNumObj.slideID), function (oneId) {
                  //if (isODF && isMasterView && model.isStandardSlideId(oneId)) { return; }
                    if (!isODF || !isMasterView || !model.isStandardSlideId(oneId)) {

                        var tempVal = isMasterView ? '<#>' : model.getSlideNumByID(oneId);
                        if (tempVal) {
                            data.push({
                                slideID: oneId,
                                fieldID: node.attr('data-fid'),
                                value: tempVal,
                                masterViewField: true
                            });
                        }
                    }
                });
            });

            if (data.length) {
                model.trigger('slideNumFields:update', data);
            }
        };

        /**
         * Method to update slide number fields on master and layout slides, when slide view is changed.
         * In ppt they are in non-placeholder drawings only, in odf, in placeholder also.
         */
        this.updateMasterSlideFields = function () {

            var // the ID of the active slide
                activeSlideId = model.getActiveSlideId(),
                // the complete target chain for the active slide
                targetChain = model.getTargetChain(activeSlideId),
                // the 'slide' family attributes of the active slide
                slideFamilyAttrs = null;

            _.each(allMasterNumFields, function (slideNumObj) {
                var node = slideNumObj.node,
                    fieldHolder, representation;

                if (_.contains(targetChain, slideNumObj.slideID)) {
                    representation = model.isMasterView() ? '<#>' : model.getSlideNumByID(activeSlideId);
                    if (node.length && representation) {
                        fieldHolder = node.children();
                        if (fieldHolder.text() !== representation + '') {
                            fieldHolder.text(representation);
                        }
                    }
                }
            });

            // in ODF format also the fields of type 'footer' and 'date' need to be updated on the master slides.

            if (app.isODF()) {

                // getting the attributes of family 'slide' of the document slide, so that the footer fields can be updated
                slideFamilyAttrs = (model.isStandardSlideId(activeSlideId) ? model.getSlideAttributesByFamily(activeSlideId, 'slide') : {}) || {};

                _.each(allFooterFields, function (footerObj) {

                    var node = footerObj.node,
                        fieldHolder, representation;

                    if (_.contains(targetChain, footerObj.slideID)) {
                        representation = model.isMasterView() ? PresentationUtils.getODFFooterRepresentation('ftr') : (slideFamilyAttrs.footerText || '');
                        if (node.length) {
                            fieldHolder = node.children();
                            if (fieldHolder.text() !== representation + '') {
                                fieldHolder.text(representation);
                            }
                        }
                    }

                });

                // handling of date fields inside footer placeholder drawing on master slide
                var masterDateFieldsInFooter = _.filter(allMasterDateFields, function (fieldObj) {
                    var node = fieldObj.node;
                    var drawing = DrawingFrame.getDrawingNode(node);
                    var drawingType = PresentationUtils.getPlaceHolderDrawingType(drawing);

                    return _.contains(model.getPresentationFooterTypes(), drawingType);
                });
                _.each(masterDateFieldsInFooter, function (footerObj) {

                    var node = footerObj.node,
                        fieldHolder, representation;

                    if (model.isMasterView()) {
                        representation = PresentationUtils.getODFFooterRepresentation('dt');
                    } else {
                        representation = (slideFamilyAttrs.dateField ? getDateTimeRepresentation(slideFamilyAttrs.dateField) : slideFamilyAttrs.dateText) || '';
                    }

                    if (node.length) {
                        fieldHolder = node.children();
                        if (fieldHolder.text() !== representation + '') {
                            fieldHolder.text(representation);
                        }
                    }

                });

                // TODO: Updating the slide pane to display content correctly
                // -> But not calling slide pane update on every slide change (only at start and if slide attributes have changed)!

            }

        };

        /**
         * Depending of selected value, insert slide number field, or open date field dialog.
         *
         * @param {String} value
         *  Value of dropdown menu for insert field. Can be 'slidenum' or 'datetime' (or 'date-time' in ODF).
         *
         * @returns {InsertDateFieldDialog}
         */
        this.dispatchInsertField = function (value) {
            if (value === 'slidenum') {
                self.insertField(value);
            } else {
                return new InsertDateFieldDialog(app.getView()).show(); // open dialog with date formats
            }
        };

        /**
         * Opens insert footer dialog.
         *
         * @returns {InsertFooterDialog}
         */
        this.openFooterDialog = function () {
            return new InsertFooterDialog(app.getView(), (app.isODF() ? getFieldOptionsOnSlideODF() : getFieldOptionsOnSlide())).show(); // open dialog with date formats
        };

        /**
         * Insert simple fields.
         *
         * @param {String} fieldType
         *  Can be slide number or date.
         * @param {Object} [options]
         *  @param {String} [fieldFormat]
         *      Optional parameter used for formatting date fields.
         *  @param {Boolean} [automatic]
         *      Optional parameter used to decide if date field is automatically updated, or fixed value.
         *  @param {Number} [formatIndex]
         *      Optional parameter used to construct datetime type.
         */
        this.insertField = function (fieldType, options) {
            var fieldFormat = Utils.getStringOption(options, 'fieldFormat', '');
            var automatic = Utils.getBooleanOption(options, 'automatic', false);
            var formatIndex = Utils.getNumberOption(options, 'formatIndex', 1);

            return model.getUndoManager().enterUndoGroup(function () {
                function doInsertField() {
                    var start = selection.getStartPosition(),
                        slideID = model.getActiveSlideId(),
                        generator = model.createOperationGenerator(),
                        operation = {},
                        representation = null,
                        repLength = 1,
                        isSelectionInTextFrame = selection.isAdditionalTextframeSelection(),
                        dateTimeType = app.isODF() ? 'date' : 'datetime';

                    if (fieldType) {
                        if (!isSelectionInTextFrame) {
                            model.insertTextFrame();
                            start = selection.getStartPosition();
                        } else {
                            model.doCheckImplicitParagraph(start);
                        }
                        if (fieldType === dateTimeType && !automatic) {
                            fieldFormat = fieldFormat || LocaleData.SHORT_DATE;
                            representation = getDateTimeRepresentation(fieldFormat);
                            operation = { start: start, text: representation };
                            generator.generateOperation(Operations.TEXT_INSERT, operation);
                            repLength = representation.length;
                        } else {
                            operation = { start: start, type: fieldType };
                            if (fieldType === dateTimeType) {
                                fieldFormat = fieldFormat || LocaleData.SHORT_DATE;
                                representation = getDateTimeRepresentation(fieldFormat);
                                operation.type = fieldType + (app.isODF() ? '' : formatIndex);
                                if (app.isODF()) {
                                    operation.attrs = { character: { field: { dateFormat: fieldFormat } } };
                                }
                            } else if (fieldType === 'slidenum') {
                                representation = model.getSlideNumByID(slideID) || '<#>'; // hash tag for master and layout slides
                                representation += ''; // convert to string if number
                                operation.type = app.isODF() ? 'page-number' : 'slidenum';
                            }
                            operation.representation = representation;
                            generator.generateOperation(Operations.FIELD_INSERT, operation);
                        }
                        model.applyOperations(generator.getOperations());
                        selection.setTextSelection(Position.increaseLastIndex(start, repLength));
                    }
                }

                if (selection.hasRange()) {
                    return model.deleteSelected()
                    .done(function () {
                        doInsertField();
                    });
                }

                doInsertField();
                return $.when();
            });
        };

        /**
         * Clicking on remove button in popup, removes highlighted simple or complex field.
         */
        this.removeField = function () {
            var node,
                rootNode = selection.getRootNode(),
                startPos;

            // hide popup
            app.getView().hideFieldFormatPopup();
            if (fieldFormatPopupTimeout) {
                fieldFormatPopupTimeout.abort();
            }

            if (currentlyHighlightedNode) { // simple field
                node = currentlyHighlightedNode;
                startPos = Position.getOxoPosition(rootNode, node);

                if (startPos) {
                    return model.getUndoManager().enterUndoGroup(function () {
                        model.deleteRange(startPos);
                    });
                }
            }
        };

        /**
         * Helper function used to make diff between starting state of checkboxes and state on clicking apply button.
         *
         * @param {Object} startOptions
         *  Options on the start state, before clicking apply button.
         * @param {Object} endOptions
         *  Options on the end state, on clicking apply button.
         * @returns {Array}
         */
        this.diffStateCheckboxes = function (startOptions, endOptions) {
            var diff = [];
            _.each(startOptions, function (startOpt) {
                if (!_.findWhere(endOptions, { type: startOpt.type })) {
                    diff.push(startOpt);
                }
            });

            return diff;
        };

        /**
         * Helper function used to make diff between starting state of checkboxes and state on clicking apply button.
         *
         * @param {Object} startOptions
         *  Options on the start state, before clicking apply button.
         * @param {Object} endOptions
         *  Options on the end state, on clicking apply button.
         * @returns {Array}
         */
        this.diffInputContent = function (startOptions, endOptions) {
            var computedOptions = [];
            _.each(endOptions, function (endOpt) {
                if (_.isString(endOpt.ftrText) && !_.findWhere(startOptions, { ftrText: endOpt.ftrText })) {
                    computedOptions.push(endOpt);
                } else if (endOpt.representation && endOpt.automatic === true && !_.findWhere(startOptions, { representation: endOpt.representation, automatic: true })) {
                    computedOptions.push(endOpt);
                } else if (_.isString(endOpt.representation) && endOpt.automatic === false && !_.findWhere(startOptions, { dtText: endOpt.representation, automatic: false })) {
                    computedOptions.push(endOpt);
                } else if (endOpt.type === 'sldNum' && !_.findWhere(startOptions, { type: 'sldNum' })) {
                    computedOptions.push(endOpt);
                }
            });

            return computedOptions;
        };

        /**
         * Insert footer with fields like date, slide number or footer text (ODF only). These values are
         * always assigned to document slides, never to master slides (but the content is inserted into
         * fields that exist only on master slides).
         *
         * @param {Array} options
         *  List of options, which placeholder fields in footer to insert, and proprietary content.
         * @param {Array} deleteOptions
         *  List of options, which placeholder fields in footer to delete.
         */
        this.insertFooterODF = function (checkedOptions, allSlides) {

            var // a container for all affected slide IDs
                slides = null,
                // the current slide pane selection
                slidePaneSelection = null,
                // the operations generator
                generator = model.createOperationGenerator();

            // generating a list of affected slides:
            // - all document slides (user checked 'Apply to all')
            // - the active slide if this is not the master view
            // - the last active document slide, if this is the master view
            // - the first document slide, if no other slide can be determined (should never occur)
            if (allSlides) {
                slides = model.getStandardSlideOrder(); // user checked 'Apply to all'
            } else {
                if (model.isStandardSlideId(model.getActiveSlideId())) {
                    slidePaneSelection = model.getSlidePaneSelection();
                    if (slidePaneSelection && slidePaneSelection.length > 1) {
                        slides = [];
                        slidePaneSelection.sort(function (a, b) { return a - b; }); // more than one selected slide in document view
                        _.each(slidePaneSelection, function (slideIndex) {
                            slides.push(model.getIdOfSlideOfActiveViewAtIndex(slideIndex));
                        });
                    } else {
                        slides = [model.getActiveSlideId()];  // active slide in document view
                    }
                } else if (model.getLastActiveStandardSlideId()) {
                    slides = [model.getLastActiveStandardSlideId()]; // in master view, the last active slide is used
                } else {
                    slides = [model.getStandardSlideOrder()[0]];  // using the first standard slide (should never happen)
                }
            }

            // iterating over the affected standard slides and generate setAttributes operations.
            // No delete or insertDrawing operation needs to be created in ODF format.
            if (slides) {

                _.each(slides, function (slideId) {

                    var // the logical position of the slide
                        pos = model.getSlidePositionById(slideId),
                        // the attribute object for all slide footer attributes (with all values resetted)
                        slideFooterAttrs = { isDate: false, isFooter: false, isSlideNum: false, footerText: null, dateText: null, dateField: null },
                        // the operation object for the setAttributes operation
                        operation = null;

                    _.each(checkedOptions, function (oneTypeObj) {

                        var // the footer type. Supported values: 'dt', 'sldNum', 'ftr'
                            type = oneTypeObj.type;

                        switch (type) {
                            case 'dt':
                                slideFooterAttrs.isDate = true;
                                if (oneTypeObj.automatic) {
                                    if (oneTypeObj.representation) { slideFooterAttrs.dateField = oneTypeObj.representation; }
                                } else {
                                    if (oneTypeObj.representation) { slideFooterAttrs.dateText = oneTypeObj.representation; }
                                }
                                break;
                            case 'sldNum':
                                slideFooterAttrs.isSlideNum = true;
                                break;
                            case 'ftr':
                                slideFooterAttrs.isFooter = true;
                                if (oneTypeObj.ftrText) { slideFooterAttrs.footerText = oneTypeObj.ftrText; }
                                break;
                        }

                    });

                    operation = { start: pos, attrs: { slide: slideFooterAttrs } };
                    if (model.isMasterView()) { operation.target = model.getOperationTargetBlocker(); } // avoid setting of target for document slide, if master view is active
                    generator.generateOperation(Operations.SET_ATTRIBUTES, operation);
                });

                model.applyOperations(generator.getOperations());

                // updating the currently active document slide immediately
                self.updateMasterSlideFields();
            }
        };

        /**
         * Insert footer with fields like date, slide number or footer text (not ODF).
         *
         * @param {Array} options
         *  List of options, which placeholder fields in footer to insert, and proprietary content.
         *
         * @param {Array} deleteOptions
         *  List of options, which placeholder fields in footer to delete.
         */
        this.insertFooter = function (options, deleteOptions) {
            var drawingStyles = model.getDrawingStyles();
            var activeSlideId = model.getActiveSlideId();
            var generator = model.createOperationGenerator();
            var operation = {};
            var isDate = false;
            var isSlideNum = false;
            var isFooter = false;
            var listOfSlideIds = [];
            var operationsDef;
            // a snapshot object
            var snapshot = null;

            function generateOpsFromOptions(currentSlideId) {
                var start = null;
                var nextAvailablePosition = null;

                _.each(options, function (opt) {
                    var paraPos;
                    var fieldPos = null;
                    var fieldType = opt.type;
                    var layoutId = model.getLayoutSlideId(currentSlideId);
                    var layoutDrawing = layoutId && drawingStyles.getAllPlaceHolderDrawingsOnSlideBySelector(layoutId, fieldType);
                    var elementAttrs = layoutDrawing && layoutDrawing.length && AttributeUtils.getExplicitAttributes(layoutDrawing); // getElementAttributes is not necessary
                    var slideDrawing = drawingStyles.getAllPlaceHolderDrawingsOnSlideBySelector(currentSlideId, fieldType);
                    var slideDrawingContent = null;
                    var len = null;

                    operation = {};
                    if (slideDrawing.length) {
                        if (fieldType === 'sldNum' && model.isLayoutOrMasterId(currentSlideId)) {
                            return; // not needed for slide number fields on layout and master slides
                        }

                        start = Position.getOxoPosition(model.getNode(), slideDrawing);
                        if (_.isArray(start)) {
                            slideDrawingContent = DOM.getChildContainerNode(slideDrawing);
                            len = slideDrawingContent.children().length;
                            while (len > 0) {
                                operation = { start: start.concat([len - 1]) };
                                model.extendPropertiesWithTarget(operation, model.isLayoutOrMasterId(currentSlideId) ? currentSlideId : null);
                                generator.generateOperation(Operations.DELETE, operation);
                                len -= 1;
                            }
                        }
                    } else {
                        if (nextAvailablePosition) {
                            start = Position.increaseLastIndex(nextAvailablePosition);
                            nextAvailablePosition = start;
                        } else {
                            nextAvailablePosition = model.getNextAvailablePositionInSlide(currentSlideId);
                            start = nextAvailablePosition;
                        }

                        // clear unnecessary properties
                        if (elementAttrs) {
                            // text frames are drawings of type shape
                            operation = { attrs: elementAttrs, start: start, type: 'shape' };
                            model.extendPropertiesWithTarget(operation, model.isLayoutOrMasterId(currentSlideId) ? currentSlideId : null);
                            generator.generateOperation(Operations.INSERT_DRAWING, operation);
                        } else {
                            return;
                        }
                    }

                    // add a paragraph into the shape
                    paraPos = _.clone(start);
                    paraPos.push(0);
                    fieldPos = _.clone(paraPos);
                    operation = { start: paraPos };
                    model.extendPropertiesWithTarget(operation, model.isLayoutOrMasterId(currentSlideId) ? currentSlideId : null);
                    generator.generateOperation(Operations.PARA_INSERT, operation);

                    if (fieldType) {
                        fieldPos.push(0);
                        operation = { start: fieldPos };
                        if (fieldType === 'dt') {
                            if (opt.automatic) {
                                operation.type = 'datetime' + opt.formatIndex;
                                operation.representation = opt.representation;
                                model.extendPropertiesWithTarget(operation, model.isLayoutOrMasterId(currentSlideId) ? currentSlideId : null);
                                generator.generateOperation(Operations.FIELD_INSERT, operation);
                            } else {
                                if (_.isString(opt.representation) && opt.representation.length > 0) {
                                    operation.text = opt.representation;
                                    model.extendPropertiesWithTarget(operation, model.isLayoutOrMasterId(currentSlideId) ? currentSlideId : null);
                                    generator.generateOperation(Operations.TEXT_INSERT, operation);
                                }
                            }
                        } else if (fieldType === 'sldNum') {
                            operation.type = app.isODF() ? 'page-number' : 'slidenum';
                            operation.representation = model.getSlideNumByID(currentSlideId) + ''; // convert to string if number
                            model.extendPropertiesWithTarget(operation, model.isLayoutOrMasterId(currentSlideId) ? currentSlideId : null);
                            generator.generateOperation(Operations.FIELD_INSERT, operation);
                        } else if (fieldType === 'ftr') {
                            if (_.isString(opt.ftrText) && opt.ftrText.length > 0) {
                                operation.text = opt.ftrText;
                                model.extendPropertiesWithTarget(operation, model.isLayoutOrMasterId(currentSlideId) ? currentSlideId : null);
                                generator.generateOperation(Operations.TEXT_INSERT, operation);
                            }
                        }
                    }
                });

                // at the end, delete unchecked fields
                if (!model.isLayoutOrMasterId(currentSlideId)) {
                    _.each(getDelOpsFromOptions(drawingStyles, deleteOptions, currentSlideId), function (operation) {
                        generator.generateOperation(Operations.DELETE, operation);
                    });
                }
            }

            if (model.isLayoutOrMasterId(activeSlideId)) {
                _.each(options, function (opt) {
                    if (opt.type === 'dt') { isDate = true; }
                    if (opt.type === 'sldNum') { isSlideNum = true; }
                    if (opt.type === 'ftr') { isFooter = true; }
                });
                operation = { start: [0], target: activeSlideId, attrs: { slide: { isDate: isDate, isFooter: isFooter, isHeader: false, isSlideNum: isSlideNum } } };
                generator.generateOperation(Operations.SET_ATTRIBUTES, operation);

                if (model.isLayoutSlideId(activeSlideId)) {
                    listOfSlideIds =  model.getAllSlideIDsWithSpecificTargetParent(activeSlideId);
                } else {
                    listOfSlideIds = model.getAllSlideIDsWithSpecificTargetAncestor(activeSlideId);
                }
            }
            generateOpsFromOptions(activeSlideId);
            _.each(listOfSlideIds, function (currId) {
                generateOpsFromOptions(currId);
            });

            if (model.isLayoutOrMasterId(activeSlideId)) {
                // blocking keyboard input during applying of operations
                model.setBlockKeyboardEvent(true);

                // creating snapshot so that the document is restored after cancelling action
                snapshot = new Snapshot(app);

                app.getView().enterBusy({
                    cancelHandler: function () {
                        if (operationsDef && operationsDef.abort) {
                            snapshot.apply();  // restoring the old state
                            app.enterBlockOperationsMode(function () { operationsDef.abort(); }); // block sending of operations
                        }
                    },
                    immediate: true,
                    warningLabel: /*#. shown while applying selected properties from dialog to all slides in document */ gt('Applying footer fields to all slides, please wait...')
                });

                // fire apply operations asynchronously
                operationsDef = model.applyOperations(generator.getOperations(), { async: true });

                // handle the result of footer fields operations
                operationsDef
                    .progress(function (progress) {
                        // update the progress bar according to progress of the operations promise
                        app.getView().updateBusyProgress(progress);
                    })
                    .always(function () {
                        app.getView().leaveBusy();
                        // allowing keyboard events again
                        model.setBlockKeyboardEvent(false);
                        if (snapshot) { snapshot.destroy(); }
                    });
            } else {
                model.applyOperations(generator.getOperations());
            }
        };

        /**
         * Insert footers with fields like date, slide num or text, but in contrary to insertFooter function,
         * this applys to all standard slides in document.
         *
         *
         * @param {Array} endOptions
         *  List of footer options which user has chosen for all slides.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the insertion of all footers
         *  ended. It is rejected, if the dialog has been canceled.
         */
        this.insertAllFooters = function (endOptions) {
            var drawingStyles = model.getDrawingStyles();
            var masterIds = _.keys(model.getAllGroupSlides('master'));
            var layoutIds = _.keys(model.getAllGroupSlides('layout'));
            var standardIds = _.keys(model.getAllGroupSlides('standard'));
            var generator = model.createOperationGenerator();
            var operation = {};
            var isDate = false;
            var isFooter = false;
            var isSlideNum = false;
            var operationsDef = null;
            // a snapshot object
            var snapshot = null;

            function handleOneSlide(currentSlideId) {
                var start = null;
                var nextAvailablePosition = null;
                var startOptions = getFieldOptionsOnSlide(currentSlideId);
                var diffContentOptions = self.diffInputContent(startOptions, endOptions);
                // if some checkbox states are dechecked, this fields needs to be deleted
                var diffStateOptions = self.diffStateCheckboxes(startOptions, endOptions);

                _.each(diffContentOptions, function (opt) {
                    var paraPos;
                    var fieldPos = null;
                    var fieldType = opt.type;
                    var slideLayoutId = model.getLayoutSlideId(currentSlideId);
                    var layoutDrawing = slideLayoutId && drawingStyles.getAllPlaceHolderDrawingsOnSlideBySelector(slideLayoutId, fieldType);
                    var elementAttrs = layoutDrawing && layoutDrawing.length && AttributeUtils.getExplicitAttributes(layoutDrawing);
                    var slideDrawing = drawingStyles.getAllPlaceHolderDrawingsOnSlideBySelector(currentSlideId, fieldType);
                    var slideDrawingContent = null;
                    var len = null;

                    if (model.isLayoutOrMasterId(currentSlideId) && !elementAttrs) {
                        elementAttrs = slideDrawing && slideDrawing.length && AttributeUtils.getExplicitAttributes(slideDrawing);
                    }

                    // appending the slide into the DOM, if necessary
                    model.appendSlideToDom(currentSlideId);

                    if (slideDrawing.length) {
                        start = Position.getOxoPosition(model.getNode(), slideDrawing);
                        if (_.isArray(start)) {
                            slideDrawingContent = DOM.getChildContainerNode(slideDrawing);
                            len = slideDrawingContent.children().length;
                            while (len > 0) {
                                operation = { start: start.concat([len - 1]) };
                                model.extendPropertiesWithTarget(operation, model.isLayoutOrMasterId(currentSlideId) ? currentSlideId : null);
                                generator.generateOperation(Operations.DELETE, operation);
                                len -= 1;
                            }
                        }
                    } else {
                        if (nextAvailablePosition) {
                            start = Position.increaseLastIndex(nextAvailablePosition);
                            nextAvailablePosition = start;
                        } else {
                            nextAvailablePosition = model.getNextAvailablePositionInSlide(currentSlideId);
                            start = nextAvailablePosition;
                        }

                        // clear unnecessary properties
                        if (elementAttrs) {
                            // text frames are drawings of type shape
                            operation = { attrs: elementAttrs, start: start, type: 'shape' };
                            model.extendPropertiesWithTarget(operation, model.isLayoutOrMasterId(currentSlideId) ? currentSlideId : null);
                            generator.generateOperation(Operations.INSERT_DRAWING, operation);
                        } else {
                            return;
                        }
                    }

                    if (!start) { // this should never happen
                        Utils.error('SlideField.insertAllFooters(): failed to find valid start position!');
                        return;
                    }

                    // add a paragraph into the shape
                    paraPos = _.clone(start);
                    paraPos.push(0);
                    fieldPos = _.clone(paraPos);
                    operation = { start: paraPos };
                    model.extendPropertiesWithTarget(operation, model.isLayoutOrMasterId(currentSlideId) ? currentSlideId : null);
                    generator.generateOperation(Operations.PARA_INSERT, operation);

                    if (fieldType) {
                        fieldPos.push(0);
                        operation = { start: fieldPos };
                        if (fieldType === 'dt') {
                            if (opt.automatic) {
                                operation.type = 'datetime' + opt.formatIndex;
                                operation.representation = opt.representation;
                                model.extendPropertiesWithTarget(operation, model.isLayoutOrMasterId(currentSlideId) ? currentSlideId : null);
                                generator.generateOperation(Operations.FIELD_INSERT, operation);
                            } else {
                                if (_.isString(opt.representation) && opt.representation.length > 0) {
                                    operation.text = opt.representation;
                                    model.extendPropertiesWithTarget(operation, model.isLayoutOrMasterId(currentSlideId) ? currentSlideId : null);
                                    generator.generateOperation(Operations.TEXT_INSERT, operation);
                                }
                            }
                        } else if (fieldType === 'sldNum') {
                            operation.type = app.isODF() ? 'page-number' : 'slidenum';
                            operation.representation = model.isLayoutOrMasterId(currentSlideId) ? '<#>' : (model.getSlideNumByID(currentSlideId) + ''); // convert to string if number
                            model.extendPropertiesWithTarget(operation, model.isLayoutOrMasterId(currentSlideId) ? currentSlideId : null);
                            generator.generateOperation(Operations.FIELD_INSERT, operation);
                        } else if (fieldType === 'ftr') {
                            if (_.isString(opt.ftrText) && opt.ftrText.length > 0) {
                                operation.text = opt.ftrText;
                                model.extendPropertiesWithTarget(operation, model.isLayoutOrMasterId(currentSlideId) ? currentSlideId : null);
                                generator.generateOperation(Operations.TEXT_INSERT, operation);
                            }
                        }
                    }
                });

                // at the end, delete unchecked fields
                if (!model.isLayoutOrMasterId(currentSlideId)) {
                    _.each(getDelOpsFromOptions(drawingStyles, diffStateOptions, currentSlideId), function (operation) {
                        generator.generateOperation(Operations.DELETE, operation);
                    });
                }
            }

            _.each(endOptions, function (opt) {
                if (opt.type === 'dt') { isDate = true; }
                if (opt.type === 'sldNum') { isSlideNum = true; }
                if (opt.type === 'ftr') { isFooter = true; }
            });

            _.each(masterIds, function (masterId) {
                operation = { start: [0], target: masterId, attrs: { slide: { isDate: isDate, isFooter: isFooter, isHeader: false, isSlideNum: isSlideNum } } };
                generator.generateOperation(Operations.SET_ATTRIBUTES, operation);
                handleOneSlide(masterId);
            });

            _.each(layoutIds, function (layoutId) {
                operation = { start: [0], target: layoutId, attrs: { slide: { isDate: isDate, isFooter: isFooter, isHeader: false, isSlideNum: isSlideNum } } };
                generator.generateOperation(Operations.SET_ATTRIBUTES, operation);
                handleOneSlide(layoutId);
            });

            _.each(standardIds, function (standardId) {
                handleOneSlide(standardId);
            });

            // blocking keyboard input during applying of operations
            model.setBlockKeyboardEvent(true);

            // creating snapshot so that the document is restored after cancelling action
            snapshot = new Snapshot(app);

            // fire apply operations asynchronously
            operationsDef = model.applyOperations(generator.getOperations(), { async: true });

            app.getView().enterBusy({
                cancelHandler: function () {
                    if (operationsDef && operationsDef.abort) {
                        snapshot.apply();  // restoring the old state
                        app.enterBlockOperationsMode(function () { operationsDef.abort(); }); // block sending of operations
                    }
                },
                immediate: true,
                warningLabel: /*#. shown while applying selected properties from dialog to all slides in document */ gt('Applying footer fields to current layout and all following standard slides, please wait...')
            });

            // handle the result of footer fields operations
            return operationsDef
                .progress(function (progress) {
                    // update the progress bar according to progress of the operations promise
                    app.getView().updateBusyProgress(progress);
                })
                .always(function () {
                    app.getView().leaveBusy();
                    // allowing keyboard events again
                    model.setBlockKeyboardEvent(false);
                    if (snapshot) { snapshot.destroy(); }
                });
        };

        /**
         * Clicking on field format from popup will update field with new formatting.
         *
         * @param {String} format
         */
        this.updateFieldFormatting = function (format) {
            var node,
                rootNode = selection.getRootNode(),
                startPos,
                fieldType = highlightedFieldType, // store values before closing popup and deleting field
                highlightedNode = currentlyHighlightedNode,
                isFixedDate;

            // hide popup
            self.destroyFieldPopup();

            if (!fieldType) {
                Utils.warn('SlideField.updateFieldFormatting(): missing field type!');
                return;
            }

            if (highlightedNode) { // simple field
                node = highlightedNode;
                startPos = Position.getOxoPosition(rootNode, node);
                isFixedDate = DOM.isFixedSimpleField(node, app.isODF());

                if (startPos) {
                    return model.getUndoManager().enterUndoGroup(function () {
                        model.deleteRange(startPos);
                        simpleField.insertField(fieldType, format, { isFixed: isFixedDate });
                    });
                }
            }
        };

        /**
         * Returns highlighted field format, or default one.
         *
         * @returns {String}
         *  Highlighted field format.
         */
        this.getSelectedFieldFormat = function () {
            return highlightedFieldInstruction || 'default';
        };

        /**
         * Gets the anchor (start) node of the given field.
         *
         * @returns {jQuery | null}
         *  The anchor node of the given field, otherwise null if its not found.
         */
        this.getSelectedFieldNode = function () {
            return currentlyHighlightedNode || null;
        };

        /**
         * On loosing edit right, close possible open field popup, and abort promise.
         *
         */
        this.destroyFieldPopup = function () {
            app.getView().hideFieldFormatPopup();
            if (fieldFormatPopupTimeout) {
                fieldFormatPopupTimeout.abort();
            }
        };

        /**
         * Add special field that is changetracked to temporary collection.
         * @param {jQuery} trackingNode
         */
        this.addToTempSpecCollection = function (trackingNode) {
            tempSpecFieldCollection.push(trackingNode);
        };

        /**
         * Return array of collected special field nodes.
         */
        this.getSpecFieldsFromTempCollection = function () {
            return tempSpecFieldCollection;
        };

        /**
         * Empty temporary collection after change track resolve is finished.
         */
        this.emptyTempSpecCollection = function () {
            tempSpecFieldCollection = [];
        };

        /**
         * Converts number for page and numPages fields into proper format code.
         *
         * @param {Number} number
         *  Number to be inserted into field.
         *
         * @param {String} format
         *  Format of the number inserted into field.
         *
         * @param {Boolean} [numPages]
         *  NumPages fields have some formats different from PageNum.
         *
         * @returns {String}
         *  Formatted number.
         */
        this.formatFieldNumber = function (pageCount, format, isNumPages) {
            return simpleField.formatPageFieldInstruction(pageCount, format, isNumPages);
        };

        /**
         * Warning: Public accessor used only for unit tests! Not meant to be called directly.
         *
         */
        this.setDateFieldToAutoUpdate = function (event, state) {
            setDateFieldToAutoUpdate(event, state);
        };

        /**
         * Public method to refresh collection of simple fields.
         */
        this.refreshSimpleFieldsCollection = function () {
            self.refreshSlideFields();
        };

        /**
         * Method for stopping operation distribution to server, and updating document's date fields.
         */
        this.updateDateFieldsOnLoad = function () {
            updateDateTimeFields();
            if (app.isODF()) { self.updateMasterSlideFields(); } // updating the fields on the active slide
            self.updateSlideNumFields();
            if (app.isODF()) { model.createMasterSlideFieldsEvent(); } // informing the side pane about the master field state of every slide
        };

        /**
         * Returns whether there are fields in presentation document.
         *
         * @returns {Boolean}
         */
        this.isEmpty = function () {
            return allFields.length === 0;
        };

        /**
         * Public getter for collection of all fields in the document.
         *
         * @returns {Array} allFields - array of objects containing slideID and DOM node of field.
         */
        this.getAllFields = function () {
            return allFields;
        };

        /**
         * Public getter for collection of all standard slide fields in the document.
         *
         * @returns {Array} allNormalSlideFields - array of objects containing slideID and DOM node of field.
         */
        this.getAllNormalSlideFields = function () {
            return allNormalSlideFields;
        };

        /**
         * Public getter that returns which footer fields and placeholders are present on slide.
         *
         * @param {String} [slideId]
         *  If ommited, active slide id is used.
         * @returns {Array}
         */
        this.getFieldOptionsOnSlide = function (slideId) {
            return getFieldOptionsOnSlide(slideId);
        };

        /**
         * Returns current date in localized short format.
         *
         * @returns {String}
         */
        this.getCurrentShortLocalDate = function () {
            return getDateTimeRepresentation(LocaleData.SHORT_DATE);
        };

        /**
         * Providing empty implementation for field manager function 'updateComplexFieldCollector'
         */
        this.updateComplexFieldCollector = _.noop;

        this.handlePasteOperationTarget = _.noop;

        this.checkRestoringSpecialFields = _.noop;

        this.checkIfSpecialFieldsSelected = _.constant(false);

        // initialization -----------------------------------------------------

        app.onInit(function () {
            model = app.getModel();
            selection = model.getSelection();
            numberFormatter = model.getNumberFormatter();

            self.waitForImportSuccess(function () {
                if (model.isLocalStorageImport() || model.isFastLoadImport()) {
                    self.refreshSlideFields();
                }

                // category codes loaded from local resource
                categoryCodesDate = numberFormatter.getCategoryCodes('date');
                // fill in date codes
                if (categoryCodesDate.length) {
                    _.each(categoryCodesDate, function (formatCode) {
                        localFormatList.date.push({ option: formatCode.value, value: getDateTimeRepresentation(formatCode.value) });
                    });
                } else {
                    localFormatList.date.push({ option: LocaleData.SHORT_DATE, value: getDateTimeRepresentation(LocaleData.SHORT_DATE) },
                            { option: LocaleData.LONG_DATE, value: getDateTimeRepresentation(LocaleData.LONG_DATE) });
                }

                // refresh and update datetime and slide number fields on document successfully imported.
                self.updateDateFieldsOnLoad();
                // listen to slide order change to update slide number fields
                self.listenTo(model, 'moved:slide inserted:slide removed:slide', self.updateSlideNumFields);
                // listen to change of slide or layout slide to update fields on master slide
                self.listenTo(model, 'change:slide change:layoutslide', self.updateMasterSlideFields);
            });

            self.listenTo(selection, 'update', checkHighlighting);

            self.listenTo(model, 'document:reloaded', function () {
                self.refreshSlideFields();
            });

            self.listenTo(app, 'docs:editmode:leave', function () {
                self.destroyFieldPopup();
            });

            self.listenTo(app.getView(), 'fielddatepopup:change', updateDateFromPopup);
            self.listenTo(app.getView(), 'fielddatepopup:autoupdate', setDateFieldToAutoUpdate);
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            simpleField.destroy();
            model = selection = numberFormatter = null;
            currentlyHighlightedNode = fieldFormatPopupTimeout = localFormatList = categoryCodesDate = allFields = allNormalSlideFields = null;
            allDateFields = allFooterFields = allSlideNumFields = allMasterNumFields = allMasterDateFields = allMasterFooterFields = tempSpecFieldCollection = null;
        });

    } // class SlideField

    // constants --------------------------------------------------------------

    /**
     * A list of those field types used in ODF footers.
     */
    SlideField.ODF_FOOTER_FIELD_TYPES = ['date-time', 'footer', 'page-number'];

    /**
     * A converter from ODF field types to place holder types.
     */
    SlideField.ODF_FIELD_PLACEHOLDER_CONVERTER = {
        'date-time': 'dt',
        footer: 'ftr',
        'page-number': 'sldNum'
    };

    // export =================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: SlideField });
});
