/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Sven Jacobi <sven.jacobi@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/presentation/app/application', [

    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/driveutils',
    'io.ox/office/tk/utils/presenterutils',

    'io.ox/office/baseframework/utils/errorcode',
    'io.ox/office/baseframework/utils/errorcontext',

    'io.ox/office/editframework/app/editapplication',

    'io.ox/office/presentation/model/model',
    'io.ox/office/presentation/view/view',
    'io.ox/office/presentation/app/controller'

], function (Utils, DriveUtils, PresenterUtils, ErrorCode, ErrorContext, EditApplication, PresentationModel, PresentationView, PresentationController) {

    'use strict';

    // class PresentationApplication ==========================================

    /**
     * The OX Presentation application.
     *
     * @constructor
     *
     * @extends EditApplication
     *
     * @param {Object} launchOptions
     *  All options passed to the core launcher (the ox.launch() method).
     */
    var PresentationApplication = EditApplication.extend(function (launchOptions) {

        // self reference
        var self = this;

        // base constructor ---------------------------------------------------

        EditApplication.call(this, PresentationModel, PresentationView, PresentationController, launchOptions, {
            importFailedHandler: importFailedHandler,
            prepareFlushHandler: prepareFlushDocument,
            prepareRenameHandler: prepareRenameHandler,
            postProcessHandler: postProcessDocument,
            postProcessHandlerStorage: postProcessHandlerStorage,
            optimizeOperationsHandler: optimizeOperations,
            fastEmptyLoadHandler: fastEmptyLoadHandler,
            sendActionsDelay: 1000,
            isMultiSelectionApp: true,
            localStorageApp: true,
            requiredStorageVersion: 3,
            supportedStorageVersion: 3
        });

        // private methods ----------------------------------------------------

        /**
         * Optimizing the actions (operations) before sending them to the server.
         *
         * @param {Object[]} actionsBuffer
         *  An array with actions.
         *
         * @returns {Object[]}
         *  The optimized array with actions.
         */
        function optimizeOperations(actionsBuffer) {
            return self.getModel().optimizeActions(actionsBuffer);
        }

        /**
         * Post-processes the document contents and formatting, after all its
         * import operations have been applied.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  document has been post-processed successfully, or rejected when an
         *  error has occurred.
         */
        function postProcessDocument() {
            return self.getModel().updateDocumentFormatting();
        }

        /**
         * Load performance: Post-processes the document contents and formatting,
         * after the document was loaded from local storage.
         * In Presentation app the function 'updateDocumentFormatting' can be
         * used also for loading documents with local storage.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  document has been post-processed successfully, or rejected when an
         *  error has occurred.
         */
        function postProcessHandlerStorage() {
            return self.getModel().updateDocumentFormatting();
        }

        /**
         * Handler will be called by base class if importing the document
         * failed. The functions handles Presentation specific errors.
         */
        function importFailedHandler(response) {

            var // specific error code sent by the server
                error = new ErrorCode(response);

            //  fulfilling task DOCS-733 - "Improve OOM Error Message in Client" - [https://jira.open-xchange.com/browse/DOCS-733]
            switch (error.getCodeAsConstant()) {
                case 'LOADDOCUMENT_COMPLEXITY_TOO_HIGH_ERROR':
                    self.setInternalError(error, ErrorContext.PRESENTATION, null, { showMessage: false });
                    break;
            }
        }

        /**
         * Preprocessing before the document will be flushed for downloading,
         * printing, sending as mail, or closing.
         * Avoiding 'sendChangedViewAttributes' if the flush is triggered by
         * starting a presentation. Otherwise the conversion to pdf is triggered
         * always before starting the presentation, if the document has once
         * been modified before.
         *
         * @param {String} reason
         *  The reason for flushing the document.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  actions (of dynamic font size calculations) have been applied.
         */
        function prepareFlushDocument(reason) {
            sendChangedViewAttributes(reason); // send changed view attributes from the slide pane
            return self.getModel().getPreFlushDocumentPromise();
        }

        /**
         * Preparations before the rename requests will be sent to the
         * backend. Must call sendChangedViewAttributes to make sure that
         * view settings are not sent while closing the document.
         */
        function prepareRenameHandler() {
            sendChangedViewAttributes();
        }

        /**
         * Send changed view attributes from the slide pane with an operation,
         * so that the attributes are stored in the document. Only send changed
         * view attributes when the user has edit rights, has modified the
         * document locally, is not on mobile or tablets.
         *
         * @param {String} reason
         *  The reason for flushing the document.
         *
         * @returns {Boolean}
         *  A boolean value specifying whether applying the operation was
         *  successful in synchronous mode (option 'async' has not been set).
         */
        function sendChangedViewAttributes(reason) {

            // don't save it on tablets/mobile, because the slide pane is not resizable there
            if (Utils.COMPACT_DEVICE) {
                return self;
            }

            // do not save slide pane width (and optionally trigger a pdf conversion), if
            // the document shall be presented.
            if (reason && reason === 'present') {
                return self;
            }

            // only send changed attributes when user has edit rights
            if (!self.isEditable()) {
                return self;
            }

            // do not send attributes, if undo/redo is running (48211)
            // Also not generating operation in long running processes (recursive call), 53641
            if (self.getModel().isUndoRedoRunning() || self.getModel().isProcessingActions()) {
                return self;
            }

            // whether the document is modified
            var modified = self.isLocallyModified();

            // if file was not modified, check if the slide pane width has been changed compared to the document
            if (!modified) {
                if (self.getModel().getSlidePaneWidthInDocAttributes() !== self.getModel().getSlidePaneWidth()) {
                    modified = true;
                }
            }
            // if the file was not modified, nothing to do
            if (!modified) {
                return self;
            }
            // send the slidePaneWidth with an operation
            return self.getModel().setSlidePaneWidthInDocAttributes();
        }

        /**
         * Handler for fast load of empty default documents
         *
         * @param {String} markup
         *  The HTML mark-up to be shown as initial document contents.
         *
         * @param {Array} actions
         *  The operation actions to be applied to finalize the fast import.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  actions have been applied.
         */
        function fastEmptyLoadHandler(markup, actions) {
            return self.getModel().fastEmptyLoadHandler(markup, actions);
        }

        // public methods -----------------------------------------------------

        /**
         * Launches the OX Presenter application with this presentation.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the new presenter application
         *  has been launched.
         */
        this.startPresenter = function () {

            // block the screen while flushing and initializing presenter
            this.getView().enterBusy({ immediate: true });

            // flush document, needed to have recent changes in the file
            var promise = this.flushDocument('present');

            // propagate changed file to the files API
            promise = promise.then(function () {
                return DriveUtils.propagateChangeFile(self.getFileDescriptor());
            });

            // resolve the file model from OX Drive
            promise = promise.then(function () {
                return DriveUtils.getFileModelFromDescriptor(self.getFileDescriptor());
            });

            // launch the Presenter application
            promise = promise.then(function (model) {
                return PresenterUtils.launchPresenter(model, launchOptions.auth_code);
            });

            // release the blocked screen of this application
            return promise.always(function () {
                self.getView().leaveBusy();
            });
        };

        // initialization -----------------------------------------------------

        // destroy all class members
        this.registerDestructor(function () {
            launchOptions = null;
        });

    }); // class PresentationApplication

    // exports ================================================================

    return PresentationApplication;

});
