/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Edy Haryono <edy.haryono@open-xchange.com>
 */
define('io.ox/office/portal/portalutils', [
    'io.ox/core/capabilities',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/utils/dateutils',
    'io.ox/office/tk/utils/driveutils',
    'io.ox/office/tk/utils/presenterutils',
    'io.ox/office/baseframework/app/extensionregistry',
    'gettext!io.ox/office/portal/main'
], function (Capabilities, Utils, KeyCodes, DateUtils, DriveUtils, PresenterUtils, ExtensionRegistry, gt) {

    'use strict';

    // TODO maybe there is a better place for this. Maybe backend office settings?
    var OFFICE_CAPABILITY_NAMES = ['text', 'spreadsheet', 'presentation'];

    // static class PortalUtils ===============================================

    // copy tk/utils methods
    var PortalUtils = _.clone(Utils);

    // static methods ---------------------------------------------------------

    /**
     * Returns the full module name of the specified portal application.
     *
     * @param {String} appBaseName
     *  The base name of the portal application ('text', 'spreadsheet', etc.).
     *
     * @returns {String}
     *  The full module name of the specified portal application.
     */
    PortalUtils.getPortalModuleName = function (appBaseName) {
        return 'io.ox/office/portal/' + appBaseName;
    };

    /**
     * Returns the full module name of the specified editor application
     * associated to a portal application.
     *
     * @param {String} appBaseName
     *  The base name of the editor application ('text', 'spreadsheet', etc.).
     *
     * @returns {String}
     *  The full module name of the specified editor application.
     */
    PortalUtils.getEditorModuleName = function (appBaseName) {
        return 'io.ox/office/' + appBaseName;
    };

    /**
     * Returns the application module name for the given filename.
     *
     * @param {String} fileName
     *  Filename as string
     *
     * @returns {String|Null}
     *  Application module name string if it exists, otherwise null.
     */
    PortalUtils.getModuleNameForFile = function (fileName) {
        var editModule = ExtensionRegistry.getEditModule(fileName);
        return (editModule && ExtensionRegistry.isEditable(fileName, editModule)) ? editModule : null;
    };

    /**
     * Detects file type and creates an corresponding icon node.
     *
     * @param {String} fileName
     *  The filename.
     *
     * @returns {jQuery}
     *  the icon node as jQuery object
     */
    PortalUtils.createDocumentIcon = function (fileName) {

        var moduleName = PortalUtils.getModuleNameForFile(fileName),
            iconClasses = 'document-icon fa ';

        switch (moduleName) {
            case 'io.ox/office/spreadsheet':
                iconClasses += 'fa-table spreadsheet';
                break;
            case 'io.ox/office/text':
                iconClasses += 'fa-file-text-o text';
                break;
            case 'io.ox/office/presentation':
                iconClasses += 'fa-list-alt presentation';
                break;
            default:
                iconClasses += 'fa-file';
        }

        return $('<i>').addClass(iconClasses);
    };

    /**
     * Formats the timestamp representation for the recents document list:
     * - show only time, if the document is created today
     * - show year only, if the document is not modified in this year.
     *
     * @param {Number} timestamp
     * - milliseconds since Jan 1, 1970
     *
     * @returns {String}
     *  the formatted date as string
     */
    PortalUtils.formatDate = function (timestamp) {

        if (!_.isNumber(timestamp)) { return gt('unknown'); }

        var now = new Date();
        var date = new Date(timestamp);

        function getTimeString() {
            return DateUtils.format(date, DateUtils.getTimeFormat());
        }

        function getDateString() {
            // get locale country default date format
            var formatString = DateUtils.getDateFormat();
            // hide year if the document is modified in this year
            if (date.getYear() === now.getYear()) {
                formatString = DateUtils.getNoYearFormat();
            }
            return DateUtils.format(date, formatString);
        }

        function isToday(date) {
            return date.getDate() === now.getDate() &&
                date.getMonth() === now.getMonth() &&
                date.getYear() === now.getYear();
        }

        // show only time if the document is last modifed today
        return isToday(date) ? getTimeString() : getDateString();

    };

    /**
     * Returns the application base name for the given filename.
     *
     * @param {String} filename
     *  Filename as string
     *
     * @returns {String | Null}
     *  Application base name string if it exists, otherwise null.
     */
    PortalUtils.getAppBaseName = function (filename) {

        var moduleName = PortalUtils.getModuleNameForFile(filename);

        if (!moduleName) { return null; }

        return moduleName.substr(moduleName.lastIndexOf('/') + 1, moduleName.length);
    };

    /**
     * Returns the given file name without the file extension.
     *
     * @param {String} filename
     *  Filename as string
     *
     * @returns {String}
     *  Specified file name as string without file extension.
     */
    PortalUtils.removeFileExtension = function (filename) {

        var shortFileName = filename,
            pos = filename.lastIndexOf('.');

        if (pos > -1) {
            shortFileName = shortFileName.substr(0, pos);
        }

        return shortFileName;
    };

    // PortalUtils.stringEndsWith moved to Utils

    /**
     * Returns application base names that are enabled in the Appsuite Capabilities.
     *
     * @returns {Array<String>}
     *  An array of application base name strings.
     */
    PortalUtils.getEnabledApps = function () {

        var enabledApps = [];

        // get enabled office apps from Capabilities
        _.each(OFFICE_CAPABILITY_NAMES, function (name) {
            if (Capabilities.has(name)) { enabledApps.push(name); }
        });

        return enabledApps;

    };

    /**
     * Returns the name of the current active application, so that it corresponds to the
     * type of the file.
     *
     * @returns {String}
     *  The application base name string.
     */
    PortalUtils.getActiveApp = function (app, options) {
        var translated = Utils.getBooleanOption(options, 'translated', true),
            string = (translated) ? app.attributes.title : app.attributes.appBaseName;

        return (string).toLowerCase();
    };

    /**
     * Move focus to from given node to its next or previous sibling. (keyboard accessibility)
     *
     * @param {jQuery} startNode
     *  Starting node.
     *
     * @param {String = 'right'} direction
     *  Available direction strings: 'right','left', 'up', 'down'.
     */
    PortalUtils.moveFocus = function (startNode, direction) {

        if (!startNode) { return; }

        var nodeToFocus = startNode;
        direction = direction || 'right';

        /**
         * Tries to find a vertical neighbor according to cursor direction (up/down)
         *
         * @returns {jQuery|Null}
         *  the vertical neighbor object or null no vertical neighbours found.
         */
        function findVerticalSibling() {
            var startNodePosition = startNode.position(),
                startNodeHeight = startNode.outerHeight(true),
                targetNodes = startNode.siblings(),
                targetNode = _.find(targetNodes, function (node) {
                    if (!$(node).attr('tabindex')) { return false; }
                    var nodePosition = $(node).position();
                    if (nodePosition.left !== startNodePosition.left) { return false; }
                    switch (direction) {
                        case 'down':
                            return nodePosition.top === (startNodePosition.top + startNodeHeight);
                        case 'up':
                            return nodePosition.top === (startNodePosition.top - startNodeHeight);
                        default :
                            return false;
                    }
                });
            if (!targetNode) { return null; }
            return targetNode;
        }

        switch (direction) {
            case 'right':
                nodeToFocus = startNode.next();
                // temp workaround: jump over inserted Bootstrap tooltip element if there is any.
                if (!nodeToFocus.attr('tabindex')) { nodeToFocus = nodeToFocus.next(); }
                break;
            case 'left':
                nodeToFocus = startNode.prev();
                break;
            case 'down':
                nodeToFocus = findVerticalSibling('down');
                break;
            case 'up':
                nodeToFocus = findVerticalSibling('up');
                break;
        }

        if (!nodeToFocus) { return; }

        Utils.setFocus(nodeToFocus);
    };

    /**
     * Keyboard accessibility handler for recent documents and templates list.
     *
     * @param {jQuery.Event} event
     *  {Boolean} event.data.grid - Whether grid navigation (up,down,left,right) is required.
     */
    PortalUtils.keydownHandler = function (event) {

        if (!event || !event.target) { return; }

        var startNode = $(event.target),
            isGrid = event.data ? event.data.grid : false;

        if (KeyCodes.matchKeyCode(event, 'ENTER') || KeyCodes.matchKeyCode(event, 'SPACE')) { startNode.trigger('click'); }
        if (KeyCodes.matchKeyCode(event, 'LEFT_ARROW')) { PortalUtils.moveFocus(startNode, 'left'); }
        if (KeyCodes.matchKeyCode(event, 'RIGHT_ARROW')) { PortalUtils.moveFocus(startNode, 'right'); }
        if (KeyCodes.matchKeyCode(event, 'DOWN_ARROW')) { PortalUtils.moveFocus(startNode, isGrid ? 'down' : 'right'); }
        if (KeyCodes.matchKeyCode(event, 'UP_ARROW')) { PortalUtils.moveFocus(startNode, isGrid ? 'up' : 'left'); }
    };

    /**
     * A handler to check if a long tap was made on iOS. It fires a 'contextmenu'
     * event on long tap (after 750 ms) and a 'click' on a normal tap (0-400 ms).
     *
     * @param {Event} event
     *  A touchstart event which started the tap.
     *
     * @param {jQuery} link
     *  A reference to the element on which the handler was called.
     */
    PortalUtils.iosContextEventHandler = function (event, link) {

        var // a flag if the the long tap was canceled (touchend event before hold() is called)
            inCancelFlag = true,
            // a flag if the a long tap is/was invoked
            inHoldFlag = true,
            // a flag if the touch position is moved while holding
            movedWhileHolding = false,
            // time from  touchstart
            timeStart = event.timeStamp,
            // position at touchstart
            xTouchstart = event.originalEvent.pageX,
            yTouchstart = event.originalEvent.pageY,
            // tracked position from touchmove
            yMoved,
            // ignore user movment from from n pixels
            ignoreMovmentInPixel = 10;

        function trackPos(event) {
            var touch = event.originalEvent.touches[0];
            yMoved = touch.pageY;
        }

        function cancel(event) {

            // to prevent the ghost click
            event.preventDefault();

            var // position at touchend
                yTouchend = event.originalEvent.pageY,
                // don't click when touchstart has moved compared to touchend ->??? scroll??
                movedOnTouchend = (Math.abs(yTouchstart - yTouchend) > ignoreMovmentInPixel),
                // prevent click when touchend event is greater than 400ms
                preventClickOnTouchHold = (Math.abs(timeStart - event.timeStamp) < 400);

            inCancelFlag = false;

            // trigger a click when there was no scrolling and the touch was released quick
            // do not trigger when a longtap was started in hold()
            if (inHoldFlag && !movedOnTouchend && preventClickOnTouchHold) {
                link.trigger('click');
            }

            // event listeners are destroyed on touchend
            removeListener();
        }

        function hold() {

            inHoldFlag = false;

            // check if moved (compare touchstart with the current touchmove pos) -> scroll (down&move) would fire otherwise
            movedWhileHolding = (Math.abs(yTouchstart - yMoved) > ignoreMovmentInPixel);

            // trigger context menu when there was no touchend before and no movement
            if (inCancelFlag && !movedWhileHolding) {

                // Create a new jQuery.Event object with specified event properties.
                var e = new $.Event('contextmenu', { pageX: xTouchstart, pageY: yTouchstart });

                // trigger an artificial keydown event with keyCode 64
                link.trigger(e);
            }
        }

        // remove all listener used in iosContextEventHandler()
        function removeListener() {
            $(document).off('touchmove', trackPos);
            link.off('touchend', cancel);
        }

        // track the position when the finger touches the touchscreen
        $(document).on('touchmove', trackPos);

        // when the finger leaves the touchscreen
        link.on('touchend', cancel);

        // check if a long tap should be triggered after 750 ms
        window.setTimeout(hold, 750);
    };

    /**
     * Add a class to a given element to mark it.
     * The highlighting is defined in the style.less.
     *
     * @param {jQuery} link
     *  A reference to the element.
     */
    PortalUtils.markElement = function (link) {
        link.addClass('permanent-hover');
    };

    /**
     * Remove a class from a given element to unmark it.
     * The highlighting is defined in the style.less.
     *
     * @param {jQuery} link
     *  A reference to the element.
     */
    PortalUtils.removeMark = function (link) {
        link.removeClass('permanent-hover');
    };

    /**
     * calls CoreMetrics Class (io.ox/metrics/main)
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {String} [options.moduleName]
     *      moduleName of the current app
     *      with help of the current module the metrics params "app" will be created
     *      and the param "action" gets the app name as prefix except actionModuleName is assigned
     *  @param {String} [options.actionModuleName=options.moduleName]
     *      and the param "action" gets the app name as prefix
     *  @param {String} [options.target]
     *      target is used as "target" and is directly sent to CoreMetrics
     *  @param {String} [options.action]
     *      action gets the prefix of the app and is sent as "action" to the CoreMetrics
     */
    PortalUtils.trackEvent = function (options) {
        require(['io.ox/metrics/main']).done(function (CoreMetrics) {
            var app = Utils.getStringOption(options, 'moduleName', '').replace('io.ox/office/', '');
            var actionApp = Utils.getStringOption(options, 'actionModuleName', Utils.getStringOption(options, 'moduleName', '')).replace('io.ox/office/', '');
            var send = {
                app: app + '-portal',
                target: Utils.getStringOption(options, 'target'),
                action: actionApp + '-' + Utils.getStringOption(options, 'action')
            };
            //console.warn('track', send);
            CoreMetrics.trackEvent(send);
        });
    };

    /**
     * Launch options to create a new document in the default 'Documents'
     * folder in Drive.
     *
     * @returns {Object}
     *  Necessary parameters for creating a new document in the default folder.
     */
    PortalUtils.getNewLaunchOptions = function () {
        return { action: 'new', target_folder_id: DriveUtils.getStandardDocumentsFolderId() };
    };

    /**
     * Launch options to create a new encrypted document in the default 'Documents'
     * folder in Drive.
     *
     * @returns {jQuery.Promise}
     *  A promise which returns the necessary parameters for creating a new encrypted document in the default folder.
     */
    PortalUtils.getNewEncryptedLaunchOptions = function () {
        // set authorization data to the model
        return DriveUtils.authorizeGuard().then(function (auth_code) {
            return _.extend(PortalUtils.getNewLaunchOptions(), { auth_code: auth_code, cryptoAction: 'Encrypt' });
        });
    };

    /**
     * runs ox-Drive with assigned folder id
     * then it calls "selectFile" in the Drive app with assigned file-descriptor
     *
     * @param {Object} file is a file-descriptor
     *
     * @param {Boolean} [default=false]
     *
     * if assigned current ox app is set to busy mode
     * and back to idle after drive is loaded
     *
     */
    PortalUtils.showInDrive = function (file, busy) {
        var win = null;
        if (busy) {
            win = ox.ui.App.getCurrentApp().getWindow();
            win.busy();
        }

        // folder from the file in drive
        ox.launch('io.ox/files/main', { folder: file.folder_id }).done(function () {
            // set proper folder and select file
            this.selectFile(file);
            this.listView.trigger('listview:shown');
            if (win) { win.idle(); }
        });
    };

    /**
     * Shows the document in the Presenter.
     *
     * @param {Object} baton
     *  A baton object which has a backbone cid.
     *  Note: Unfortunately there are cases where the cid is the file-descriptor (e.g. in Drive),
     *  so better take a look at the baton before using it here.
     *
     * @param {Boolean} [busy=false]
     *  If assigned current ox app is set to busy mode
     *  and back to idle after the Presenter is loaded.
     *
     */
    PortalUtils.showInPresenter = function (baton, busy) {
        var win = null;
        if (busy) {
            win = ox.ui.App.getCurrentApp().getWindow();
            win.busy();
        }
        // There are two different kinds of batons, one with a backbone cid and
        // one which has a file descriptor as a cid (e.g. in Drive). This baton
        // is expected to have a backbone cid. To launch the Presenter, a baton
        // with a file descriptor as a cid is needed. Therefore 'getFileModelFromDescriptor'
        // is used to create it.
        DriveUtils.getFileModelFromDescriptor(baton.data).then(PresenterUtils.launchPresenter).then(function () {

            if (win) { win.idle(); }
        });
    };

    // exports ================================================================

    return PortalUtils;

});
