/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

/* eslint no-new: 0 */

define('io.ox/office/portal/action/toolbaractionsimpl', [
    'io.ox/core/extensions',
    'io.ox/core/extPatterns/links',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/driveutils',
    'io.ox/office/baseframework/utils/baseconfig',
    'io.ox/office/portal/portalutils',
    'gettext!io.ox/office/portal/main'
], function (Ext, Links, Utils, DriveUtils, Config, PortalUtils, gt) {

    'use strict';

    /**
     * Creates all extensions points and bind actions for the portal toolbar.
     *
     * @constructor
     *
     * @param {Object} options
     *  @param {String} [options.appBaseName]
     *  @param {String} [options.newDocText]
     *  @param {String} [options.openDocText]
     */
    function ToolbarActionsImpl(options) {

        var // the name of the application, used for filtering files in file picker
            appBaseName =  Utils.getStringOption(options, 'appBaseName'),
            // identifier for the appBaseName toolbar
            toolbarPointId = 'io.ox/office/portal/toolbar' + appBaseName,
            // configuration key to store the path to the recently opened documents
            preselectKey = appBaseName + '/recentOpenDocPath',
            // deferred is set when "newdoc" & "loaddoc" is called, so we can allow only one new call at once
            newDeferred = null, loadDeferred = null;

        this.toolbarPoint = Ext.point(toolbarPointId);

        // create extension point for the 'new appBaseName' link
        Ext.point(toolbarPointId + '-newblank').extend(new Links.Link({
            id: 'new_' + appBaseName,
            index: 100,
            prio: 'hi',
            label: Utils.getStringOption(options, 'newDocText'),
            cssClasses: 'new-document',
            role: 'button',
            ref: toolbarPointId + '/actions/new/' + appBaseName
        }));

        // add link to the toolbar point
        this.toolbarPoint.extend({
            id: 'new-' + appBaseName + 'document',
            index: 100,
            draw: function (baton) {
                Ext.point(toolbarPointId + '-newblank').invoke('draw', this, baton);
            }
        });

        if (Config.GUARD_AVAILABLE && !PortalUtils.SMALL_DEVICE) {
            // create extension point for the 'new appBaseName' link
            Ext.point(toolbarPointId + '-newblank-encrypted').extend(new Links.Link({
                id: 'new_' + appBaseName + '_encrypted',
                index: 150,
                prio: 'hi',
                label: Utils.getStringOption(options, 'newDocTextEncrypted'),
                cssClasses: 'new-document',
                role: 'button',
                ref: toolbarPointId + '/actions/newencrypted/' + appBaseName
            }));

            // add link to the toolbar point
            this.toolbarPoint.extend({
                id: 'new-' + appBaseName + 'document-encrypted',
                index: 150,
                draw: function (baton) {
                    Ext.point(toolbarPointId + '-newblank-encrypted').invoke('draw', this, baton);
                }
            });
        }
        //add open document link to  the toolbar point
        this.toolbarPoint.extend(new Links.Link({
            id: 'open-' + appBaseName + 'document',
            index: 200,
            prio: 'hi',
            label: gt('Open document'),
            cssClasses: 'open-document',
            role: 'button',
            ref: 'io.ox/office/portal/open' + appBaseName
        }));

        // define action that will be fired on the 'new appBaseName', 'new text' and 'open document' link
        new Links.Action(toolbarPointId + '/actions/new/' + appBaseName, {
            capabilities: appBaseName,
            action: function () {
                if (newDeferred && newDeferred.state() === 'pending') {
                    Utils.warn('cant run new document, because old call is still pending');
                    return;
                }
                var moduleName = PortalUtils.getEditorModuleName(appBaseName);
                PortalUtils.trackEvent({ moduleName: moduleName, target: 'toolbar', action: 'newblank' });
                newDeferred = ox.launch(moduleName + '/main', PortalUtils.getNewLaunchOptions());
            }
        });
        if (Config.GUARD_AVAILABLE) {
            // define action that will be fired on the 'new appBaseName', 'new text' and 'open document' link encrypted
            new Links.Action(toolbarPointId + '/actions/newencrypted/' + appBaseName, {
                capabilities: appBaseName,
                action: function () {

                    if (newDeferred && newDeferred.state() === 'pending') {
                        Utils.warn('cant run new document, because old call is still pending');
                        return;
                    }

                    var promise = PortalUtils.getNewEncryptedLaunchOptions();

                    // set authorization data to the model
                    promise.then(function (launch_options) {
                        var moduleName = PortalUtils.getEditorModuleName(appBaseName);
                        PortalUtils.trackEvent({ moduleName: moduleName, target: 'toolbar', action: 'newblank' });
                        newDeferred = ox.launch(moduleName + '/main', launch_options);
                    });
                }
            });
        }
        new Links.Action('io.ox/office/portal/open' + appBaseName, {
            action: function () {
                require([
                    'io.ox/files/filepicker',
                    'io.ox/office/baseframework/app/extensionregistry',
                    // prefetch
                    'io.ox/files/mobile-navbar-extensions'
                ]).done(function (FilePicker, ExtensionRegistry) {
                    var preselect = Config.get(preselectKey) || DriveUtils.getStandardDocumentsFolderId();
                    var extensions = ExtensionRegistry.getEditableExtensions();
                    var filter = extensions.map(function (ext) { return '.' + ext; }).join(',');

                    var dialog = new FilePicker({
                        // filter files of disabled applications (capabilities)
                        filter: function (file) {
                            // Bug 47431: check if there is a file name (String), otherwise return false
                            // Bug 49564 DriveUtils.canModify use for new folders like "My Attachments"
                            return _.isString(file.filename) && ExtensionRegistry.isEditable(file.filename) && DriveUtils.canModify(file);
                        },
                        sorter: function (file) {
                            return (file.filename || file.title).toLowerCase();
                        },
                        primaryButtonText: gt('Open'),
                        header: gt('Open document'),
                        folder: preselect,
                        uploadButton: true,
                        multiselect: false,
                        //uploadFolder: DriveUtils.getStandardDocumentsFolderId(),
                        hideTrashfolder: true,
                        acceptLocalFileType: filter,
                        createFolderButton: false
                    });
                    dialog.done(function (selectedFiles) {
                        var selectedFile = selectedFiles[0];
                        if (selectedFile) {
                            if (selectedFile.creation_date) {
                                require(['io.ox/office/baseframework/utils/infomessages',  'io.ox/core/notifications']).done(function (InfoMessages, Notifications) {
                                    DriveUtils.getPath(selectedFile.folder_id).done(function (paths) {
                                        var path = DriveUtils.preparePath(paths);
                                        var messageData = InfoMessages.getMessageData('INFO_DOC_SAVED_IN_FOLDER', { fullFileName: selectedFile.title, fullPathName: path, type: 'success', duration: 10000 });
                                        Notifications.yell(messageData.type, messageData.message);
                                    });
                                });

                            }

                            if (loadDeferred && loadDeferred.state() === 'pending') {
                                Utils.warn('cant run load document, because old call is still pending');
                                return;
                            }

                            var actionModuleName = PortalUtils.getModuleNameForFile(selectedFile.filename);
                            PortalUtils.trackEvent({ moduleName: appBaseName, actionModuleName: actionModuleName, target: 'toolbar', action: 'open' });

                            var launchOptions = ExtensionRegistry.isNative(selectedFile.filename) ?
                                { action: 'load', file: selectedFile } :
                                { action: 'convert', target_folder_id: selectedFile.folder_id, templateFile: selectedFile, preserveFileName: true };
                            loadDeferred = ox.launch(actionModuleName + '/main', launchOptions);
                            Config.set(preselectKey, selectedFile.folder_id);
                        }
                    });
                });
            }
        });

    } // class ToolbarActionsImpl

    return ToolbarActionsImpl;

});
