/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/editframework/view/mobilesearchgroup', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/baseframework/view/toolpane',
    'io.ox/office/baseframework/view/toolbar',
    'io.ox/office/baseframework/view/basecontrols',
    'io.ox/office/editframework/view/editlabels',
    'gettext!io.ox/office/editframework/main'
], function (Utils, TriggerObject, ToolPane, ToolBar, Controls, Labels, gt) {

    'use strict';

    // convenience shortcuts
    var Button = Controls.Button;
    var TextField = Controls.TextField;
    var CompoundSplitButton = Controls.CompoundSplitButton;

    // class MobileSearchGroup ================================================

    var MobileSearchGroup = TriggerObject.extend({ constructor: function (docView) {

        var searchBar = docView.createToolBar('view/searchgroup');
        var searchReplaceBar = docView.createToolBar('view/replacegroup');
        var replacePane = new ToolPane(docView, { position: 'top', classes: 'small-distance standard-design' });
        var replaceBar = new ToolBar(docView);
        var replaceSplit = null;

        var searchBtn = null;
        var searchReplaceBtn = null;

        // base constructor ---------------------------------------------------

        TriggerObject.call(this, docView);

        // private methods ---------------------------------------------------

        function addSearch(bar) {
            bar.addGroup('document/search/text',  new TextField(docView, Labels.SEARCH_TEXT).setWidth(Labels.SEARCH_TEXT.smallerVersion.css.width));
            bar.addGroup('document/search/start', new Button(docView, Labels.SEARCH_START));
            bar.addGroup('document/search/prev',  new Button(docView, Labels.SEARCH_PREV));
            bar.addGroup('document/search/next',  new Button(docView, Labels.SEARCH_NEXT));
        }

        function showSearch() {
            searchBtn.show();
            searchReplaceBtn.hide();
        }

        function showSearchReplace() {
            searchBtn.hide();
            searchReplaceBtn.show();
        }

        var delayClose = this.createDebouncedMethod('MobileSearchGroup.delayClose', null, function () {
            if (!searchBar.isVisible() && !searchReplaceBar.isVisible()) {
                docView.executeControllerItem('document/search/end');
                showSearch();
            }
        });

        // public methods ---------------------------------------------------

        this.handleTabListButtons = function () {

            docView.createToolBarTab('view/searchgroup', { label: gt('Search'), icon: 'fa-search', priority: 2 });
            docView.createToolBarTab('view/replacegroup', { label: gt('Replace'), icon: 'fa-search', priority: 2 });

            var flatTabView = docView.getToolPane().getFlatTabView();
            var tabViewNode = flatTabView.getMenu().getNode();
            searchBtn = tabViewNode.find('[data-value="view/searchgroup"]').parent();
            searchReplaceBtn = tabViewNode.find('[data-value="view/replacegroup"]').parent();

            $('<div class="separator-line">').insertBefore(searchBtn);

            showSearch();

            tabViewNode.find('[data-key="view/searchpane"]').hide();
        };

        //toggle should be never called
        this.toggle = Utils.NOOP;

        this.hide = function () {
            searchBar.hide();
            searchReplaceBar.hide();
        };

        this.isVisible = function () {
            return searchBar.isVisible() || searchReplaceBar.isVisible();
        };

        // init ---------------------------------------------------------------

        addSearch(searchBar);
        addSearch(searchReplaceBar);

        replaceBar.addGroup('dummy', new Button(docView, { icon: 'fa' }));
        replaceBar.addGap();
        replaceBar.addGroup('document/replace/text', new TextField(docView, Labels.REPLACE_TEXT).setWidth(Labels.REPLACE_TEXT.smallerVersion.css.width));

        replaceSplit = new CompoundSplitButton(docView, Labels.REPLACE_NEXT);
        replaceSplit.addGroup('document/replace/next', new Button(docView, Labels.REPLACE_NEXT));
        replaceSplit.addGroup('document/replace/all',  new Button(docView, Labels.REPLACE_ALL));
        replaceBar.addGroup('document/replace/next', replaceSplit);

        replacePane.addViewComponent(replaceBar);
        replacePane.hide();
        docView.addPane(replacePane);

        // listeners ---------------------------------------------------------------

        searchBar.on('component:show', function () {
            replacePane.hide();
            showSearchReplace();
        });

        searchBar.on('component:hide', function () {
            replacePane.hide();
            delayClose();
        });

        searchReplaceBar.on('component:show', function () {
            replacePane.show();
            showSearch();
        });

        searchReplaceBar.on('component:hide', function () {
            replacePane.hide();
            delayClose();
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            searchBar.destroy();
            searchReplaceBar.destroy();
            docView = searchBar = searchReplaceBar = null;
            replacePane = replaceBar = replaceSplit = null;
            searchBtn = searchReplaceBtn = null;
        });

    } }); // class MobileSearchGroup

    // exports ================================================================

    return MobileSearchGroup;

});
