/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/control/borderstylepicker', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/tk/render/canvas',
    'io.ox/office/tk/control/radiolist',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/view/editlabels'
], function (Utils, Forms, ValueMap, Canvas, RadioList, Border, Labels) {

    'use strict';

    // the canvas element used to generate the bitmaps
    var canvas = new Canvas(Canvas.SINGLETON).initialize({ width: 32, height: 18 });

    // maps unique bitmap keys to the data URLs
    var bitmapUrlMap = new ValueMap();

    // private global functions ===============================================

    /**
     * Creates a bitmap for the specified border line style, and returns its
     * data URL.
     *
     * @param {String} lineStyle
     *  The effective line style. MUST be one of 'none', 'solid', 'dashed',
     *  'dotted', 'dashDot', or 'dashDotDot'.
     *
     * @param {String} lineColor
     *  The effective CSS line color.
     *
     * @param {Number} [lineWidth=1]
     *  Width of a (single) line in the border style, in pixels. The value 0
     *  will be interpreted as hair line.
     *
     * @param {Number} [lineCount=1]
     *  The number of parallel lines shown in the border.
     *
     * @returns {String}
     *  The data URL of the generated bitmap.
     */
    function getBorderStyleBitmapUrl(lineStyle, lineColor, lineWidth, lineCount) {

        // validate the optional input parameters
        if (typeof lineWidth !== 'number') { lineWidth = 1; }
        if (typeof lineCount !== 'number') { lineCount = 1; }

        // return data URL of a bitmap already created, or create a new bitmap
        return bitmapUrlMap.getOrCreate(lineStyle + ',' + lineColor + ',' + lineWidth + ',' + lineCount, function () {

            // clear the canvas (also for style 'none')
            canvas.clear();

            // create the bitmap (nothing to do for style 'none')
            if (lineStyle !== 'none') {
                canvas.render(function (context, width, height) {

                    // effective line width
                    var lineWidthPx = Math.max(1, lineWidth);
                    // dash pattern
                    var pattern = Border.getBorderPattern(lineStyle, lineWidthPx);
                    // Y offset of the first line
                    var y = Math.floor(height / 2) - lineWidthPx * (lineCount - 1) - (lineWidthPx % 2) / 2;

                    // initialize line settings
                    context.setLineStyle({ style: lineColor, width: lineWidthPx, pattern: pattern });
                    if (lineWidth < 1) { context.setGlobalAlpha(0.4); }

                    // draw the lines
                    for (; lineCount > 0; lineCount -= 1) {
                        context.drawLine(0, y, width, y);
                        y += 2 * lineWidthPx;
                    }
                });
            }

            // convert the bitmap to a data URL
            return canvas.getDataURL();
        });
    }

    // class BorderStylePicker ================================================

    /**
     * A generic drop-down list control for border styles.
     *
     * @constructor
     *
     * @extends RadioList
     *
     * @param {Array<Object>} listEntries
     *  An array of descriptors for the entries of the drop-down list. Each
     *  descriptor MUST be an object with the following properties:
     *  - {Any} value
     *      The value associated to the list item.
     *  - {String} label
     *      The text label for the list item.
     *  - {String} style
     *      The line style: one of 'solid', 'dashed', 'dotted', 'dashDot', or
     *      'dashDotDot'.
     *  - {Number} [width=1]
     *      Width of a (single) line, in pixels. The value 0 will be
     *      interpreted as hair line.
     *  - {Number} [count=1]
     *      Number of parallel lines.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options supported by the base class
     *  RadioList.
     */
    var BorderStylePicker = RadioList.extend({ constructor: function (docView, listEntries, initOptions) {

        // self reference
        var self = this;

        // base constructor ---------------------------------------------------

        RadioList.call(this, docView, Utils.extendOptions({
            icon: 'docs-line-style',
            tooltip: Labels.BORDER_STYLE_LABEL,
            updateCaptionMode: 'none',
            dropDownVersion: { label: Utils.getStringOption(initOptions, 'tooltip') }
        }, initOptions));

        // initialization -----------------------------------------------------

        // lazy initialization of the drop-down menu items
        this.listenOnceTo(this.getMenu(), 'popup:beforeshow', function () {
            listEntries.forEach(function (entry) {
                var buttonNode = self.getMenu().createItemNode(entry.value, { label: entry.label });
                var bitmapUrl1 = getBorderStyleBitmapUrl(entry.style, Forms.LessVar.TEXT_COLOR, entry.width, entry.count);
                var bitmapUrl2 = getBorderStyleBitmapUrl(entry.style, Forms.LessVar.LIST_SELECTED_TEXT_COLOR, entry.width, entry.count);
                Forms.setCaptionBitmapIcon(buttonNode, bitmapUrl1, bitmapUrl2);
            });
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docView = listEntries = initOptions = null;
        });

    } }); // class BorderStylePicker

    // exports ================================================================

    return BorderStylePicker;

});
