/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/control/applicationstatuslabel', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/control/label',
    'gettext!io.ox/office/editframework/main'
], function (Utils, Label, gt) {

    'use strict';

    var LOADING_LABEL = /*#. label shown in the toolbar while the document is loading */ gt('Loading document');
    var SAVING_LAVEL = /*#. label shown in the toolbar while user changes are sent to the server */ gt('Saving changes');
    var ALL_SAVED_LABEL = gt('All changes saved');

    // class ApplicationStatusLabel ===========================================

    /**
     * Shows the current status of the application when it has changed.
     *
     * @constructor
     *
     * @extends Label
     *
     * @param {EditView} docView
     *  The document view instance containing this control.
     */
    function ApplicationStatusLabel(docView, initOptions) {

        // self reference
        var self = this;

        // caption options, mapped by different application states
        var captions = {};

        // timer for deferred caption update ('busy' captions with spinning icon)
        var timer = null;

        // base constructor ---------------------------------------------------

        Label.call(this, docView, _.extend(initOptions, { classes: 'app-status-label', minWidth: 200 }));

        // private methods ----------------------------------------------------

        function updateCaption(state) {

            // the registered caption settings for the new application state
            var captionData = captions[state];
            // the caption options
            var caption = Utils.getObjectOption(captionData, 'caption', null);
            // the delay timer for showing the caption
            var delay = Utils.getIntegerOption(captionData, 'delay', 0, 0);

            // cancel pending timer for a busy caption
            if (timer) { timer.abort(); }

            // if the application state is a temporary 'busy' state, defer showing the caption
            if (delay > 0) {
                timer = self.executeDelayed(function () { self.setCaption(caption); }, 'ApplicationStatusLabel.updateCaption', delay);
                timer.always(function () { timer = null; });
            } else {
                self.setCaption(caption);
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Registers a label text and optional icon for a specific user defined
         * application state.
         *
         * @param {String} state
         *  The identifier of an application state.
         *
         * @param {String} label
         *  The label text to be shown while the application state is active.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.icon]
         *      The CSS class of an icon to be shown in front of the label.
         *  @param {Boolean} [options.busy=false]
         *      If set to true, a rotating busy icon will be shown in front of
         *      the label. This option overrides the option 'icon'.
         *  @param {Integer} [options.delay=0]
         *      The delay time in milliseconds to show the caption.
         *
         * @returns {ApplicationStatusLabel}
         *  A reference to this instance.
         */
        this.registerCaption = function (state, label, options) {

            // create the caption settings
            var captionData = captions[state] = {
                caption: { label: label },
                delay: Utils.getIntegerOption(options, 'delay', 0, 0)
            };

            // add a busy spinner icon, or another custom icon
            if (Utils.getBooleanOption(options, 'busy', false)) {
                captionData.caption.icon = 'fa-refresh';
                captionData.caption.iconClasses = 'fa-spin';
            } else {
                captionData.caption.icon = Utils.getStringOption(options, 'icon', null);
            }

            return this;
        };

        // initialization -----------------------------------------------------

        this.registerCaption('preview', LOADING_LABEL, { busy: true });
        this.registerCaption('sending', SAVING_LAVEL, { busy: true, delay: 200 });
        this.registerCaption('ready', ALL_SAVED_LABEL, { icon: 'fa-check' });

        // make label node an ARIA live region
        this.getLabelNode().attr({
            role: 'status',
            'aria-live': 'polite',
            'aria-relevant': 'additions',
            'aria-atomic': true,
            'aria-readonly': true
        });

        // update caption according to application status
        this.listenTo(docView.getApp(), 'docs:state', function (state) { updateCaption(state); });

        // destroy class members on destruction
        this.registerDestructor(function () {
            docView = self = captions = timer = null;
        });

    } // class ApplicationStatusLabel

    // exports ================================================================

    // derive this class from class Label
    return Label.extend({ constructor: ApplicationStatusLabel });

});
