/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Carsten Driesner <carsten.driesner@open-xchange.com>
 */

define('io.ox/office/editframework/utils/operationutils', [
    'io.ox/office/tk/utils'
], function (Utils) {

    'use strict';

    // static class OperationUtils ============================================

    var OperationUtils = {};

    // constants --------------------------------------------------------------

    /**
     * Predefined property name of the operation state number.
     *
     * @constant
     */
    OperationUtils.OSN = 'osn';

    /**
     * Predefined property name of the operation length.
     *
     * @constant
     */
    OperationUtils.OPL = 'opl';

    /**
     * Predefined property name of actions array.
     *
     * @constant
     */
    OperationUtils.ACTIONS = 'actions';

    /**
     * Predefined property name of the operations array within an action object
     *
     * @constant
     */
    OperationUtils.OPERATIONS = 'operations';

    /**
     * Minifying the operations (DOCS-892).
     * Collector for all conversions to minify the operations (and data attribute objects).
     *
     * Syntax:
     *
     * 'keyChanges': The 'key' describes the string to be substituted. The value is an object,
     *               that must contain the key 'n' for 'new'.
     *               Example: "start: { n: 's' }"
     *               The key 'start' is replaced by the string 's'.
     *               Optionally the value object can contain a key 'p' for 'parent'. If this is
     *               specified, the key is only replaced, if the object has the specified parent.
     *               Example: "bold: { n: 'bo', p: { character: 1 } }"
     *               The key 'bold' is replaced by the string 'bo', only if the parent of the
     *               object is 'character'. In this case this is the family of the attribute.
     *               Info: As parent the original (long) key name can always be used!
     *               Info: More than one parent can be specified inside the parent object.
     *
     * 'valueChanges': The 'key' describes the string to be substituted. The value is an object,
     *               that must contain the key 'n' for 'new' and the key 'k' for 'key'.
     *               Example: "insertText: { n: 'it', k: 'n' }"
     *               The value 'insertText' is replaced by the string 'it', if the corresponding
     *               key is 'n' ('n' describes 'name' that is already replaced, because keys are
     *               exchanged before values during minification. During expansion values are first
     *               expanded, then keys).
     *               Optionally the value object can contain a key 'p' for 'parent'. If this is
     *               specified, the value is only replaced, if the object has the specified parent.
     *               Example: "right: { n: 'ri', k: 'al', p: { paragraph: 1 } }"
     *               The value 'right' is replaced by the string 'ri', only if the key is 'alignment'
     *               AND the parent of the object is 'paragraph'. In this case this is the family of
     *               the attribute.
     *               Info: As parent the original (long) key name can always be used!
     *               Info: More than one parent can be specified inside the parent object.
     */
    OperationUtils.CONVERSIONS = {

        keyChanges: {

            // top level keys
            name: { n: 'n' },
            start: { n: 'o' },
            end: { n: 'q' },
            text: { n: 'w' },
            attrs: { n: 'a' },

            styleId: { n: 'st' },
            styleName: { n: 'sn' },
            target: { n: 'ta' },
            type: { n: 'ty' },

            // families

            changes: { n: 'cg' },
            character: { n: 'ch' },
            drawing: { n: 'dr' },
            paragraph: { n: 'pa' },
            presentation: { n: 'pr' },
            shape: { n: 'sh' },
            table: { n: 'tb' },

            // character attributes

            // bold: { n: 'bo', p: { character: 1 } }, // example for parent handling -> only replacing in family 'character'
//            bold: { n: 'bo' },
//            italic: { n: 'it' },
//            underline: { n: 'un' },
//
//            fontSize: { n: 'fs' },
//
//            fontName: { n: 'fn' },
            fontNameAsian: { n: 'fna' },
            fontNameComplex: { n: 'fnc' },
            fontNameEastAsia: { n: 'fnea' },
            fontNameSymbol: { n: 'fns' },
//
//            anchor: { n: 'an' },
//            baseline: { n: 'bl' },
//            vertAlign: { n: 'va' },
//            caps: { n: 'ca' },
//
//            color: { n: 'co' },
//            fillColor: { n: 'fco' },
//
//            type: { n: 'ty' },
//
//            // paragraph attributes
//
//            alignment: { n: 'al' },
//            lineHeight: { n: 'lh' },
//
//            indentLeft: { n: 'il' },
//            indentRight: { n: 'iri' },
//            indentFirstLine: { n: 'ifl' },
//
//            listStyleId: { n: 'lsi' },
//            listStartValue: { n: 'lsv' },
//            listLevel: { n: 'll' },
//            listLabelHidden: { n: 'llh' },
//
//            outlineLevel: { n: 'oll' },
//            tabStops: { n: 'ts' }

            // drawing attributes

            minFrameHeight: { n: 'mfh' }
        },

        valueChanges: {

            // operation names
            changeAutoStyle: { n: 'cas', k: 'n' },
            changeCells: { n: 'cc', k: 'n' },
            changeCFRule: { n: 'ccf', k: 'n' },
            changeColumns: { n: 'ccol', k: 'n' },
            changeComment: { n: 'cco', k: 'n' },
            changeLayout: { n: 'cla', k: 'n' },
            changeMaster: { n: 'cma', k: 'n' },
            changeName: { n: 'cn', k: 'n' },
            changeRows: { n: 'cro', k: 'n' },
            changeStyleSheet: { n: 'css', k: 'n' },
            changeTable: { n: 'ct', k: 'n' },
            changeTableColumn: { n: 'ctc', k: 'n' },
            changeValidation: { n: 'cva', k: 'n' },
            copySheet: { n: 'cs', k: 'n' },
            createError: { n: 'ce', k: 'n' },
            delete: { n: 'd', k: 'n' },
            deleteAutoStyle: { n: 'das', k: 'n' },
            deleteCFRule: { n: 'dcf', k: 'n' },
            deleteChartDataSeries: { n: 'dcd', k: 'n' },
            deleteColumns: { n: 'dcol', k: 'n' },
            deleteComment: { n: 'dco', k: 'n' },
            deleteDrawing: { n: 'dd', k: 'n' },
            deleteHeaderFooter: { n: 'dhf', k: 'n' },
            deleteHyperlink: { n: 'dhy', k: 'n' },
            deleteName: { n: 'dn', k: 'n' },
            deleteNumberFormat: { n: 'dnf', k: 'n' },
            deleteRows: { n: 'dro', k: 'n' },
            deleteSheet: { n: 'dsh', k: 'n' },
            deleteStyleSheet: { n: 'dss', k: 'n' },
            deleteTable: { n: 'dta', k: 'n' },
            deleteValidation: { n: 'dva', k: 'n' },
            group: { n: 'g', k: 'n' },
            insertAutoStyle: { n: 'ias', k: 'n' },
            insertBookmark: { n: 'ibm', k: 'n' },
            insertCells: { n: 'ic', k: 'n' },
            insertCFRule: { n: 'icr', k: 'n' },
            insertChartDataSeries: { n: 'icd', k: 'n' },
            insertColumn: { n: 'ico', k: 'n' },
            insertColumns: { n: 'icol', k: 'n' },
            insertComment: { n: 'icom', k: 'n' },
            insertComplexField: { n: 'icf', k: 'n' },
            insertDrawing: { n: 'id', k: 'n' },
            insertField: { n: 'if', k: 'n' },
            insertFontDescription: { n: 'ifd', k: 'n' },
            insertHardBreak: { n: 'ihb', k: 'n' },
            insertHeaderFooter: { n: 'ihf', k: 'n' },
            insertHyperlink: { n: 'ihl', k: 'n' },
            insertLayoutSlide: { n: 'ils', k: 'n' },
            insertListStyle: { n: 'ili', k: 'n' },
            insertMasterSlide: { n: 'ims', k: 'n' },
            insertName: { n: 'in', k: 'n' },
            insertNumberFormat: { n: 'inf', k: 'n' },
            insertParagraph: { n: 'ip', k: 'n' },
            insertRange: { n: 'ira', k: 'n' },
            insertRows: { n: 'iro', k: 'n' },
            insertSheet: { n: 'ish', k: 'n' },
            insertSlide: { n: 'isl', k: 'n' },
            insertStyleSheet: { n: 'iss', k: 'n' },
            insertTab: { n: 'itb', k: 'n' },
            insertTable: { n: 'ita', k: 'n' },
            insertText: { n: 'it', k: 'n' },
            insertTheme: { n: 'ith', k: 'n' },
            insertValidation: { n: 'iva', k: 'n' },
            mergeCells: { n: 'mc', k: 'n' },
            mergeParagraph: { n: 'mp', k: 'n' },
            mergeTable: { n: 'mt', k: 'n' },
            move: { n: 'mo', k: 'n' },
            moveDrawing: { n: 'md', k: 'n' },
            moveLayoutSlide: { n: 'mls', k: 'n' },
            moveSheet: { n: 'msh', k: 'n' },
            moveSlide: { n: 'msl', k: 'n' },
            noOp: { n: 'no', k: 'n' },
            setAttributes: { n: 'sa', k: 'n' },
            setChartAxisAttributes: { n: 'sca', k: 'n' },
            setChartDataSeriesAttributes: { n: 'scd', k: 'n' },
            setChartGridlineAttributes: { n: 'scg', k: 'n' },
            setChartLegendAttributes: { n: 'scl', k: 'n' },
            setChartTitleAttributes: { n: 'sct', k: 'n' },
            setDocumentAttributes: { n: 'sda', k: 'n' },
            setDrawingAttributes: { n: 'sdr', k: 'n' },
            setSheetAttributes: { n: 'ssa', k: 'n' },
            setSheetName: { n: 'ssn', k: 'n' },
            splitParagraph: { n: 'sp', k: 'n' },
            splitTable: { n: 'st', k: 'n' },
            ungroup: { n: 'ug', k: 'n' },
            unknownValue: { n: 'uk', k: 'n' },
            updateComplexField: { n: 'ucf', k: 'n' },
            updateField: { n: 'uf', k: 'n' }

            // property values

//            baseline: { n: 'bl', k: 'va' },
//            auto: { n: 'au', k: 'ty' },
//            percent: { n: 'pc', k: 'ty' },
//
//            left: { n: 'le', k: 'al' },
//            // right: { n: 'ri', k: 'al', p: { paragraph: 1 } } // example for parent handling -> only replacing in family 'paragraph'
//            right: { n: 'ri', k: 'al' }

        }
    };

    /**
     * Expanding the operations (DOCS-892)
     * Collector for all conversions to expand the minified operations (and other objects).
     * This object is generated automatically within the function
     * OperationUtils.generateExpandConversions.
     * The base for the expansion is the object OperationUtils.CONVERSIONS.
     */
    OperationUtils.EXPAND_CONVERSIONS = null;

    // static methods ---------------------------------------------------------

    /**
     * Retrieves the operation state number from an operation object.
     *
     * @param {Object} operation
     *  A JSON operation object with properties.
     *
     * @returns {Number}
     *  The operation state number or -1, if it cannot be retrieved.
     */
    OperationUtils.getOSN = function (operation) {
        return Utils.getIntegerOption(operation, OperationUtils.OSN, -1);
    };

    /**
     * Retrieves the operation length from an operation object.
     *
     * @param {Object} operation
     *  A JSON operation object with properties.
     *
     * @returns {Number}
     *  The operation length or -1, if it cannot be retrieved.
     */
    OperationUtils.getOPL = function (operation) {
        return Utils.getIntegerOption(operation, OperationUtils.OPL, -1);
    };

    /**
     * Retrieves the actions array from an actions object or the actions array
     * itself.
     *
     * @param {Array<Object>|Object} actions
     *
     * @returns {Array<Object>}
     */
    OperationUtils.getActionsArray = function (actions) {
        return (_.isObject(actions) && _.isArray(actions[OperationUtils.ACTIONS])) ?
               actions[OperationUtils.ACTIONS] : actions;
    };

    /**
     * Returns the next operation state number which must follow
     * on the operations stored in the actions array.
     *
     * @param {Array<Object>} opsArray
     *  The array with JSON operation objects.
     *
     * @returns {Number}
     *  The next operation state number or -1 if the next number
     *  cannot be determined (e.g. empty array).
     */
    OperationUtils.getNextOperationStateNumber = function (opsArray) {
        var // calculated osn
            osn = -1,
            // the operation length
            opl = -1,
            // last operation
            operation = null;

        if (_.isArray(opsArray)) {
            operation = _.last(opsArray);
            osn = OperationUtils.getOSN(operation);
            opl = OperationUtils.getOPL(operation);
            osn = ((osn !== -1) && (opl !== -1)) ? osn + opl : -1;
        }

        return osn;
    };

    /**
     * Returns the starting operation state number from an operations
     * array.
     *
     * @params {Array<Object>} opsArray
     *  An array with JSON operation objects.
     *
     * @returns {Number}
     *  The operation state number of the first operation in the array, or -1
     *  if the operation state number cannot be determined (e.g. empty array).
     */
    OperationUtils.getStartingOperationStateNumber = function (opsArray) {
        return (_.isArray(opsArray)) ? OperationUtils.getOSN(_.first(opsArray)) : -1;
    };

    /**
     * Generating the conversion object for expanding the operations (DOCS-892).
     * This function is only called once and creates automatically the object that can
     * be used to expand minified operations.
     * Using the specified object 'OperationUtils.CONVERSIONS' for minifying operations
     * and objects, within this function the object 'OperationUtils.EXPAND_CONVERSIONS'
     * is generated, so that operations and other objects can be expanded again.
     */
    OperationUtils.generateExpandConversions = function () {

        var keyChanges = OperationUtils.CONVERSIONS.keyChanges;
        var valueChanges = OperationUtils.CONVERSIONS.valueChanges;

        OperationUtils.EXPAND_CONVERSIONS = { keyChanges: {}, valueChanges: {} };

        _.each(keyChanges, function (val, key) {

            var newVal = key;
            var newKey = val.n;

            OperationUtils.EXPAND_CONVERSIONS.keyChanges[newKey] = _.copy(val, true); // deep copy of value
            OperationUtils.EXPAND_CONVERSIONS.keyChanges[newKey].n = newVal;
        });

        _.each(valueChanges, function (val, key) {

            var newVal = key;
            var newKey = val.n;

            OperationUtils.EXPAND_CONVERSIONS.valueChanges[newKey] = _.copy(val, true); // deep copy of value
            OperationUtils.EXPAND_CONVERSIONS.valueChanges[newKey].n = newVal;
        });

    };

    /**
     * Minifying or expanding one specified object (DOCS-892).
     *
     * @params {Object} obj
     *  An object (for example an operation or attribute object) that shall be minified or
     *  expanded.
     *
     * @params {String} parent
     *  A string that specifies a parent for the given object. This can be used to specify
     *  the minification of the object to selected parents.
     *  Example: Using parent 'character' reduces the minification to those keys that are
     *  inside the character attribute object. If same keys or values are specified also
     *  inside another family, they are not replaced there.
     *  For more information look at the description of the object 'OperationUtils.CONVERSIONS'.
     *
     * @params {Boolean} [expand]
     *  Whether the specified object shall be expanded or minified. If this parameter is
     *  set to a truthy value, the object will be expanded. Otherwise (or if not specified)
     *  the object will be minified.
     */
    OperationUtils.handleMinifiedObject = function (obj, parent, expand) {

        if (!OperationUtils.isConversionDefined()) { return; }

        if (expand && !OperationUtils.EXPAND_CONVERSIONS) { OperationUtils.generateExpandConversions(); }

        var keyChanges = expand ? OperationUtils.EXPAND_CONVERSIONS.keyChanges : OperationUtils.CONVERSIONS.keyChanges;
        var valueChanges = expand ? OperationUtils.EXPAND_CONVERSIONS.valueChanges : OperationUtils.CONVERSIONS.valueChanges;

        function changeKeys(key) {

            var newKey = null;
            var doReplace = false;

            if (keyChanges[key]) {

                doReplace = !parent || !keyChanges[key].p || keyChanges[key].p[parent] === 1;

                if (doReplace) {
                    newKey = keyChanges[key].n;

                    obj[newKey] = obj[key];
                    delete obj[key];
                    key = newKey;
                }
            }

            return key;
        }

        function changeValues(key, val) {

            var oneKey = null;
            var doReplace = false;

            if (valueChanges[val]) {

                doReplace = !parent || !valueChanges[val].p || valueChanges[val].p[parent] === 1;

                if (doReplace) {

                    oneKey = valueChanges[val].k;
                    if (obj[oneKey] && oneKey === key) {
                        obj[oneKey] = valueChanges[val].n;
                    }
                }
            }

        }

        _.each(obj, function (val, key) {

            var currentKey = null;

            if (expand) {
                changeValues(key, val);
                changeKeys(key);
            } else {
                currentKey = changeKeys(key);
                changeValues(currentKey, val);
            }

            if (_.isObject(val) && (!_.isArray(val) || _.isObject(val[0]))) { // TODO: check for more optimal solution
                OperationUtils.handleMinifiedObject(val, key, expand); // recursive call (using original, not shortened key)
            }

        });

    };

    /**
     * Minifying or expanding the specified action(s) (DOCS-892).
     *
     * In the case of expanding the operations (doExpand is true) the original operations
     * array must be used and modified. The version with the shortened operations cannot
     * be used on client side.
     *
     * This is different in the case of shrinking operations (doExpand is a falsy value).
     * This call is used, before operations are sent to the server. In this case are
     * problems, if the original array with the original operations is modified (55226).
     *
     * @params {Object[]} opsArray
     *  An array of actions as it is used to send actions to the server or when
     *  receiving actions from the server.
     *
     * @params {Boolean} doExpand
     *  Whether the specified actions shall be expanded or minified. If this parameter is
     *  set to a truthy value, the actions will be expanded. Otherwise (or if not specified)
     *  the actions will be minified.
     *
     * @returns {Object[]}
     *  An array of actions.
     *  If 'doExpand' is truthy this is the reference to the original parameter 'opsArray'.
     *  If 'doExpand' is falsy, this must be a reference to a new generated array.
     */
    OperationUtils.handleMinifiedActions = function (opsArray, doExpand) {

        function iterateOneAction(action) {
            var allOps = action.operations;
            if (!allOps) { return; }
            _.each(allOps, function (op) {
                OperationUtils.handleMinifiedObject(op, '', doExpand);
            });
        }

        if (!opsArray || !OperationUtils.isConversionDefined()) { return; }

        // working on the original array for expansion, but using a clone for shrinking
        // -> a clone is required to avoid side effects with objects that directly
        //    included into operations during operation generation.
        var localOps = doExpand ? opsArray : _.copy(opsArray, true);

        _.each(localOps, function (action) {
            iterateOneAction(action);
        });

        return localOps;
    };

    /**
     * Whether key or value changes are defined.
     *
     * @returns {Boolean}
     *  Whether at least one key or value change is defined.
     */
    OperationUtils.isConversionDefined = function () {
        return !_.isEmpty(OperationUtils.CONVERSIONS.keyChanges) || !_.isEmpty(OperationUtils.CONVERSIONS.valueChanges);
    };

    // exports ================================================================

    return OperationUtils;

});
