/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Peter Seliger <peter.seliger@open-xchange.com>
 */

define('io.ox/office/editframework/utils/gradient', [

    'io.ox/office/tk/utils',

    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/utils/color'

], function (Utils, AttributeUtils, Color) {

    'use strict';

    // locally scoped helper methods and object shortcuts =====================

    var
        global          = window,

        Object          = global.Object,
        Array           = global.Array,

        NOOP            = global.Function.prototype,

        JSON            = global.JSON,
        Math            = global.Math,

        document        = global.document,

        expose_internal_class_name  = Object.prototype.toString,
        regXObjectObject            = (/^\[object\s+Object]$/),
        regXCanvasElement           = (/^\[object\s+HTMLCanvasElement]$/),

        isEqualValue    = _.isEqual,

        isFunction      = _.isFunction,
      //isArray         = _.isArray,
      //isObject        = _.isObject,

        isObjectObject  = function (type) {
            return (!!type && regXObjectObject.test(expose_internal_class_name.call(type)));
        },
        object_extend   = _.extend,

        array_from      = (isFunction(Array.from) && Array.from) || (function (array_prototype_slice) {
            return function (type) {

                return array_prototype_slice.call(type);
            };
        }(Array.prototype.slice)),

      //json_parse      = JSON.parse,
        json_stringify  = JSON.stringify,

        math_abs        = Math.abs,

        math_min        = Math.min,
        math_max        = Math.max,

      //math_ceil       = Math.ceil,
        math_round      = Math.round,

        math_radix      = function getRadix(angleDegree) {
            return (angleDegree * MATH_PI / 180);
        },
        math_degree     = function getRadix(angleRadix) {
            return (angleRadix * 180 / MATH_PI);
        },
        math_cos        = Math.cos,
        math_sin        = Math.sin,
        math_tan        = Math.tan,
        math_atan       = Math.atan,

        MATH_PI         = Math.PI,

        IMAGE_BLUEPRINT = document.createElement('img'),

        CANVAS_ELEMENT  = document.createElement('canvas'),
        CANVAS_CONTEXT  = CANVAS_ELEMENT.getContext('2d'),

        GRADIENT_FILLTYPE_OPERATIONS  = {
            linear_and_bound_false:       getGradientFillLinearUnbound,
            linear_and_bound_true:        getGradientFillLinearBound,

            path_circle_and_bound_false:  getGradientFillRadialUnbound,
            path_circle_and_bound_true:   getGradientFillRadialBound,

            path_rect_and_bound_false:    getGradientFillRectangularUnbound,
            path_rect_and_bound_true:     getGradientFillRectangularBound,

            path_shape_and_bound_false:   getGradientFillShapeUnbound,
            path_shape_and_bound_true:    getGradientFillShapeBound
        },
        LITERAL_PREFIX__PATH_TYPE     =   'path_',
        LITERAL_BRIDGE__AND_BOUND     =   '_and_bound_';

    // private static (locally scoped) list of gradient references ============

    // var gradientList = []; // @NOTIFICATION: for debugging only.

    // private static (locally scoped) class methods ==========================

    function returnCleanSketchSheet(width, height) {
        CANVAS_ELEMENT.width  = width;
        CANVAS_ELEMENT.height = height;
      //CANVAS_ELEMENT.style.width  = (width  + 'px');
      //CANVAS_ELEMENT.style.height = (height + 'px');

      //CANVAS_CONTEXT = CANVAS_ELEMENT.getContext('2d');
        CANVAS_CONTEXT.clearRect(0, 0, width, height);

        return {
            canvas:   CANVAS_ELEMENT,
            context:  CANVAS_CONTEXT
        };
    }

    function createCleanSketchSheetClone(width, height) {
        var
            context,
            canvas = CANVAS_ELEMENT.cloneNode();

        canvas.width  = width   || (width   = CANVAS_ELEMENT.width);
        canvas.height = height  || (height  = CANVAS_ELEMENT.height);

        context = canvas.getContext('2d');
        context.clearRect(0, 0, width, height);

        return {
            canvas:   canvas,
            context:  context
        };
    }

    // function createImageElementFromCanvas/*Context*/(canvas, width, height) {
    //     var
    //       //imageData   = context.getImageData(0, 0, width, height),
    //       //elmImage    = document.createElement('img');
    //
    //         elmImage    = IMAGE_BLUEPRINT.cloneNode();
    //
    //     elmImage.width  = /*imageData.*/width || canvas.width;
    //     elmImage.height = /*imageData.*/height || canvas.height;
    //
    //     elmImage.src    = canvas.toDataURL();
    //   //elmImage.src    = ['data:', imageData.data].join('');
    //
    //     return elmImage;
    // }

    /**
     *
     * @param sketchCanvas
     * @param drawingContext
     * @returns {CanvasPattern}
     */
    function createPatternFillFromCanvas(sketchCanvas, drawingContext) {

        return drawingContext.createPattern(sketchCanvas, 'no-repeat');
    }

    /**
     *
     * @param context
     * @param gradientFill
     * @param rotation
     * @param boxMeasures
     * @returns {CanvasPattern}
     */
    function renderRotatedPatternFromGradientFill(context, gradientFill, rotation, boxMeasures) {
        var
            canvas          = context.canvas,

            fillRextWidth   = canvas.width,
            fillRextHeight  = canvas.height,
            halfRectWidth   = math_round(fillRextWidth / 2),
            halfRectHeight  = math_round(fillRextHeight / 2),

            snapBoxWidth    = boxMeasures.width,
            snapBoxHeight   = boxMeasures.height,
            halfBoxWidth    = math_round(snapBoxWidth / 2),
            halfBoxHeight   = math_round(snapBoxHeight / 2),

            // translateX      = fillRextWidth - boxMeasures.mx,
            // translateY      = fillRextHeight - boxMeasures.my,

            bitmapSheet     = createCleanSketchSheetClone(snapBoxWidth, snapBoxHeight),
            bitmapCanvas    = bitmapSheet.canvas,
            bitmapContext   = bitmapSheet.context,

            sketchSheet     = returnCleanSketchSheet(fillRextWidth, fillRextHeight),
            sketchCanvas    = sketchSheet.canvas,
            sketchContext   = sketchSheet.context;

        sketchContext.setTransform(1, 0, 0, 1, 0, 0);

        bitmapContext.fillStyle = gradientFill;
        bitmapContext.fillRect(0, 0, snapBoxWidth, snapBoxHeight);

        sketchContext.translate(halfRectWidth, halfRectHeight);

      //sketchContext.translate((halfRectWidth + boxMeasures.x1), (halfRectHeight - boxMeasures.y1));
      //sketchContext.translate(translateX, translateY);

        sketchContext.rotate(math_radix(rotation));
      //sketchContext.translate(math_round(boxMeasures.x1 / 2), -math_round(boxMeasures.y1 / 2));

        sketchContext.drawImage(bitmapCanvas, -halfBoxWidth, -halfBoxHeight, snapBoxWidth, snapBoxHeight);

        sketchContext.setTransform(1, 0, 0, 1, 0, 0);
      //sketchContext.rotate(-math_radix(rotation));
      //sketchContext.translate(-halfRectWidth, -halfRectHeight);

      //sketchContext.translate(-translateX, -translateY);
      //sketchContext.translate((0 - boxMeasures.mx), (0 - boxMeasures.my));

        // // debug
        // //
        // var
        //     dumpCanvas  = document.createElement('canvas'),
        //     dumpContext;
        //
        // dumpCanvas.width  = boxMeasures.width;
        // dumpCanvas.height = boxMeasures.height;
        //
        // dumpContext = dumpCanvas.getContext('2d');
        // dumpContext.fillStyle = gradientFill;
        // dumpContext.fillRect(0, 0, boxMeasures.width, boxMeasures.height);
        //
        // document.body.appendChild(dumpCanvas);
        // dumpCanvas.style.position = 'absolute';
        // //
        // // debug

        return createPatternFillFromCanvas(sketchCanvas, context);
    }

    function getFillPathMeasures(rectWidth, rectHeight, gradientValue) {
        var
            x1 = math_round(gradientValue.pathFillLeft  * rectWidth),
            y1 = math_round(gradientValue.pathFillTop   * rectHeight),

            x2 = math_round(rectWidth   - (gradientValue.pathFillRight  * rectWidth)),
            y2 = math_round(rectHeight  - (gradientValue.pathFillBottom * rectHeight));

        return {
            x1:     x1,
            y1:     y1,
            x2:     x2,
            y2:     y2,
            width:  math_abs(x2 - x1),
            height: math_abs(y2 - y1)
        };
    }

    function getTileRectMeasures(rectWidth, rectHeight, gradientValue) {
        var
            x1 = math_round(gradientValue.left  * rectWidth),
            y1 = math_round(gradientValue.top   * rectHeight),

            x2 = math_round(rectWidth   - (gradientValue.right  * rectWidth)),
            y2 = math_round(rectHeight  - (gradientValue.bottom * rectHeight));

        return {
            x1:     x1,
            y1:     y1,
            x2:     x2,
            y2:     y2,
            width:  math_abs(x2 - x1),
            height: math_abs(y2 - y1)
        };
    }

    function getSnapBoxMeasuresForRotatedShape(pathCoords, rotation) {
        var
            mx  = pathCoords.mx,
            my  = pathCoords.my,

            deltaY, // a - cathetus
            deltaX, // b - cathetus
            radius, // c - hypotenuse

            radixA, // alpha angle (in radix)
            q,      // ordinal number of the Cartesian quadrant.

            xMin, yMin, xMax, yMax;

        // calculate rotated values for each {x, y} coordinate tuple.
        pathCoords.xValues.forEach(function (x, idx/*, pathCoords.xValues*/) {
            var
                y = pathCoords.yValues[idx];

            deltaY = my - y;
            deltaX = mx - x;
            radius = Utils.radius(deltaX, deltaY);

            q = (deltaX >= 0) ? ((deltaY >= 0) ? 2 : 3) : ((deltaY >= 0) ? 1 : 4);

            deltaY = math_abs(deltaY);
            deltaX = math_abs(deltaX);

            radixA = (deltaY === 0) ? 0 : Math.atan(deltaY / deltaX);

            if (q === 2) {
                radixA = math_radix(180) - radixA;
            } else if (q === 3) {
                radixA = math_radix(180) + radixA;
            } else if (q === 4) {
                radixA = math_radix(360) - radixA;
            }
            radixA = math_radix((math_degree(radixA) - rotation + 360) % 360);

            pathCoords.xValues[idx] = mx + math_round(math_cos(radixA) * radius);
            pathCoords.yValues[idx] = my + math_round(math_sin(radixA) * radius);
        });

        xMin = math_min.apply(Math, pathCoords.xValues);
        yMin = math_min.apply(Math, pathCoords.yValues);

        xMax = math_max.apply(Math, pathCoords.xValues);
        yMax = math_max.apply(Math, pathCoords.yValues);

        return {
            x1: xMin,
            y1: yMin,

            x2: xMax,
            y2: yMax,

          //mx: math_round((xMin + xMax) / 2),
          //my: math_round((yMin + yMax) / 2),
            mx: (xMax - math_round((xMax - xMin) / 2)),
            my: (yMax - math_round((yMax - yMin) / 2)),

            width:  (xMax - xMin),
            height: (yMax - yMin)
          //width:  (math_abs(xMin) + math_abs(xMax)),
          //height: (math_abs(yMin) + math_abs(yMax))
        };
    }

    /**
     * TODO - <peter.seliger@open-xchange.com> needs to provide documentation entirely for this module.
     *
     * @param rectWidth
     * @param rectHeight
     * @param rotation
     * @returns {{width: *, height: *}}
     */
    function getBoundingBoxMeasuresForRotatedDrawingRect(rectWidth, rectHeight, rotation) {
        var
            radius = Utils.radius(rectWidth, rectHeight) / 2,

            // adjusting the values to a system different from the Cartesian one.
            radixD = math_radix(((90  + 45 - rotation) + 360) % 360),
            radixC = math_radix(((0   + 45 - rotation) + 360) % 360),
            radixB = math_radix(((270 + 45 - rotation) + 360) % 360),
            radixA = math_radix(((180 + 45 - rotation) + 360) % 360),

            cornerPointList = [{
                // corner D
                x: math_round(math_cos(radixD) * radius),
                y: math_round(math_sin(radixD) * radius)
            }, {
                // corner C
                x: math_round(math_cos(radixC) * radius),
                y: math_round(math_sin(radixC) * radius)
            }, {
                // corner B
                x: math_round(math_cos(radixB) * radius),
                y: math_round(math_sin(radixB) * radius)
            }, {
                // corner A
                x: math_round(math_cos(radixA) * radius),
                y: math_round(math_sin(radixA) * radius)
            }],
            cornerPointExtrema = cornerPointList.reduce(function (collector, point) {

                collector.xMin = (collector.xMin === null) ? point.x : math_min(collector.xMin, point.x);
                collector.xMax = (collector.xMax === null) ? point.x : math_max(collector.xMax, point.x);
                collector.yMin = (collector.yMin === null) ? point.y : math_min(collector.yMin, point.y);
                collector.yMax = (collector.yMax === null) ? point.y : math_max(collector.yMax, point.y);

                return collector;

            }, { xMin: null, xMax: null, yMin: null, yMax: null });

        return {
            width: (math_abs(cornerPointExtrema.xMax) + math_abs(cornerPointExtrema.xMin)),
            height: (math_abs(cornerPointExtrema.yMax) + math_abs(cornerPointExtrema.yMin))
        };
    }/*

    function getLinearGradientCoordinates(drawingRectWidth, drawingRectHeight, boundingBoxWidth, boundingBoxHeight, rotation) {
        var
            mx      = math_round(drawingRectWidth / 2),   // center of drawing rect or canvas.
            my      = math_round(drawingRectHeight / 2),  //

            deltaY  = math_round(math_abs(boundingBoxHeight - drawingRectHeight) / 2),  // later used for calculating of cathetus (a).
            deltaX  = math_round(math_abs(boundingBoxWidth - drawingRectWidth) / 2),    // later used for calculating of cathetus (b).

            radius, // c - hypotenuse

            xValues = [(0 - deltaX), (drawingRectWidth + deltaX)],
          //yValues = [my, my],
            yValues = [0, 0],

            radixA, // alpha angle (in radix)
            q;      // ordinal number of the Cartesian quadrant.

        // calculate rotated values for each {x, y} coordinate tuple.
        xValues.forEach(function (x, idx/ *, xValues* /) {
            var
                y = yValues[idx];

            deltaY = my - y;
            deltaX = mx - x;

            radius = Utils.radius(deltaX, deltaY);

            q = (deltaX >= 0) ? ((deltaY >= 0) ? 2 : 3) : ((deltaY >= 0) ? 1 : 4);

            deltaY = math_abs(deltaY);
            deltaX = math_abs(deltaX);

            radixA = (deltaY === 0) ? 0 : Math.atan(deltaY / deltaX);

            if (q === 2) {
                radixA = math_radix(180) - radixA;
            } else if (q === 3) {
                radixA = math_radix(180) + radixA;
            } else if (q === 4) {
                radixA = math_radix(360) - radixA;
            }
            radixA = math_radix((math_degree(radixA) - rotation + 360) % 360);

            xValues[idx] = (mx + math_round(math_cos(radixA) * radius));
            yValues[idx] = (my + math_round(math_sin(radixA) * radius));
        });

        return {
            x1: xValues[0],
            y1: yValues[0],

            x2: xValues[1],
            y2: yValues[1]
        };
    }*/

    // /**
    //  *
    //  * @param halfWidth
    //  * @param halfHeight
    //  * @param rotationValue
    //  * @returns {{x1: *, y1: *, x2: *, y2: *}}
    //  */
    // function getGradientLineCoordinates(halfWidth, halfHeight, rotationValue) {
    //     var
    //         // gradient path computation seems to be done strictly within the boundaries
    //         // of a rectangular box that encloses its shape/drawing.
    //         // the rectangle's geometric centre most probably is the zero point of a
    //         // strangely flipped Cartesian coordinate system.
    //         //
    //         // - the assumption did prove right.
    //         // - all test documents render according to theirs powerpoint pendants and
    //         //   do pass the document roundtrip as well.
    //         //
    //         x1, y1, x2, y2,
    //
    //         a,
    //         b,
    //         angleDegree = (rotationValue % 90); // ALPHA angel.
    //
    //     if (
    //         ((rotationValue >=   0) && (rotationValue <  90)) ||
    //         ((rotationValue >= 180) && (rotationValue < 270))
    //     ) {
    //         // "adjacent side" computing / de: Ankatheten-Berechnung
    //         a = halfWidth;
    //         b = (math_tan(math_radix(angleDegree)) * a);
    //
    //         if (b > halfHeight) {
    //             b = halfHeight;
    //             a = (math_tan(math_radix(90 - angleDegree)) * b);
    //         }
    //         // flipping the values according to a system different from the Cartesian one.
    //         if ((rotationValue >= 0) && (rotationValue <  90)) {
    //             x1 = halfWidth  - a;
    //             y1 = halfHeight - b;
    //             x2 = halfWidth  + a;
    //             y2 = halfHeight + b;
    //         } else {
    //             x1 = halfWidth  + a;
    //             y1 = halfHeight + b;
    //             x2 = halfWidth  - a;
    //             y2 = halfHeight - b;
    //         }
    //     } else {
    //         // "adjacent side" computing / de: Ankatheten-Berechnung
    //         b = halfHeight;
    //         a = (math_tan(math_radix(angleDegree)) * b);
    //
    //         if (a > halfWidth) {
    //             a = halfWidth;
    //             b = (math_tan(math_radix(90 - angleDegree)) * a);
    //         }
    //         // flipping the values according to a system different from the Cartesian one.
    //         if ((rotationValue >= 90) && (rotationValue <  180)) {
    //             x1 = halfWidth  + a;
    //             y1 = halfHeight - b;
    //             x2 = halfWidth  - a;
    //             y2 = halfHeight + b;
    //         } else {
    //             x1 = halfWidth  - a;
    //             y1 = halfHeight + b;
    //             x2 = halfWidth  + a;
    //             y2 = halfHeight - b;
    //         }
    //     }
    //     return { x1: x1, y1: y1, x2: x2, y2: y2 };
    // }

    /**
     *
     * @param rectWidth
     * @param rectHeight
     * @param rotation
     * @returns {{x1: *, y1: *, x2: *, y2: *}}
     */
    function getLinearGradientCoordinates(rectWidth, rectHeight, rotation) {
        var
            // gradient path computation seems to be done strictly within the boundaries
            // of a rectangular box that encloses its shape/drawing.
            // the rectangle's geometric centre most probably is the zero point of a
            // strangely flipped Cartesian coordinate system.
            //
            // - the assumption did prove right.
            // - all test documents render according to theirs powerpoint pendants and
            //   do pass the document roundtrip as well.
            //
            x1, y1, x2, y2,

            angle,
            hypotenuse,
            oppositeLeg,

            partialSquareDegree = (rotation % 90),
            boundaryAngleDegree = math_atan(rectHeight / rectWidth);

        angle = math_radix(partialSquareDegree);

        if ((rotation >= 0) && (rotation <  90)) {
            if (rotation < boundaryAngleDegree) {

                x1 = 0;
                y1 = 0;
                x2 = math_round(rectWidth / math_cos(angle));
                y2 = math_round((rectWidth * math_tan(angle)) + ((x2 - rectWidth) * math_tan(angle)));

            } else {
                angle       = math_radix(90 - partialSquareDegree);
                hypotenuse  = (rectWidth - (rectHeight * math_tan(angle)));
                oppositeLeg = (hypotenuse * math_sin(angle));

                x2 = rectWidth;
                y2 = rectHeight;
                x1 = math_round(hypotenuse - (oppositeLeg * math_sin(angle)));
                y1 = (0 - math_round(oppositeLeg * math_cos(angle)));
            }
        } else if ((rotation >= 90) && (rotation <  180)) {
            if (rotation < boundaryAngleDegree) {

                x1 = rectWidth;
                y1 = 0;
                y2 = math_round(rectHeight / math_cos(angle));
                x2 = math_round((rectHeight * math_tan(angle)) + ((y2 - rectHeight) * math_tan(angle)));

            } else {
                angle       = math_radix(90 - partialSquareDegree);
                hypotenuse  = (rectHeight - (rectWidth * math_tan(angle)));
                oppositeLeg = (hypotenuse * math_sin(angle));

                x2 = 0;
                y2 = rectHeight;
                y1 = math_round(hypotenuse - (oppositeLeg * math_sin(angle)));
                x1 = (rectWidth + math_round(oppositeLeg * math_cos(angle)));
            }
        } else if ((rotation >= 180) && (rotation < 270)) {
            if (rotation < boundaryAngleDegree) {

                x1 = rectWidth;
                y1 = rectHeight;
                x2 = (rectWidth - math_round(rectWidth / math_cos(angle)));
                y2 = (rectHeight - math_round((rectWidth * math_tan(angle)) + ((x2 - rectWidth) * math_tan(angle))));

            } else {
                angle       = math_radix(90 - partialSquareDegree);
                hypotenuse  = (rectWidth - (rectHeight * math_tan(angle)));
                oppositeLeg = (hypotenuse * math_sin(angle));

                x2 = 0;
                y2 = 0;
                x1 = (rectWidth - math_round(hypotenuse - (oppositeLeg * math_sin(angle))));
                y1 = (rectHeight + math_round(oppositeLeg * math_cos(angle)));
            }
        } else {
            if (rotation < boundaryAngleDegree) {

                x1 = 0;
                y1 = rectHeight;
                y2 = math_round(rectHeight / math_cos(angle));
                x2 = math_round((rectHeight * math_tan(angle)) + ((y2 - rectHeight) * math_tan(angle)));

            } else {
                angle       = math_radix(90 - partialSquareDegree);
                hypotenuse  = (rectHeight - (rectWidth * math_tan(angle)));
                oppositeLeg = (hypotenuse * math_sin(angle));

                x2 = rectWidth;
                y2 = 0;
                y1 = (rectHeight - math_round(hypotenuse - (oppositeLeg * math_sin(angle))));
                x1 = (0 - math_round(oppositeLeg * math_cos(angle)));
            }
        }

        return { x1: x1, y1: y1, x2: x2, y2: y2 };
    }

    function parseColorStopsForFillPathRectangular(colorStopList) {
        // halven of all positions and inverse duplicating the before halvened range.

        colorStopList = array_from(colorStopList).map(function (colorStopValue) {
            return {
                position: (math_round(((1 - colorStopValue.position) / 2) * 100000) / 100000),
                color:    colorStopValue.color
            };
        });

        return colorStopList.concat(
            array_from(colorStopList).reverse().map(function (colorStopValue) {
                return {
                    position: (1 - colorStopValue.position),
                    color:    colorStopValue.color
                };
            })
        );
    }

    function renderGradientFillRectangular(context, fillRect, gradientFillVertical, gradientFillHorizontal) {

        // // draw gradient fill-path "bow tie"-like :: fill path vertically with vertical gradient
        // context.beginPath();
        // context.moveTo(fillRect.x2, fillRect.y1); // "bow tie" start
        // context.lineTo(fillRect.x1, fillRect.y2); // ...
        // context.lineTo(fillRect.x2, fillRect.y2); // ...
        // context.lineTo(fillRect.x1, fillRect.y1); // "bow tie" end.

        // draw gradient fill-path "bow tie"-like :: fill path vertically with vertical gradient
        //
        context.beginPath();                            // "bow tie" start for the first (hvertical) "bow tie" ...
        context.moveTo((fillRect.x2 + 1), fillRect.y1); // ... with 1px overlapping of its edges
        context.lineTo((fillRect.x1 - 1), fillRect.y2); // ... in order to prevent sparkling pixels
        context.lineTo((fillRect.x2 + 1), fillRect.y2); // ... along the 2 just colliding fill paths.
        context.lineTo((fillRect.x1 - 1), fillRect.y1); // ... "bow tie" end.

        context.fillStyle = gradientFillVertical;
        context.fill();

        // draw gradient fill-path "bow tie"-like :: fill path horizontally with horizontal gradient
        //
        context.beginPath();                            // "bow tie" start for the second (horizontal) "bow tie" ...
        context.moveTo((fillRect.x1 + 1), fillRect.y1); // ...
        context.lineTo((fillRect.x2 - 1), fillRect.y2); // ... with 1px overlapping of its edges
        context.lineTo(fillRect.x2, fillRect.y2);       // ... in order to prevent sparkling pixels
        context.lineTo(fillRect.x2, fillRect.y1);       // ... along the 2 just colliding fill paths.
        context.lineTo((fillRect.x2 - 1), fillRect.y1); // ...
        context.lineTo((fillRect.x1 + 1), fillRect.y2); // ...
        context.lineTo(fillRect.x1, fillRect.y2);       // ... "bow tie" end.

        context.fillStyle = gradientFillHorizontal;
        context.fill();
    }

    // function renderGradientFillLinear(context, colorStopList, canvasWidth, canvasHeight, rotation) {
    //     var
    //       //boxMeasures   = getBoundingBoxMeasuresForRotatedDrawingRect(canvasWidth, canvasHeight, rotation),
    //         boxMeasures   = getSnapBoxMeasuresForRotatedShape({
    //
    //             mx: (canvasWidth / 2),
    //             my: (canvasHeight / 2),
    //             xValues: [0, canvasWidth, canvasWidth, 0],
    //             yValues: [0, 0, canvasHeight, canvasHeight]
    //
    //         }, rotation),
    //
    //         boxWidth      = boxMeasures.width,
    //         boxHeight     = boxMeasures.height,
    //
    //         halfBoxWidth  = (boxWidth / 2),
    //         halfBoxHeight = (boxHeight / 2),
    //
    //         // halfDeltaX    = math_round(math_abs(boxWidth - canvasWidth) / 2),
    //         // halfDeltaY    = math_round(math_abs(boxHeight - canvasHeight) / 2),
    //
    //         // pivotX        = (0 - halfDeltaX), // - canvas pivot-x default-left: 0
    //         // pivotY        = (0 - halfDeltaY), // - canvas pivot-y default-top : 0
    //
    //         fillSheet     = createCleanSketchSheetClone(boxWidth, boxHeight),
    //       //fillCanvas,   = fillSheet.canvas,
    //         fillContext   = fillSheet.context,
    //
    //         sketchSheet   = returnCleanSketchSheet(canvasWidth, canvasHeight),
    //         sketchCanvas  = sketchSheet.canvas,
    //         sketchContext = sketchSheet.context,
    //
    //         // fill outer boundaries.
    //         gradientFill = createGradientFillLinear(fillContext, colorStopList, 0, 0, boxWidth, 0);
    //
    //     sketchContext.save();                                       // - first save the untranslated/unrotated context.
    //     sketchContext.beginPath();                                  //
    //                                                                 //
    //     sketchContext.translate(halfBoxWidth, halfBoxHeight);       // - move the pivot point to the center of the rect.
    //     sketchContext.rotate(math_radix(rotation));                 // - rotate the rect.
    //                                                                 //
    //     sketchContext.rect((-1 * halfBoxWidth), (-1 * halfBoxHeight), boxWidth, boxHeight); // - draw the rect on the transformed context.
    //                                                                                         //   Note: after transforming [0, 0] is visually [x, y],
    //                                                                                         //   so the rect needs to be offset accordingly when drawn.
    //     sketchContext.fillStyle = gradientFill;                     //
    //     sketchContext.fill();                                       //
    //                                                                 //
    //     sketchContext.restore();                                    // - restore the context to its untranslated/unrotated state.
    //
    //     return createPatternFillFromCanvas(sketchCanvas, context);
    // }

    /**
     *
     * @param context
     * @param colorStopList
     * @param x1
     * @param y1
     * @param x2
     * @param y2
     * @returns {CanvasGradient}
     */
    function createGradientFillLinear(context, colorStopList, x1, y1, x2, y2) {
        var
            gradientFill = context.createLinearGradient(x1, y1, x2, y2);

        colorStopList.forEach(function (colorStopValue) {
            gradientFill.addColorStop(colorStopValue.position, colorStopValue.color);
        });
        return gradientFill;
    }

    function getGradientFillShapeUnbound(/* context, gradientValue, parentRotation, pathCoords */) {
        // Utils.info('+++ getGradientFillShapeUnbound +++ [gradientValue, parentRotation, pathCoords] : ', gradientValue, parentRotation, pathCoords);
    }

    function getGradientFillShapeBound(/* context, gradientValue */) {
        // Utils.info('+++ getGradientFillShapeBound +++ [gradientValue] : ', gradientValue);
    }

    function getGradientFillRectangularUnbound(context, gradientValue, parentRotation, pathCoords) {
        // Utils.info('+++ getGradientFillRectangularUnbound +++ [gradientValue, parentRotation, pathCoords] : ', gradientValue, parentRotation, pathCoords);
        var
          //canvas            = context.canvas,
          //canvasWidth       = canvas.width,
          //canvasHeight      = canvas.height,

          //boxMeasures       = getBoundingBoxMeasuresForRotatedDrawingRect(canvasWidth, canvasHeight, parentRotation),
            boxMeasures       = getSnapBoxMeasuresForRotatedShape(pathCoords, parentRotation),
            boundingWidth     = boxMeasures.width,
            boundingHeight    = boxMeasures.height,

            sketchSheet       = returnCleanSketchSheet(boundingWidth, boundingWidth),
          //sketchSheet       = returnCleanSketchSheet(canvasWidth, canvasHeight),
            sketchCanvas      = sketchSheet.canvas,
            sketchContext     = sketchSheet.context,

            tileRectMeasures  = getTileRectMeasures(boundingWidth, boundingHeight, gradientValue),
          //fillPathMeasures  = getFillPathMeasures(boundingWidth, boundingHeight, gradientValue),

          //tileRectMeasures  = getTileRectMeasures(canvasWidth, canvasHeight, gradientValue),
          //fillPathMeasures  = getFillPathMeasures(canvasWidth, canvasHeight, gradientValue),

            halfWidth         = math_round(tileRectMeasures.width / 2),
            halfHeight        = math_round(tileRectMeasures.height / 2),

            vertical_x        = (tileRectMeasures.x1 + halfWidth),
            vertical_y1       = tileRectMeasures.y1,
            vertical_y2       = tileRectMeasures.y2,

            horizontal_x1     = tileRectMeasures.x1,
            horizontal_x2     = tileRectMeasures.x2,
            horizontal_y      = (tileRectMeasures.y1 + halfHeight),

            colorStopList     = parseColorStopsForFillPathRectangular(gradientValue.colorStops);

        // debugging
        //
        // document.body.appendChild(sketchCanvas);
        // sketchCanvas.style.position = 'absolute';

        renderGradientFillRectangular(
            sketchContext,                                                                                                    // context,
            tileRectMeasures,                                                                                                 // fillRect,
            createGradientFillLinear(sketchContext, colorStopList, vertical_x, vertical_y1, vertical_x, vertical_y2),         // gradientFillVertical,
            createGradientFillLinear(sketchContext, colorStopList, horizontal_x1, horizontal_y, horizontal_x2, horizontal_y)  // gradientFillHorizontal
        );

      //// Utils.info('+++ getGradientFillRectangularUnbound +++ [canvas.width, canvas.height] : ', canvasWidth, canvasHeight);
      //// Utils.info('+++ getGradientFillRectangularUnbound +++ [tileRectMeasures] : ', tileRectMeasures);
      //// Utils.info('+++ getGradientFillRectangularUnbound +++ [fillPathMeasures] : ', fillPathMeasures);
      //// Utils.info('+++ getGradientFillRectangularUnbound +++ [gradientFill] : ', gradientFill);

        return createPatternFillFromCanvas(sketchCanvas, context);
    }

    function getGradientFillRectangularBound(context, gradientValue) {
        // Utils.info('+++ getGradientFillRectangularBound +++ [gradientValue] : ', gradientValue);
        var
            canvas            = context.canvas,
            canvasWidth       = canvas.width,
            canvasHeight      = canvas.height,

            sketchSheet       = returnCleanSketchSheet(canvasWidth, canvasHeight),
            sketchCanvas      = sketchSheet.canvas,
            sketchContext     = sketchSheet.context,

            tileRectMeasures  = getTileRectMeasures(canvasWidth, canvasHeight, gradientValue),
          //fillPathMeasures  = getFillPathMeasures(canvasWidth, canvasHeight, gradientValue),

            halfWidth         = math_round(tileRectMeasures.width / 2),
            halfHeight        = math_round(tileRectMeasures.height / 2),

            vertical_x        = (tileRectMeasures.x1 + halfWidth),
            vertical_y1       = tileRectMeasures.y1,
            vertical_y2       = tileRectMeasures.y2,

            horizontal_x1     = tileRectMeasures.x1,
            horizontal_x2     = tileRectMeasures.x2,
            horizontal_y      = (tileRectMeasures.y1 + halfHeight),

            colorStopList     = parseColorStopsForFillPathRectangular(gradientValue.colorStops);

        // debugging
        //
        // document.body.appendChild(sketchCanvas);
        // sketchCanvas.style.position = 'absolute';

        renderGradientFillRectangular(
            sketchContext,                                                                                                    // context,
            tileRectMeasures,                                                                                                 // fillRect,
            createGradientFillLinear(sketchContext, colorStopList, vertical_x, vertical_y1, vertical_x, vertical_y2),         // gradientFillVertical,
            createGradientFillLinear(sketchContext, colorStopList, horizontal_x1, horizontal_y, horizontal_x2, horizontal_y)  // gradientFillHorizontal
        );

      //// Utils.info('+++ getGradientFillRectangularBound +++ [canvas.width, canvas.height] : ', canvasWidth, canvasHeight);
      //// Utils.info('+++ getGradientFillRectangularBound +++ [tileRectMeasures] : ', tileRectMeasures);
      //// Utils.info('+++ getGradientFillRectangularBound +++ [fillPathMeasures] : ', fillPathMeasures);
      //// Utils.info('+++ getGradientFillRectangularBound +++ [gradientFill] : ', gradientFill);

        return createPatternFillFromCanvas(sketchCanvas, context);
    }

    function getGradientFillRadialUnbound(context, gradientValue, parentRotation) {
        // Utils.info('+++ getGradientFillRadialUnbound +++ [gradientValue, parentRotation] : ', gradientValue, parentRotation);
        var
            canvas            = context.canvas,

          //tileRectMeasures  = getTileRectMeasures(canvas.width, canvas.height, gradientValue),
          //fillPathMeasures  = getFillPathMeasures(canvas.width, canvas.height, gradientValue),

            boxMeasures       = getBoundingBoxMeasuresForRotatedDrawingRect(canvas.width, canvas.height, parentRotation),
            boundingWidth     = boxMeasures.width,
            boundingHeight    = boxMeasures.height,

            tileRectMeasures  = getTileRectMeasures(boundingWidth, boundingHeight, gradientValue),
            fillPathMeasures  = getFillPathMeasures(boundingWidth, boundingHeight, gradientValue),

            path_x1           = fillPathMeasures.x1,
            path_y1           = fillPathMeasures.y1,
          //path_x2           = fillPathMeasures.x2,
          //path_y2           = fillPathMeasures.y2,

            cathetus_1a       = math_abs(path_x1 - tileRectMeasures.x1),
            cathetus_1b       = math_abs(path_y1 - tileRectMeasures.y1),
            radius_1          = math_round(Utils.radius(cathetus_1a, cathetus_1b)),

            colorStopList     = gradientValue.colorStops,
          //rotationValue     = (gradientValue.rotation % 360),

            gradientFill      = context.createRadialGradient(path_x1, path_y1, 0, path_x1, path_y1, radius_1);/*

        if ((path_x1 === path_x2) && (path_y1 === path_y2)) {

            gradientFill      = context.createRadialGradient(path_x1, path_y1, radius_1, path_x1, path_y1, 0);
        } else {
            var
                cathetus_2a   = math_abs(path_x2 - tileRectMeasures.x2),
                cathetus_2b   = math_abs(path_y2 - tileRectMeasures.y2),
                radius_2      = math_round(Utils.radius(cathetus_2a, cathetus_2b) / 2);

            gradientFill      = context.createRadialGradient(path_x1, path_y1, radius_1, path_x2, path_y2, radius_2);
        }*/

        colorStopList.forEach(function (colorStopValue) {
            gradientFill.addColorStop(colorStopValue.position, colorStopValue.color);
        });

      //// Utils.info('+++ getGradientFillRadialUnbound +++ [canvas.width, canvas.height] : ', canvas.width, canvas.height);
      //// Utils.info('+++ getGradientFillRadialUnbound +++ [tileRectMeasures] : ', tileRectMeasures);
      //// Utils.info('+++ getGradientFillRadialUnbound +++ [fillPathMeasures] : ', fillPathMeasures);
      //// Utils.info('+++ getGradientFillRadialUnbound +++ [gradientFill] : ', gradientFill);

        return gradientFill;
    }

    function getGradientFillRadialBound(context, gradientValue) {
        // Utils.info('+++ getGradientFillRadialBound +++ [gradientValue] : ', gradientValue);
        var
            canvas            = context.canvas,

            tileRectMeasures  = getTileRectMeasures(canvas.width, canvas.height, gradientValue),
            fillPathMeasures  = getFillPathMeasures(canvas.width, canvas.height, gradientValue),

            path_x1           = fillPathMeasures.x1,
            path_y1           = fillPathMeasures.y1,
          //path_x2           = fillPathMeasures.x2,
          //path_y2           = fillPathMeasures.y2,

            cathetus_1a       = math_abs(path_x1 - tileRectMeasures.x1),
            cathetus_1b       = math_abs(path_y1 - tileRectMeasures.y1),
            radius_1          = math_round(Utils.radius(cathetus_1a, cathetus_1b)),

            colorStopList     = gradientValue.colorStops,
          //rotationValue     = (gradientValue.rotation % 360),

            gradientFill      = context.createRadialGradient(path_x1, path_y1, 0, path_x1, path_y1, radius_1);/*

         if ((path_x1 === path_x2) && (path_y1 === path_y2)) {

            gradientFill      = context.createRadialGradient(path_x1, path_y1, radius_1, path_x1, path_y1, 0);
         } else {
            var
                cathetus_2a   = math_abs(path_x2 - tileRectMeasures.x2),
                cathetus_2b   = math_abs(path_y2 - tileRectMeasures.y2),
                radius_2      = math_round(Utils.radius(cathetus_2a, cathetus_2b) / 2);

            gradientFill      = context.createRadialGradient(path_x1, path_y1, radius_1, path_x2, path_y2, radius_2);
         }*/

        colorStopList.forEach(function (colorStopValue) {
            gradientFill.addColorStop(colorStopValue.position, colorStopValue.color);
        });

      //// Utils.info('+++ getGradientFillRadialBound +++ [canvas.width, canvas.height] : ', canvas.width, canvas.height);
      //// Utils.info('+++ getGradientFillRadialBound +++ [tileRectMeasures] : ', tileRectMeasures);
      //// Utils.info('+++ getGradientFillRadialBound +++ [fillPathMeasures] : ', fillPathMeasures);
      //// Utils.info('+++ getGradientFillRadialBound +++ [gradientFill] : ', gradientFill);

        return gradientFill;
    }

    /**
     *
     * @param context
     * @param gradientValue
     * @param parentRotation
     * @param pathCoords
     * @returns {CanvasPattern}
     */
    function getGradientFillLinearUnbound(context, gradientValue, parentRotation, pathCoords) {
        // Utils.info('+++ getGradientFillLinearUnbound +++ [gradientValue, parentRotation] : ', gradientValue, parentRotation, pathCoords);
        var
            gradientFill,

            boxMeasures     = getSnapBoxMeasuresForRotatedShape(pathCoords, parentRotation),
            boxWidth        = boxMeasures.width,
            boxHeight       = boxMeasures.height,

            colorStopList   = gradientValue.colorStops,
          //isScaled        = gradientValue.isScaled,
            rotation        = (gradientValue.rotation % 360);

        if (rotation > 0) {
            var
                lineCoords  = getLinearGradientCoordinates(boxWidth, boxHeight, rotation);

            gradientFill    = createGradientFillLinear(context, colorStopList, lineCoords.x1, lineCoords.y1, lineCoords.x2, lineCoords.y2);
        } else {
            gradientFill    = createGradientFillLinear(context, colorStopList, 0, 0, boxWidth, 0);
        }
        return renderRotatedPatternFromGradientFill(context, gradientFill, ((360 - parentRotation) % 360), boxMeasures);
    }

    /**
     *
     * @param context
     * @param gradientValue
     * @returns {CanvasGradient}
     */
    function getGradientFillLinearBound(context, gradientValue) {
      //// Utils.info('+++ getGradientFillLinearBound +++ [gradientValue] : ', gradientValue);
        var
            gradientFill,

            canvas          = context.canvas,
            canvasWidth     = canvas.width,
            canvasHeight    = canvas.height,

            colorStopList   = gradientValue.colorStops,
          //isScaled        = gradientValue.isScaled,
            rotation        = (gradientValue.rotation % 360);

        if (rotation > 0) {
            var
                lineCoords  = getLinearGradientCoordinates(canvasWidth, canvasHeight, rotation);

            gradientFill    = createGradientFillLinear(context, colorStopList, lineCoords.x1, lineCoords.y1, lineCoords.x2, lineCoords.y2);
        } else {
            gradientFill    = createGradientFillLinear(context, colorStopList, 0, 0, canvasWidth, 0);
        }
        return gradientFill;
    }

    /**
     *
     * @param context
     * @param gradientValue
     * @returns {CanvasGradient|CanvasPattern}
     */
    function getGradientFill(context, gradientValue, parentRotation, pathCoords) {
        var
            operation,
            identifier    = '',

            gradientType  = gradientValue.type,
            pathType      = gradientValue.pathType,
            isBound       = (!pathCoords || !!gradientValue.isRotateWithShape || (parentRotation === 0));

        if (gradientType === 'linear') {
            identifier    = [gradientType, LITERAL_BRIDGE__AND_BOUND, isBound].join('');
        } else if (pathType) {
            identifier    = [LITERAL_PREFIX__PATH_TYPE, pathType, LITERAL_BRIDGE__AND_BOUND, isBound].join('');
        }
        operation         = GRADIENT_FILLTYPE_OPERATIONS[identifier] || NOOP;

        return operation(context, gradientValue, parentRotation, pathCoords);
    }

    /**
     *
     * @param colorStop
     * @returns {*}
     */
    function parseCssColorStop(colorStop, docModel, target) {

        colorStop.color = docModel.getCssColor(colorStop.color, 'fill', target);

        return colorStop;
    }

    /**
     *
     * @param colorStopList
     * @returns {*}
     */
    function getMappedColorStopValues(colorStopList) {
        return colorStopList.map(function (colorStop) {

            return colorStop.valueOf();
        });
    }

    /**
     *
     * @param gradient
     * @param type
     * @returns {Boolean|boolean}
     */
    function isEqualGradients(gradient, type) {
        var
            gradientType  = gradient.getType(),
            isEqual       = (
                isBaseGradient(type) && ((type === gradient) || (

                (gradientType === type.getType()) &&

                (gradient.getRotation()       === type.getRotation()) &&
                (gradient.isRotateWithShape() === type.isRotateWithShape()) &&

                isEqualValue(getMappedColorStopValues(gradient.getColorStops()), getMappedColorStopValues(type.getColorStops()))
            )));

        if (isEqual) {
            isEqual = false;

            if ((gradientType === 'linear') && hasLinearTypeTrait(type)) {

                isEqual = (gradient.isScaled() === type.isScaled());

            } else if ((gradientType === 'path') && hasPathTypeTrait(type)) {
                var
                    pathType = gradient.getPathType();
                if (pathType === type.getPathType()) {

                    if (pathType === 'shape') {

                        isEqual = true;

                    } else if (((pathType === 'circle') || (pathType === 'rect')) && hasWithFillRectTrait(type)) {
                        isEqual = (
                            isEqualValue(gradient.getEdgeValues(),      type.getEdgeValues()) &&
                            isEqualValue(gradient.getFillTileValues(),  type.getFillTileValues())
                        );
                    }
                }
            }
        }
        return isEqual;
    }

    // BEGIN OF :: locally scoped helper module(s) / class(es) ----------------
    //

    // module ColorStop =======================================================

    /**
     *
     * @param position
     * @param color
     * @constructor
     */
    function ColorStop(position, color) {
        var // locally stored initial `this` reference.
            colorStop = this;

        colorStop.getPosition = function () {
            return position;
        };
      //colorStop.setPosition = function (value) {
      //    position = value;
      //};

        colorStop.getColor = function () {
            return color.clone();
        };
      //colorStop.setColor = function (colorDescriptor) {
      //    color = Color.parseJSON(colorDescriptor);
      //};

        /**
         * Returns the type-descriptor representation of this color-stop.
         *
         * @returns {Object}
         *  The type-descriptor representation of this color-stop.
         */
        colorStop.valueOf = function () {
            return {
                position:   position,
                color:      color.toJSON()
            };
        };

        /**
         * Returns the JSON-style stringified type-descriptor representation
         * of this color-stop.
         *
         * @returns {Object}
         *  The JSON-style stringified type-descriptor representation of this color-stop.
         */
        colorStop.toString = function () {
            return json_stringify(colorStop.valueOf());
        };

    } // constructor ColorStop

    /**
     *
     * @param typeDescriptor
     * @returns {ColorStop}
     */
    function createColorStop(typeDescriptor) {
        if (!isObjectObject(typeDescriptor)) {

            typeDescriptor = {};
        }
        var
            position  = Utils.getNumberOption(typeDescriptor, 'position', 0,    0, 1, 0.00001),
            color     = Color.parseJSON(
                Utils.getObjectOption(typeDescriptor,         'color',    {})
            );
      //global.console.log('+++ createColorStop :: [position, color] : ', position, color);

        return (new ColorStop(position, color));

    } // factory ColorStop

    var ColorStopModule = {
        create: createColorStop
    };
    // module ColorStop =======================================================

    //
    // END OF :: locally scoped helper module(s) / class(es) ------------------

    // constructor Gradient ===================================================
    /**
     * Runtime representation of a gradient of any type supported in document
     * operations.
     *
     * @param state
     * @param traitList
     * @constructor
     */
    function Gradient(state, colorStopList, traitList) {
        var // locally stored initial `this` reference.
            gradient = this;

        // public methods -----------------------------------------------------

        /**
         * Returns the type of this gradient.
         *
         * @returns {String}
         *  The type of this gradient. See constructor parameter 'type' for
         *  details.
         */
        gradient.getType = function () {
            return state.type;
        };
      //gradient.setType = function (value) {
      //    state.type = value;
      //};

        /**
         * Returns a gradient's color stop list.
         *
         * @returns {Array}
         *  A copy of this gradient's internal list of color stop types.
         */
        gradient.getColorStops = function () {
            return array_from(colorStopList);
        };

        /**
         * Returns the type-descriptor representation of this gradient,
         * as it gets used in document operations.
         *
         * @returns {Object}
         *  The type-descriptor representation of this gradient.
         */
        gradient.valueOf = function () {
            var
                gadientValue = object_extend({}, state);

            gadientValue.colorStops = getMappedColorStopValues(colorStopList);

            return gadientValue;
        };

        /**
         * Returns the JSON-style stringified type-descriptor representation
         * of this gradient, as it gets used in document operations.
         *
         * @returns {Object}
         *  The JSON-style stringified type-descriptor representation of this gradient.
         */
        gradient.toString = function () {
            return json_stringify(gradient.valueOf());
        };

        /**
         * Returns whether this gradient equals the passed gradient type.
         *
         * @param {Gradient} gradient
         *  The gradient type to be compared with this gradient.
         *
         * @returns {Boolean}
         *  Whether this gradient is equal to the passed gradient type.
         */
        gradient.equals = function (type) {
            return isEqualGradients(gradient, type);
        };

        /**
         * Returns a deep clone of this gradient.
         *
         * @returns {Gradient}
         *  A deep clone of this gradient.
         */
        gradient.clone = function () {
            return (new Gradient(object_extend({}, state), array_from(colorStopList), array_from(traitList)));
        };

        applyGradientSpecificTraits(traitList, gradient, state);

    } // constructor Gradient

    // trait based gradient composition =======================================
    function applyGradientSpecificTraits(traitList, gradient, state) {
        traitList.forEach(function (trait/*, idx, list*/) {

            trait.call(gradient, state);
        });
    } // trait based gradient composition

    // trait WithScalingTrait =================================================
    function WithScalingTrait(state) {
        var
            type = this;

        type.isScaled = function () {
            return state.isScaled;
        };
    } // trait WithScalingTrait

    // trait WithRotationTrait ================================================
    function WithRotationTrait(state) {
        var
            type = this;

        type.getRotation = function () {
            return state.rotation;
        };
      //type.setRotation = function (value) {
      //    state.rotation = value;
      //};

        type.isRotateWithShape = function () {
            return state.isRotateWithShape;
        };
    } // trait WithRotationTrait

    // trait WithFillRectTrait ================================================
    function WithFillRectTrait(state) {
        var
            type = this;

        type.getEdgeValues = function () {
            return {
                top:    state.top,
                right:  state.right,
                bottom: state.bottom,
                left:   state.left
            };
        };
        type.getFillTileValues = function () {
            return {
                top:    state.pathFillTop,
                right:  state.pathFillRight,
                bottom: state.pathFillBottom,
                left:   state.pathFillLeft
            };
        };
    } // trait WithFillRectTrait

    // trait PathTypeTrait ====================================================
    function PathTypeTrait(state) {
        var
            type = this;

        type.getPathType = function () {
            return state.pathType;
        };
      //WithRotationTrait.call(type, state);

    } // trait PathTypeTrait

    // trait PathTypeShapeTrait ===============================================
    function PathTypeShapeTrait(state) {
        var
            type = this;

        PathTypeTrait.call(type, state);

    } // trait PathTypeShapeTrait

    // trait PathTypeWithFillRectTrait =========================================
    function PathTypeWithFillRectTrait(state) {
        var
            type = this;

        PathTypeTrait.call(type, state);
        WithFillRectTrait.call(type, state);

    } // trait PathTypeWithFillRectTrait

    // trait LinearTypeTrait ==================================================
    function LinearTypeTrait(state) {
        var
            type = this;

        WithScalingTrait.call(type, state);
      //WithRotationTrait.call(type, state);

    } // trait LinearTypeTrait

    // public static factory method ===========================================

    /**
     * Evaluates the passed type-descriptor representation of a gradient and
     * returns a new instance of the class `Gradient` or returns the `NULL` value
     * in case of `typeDescriptor` not being an `Object` type.
     *
     * @param {Object} typeDescriptor
     *  The type-descriptor to be evaluated. Should be an object with the
     *  properties supported in document operations.
     *
     * @returns {Gradient|null}
     *  A new gradient instance or the `NULL` value.
     */
    function createGradient(typeDescriptor) {
        var
            gradient        = null,

            state           = {},
            traitList       = [],

            colorStopList,
            createColorStop = ColorStopModule.create;

        if (isObjectObject(typeDescriptor)) {

            colorStopList = Utils.getArrayOption(typeDescriptor, 'colorStops', null, true);

            if (colorStopList) {
                colorStopList = colorStopList.map(createColorStop);
            } else {
                /**
                 *  please intercept here in case of omitted "color"/"color2" properties
                 *  in order to retrieve feasible (e.g. theme depended) default/fallback values
                 *  before pushing them into this "colorStopList".
                 */
                colorStopList   = [
                    createColorStop({ position: 0, color: typeDescriptor.color }),
                    createColorStop({ position: 1, color: typeDescriptor.color2 })
                ];
            }
          //global.console.log('+++ createGradient :: colorStopList : ', colorStopList);

            var
                // either 'linear' or 'path'.   // Module.LINEAR.type -  MINIMAL_GRADIENT_DESCRIPTOR.type
                gradientType        = Utils.getStringOption(typeDescriptor,   'type',               'linear',   true),
                pathType;

            state.type              = gradientType;

            state.rotation          = Utils.getNumberOption(typeDescriptor,   'rotation',           0,          0,      360,    0.00001);
            state.isRotateWithShape = Utils.getBooleanOption(typeDescriptor,  'isRotateWithShape',  false);

        //  state.flipH             = Utils.getBooleanOption(typeDescriptor,  'flipH',              false);
        //  state.flipV             = Utils.getBooleanOption(typeDescriptor,  'flipV',              false);
        //  // 'none' or 'x' or 'y' or 'xy'.
        //  //state.flip            = Utils.getStringOption(typeDescriptor,   'flip',               'none',     true);

            if (gradientType === 'linear') {

                // "linear" type exclusively:
                state.isScaled      = Utils.getBooleanOption(typeDescriptor,  'isScaled',           false);

                traitList.push(LinearTypeTrait);

            } else if (gradientType === 'path') {

                // "path" type exclusively:
                // 'circle' or 'rect' or 'shape'.
                pathType            = Utils.getStringOption(typeDescriptor,   'pathType',           'rect',     true);

                state.pathType      = pathType;

                if (pathType === 'shape') {
                    traitList.push(PathTypeShapeTrait);

                } else {
                    // (pathType === 'circle') || (pathType === 'rect')

                    state.left           = Utils.getNumberOption(typeDescriptor, 'left',            0,          -1,     1);
                    state.top            = Utils.getNumberOption(typeDescriptor, 'top',             0,          -1,     1);
                    state.right          = Utils.getNumberOption(typeDescriptor, 'right',           0,          -1,     1);
                    state.bottom         = Utils.getNumberOption(typeDescriptor, 'bottom',          0,          -1,     1);

                    state.pathFillLeft   = Utils.getNumberOption(typeDescriptor, 'pathFillLeft',    0.5,        -1,     1); // - new "pathFill" default values ...
                    state.pathFillTop    = Utils.getNumberOption(typeDescriptor, 'pathFillTop',     0.5,        -1,     1); //
                    state.pathFillRight  = Utils.getNumberOption(typeDescriptor, 'pathFillRight',   0.5,        -1,     1); //   see: Bug 51597 - ODP: gradient background are not displayed
                    state.pathFillBottom = Utils.getNumberOption(typeDescriptor, 'pathFillBottom',  0.5,        -1,     1); //   see: Gradient Data TYpe - https://intranet.open-xchange.com/wiki/documents-team:operations:general#gradient

                    // // NOTE:
                    // // can be omitted as soon as Sven changes the total of top/left/right/bottom values to the total of pathFill(Top/Left/Right/Bottom)
                    // //
                    // if (('left' in typeDescriptor) && !('pathFillRight' in typeDescriptor)) {
                    //
                    //     state.pathFillRight = (-1 * state.left) || 0;
                    //   //state.pathFillLeft = 0.5;
                    //
                    // } else if (('right' in typeDescriptor) && !('pathFillLeft' in typeDescriptor)) {
                    //
                    //     state.pathFillLeft = (-1 * state.right) || 0;
                    //   //state.pathFillRight = 0.5;
                    // }
                    // if (('top' in typeDescriptor) && !('pathFillBottom' in typeDescriptor)) {
                    //
                    //     state.pathFillBottom = (-1 * state.top) || 0;
                    //   //state.pathFillTop = 0.5;
                    //
                    // } else if (('bottom' in typeDescriptor) && !('pathFillTop' in typeDescriptor)) {
                    //
                    //     state.pathFillTop = (-1 * state.bottom) || 0;
                    //   //state.pathFillBottom = 0.5;
                    // }

                    traitList.push(PathTypeWithFillRectTrait);
                }
            }
            traitList.unshift(WithRotationTrait); // make rotation trait a generic behavior of any gradient type

            gradient = (new Gradient(state, colorStopList, traitList));
        }
        //if (gradient) { gradientList.push(gradient); } // @NOTIFICATION: for debugging only.

        return gradient;
    }

    // private static (locally scoped) trait and type detection helpers =======

    function hasWithRotationTrait(type) {
        return (
            isFunction(type.getRotation) &&
            isFunction(type.isRotateWithShape)
        );
    }
    function hasWithScalingTrait(type) {
        return isFunction(type.isScaled);
    }
    function hasWithFillRectTrait(type) {
        return (
            isFunction(type.getEdgeValues) &&
            isFunction(type.getFillTileValues)
        );
    }

    function hasGenericTraits(type) {
        return (
            isFunction(type.equals) &&
            isFunction(type.clone) &&

            isFunction(type.getType) &&
            isFunction(type.getColorStops) &&

            hasWithRotationTrait(type)
        );
    }

    function hasLinearTypeTrait(type) {
        return hasWithScalingTrait(type);
    }
    function hasPathTypeTrait(type) {
        return isFunction(type.getPathType);
    }

    // function hasPathTypeWithFillRectTrait(type) {
    //     return (
    //         hasPathTypeTrait(type) &&
    //
    //         hasWithFillRectTrait(type)
    //     );
    // }
    // function hasPathTypeShapeTrait(type) {
    //     return hasPathTypeTrait(type);
    // }

    function isBaseGradient(type) {
        return (isObjectObject(type) && ((type instanceof Gradient) || hasGenericTraits(type)));  // - shortcut variant.
      //return (isObjectObject(type) && hasGenericTraits(type));                                  // - spoof pr variant.
    }

    // public static utility methods ==========================================

    /**
     * Returns whether the passed type does match all criteria
     * for being a valid `Gradient` type or not.
     *
     * @param {any} type
     * The type that is going to be checked for being a valid `Gradient` type.
     *
     * @returns {Boolean}
     *  Whether the passed type does match all criteria for being a valid `Gradient` type.
     */
    function isGradient(type) {
        var
            isValidType   = isBaseGradient(type),
            gradientType  = (isValidType && type.getType()),
            pathType;

        if (isValidType) {
            isValidType   = false;

            if (gradientType === 'linear') {

                isValidType = hasLinearTypeTrait(type);

            } else if (gradientType === 'path') {

                isValidType = hasPathTypeTrait(type);
                pathType    = (isValidType && type.getPathType());

                isValidType = (isValidType && (

                    ((pathType === 'shape') && true) ||                     // - generic approach is feasible enough yet.
                  //((pathType === 'shape') && hasPathTypeShapeTrait()) ||  // - use that, ones it has been specified more precisely.

                    (((pathType === 'circle') || (pathType === 'rect')) && hasWithFillRectTrait(type))
                ));
            }
        }
        return isValidType;
    }

    /**
     *
     * @param type
     * @returns {boolean}
     */
    function isHTML5Canvas(type) {
        return (
            !!type && (

            regXCanvasElement.test(expose_internal_class_name.call(type)) ||
            (isFunction(type.getContext) && isFunction(type.toDataURL))
        ));
    }

    /**
     *
     * @param type
     * @returns {boolean}
     */
    function isCanvasGradient(type) {
        return (
            !!type && (

            (/^\[object\s+CanvasGradient]$/).test(expose_internal_class_name.call(type)) ||
            isFunction(type.addColorStop)
        ));
      //return (!!type && isFunction(type.addColorStop));
    }

    /**
     *
     * @param type
     * @returns {boolean}
     */
    function isCanvasPattern(type) {
        return (
            !!type && (

            (/^\[object\s+CanvasPattern]$/).test(expose_internal_class_name.call(type)) ||
            isFunction(type.setTransform)
        ));
    }

    /**
     * Returns whether the type-descriptor representations of two gradients are equal or not.
     *
     * @param {Object} typeDescriptorA
     * The type-descriptor representation of the first gradient to be compared
     * to the other gradient.
     *
     * @param {Object} typeDescriptorB
     * The type-descriptor representation of the second gradient to be compared
     * to the other gradient.
     *
     * @returns {Boolean}
     *  Whether the type-descriptor representations of two gradients are equal.
     */
    function isEqualDescriptors(typeDescriptorA, typeDescriptorB) {
        return createGradient(typeDescriptorA).equals(createGradient(typeDescriptorB));
    }

    /**
     *
     * @param fillAttrs
     * @returns {*}
     */
    function getDescriptorFromFillAttributes(fillAttrs) {
        var
            gradientDescriptor = object_extend({}, fillAttrs.gradient);  // ensures always an object.

        // enable the descriptor to cover a minimal gradient (color transition) render fallback.
        gradientDescriptor.color  = fillAttrs.color;
        gradientDescriptor.color2 = fillAttrs.color2;

        return gradientDescriptor;
    }

    /**
     *
     * @param gradient
     * @param docModel
     * @param target
     * @returns {*}
     */
    function parseCssColors(gradient, docModel, target) {
        var
            gradientValue = gradient.valueOf();

        gradientValue.colorStops.forEach(function (colorStop) {
            parseCssColorStop(colorStop, docModel, target);
        });

        return gradientValue;
    }

    /**
     *
     * @param docModel
     * @param gradient
     * @param pathCoords
     * @param context
     * @param target
     * @returns {*}
     */
    function parseFillStyle(docModel, gradient, context, target, pathCoords) {
      //// Utils.info('+++ Gradient.parseFillStyle +++ pathCoords : ', pathCoords);
        var                                                                           //  - refactor `CanvasWrapper` making it aware of a parent `Shape` abstraction
            $drawingNode    = $(context.canvas).closest('.drawing.rotated-drawing'),  //  - ones a canva wrapper type can acces it parent shape type, retrieving
            drawingAttrs    = AttributeUtils.getExplicitAttributes($drawingNode),     //    values like e.g. `parentRotation` will be much easier and far cleaner.

            gradientValue   = parseCssColors(gradient, docModel, target),
            gradientFill,

            parentRotation  = (drawingAttrs && drawingAttrs.drawing && drawingAttrs.drawing.rotation) || 0;
          //parentRotation  = DrawingFrame.getDrawingRotationAngle(docModel, $drawingNode) || 0;

        if ((context.canvas.width <= 0) || (context.canvas.height <= 0)) {
            gradientFill = 'transparent';
        } else {
            gradientFill = getGradientFill(context, gradientValue, parentRotation, pathCoords);
        }
        return gradientFill;
    }

    function parseImageDataUrl(gradient, docModel, canvasOrDescriptor, target) {
        var
            dataUrl,

            canvas,
            context;

        if (isObjectObject(canvasOrDescriptor)) {
            var
                sketchSheet = createCleanSketchSheetClone(canvasOrDescriptor.width, canvasOrDescriptor.height);

            canvas  = sketchSheet.canvas;
            context = sketchSheet.context;

        } else if (isHTML5Canvas(canvasOrDescriptor)) {

            canvas  = canvasOrDescriptor;
            context = canvas.getContext('2d');
        }
        if (context) {

            context.fillStyle = parseFillStyle(docModel, gradient, context, target/*, null*/);  // @NOTE - Do never ever alter
            context.fillRect(0, 0, canvas.width, canvas.height);                                // precedence of this two lines.

            dataUrl = canvas.toDataURL();
        }

        return dataUrl;
      //return (dataUrl || '');
    }

    function parseImageElement(gradient, docModel, canvasOrDescriptor, target) {
        var
            elmImage  = IMAGE_BLUEPRINT.cloneNode(),

            dataUrl   = parseImageDataUrl(gradient, docModel, canvasOrDescriptor, target);
        if (dataUrl) {

            elmImage.width  = canvasOrDescriptor.width;
            elmImage.height = canvasOrDescriptor.height;

            elmImage.src    = dataUrl;
        }
        return elmImage;
    }

    // module Gradient --------------------------------------------------------

    var Module = {      //  assign locally scoped static method implementations.

      //all:                function () { return array_from(gradientList); },       // @NOTIFICATION: for debugging only.
        create:             createGradient,

        parseCssColors:     parseCssColors,
        parseFillStyle:     parseFillStyle,

        parseImageDataUrl:  parseImageDataUrl,
        parseImageElement:  parseImageElement,

        isCanvasPattern:    isCanvasPattern,
        isCanvasGradient:   isCanvasGradient,

        isGradient:         isGradient,
        isEqualDescriptors: isEqualDescriptors,

        getDescriptorFromAttributes: getDescriptorFromFillAttributes
    };

    // constants --------------------------------------------------------------

    /**
     * The config object according to the default gradient type.
     *
     * @constant
     */
    Module.LINEAR = { type: 'linear' }; // MINIMAL_GRADIENT_DESCRIPTOR

    // exports ================================================================

    return Module;

});
