/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/stylecollection', [
    'io.ox/office/tk/config',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/editframework/utils/operations',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/model/operationcontext',
    'gettext!io.ox/office/editframework/main'
], function (Config, Utils, ValueMap, ModelObject, Operations, AttributeUtils, OperationContext, gt) {

    'use strict';

    // class StyleCollection ==================================================

    /**
     * Collection for hierarchical style sheets of a specific attribute family.
     *
     * Instances will trigger the following events:
     *  - 'insert:stylesheet': After a new style sheet has been inserted into
     *      this collection. The event handler receives the identifier of the
     *      new style sheet.
     *  - 'delete:stylesheet': After a style sheet has been deleted from this
     *      collection. The event handler receives the identifier of the
     *      deleted style sheet.
     *  - 'change:stylesheet': After the formatting attributes, or other
     *      properties of a style sheet in this collection have been changed.
     *      The event handler receives the identifier of the changed style
     *      sheet.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {EditModel} docModel
     *  The document model containing this instance.
     *
     * @param {String} styleFamily
     *  The main attribute family represented by the style sheets contained by
     *  instances of the derived collection class. The style sheets and the DOM
     *  elements referring to the style sheets must support all attributes of
     *  this attribute family.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. All callback functions will be called in the
     *  context of this style sheet collection.
     *  @param {Boolean} [initOptions.styleSheetSupport=true]
     *      If set to false, this collection is not allowed to contain style
     *      sheets. It can only be used to format DOM elements with explicit
     *      formatting attributes. DEPRECATED!
     *  @param {String} [initOptions.poolId]
     *      The identifier of an attribute pool to be used. If omitted, the
     *      default attribute pool of the document will be used.
     *  @param {String} [initOptions.families]
     *      The names of additional attribute families supported by the style
     *      sheets of this collection (space-separated). It is not needed to
     *      repeat the name of the style family (as passed to the 'styleFamily'
     *      parameter) here.
     *  @param {Object<String,Function>} [initOptions.parentResolvers]
     *      The parent style families whose associated style sheets can contain
     *      attributes of the family supported by this style sheet collection.
     *      The DOM elements referring to the style sheets of the specified
     *      style families must be ancestors of the DOM elements referring to
     *      the style sheets of this container. The passed object maps the
     *      attribute family to an ancestor element resolver function. The
     *      function receives the descendant DOM element as jQuery object in
     *      the first parameter, and returns the ancestor element of that DOM
     *      element which is associated to the parent style family used as map
     *      key.
     *  @param {Function} [initOptions.baseAttributesResolver]
     *      A function that can return additional base attributes for the
     *      specified DOM element. The callback function receives the following
     *      parameters:
     *      (1) {jQuery} element
     *          The DOM element to resolve the base attributes for.
     *      (2) {Object} baseAttributes
     *          The base attribute set already collected for the specified DOM
     *          element (e.g. from its parent elements). MUST NOT be changed by
     *          the callback function!
     *      (3) {Object} [newAttributes]
     *          An optional object containing the new attributes that will be
     *          assigned to the specified DOM element but are not yet assigned.
     *          This is especially important for resolving the attributes from
     *          layout or master slide, if the paragraph level has changed.
     *  @param {Function} [initOptions.styleAttributesResolver]
     *      A function that extracts the effective explicit attribute set from
     *      the original attribute definition of a style sheet. The attributes
     *      returned by this function may depend on a source element (e.g. on
     *      the position of this source element in its parents). The callback
     *      function receives the following parameters:
     *      (1) {Object} styleAttributes
     *          The original attribute definition of the style sheet. MUST NOT
     *          be changed by the callback function!
     *      (2) {jQuery} element
     *          The element referring to the style sheet.
     *      (3) {jQuery} [sourceNode]
     *          The descendant source node.
     *  @param {Function} [initOptions.elementAttributesResolver]
     *      A function that extracts and returns specific attributes from the
     *      explicit attribute set of an element. The attributes returned by
     *      this function may depend on a source element (e.g. on the position
     *      of this source element in its parents). The callback function
     *      receives the following parameters:
     *      (1) {Object} elementAttributes
     *          The explicit attribute set of the element. MUST NOT be changed
     *          by the callback function!
     *      (2) {jQuery} element
     *          The element whose explicit attributes have been passed in the
     *          first parameter.
     *      (3) {jQuery} [sourceNode]
     *          The descendant source node.
     */
    function StyleCollection(docModel, styleFamily, initOptions) {

        // self reference
        var self = this;

        // all registered style sheets, mapped by unique identifier
        var styleSheetMap = new ValueMap();

        // the attribute pool to be used for the formatting attributes
        var attributePool = docModel.getAttributePool(Utils.getStringOption(initOptions, 'poolId', null));

        // whether this collection represents the default style family (no "type" property in operations)
        var isDefaultFamily = styleFamily === docModel.getDefaultStyleFamily();

        // the names of all supported attribute families, as flag set
        var supportedFamilySet = null;

        // identifier of the default style sheet
        var defaultStyleId = null;

        // whether the default style shall be used, if no style is specified at an element
        var useDefaultStyleIfMissing = true;

        // whether this collection supports style sheets
        var styleSheetSupport = Utils.getBooleanOption(initOptions, 'styleSheetSupport', true);

        // the resolver functions for DOM ancestor elements, keyed by attribute family
        var parentResolvers = Utils.getObjectOption(initOptions, 'parentResolvers', {});

        // custom resolver for base attributes
        var baseAttributesResolver = Utils.getFunctionOption(initOptions, 'baseAttributesResolver', null);

        // custom resolver for style attributes depending on a context element
        var styleAttributesResolver = Utils.getFunctionOption(initOptions, 'styleAttributesResolver', null);

        // custom resolver for explicit element attributes depending on a context element
        var elementAttributesResolver = Utils.getFunctionOption(initOptions, 'elementAttributesResolver', null);

        // all registered DOM formatting handlers
        var domFormatHandlers = [];

        // all registered DOM preview formatting handlers
        var domPreviewHandlers = [];

        // the default name for unnamed ODF table styles
        var DEFAULT_TABLE_STYLE_NAME = '[' + gt('Unnamed Style') + ']';

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel, { trigger: 'always' });

        // private methods ----------------------------------------------------

        /**
         * Returns whether a style sheet with the specified identifier exists
         * in this collection.
         *
         * @param {String|Null} styleId
         *  The identifier of a style sheet.
         *
         * @returns {Boolean}
         *  Whether a style sheet with the specified identifier exists in this
         *  collection.
         */
        function hasStyleSheet(styleId) {
            return !!styleId && styleSheetMap.has(styleId);
        }

        /**
         * Returns the style sheet with the specified identifier.
         *
         * @param {String|Null} styleId
         *  The identifier of a style sheet.
         *
         * @returns {Object|Null}
         *  The style sheet model with the specified identifier.
         */
        function getStyleSheet(styleId) {
            return styleId ? styleSheetMap.get(styleId, null) : null;
        }

        /**
         * Invokes a callback function, if the style sheet with the specified
         * identifier exists.
         *
         * @param {String|Null} styleId
         *  The identifier of a style sheet.
         *
         * @param {Function} callback
         *  The callback function that will be invoked if the specified style
         *  sheet exists. Receives the style sheet mdoel as first parameter.
         */
        function withStyleSheet(styleId, callback) {
            if (styleId) { styleSheetMap.with(styleId, callback, self); }
        }

        /**
         * Returns the identifier of the style sheet that will be effectively
         * used for the specified style sheet identifier. If the specified
         * style sheet does not exist, but this collection contains a default
         * style sheet, its unique identifier will be returned instead.
         * Otherwise, the passed style sheet identifier will be returned.
         *
         * @param {String} styleId
         *  The unique identifier of a style sheet.
         *
         * @returns {String}
         *  The passed style sheet identifier, if such a style sheet exists in
         *  this collection, otherwise the identifier of the default style
         *  sheet.
         */
        function getEffectiveStyleId(styleId) {
            return (!hasStyleSheet(styleId) && hasStyleSheet(defaultStyleId)) ? defaultStyleId : styleId;
        }

        /**
         * Returns whether the passed style sheet is a descendant of the other
         * passed style sheet.
         */
        function isDescendantStyleSheet(styleSheet, ancestorStyleSheet) {
            while (styleSheet) {
                if (styleSheet === ancestorStyleSheet) { return true; }
                styleSheet = getStyleSheet(styleSheet.parentId);
            }
            return false;
        }

        /**
         * Delete all cached merged attribute sets in all style sheets.
         */
        function deleteCachedAttributeSets() {
            styleSheetMap.forEach(function (styleSheet) {
                delete styleSheet.sparseAttributes;
                delete styleSheet.mergedAttributes;
            });
        }

        /**
         * Returns the complete attributes of an ancestor of the passed
         * element, if a parent style family has been registered and its
         * element resolver function returns a valid ancestor element, and
         * extends missing attributes by their current default values.
         *
         * @param {HTMLElement|jQuery} [element]
         *  An element whose ancestor will be searched and queried for its
         *  attributes. If this object is a jQuery collection, uses the first
         *  DOM node it contains. If missing, returns just the current default
         *  values of all supported attribute families.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.noDefaults=false]
         *      If set to true, the default attributes will not be included
         *      into the generated base attribute set. This is especially useful,
         *      when operations shall be generated with all available attributes,
         *      for example after copy/paste of template drawings. In this case
         *      it is better to omit those attributes, that are set by default.
         *
         * @returns {Object}
         *  The formatting attributes of an ancestor of the passed element, as
         *  map of attribute value maps (name/value pairs), keyed by attribute
         *  family. If no ancestor element has been found, returns the current
         *  default values for all supported attribute families.
         */
        function resolveBaseAttributeSet(element, options) {

            // passed element, as jQuery object
            var $element = $(element);
            // the resulting merged attributes of the ancestor element
            var baseAttributeSet = {};
            // whether the default attributes shall be part of the base attribute set
            var noDefaults = Utils.getBooleanOption(options, 'noDefaults', false);

            // collect attributes from ancestor element if specified (only one parent style family must match at a time)
            if ($element.length > 0) {

                // find a matching ancestor element and its style family (only one parent must match at a time)
                var parentElement = null;
                var parentFamily = null;
                _.some(parentResolvers, function (elementResolver, family) {
                    parentElement = $(elementResolver.call(self, $element));
                    parentFamily = family;
                    return parentElement.length > 0;
                });

                // add the element attributes of the ancestor element (only the supported attribute families)
                if (parentElement && (parentElement.length > 0)) {
                    var parentStyleCollection = docModel.getStyleCollection(parentFamily);
                    var parentAttributes = parentStyleCollection.getElementAttributes(parentElement, { special: true, sourceNode: $element, noDefaults: noDefaults });
                    attributePool.extendAttributeSet(baseAttributeSet, parentAttributes, { special: true });
                }
            }

            // add missing default attribute values of supported families not found in the parent element
            _.each(supportedFamilySet, function (flag, family) {
                if (!(family in baseAttributeSet)) {
                    baseAttributeSet[family] = noDefaults ? null : attributePool.getDefaultValues(family);
                }
            });

            // remove attribute maps of unsupported families (this also removes the style identifier of the parent element)
            self.filterBySupportedFamilies(baseAttributeSet);

            return baseAttributeSet;
        }

        /**
         * Invokes the custom resolver callback for additional base attributes.
         *
         * @param {jQuery|HTMLElement} element
         *  The affected DOM element. Will be passed to the custom attributes
         *  resolver (see the option 'baseAttributesResolver' passed to the
         *  constructor). If this object is a jQuery collection, uses the first
         *  DOM node it contains.
         *
         * @param {Object} baseAttributes
         *  An attribute set with the base attributes already collected for the
         *  passed DOM element.
         *
         * @param {Object} [newAttributes]
         *  An optional object containing the new attributes that will be assigned
         *  to the DOM element but are not yet assigned. This is especially important
         *  for resolving the attributes from layout or master slide, if the paragraph
         *  level has changed (resolveLayoutAttributes is called before the new
         *  attributes are assigned to the paragraph).
         */
        function resolveCustomBaseAttributeSet(element, baseAttributes, newAttributes) {
            return baseAttributesResolver ? baseAttributesResolver.call(self, $(element), baseAttributes, newAttributes) : null;
        }

        /**
         * Returns the attributes from the specified style sheet and its parent
         * style sheets. Does not add default attribute values, or attributes
         * from ancestors of the passed element.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @param {HTMLElement|jQuery} [element]
         *  An element referring to a style sheet in this collection whose
         *  attributes will be extracted. Will be passed to a custom style
         *  attributes resolver (see the option 'styleAttributesResolver'
         *  passed to the constructor). If this object is a jQuery collection,
         *  uses the first DOM node it contains.
         *
         * @param {HTMLElement|jQuery} [sourceNode]
         *  The source DOM node corresponding to a child attribute family that
         *  has initiated the call to this method. Will be passed to a custom
         *  style attributes resolver (see the option 'styleAttributesResolver'
         *  passed to the constructor). If this object is a jQuery collection,
         *  uses the first DOM node it contains.
         *
         * @returns {Object}
         *  The formatting attributes contained in the style sheet and its
         *  ancestor style sheets, as map of attribute value maps (name/value
         *  pairs), keyed by attribute family.
         */
        function resolveStyleAttributeSet(styleId, element, sourceNode) {

            // passed element, as jQuery object
            var $element = $(element);
            // passed source node, as jQuery object
            var $sourceNode = $(sourceNode);
            // the resulting merged attributes of the style sheet and its ancestors
            var styleAttributeSet = {};

            // collects style sheet attributes recursively through parents
            function collectStyleAttributes(styleSheet) {

                // call recursively to get the attributes of the parent style sheets
                withStyleSheet(styleSheet.parentId, collectStyleAttributes);

                // add all attributes of the current style sheet, mapped directly by attribute family
                attributePool.extendAttributeSet(styleAttributeSet, styleSheet.attributes);

                // try user-defined resolver for style attributes mapped in non-standard structures
                if (styleAttributesResolver && ($element.length > 0)) {
                    attributePool.extendAttributeSet(styleAttributeSet, styleAttributesResolver.call(self, styleSheet.attributes, $element, $sourceNode));
                }
            }

            // returning empty object, if styleId not specified and fallback to default style not allowed (55547)
            if (!styleId && !useDefaultStyleIfMissing) { return styleAttributeSet; }

            // fall-back to default style sheet if passed identifier is invalid
            styleId = getEffectiveStyleId(styleId);

            // collect attributes from the style sheet and its parents
            withStyleSheet(styleId, collectStyleAttributes);

            // remove attribute maps of unsupported families (and the style identifier)
            self.filterBySupportedFamilies(styleAttributeSet);

            // restore identifier of the style sheet
            styleAttributeSet.styleId = styleId;

            return styleAttributeSet;
        }

        /**
         * Returns the explicit attributes from the specified element. Does not
         * add default attribute values. Uses a custom element attributes
         * resolver function if specified in the constructor of this style
         * sheet collection.
         *
         * @param {HTMLElement|jQuery} element
         *  The element whose explicit attributes will be extracted. Will be
         *  passed to a custom attributes resolver (see the option
         *  'elementAttributesResolver' passed to the constructor). If this
         *  object is a jQuery collection, uses the first DOM node it contains.
         *
         * @param {HTMLElement|jQuery} [sourceNode]
         *  The source DOM node corresponding to a child attribute family that
         *  has initiated the call to this method. Will be passed to a custom
         *  element attributes resolver (see the option
         *  'elementAttributesResolver' passed to the constructor). If this
         *  object is a jQuery collection, uses the first DOM node it contains.
         *
         * @returns {Object}
         *  The explicit formatting attributes contained in the element, as map
         *  of attribute value maps (name/value pairs), keyed by attribute
         *  family.
         */
        function resolveElementAttributeSet(element, sourceNode) {

            // the explicit attributes of the element
            var explicitAttributes = AttributeUtils.getExplicitAttributes(element, { direct: true });

            // call custom element attribute resolver
            if (elementAttributesResolver) {
                explicitAttributes = elementAttributesResolver.call(self, explicitAttributes, $(element), $(sourceNode));
            }

            return explicitAttributes;
        }

        /**
         * In ODF format there is not always a style name specified. In this case the filter
         * sends the ID as style sheet name. The user should not see the ID in the dialogs.
         * Therefore a default table style name needs to be used (56231).
         *
         * @param {Object} styleSheet
         *  The object representing a style sheet. At least the properties 'name' and 'id'
         *  are required.
         */
        function checkStyleSheetName(styleSheet) {
            if (styleSheet.name && styleSheet.id && styleSheet.name === styleSheet.id && Utils.isUUID(styleSheet.name)) {
                styleSheet.name = DEFAULT_TABLE_STYLE_NAME;
            }
        }

        function createOperationProperties(styleId, properties) {
            properties = _.extend({ styleId: styleId }, properties);
            if (!isDefaultFamily) { properties.type = styleFamily; }
            return properties;
        }

        // protected methods --------------------------------------------------

        /**
         * Registers a callback function that will be called for every DOM
         * element whose attributes have been changed, and that will format the
         * DOM element according to the new attribute values.
         *
         * @param {Function} formatHandler
         *  The callback function that implements formatting of a DOM element.
         *  Receives the following parameters:
         *  (1) {jQuery} element
         *      The element whose attributes have been changed.
         *  (2) {Object} mergedAttributes
         *      The complete attribute set merged from style sheet attributes,
         *      and explicit attributes.
         *  (3) {Boolean} async
         *      If set to true, the format handler may execute asynchronously
         *      in a browser timeout loop. In this case, the handler MUST
         *      return a promise that will be resolved after execution.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.duringImport=false]
         *      If set to true, the format handler will also be called when the
         *      attributes of a DOM element have been changed while importing
         *      the document. By default, registered formatting handlers will
         *      not be called during import.
         *
         * @returns {StyleCollection}
         *  A reference to this instance.
         */
        this.registerFormatHandler = function (formatHandler, options) {
            domFormatHandlers.push({
                callback: formatHandler,
                import: Utils.getBooleanOption(options, 'duringImport', false)
            });
            return this;
        };

        /**
         * Registers a callback function that will be called if a DOM element
         * used in the GUI to show a preview of a style sheet needs to be
         * formatted.
         *
         * @param {Function} previewHandler
         *  The callback function that implements preview formatting of a DOM
         *  element. Receives the following parameters:
         *  (1) {jQuery} element
         *      The element whose attributes have been changed.
         *  (2) {Object} mergedAttributes
         *      The complete attribute set merged from style sheets attributes,
         *      and explicit attribute.
         *  The function MUST run synchronously.
         *
         * @returns {StyleCollection}
         *  A reference to this instance.
         */
        this.registerPreviewHandler = function (previewHandler) {
            domPreviewHandlers.push({
                callback: previewHandler,
                import: true
            });
            return this;
        };

        /**
         * Callback handler for the document operation 'insertStyleSheet'.
         * Inserts a new style sheet into this collection.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'insertStyleSheet' document operation.
         *
         * @param {Function} [callback]
         *  A callback that will be invoked with the new style sheet as first
         *  parameter. Used by internal code to create specialized style sheets
         *  with some non-standard properties.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyInsertStyleSheetOperation = function (context, callback) {

            // check that style sheet handling is enabled
            context.ensure(styleSheetSupport, 'creating style sheets not allowed for style family \'%s\'', styleFamily);

            // the unique identifier of the style sheet
            var styleId = context.getStr('styleId');
            // whether the new style sheet shall be registered
            var doRegisterStyleSheet = true;

            // TODO: disallow multiple insertion of style sheets? (predefined style sheets may be created on demand without checking existence)
            withStyleSheet(styleId, function (styleSheet) {
                if (!styleSheet.dirty) {
                    if (context.operation.dirty) {
                        doRegisterStyleSheet = false; // not overwriting an existing registered style sheet with a dirty style sheet (55470)
                    } else {
                        Utils.warn('StyleCollection.applyInsertStyleSheetOperation(): style sheet \'%s\' exists already', styleId);
                    }
                }
            });

            if (!doRegisterStyleSheet) { return; }

            // get or create a style sheet object, set identifier and user-defined name of the style sheet
            var styleSheet = styleSheetMap.getOrConstruct(styleId, Object);
            styleSheet.id = styleId;
            styleSheet.name = context.getOptStr('styleName', styleId, true); // Empty string is allowed

            if (docModel.getApp().isODF()) { checkStyleSheetName(styleSheet); } // 56231

            // set parent of the style sheet, check for cyclic references
            styleSheet.parentId = context.getOptStr('parent');
            if (isDescendantStyleSheet(getStyleSheet(styleSheet.parentId), styleSheet)) {
                Utils.warn('StyleCollection.applyInsertStyleSheetOperation(): cyclic reference, cannot set style sheet parent "' + styleSheet.parentId + '"');
                styleSheet.parentId = null;
            }

            // set style sheet options
            styleSheet.hidden = context.getOptBool('hidden');
            styleSheet.priority = context.getOptInt('uiPriority');
            styleSheet.custom = context.getOptBool('custom');

            // bug 27716: set first visible style as default style, if imported default style is hidden
            withStyleSheet(defaultStyleId, function (defStyleSheet) {
                if (defStyleSheet.hidden && !styleSheet.hidden) {
                    defaultStyleId = styleId;
                }
            });

            // set default style sheet
            if (context.getOptBool('default')) {
                if (defaultStyleId === null) {
                    defaultStyleId = styleId;
                } else if (defaultStyleId !== styleId) {
                    Utils.warn('StyleCollection.applyInsertStyleSheetOperation(): multiple default style sheets "' + defaultStyleId + '" and "' + styleId + '"');
                }
            }

            // store a deep clone of the passed attributes
            styleSheet.attributes = _.copy(context.getOptObj('attrs', {}), true);

            // delete all cached merged attributes (parent tree may have changed)
            deleteCachedAttributeSets();

            // invoke user callback
            if (_.isFunction(callback)) { callback.call(this, styleSheet); }

            // notify listeners (Performance: Do not do this, when missing styles sheets are added after document is loaded)
            if (!docModel.isBlockedGuiUpdate()) { this.trigger('insert:stylesheet', styleId); }
        };

        /**
         * Callback handler for the document operation 'deleteStyleSheet'.
         * Removes an existing style sheet from this collection, and triggers a
         * 'delete:stylesheet' on success.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'deleteStyleSheet' document operation.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyDeleteStyleSheetOperation = function (context) {

            // get the style sheet to be deleted
            var styleId = context.getStr('styleId');
            var styleSheet = getStyleSheet(styleId);
            context.ensure(styleSheet, 'style sheet \'%s\' does not exist', styleId);

            // default style sheet cannot be deleted
            context.ensure(styleId !== defaultStyleId, 'cannot remove default style sheet');

            // update parent of all style sheets referring to the removed style sheet
            styleSheetMap.forEach(function (childSheet) {
                if (styleId === childSheet.parentId) {
                    childSheet.parentId = styleSheet.parentId;
                }
            });

            // special behavior: built-in style sheets will not be deleted but invalidated
            if (styleSheet.builtIn) {
                styleSheet.dirty = true;
                return;
            }

            // remove the style sheet from the map, and delete all cached merged attributes
            // (parent tree may have changed)
            styleSheetMap.remove(styleId);
            deleteCachedAttributeSets();

            // notify listeners
            this.trigger('delete:stylesheet', styleId);
        };

        /**
         * Callback handler for the document operation 'changeStyleSheet'.
         * Changes the properties, or formatting attributes of an existing
         * style sheet in this collection, and triggers a 'change:stylesheet'
         * on success.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'changeStyleSheet' document operation.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyChangeStyleSheetOperation = function (context) {

            // get the style sheet to be changed
            var styleId = context.getStr('styleId');
            var styleSheet = getStyleSheet(styleId);
            context.ensure(styleSheet, 'style sheet \'%s\' does not exist', styleId);

            // change the readable name of the style sheet
            if (context.has('styleName')) { styleSheet.name = context.getStr('styleName'); }

            // change the formatting attributes
            if (context.has('attrs')) { styleSheet.attributes = context.getObj('attrs'); }

            // delete all cached merged attributes (parent tree, or descendant style sheets may have changed)
            deleteCachedAttributeSets();

            // notify listeners
            this.trigger('change:stylesheet', styleId);
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the attribute pool used by this style sheet collection.
         *
         * @returns {AttributePool}
         *  The attribute pool used by this style sheet collection.
         */
        this.getAttributePool = function () {
            return attributePool;
        };

        /**
         * Returns the style family name of this style sheet collection.
         *
         * @returns {String}
         *  The style family name of this style sheet collection.
         */
        this.getStyleFamily = function () {
            return styleFamily;
        };

        /**
         * Returns a set object with the names of all attribute families
         * supported by style sheets contained in this instance, including the
         * main style family.
         *
         * @returns {Object}
         *  A set object containing the names of all supported attribute
         *  families as property keys.
         */
        this.getSupportedFamilies = function () {
            return supportedFamilySet;
        };

        /**
         * Removes all attribute maps from the passed attribute set in-place
         * that are not mapped by an attribute family supported by this style
         * sheet collection. Additionally, all existing but empty attribute
         * maps will be removed.
         *
         * @param {Object} attributeSet
         *  (in/out parameter) The attribute set to be filtered. All properties
         *  of this object whose key is not a supported attribute family will
         *  be deleted.
         *
         * @returns {Object}
         *  The filtered attribute set passed to this method (has been modified
         *  in-place), for convenience.
         */
        this.filterBySupportedFamilies = function (attributeSet) {

            // remove all properties not keyed by a supported attribute family
            _.each(attributeSet, function (attributes, family) {
                if (!(family in supportedFamilySet) || !_.isObject(attributes) || _.isEmpty(attributes)) {
                    delete attributeSet[family];
                }
            });

            // return the modified attribute set for convenience
            return attributeSet;
        };

        /**
         * Removes all unsupported attributes from the passed attribute set
         * in-place. Additionally, all existing but empty attribute maps will
         * be removed.
         *
         * @param {Object} attributeSet
         *  (in/out parameter) The attribute set to be filtered.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.special=false]
         *      If set to true, special attributes (attributes that are marked
         *      with the 'special' flag in the attribute definition) are
         *      considered to be supported. Otherwise, special attributes will
         *      be filtered.
         *
         * @returns {Object}
         *  The filtered attribute set passed to this method (has been modified
         *  in-place), for convenience.
         */
        this.filterBySupportedAttributes = function (attributeSet, options) {

            _.each(attributeSet, function (attributes, family) {

                // attribute definitions of the current family
                var definitionMap = (family in supportedFamilySet) ? attributePool.getRegisteredAttributes(family) : null;

                // process all attributes, if the current attribute family is supported
                if (definitionMap) {
                    _.each(attributes, function (value, name) {
                        if (!definitionMap.isSupported(name, options)) {
                            delete attributes[name];
                        }
                    });

                    if (_.isEmpty(attributes)) {
                        delete attributeSet[family];
                    }
                    return;
                }

                // delete the current property, unless it is a (supported) style sheet reference
                if (!styleSheetSupport || (family !== 'styleId')) {
                    delete attributeSet[family];
                }
            });

            // return the modified attribute set for convenience
            return attributeSet;
        };

        /**
         * Returns the identifier of the default style sheet.
         *
         * @returns {String|Null}
         *  The identifier of the default style sheet.
         */
        this.getDefaultStyleId = function () {
            return defaultStyleId;
        };

        /**
         * Specifying whether the default style shall be used, if no style ID is specified
         * for an element.
         *
         * @param {Boolean} value
         *  Whether the default style shall be used, if no style ID is specified for an element.
         */
        this.setUseDefaultStyleIfMissing = function (value) {
            useDefaultStyleIfMissing = value;
        };

        /**
         * Inserts a new 'dirty' (predefined) style sheet into this collection.
         *
         * @param {String} styleId
         *  The unique identifier of of the new style sheet.
         *
         * @param {String} styleName
         *  The user-defined name of of the new style sheet.
         *
         * @param {Object} attributes
         *  The formatting attributes contained in the new style sheet. The
         *  structure of this object is dependent on the style family of this
         *  collection.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.parent]
         *      The identifier of of the parent style sheet the new style sheet
         *      will derive undefined attributes from.
         *  @param {String} [options.category]
         *      If specified, the name of a category that can be used to group
         *      style sheets into several sections in the user interface.
         *  @param {Number} [options.priority=0]
         *      The sorting priority of the style (the lower the value the
         *      higher the priority).
         *  @param {Boolean} [options.defStyle=false]
         *      True, if the new style sheet is the default style sheet of this
         *      style sheet collection. The default style will be used for all
         *      elements without explicit style sheet. Only the first style
         *      sheet that will be inserted with this flag will be permanently
         *      registered as default style sheet.
         *  @param {Boolean} [options.hidden=false]
         *      If set to true, the style sheet should be displayed in the user
         *      interface.
         *  @param {Boolean} [options.builtIn=false]
         *      Whether the style sheet is predefined (provided in a new empty
         *      document).
         *  @param {Boolean} [options.custom=false]
         *      Whether the style sheet has been customized (changed by the
         *      user, especially useful for built-in style sheets).
         *
         * @returns {Boolean}
         *  Whether the new style sheet has been created successfully.
         */
        this.createDirtyStyleSheet = function (styleId, styleName, attributeSet, options) {

            // the document operation that will be executed to insert the style sheet
            var operation = createOperationProperties(styleId, { styleName: styleName, attrs: attributeSet, dirty: true });

            // add parent style identifier
            var parentStyle = Utils.getStringOption(options, 'parent', null);
            if (parentStyle) { operation.parent = parentStyle; }

            // add other options supported by the operation
            operation.uiPriority = Utils.getIntegerOption(options, 'priority', 0);
            operation.hidden = Utils.getBooleanOption(options, 'hidden', false);
            operation.custom = Utils.getBooleanOption(options, 'custom', false);

            // apply the operation (should not throw, just in case)
            try {
                var context = new OperationContext(docModel, operation, false);
                this.applyInsertStyleSheetOperation(context, function (styleSheet) {
                    styleSheet.category = Utils.getStringOption(options, 'category', null);
                    styleSheet.builtIn = Utils.getBooleanOption(options, 'builtIn', false);
                    styleSheet.dirty = true;
                });
                return true;
            } catch (error) {
                Utils.exception(error);
                return false;
            }
        };

        /**
         * Returns whether this collection contains the specified style sheet.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @returns {Boolean}
         *  Whether this instance contains a style sheet with the passed
         *  identifier.
         */
        this.containsStyleSheet = function (styleId) {
            return hasStyleSheet(styleId);
        };

        /**
         * Returns whether this collection contains a style sheet with the
         * specified name (not identifier).
         *
         * @param {String} styleName
         *  The name of the style sheet.
         *
         * @returns {Boolean}
         *  Whether this instance contains a style sheet with the passed name.
         */
        this.containsStyleSheetByName = function (styleName) {
            styleName = styleName.toLowerCase();
            return styleSheetMap.some(function (styleSheet) {
                return styleSheet.name.toLowerCase() === styleName;
            });
        };

        /**
         * Returns the style identifier of the first style sheet found with the
         * specified name.
         *
         * @param {String} styleName
         *  The name of the style sheet.
         *
         * @returns {String|Null}
         *  The identifier of a style sheet with the passed name; or null, if
         *  no such style sheet exists.
         */
        this.getStyleIdByName = function (styleName) {
            var resultId = null;
            styleName = styleName.toLowerCase();
            styleSheetMap.some(function (styleSheet, styleId) {
                if (styleSheet.name.toLowerCase() === styleName) {
                    resultId = styleId;
                    return true;
                }
            });
            return resultId;
        };

        /**
         * Returns the names of all style sheets in a map, keyed by their
         * unique identifiers.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.skipHidden=false]
         *      If set to true, the names of hidden style sheets will not be
         *      returned.
         *
         * @returns {Object}
         *  A map with all style sheet names, keyed by style sheet identifiers.
         */
        this.getStyleSheetNames = function (options) {

            // whether to skip hidden style sheets
            var skipHidden = Utils.getBooleanOption(options, 'skipHidden', false);
            // the result map
            var names = {};

            styleSheetMap.forEach(function (styleSheet, styleId) {
                if (!skipHidden || !styleSheet.hidden) {
                    names[styleId] = styleSheet.name;
                }
            });
            return names;
        };

        /**
         * Returns whether a specific style sheet is a descendant of another
         * style sheet, including the case that the specified style sheets are
         * identical.
         *
         * @param {String} styleId
         *  The identifier of a style sheet that may be a descendant of the
         *  other style sheet.
         *
         * @param {String} parentStyleId
         *  The identifier of a style sheet that may be a parent of the other
         *  style sheet.
         *
         * @returns {Boolean}
         *  Whether the first style sheet is a descendant of the second style
         *  sheet, or both style sheet identifiers are equal.
         */
        this.isChildOfOther = function (styleId, parentStyleId) {

            // check the identifier itself, and all existing parent identifiers
            for (; styleId; styleId = this.getParentId(styleId)) {
                if (styleId === parentStyleId) { return; }
            }

            return false;
        };

        /**
         * Returns the merged attributes from the specified style sheet and its
         * parent style sheets.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.skipDefaults=false]
         *      If set to true, the returned attribute set will only contain
         *      the explicit attributes of the style sheet and its ancestors,
         *      but NOT the default values for missing attributes. This means
         *      the attribute set returned from this method MAY NOT be
         *      complete.
         *
         * @returns {Object}
         *  A complete (except when using the option 'skipDefaults', see above)
         *  attribute set (a map of attribute maps with name/value pairs, keyed
         *  by the attribute families, additionally containing the style sheet
         *  reference in the 'styleId' property), containing values for all
         *  supported attributes, collected from the specified style sheet, and
         *  all its parent style sheets, up to the map of default attributes.
         *  For efficiency, a reference to the cached attribute set will be
         *  returned. ATTENTION! For efficiency, a reference to the cached
         *  attribute set will be returned. This object MUST NOT be changed!
         */
        this.getStyleAttributeSet = function (styleId, options) {

            // the style sheet entry
            var styleSheet = getStyleSheet(getEffectiveStyleId(styleId));
            // whether to skip the default attribute values
            var skipDefaults = Utils.getBooleanOption(options, 'skipDefaults', false);
            // the cached incomplete merged style sheet attributes
            var sparseAttributeSet = styleSheet ? styleSheet.sparseAttributes : null;
            // the cached complete merged style sheet attributes
            var mergedAttributeSet = styleSheet ? styleSheet.mergedAttributes : null;

            // the effective style sheet attributes according to passed options
            var attributeSet = skipDefaults ? sparseAttributeSet : mergedAttributeSet;
            if (attributeSet) { return attributeSet; }

            // calculate the existing attributes of the style sheet, and its ancestors
            if (!sparseAttributeSet) {
                sparseAttributeSet = resolveStyleAttributeSet(styleId);
                if (styleSheet) { styleSheet.sparceAttributes = sparseAttributeSet; }
            }

            // return the incomplete attributes if requested
            if (skipDefaults) { return sparseAttributeSet; }

            // start with the attribute default values, extend with attributes of specified style sheet
            attributeSet = resolveBaseAttributeSet();
            attributePool.extendAttributeSet(attributeSet, sparseAttributeSet);
            if (styleSheet) { styleSheet.mergedAttributes = attributeSet; }
            return attributeSet;
        };

        /**
         * Returns the merged attribute set of the default style sheet of this
         * collection.
         *
         * @returns {Object}
         *  The complete merged attribute set of the default style sheet of
         *  this collection.
         *  ATTENTION! For efficiency, a reference to the cached attribute set
         *  will be returned. This object MUST NOT be changed!
         */
        this.getDefaultStyleAttributeSet = function () {
            return this.getStyleAttributeSet(defaultStyleId || '');
        };

        /**
         * Returns the UI category for the specified style sheet.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @returns {String}
         *  The UI category. If the style sheet does not exist, returns null.
         */
        this.getUICategory = function (styleId) {
            var styleSheet = getStyleSheet(styleId);
            return styleSheet ? styleSheet.category : null;
        };

        /**
         * Returns the UI priority for the specified style sheet.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @returns {Number}
         *  The UI priority. If the style sheet does not exist, returns 0.
         */
        this.getUIPriority = function (styleId) {
            var styleSheet = getStyleSheet(styleId);
            return styleSheet ? styleSheet.priority : 0;
        };

        /**
         * Returns the identifier of the parent style sheet for the specified
         * style sheet.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @returns {String|Null}
         *  The parent id of the style sheet; or null, if no parent exists.
         */
        this.getParentId = function (styleId) {
            var styleSheet = getStyleSheet(styleId);
            return styleSheet ? styleSheet.parentId : null;
        };

        /**
         * Returns whether the specified style sheet is hidden in the user
         * interface.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @returns {Boolean|Null}
         *  True, if the specified style sheet is hidden; false, if the style
         *  sheet is visible; or null, if the style sheet does not exist.
         */
        this.isHidden = function (styleId) {
            var styleSheet = getStyleSheet(styleId);
            return styleSheet ? styleSheet.hidden : null;
        };

        /**
         * Returns the user defined name for the specified style sheet.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @returns {String|Null}
         *  The user defined name of the style sheet; or null, if the style
         *  sheet does not exist.
         */
        this.getName = function (styleId) {
            var styleSheet = getStyleSheet(styleId);
            return styleSheet ? styleSheet.name : null;
        };

        /**
         * @param {String} styleId
         *
         * @returns {Boolean|Null}
         */
        this.isCustom = function (styleId) {
            var styleSheet = getStyleSheet(styleId);
            return styleSheet ? styleSheet.custom : null;
        };

        /**
         * Returns whether the specified style sheet is dirty.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @returns {Boolean|Null}
         *  The dirty state of the style sheet; or null, if the style sheet
         *  does not exist.
         */
        this.isDirty = function (styleId) {
            var styleSheet = getStyleSheet(styleId);
            return styleSheet ? styleSheet.dirty : null;
        };

        /**
         * Changes the dirty state of the specified style sheet.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @param {Boolean} dirty
         *
         * @returns {StyleCollection}
         *  A reference to this instance.
         */
        this.setDirty = function (styleId, dirty) {
            withStyleSheet(styleId, function (styleSheet) {
                styleSheet.dirty = dirty;
            });
            return this;
        };

        /**
         * Returns the explicit attribute set of the specified style sheet,
         * without resolving the parent style sheets, or converting to the
         * attributes of a specific attribute family.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @param {Boolean} [direct=false]
         *  If set to true, the returned attribute set will be a reference to
         *  the original map stored in this instance, which MUST NOT be
         *  modified! By default, a deep clone of the attribute set will be
         *  returned that can be freely modified.
         *
         * @returns {Object}
         *  The explicit attribute set contained in the style sheet.
         */
        this.getStyleSheetAttributeMap = function (styleId, direct) {
            var styleSheet = getStyleSheet(styleId);
            return !styleSheet ? {} : (direct === true) ? styleSheet.attributes : _.copy(styleSheet.attributes, true);
        };

        /**
         * Changes all given attributes in the style sheet, does not trigger
         * any listener or change anything in the DOM element is in use for
         * attributes: priority, dirty, builtIn.
         *
         * @param {String} styleId
         *
         * @param {Object} [options]
         *
         * @returns {StyleCollection}
         *  A reference to this instance.
         */
        this.setStyleOptions = function (styleId, options) {
            withStyleSheet(styleId, function (styleSheet) {
                _.extend(styleSheet, options);
            });
            return this;
        };

        /**
         * Builds a complete attribute set containing all formatting attributes
         * of the own style family, and of the additional attribute families
         * supported by the style sheets in this collection, all set to the
         * null value.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.style=false]
         *      If set to true, the resulting attribute set will contain an
         *      additional property 'styleId' (reference to the style sheet),
         *      also set to the value null.
         *
         * @returns {Object}
         *  A complete attribute set with null values for all attributes
         *  registered for all supported attribute families.
         */
        this.buildNullAttributeSet = function (options) {
            var attributeSet = attributePool.buildNullValueSet(supportedFamilySet);
            if (styleSheetSupport && Utils.getBooleanOption(options, 'style', false)) {
                attributeSet.styleId = null;
            }
            return attributeSet;
        };

        /**
         * Convenience shortcut to extend the passed attribute sets using the
         * attribute pool of this instance. See description of the method
         * AttributePool.extendAttributeSet() for more details.
         *
         * @param {Object} attributeSet1
         *  (in/out) The first attribute set that will be extended in-place.
         *
         * @param {Object|Null} attributeSet2
         *  The second attribute set, whose attribute values will be inserted
         *  into the first attribute set. If set to null, the first attribute
         *  set will bot be changed.
         *
         * @param {Object} [options]
         *  Optional parameters. See method AttributePool.extendAttributeSet()
         *  for more details.
         *
         * @returns {Object}
         *  A reference to the first passed and extended attribute set.
         *
         * @returns {Object}
         */
        this.extendAttributeSet = function (attributeSet1, attributeSet2, options) {
            return attributePool.extendAttributeSet(attributeSet1, attributeSet2, options);
        };

        /**
         * Returns the merged values of the formatting attributes from a style
         * sheet and explicit attributes.
         *
         * @param {Object} attributes
         *  An (incomplete) attribute set containing an optional style sheet
         *  identifier in the property 'styleId', and explicit attributes.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.special=false]
         *      If set to true, includes special attributes (attributes that
         *      are marked with the 'special' flag in the attribute definitions
         *      passed to the constructor) to the result set.
         *  @param {Boolean} [options.skipDefaults=false]
         *      If set to true, the returned attribute set will NOT contain the
         *      default values of attributes not contained in the style sheet
         *      (or its ancestors). This means the attribute set returned from
         *      this method MAY NOT be complete.
         *
         * @returns {Object}
         *  A complete (except when using the option 'skipDefaults', see above)
         *  merged attribute set with the values of all formatting attributes
         *  supported by this collection.
         */
        this.getMergedAttributes = function (attributes, options) {

            // the identifier of the style sheet
            var styleId = Utils.getStringOption(attributes, 'styleId', '');
            // resulting merged attributes (start with defaults and style sheet attributes)
            var mergedAttributes = _.copy(this.getStyleAttributeSet(styleId, options), true);

            // add the passed explicit attributes (protect effective style sheet identifier)
            styleId = mergedAttributes.styleId;
            attributePool.extendAttributeSet(mergedAttributes, attributes, options);
            mergedAttributes.styleId = styleId;

            // filter by supported attributes, according to the passed options
            return this.filterBySupportedAttributes(mergedAttributes, options);
        };

        /**
         * Updates the CSS formatting of the specified DOM element, according
         * to the passed merged attribute set.
         *
         * @param {HTMLElement|jQuery} element
         *  The element to be updated. If this object is a jQuery collection,
         *  uses the first DOM node it contains.
         *
         * @param {Object} mergedAttributes
         *  A complete attribute set with all attribute values merged from
         *  style sheet and explicit attributes.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.preview=false]
         *      If set to true, a GUI element used to preview the formatting of
         *      a style sheet will be rendered. Executes the 'preview' callback
         *      functions instead of the regular 'format' callback functions
         *      registered in the attribute definitions, and calls the preview
         *      handler instead of the format handler registered at this class
         *      instance.
         *  @param {Boolean} [options.async=false]
         *      If set to true, the registered format handler will be asked to
         *      execute asynchronously in a browser timeout loop. MUST NOT be
         *      used together with the option 'preview'.
         *  @param {Boolean} [options.duringImport=false]
         *      If set to true, all formatting handlers will be invoked while
         *      importing the document. By default, only the formatting
         *      handlers that have been registered with the 'duringImport' flag
         *      will be invoked in that case.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the registered format or
         *  preview handlers have finished. If the option 'async' is set, and
         *  the format handlers support asynchronous execution, the promise
         *  will be resolved sometime in the future. Otherwise, the returned
         *  promise is already resolved when this method returns.
         */
        this.applyElementFormatting = function (element, mergedAttributes, options) {

            // the passed elemenet, as jQuery object
            var $element = $(element);
            // whether to render a GUI preview element
            var preview = Utils.getBooleanOption(options, 'preview', false);
            // pass the 'async' flag to the format handlers
            var async = !preview && Utils.getBooleanOption(options, 'async', false);
            // the array with all registered format handlers (according to preview mode)
            var currFormatHandlers = preview ? domPreviewHandlers : domFormatHandlers;
            // whether to skip element formatting while importing the document
            var skipDuringImport = !this.isImportFinished() && !Utils.getBooleanOption(options, 'duringImport', false);
            // the return value of a format handler
            var formatHandlerValue = null;

            // update information for debugging
            if (!preview && Config.DEBUG) {
                self.DBG_COUNT = (self.DBG_COUNT || 0) + 1;
            }

            // invokes the specified format handler
            function invokeHandler(handlerData) {
                if (!skipDuringImport || handlerData.import) {
                    return handlerData.callback.call(self, $element, mergedAttributes, async);
                }
            }

            // invoke all registered formatting handlers asynchronously if specified
            // Info: Async is only used for table formatting during loading the (text) document.
            if (async) {
                return this.iterateArraySliced(currFormatHandlers, invokeHandler, 'StyleCollection.applyElementFormatting');
            }

            // invoke all registered formatting handlers synchronously (TODO: some handlers still return pending promises!)
            currFormatHandlers.forEach(function (handlerData) {
                formatHandlerValue = invokeHandler(handlerData);
            });

            // INFO: Only the table format handler returns a promise.
            // TODO: There are two problems:
            //       1. Although async is NOT specified, this function returns an unresolved promise.
            //       2. If more than one format handler is specified, only the return value of the last
            //          format handler is used here.

            return (formatHandlerValue && Utils.isPromise(formatHandlerValue)) ? formatHandlerValue : self.createResolvedPromise();
        };

        /**
         * Calculates the merged attribute set, and updates the CSS formatting
         * of the specified DOM element, according to its current explicit
         * attributes and style settings.
         *
         * @param {HTMLElement|jQuery} element
         *  The element to be updated. If this object is a jQuery collection,
         *  uses the first DOM node it contains.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Object} [options.baseAttributes]
         *      An attribute set with all attribute values of the ancestor of
         *      the passed element, merged from style sheet and explicit
         *      attributes, keyed by attribute family. Used internally while
         *      updating the formatting of all child elements of a node.
         *  @param {Boolean} [options.preview=false]
         *      If set to true, a GUI element used to preview the formatting of
         *      a style sheet will be rendered. Executes the 'preview' callback
         *      functions instead of the regular 'format' callback functions
         *      registered in the attribute definitions, and calls the preview
         *      handler instead of the format handler registered at this class
         *      instance.
         *  @param {Boolean} [options.async=false]
         *      If set to true, the registered format handler will be asked to
         *      execute asynchronously in a browser timeout loop. MUST NOT be
         *      used together with the option 'preview'.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the registered format or
         *  preview handlers have finished. If the option 'async' is set, and
         *  the format handlers support asynchronous execution, the promise
         *  will be resolved sometime in the future. Otherwise, the returned
         *  promise is already resolved when this method returns.
         */
        this.updateElementFormatting = function (element, options) {

            // the resulting merged attribute set
            var mergedAttributes = null;

            // caller may pass precalculated base attributes, e.g. of a parent DOM element
            var baseAttributes = Utils.getObjectOption(options, 'baseAttributes', null);
            if (baseAttributes) {
                mergedAttributes = attributePool.getDefaultValueSet(supportedFamilySet); // all families required
                attributePool.extendAttributeSet(mergedAttributes, baseAttributes);
            } else {
                mergedAttributes = resolveBaseAttributeSet(element);
            }

            // extend with custom base attributes returned by user callback function
            attributePool.extendAttributeSet(mergedAttributes, resolveCustomBaseAttributeSet(element, mergedAttributes));

            // add attributes of the style sheet and its parents
            var styleId = AttributeUtils.getElementStyleId(element);
            attributePool.extendAttributeSet(mergedAttributes, resolveStyleAttributeSet(styleId, element));

            // add the explicit attributes of the element
            attributePool.extendAttributeSet(mergedAttributes, resolveElementAttributeSet(element), { special: true });

            // apply the CSS formatting of the DOM element
            return this.applyElementFormatting(element, mergedAttributes, _.extend({}, options, { duringImport: true }));
        };

        /**
         * Returns the values of the formatting attributes in the specified DOM
         * element.
         *
         * @param {HTMLElement|jQuery} element
         *  The element whose attributes will be returned. If this object is a
         *  jQuery collection, uses the first DOM node it contains.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {HTMLElement|jQuery} [options.sourceNode]
         *      A descendant of the passed element associated to a child
         *      attribute family. Will be passed to a style attribute resolver
         *      callback function where it might be needed to resolve the
         *      correct attributes according to the position of this source
         *      node.
         *  @param {Object} [options.baseAttributes]
         *      An attribute set with all attribute values of the ancestor of
         *      the passed element, merged from style sheet and explicit
         *      attributes, keyed by attribute family. Used internally while
         *      updating the formatting of all child elements of a node.
         *  @param {Boolean} [options.special=false]
         *      If set to true, includes special attributes (attributes that
         *      are marked with the 'special' flag in the attribute definitions
         *      passed to the constructor) to the result map.
         *  @param {Boolean} [options.noDefaults=false]
         *      If set to true, the default attributes will not be included
         *      into the generated merged attributes. This is especially useful,
         *      when operations shall be generated with all available attributes,
         *      for example after copy/paste of template drawings. In this case
         *      it is better to omit those attributes, that are set by default.
         *
         * @returns {Object}
         *  A complete merged attribute set with the values of all formatting
         *  attributes supported by this collection.
         */
        this.getElementAttributes = function (element, options) {

            // the descendant source node
            var sourceNode = Utils.getOption(options, 'sourceNode');
            // the identifier of the style sheet referred by the element
            var styleId = AttributeUtils.getElementStyleId(element);
            // resulting merged attributes (start with defaults, parent element, and style sheet attributes)
            var mergedAttributes = null;
            // caller may pass precalculated base attributes, e.g. of a parent DOM element
            var baseAttributes = Utils.getObjectOption(options, 'baseAttributes', null);
            // whether the default values should be part of the merged attributes
            var noDefaults = Utils.getBooleanOption(options, 'noDefaults', false);

            if (baseAttributes) {
                mergedAttributes = noDefaults ? null : attributePool.getDefaultValueSet(supportedFamilySet); // all families required
                attributePool.extendAttributeSet(mergedAttributes, baseAttributes);
            } else {
                mergedAttributes = resolveBaseAttributeSet(element, { noDefaults: noDefaults });
            }

            // extend with custom base attributes returned by user callback function
            attributePool.extendAttributeSet(mergedAttributes, resolveCustomBaseAttributeSet(element, mergedAttributes));

            // add attributes of the style sheet and its parents
            attributePool.extendAttributeSet(mergedAttributes, resolveStyleAttributeSet(styleId, element, sourceNode));

            // add the explicit attributes of the element (protect effective style sheet identifier)
            styleId = mergedAttributes.styleId;
            attributePool.extendAttributeSet(mergedAttributes, resolveElementAttributeSet(element, sourceNode), options);
            mergedAttributes.styleId = styleId;

            // filter by supported attributes, according to the passed options
            return this.filterBySupportedAttributes(mergedAttributes, options);
        };

        /**
         * Changes specific formatting attributes in the specified DOM element.
         *
         * @param {HTMLElement|jQuery} element
         *  The element whose attributes will be changed. If this object is a
         *  jQuery collection, uses the first DOM node it contains.
         *
         * @param {Object} attributes
         *  An (incomplete) attribute set with all formatting attributes to be
         *  changed. To clear an explicit attribute value (thus defaulting to
         *  the current style sheet), the respective value in this attribute
         *  set has to be set explicitly to the null value. Attributes missing
         *  in this attribute set will not be modified at the element.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.clear=false]
         *      If set to true, explicit element attributes that are equal to
         *      the attributes of the current style sheet will be removed from
         *      the element.
         *  @param {Boolean} [options.special=false]
         *      If set to true, allows to change special attributes (attributes
         *      that are marked with the 'special' flag in the attribute
         *      definitions passed to the constructor).
         *  @param {Boolean} [options.preview=false]
         *      If set to true, a GUI element used to preview the formatting of
         *      a style sheet will be rendered. Executes the 'preview' callback
         *      functions instead of the regular 'format' callback functions
         *      registered in the attribute definitions, and calls the preview
         *      handler instead of the format handler registered at this class
         *      instance.
         *  @param {Function} [options.changeListener]
         *      If specified, will be called if the attributes of the element
         *      have been changed. Will be called in the context of this style
         *      sheet collection. Receives the passed element as first
         *      parameter, the old explicit attribute set as second parameter,
         *      and the new explicit attribute set as third parameter.
         *
         * @returns {StyleCollection}
         *  A reference to this instance.
         */
        this.setElementAttributes = function (element, attributes, options) {

            // whether to remove element attributes equal to style attributes
            var clear = Utils.getBooleanOption(options, 'clear', false);
            // whether a preview element is rendered
            var preview = Utils.getBooleanOption(options, 'preview', false);
            // change listener notified for changed attributes
            var changeListener = Utils.getFunctionOption(options, 'changeListener');
            // new style sheet identifier
            var styleId = attributes.styleId;

            // the existing explicit element attributes
            var oldElementAttributes = AttributeUtils.getExplicitAttributes(element, { direct: true });
            // new explicit element attributes (clone, there may be multiple elements pointing to the same data object)
            var newElementAttributes = _.copy(oldElementAttributes, true);
            // merged attribute values from style sheets and explicit attributes
            var mergedAttributes = resolveBaseAttributeSet(element);

            // add/remove new style sheet identifier, or get current style sheet identifier
            if (_.isString(styleId)) {
                newElementAttributes.styleId = styleId;
            } else if (_.isNull(styleId)) {
                delete newElementAttributes.styleId;
            } else {
                styleId = AttributeUtils.getElementStyleId(element);
            }

            // extend with custom base attributes returned by user callback function
            attributePool.extendAttributeSet(mergedAttributes, resolveCustomBaseAttributeSet(element, mergedAttributes, attributes));

            // collect all attributes of the new or current style sheet, and its parents
            attributePool.extendAttributeSet(mergedAttributes, resolveStyleAttributeSet(styleId, element));

            // add or remove the passed explicit attributes
            attributes = _.copy(attributes, true);
            this.filterBySupportedFamilies(attributes);
            _.each(attributes, function (attributeValues, family) {
                if (_.isObject(attributeValues)) {

                    // definitions of own attributes
                    var definitionMap = attributePool.getRegisteredAttributes(family);
                    // add an attribute map for the current family
                    var elementAttributeValues = newElementAttributes[family] || (newElementAttributes[family] = {});

                    // update the attribute map with the passed attributes
                    definitionMap.forEachSupported(attributeValues, function (value, name) {
                        // check whether to clear the attribute
                        if ((value === null) || (clear && _.isEqual(mergedAttributes[family][name], value))) {
                            delete elementAttributeValues[name];
                        } else {
                            elementAttributeValues[name] = value;
                        }
                    }, options);

                    // remove empty attribute value maps completely
                    if (_.isEmpty(elementAttributeValues)) {
                        delete newElementAttributes[family];
                    }
                }
            });

            // check if any attributes have been changed
            if (preview || !_.isEqual(oldElementAttributes, newElementAttributes)) {

                // write back new explicit attributes to the element
                AttributeUtils.setExplicitAttributes(element, newElementAttributes);

                // merge explicit attributes into style attributes
                attributePool.extendAttributeSet(mergedAttributes, resolveElementAttributeSet(element), options);

                // apply CSS formatting of the DOM element
                this.applyElementFormatting($(element), mergedAttributes, { preview: preview });

                // call the passed change listener
                if (_.isFunction(changeListener)) {
                    changeListener.call(this, element, oldElementAttributes, newElementAttributes);
                }
            }

            return this;
        };

        // operation generators -----------------------------------------------

        /**
         * Generates an 'insertStyleSheet' operation, if the specified style
         * sheet exists in this style sheet collection, and is marked dirty.
         *
         * @param {OperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {String} styleId
         *  The identifier of the style sheet.
         *
         * @returns {StyleCollection}
         *  A reference to this instance.
         */
        this.generateMissingStyleSheetOperations = function (generator, styleId) {
            withStyleSheet(styleId, function (styleSheet) {

                // nothing to do, if the style sheet is not dirty
                if ((styleId === defaultStyleId) || !styleSheet.dirty) { return; }

                // the properties for the shyle sheet operations
                var properties = createOperationProperties(styleId);

                // generate the 'deleteStyleSheet' operation for undo
                generator.generateOperation(Operations.DELETE_STYLESHEET, properties, { undo: true });

                // add more properties for style sheet insertion
                properties.styleName = styleSheet.name;
                properties.attrs = styleSheet.attributes;
                if (styleSheet.parentId) { properties.parent = styleSheet.parentId; }
                if (_.isNumber(styleSheet.priority)) { properties.uiPriority = styleSheet.priority; }
                if (styleSheet.hidden) { properties.hidden = true; }
                if (styleSheet.custom) { properties.custom = true; }

                // generate the 'insertStyleSheet' operation, remove the dirty flag from the style sheet
                generator.generateOperation(Operations.INSERT_STYLESHEET, properties);
                styleSheet.dirty = false;
            });

            return this;
        };

        // initialization -----------------------------------------------------

        // add additional attribute families passed in the options
        supportedFamilySet = Utils.makeSet(Utils.getTokenListOption(initOptions, 'families', []));
        supportedFamilySet[styleFamily] = true;

        // destroy class members on destruction
        this.registerDestructor(function () {
            self = docModel = initOptions = attributePool = styleSheetMap = supportedFamilySet = null;
            parentResolvers = styleAttributesResolver = elementAttributesResolver = baseAttributesResolver = null;
            domFormatHandlers = domPreviewHandlers = null;
        });

    } // class StyleCollection

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: StyleCollection });

});
